/*
 * $Id: attrib.h,v 1.26 2001/12/15 05:13:08 antona Exp $
 *
 * attrib.h - Exports for attribute handling. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef ATTRIB_H
#define ATTRIB_H

#include "types.h"
#include "support.h"
#include "unistr.h"

/* Attr flag bit values */
#define AT_mapped		0
#define AT_dirty		1
#define AT_error		2
#define AT_nonresident		3
#define AT_in_attr_list		4
#define AT_in_ext_record	5
				/* bits 6-31 reserved for future use */

#define AttrMapped(a)		test_bit(AT_mapped, (a)->a_flags)
#define SetAttrMapped(a)	((a)->a_flags |= 1 << AT_mapped)
#define ClearAttrMapped(a)	clear_bit(AT_mapped, (a)->a_flags)

#define AttrDirty(a)		test_bit(AT_dirty, (a)->a_flags)
#define SetAttrDirty(a)		set_bit(AT_dirty, (a)->a_flags)
#define ClearAttrDirty(a)	clear_bit(AT_dirty, (a)->a_flags)

#define AttrError(a)		test_bit(AT_error, (a)->a_flags)
#define SetAttrError(a)		set_bit(AT_error, (a)->a_flags)
#define ClearAttrError(a)	clear_bit(AT_error, (a)->a_flags)

#define AttrNonResident(a)	test_bit(AT_nonresident, (a)->a_flags)
#define SetAttrNonResident(a)	set_bit(AT_nonresident, (a)->a_flags)
#define ClearAttrNonResident(a)	clear_bit(AT_nonresident, (a)->a_flags)

#define AttrInAttrList(a)	test_bit(AT_in_attr_list, (a)->a_flags)
#define SetAttrInAttrList(a)	set_bit(AT_in_attr_list, (a)->a_flags)
#define ClearAttrInAttrList(a)	clear_bit(AT_in_attr_list, (a)->a_flags)

#define AttrInExtRecord(a)	test_bit(AT_in_ext_record, (a)->a_flags)
#define SetAttrInExtRecord(a)	set_bit(AT_in_ext_record, (a)->a_flags)
#define ClearAttrInExtRecord(a)	clear_bit(AT_in_ext_record, (a)->a_flags)

int write_non_resident_attr(attr *a);

int flush_attr(attr *a);

BOOL __set_attr_dirty(attr *a);

static __inline__ BOOL set_attr_dirty(attr *a)
{
	if (test_and_set_bit(AT_dirty, a->a_flags) || __set_attr_dirty(a))
		return TRUE;
	return flush_attr(a) == 0;
}

extern __inline__ attr *__allocate_attr(void);

extern __inline__ int __free_attr(attr *a);

int __map_attr_value(attr *me);

static __inline__ int map_attr_value(attr *a)
{
	a->a_count++;
	if (AttrMapped(a))
		return 0;
	return __map_attr_val(a);
}

static __inline__ void unmap_attr_value(attr *a)
{
	if (AttrMapped(a) && a->a_count > 0)
		a->a_count--;
}

int __unmap_attr_value(attr *a);

/**
 *
 * Into mft_entry.
 */
int insert_attr_in_mft_entry(mft_entry *me, attr **a, const BOOL dirty);

int remove_attr(attr *a);

/**
 * decompress_run_list - decompress a mapping pairs array into a run list
 * @attr:	non-resident attribute whose mapping pairs array to decompress
 * 
 * Allocate a run_list array and decompress the run-list of the non-resident
 * attribute @attr into the run_list array.
 * 
 * Return a pointer to the run_list array or NULL on error.
 * Notes: caller has to free the run_list array when finished.
 */
run_list *decompress_run_list(const ATTR_RECORD *attr);

/**
 * vcn_to_lcn - determine the lcn corresponding to a particular vcn
 * @rl:		decompressed mapping pairs array (i.e. run list)
 * @vcn:	virtucal cluster number for which to determine the lcn
 *
 * Determine the logical cluster number given the virtual cluster number @vcn
 * and the run list @rl describing the mapping between the LCNs and VCNs. 
 * 
 * Return values:
 * 
 * 	> 0: The lcn corresponding to the vcn.
 * 	
 * 	0: Check errno:
 * 		0	The lcn corresponding to the vcn is really zero.
 *		ENOENT	We are looking for a vcn which doesn't exist.
 * 		EINVAL	Either @rl or @vcn are invalid.
 * 		
 *	-1: The lcn is sparse, i.e. it's contents are zero and there is no disk
 *	    space allocated for it.
 */
LCN vcn_to_lcn(const run_list *rl, const VCN vcn);

/**
 * set_ntfs_volume_flags - set the flags of an ntfs volume
 * @v:		ntfs volume the mft record @b is from
 * @b:		pointer to a buffer containing the mft record of FILE_$Volume
 * @flags:	new flags
 * 
 * Set the volume flags in the mft record buffer @b and on volume @v to @flags.
 * Return 1 on success or 0 on error.
 */
int set_ntfs_volume_flags(ntfs_volume *v, MFT_RECORD *b, const __u16 flags);

/**
 * attr_search_context - used in attribute search functions
 * @mrec:	buffer containing mft record to search
 * @attr:	attribute record in @mrec where to begin/continue search
 * @is_first:	if true lookup_attr() begins search with @attr, else after @attr
 *
 * Structure must be initialized to zero before the first call to one of the
 * attribute search functions. If the mft record in which to search has already
 * been loaded into memory, then initialize @mrec to point to it, @attr to
 * point to the first attribute within @mrec, and set @is_first to TRUE.
 *
 * @is_first is only honoured when @mrec is not NULL. Then, if @is_first is
 * TRUE, the search begins with @attr. If @is_first is FALSE, the search begins
 * after @attr. This is so that, after the first call to one of the search
 * attribute functions, we can call the function again, without any
 * modification of the search context, to automagically get the next matching
 * attribute.
 */
typedef struct {
	MFT_RECORD *mrec;
	ATTR_RECORD *attr;
	BOOL is_first;
} attr_search_context;

/*
 * See attrib.c for descriptions of the below functions.
 */

extern __inline__ BOOL find_first_attr(const ATTR_TYPES type,
		const uchar_t *name, const __u32 name_len,
		const IGNORE_CASE_BOOL ic, const uchar_t *upcase,
		const __u32 upcase_len, const __u8 *val, const __u32 val_len,
		attr_search_context *ctx);

BOOL find_attr(const ATTR_TYPES type, const uchar_t *name, const __u32 name_len,
		const IGNORE_CASE_BOOL ic, const uchar_t *upcase,
		const __u32 upcase_len, const __u8 *val, const __u32 val_len,
		attr_search_context *ctx);

/**
 * get_attribute_value_length - return the length of the value of an attribute
 * @a:	pointer to a buffer containing the attribute record
 *
 * Return the byte size of the attribute value of the attribute @a (as it
 * would be after eventual decompression and filling in of holes if sparse).
 * If we return 0, check errno. If errno is 0 the actual length was 0, 
 * otherwise errno describes the error.
 *
 * FIXME: Describe possible errnos.
 */
__s64 get_attribute_value_length(const ATTR_RECORD *a);

/**
 * get_attribute_value - return the attribute value of an attribute
 * @vol:	volume on which the attribute is present
 * @a:		attribute to get the value of
 * @b:		destination buffer for the attribute value
 * 
 * Make a copy of the attribute value of the attribute @a into the destination
 * buffer @b. Note, that the size of @b has to be at least equal to the value
 * returned by get_attribute_value_length(@a).
 *
 * Return number of bytes copied. If this is zero check errno. If errno is 0
 * then nothing was read due to a zero-length attribute value, otherwise
 * errno describes the error.
 */
__s64 get_attribute_value(const ntfs_volume *vol, const MFT_RECORD *m,
			  const ATTR_RECORD *a, __u8 *b);

/**
 * set_attribute_value - set the attribute value of an attribute
 * @vol:	volume on which the attribute is present
 * @a:		attribute to set the value of
 * @b:		buffer containing the new attribute value
 * @l:		length in bytes of the new attribute value @b
 *
 * Set the value of the attribute @a to the contents of the buffer @b which has
 * a byte-size of @l and write it to the NTFS volume @vol. 
 * Return 1 on success and 0 on error, errno will be set in the latter case.
 *
 * FIXME: This is only a temporary implementation which doesn't cope with
 * changes in the length of the resident part of the attribute which in turn
 * means it doesn't cope with changes in length or on disk location of the
 * non-resident part of the attribute either! See ../ntfstools/ntfsfix.c for
 * details.
 * FIXME: Document possible errnos.
 */
int set_attribute_value(ntfs_volume *vol, ATTR_RECORD *a,
			const __u8 *b, __s64 l);

#endif /* defined ATTRIB_H */

