/*
 * $Id: bootsect.c,v 1.5 2001/06/01 02:07:26 antona Exp $
 *
 * bootsect.c - Boot sector handling code. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>

#include "types.h"
#include "bootsect.h"
#include "endians.h"

int is_boot_sector_ntfs(const NTFS_BOOT_SECTOR *b, const int silent)
{
	__u32 i;

	if (!silent)
		printf("\nBeginning bootsector check...\n");
	
	/* Calculate the checksum. Note, this is just a simple addition of
	   all __u32 values in the bootsector starting at the beginning and
	   finishing at the offset of the checksum itself (i.e. not including
	   the checksum...). */
	if ((void*)b < (void*)&b->checksum) {
		__u32 *u = (__u32 *)b;
		__u32 *bi = (__u32 *)(&b->checksum);

		if (!silent)
			printf("Calculating bootsector checksum... ");
		
		for (i = 0; u < bi; ++u)
			i += le32_to_cpup(u);
		
		if (le32_to_cpu(b->checksum) && le32_to_cpu(b->checksum) != i)
			goto not_ntfs;
		if (!silent)
			puts("OK");
	}

	/* Check OEMidentifier is "NTFS    " */
	if (!silent)
		printf("Checking OEMid... ");
	if (b->oem_id != cpu_to_le64(0x202020205346544e)) /* "NTFS    " */
		goto not_ntfs;
	if (!silent)
		puts("OK");

	/* Check bytes per sector value is between 256 and 4096. */
	if (!silent)
		printf("Checking bytes per sector... ");
	if (le16_to_cpu(b->bpb.bytes_per_sector) <  0x100 ||
	    le16_to_cpu(b->bpb.bytes_per_sector) > 0x1000)
		goto not_ntfs;
	if (!silent)
		puts("OK");

	/* Check sectors per cluster value is valid. */
	if (!silent)
		printf("Checking sectors per cluster... ");
	switch (b->bpb.sectors_per_cluster) {
	case 1: case 2: case 4: case 8: case 16:
	case 32: case 64: case 128:
		break;
	default:
		goto not_ntfs;
	}
	if (!silent)
		puts("OK");

	/* Check the cluster size is not above 65536 bytes. */
	if (!silent)
		printf("Checking cluster size... ");
	if ((__u32)le16_to_cpu(b->bpb.bytes_per_sector) *
	    b->bpb.sectors_per_cluster > 0x10000)
		goto not_ntfs;
	if (!silent)
		puts("OK");

	/* Check reserved/unused fields are really zero. */
	if (!silent)
		printf("Checking reserved fields are zero... ");
	if (le16_to_cpu(b->bpb.reserved_sectors) ||
	    le16_to_cpu(b->bpb.root_entries) ||
	    le16_to_cpu(b->bpb.sectors) ||
	    le16_to_cpu(b->bpb.sectors_per_fat) ||
	    le32_to_cpu(b->bpb.large_sectors) ||
	    b->bpb.fats)
		goto not_ntfs;
	if (!silent)
		puts("OK");

	/* Check clusters per file mft record value is valid. */
	if (!silent)
		printf("Checking clusters per mft record... ");
	if ((__u8)b->clusters_per_mft_record < 0xe1 || 
	    (__u8)b->clusters_per_mft_record > 0xf7) {
		switch (b->clusters_per_mft_record) {
		case 1: case 2: case 4: case 8: case 0x10: case 0x20: case 0x40:
			break;
		default:
			goto not_ntfs;
		}
	}
	if (!silent)
		puts("OK");

	/* Check clusters per index block value is valid. */
	if (!silent)
		printf("Checking clusters per index block... ");
	if ((__u8)b->clusters_per_index_block < 0xe1 || 
	    (__u8)b->clusters_per_index_block > 0xf7) {
		switch (b->clusters_per_index_block) {
		case 1: case 2: case 4: case 8: case 0x10: case 0x20: case 0x40:
			break;
		default:
			goto not_ntfs;
		}
	}
	if (!silent)
		puts("OK");

	if (b->end_of_sector_marker != cpu_to_le16(0xaa55))
		fprintf(stderr, "Warning: Bootsector has invalid end of " \
				"sector marker.\n");

	if (!silent)
		puts("Bootsector check completed successfully.");

	return 1;
not_ntfs:
	if (!silent) {
		puts("FAILED");
		puts("Bootsector check failed. Aborting...");
	}
	return 0;
}

