#pragma implementation
#include <string.h>
#include <stdlib.h>
#include "dictionary.h"

/* #Specification: DICTIONARY / principles
	A dictionnary is an array of value. Each entry in the array is keyed
	by a name instead of a numerical index. It is often called an
	associative array.

	Two functions exist to set and retrieve values (setting will add a new
	entry if a corresponding entry does not exist

	You can enter various information in the array in a type safe maner.
	In fact the set and get function are organised by type. For example
	you have setstr() which set a string value. You have getstr() to
	retrieve it. You have setint() and getint() to manipulate integer
	data.

	The "type safe" come from the fact that you can't retrieve a value
	using a different type function than the one used to encode it. So
	you can't do setstr("name","jack") and then getint("name").

	In fact, you can do

	#
		setstr ("name","jack");
		setint ("name",22);
	#

	And then the corresponding function will produce appropriate results

	#
		getstr("name") will produce "jack"
		getint("name") will produce 22
	#
*/
#
PUBLIC DICTIONARY::DICTIONARY()
{
}

PRIVATE void DICTIONARY::set_type (
	const char prefix,
	const char *var,
	const char *val)
{
	char newvar[1 + strlen(var)+1];
	newvar[0] = prefix;
	strcpy (newvar+1,var);
	int no = vars.lookup (newvar);
	if (no == -1){
		vars.add (new SSTRING(newvar));
		vals.add (new SSTRING(val));
	}else{
		vals.getitem(no)->setfrom (val);
	}
}

PRIVATE const char *DICTIONARY::getval_type (
	const char prefix,
	const char *var) const
{
	char newvar[1 + strlen(var)+1];
	newvar[0] = prefix;
	strcpy (newvar+1,var);
	const char *ret = NULL;
	int no = vars.lookup (newvar);
	if (no != -1){
		ret = vals.getitem(no)->get();
	}
	return ret;
}

static const char dict_str='s';
static const char dict_num='n';
static const char dict_obj='o';
static const char dict_bool='b';



/*
	Add/redefine a string value in the dictionary
	get_str() must be used to retrieve it.
*/
PUBLIC void DICTIONARY::set_str (const char *var, const char *val)
{
	set_type (dict_str,var,val);
}

PUBLIC const char *DICTIONARY::get_str (const char *var) const
{
	return getval_type (dict_str,var);
}

/*
	Add/redefine a numerical value in the dictionary
	get_int() must be used to retrieve it.
*/
PUBLIC void DICTIONARY::set_int (const char *var, int val)
{
	char str[30];
	sprintf (str,"%d",val);
	set_type (dict_num,var,str);
}


PUBLIC int DICTIONARY::get_int (const char *var) const
{
	int ret = -1;
	const char *str = getval_type (dict_num,var);
	if (str != NULL) ret = atoi(str);
	return ret;
}

/*
	Add/redefine a boolean value in the dictionary
	get_bool() must be used to retrieve it.
*/
PUBLIC void DICTIONARY::set_bool (const char *var, bool val)
{
	set_type (dict_bool,var,val ? "1" : "0");
}


PUBLIC bool DICTIONARY::get_bool (const char *var) const
{
	bool ret = false;
	const char *str = getval_type (dict_bool,var);
	if (str != NULL) ret = str[0] == '1';
	return ret;
}

/*
	Add/redefine a pointer to an object
	get_obj() must be used to retrieve it.
*/
PUBLIC void DICTIONARY::set_obj (const char *var, void *obj)
{
	char str[30];
	sprintf (str,"%p",obj);
	set_type (dict_obj,var,str);
}


PUBLIC void *DICTIONARY::get_obj (const char *var) const
{
	void *ret;
	const char *str = getval_type (dict_obj,var);
	if (str != NULL){
		sscanf (str,"%x",(int*)&ret);
	}
	return ret;
}

/*
	Return the name of the variable "no" of the dictionary.
	This is used by some functions which wants to iterate all
	variables

	Return NULL if no is out of range.
*/
PUBLIC const char *DICTIONARY::get_var (int no) const
{
	const char *ret = NULL;
	SSTRING *s = vars.getitem(no);
	if (s != NULL){
		 ret = s->get()+1;		// Skip the type
	}
	return ret;
}
/*
	Return the value (string) of the variable "no" of the dictionary.
	This is used by some functions which wants to iterate all
	variables

	Return NULL if no is out of range.
*/
PUBLIC const char *DICTIONARY::get_val (int no) const
{
	const char *ret = NULL;
	SSTRING *s = vals.getitem(no);
	if (s != NULL){
		 ret = s->get();
	}
	return ret;
}
/*
	Return the number of variables in the dictionary
*/
PUBLIC int DICTIONARY::getnb () const
{
	return vars.getnb();
}

