/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: meta.c,v 1.3 2001/11/15 23:03:01 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h> /* atoi */

#if XMMS_TYPE == XMMS_TYPE_LINK
#include <xmmsctrl.h>
#elif XMMS_TYPE == XMMS_TYPE_RUNTIME
#include <dlfcn.h> /* dlopen */
#elif XMMS_TYPE == XMMS_TYPE_ERROR
#error XMMS_TYPE is set incorrectly by configure.in.  This is a bug!  Report it!
#endif

#include "dotconf.h"
#include "meta.h"

/* a metadata_type is static information.
 * metadata_item is an instance of that information. 
 */ 
typedef void (*create_view_f)(metadata_item* mi);
typedef void (*add_to_request_f)(metadata_item* mi, 
		                         GHashTable *request);
typedef void (*load_from_request_f)(metadata_item* mi, GHashTable *hash);
typedef char* (*get_f)(metadata_item* mi);
struct _metadata_item {
	MetaMgr *mm;
	GtkWidget *menuitem;

	create_view_f create_view;
	add_to_request_f add_to_request;
	load_from_request_f load_from_request;
	get_f get;

	/* box is NULL when the item isn't displayed. */
	GtkWidget *box;
	gpointer input;
};

typedef struct _metadata_type metadata_type;
typedef metadata_item* (*create_item_f)(MetaMgr *mm,
		                                metadata_type *mt, 
                                        GtkWidget *menuitem);
struct _metadata_type {
	char *name;
	create_item_f create_item;
};

/* mood */
static metadata_item* 
mood_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem);

/* music */
static metadata_item* 
music_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem);

/* preformatted */
static metadata_item* 
preformat_create_item(MetaMgr *mm, metadata_type *mt, 
		GtkWidget *menuitem);

static metadata_item* 
nocomments_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem);

static metadata_item* 
pickeyword_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem);

static metadata_item* 
backdated_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem);

/* and finally, the metadata types. 
 * must correspond to the METADATA_ enums in meta.h. */
static metadata_type metadata_types[] = {
	{ "Current Mood", mood_create_item },
	{ "Current Music", music_create_item },
	{ "Preformatted HTML", preformat_create_item },
	{ "Disable Comments", nocomments_create_item },
 	{ "User Icon", pickeyword_create_item },
 	{ "Backdated", backdated_create_item },
};

static void add_view(metadata_item *mi);
static void remove_view(metadata_item *mi);
static void metadata_create_view(metadata_item *mi, char *label);
static metadata_item* create_view_by_name(MetaMgr *mm, char *name);
static metadata_item* create_view_by_index(MetaMgr *mm, int idx);
static void metadata_create_view_cb(GtkWidget *w, MetaMgr *mm);

static void metamgr_init(MetaMgr *mm);

/* gtk stuff */
guint
metamgr_get_type() {
	static guint mm_type = 0;
	if (!mm_type) {
		GtkTypeInfo mm_info = {
			"MetaMgr",
			sizeof (MetaMgr),
			sizeof (GtkOptionMenuClass),
			(GtkClassInitFunc) NULL,
			(GtkObjectInitFunc) metamgr_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		mm_type = gtk_type_unique(gtk_option_menu_get_type(), &mm_info);
	}
	return mm_type;
}

static void
metamgr_destroy(MetaMgr *mm) {
	int i;
	for (i = 0; i < METADATACOUNT; i++) {
		g_free(mm->items[i]);
	}
}

static void
metamgr_init(MetaMgr *mm) {
	GtkWidget *menu, *item;

	gtk_signal_connect(GTK_OBJECT(mm), "destroy",
			GTK_SIGNAL_FUNC(metamgr_destroy), NULL);

	menu = gtk_menu_new(); {
		int i;
		metadata_type *mt;
		metadata_item *mi;

		item = gtk_menu_item_new_with_label("Additional data:");
		gtk_menu_append(GTK_MENU(menu), item);

		for (i = 0; i < METADATACOUNT; i++) {
			mt = &metadata_types[i];
			item = gtk_menu_item_new_with_label(mt->name);
			gtk_signal_connect(GTK_OBJECT(item), "activate",
					metadata_create_view_cb, mm);
			gtk_menu_append(GTK_MENU(menu), item);

			mi = mt->create_item(mm, mt, item);
			mm->items[i] = mi;
		}
	}
	gtk_option_menu_set_history(GTK_OPTION_MENU(mm), 0);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(mm), menu);
}

GtkWidget*
metamgr_new(GtkBox *insertbox) {
	MetaMgr *mm;
	mm = METAMGR(gtk_type_new(metamgr_get_type()));
	mm->insertbox = insertbox;

	return GTK_WIDGET(mm);
}

void 
metamgr_append_to_request(MetaMgr *mm, GHashTable* request) {
	metadata_item *mi;
	int i;

	for (i = 0; i < METADATACOUNT; i++) {
		mi = mm->items[i];
		/* add to the request, even if the box isn't visible.
		 * this makes deletion work. */
		mi->add_to_request(mi, request);
	}
}

void
metamgr_load_from_request(MetaMgr *mm, GHashTable *request) {
	metadata_item *mi;
	int i;

	for (i = 0; i < METADATACOUNT; i++) {
		mi = mm->items[i];
		mi->load_from_request(mi, request);
	}
}

void 
metamgr_clear(MetaMgr *mm) {
	metadata_item *mi;
	int i;

	for (i = 0; i < METADATACOUNT; i++) {
		mi = mm->items[i];
		if (mi->box)
			remove_view(mi);
	}
}

int
meta_index(char *name) {
	metadata_type *mt;
	int i;
	for (i = 0; i < METADATACOUNT; i++) {
		mt = &metadata_types[i];
		if (strncasecmp(mt->name, name, strlen(mt->name)) == 0) 
			return i;
	}
	g_warning("unknown metadata type: '%s'\n", name);
	return -1;
}

static void
add_view(metadata_item *mi) {
	gtk_widget_show_all(mi->box);
	gtk_box_pack_start(mi->mm->insertbox, mi->box, FALSE, FALSE, 0);
	gtk_widget_hide(mi->menuitem);
}

static void
remove_view(metadata_item *mi) {
	if (mi->box == NULL) return;
	gtk_widget_destroy(mi->box);   /* kill the entry, and   */
	mi->box = NULL;
	gtk_widget_show(mi->menuitem); /* reshow the menu item. */
}

static void 
toggle_cb(GtkWidget *w, gpointer d) {
	metadata_item *mi = d;
	remove_view(mi);
}

static metadata_item*
create_view_by_index(MetaMgr *mm, int idx) {
	metadata_item *mi;

	mi = mm->items[idx];
	if (!mi->box)
		mi->create_view(mi);
	return mi;
}

static metadata_item*
create_view_by_name(MetaMgr *mm, char *name) {
	int i;

	i = meta_index(name);
	if (i < 0) return NULL;
	return create_view_by_index(mm, i);
}

static void 
metadata_create_view_cb(GtkWidget *w, MetaMgr *mm) {
	char *text;
	metadata_item *mi;

	gtk_option_menu_set_history(GTK_OPTION_MENU(mm), 0);
	gtk_label_get(GTK_LABEL(GTK_BIN(w)->child), &text);

	mi = create_view_by_name(mm, text);
	add_view(mi);
}

/* -------------------- shared functions -------------------- */
static void
metadata_create_view(metadata_item *mi, char *label) {
	GtkWidget *button;

	mi->box = gtk_hbox_new(FALSE, 5); {
		button = gtk_toggle_button_new_with_label(label);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
		gtk_signal_connect(GTK_OBJECT(button), "toggled",
				toggle_cb, mi);
		gtk_box_pack_start(GTK_BOX(mi->box), button, FALSE, FALSE, 0);
	}
}

static char*
metadata_default_get(metadata_item *mi) {
	return gtk_entry_get_text(GTK_ENTRY(mi->input));
}

/* -------------------- mood -------------------- */
static void
mood_create_view(metadata_item *mi) {
	GtkWidget *combo;

	metadata_create_view(mi, "Mood");
	combo = gtk_combo_new(); {
		GList *moodlist = NULL, *l;
		mood *m;

		for (l = conf.moods; l != NULL; l = l->next) {
			m = l->data;
			moodlist = g_list_append(moodlist, m->name);
		}
		gtk_combo_set_popdown_strings(GTK_COMBO(combo), moodlist);
		g_list_free(moodlist);
	}
	mi->input = GTK_COMBO(combo)->entry;
	gtk_entry_set_text(GTK_ENTRY(mi->input), "");
	gtk_box_pack_start(GTK_BOX(mi->box), combo, TRUE, TRUE, 0);
}

static void 
mood_add_to_request(metadata_item *mi, GHashTable *request) {
	char *text;
	GList *l;
	mood *m;

	if (!mi->box) {
		g_hash_table_insert(request, g_strdup("prop_current_mood"), 
				g_strdup(""));
		g_hash_table_insert(request, g_strdup("prop_current_moodid"), 
				g_strdup(""));
		return;
	}
	text = mi->get(mi);
	/* if we're using a predefined mood, we just want to send the id. */
	for (l = conf.moods; l != NULL; l = l->next) {
		m = l->data;
		if (strcasecmp(m->name, text) == 0)
			break;
	}

	if (l == NULL) { /* the mood wasn't found. */
		/* send it as text */
		g_hash_table_insert(request, g_strdup("prop_current_mood"),
				g_strdup(text));
	} else {
		g_hash_table_insert(request, g_strdup("prop_current_moodid"),
				g_strdup_printf("%d", m->id));
	}
}

static void 
mood_load_from_request(metadata_item *mi, GHashTable *request) {
	char *moodtext;

	moodtext = g_hash_table_lookup(request, "prop_current_mood");
	if (moodtext == NULL) {
		int id = -1;
		char *moodid;
		GList *l;
		mood *m;

		moodid = g_hash_table_lookup(request, "prop_current_moodid");
		if (moodid == NULL) return;

		id = atoi(moodid);
		for (l = conf.moods; l != NULL; l = l->next) {
			m = l->data;
			if (m->id == id)
				break;
		}
		if (l == NULL) return; /* mood id not found. */

		moodtext = m->name;
	}

	if (!mi->box) {
		mi->create_view(mi);
		add_view(mi);
	}
	gtk_entry_set_text(GTK_ENTRY(mi->input), moodtext);
}

static metadata_item*
mood_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem) {
	metadata_item* mi;

	mi = g_new0(metadata_item, 1);
	mi->menuitem = menuitem;
	mi->mm = mm;
	mi->create_view = mood_create_view;
	mi->add_to_request = mood_add_to_request;
	mi->load_from_request = mood_load_from_request;
	mi->get = metadata_default_get;
	return mi;
}


/* -------------------- music -------------------- */

static void 
music_add_to_request(metadata_item *mi, GHashTable *request) {
	char *text = "";

	if (mi->box)
		text = mi->get(mi);
		
	g_hash_table_insert(request, g_strdup("prop_current_music"),
			g_strdup(text));
}

static void 
music_load_from_request(metadata_item *mi, GHashTable *request) {
	char *music;

	music = g_hash_table_lookup(request, "prop_current_music");
	if (music) {
		if (!mi->box) {
			mi->create_view(mi);
			add_view(mi);
		}
		gtk_entry_set_text(GTK_ENTRY(mi->input), music);
	}
}

#if XMMS_TYPE == XMMS_TYPE_RUNTIME
static void *xmms_dl = NULL; /* for dlopen(). */

static void
music_xmms_grab(GtkWidget *w, gpointer d) {
	gint pos;
	gchar *text;
	/*
	 * we do some wackyness here to avoid statically linking with XMMS.
gint xmms_remote_get_playlist_pos(gint session);
gchar *xmms_remote_get_playlist_title(gint session, gint pos);
	*/
	static gint (*xrgpp)(gint) = NULL;
	static gchar* (*xrgpt)(gint, gint) = NULL;

	if (xrgpp == NULL || xrgpt == NULL) {
		xrgpp = dlsym(xmms_dl, "xmms_remote_get_playlist_pos");
		if (dlerror()) return;
		xrgpt = dlsym(xmms_dl, "xmms_remote_get_playlist_title");
		if (dlerror()) return;
	}
	
	pos = xrgpp(0);
	text = xrgpt(0, pos);
	if (text != NULL) {
		gtk_entry_set_text(GTK_ENTRY(d), text);
		g_free(text);
	} else {
		gtk_entry_set_text(GTK_ENTRY(d), "[unable to connect to XMMS]");
	}
}

static gboolean xmms_load_ok() {
	if (xmms_dl == NULL) {
		xmms_dl = dlopen("libxmms.so", RTLD_LAZY);
		if (!xmms_dl) 
			xmms_dl = dlopen("libxmms.so.1", RTLD_LAZY);
			if (!xmms_dl)
				return FALSE; /* no library; don't use xmms. */
	}
	return TRUE;
}
/* XMMS_TYPE_RUNTIME */
#elif XMMS_TYPE == XMMS_TYPE_LINK
static void
music_xmms_grab(GtkWidget *w, gpointer d) {
	gint pos;
	gchar *text;

	pos = xmms_remote_get_playlist_pos(0);
	text = xmms_remote_get_playlist_title(0, pos);
	if (text != NULL) {
		gtk_entry_set_text(GTK_ENTRY(d), text);
		g_free(text);
	} else {
		gtk_entry_set_text(GTK_ENTRY(d), "[unable to connect to XMMS]");
	}
}
#endif /* XMMS_TYPE */ 

static void
music_create_view(metadata_item *mi) {
	metadata_create_view(mi, "Music");

	mi->input = gtk_entry_new();
	gtk_widget_set_usize(mi->input, 100, -1);
	gtk_box_pack_start(GTK_BOX(mi->box), mi->input, TRUE, TRUE, 0);

#if XMMS_TYPE != XMMS_TYPE_NONE

#if XMMS_TYPE == XMMS_TYPE_RUNTIME
	if (xmms_load_ok()) 
#endif
	{
		GtkWidget *button;
		button = gtk_button_new_with_label("<- XMMS");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				music_xmms_grab, mi->input);
		gtk_box_pack_start(GTK_BOX(mi->box), button, FALSE, FALSE, 0);
	}
#endif /* XMMS_TYPE != XMMS_TYPE_NONE */
}

static metadata_item* 
music_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem) {
	metadata_item *mi;

	mi = g_new0(metadata_item, 1);
	mi->menuitem = menuitem;
	mi->mm = mm;
	mi->create_view = music_create_view;
	mi->add_to_request = music_add_to_request;
	mi->load_from_request = music_load_from_request;
	mi->get = metadata_default_get;

	return mi;
}

/* -------------------- preformat -------------------- */
static char*
preformat_get(metadata_item *mi) {
	int active;
	static char* values[] = { "0", "1" };
	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mi->input));
	if (active == 0) return values[0];
	else return values[1];
}

static void 
preformat_add_to_request(metadata_item *mi, GHashTable *request) {
	char *text = "0";

	if (mi->box)
		text = mi->get(mi);
		
	g_hash_table_insert(request, g_strdup("prop_opt_preformatted"),
			g_strdup(text));
}

static void 
preformat_load_from_request(metadata_item *mi, GHashTable *request) {
	char *pre;
	int val;
	
	pre = g_hash_table_lookup(request, "prop_opt_preformatted");
	if (pre) {
		val = atoi(pre);
		if (!mi->box) {
			mi->create_view(mi);
			add_view(mi);
		}
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mi->input), val);
	}
}

static void
preformat_create_view(metadata_item *mi) {
	metadata_create_view(mi, "Preformat");
	mi->input = gtk_check_button_new_with_label("Don't auto-insert <br>");
	gtk_box_pack_start(GTK_BOX(mi->box), mi->input, TRUE, TRUE, 0);
}

static metadata_item* 
preformat_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem) {
	metadata_item* mi;

	mi = g_new0(metadata_item, 1);
	mi->menuitem = menuitem;
	mi->mm = mm;
	mi->create_view = preformat_create_view;
	mi->add_to_request = preformat_add_to_request;
	mi->load_from_request = preformat_load_from_request;
	mi->get = preformat_get;

	return mi;
}


/* -------------------- nocomments -------------------- */
static char*
nocomments_get(metadata_item *mi) {
	int active;
	static char* values[] = { "0", "1" };
	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mi->input));
	if (active == 0) return values[0];
	else return values[1];
}

static void 
nocomments_add_to_request(metadata_item *mi, GHashTable *request) {
	char *text = "0";

	if (mi->box)
		text = mi->get(mi);
		
	g_hash_table_insert(request, g_strdup("prop_opt_nocomments"),
			g_strdup(text));
}

static void 
nocomments_load_from_request(metadata_item *mi, GHashTable *request) {
	char *pre;
	int val;
	
	pre = g_hash_table_lookup(request, "prop_opt_nocomments");
	if (pre) {
		val = atoi(pre);
		if (!mi->box) {
			mi->create_view(mi);
			add_view(mi);
		}
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mi->input), val);
	}
}

static void
nocomments_create_view(metadata_item *mi) {
	metadata_create_view(mi, "Comments");
	mi->input = gtk_check_button_new_with_label("Disallow comments to this journal entry");
	gtk_box_pack_start(GTK_BOX(mi->box), mi->input, TRUE, TRUE, 0);
}

static metadata_item* 
nocomments_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem) {
	metadata_item* mi;

	mi = g_new0(metadata_item, 1);
	mi->menuitem = menuitem;
	mi->mm = mm;
	mi->create_view = nocomments_create_view;
	mi->add_to_request = nocomments_add_to_request;
	mi->load_from_request = nocomments_load_from_request;
	mi->get = nocomments_get;

	return mi;
}

/* -------------------- picture keyword -------------------- */
static void 
pickeyword_activate(GtkWidget *w, metadata_item *mi) {
	gpointer data;
	data = gtk_object_get_user_data(GTK_OBJECT(w));
	gtk_object_set_user_data(GTK_OBJECT(mi->input), data);
}

static void
pickeyword_create_view(metadata_item *mi) {
	GtkWidget *menu, *item;
	GtkWidget *om;

	metadata_create_view(mi, "User Picture");

	mi->input = om = gtk_option_menu_new();
	menu = gtk_menu_new(); {
		GList *l;

		item = gtk_menu_item_new_with_label("(default)");
		gtk_signal_connect(GTK_OBJECT(item), "activate",
				pickeyword_activate, mi);
		gtk_object_set_user_data(GTK_OBJECT(item), "");
		gtk_menu_append(GTK_MENU(menu), item);

		for (l = conf.pickws; l != NULL; l = l->next) {
			item = gtk_menu_item_new_with_label(l->data);
			gtk_signal_connect(GTK_OBJECT(item), "activate",
					pickeyword_activate, mi);
			gtk_object_set_user_data(GTK_OBJECT(item), l->data);
			gtk_menu_append(GTK_MENU(menu), item);
		}
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(om), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(om), 0);
	gtk_object_set_user_data(GTK_OBJECT(om), "");

	gtk_box_pack_start(GTK_BOX(mi->box), om, TRUE, TRUE, 0);
}

static char*
pickeyword_get(metadata_item *mi) {
	return gtk_object_get_user_data(GTK_OBJECT(mi->input));
}

static void 
pickeyword_add_to_request(metadata_item *mi, GHashTable *request) {
	char *text;

	if (!mi->box) {
		g_hash_table_insert(request, g_strdup("prop_picture_keyword"), 
				g_strdup(""));
		return;
	}
	text = mi->get(mi);

	g_hash_table_insert(request, g_strdup("prop_picture_keyword"),
			    g_strdup(text));
}

static void 
pickeyword_load_from_request(metadata_item *mi, GHashTable *request) {
	char *kwtext;
	GList *l;
	int i;

	kwtext = g_hash_table_lookup(request, "prop_picture_keyword");
	if (kwtext == NULL) return;

	if (!mi->box) {
		mi->create_view(mi);
		add_view(mi);
	}
	
	for (i = 1, l = conf.pickws; l != NULL; i++, l = l->next) {
		if (strcasecmp(l->data, kwtext) == 0) {
			gtk_option_menu_set_history(GTK_OPTION_MENU(mi->input), i);
			gtk_object_set_user_data(GTK_OBJECT(mi->input), l->data);
		}
	}
}

static metadata_item*
pickeyword_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem) {
	metadata_item* mi;

	mi = g_new0(metadata_item, 1);
	mi->menuitem = menuitem;
	mi->mm = mm;
	mi->create_view = pickeyword_create_view;
	mi->add_to_request = pickeyword_add_to_request;
	mi->load_from_request = pickeyword_load_from_request;
	mi->get = pickeyword_get;
	return mi;
}

/* -------------------- backdated -------------------- */
static char*
backdated_get(metadata_item *mi) {
	int active;
	static char* values[] = { "0", "1" };
	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mi->input));
	if (active == 0) return values[0];
	else return values[1];
}

static void 
backdated_add_to_request(metadata_item *mi, GHashTable *request) {
	char *text = "0";

	if (mi->box)
		text = mi->get(mi);
		
	g_hash_table_insert(request, g_strdup("prop_opt_backdated"),
			g_strdup(text));
}

static void 
backdated_load_from_request(metadata_item *mi, GHashTable *request) {
	char *pre;
	int val;
	
	pre = g_hash_table_lookup(request, "prop_opt_backdated");
	if (pre) {
		val = atoi(pre);
		if (!mi->box) {
			mi->create_view(mi);
			add_view(mi);
		}
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mi->input), val);
	}
}

static void
backdated_create_view(metadata_item *mi) {
	metadata_create_view(mi, "Backdated");
	mi->input = gtk_check_button_new_with_label("Backdated entry (don't show on friends lists)");
	gtk_box_pack_start(GTK_BOX(mi->box), mi->input, TRUE, TRUE, 0);
}

static metadata_item* 
backdated_create_item(MetaMgr *mm, metadata_type *mt, GtkWidget *menuitem) {
	metadata_item* mi;

	mi = g_new0(metadata_item, 1);
	mi->menuitem = menuitem;
	mi->mm = mm;
	mi->create_view = backdated_create_view;
	mi->add_to_request = backdated_add_to_request;
	mi->load_from_request = backdated_load_from_request;
	mi->get = backdated_get;

	return mi;
}

