// -*- mode: c++; tab-width: 4; indent-tabs-mode: t -*-
/**
 * @file cache/debtags/serializer.h
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

#ifndef EPT_CACHE_DEBTAGS_SERIALIZER_TCC
#define EPT_CACHE_DEBTAGS_SERIALIZER_TCC

#include <ept/cache/debtags/serializer.h>
#if 0
#include <ept/cache/debtags/pkgidx.h>
#include <ept/cache/debtags/vocabulary.h>
#include <ept/cache/package.h>
//#include <ept/cache/cache.h>
#endif

namespace ept {
namespace t {
namespace cache {
namespace debtags {

template<typename C, typename OUT> template<typename ITEMS, typename TAGS>
StringToEpt<C, OUT>& StringToEpt<C, OUT>::operator=(const std::pair<ITEMS, TAGS>& data)
{
	std::set< typename C::Package > ritems;
	std::set< typename C::Tag > rtags;

	for (typename ITEMS::const_iterator i = data.first.begin();
			i != data.first.end(); ++i)
	{
		typename C::Package p = packages.index().packageByName(*i);
		if (p.valid())
			ritems.insert(p);
	}

	for (typename TAGS::const_iterator i = data.second.begin();
			i != data.second.end(); ++i)
	{
		typename C::Tag t = tags.tagByName(*i);
		if (t.valid())
			rtags.insert(t);
	}

	if (!ritems.empty() && !rtags.empty())
	{
		*out = make_pair(ritems, rtags);
		++out;
	}
	return *this;
}

template<typename C, typename OUT> template<typename ITEMS, typename TAGS>
IntToEpt<C, OUT>& IntToEpt<C, OUT>::operator=(const std::pair<ITEMS, TAGS>& data)
{
	std::set< typename C::Package > ritems;
	std::set< typename C::Tag > rtags;

	for (typename ITEMS::const_iterator i = data.first.begin();
			i != data.first.end(); ++i)
	{
		const char* name = pkgidx.name(*i);
		typename C::Package p = packages.index().packageByName(name);
		if (p.valid())
			ritems.insert(p);
	}

	for (typename TAGS::const_iterator i = data.second.begin();
			i != data.second.end(); ++i)
	{
		typename C::Tag t = tags.tagByID(*i);
		if (t.valid())
			rtags.insert(t);
	}

	if (!ritems.empty() && !rtags.empty())
	{
		*out = make_pair(ritems, rtags);
		++out;
	}
	return *this;
}


#if 0
string FacetIntConverter::componentName() { return "FacetIntConverter"; }

int FacetIntConverter::operator()(const aptFront::cache::entity::Facet& item) const
{
	if (!item.valid()) return -1;
	return item.id();
}
aptFront::cache::entity::Facet FacetIntConverter::operator()(const int& item) const
{
	return cache().tags().facetByID(item);
}

string FacetStringConverter::componentName() { return "FacetStringConverter"; }

std::string FacetStringConverter::operator()(const aptFront::cache::entity::Facet& item) const
{
	if (!item.valid()) return string();
	return item.name();
}
aptFront::cache::entity::Facet FacetStringConverter::operator()(const std::string& item) const
{
	return cache().tags().facetByName(item);
}

string TagIntConverter::componentName() { return "TagIntConverter"; }

int TagIntConverter::operator()(const aptFront::cache::entity::Tag& item) const
{
	if (!item.valid()) return -1;
	return item.id();
}
aptFront::cache::entity::Tag TagIntConverter::operator()(const int& item) const
{
	return cache().tags().tagByID(item);
}

string TagStringConverter::componentName() { return "TagStringConverter"; }

std::string TagStringConverter::operator()(const aptFront::cache::entity::Tag& item) const
{
	if (!item.valid()) return string();
	return item.fullname();
}
aptFront::cache::entity::Tag TagStringConverter::operator()(const std::string& item) const
{
	return cache().tags().tagByName(item);
}

Tagcoll::OpSet<entity::Tag> TagStringConverter::parseTagList(const std::string& str) const
{
	if (str.empty())
		return Tagcoll::OpSet<entity::Tag>();

	size_t i = str.find(", ");
	if (i == string::npos)
	{
		// Check if we need curly brace expansion
		if (str[str.size() - 1] == '}')
		{
			using namespace std;
			Tagcoll::OpSet<entity::Tag> res;
			size_t begin = str.find('{');
			if (begin == string::npos)
				return res;
			string prefix(str, 0, begin);
			++begin;
			size_t end;
			while ((end = str.find(',', begin)) != string::npos)
			{
				res += (*this)(prefix + str.substr(begin, end-begin));
				begin = end + 1;
			}
			res += (*this)(prefix + str.substr(begin, str.size() - 1 - begin));
			return res;
		} else {
			entity::Tag t = (*this)(str);
			if (t.valid())
				return Tagcoll::OpSet<entity::Tag>() + t;
			else
				return Tagcoll::OpSet<entity::Tag>();
		}
	} else {
		return parseTagList(string(str, 0, i)) + parseTagList(string(str, i+2));
	}
}

string PackageIntConverter::componentName() { return "PackageIntConverter"; }

int PackageIntConverter::operator()(const aptFront::cache::entity::Package& item) const
{
	if (!item.valid()) return -1;
	return item.id();
}
aptFront::cache::entity::Package PackageIntConverter::operator()(const int& item) const
{
	PkgIdx& p = cache().pkgidx();
	return cache().packages().packageByName(string(p.name(item), p.size(item)));
}

string PackageStringConverter::componentName() { return "PackageStringConverter"; }

std::string PackageStringConverter::operator()(const aptFront::cache::entity::Package& item) const
{
	if (!item.valid()) return string();
	return item.name();
}
aptFront::cache::entity::Package PackageStringConverter::operator()(const std::string& item) const
{
	return cache().packages().packageByName(item);
}
#endif

}
}
}
}

#endif

#if 0
#ifdef COMPILE_TESTSUITE
//#include <apt-front/cache/component/debtags/update.h>
#include <iostream>
#include "test-utils.h"

namespace tut {
using namespace aptFront::cache;
using namespace component;
using namespace debtags;
using namespace std;

struct cache_component_debtags_serializer_shar {
    cache_component_debtags_serializer_shar () {
        aptInit ();
        ok = true;
        debtags::fetchNewData();
        c.open( Cache::OpenDefault |
                Cache::OpenReadOnly | Cache::OpenDebtags );
    }
    void check() {
        if (ok) return;
        ok = true;
        throw warning( "debtags init failed, cancelling" );
    }
    ~cache_component_debtags_serializer_shar() {
        check();
    }
    Cache c;
    bool ok;
};

TESTGRP( cache_component_debtags_serializer );

using namespace Tagcoll;

template<> template<>
void to::test<1> ()
{
    check();

	PackageStringConverter& psc = c.packagestringconverter();

	ensure(psc("Slartibartsfart") == entity::Package());

    /* Get the 'debtags' package */
    entity::Package p = c.packages().packageByName( "debtags" );
    ensure(p.valid());

	/* Get the 'debtags' package using the serializer */
	entity::Package p1 = psc("debtags");
	ensure(p1.valid());

	/* They must be the same */
	ensure(p == p1);

	ensure_equals(psc(p), "debtags");
	ensure_equals(psc(p1), "debtags");
	ensure_equals(psc(p), psc(p1));

	/* If there is an invalid package to serialize, it should be discarded */
	{
		Tagcoll::OpSet<entity::Package> pkgs;
		pkgs += c.packages().packageByName( "debtags" );
		pkgs += c.packages().packageByName( "tagcoll" );
		pkgs += entity::Package();

		ensure_equals (pkgs.size(), 3u);
		ensure_equals (psc(pkgs).size(), 2u);
		ensure (psc(pkgs).contains("debtags"));
		ensure (psc(pkgs).contains("tagcoll"));
	}

	/* If there is an invalid package to serialize, it should be discarded */
	{
		Tagcoll::OpSet<std::string> pkgs;
		pkgs += "debtags";
		pkgs += "tagcoll";
		pkgs += "Slartibartsfart";

		ensure_equals (pkgs.size(), 3u);
		ensure_equals (psc(pkgs).size(), 2u);
		ensure (psc(pkgs).contains(psc("debtags")));
		ensure (psc(pkgs).contains(psc("tagcoll")));
		ensure (!psc(pkgs).contains(entity::Package()));
	}
}

ostream& operator<<(ostream& out, const entity::Package& pkg)
{
	if (pkg.valid())
		return out << pkg.name();
	else
		return out << "(invalid package)";
}

// Check that package conversions work two-way
template<> template<>
void to::test<2> ()
{
	PackageStringConverter& psc = c.packagestringconverter();
	for (component::Aggregator::iterator i = c.packages().packagesBegin();
			i != c.packages().packagesEnd(); ++i)
	{
		try {
			ensure_equals(*i, psc(psc(*i)));
		} catch (...) {
			cerr << "Note: exception thrown during processing[string] of package " << i->name(string("(invalid package)")) << endl;
			throw;
		}
	}

	PackageIntConverter& pic = c.packageintconverter();
	for (component::Aggregator::iterator i = c.packages().packagesBegin();
			i != c.packages().packagesEnd(); ++i)
	{
		try {
			ensure_equals(*i, pic(pic(*i)));
		} catch (...) {
			cerr << "Note: exception thrown during processing[int] of package " << i->name(string("(invalid package)")) << endl;
			throw;
		}
	}
}

// Check that facet conversions work two-way
template<> template<>
void to::test<3> ()
{
	typedef Tagcoll::OpSet<entity::Facet> FacetSet;

	FacetStringConverter& fsc = c.facetstringconverter();
	FacetSet allFacets(c.tags().facets());
	for (FacetSet::const_iterator i = allFacets.begin(); i != allFacets.end(); i++)
	{
		try {
			ensure_equals(*i, fsc(fsc(*i)));
		} catch (...) {
			cerr << "Note: exception thrown during processing[string] of facet " << i->name() << endl;
			throw;
		}
	}

	FacetIntConverter& fic = c.facetintconverter();
	for (FacetSet::const_iterator i = allFacets.begin(); i != allFacets.end(); i++)
	{
		try {
			ensure_equals(*i, fic(fic(*i)));
		} catch (...) {
			cerr << "Note: exception thrown during processing[int] of facet " << i->name() << endl;
			throw;
		}
	}
}

// Check that tag conversions work two-way
template<> template<>
void to::test<4> ()
{
	typedef Tagcoll::OpSet<entity::Tag> TagSet;

	TagStringConverter& tsc = c.tagstringconverter();
	TagSet allTags(c.tags().tags());
	for (TagSet::const_iterator i = allTags.begin(); i != allTags.end(); i++)
	{
		try {
			ensure_equals(*i, tsc(tsc(*i)));
		} catch (...) {
			cerr << "Note: exception thrown during processing[string] of tag " << i->fullname() << endl;
			throw;
		}
	}

	TagIntConverter& tic = c.tagintconverter();
	for (TagSet::const_iterator i = allTags.begin(); i != allTags.end(); i++)
	{
		try {
			ensure_equals(*i, tic(tic(*i)));
		} catch (...) {
			cerr << "Note: exception thrown during processing[int] of tag " << i->fullname() << endl;
			throw;
		}
	}
}

// Check TagStringConverter::parseTagList
template<> template<>
void to::test<5> ()
{
	TagStringConverter& tsc = c.tagstringconverter();
	OpSet<entity::Tag> ts;

	// First ensure that we're using existing tags as samples
	ensure(tsc("accessibility::TODO") != entity::Tag());
	ensure(tsc("role::sw:devel-lib") != entity::Tag());
	ensure(tsc("x11::xserver") != entity::Tag());
	ensure(tsc("antani") == entity::Tag());
	ensure(tsc("blinda") == entity::Tag());
	ensure(tsc("supercazzola") == entity::Tag());

	ts = tsc.parseTagList("role::sw:devel-lib");
	ensure_equals(ts.size(), 1u);
	ensure(ts.contains(tsc("role::sw:devel-lib")));

	ts = tsc.parseTagList("accessibility::TODO, x11::xserver, role::sw:devel-lib");
	ensure_equals(ts.size(), 3u);
	ensure(ts.contains(tsc("accessibility::TODO")));
	ensure(ts.contains(tsc("role::sw:devel-lib")));
	ensure(ts.contains(tsc("x11::xserver")));

	ts = tsc.parseTagList("antani");
	ensure_equals(ts.size(), 0u);

	ts = tsc.parseTagList("antani, blinda, supercazzola");
	ensure_equals(ts.size(), 0u);

	ts = tsc.parseTagList("antani, x11::xserver, blinda");
	ensure_equals(ts.size(), 1u);
	ensure(ts.contains(tsc("x11::xserver")));
}

// Check TagStringConverter::parseTagList's handling of curly brace expansion
template<> template<>
void to::test<6> ()
{
	TagStringConverter& tsc = c.tagstringconverter();
	OpSet<entity::Tag> ts;

	// First ensure that we're using existing tags as samples
	ensure(tsc("role::TODO") != entity::Tag());
	ensure(tsc("role::sw:server") != entity::Tag());
	ensure(tsc("role::aux:dummy") != entity::Tag());
	ensure(tsc("role::sw:amusement") != entity::Tag());
	ensure(tsc("role::sw:server{}") == entity::Tag());
	ensure(tsc("role::{}") == entity::Tag());
	ensure(tsc("role::{") == entity::Tag());
	ensure(tsc("role::}") == entity::Tag());

	ts = tsc.parseTagList("role::{TODO,sw:server,aux:dummy,sw:amusement}");
	ensure_equals(ts.size(), 4u);
	ensure(ts.contains(tsc("role::TODO")));
	ensure(ts.contains(tsc("role::sw:server")));
	ensure(ts.contains(tsc("role::aux:dummy")));
	ensure(ts.contains(tsc("role::sw:amusement")));

	ts = tsc.parseTagList("role::{TODO,aux:dummy}, role::sw:{server,amusement}");
	ensure_equals(ts.size(), 4u);
	ensure(ts.contains(tsc("role::TODO")));
	ensure(ts.contains(tsc("role::sw:server")));
	ensure(ts.contains(tsc("role::aux:dummy")));
	ensure(ts.contains(tsc("role::sw:amusement")));
}

}
#endif
#endif
// vim:set ts=4 sw=4:
