/**
 * @file libgnurdf/property.h Property functions
 *
 * $Id: property.h,v 1.7 2001/09/20 22:47:28 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBGNURDF_PROPERTY_H_
#define _LIBGNURDF_PROPERTY_H_

#include <libgnurdf/types.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Returns the element or its value with the specified name in a
 * description or container.
 *
 * To retrieve only the property's value, pass a pointer to a string for
 * \a value and NULL for \a element.
 *
 * Likewise, you can retrieve only the element by passing a pointer to a
 * pointer of an RdfElement for \a element and NULL for \a value.
 *
 * If you wish to retrieve both the value and the element, pass a pointer
 * to a string for \a value and a pointer to a pointer of an RdfElement to
 * \a element.
 * 
 * @param parent   The description or container containing the property.
 * @param property The name of the property to retrieve.
 * @param ns       The namespace of the property.
 * @param value    The returned value, unless NULL is specified.
 * @param element  The returned element, unless NULL is specified.
 *
 * @return RDF_BAG if the property is an <RDF:Bag>.
 * @return RDF_SEQ if the property is an <RDF:Seq>.
 * @return RDF_ALT if the property is an <RDF:Alt>.
 * @return RDF_DESC if the property is an <RDF:Description>.
 * @return 0 if the property is a normal property.
 * @return -1 if not found.
 */
int rdfGetProperty(RdfElement *parent, const char *property,
				   RdfNamespace *ns, char **value, RdfElement **element);

/**
 * Sets the value of a property in a description or container.
 *
 * @param parent   The description or container which will hold the property.
 * @param property The name of the property.
 * @param ns       The namespace of the property.
 * @param value    The property's value.
 *
 * @return The new property's element.
 * 
 * @see rdfSetElement()
 */
RdfElement *rdfSetProperty(RdfElement *parent, const char *property,
						   RdfNamespace *ns, const char *value);

/**
 * Sets the element of a property in a description or container.
 *
 * @param parent   The description or container which will hold the property.
 * @param property The name of the property.
 * @param ns       The namespace of the property.
 * @param element  The property's element.
 *
 * @return The new property's element.
 *
 * @see rdfSetProperty()
 */
RdfElement *rdfSetElement(RdfElement *parent, const char *property,
						  RdfNamespace *ns, RdfElement *element);

/**
 * Removes a property from a description or container.
 *
 * @param desc     The description or container which holds the property.
 * @param property The property's name.
 * @param ns       The property's namespace.
 */
void rdfRemoveProperty(RdfElement *parent, const char *property,
					   RdfNamespace *ns);
/**
 * Removes an element from a description or container.
 *
 * @param element The element to remove.
 */
void rdfRemoveElement(RdfElement *element);

/*************************************************************************/
/** @name Backwards-compatibility functions                              */
/*************************************************************************/
/*@{*/

/**
 * Returns the element or its value with the specified name in a Description.
 *
 * @deprecated This function has been replaced with rdfGetProperty().
 *
 * @see rdfGetProperty()
 */
int rdfGetValue(RdfDescription *desc, const char *property,
				RdfNamespace *ns, char **value, RdfElement **element);

/**
 * Sets the value of a property in a description.
 *
 * @deprecated This function has been replaced with rdfSetProperty().
 *
 * @see rdfSetProperty()
 */
RdfElement *rdfSetValue(RdfDescription *desc, const char *property,
						RdfNamespace *ns, const char *value);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _LIBGNURDF_PROPERTY_H_ */

