/*
 * Copyright (c) 2007, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.logging;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.Format;
import java.util.Date;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;

import de.intarsys.tools.format.TrivialDateFormat;

/**
 * A simple usable formatter for java logging
 */
public class SimpleFormatter extends Formatter {
	/** The line separator. */
	private static final String LF = System.getProperty("line.separator");

	/** A thread safe date format. */
	private Format dateFormat = new TrivialDateFormat();

	/** The StringBuffer to format the message (this is not thread safe). */
	private final StringBuffer sb = new StringBuffer();

	private boolean showDate = true;

	private boolean showLevel = true;

	private boolean showName = true;

	private boolean showThread = true;

	/**
	 * 
	 */
	public SimpleFormatter() {
		super();
	}

	@Override
	public synchronized String format(LogRecord record) {
		sb.setLength(0);
		if (isShowDate()) {
			sb.append("[");
			int index = sb.length();
			dateFormat.format(new Date(record.getMillis()), sb, null);
			for (int i = sb.length() - index; i < 24; i++) {
				sb.append(' ');
			}
			sb.append("]");
		}
		if (isShowLevel()) {
			sb.append("[");
			String levelString = record.getLevel().toString();
			sb.append(levelString);
			for (int i = levelString.length(); i < 10; i++) {
				sb.append(' ');
			}
			sb.append("]");
		}
		if (isShowName()) {
			String loggerName = record.getLoggerName();
			if (loggerName == null) {
				loggerName = "<unknown>";
			}
			sb.append("[");
			sb.append(loggerName);
			sb.append("]");
		}
		if (isShowThread()) {
			sb.append("[");
			sb.append(Thread.currentThread().getName());
			sb.append("]");
		}
		sb.append(" ");
		sb.append(record.getMessage());
		sb.append(LF);
		if (record.getThrown() != null) {
			try {
				StringWriter sw = new StringWriter();
				PrintWriter pw = new PrintWriter(sw);
				record.getThrown().printStackTrace(pw);
				pw.close();
				sb.append(sw.toString());
				sb.append(LF);
			} catch (Exception ex) {
				// ignore
			}
		}
		return sb.toString();
	}

	public boolean isShowDate() {
		return showDate;
	}

	public boolean isShowLevel() {
		return showLevel;
	}

	public boolean isShowName() {
		return showName;
	}

	public boolean isShowThread() {
		return showThread;
	}

	public void setShowDate(boolean showDate) {
		this.showDate = showDate;
	}

	public void setShowLevel(boolean showLevel) {
		this.showLevel = showLevel;
	}

	public void setShowName(boolean showName) {
		this.showName = showName;
	}

	public void setShowThread(boolean showThread) {
		this.showThread = showThread;
	}
}
