/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef _SML_SESSION_H_
#define _SML_SESSION_H_

typedef enum SmlSessionType {
	SML_SESSION_TYPE_SERVER,
	SML_SESSION_TYPE_CLIENT
} SmlSessionType;

typedef enum SmlSessionEventType {
	SML_SESSION_EVENT_ERROR,
	SML_SESSION_EVENT_COMMAND,
	SML_SESSION_EVENT_CHILD_COMMAND,
	SML_SESSION_EVENT_HEADER_REPLY,
	SML_SESSION_EVENT_FINAL,
	SML_SESSION_EVENT_END,
	SML_SESSION_EVENT_FLUSH
} SmlSessionEventType;

typedef void (* SmlSessionEventCallback) (SmlSession *session, SmlSessionEventType type, SmlCommand *command, SmlCommand *parent, SmlStatus *reply, SmlError *error, void *userdata);
typedef void (* SmlSessionDataCallback) (SmlSession *session, SmlTransportData *data, void *userdata);

/* Session Management */
SmlSession *smlSessionNew(SmlSessionType sessionType, SmlMimeType mimetype, SmlProtocolVersion version, SmlProtocolType protocol, SmlLocation *target, SmlLocation *source, const char *sessionID, unsigned int messageID, SmlError **error);
void smlSessionRef(SmlSession *session);
void smlSessionUnref(SmlSession *session);
void smlSessionSetEventCallback(SmlSession *session, SmlSessionEventCallback callback, void *userdata);
void smlSessionSetDataCallback(SmlSession *session, SmlSessionDataCallback callback, void *userdata);

/* Session Control */
SmlBool smlSessionFlush(SmlSession *session, SmlBool final, SmlError **error);
SmlBool smlSessionEnd(SmlSession *session, SmlError **error);
void smlSessionUseStringTable(SmlSession *session, SmlBool useStringtable);
void smlSessionUseOnlyReplace(SmlSession *session, SmlBool onlyReplace);
void smlSessionUseNumberOfChanges(SmlSession *session, SmlBool support);
void smlSessionUseLargeObjects(SmlSession *session, SmlBool support);
void smlSessionSetSendingLimit(SmlSession *session, unsigned int limit);
void smlSessionSetReceivingLimit(SmlSession *session, unsigned int limit);
void smlSessionSetSendingMaxObjSize(SmlSession *session, int limit);
void smlSessionSetRequestedMaxObjSize(SmlSession *session, int limit);
int smlSessionGetSendingMaxObjSize(SmlSession *session);
void smlSessionSetReceivingMaxObjSize(SmlSession *session, unsigned int limit);
unsigned int smlSessionGetReceivingMaxObjSize(SmlSession *session);
const char *smlSessionGetSessionID(SmlSession *session);
void smlSessionSetSessionID(SmlSession *session, const char *sessionID);
SmlProtocolVersion smlSessionGetVersion(SmlSession *session);
void smlSessionDispatchEvent(SmlSession *session, SmlSessionEventType type, SmlCommand *cmd, SmlCommand *parent, SmlStatus *headerreply, SmlError *error);
SmlLocation *smlSessionGetSource(SmlSession *session);
SmlLocation *smlSessionGetTarget(SmlSession *session);

/* Session Data input */
SmlBool smlSessionReceiveHeader(SmlSession *session, SmlHeader *header, SmlError **error);
SmlBool smlSessionReceiveBody(SmlSession *session, SmlParser *parser, SmlError **error);

/* Session Command input */
SmlBool smlSessionSendCommand(SmlSession *session, SmlCommand *cmd, SmlCommand *parent, SmlStatusReplyCb callback, void *userdata, SmlError **error);
SmlBool smlSessionStartCommand(SmlSession *session, SmlCommand *cmd, SmlCommand *parent, SmlStatusReplyCb callback, void *userdata, SmlError **error);
SmlBool smlSessionEndCommand(SmlSession *session, SmlCommand *parent, SmlError **error);
SmlBool smlSessionSendReply(SmlSession *session, SmlStatus *status, SmlError **error);

SmlBool smlSessionCheck(SmlSession *session);
void smlSessionDispatch(SmlSession *session);

#endif //_SML_SESSION_H_
