/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_mib.h>
#include <string.h>

#include "libsysactivity.h"

int sa_count_network_interfaces(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	u_int count;
	size_t len = sizeof count;
	if (sysctlbyname("net.link.generic.system.ifcount", &count, &len, NULL, 0) == -1)
		return ENOSYS;

	*number = count;
	return 0;
}

int sa_get_network_interface(char* name, struct sa_network_interface* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	int mib[6];
	struct ifmibdata if_data;
	size_t len = sizeof if_data;

	mib[0] = CTL_NET;
	mib[1] = PF_LINK;
	mib[2] = NETLINK_GENERIC;
	mib[3] = IFMIB_IFDATA;
	mib[5] = IFDATA_GENERAL;
	for (mib[4] = 1; sysctl(mib, 6, &if_data, &len, NULL, 0) != -1; mib[4]++) {
		if (strncmp(if_data.ifmd_name, name, sizeof if_data.ifmd_name) != 0)
			continue;

		//char	ifmd_name[IFNAMSIZ]; /* name of interface */
		strlcpy(dst->name, if_data.ifmd_name, sizeof dst->name);
		dst->received_bytes = if_data.ifmd_data.ifi_ibytes;
		dst->received_packets = if_data.ifmd_data.ifi_ipackets;
		dst->received_errors = if_data.ifmd_data.ifi_ierrors;
		dst->received_drop = if_data.ifmd_data.ifi_iqdrops;
		dst->received_multicast = if_data.ifmd_data.ifi_imcasts;
		dst->sent_bytes = if_data.ifmd_data.ifi_obytes;
		dst->sent_packets = if_data.ifmd_data.ifi_opackets;
		dst->sent_errors = if_data.ifmd_data.ifi_oerrors;
		dst->sent_multicast = if_data.ifmd_data.ifi_omcasts;

		return 0;
	}
	return ENODEV;
}

int sa_get_network_interfaces(struct sa_network_interface* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	int i;
	int mib[6];
	struct ifmibdata if_data;
	size_t len = sizeof if_data;

	*written = 0;
	mib[0] = CTL_NET;
	mib[1] = PF_LINK;
	mib[2] = NETLINK_GENERIC;
	mib[3] = IFMIB_IFDATA;
	mib[5] = IFDATA_GENERAL;
	for (i = 0; i < dst_size; i++) {
		mib[4] = i + 1;
		if (sysctl(mib, 6, &if_data, &len, NULL, 0) == -1)
			return ENODEV;

		//char	ifmd_name[IFNAMSIZ]; /* name of interface */
		strlcpy(dst[i].name, if_data.ifmd_name, sizeof dst->name);
		dst[i].received_bytes = if_data.ifmd_data.ifi_ibytes;
		dst[i].received_packets = if_data.ifmd_data.ifi_ipackets;
		dst[i].received_errors = if_data.ifmd_data.ifi_ierrors;
		dst[i].received_drop = if_data.ifmd_data.ifi_iqdrops;
		dst[i].received_multicast = if_data.ifmd_data.ifi_imcasts;
		dst[i].sent_bytes = if_data.ifmd_data.ifi_obytes;
		dst[i].sent_packets = if_data.ifmd_data.ifi_opackets;
		dst[i].sent_errors = if_data.ifmd_data.ifi_oerrors;
		dst[i].sent_multicast = if_data.ifmd_data.ifi_omcasts;

		(*written)++;
	}
	return 0;
}
