/*  tiffep - TI File Format Engine and Proxy
 *  Copyright (C) 2000-2001  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
  This unit contains some miscellaneous but useful functions.
*/

#include "hfiles.h"
#include "tiffep.h"


/*************************************/
/* Functions: calc_type -> extension */
/*************************************/

/* Return the extension of a group file according to the calc type */
TIEXPORT
const char *tiffep_group_file_ext(gint calc_type)
{
  if(calc_type > NCALCS) return NULL;
  return GROUP_FILE_EXT[calc_type];
}

/* Return the extension of a backup file according to the calc type */
TIEXPORT
const char *tiffep_backup_file_ext(gint calc_type)
{
  if(calc_type > NCALCS) return NULL;
  return BACKUP_FILE_EXT[calc_type];
}

/* Return the extension of a FLASHapp file according to the calc type */
TIEXPORT
const char *tiffep_flash_app_file_ext(gint calc_type)
{
  if(calc_type > NCALCS) return NULL;
  return FLASH_APP_FILE_EXT[calc_type];
}

/* Return the extension of a FLASHams file according to the calc type */
TIEXPORT
const char *tiffep_flash_os_file_ext(gint calc_type)
{
  return FLASH_OS_FILE_EXT[calc_type];
}

/************************************/
/* Functions: var_type -> extension */
/************************************/

// Return the type corresponding to the value
static const char *ti82_byte2type(byte data)
{
  if(data > TI82_MAXTYPES) return NULL;
  return TI82_TYPES[data];
}

// Return the value corresponding to the type
static byte ti82_type2byte(char *s)
{
  int i;

  for(i=0; i<TI82_MAXTYPES; i++)
    {
      if(!strcmp(TI82_TYPES[i], s)) break;
    }
  if(i>TI82_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
static const char *ti82_byte2fext(byte data)
{
  if(data > TI82_MAXTYPES) return NULL;
  return TI82_EXT[data];
}

// Return the value corresponding to the file extension
static byte ti82_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI82_MAXTYPES; i++)
    {
      if(!strcmp(TI82_EXT[i], s)) break;
    }
  if(i > TI82_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the type corresponding to the value
static const char *ti83_byte2type(byte data)
{
  if(data > TI83_MAXTYPES) return NULL;
  if(data==TI83_DIR) return "DIR";
  return TI83_TYPES[data];
}

// Return the value corresponding to the type
static byte ti83_type2byte(char *s)
{
  int i;
  
  for(i=0; i<TI83_MAXTYPES; i++)
    {
      if(!strcmp(TI83_TYPES[i], s)) break;
    }
  if(i > TI83_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
static const char *ti83_byte2fext(byte data)
{
  if(data > TI83_MAXTYPES) return NULL;
  return TI83_EXT[data];
}

// Return the value corresponding to the file extension
static byte ti83_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI83_MAXTYPES; i++)
    {
      if(!strcmp(TI83_EXT[i], s)) break;
    }
  if(i>TI83_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the type corresponding to the value
static const char *ti83p_byte2type(byte data)
{
  if(data > TI83p_MAXTYPES) return NULL;
  if(data==TI83p_DIR) return "DIR";
  return TI83p_TYPES[data];
}

// Return the value corresponding to the type
static byte ti83p_type2byte(char *s)
{
  int i;
  
  for(i=0; i<TI83p_MAXTYPES; i++)
    {
      if(!strcmp(TI83p_TYPES[i], s)) break;
    }
  if(i >= TI83p_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
static const char *ti83p_byte2fext(byte data)
{
  if(data > TI83p_MAXTYPES) return NULL;
  return TI83p_EXT[data];
}

// Return the value corresponding to the file extension
static byte ti83p_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI83p_MAXTYPES; i++)
    {
      if(!strcmp(TI83p_EXT[i], s)) break;
    }
  if(i>=TI83p_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the type corresponding to the value
static const char *ti85_byte2type(byte data)
{
  if(data > TI85_MAXTYPES) return NULL;
  return TI85_TYPES[data];
}

// Return the value corresponding to the type
static byte ti85_type2byte(char *s)
{
  int i;

  for(i=0; i<TI85_MAXTYPES; i++)
    {
      if(!strcmp(TI85_TYPES[i], s)) break;
    }
  if(i>TI85_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
static const char *ti85_byte2fext(byte data)
{
  if(data > TI85_MAXTYPES) return NULL;
  return TI85_EXT[data];
}

// Return the value corresponding to the file extension
static byte ti85_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI85_MAXTYPES; i++)
    {
      if(!strcmp(TI85_EXT[i], s)) break;
    }
  if(i > TI85_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.");
      return 0;
    }

  return i;
}

// Return the type corresponding to the value
static const char *ti86_byte2type(byte data)
{
  if(data > TI86_MAXTYPES) return NULL;
  return TI86_TYPES[data];
}

// Return the value corresponding to the type
static byte ti86_type2byte(char *s)
{
  int i;

  for(i=0; i<TI86_MAXTYPES; i++)
    {
      if(!strcmp(TI86_TYPES[i], s)) break;
    }
  if(i>TI86_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
static const char *ti86_byte2fext(byte data)
{
  if(data > TI86_MAXTYPES) return NULL;
  return TI86_EXT[data];
}

// Return the value corresponding to the file extension
static byte ti86_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI86_MAXTYPES; i++)
    {
      if(!strcmp(TI86_EXT[i], s)) break;
    }
  if(i > TI86_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.");
      return 0;
    }

  return i;
}

// Return the type corresponding to the value
static const char *ti89_byte2type(byte data)
{
  if(data > TI89_MAXTYPES) return NULL;
  return TI89_TYPES[data];
}

// Return the value corresponding to the type
static byte ti89_type2byte(char *s)
{
  int i;

  for(i=0; i<TI89_MAXTYPES; i++)
    {
      if(!strcmp(TI89_TYPES[i], s)) break;
    }
  if(i>TI89_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
static const char *ti89_byte2fext(byte data)
{
  if(data > TI89_MAXTYPES) return NULL;
  return TI89_EXT[data];
}

// Return the value corresponding to the file extension
static byte ti89_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI89_MAXTYPES; i++)
    {
      if(!strcmp(TI89_EXT[i], s)) break;
    }
  if(i > TI89_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the type corresponding to the value
static const char *ti92_byte2type(byte data)
{
  if(data > TI92_MAXTYPES) return NULL;
  return TI92_TYPES[data];
}

// Return the value corresponding to the type
/*TIEXPORT*/
static byte ti92_type2byte(char *s)
{
  int i;

  for(i=0; i<TI92_MAXTYPES; i++)
    {
      if(!strcmp(TI92_TYPES[i], s)) break;
    }
  if(i>TI92_MAXTYPES)
    {
      DISPLAY2("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
/*TIEXPORT*/
static const char *ti92_byte2fext(byte data)
{
  if(data > TI92_MAXTYPES) return NULL;
  return TI92_EXT[data];
}

// Return the value corresponding to the file extension
/*TIEXPORT*/
static byte ti92_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI92_MAXTYPES; i++)
    {
      if(!strcmp(TI92_EXT[i], s)) break;
    }
  if(i > TI92_MAXTYPES)
    {
      DISPLAY2("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the type corresponding to the value
static const char *ti92p_byte2type(byte data)
{
  if(data > TI92p_MAXTYPES) return NULL;
  return TI92p_TYPES[data];
}

// Return the value corresponding to the type
static byte ti92p_type2byte(char *s)
{
  int i;

  for(i=0; i<TI92p_MAXTYPES; i++)
    {
      if(!strcmp(TI92p_TYPES[i], s)) break;
    }
  if(i>TI92p_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

// Return the file extension corresponding to the value
static const char *ti92p_byte2fext(byte data)
{
  if(data > TI92p_MAXTYPES) return NULL;
  return TI92p_EXT[data];
}

// Return the value corresponding to the file extension
static byte ti92p_fext2byte(char *s)
{
  int i;

  for(i=0; i<TI92p_MAXTYPES; i++)
    {
      if(!strcmp(TI92p_EXT[i], s)) break;
    }
  if(i > TI92p_MAXTYPES)
    {
      printf("Warning: unknown type. It is a bug. Please report this information.\n");
      return 0;
    }

  return i;
}

/*****************************/
/* Value -> String functions */
/*****************************/

/* 
   Return a string describing the calculator type
   - calc_type [in]: a calculator type
   - const char* [out]: the string
*/
TIEXPORT
const char *tiffep_calc_type_to_string(gint calc_type)
{
  switch(calc_type)
    {
    case TYPE_NONE: return "none";
    case TYPE_TI92P: return "92+";
    case TYPE_TI92: return "92";
    case TYPE_TI89: return "89";
    case TYPE_TI86: return "86";
    case TYPE_TI85: return "85";
    case TYPE_TI83P: return "83+";
    case TYPE_TI83: return "83";
    case TYPE_TI82: return "82";
    case TYPE_TI73: return "73";
    default: return NULL;
    }

  return NULL;
}

/* 
   Return a string describing the variable type
   - var_type [in]: a var type
   - const char* [out]: the string
*/
TIEXPORT
const char *tiffep_var_type_to_string(gint var_type)
{
  switch(var_type)
    {
    case TYPE_EXPRESSION: return "expression";
    case TYPE_LIST      : return "list";
    case TYPE_MATRIX    : return "matrix"; 
    case TYPE_DATA      : return "data";
    case TYPE_TEXT      : return "text";
    case TYPE_STRING    : return "string";
    case TYPE_GDB       : return "graphic data base";
    case TYPE_FIGURE    : return "figure";
    case TYPE_PICTURE   : return "picture";
    case TYPE_PROGRAM   : return "program";
    case TYPE_FUNCTION  : return "function";
    case TYPE_MACRO     : return "macro";
    case TYPE_BACKUP    : return "backup";
    case TYPE_DIRECTORY : return "folder";
    case TYPE_GROUP     : return "group";
    case TYPE_FLASH     : return "flash";
    case TYPE_ASM       : return "assembly program";
    default: return NULL;
    }

  return NULL;
}

/* 
   Return a string describing the variable attribute
   - var_atrb [in]: a variable attribute
   - const char* [out]: the string
*/
TIEXPORT
const char *tiffep_attribute_to_string(gint var_atrb)
{
  switch(var_atrb)
    {
    case ATTRB_NONE:      return "none"; break;
    case ATTRB_LOCK:      return "locked"; break;
    case ATTRB_ARCHIVE:   return "archived"; break;
    case ATTRB_PROTECTED: return "protected"; break;
    default: return NULL;
    }

  return NULL;
}

/* 
   Return a string describing the file type (.82s for instance)
   - file_type [in]: a file type
   - const char* [out]: the string
*/
TIEXPORT
const char *tiffep_file_type_to_string(gint file_type)
{
  /* To do */

  return "none";
}

/*
   Return a string describing the file type (.82s for instance)
   - calc_type [in]: a calculator type
   - var_type [in]: a variable type (independant format)
   - const char* [out]: the string
*/
TIEXPORT
const char* tiffep_vartype_to_file_extension(gint calc_type, gint var_type)
{
  switch(calc_type)
    {
    case TYPE_TI73:  return "none";
    case TYPE_TI82:  return ti82_byte2fext(var_type);
    case TYPE_TI83:  return ti83_byte2fext(var_type);
    case TYPE_TI83P: return ti83p_byte2fext(var_type);
    case TYPE_TI85:  return ti85_byte2fext(var_type);
    case TYPE_TI86:  return ti86_byte2fext(var_type);
    case TYPE_TI89:  return ti89_byte2fext(var_type);
    case TYPE_TI92:  return ti92_byte2fext(var_type);
    case TYPE_TI92P: return ti92p_byte2fext(var_type);
    default: return NULL;
    }

  return NULL;
}


/*******************/
/* Query functions */
/*******************/

/* 
   Check whether it is a group file
   - filename [in]: a file name
   - int [out]: TRUE if group file, FALSE otherwise
*/
TIEXPORT
int tiffep_is_a_group_file(gchar *filename)
{
  int i;
  gchar *s = g_strdup(filename);

  g_strdown(s);
  for(i=1; i<CALC_MAX; i++)
    {
      if(strstr(s, GROUP_FILE_EXT[i]) != NULL)
	{
	  g_free(s);
	  return TRUE;
	}
    }
  g_free(s);
  
  return FALSE;
}

/* 
   Check whether it is a backup file
   - filename [in]: a file name
   - int [out]: TRUE if backup file, FALSE otherwise
*/
TIEXPORT
int tiffep_is_a_backup_file(gchar *filename)
{
  gint i;
  gchar *s = g_strdup(filename);

  g_strdown(s);
  for(i=1; i<CALC_MAX; i++)
    {
      if(strstr(s, BACKUP_FILE_EXT[i]) != NULL)
	{
	  g_free(s);
	  return TRUE;
	}
    }
  g_free(s);

  return FALSE;
}

/* 
   Check whether it is a flash file
   - filename [in]: a file name
   - int [out]: TRUE if flash file, FALSE otherwise
*/
TIEXPORT
int tiffep_is_a_flash_file(gchar *filename)
{
  return FALSE;
}



/* 
   Return the calc type corresponding to the file
   - filename [in]: a filename
   - int [out]: the calculator type
*/
TIEXPORT
int tiffep_which_calc_type_from_file(char *filename)
{
  gchar *ext;

  ext = strrchr(filename, '.');

  if(strstr(ext, "82") != NULL)
    return TYPE_TI82;
  else if(strstr(ext, "83") != NULL)
    return TYPE_TI83;
  else if(strstr(ext, "8x") != NULL)
    return TYPE_TI83P;
  else if(strstr(ext, "85") != NULL)
    return TYPE_TI85;
  else if(strstr(ext, "86") != NULL)
    return TYPE_TI86;
  else if(strstr(ext, "89") != NULL)
    return TYPE_TI89;
  else if(strstr(ext, "92") != NULL)
    return TYPE_TI92;
  else if(strstr(ext, "9x") != NULL)
    return TYPE_TI92P;
  else return TYPE_NONE;

  return TYPE_NONE;
}

/* 
   Return the var type corresponding to the file
   - filename [in]: a filename
   - int [out]: the var type
*/
TIEXPORT
int tiffep_which_var_type_from_file(char *filename)
{
  gchar *ext;

  /* For checking the extension & type file */
  if(tiffep_which_calc_type_from_file(filename) == TYPE_NONE)
    return TYPE_NONE;

  ext = strrchr(filename, '.');
#ifdef __WIN32__
  strupr(ext);
#endif
  switch(tiffep_which_calc_type_from_file(filename))
    {
    case TYPE_TI82:
    case TYPE_TI83:
    case TYPE_TI83P:
    case TYPE_TI85:
    case TYPE_TI86:
      return TYPE_NONE;
      break;

    case TYPE_TI89:
    case TYPE_TI92:
    case TYPE_TI92P:
      if(strstr(ext, "E") || strstr(ext, "e"))      return TYPE_EXPRESSION;
      else if(strstr(ext, "L") || strstr(ext, "l")) return TYPE_LIST;
      else if(strstr(ext, "M") || strstr(ext, "m")) return TYPE_MATRIX;
      else if(strstr(ext, "C") || strstr(ext, "c")) return TYPE_DATA;
      else if(strstr(ext, "T") || strstr(ext, "t")) return TYPE_TEXT;
      else if(strstr(ext, "S") || strstr(ext, "s")) return TYPE_STRING;
      else if(strstr(ext, "D") || strstr(ext, "d")) return TYPE_GDB;
      else if(strstr(ext, "A") || strstr(ext, "a")) return TYPE_FIGURE;
      else if(strstr(ext, "I") || strstr(ext, "i")) return TYPE_PICTURE;
      else if(strstr(ext, "P") || strstr(ext, "p")) return TYPE_PROGRAM;
      else if(strstr(ext, "F") || strstr(ext, "f")) return TYPE_FUNCTION;
      else if(strstr(ext, "X") || strstr(ext, "x")) return TYPE_MACRO;
      else if(strstr(ext, "B") || strstr(ext, "b")) return TYPE_BACKUP;
      else if(strstr(ext, "G") || strstr(ext, "g")) return TYPE_GROUP;
      else if(strstr(ext, "Z") || strstr(ext, "z")) return TYPE_ASM;
      else if(strstr(ext, "K") || strstr(ext, "k")) return TYPE_FLASH;
      else return TYPE_NONE;
  break;
    default:
      return TYPE_NONE;
      break;
    }

  return TYPE_NONE;
}

/* 
   Return the file = calc & var type corresponding to the file
   - filename [in]: a filename
   - [out]: the var type ORed with the calc type
*/
TIEXPORT
int tiffep_which_file_type_from_file(char *filename)
{
  return (tiffep_which_var_type_from_file(filename) |
	  tiffep_which_calc_type_from_file(filename));
}

/* 
   Read a string of 8 chars from a file
   - s [out]: a buffer for storing the string
   - f [in]: a file descriptor
   - [out]: the result of the operation (0 if failed)
*/
gint read_string8(gchar *s, FILE *f)
{
  return ((fgets(s, 9, f)==NULL) ? 0 : 1);
}

/* 
   Write a string of 8 chars (NULL padded) to a file
   - s [in]: a string
   - f [in]: a file descriptor
   - [out]: always different of 0
*/
gint write_string8(gchar *s, FILE *f)
{
  gint i;
  gint l = 8;

  l=strlen(s);
  if(l > 8)
    {
      g_warning("String passed in 'write_string8' is too long (>8 chars).");
      l = 8;
    }  

  for(i=0; i<l; i++)
    fputc(s[i], f);
  for(i=l; i<8; i++)
    fputc(0x00, f);
  
  return !0;
}

/* 
   Read a string of 'n' chars from a file
   - s [out]: a buffer for storing the string
   - f [in]: a file descriptor
   - [out]: the result of the operation (0 if failed)
*/
gint read_string_n(gchar *s, gint n, FILE *f)
{
  return ((fgets(s, n+1, f)==NULL) ? 0 : 1);
}

/* 
   Write a string of 'n' chars (NULL padded) to a file
   - s [in]: a string
   - f [in]: a file descriptor
   - [out]: always different of 0
*/
gint write_string_n(gchar *s, gint n, FILE *f)
{
  gint i;
  gint l = n;

  l=strlen(s);
  if(l > n)
    {
      g_warning("String passed in 'write_string8' is too long (>n chars).");
      l = n;
    }  

  for(i=0; i<l; i++)
    fputc(s[i], f);
  for(i=l; i<n; i++)
    fputc(0x00, f);
  
  return !0;
}

/* 
   Compute the checksum of a byte array. Returns a word value.
   - buffer [in]: an array of byte values
   - size [in]: the array size
   - chk [out]: the computed checksum
   - [out]: the computed checksum
*/
word compute_checksum(byte *buffer, gint size, word *chk)
{
  gint i;
  word c = 0;
 
  for(i=0; i<size; i++)
    c += buffer[i];
  *chk = c;

  return c;
}


