/*  tiffep - TI File Format Engine and Proxy
 *  Copyright (C) 2000-2001  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "hfiles.h"
#include "tiffep.h"


/*
  Allocate a TiffepFolderInfo structure and fill it with null values.
  - TiffepFolderInfo [out]: the allocated structure, NULL otherwise.
*/
TIEXPORT
TiffepFolderInfo* tiffep_alloc_di_struct(void)
{ 
  TiffepFolderInfo *fldinfo = (TiffepFolderInfo *)g_malloc0(sizeof(TiffepFolderInfo));
  return fldinfo;
}

/*
  Free all allocated memory
  - di [in]: the informations stored in an allocated structure
  for being freed
  - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_free_di_struct(TiffepFolderInfo **fldinfo)
{
  TiffepFolderInfo *di = *fldinfo;

  if(di == NULL) 
    {
      g_warning(_("tiffep_free_di_struct: NULL pointer.\n"));
      return -1;
    }

  // free the fields first
  g_free(DI_NAME(di));
  
  // free the structure itself
  g_free(di);
  *fldinfo = NULL;

  return 0;
}

/*
  Duplicate a TiffepFolderInfo structure
  - di [in]: the TiffepFolderInfo structure to duplicate
  - TiffepFolderInfo [out]: the newly allocated structure if success, 
  NULL otherwise.
*/
TIEXPORT
TiffepFolderInfo* tiffep_dup_di_struct(TiffepFolderInfo *di)
{
  TiffepFolderInfo *src = di;
  TiffepFolderInfo *dst = NULL;

  dst = tiffep_alloc_di_struct();

  DI_NAME(dst)    = g_strdup(DI_NAME(src));
  DI_NUMVARS(dst) = DI_NUMVARS(src);
  DI_OFFSET(dst)  = DI_OFFSET(src); 

  return dst;
}

/*
  Clear a TiffepFolderInfo structure
  - di [in]: the TiffepFolderInfo structure to duplicate
  - TiffepFolderInfo [out]: the newly allocated structure if success, 
  NULL otherwise.
*/
TIEXPORT
TiffepFolderInfo* tiffep_clear_di_struct(TiffepFolderInfo *di)
{
  memset((void *)di, 0, sizeof(TiffepFolderInfo));
  return di;
}

/*
  DISPLAY2 a TiffepFolderInfo structure
  - di [in]: the TiffepFolderInfo structure to print
  - int [out]: none
*/
TIEXPORT
int tiffep_print_di_struct(TiffepFolderInfo *di)
{
  if(di == NULL)
    {
      g_warning("tiffep_print_di_struct: NULL pointer.\n");
      return -1;
    }
  DISPLAY2("fld_path    = <%s>\n", di->name);
  DISPLAY2("fld_numvars = %i\n", di->num_vars);
  DISPLAY2("offset     = 0x%04x\n", di->offset);

  return 0;
}
