program shanks;
(* file exemples/pascal/shanks.p: square root modulo an odd prime
 *-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |            Racine carre modulo p, algorithme de Shanks               |
 |                                                                       |
 +-----------------------------------------------------------------------*)

uses _name_;
{$ifdef __GPC__}{$X+}{$endif}

                       (* +------------------------+
                          |  Symbole de Kronecker  |
                          +------------------------+ *)

(* a = naturel quelconque, b = naturel impair *)
function kronecker(a,b:xint):int;
var x,y,z : xint;
    i,k   : int;

begin
   
   x := f_copy(a);
   y := f_copy(b);
   
   (* Algorithme d'Euclide + rciprocit quadratique *)
   k := 1;
   while neq_1(x,0) do begin

      (* i <- valuation 2-adique de x, x <- x/2^i *)
      i := 0; while not(nth_bit(x,i)) do i := i+1;
      shiftr(x, x, i);

      (* (2^i/y) = -1 ssi i est impair et y = 3 ou 5 mod 8 *)
      if ((i and 1) = 1) and (((nth_word(y,0) + 2) and 7) > 4) then k := -k;

      (* (x/y) = (y/x) si x ou y = 1 mod 4, -(y/x) sinon *)
      if nth_bit(x,1) and nth_bit(y,1) then k := -k;

      (* y <- x, x <- y mod x *)
      z := x; x := y; y := z; gmod(x, x, y, 1);

      (* (-1/y) = 1 ssi y = 1 mod 4 *)
      if sgn(x) < 0 then begin
         abs(x,x);
         if nth_bit(y,1) then k := -k;
      end
      
   end;

   (* si pgcd(x,y) != 1 alors (x/y) = 0 *)
   if neq_1(y,1) then k := 0;

   xfree(x);
   xfree(y);
   kronecker := k;

end;



                      (* +---------------------------+
                         |  Racine carre modulaire  |
                         +---------------------------+ *)

(* p = premier impair, a = b^2 mod p, retourne b (compris entre 0 et p/2) *)
procedure sqrtmod(var b:xint; a,p:xint);
var q,x,y,z : xint;
    k,l     : int;
begin
   
   q := xnew();
   x := xnew();
   y := xnew();
   z := xnew();

   (* dcompose p = 1 + 2^k*(2*q+1) *)
   k := 1; while not(nth_bit(p,k)) do k := k+1;
   shiftr(q, p, k+1);

   (* b <- a^(q+1) mod p, x <- a^(2*q+1) mod p (donc b^2 = a*x mod p) *)
   powmod(x, a, q, p);
   mul(b, x, a); modulo(b, b, p);
   mul(x, x, b); modulo(x, x, p);

   if sup_1(x,1) then begin

      (* y <- lment d'ordre 2^k *)
      repeat nrandom(y, nbits(p)) until kronecker(y,p) = -1;
      shiftl(q, q, 1); add_1(q, q, 1);
      powmod(y, y, q, p);

      repeat
       
         (* ordre de x = 2^l *)
         sqr(z, x); modulo(z, z, p);
         l := 1; while (l < k) and neq_1(z,1) do begin
            l := l+1;
            sqr(z, z); modulo(z, z, p);
         end;
         if l >= k then begin
            writeln('internal error: l >= k');
            exit;
         end;

         (* b <- b*y^(2^(k-l-1))
            x <- x*y^(2^(k-l)), 
            y <- y^(2^(k-l))
            donc b^2 = a*x mod p, ordre(y) = 2^l et ordre(x) < 2^l *)
         while l < k-1 do begin
            k := k-1;
            sqr(y, y); modulo(y, y, p);
         end;
         mul(b, b, y); modulo(b, b, p);
         sqr(y, y);    modulo(y, y, p); k := k-1;
         mul(x, x, y); modulo(x, x, p);

      until eq_1(x,1);
      
   end;

   (* rduit b  l'intervalle [0,p/2] *)
   gmod(b, b, p, 1); abs(b, b);

   (* termin *)
   xfree(x);
   xfree(y);
   xfree(z);

end;

                        (* +-----------------------+
                           |  Fonction principale  |
                           +-----------------------+ *)

var help,test,p_ok,a_ok : boolean;
    n,i   : int;
    p,a,b : xint;
    s     : pchar;
    c     : word;
   
const test_p = '100000000000000000000000000000000000133';
      test_a = '123456';
      test_b = '36705609512061164177523976477230978260';
begin
   help := false; test := false; p_ok := false; a_ok := false;
   n := 200;
  
    p := xnew;
    a := xnew;
    b := xnew;

   (* dcode les arguments *)
   i := 1;
   while not(help) and not(test) and (i <= paramcount) do begin
      if paramstr(i) = '-p' then begin
         i := i+1;
         if i <= paramcount then begin
            copy_pstring(p,paramstr(i)); p_ok := true;
         end
         else help := true;
      end
      else if paramstr(i) = '-a' then begin
         i := i+1;
         if i <= paramcount then begin
            copy_pstring(a,paramstr(i)); a_ok := true;
         end
         else help := true;
      end
      else if paramstr(i) = '-bits' then begin
         i := i+1;
	 c := 1;
         if i <= paramcount then val(paramstr(i),n,c);
         if c <> 0 then help := true;
      end
      else if paramstr(i) = '-test' then begin
         copy_string(p,test_p);
         copy_string(a,test_a);
         test := true; p_ok := true; a_ok := true;
      end
      else help := true;
      i := i+1;
   end;
   
   if help then writeln('usage: ',paramstr(0),' [-p <odd prime>] [-a <quadres mod p>] [-test] [-bits n]')
   else begin

      random_init(0);

      (* si p est dfini, vrifie qu'il est premier impair.
         sinon tire un nombre premier impair au hasard *)
      if p_ok then begin
         if not(nth_bit(p,0)) or (isprime(p) = t_false) then begin
            writeln('p is not an odd prime');
            exit;
         end
      end
      else begin
         nrandom1(p, n); if not(nth_bit(p,0)) then add_1(p,p,1);
         repeat add_1(p, p, 2) until isprime(p) <> t_false;
	 s := string_of(p); writeln('p = ',s); strfree(s);
      end;

      (* si a est dfini, vrifie que c'est un carr non nul modulo p.
         sinon tire un rsidu quadratique au hasard.*)
      if a_ok then begin
         if kronecker(a,p) <> 1 then begin
            writeln('kronecker(a,p) != 1');
            exit;
         end
      end
      else begin
         nrandom(a,n);
         repeat add_1(a, a, 1) until kronecker(a,p) = 1;
         s := string_of(a); writeln('a = ',s); strfree(s);
      end;

      (* calcule la racine carre *)
      sqrtmod(b, a, p);
      s := string_of(b);
      if test then begin
         if strcmp(s,test_b) <> 0 then writeln('error in the ',paramstr(0),' test')
         else                          writeln(paramstr(0),#9'test ok');
      end
      else writeln('b = ',s);
      strfree(s);

   end;

   xfree(p);
   xfree(a);
   xfree(b);

end.
