// Copyright (C) 2003 Shai Ayal <shaiay@users.sourceforge.net>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include <FL/gl.h>
#include "gl2ps.h"
#include "axes.h"
#include "text.h"

#define OBJ_NAME "text"

Text::Text(ocpl::Handle Parent, 
	   const char* _text,
	   ocpl::Real _x,
	   ocpl::Real _y,
	   ocpl::Real _z
	   ) : Object(Parent) 
{
  Properties["String"] = new String(_text);
  Properties["Position"] = Real2Matrix(_x,_y,_z);
  SET_TYPE;
  COPY_DEFAULT(Color,Color);
  COPY_DEFAULT(Clipping,Radio);
  COPY_DEFAULT(FontName,Radio);
  COPY_DEFAULT(FontSize,Scalar);
  COPY_DEFAULT(Rotation,Scalar);
  COPY_DEFAULT(HorizontalAlignment,Radio);
  COPY_DEFAULT(VerticalAlignment,Radio);

  Font = 0L;
  _w = 0;
  _h = 0;
  for(int i=0;i<3;i++) {
    lmin[i]=min[i]=ocpl::nan;
    lmax[i]=max[i]=ocpl::nan;
  }
}

void Text::PostSet(ocpl::command& com)
{
  std::string prop(tolower(com.argin(1)->data));
  
  if(prop=="string" || prop=="fontname" || prop=="fontsize") {
        CalcWH();
  }
}

void Text::CalcWH()
{

  MAKE_REF(string,String);
  float hh;
  
  if(!Font) {
    std::string ttt;
    SetFont( ttt);
  }

  if(Font) {
    float llx,lly,llz,urx,ury,urz;
    
    // width is the exact width of the string
    Font->BBox(string(),llx,lly,llz,urx,ury,urz);
    _w = urx-llx;
    hh = ury-lly;
    // height is the heigth of the letter "09M"
    Font->BBox("09M",llx,lly,llz,urx,ury,urz);
    _h  = ury-lly;
    // line heigth contains both the highest & lowest character
    _lh = Font->Ascender()-Font->Descender(); // heigth is taken as "line height"
  }
  else {
    _w = 0;
    _h = 0;
    _lh = 0;
    hh = 0;
  }

  // Thsi is the exact bounding box of the string
  delete Properties["WidthHeight"];
  Properties["WidthHeight"] = Real2Matrix(_w,hh);
}

int Text::SetFont(std::string& psfontname)
{ 
  MAKE_REF(fontname,Radio);
  MAKE_REF(fontsize,Scalar);

  int fs=static_cast<int>(fontsize());
  ocpl::Handle hnd = FindParentOfType("Figure")->GetHandle();

  if(fontname()=="mono") {
    Font = FTGLFontManager::Instance().GetFont("n022003l.pfb" , fs, hnd);
    psfontname = "Courier";
  }
  else if(fontname()=="serif") {
    Font = FTGLFontManager::Instance().GetFont("n021003l.pfb" , fs, hnd);
    psfontname = "Times-Roman";
  }
  else { // sans
    Font = FTGLFontManager::Instance().GetFont("n019003l.pfb" , fs, hnd);
    psfontname = "Helvetica";
  }

  CalcWH();
  return fs;
}

void Text::draw()
{
  IS_VISIBLE;

  MAKE_REF(position,Matrix);
  MAKE_REF(string,String);
  MAKE_REF(clipping,Radio);
  MAKE_REF(horizontalalignment,Radio);
  MAKE_REF(verticalalignment,Radio);
  MAKE_REF(rotation,Scalar);
  std::string psfontname;
  
  if(string()=="") return;

  SET_CLIPPING;
  
  SET_COLOR(color);

  Axes* axes = dynamic_cast<Axes*>(FindParentOfType("axes"));
  Radio& xscale = ::Get<Radio>(axes,"xscale");
  Radio& yscale = ::Get<Radio>(axes,"yscale");

  int fs=SetFont(psfontname);

  if (printing) {
    int gl2psa=GL2PS_TEXT_BL;
    if(horizontalalignment()=="left") {
      if(verticalalignment()==  "bottom") gl2psa=GL2PS_TEXT_BL;
      else if(verticalalignment()==  "top") gl2psa=GL2PS_TEXT_TL;
      else if(verticalalignment()==  "middle") gl2psa=GL2PS_TEXT_CL;
    }
    else if(horizontalalignment()=="right") {
      if(verticalalignment()==  "bottom") gl2psa=GL2PS_TEXT_BR;
      else if(verticalalignment()==  "top") gl2psa=GL2PS_TEXT_TR;
      else if(verticalalignment()==  "middle") gl2psa=GL2PS_TEXT_CR;
    }
    else if(horizontalalignment()=="center") {
      if(verticalalignment()==  "bottom") gl2psa=GL2PS_TEXT_B;
      else if(verticalalignment()==  "top") gl2psa=GL2PS_TEXT_T;
      else if(verticalalignment()==  "middle") gl2psa=GL2PS_TEXT_C;
    }
    
    glRasterPos3d(LOGIT(position(0),xscale()=="log"),
		  LOGIT(position(1),yscale()=="log"),
		  0);
    gl2psTextOpt(string(), 
		 psfontname.c_str(), 
		 fs,
		 gl2psa , 
		 rotation() ) ;
  }
  else {
    double px,py;
    axes->GetPixel(px,py);

    double dx=0,dy=0;

    if(horizontalalignment()=="left")        dx = 0;
    else if(horizontalalignment()=="right")  dx = -_w;
    else if(horizontalalignment()=="center") dx = -_w/2;

    if(verticalalignment()==  "bottom")      dy = -Font->Descender();
    else if(verticalalignment()==  "top")    dy = -h();
    else if(verticalalignment()==  "middle") dy = -h()/2.0;
  
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslated(LOGIT(position(0),xscale()=="log"),
		 LOGIT(position(1),yscale()=="log"),
		 0);
    glScaled(px,py,1.0);
    glRotated(rotation(),0,0,1);
    glTranslated(dx,dy,0);

    glEnable(GL_TEXTURE_2D); 
    if(Font) Font->Render(string());
    else std::cerr << "OctPlot: can't find font -- check font path\n";
    glDisable(GL_TEXTURE_2D); 

    glPopMatrix();
  }

  UNSET_CLIPPING;
}
