/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "OmModule.h"
#include <cassert>
#include "OmGtk.h"
#include "Controller.h"
#include "OmPatchBayArea.h"
#include "NodeControlWindow.h"
#include "PatchModel.h"
#include "NodeModel.h"
#include "OmPort.h"
#include "MetadataModel.h"
#include "GladeFactory.h"
#include "RenameWindow.h"
#include "PatchController.h"
#include "PatchWindow.h"
#include "OmGtkApp.h"

namespace OmGtk {


OmModule::OmModule(OmPatchBayArea* patch_bay, NodeModel* node_model, PatchController* parent)
: PatchBay::Module(patch_bay, node_model->name(), node_model->x(), node_model->y()),
  m_parent(parent),
  m_node_model(node_model),
  m_dialog(NULL)
{
	assert(m_node_model != NULL);

	Gtk::Menu::MenuList& items = m_menu.items();
	
	items.push_back(Gtk::Menu_Helpers::MenuElem("Controls...",
		sigc::mem_fun(this, &OmModule::show_control_window)));
	items.push_back(Gtk::Menu_Helpers::MenuElem("Rename...",
		sigc::mem_fun(this, &OmModule::menu_rename)));
	items.push_back(Gtk::Menu_Helpers::SeparatorElem());
	items.push_back(Gtk::Menu_Helpers::MenuElem("Disconnect All",
		sigc::mem_fun(this, &OmModule::menu_disconnect_all)));
	items.push_back(Gtk::Menu_Helpers::MenuElem("Destroy",
		sigc::mem_fun(this, &OmModule::menu_remove)));
	
	items[0].property_sensitive() = false;

	if (node_model->polyphonic() && node_model->parent() != NULL
			&& node_model->parent()->poly() > 1) {
		m_module_box.property_width_units() = 2.0;
		m_module_box.property_outline_color_rgba() = 0x99AABBFF;
	}

	if (node_model->plugin_model()->type() == PluginModel::Internal
			&& node_model->plugin_model()->plug_label() == "midi_control_in") {
		Gtk::Menu::MenuList& items = m_menu.items();
		items.push_back(Gtk::Menu_Helpers::MenuElem("Learn",
			sigc::mem_fun(this, &OmModule::menu_learn)));
	}
}


OmModule::~OmModule()
{
	if (m_dialog != NULL) {
		m_dialog->hide();
		delete m_dialog;
	}
}


void
OmModule::store_location()
{
	char temp_buf[16];
	
	if (m_node_model->x() != property_x()) {
		m_node_model->x(property_x());
		snprintf(temp_buf, 16, "%f", m_node_model->x());
		controller->set_metadata(m_node_model->path(), "module-x", temp_buf);
	}

	if (m_node_model->y() != property_y()) {
		m_node_model->y(property_y());
		snprintf(temp_buf, 16, "%f", m_node_model->y());
		controller->set_metadata(m_node_model->path(), "module-y", temp_buf);
	}
}


void
OmModule::move_to(double x, double y)
{
	Module::move_to(x, y);
	m_node_model->x(x);
	m_node_model->y(y);
	//store_location();
}


void
OmModule::path(const string& path)
{
	m_patch_bay->rename_module(OmPath::name(m_node_model->path()),
		OmPath::name(path));
	
	if (m_node_model->parent() != NULL)
		m_node_model->parent()->rename_node(m_node_model->path(), path);

	m_node_model->path(path);
}


void
OmModule::add_om_port(PortModel* pm, bool resize_to_fit)
{
	OmPort* port = new OmPort(this, pm);

	port->signal_event().connect(
		sigc::bind<Port*>(sigc::mem_fun(m_patch_bay, &OmPatchBayArea::port_event), port));
	
	if (pm->is_control() && pm->is_input()) {
		m_menu.items()[0].property_sensitive() = true;

		if (m_dialog != NULL) {
			m_dialog->control_panel()->add_port(pm);
			m_dialog->resize();
			port->add_control_panel(m_dialog->control_panel());
		}
	}
	
	Module::add_port(port, resize_to_fit);
}


void
OmModule::metadata_update(const MetadataModel* const mm)
{
	assert(mm->path() == m_node_model->path());
	
	if (mm->key() == "module-x") {
		float x = atof(mm->value().c_str());
		if (x > 0 && x < m_patch_bay->width())
			move_to(x, property_y().get_value());
	} else if (mm->key() == "module-y") {
		float y = atof(mm->value().c_str());
		if (y > 0 && y < m_patch_bay->height())
			move_to(property_x().get_value(), y);
	} else {
		//cerr << "[OmModule] Unknown metadata key " << mm->key() << endl;
	}
	m_node_model->set_metadata(mm->key(), mm->value());
}


void
OmModule::show_control_window()
{
	assert(m_parent != NULL);
	assert(m_parent->window() != NULL);

	if (m_dialog == NULL) {
		m_dialog = new NodeControlWindow(m_parent, m_node_model);
		m_dialog->set_transient_for(*m_parent->window());
		for (PortList::iterator i = m_ports.begin(); i != m_ports.end(); ++i)
			if (((OmPort*)(*i))->port_model()->is_input()
					&& ((OmPort*)(*i))->port_model()->is_control())
				((OmPort*)(*i))->add_control_panel(m_dialog->control_panel());
	}
	
	if (m_dialog->control_panel()->num_controls() > 0) {
		m_dialog->show();
		m_dialog->raise();
	} else {
		m_dialog->hide();
	}
}


void
OmModule::menu_rename()
{
	assert(m_parent != NULL);

	// FIXME: will this be magically cleaned up?
	RenameWindow* win = NULL;
	Glib::RefPtr<Gnome::Glade::Xml> xml = glade_factory->new_glade_reference("rename_win");
	xml->get_widget_derived("rename_win", win);
	if (m_parent->window() != NULL)
		win->set_transient_for(*m_parent->window());
	else
		win->set_transient_for(*app->window());

	win->set_object(this);
	win->show();
}


void
OmModule::menu_remove()
{
	controller->remove_node(m_node_model->path());
}


void
OmModule::menu_learn()
{
	controller->midi_learn(m_node_model->path());
}

void
OmModule::menu_disconnect_all()
{
	controller->disconnect_all(m_node_model->path());
}


} // namespace OmGtk
