/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "PatchController.h"
#include <cassert>
#include <cstdlib>
#include "OmGtk.h"
#include "GladeFactory.h"
#include "StateManager.h"
#include "OmPath.h"
#include "ControlPanel.h"
#include "ConnectionModel.h"
#include "OmPatchBayArea.h"
#include "canvas/Module.h"
#include "PatchModel.h"
#include "Controller.h"
#include "SubpatchModule.h"
#include "DSSIModule.h"
#include "PatchWindow.h"
#include "NodeModel.h"
#include "OmModule.h"
#include "OmPort.h"
#include "MetadataModel.h"
#include "ControlModel.h"
#include "PluginModel.h"
#include "NodeControlWindow.h"


using std::cerr; using std::cout; using std::endl;

namespace OmGtk {


PatchController::PatchController(PatchModel* patch_model, Controller* controller, PatchController* parent, bool load_widgets)
: m_parent(parent),
  m_patch_model(patch_model),
  m_patch_bay(new OmPatchBayArea(this, patch_model->path(), 1600*2, 1200*2)),
  m_controller(controller),
  m_window(NULL),
  m_module(NULL),
  m_control_window(NULL),
  m_module_x(0),
  m_module_y(0)
{
	if (load_widgets) {
		Glib::RefPtr<Gnome::Glade::Xml> xml = glade_factory->new_glade_reference();
		xml->get_widget_derived("patch_win", m_window);
		assert(m_window != NULL);
		m_window->patch_controller(this);
		
		xml->get_widget_derived("control_panel_vbox", m_control_panel);
		m_control_panel->reparent(m_control_panel_bin);
		m_control_panel->init(this, patch_model, (patch_model->poly() > 1));
	
		m_control_window = new NodeControlWindow(this, m_patch_model);
		m_control_window->control_panel()->mirror(m_control_panel);
		m_control_panel->mirror(m_control_window->control_panel());
	}
}


PatchController::~PatchController()
{
	if (m_control_window != NULL) {
		m_control_window->hide();
		delete m_control_window;
		m_control_window = NULL;
		m_module->control_window(NULL);
	}

	if (m_window != NULL) {
		m_window->hide();
		delete m_window;
	}

	delete m_patch_bay;
}


void
PatchController::path(const string& new_path)
{
	assert(m_patch_model != NULL);
	
	OmModule* module = NULL;
	for (ModuleMap::iterator i = m_patch_bay->modules().begin();
			i != m_patch_bay->modules().end(); ++i) {
		module = ((OmModule*)((*i).second));
		assert(module != NULL);
		assert(module->node_model() != NULL);
		module->path(m_patch_model->path() +"/"+ module->node_model()->name());
	}

	if (m_module != NULL)
		m_module->path(new_path);

	if (m_window != NULL)
		m_window->set_title(string("Om - ").append(new_path));

	if (m_control_window != NULL)
		m_control_window->set_title(string("Om - ").append(new_path));
	
	m_patch_model->path(new_path);
}


void
PatchController::enabled(bool e)
{
	if (m_window != NULL)
		m_window->enabled(e);
	m_patch_model->enabled(e);
}


/** Adds a node to the patch bay.
 */
void
PatchController::new_node(NodeModel* const nm)
{
	if (m_patch_model->get_node(nm->name()) != NULL) {
		// Patch already exists, ignore
		delete nm;
	} else {
		m_patch_model->add_node(nm);

		OmModule* module = NULL;
		
		// Set zoom to 1.0 so module isn't all messed up (gnomecanvas is worthless trash)
		float old_zoom = m_patch_bay->zoom();
		m_patch_bay->set_pixels_per_unit(1.0);
		
		if (nm->plugin_model()->type() == PluginModel::DSSI)
			module = new DSSIModule(m_patch_bay, nm, this);
		else
			module = new OmModule(m_patch_bay, nm, this);
	
		for (PortModelList::const_iterator i = nm->port_models().begin(); i != nm->port_models().end(); ++i)
			module->add_om_port(*i, false);
	
		module->resize();

		m_patch_bay->add_module(module);

		// Reset zoom
		m_patch_bay->set_pixels_per_unit(old_zoom);
		module->zoom(old_zoom);
	}
}


/** Removes a node from this patch.
 */
void
PatchController::remove_node(const string& name)
{
	m_patch_bay->remove_module(name);
	m_patch_model->remove_node(name);
}


/** Add a subpatch to this patch.
 *
 * Unlike new_node, the subpatch module is created elsewhere and simply passed
 * here.  Subpatch modules are more clever and important..
 */
void
PatchController::new_subpatch(SubpatchModule* module)
{
	m_patch_model->add_node(module->patch_model());
	m_patch_bay->add_module(module);
}


/** Adds a port to this patch.
 *
 * Will add a port to the subpatch module and the control window, if they
 * exist.
 */
void
PatchController::new_port(PortModel* const pm)
{
	if (m_module != NULL) 
		m_module->add_om_port(pm, true);

	if (pm->is_control() && pm->is_input()) {
		m_control_panel->add_port(pm);
		m_control_window->control_panel()->add_port(pm);
		m_control_window->resize();
	}

	if (m_window != NULL && pm->is_control() && pm->is_input())
		m_window->menu_view_control_window()->property_sensitive() = true;

	m_patch_model->add_port_model(pm);
}


/** Removes a port from this patch
 */
void
PatchController::remove_port(const string& path)
{
	assert(OmPath::parent(path) == m_patch_model->path());

	if (m_module != NULL)
		m_module->remove_port(OmPath::name(path), true);

	if (m_control_window != NULL) {
		m_control_window->control_panel()->remove_port(path);
		m_control_window->resize();
	
		m_control_panel->remove_port(path);
	}

	m_patch_model->remove_port_model(path);
}


void
PatchController::connection(ConnectionModel* const cm)
{
	//std::cerr << "[PatchController] Connection: " << port1_name << " -> " << port2_name << std::endl;
	m_patch_bay->add_connection(cm->src_node_name(), cm->src_port_name(), cm->dst_node_name(), cm->dst_port_name());
	m_patch_model->add_connection(cm);

	// Disable control slider from destination node control window
	OmModule* m = (OmModule*)m_patch_bay->find_module(cm->dst_node_name());
	if (m != NULL) {
		OmPort* p = m->port(cm->dst_port_name());
		if (p != NULL && p->connections().size() == 1) {
			p->port_model()->connected(true);
			if (m->control_window() != NULL) {
				m->control_window()->control_panel()->disable_port(cm->dst_port_path());
				if (m->control_window()->control_panel()->mirror() != NULL)
					m->control_window()->control_panel()->mirror()->disable_port(cm->dst_port_path());
			}
		}
	}
}



void
PatchController::disconnection(const string& src_port_path, const string& dst_port_path)
{
	string src_node_name = OmPath::name(OmPath::parent(src_port_path));
	string src_port_name = OmPath::name(src_port_path);
	string dst_node_name = OmPath::name(OmPath::parent(dst_port_path));
	string dst_port_name = OmPath::name(dst_port_path);
	
	m_patch_bay->remove_connection(src_node_name, src_port_name, dst_node_name, dst_port_name);
	m_patch_model->remove_connection(src_port_path, dst_port_path);
	
	// Enable control slider in destination node control window
	OmModule* m = (OmModule*)m_patch_bay->find_module(dst_node_name);
	if (m != NULL) {
		OmPort* p = m->port(dst_port_name);
		if (p != NULL && p->connections().size() == 0) {
			p->port_model()->connected(false);
			if (m->control_window() != NULL) {
				m->control_window()->control_panel()->enable_port(dst_port_path);
				if (m->control_window()->control_panel()->mirror() != NULL)
					m->control_window()->control_panel()->mirror()->enable_port(dst_port_path);
			}
		}
	}
}


void
PatchController::metadata_update(const MetadataModel* const mm)
{
}


/** Try to guess a suitable location for a new module.
 */
void
PatchController::get_new_module_location(int& x, int& y)
{
	int scroll_x;
	int scroll_y;
	m_patch_bay->get_scroll_offsets(scroll_x, scroll_y);

	x = scroll_x + 20;
	y = scroll_y + 20;
}


void
PatchController::show_control_window()
{
	if (m_control_window == NULL)
		return;

	if (m_control_window->control_panel()->num_controls() > 0) {
		m_control_window->show();
		m_control_window->raise();
	} else {
		m_control_window->hide();
		m_window->show();
	}
}


/** Become the parent of the control panel.
 *
 * Used by OmGtkApp to 'get rid' of a control panel so it can display a new
 * one (a Gtk::Widget can not have no parent)
 */
void
PatchController::claim_control_panel()
{
	m_control_panel->reparent(m_control_panel_bin);
}


void
PatchController::module(SubpatchModule* sm)
{
	m_module = sm;
	sm->control_window(m_control_window);
}


} // namespace OmGtk
