/*
 * transcoder-nv.cc --
 *
 *      Netvideo Transcoder
 */

/*
 * This module was originally derived from:
 *
 * Netvideo version 3.2
 * Written by Ron Frederick <frederick@parc.xerox.com>
 *
 * Video decode routines
 *
 * Copyright (c) Xerox Corporation 1992. All rights reserved.
 *
 * License is granted to copy, to use, and to make and to use derivative
 * works for research and evaluation purposes, provided that Xerox is
 * acknowledged in all documentation pertaining to any such copy or derivative
 * work. Xerox grants no other licenses expressed or implied. The Xerox trade
 * name should not be used in any advertising without its written permission.
 *
 * XEROX CORPORATION MAKES NO REPRESENTATIONS CONCERNING EITHER THE
 * MERCHANTABILITY OF THIS SOFTWARE OR THE SUITABILITY OF THIS SOFTWARE
 * FOR ANY PARTICULAR PURPOSE.  The software is provided "as is" without
 * express or implied warranty of any kind.
 *
 * These notices must be retained in any copies of any part of this software.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/video/transcoder-nv.cc,v 1.6 2001/04/05 17:17:12 lim Exp $";

#include <stdio.h>
#include <string.h>
#include <sys/param.h>
#include "config.h"
#include "inet.h"
#include "rtp.h"
#include "transcoder.h"
#include "transmitter.h"
#include "encoder.h"
#include "crdef.h"

#ifdef INFOPAD
#include "encoder-ipadvq.h"
#include "ipadchan.h"
#endif /* INFOPAD */

class NvTranscoder : public VideoTranscoder {
public:
	NvTranscoder();
	int recv(const rtphdr* rh, const u_char* bp, int cc, Transmitter& tx);
protected:
	void process_hdr(const nvhdr*);
	virtual void configure();
	void info(char* wrk) const;
	const u_char* decode_run(const u_char* data, const u_char* end,
				 int color, int mark);
	void VidTransform_Rev(u_int* inp, u_char* yp, char* up, char* vp,
			      int width);
	void NVDCT_RevTransform(u_int* inp, u_char *yp,
				char* up, char* vp, int width);
	int compute_margins(void);
	struct margin {
		int top;
		int right;
		int left;
	} margin_;
	int rcrop_;
	int lcrop_;
	int topcrop_;
	int botcrop_;

	int outw_;
	int outh_;
	int use_dct_;
	u_char* frm_;
	int color_;
};

static class NvTranscoderMatcher : public Matcher {
public:
	NvTranscoderMatcher() : Matcher("transcoder") {}
	TclObject* match(const char* id) {
		if (strcmp(id, "nv") == 0)
			return (new NvTranscoder());
		return (0);
	}
} nv_transcoder_matcher;

NvTranscoder::NvTranscoder()
  	     :VideoTranscoder(sizeof(nvhdr)), use_dct_(0), frm_(0), color_(1)
{}

void NvTranscoder::configure()
{
	if (compute_margins() != 0)
		fprintf(stderr, "NvTranscoder: Couldn't compute margins.\n");
	delete frm_;
	int size = outw_ * outh_ * 2;
	frm_ = new u_char[size];
	memset(frm_, 0x80, size);
	crinit(outw_, outh_);
}

int NvTranscoder::compute_margins()
{
	/*
	 * Embed one image size in a bigger one.
	 * (extra areas will already be gray from init() )
	 */
	margin& m = margin_;
	m.top = m.left = m.right = 0;
	lcrop_ = 0;
	rcrop_ = inw_ >> 3;
	topcrop_ = 0;
	botcrop_ = inh_ >> 3;

	int dx = outw_ - inw_;
	int dy = outh_ - inh_;

	/*
	 * If output size is smaller that intput, crop input image to fit
	 * output size, otherwise try and center the image in the larger
	 * frame, making sure to align on macroblock (16x16) boundaries.
	 */
	if (dx < 0) {
		dx = -dx;
		/* Bail if can't align on block boundary. */
		if (dx % 8 != 0)
			return (-1);

		int q = dx / 16;
		if (q % 2 != 0)
			/* Can't center crop horizontally */
			lcrop_ = (dx - 16) / 2;
		else
			/* Center crop horizontally */
			lcrop_ = dx / 2;

		lcrop_ >>= 3;
		rcrop_ -= lcrop_;
	} else {
		if (dx % 8 != 0)
			return (-1);

		/* Know dx is multiple of 8 at least */
		if (dx % 16 == 8) {
			/* crop one block and bias dx */
			rcrop_--;
			dx -= 8;
		}

		int q = dx / 16;
		if (q % 2 != 0) {
			/* Can't center horizontally */
			m.right = ((dx - 16) / 2) >> 3;
			m.left = ((dx + 16) / 2) >> 3;
		} else
			/* Center horizontally */
			m.right = m.left = (dx / 2) >> 3;
	}

	if (dy < 0) {
		dy = -dy;
		/* Bail if can't align on block boundary. */
		if (dy % 8 != 0)
			return (-1);
		int q = dy / 16;
		if (q % 2 != 0)
			/* Can't center crop vertically */
			topcrop_ = (dy - 16) / 2;
		else
			/* Center crop vertically */
			topcrop_ = dy / 2;
		topcrop_ >>= 3;
		botcrop_ -= topcrop_;
	} else {
		/* Bail if can't align on block boundary. */
		if (dy % 8 != 0)
			return (-1);

		/* Know dy is multiple of 8 at least */
		if (dy % 16 == 8) {
			/* crop one block and bias dy */
			botcrop_--;
			dy -= 8;
		}

		int q = dy / 16;
		if (q % 2 != 0)
			/* Can't center vertically */
			m.top = ((dy - 16) / 2) >> 3;
		else
			/* Center vertically */
			m.top = (dy / 2) >> 3;
	}
	return (0);
}

void NvTranscoder::info(char* wrk) const
{
	strcpy(wrk, use_dct_ ? "[dct]" : "[haar]");
}

/* Sick little macro which will limit x to [0..255] with logical ops */
#define UCLIMIT(x) ((t = (x)), (t &= ~(t>>31)), (t | ~((t-256) >> 31)))
/* A variant of above which will limit x to [-128..127] */
#define SCLIMIT(x) (UCLIMIT((x)+128)-128)

void NvTranscoder::VidTransform_Rev(u_int* inp, u_char *yp,
				 char* up, char* vp, int width)
{
    register int i, t, t0, t1, t2, t3, t4, t5, *dataptr;
    register int width2=2*width, width6=6*width;
    register signed char *inpcptr=(signed char *)inp;
    static int block[64];
#define SIGN_EXTEND(c) c

    dataptr = block;
    for (i=0; i<8; i++) {
	if ((inp[0] << 8) == 0) {
	    t2 = t3 = t4 = t5 = SIGN_EXTEND(inpcptr[0]);
	} else {
	    t4 = SIGN_EXTEND(inpcptr[0]);
	    t5 = SIGN_EXTEND(inpcptr[1]);
	    t0 = t4 - t5;
	    t1 = t4 + t5;

	    t4 = SIGN_EXTEND(inpcptr[2]);
	    t5 = SIGN_EXTEND(inpcptr[3]);
	    t2 = t0 - t4;
	    t3 = t0 + t4;
	    t4 = t1 - t5;
	    t5 = t1 + t5;
	}

	if (inp[1] == 0) {
	    dataptr[0] = dataptr[1] = t2;
	    dataptr[2] = dataptr[3] = t3;
	    dataptr[4] = dataptr[5] = t4;
	    dataptr[6] = dataptr[7] = t5;
	} else {
	    t0 = SIGN_EXTEND(inpcptr[4]);
	    t1 = SIGN_EXTEND(inpcptr[5]);
	    dataptr[0] = t2 - t0;
	    dataptr[1] = t2 + t0;
	    dataptr[2] = t3 - t1;
	    dataptr[3] = t3 + t1;

	    t0 = SIGN_EXTEND(inpcptr[6]);
	    t1 = SIGN_EXTEND(inpcptr[7]);
	    dataptr[4] = t4 - t0;
	    dataptr[5] = t4 + t0;
	    dataptr[6] = t5 - t1;
	    dataptr[7] = t5 + t1;
	}

	inp += 2;
	inpcptr += 8;
	dataptr += 8;
    }

    dataptr = block;
    for (i=0; i<8; i++) {
	t4 = dataptr[0] + 128; /* Add back DC offset */
	t5 = dataptr[8];
	t0 = t4 - t5;
	t1 = t4 + t5;

	t4 = dataptr[16];
	t5 = dataptr[24];
	t2 = t0 - t4;
	t3 = t0 + t4;
	t4 = t1 - t5;
	t5 = t1 + t5;

	t0 = dataptr[32];
	t1 = dataptr[40];
	yp[0] = UCLIMIT(t2 - t0);
	yp[width] = UCLIMIT(t2 + t0);
	yp += width2;
	yp[0] = UCLIMIT(t3 - t1);
	yp[width] = UCLIMIT(t3 + t1);
	yp += width2;

	t0 = dataptr[48];
	t1 = dataptr[56];
	yp[0] = UCLIMIT(t4 - t0);
	yp[width] = UCLIMIT(t4 + t0);
	yp += width2;
	yp[0] = UCLIMIT(t5 - t1);
	yp[width] = UCLIMIT(t5 + t1);
	yp -= width6;

	yp++;
	dataptr++;
    }

    if (up) {
	int uswitch = 1;
	signed char* p = (signed char*)up++;
	dataptr = block;
	for (i=0; i<8; i++) {
	    if ((inp[0] << 16) == 0) {
		t2 = t3 = SIGN_EXTEND(inpcptr[0]);
		t4 = t5 = SIGN_EXTEND(inpcptr[1]);
	    } else {
		t0 = SIGN_EXTEND(inpcptr[0]);
		t1 = SIGN_EXTEND(inpcptr[1]);

		t4 = SIGN_EXTEND(inpcptr[2]);
		t5 = SIGN_EXTEND(inpcptr[3]);
		t2 = t0 - t4;
		t3 = t0 + t4;
		t4 = t1 - t5;
		t5 = t1 + t5;
	    }

	    if (inp[1] == 0) {
		dataptr[0] = dataptr[2] = t2;
		dataptr[4] = dataptr[6] = t3;
		dataptr[1] = dataptr[3] = t4;
		dataptr[5] = dataptr[7] = t5;
	    } else {
		t0 = SIGN_EXTEND(inpcptr[4]);
		t1 = SIGN_EXTEND(inpcptr[5]);
		dataptr[0] = t2 - t0;
		dataptr[2] = t2 + t0;
		dataptr[4] = t3 - t1;
		dataptr[6] = t3 + t1;

		t0 = SIGN_EXTEND(inpcptr[6]);
		t1 = SIGN_EXTEND(inpcptr[7]);
		dataptr[1] = t4 - t0;
		dataptr[3] = t4 + t0;
		dataptr[5] = t5 - t1;
		dataptr[7] = t5 + t1;
	    }

	    inp += 2;
	    inpcptr += 8;
	    dataptr += 8;
	}

	dataptr = block;
	for (i=0; i<8; i++) {
	    t4 = dataptr[0];
	    t5 = dataptr[8];
	    t0 = t4 - t5;
	    t1 = t4 + t5;

	    t4 = dataptr[16];
	    t5 = dataptr[24];
	    t2 = t0 - t4;
	    t3 = t0 + t4;
	    t4 = t1 - t5;
	    t5 = t1 + t5;

	    t0 = dataptr[32];
	    t1 = dataptr[40];

	    *p = SCLIMIT(t2 - t0) + 0x80;
	    p += width / 2;
	    *p = SCLIMIT(t2 + t0) + 0x80;
	    p += width / 2;
	    *p = SCLIMIT(t3 - t1) + 0x80;
	    p += width / 2;
	    *p = SCLIMIT(t3 + t1) + 0x80;
	    p += width / 2;

	    t0 = dataptr[48];
	    t1 = dataptr[56];
	    *p = SCLIMIT(t4 - t0) + 0x80;
	    p += width / 2;
	    *p = SCLIMIT(t4 + t0) + 0x80;
	    p += width / 2;
	    *p = SCLIMIT(t5 - t1) + 0x80;
	    p += width / 2;
	    *p = SCLIMIT(t5 + t1) + 0x80;
	    p += width / 2;

	    if (uswitch) {
		    uswitch = 0;
		    p = (signed char*)vp++;
	    } else {
		    uswitch = 1;
		    p = (signed char*)up++;
	    }
	    dataptr++;
	}
    }
}

void NvTranscoder::NVDCT_RevTransform(u_int* inp, u_char *yp,
				   char* up, char* vp, int width)
{
    int i, t, *dataptr;
    int a0, a1, a2, a3, b0, b1, b2, b3, c0, c1, c2, c3;
    int8_t *inpcptr=(int8_t *)inp;
    static int block[64];

    dataptr = block;
    for (i=0; i<8; i++) {
	if ((inp[0]|inp[1]) == 0) {
	    dataptr[0] = dataptr[1] = dataptr[2] = dataptr[3] =
		dataptr[4] = dataptr[5] = dataptr[6] = dataptr[7] = 0;
	} else {
	    b0 = inpcptr[0] << 4;
	    b1 = inpcptr[4] << 4;
	    b2 = inpcptr[2] << 4;
	    b3 = inpcptr[6] << 4;

	    a0 = (362 * (b0+b1)) >> 9;
	    a1 = (362 * (b0-b1)) >> 9;
	    a2 = (196*b2 - 473*b3) >> 9;
	    a3 = (473*b2 + 196*b3) >> 9;

	    b0 = a0+a3;
	    b1 = a1+a2;
	    b2 = a1-a2;
	    b3 = a0-a3;

	    a0 = inpcptr[1] << 4;
	    a1 = inpcptr[3] << 4;
	    a2 = inpcptr[5] << 4;
	    a3 = inpcptr[7] << 4;

	    c0 = (100*a0 - 502*a3) >> 9;
	    c1 = (426*a2 - 284*a1) >> 9;
	    c2 = (426*a1 + 284*a2) >> 9;
	    c3 = (502*a0 + 100*a3) >> 9;

	    a0 = c0+c1;
	    a1 = c0-c1;
	    a2 = c3-c2;
	    a3 = c3+c2;

	    c0 = a0;
	    c1 = (362 * (a2-a1)) >> 9;
	    c2 = (362 * (a1+a2)) >> 9;
	    c3 = a3;

	    dataptr[0] = b0+c3;
	    dataptr[1] = b1+c2;
	    dataptr[2] = b2+c1;
	    dataptr[3] = b3+c0;
	    dataptr[4] = b3-c0;
	    dataptr[5] = b2-c1;
	    dataptr[6] = b1-c2;
	    dataptr[7] = b0-c3;
	}

	inp += 2;
	inpcptr += 8;
	dataptr += 8;
    }

    dataptr = block;
    for (i=0; i<8; i++) {
	b0 = dataptr[0]+1448; /* Add back DC offset */
	b1 = dataptr[32];
	b2 = dataptr[16];
	b3 = dataptr[48];

	a0 = (362 * (b0+b1)) >> 9;
	a1 = (362 * (b0-b1)) >> 9;
	a2 = (196*b2 - 473*b3) >> 9;
	a3 = (473*b2 + 196*b3) >> 9;

	b0 = a0+a3;
	b1 = a1+a2;
	b2 = a1-a2;
	b3 = a0-a3;

	a0 = dataptr[8];
	a1 = dataptr[24];
	a2 = dataptr[40];
	a3 = dataptr[56];

	c0 = (100*a0 - 502*a3) >> 9;
	c1 = (426*a2 - 284*a1) >> 9;
	c2 = (426*a1 + 284*a2) >> 9;
	c3 = (502*a0 + 100*a3) >> 9;

	a0 = c0+c1;
	a1 = c0-c1;
	a2 = c3-c2;
	a3 = c3+c2;

	c0 = a0;
	c1 = (362 * (a2-a1)) >> 9;
	c2 = (362 * (a2+a1)) >> 9;
	c3 = a3;

	yp[0]     = UCLIMIT((b0+c3+4) >> 3);
	yp[width] = UCLIMIT((b1+c2+4) >> 3);
	yp += 2*width;

	yp[0]     = UCLIMIT((b2+c1+4) >> 3);
	yp[width] = UCLIMIT((b3+c0+4) >> 3);
	yp += 2*width;

	yp[0]     = UCLIMIT((b3-c0+4) >> 3);
	yp[width] = UCLIMIT((b2-c1+4) >> 3);
	yp += 2*width;

	yp[0]     = UCLIMIT((b1-c2+4) >> 3);
	yp[width] = UCLIMIT((b0-c3+4) >> 3);
	yp -= 6*width;

	yp++;
	dataptr++;
    }

    if (up) {
	u_int8_t uvblk[64];
	u_int8_t* uvp = uvblk;
#define width 8
	dataptr = block;
	for (i=0; i<8; i++) {
	    if ((inp[0]|inp[1]) == 0) {
		dataptr[0] = dataptr[1] = dataptr[2] = dataptr[3] =
		    dataptr[4] = dataptr[5] = dataptr[6] = dataptr[7] = 0;
	    } else {
		b0 = inpcptr[0] << 4;
		b2 = inpcptr[2] << 4;
		b1 = inpcptr[4] << 4;
		b3 = inpcptr[6] << 4;

		a0 = (362 * (b0+b1)) >> 9;
		a1 = (362 * (b0-b1)) >> 9;
		a2 = (196*b2 - 473*b3) >> 9;
		a3 = (473*b2 + 196*b3) >> 9;

		dataptr[0] = a0+a3;
		dataptr[2] = a1+a2;
		dataptr[4] = a1-a2;
		dataptr[6] = a0-a3;

		b0 = inpcptr[1] << 4;
		b2 = inpcptr[3] << 4;
		b1 = inpcptr[5] << 4;
		b3 = inpcptr[7] << 4;

		a0 = (362 * (b0+b1)) >> 9;
		a1 = (362 * (b0-b1)) >> 9;
		a2 = (196*b2 - 473*b3) >> 9;
		a3 = (473*b2 + 196*b3) >> 9;

		dataptr[1] = a0+a3;
		dataptr[3] = a1+a2;
		dataptr[5] = a1-a2;
		dataptr[7] = a0-a3;
	    }

	    inp += 2;
	    inpcptr += 8;
	    dataptr += 8;
	}

	dataptr = block;
	for (i=0; i<8; i++) {
	    b0 = dataptr[0];
	    b1 = dataptr[32];
	    b2 = dataptr[16];
	    b3 = dataptr[48];

	    a0 = (362 * (b0+b1)) >> 9;
	    a1 = (362 * (b0-b1)) >> 9;
	    a2 = (196*b2 - 473*b3) >> 9;
	    a3 = (473*b2 + 196*b3) >> 9;

	    b0 = a0+a3;
	    b1 = a1+a2;
	    b2 = a1-a2;
	    b3 = a0-a3;

	    a0 = dataptr[8];
	    a1 = dataptr[24];
	    a2 = dataptr[40];
	    a3 = dataptr[56];

	    c0 = (100*a0 - 502*a3) >> 9;
	    c1 = (426*a2 - 284*a1) >> 9;
	    c2 = (426*a1 + 284*a2) >> 9;
	    c3 = (502*a0 + 100*a3) >> 9;

	    a0 = c0+c1;
	    a1 = c0-c1;
	    a2 = c3-c2;
	    a3 = c3+c2;

	    c0 = a0;
	    c1 = (362 * (a2-a1)) >> 9;
	    c2 = (362 * (a2+a1)) >> 9;
	    c3 = a3;

	    uvp[0]     = SCLIMIT((b0+c3+4) >> 3);
	    uvp[width] = SCLIMIT((b1+c2+4) >> 3);
	    uvp += 2*width;

	    uvp[0]     = SCLIMIT((b2+c1+4) >> 3);
	    uvp[width] = SCLIMIT((b3+c0+4) >> 3);
	    uvp += 2*width;

	    uvp[0]     = SCLIMIT((b3-c0+4) >> 3);
	    uvp[width] = SCLIMIT((b2-c1+4) >> 3);
	    uvp += 2*width;

	    uvp[0]     = SCLIMIT((b1-c2+4) >> 3);
	    uvp[width] = SCLIMIT((b0-c3+4) >> 3);
	    uvp -= 6*width;

	    uvp++;
	    dataptr++;
	}
#undef width
	uvp = uvblk;
	for (i = 0; i < 8; ++i) {
		up[0] = uvp[0] + 0x80;
		vp[0] = uvp[1] + 0x80;
		up[1] = uvp[2] + 0x80;
		vp[1] = uvp[3] + 0x80;
		up[2] = uvp[4] + 0x80;
		vp[2] = uvp[5] + 0x80;
		up[3] = uvp[6] + 0x80;
		vp[3] = uvp[7] + 0x80;
		up += width >> 1;
		vp += width >> 1;
		uvp += 8;
	}
    }
}

#define VIDCODE_COLORFLAG 0x8000
#define VIDCODE_WIDTHMASK 0x0fff

const u_char* NvTranscoder::decode_run(const u_char* data, const u_char* end,
				    int color, int mark)
{
	if (end - data < 3)
		return (end);

	int w0, w;
	w0 = w = *data++;
	int x0 = *data++;
	int y0 = *data++;

	margin& m = margin_;
	if ((x0+w > inw_/8) || (y0 >= inh_/8))
		return (end);

	/* Remember original coords */
	int y = y0;
	int x = x0;

	/* Skip margins */
	y0 += m.top;
	x0 += m.left;

	/* Adjust for crops */
	y0 -= topcrop_;
	x0 -= lcrop_;

	int offset = y0 * outw_;
#ifdef notdef
	u_char* ts = &rvts_[(offset >> 3) + x0];
#endif
	u_char* crv = &crvec_[(offset >> 3) + x0];

	offset = (offset << 3) + (x0 << 3);
	u_char* yp = &frm_[offset];
	int s = outw_ * outh_;/* FIXME */
	u_char* up = frm_ + s + (offset >> 1);
	u_char* vp = up + (s >> 1);
	while (w-- > 0) {
		if (data >= end) {
			return (end);
		}
		static u_int block[32];
		int lim=color? 128 : 64;
		for (int i=0; i<lim; ) {
			int run = *data++;
			int j = run >> 6;
			int k = run & 0x3f;
			if (i+j+k > lim) {
				return (end);
			}
			char* blkp = (char*)block;
			while (j--) blkp[i++] = (char) *data++;
			while (k--) blkp[i++] = 0;
		}
		if (y >= topcrop_ && y < botcrop_ && x >= lcrop_ &&
		    x < rcrop_) {
			if (use_dct_)
				NVDCT_RevTransform(block, yp, (char*)up,
						   (char*)vp, outw_);
			else
				VidTransform_Rev(block, yp, (char*)up,
						 (char*)vp, outw_);
			*crv = mark;
		}

		crv += 1;
		yp += 8;
		x++;
		if (up) {
			up += 4;
			vp += 4;
		}
#ifdef notdef
		*ts++ = now_;
#endif
	}
	return (data);
}

void NvTranscoder::process_hdr(const nvhdr* ph)
{
	int reconfig = 0;

	int h = ntohs(ph->height);
	int w = ntohs(ph->width);
	color_ = w >> 15;
	w &= 0xfff;
	int fmt = h >> 12;
	h &= 0xfff;
	if (fmt == 0)
		use_dct_ = 0;
	else if (fmt == 1)
		use_dct_ = 1;
#ifdef notyet
	if (color != color_)
		/*FIXME should do this by invoking tcl command */
		setcolor(color);
#endif
	if (w != inw_ || h != inh_) {
		if ((unsigned)w > 1000 || (unsigned)h > 1000) {
			/*FIXME*/
			w = 320;
			h = 240;
		}
		inw_ = w;
		inh_ = h;
		encoder_->framesize(inw_, inh_, outw_, outh_);
		reconfig = 1;
	}
	if (reconfig)
		configure();

}

int NvTranscoder::recv(const rtphdr* rh, const u_char* bp, int cc,
		       Transmitter& tx)
{
	const nvhdr* ph = (nvhdr*)(rh + 1);
	process_hdr(ph);

	bp += sizeof(nvhdr);
	cc -= sizeof(nvhdr);
	const u_char* end = bp + cc;
	while (bp < end)
		bp = decode_run(bp, end, color_, CR_SEND|CR_MOTION);

	if (!(ntohs(rh->rh_flags) & RTP_M))
		return (0);
	/*
	 * Have a frame.
	 */
	if (tx.bps() == 0)
		return (1);

	double now = gettimeofday();
	if (fc_ <= now) {
		/* If we have fallen behind (>200ms), re-sync. */
		if (now - fc_ > 200000.)
			fc_ = now;
		VideoFrame vf(ntohl(rh->rh_ts), frm_, outw_, outh_, 422,
			      crvec_);
		int ecc = encoder_->consume(&vf, tx);
		age_blocks();
		double bits = 8 * double(ecc);
		fc_ += 1e6 * bits / tx.bps();
	}
	return (1);
}

#ifdef INFOPAD
void NvIpadVQTranscoder::configure(const u_char* p, Network* n)
{
	NvTranscoder::configure(p, n);

	IpadChannel* c = (IpadChannel*)n;
	encoder_ = new IpadVQEncoder(c, outw_, outh_, 422, 0);
}

void NvIpadVQTranscoder::resize(int w, int h)
{
	outw_ = inw_ = w;
	outh_ = inh_ = h;
	/* FIXME */
	u_int bps = encoder_->bps();
	delete encoder_;
	encoder_ = new IpadVQEncoder((IpadChannel*)net_, outw_, outh_, 422, 0);
	encoder_->bps(bps);
}

int NvIpadVQTranscoder::recv(const rtphdr* rh, const u_char* bp, int cc, int)
{
	const nvhdr* ph = (nvhdr*)(rh + 1);
	int reconfig = process_hdr(ph);
	if (reconfig)
		reconfigure();

	bp += sizeof(nvhdr);
	cc -= sizeof(nvhdr);
	const u_char* end = bp + cc;
	while (bp < end)
		bp = decode_run(bp, end, color_, CR_SEND|CR_MOTION);

	if (!(ntohs(rh->rh_flags) & RTP_M))
		return (0);
	/*
	 * Have a frame.
	 */
	if (bps() == 0)
		return (1);
	double now = gettimeofday();
	if (fc_ <= now) {
		/* If we have fallen behind (>200ms), re-sync. */
		if (now - fc_ > 200000.)
			fc_ = now;
		int ecc = encoder()->encode(frm_, outw_, outh_, 0);
		age_blocks();
		double bits = 8 * double(ecc);
		obits_ += int(bits);
		ofrms_ += 1;
		opkts_ = encoder()->np();
		fc_ += 1e6 * bits / double(bps());
	}
	return (1);
}
#endif /* INFOPAD */

