/*
 * module-jpeg-size-recorder.cc --
 *
 *      This file contains the definition and methods for the JpegSizeRecorder
 *      class.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "rep-decoder.h"
#include "bsd-endian.h"
#include "tclcl.h"
#include "jpeg/jpeg.h"
#include "renderer.h"
#include "pktbuf.h"
#include "vidreps.h"

/*
 * The JpegSizeRecorder class is used as a measurment tool to
 * record the size of JPEG frame data. This size includes all
 * RTP and JPEG payload specific headers. The packets are released
 * immediately upon reception and no decoding or other special
 * handling of the packets is done. The timestamp and size of
 * each JPEG frame is printed to stderr. Partially completed
 * frames are not reported.
 */

class JpegSizeRecorder : public PacketHandler {
public:
  JpegSizeRecorder() : PacketHandler(sizeof(jpeghdr)), reset_state_(1) {};

  virtual void recv(pktbuf *pb);

  int reset_state_;
  int cur_size_;
  u_int32_t last_ts_;
  u_int16_t last_seq_no_;
  int q_;
};

void
JpegSizeRecorder::recv(pktbuf *pb)
{
  rtphdr *rh = (rtphdr*)pb->dp;
  jpeghdr *jh = (jpeghdr *) (rh+1);

  int q = jh->q;

  u_int32_t ts = ntohl(rh->rh_ts);
  u_int16_t sn = ntohs(rh->rh_seqno);

  int len = pb->len;

  if (reset_state_ == 1) {
    cur_size_ = 0;
    last_ts_ = ts;
    last_seq_no_ = sn;
    reset_state_ = 2;
    goto forward;
  } else if (reset_state_ == 2) {
    if (last_ts_ == ts) {
      last_seq_no_ = sn;
      goto forward;
    }
    if (last_seq_no_ != sn - 1) {
      cur_size_ = 0;
      last_ts_ = ts;
      last_seq_no_ = sn;
      reset_state_ = 2;
      goto forward;
    }

    cur_size_ = len;
    last_ts_ = ts;
    last_seq_no_ = sn;
    q_ = q;
    reset_state_ = 0;
    goto forward;

  } else {
    if (last_ts_ == ts) {
      if (last_seq_no_ == sn - 1) {
	cur_size_ += len;
	last_seq_no_ = sn;
	goto forward;
      } else {
	cur_size_ = 0;
	last_ts_ = ts;
	last_seq_no_ = sn;
	reset_state_ = 2;
	goto forward;
      }
    } else {
      if (last_seq_no_ == sn - 1) {
	fprintf(stderr, "%u %d %d\n", last_ts_, q_, cur_size_);

	cur_size_ = len;
	last_ts_ = ts;
	last_seq_no_ = sn;
	q_ = q;
	reset_state_ = 0;
	goto forward;
      } else {
	cur_size_ = 0;
	last_ts_ = ts;
	last_seq_no_ = sn;
	reset_state_ = 2;
	goto forward;
      }
    }
  }

forward:
  if (target_ == 0) {
      pb->release();
  } else {
    target_->recv(pb);
  }
}

static class JpegSizeRecorderClass : public TclClass {
public:
  JpegSizeRecorderClass() : TclClass("Module/VideoDecoder/JPEGSizeRecorder") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new JpegSizeRecorder());
  }
} swd_jpeg_size_recorder_;

