# tcl-expand.tcl --
#
#       This script expands the header file and main file for a Mash tool.  It
#       substitutes "VERSION" and "TOOLNAME" in the header file and expands
#       source and import commands in the main file.
#
#       The command-line syntax is:
#           tclsh tcl-expand.tcl [-v] headfile mainfile toolname
#
#       The verbose option (-v) prints what is being expanded and imported.
#
#       If you don't want a source or import command expanded, you can avoid
#       expansion by using "\source" and "\import".
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl-expand.tcl,v 1.28 2002/02/03 04:41:07 lim Exp $


# main --
#
#       Expand header file and main file for a Mash tool.
#
# Arguments:
#       argv    Command-line arguments.
#
# Results:
#       Prints expanded Tcl files to standard output.

proc main {} {
    global argv
    global Verbose
    global IgnoreImports
    global versionFilename

    # Process command-line options.

    set versionFilename "VERSION"
    set Verbose 0
    set args {}
    set state "normal"
    foreach arg $argv {
        if {($arg == {-v}) || ($arg == {--verbose})} {
            set Verbose 1
        } elseif {($arg == {-h}) || ($arg == {--help})} {
            PrintUsage
        } elseif {($arg == {-V}) || ($arg == {--version})} {
	    # we assume the next arg is the file of the version 
	    set state "version"
        } else {
	    if {$state == "version"} {
		set versionFilename $arg
		set state "normal"
	    } else {
		lappend args $arg
	    }
        }
    }

    if {[llength $args] == 3} {

        # The output of this script is always redirected to a file so we use
        # full buffering.  On Windows, this increases performance by one to two
        # orders of magnitude.

        fconfigure stdout -buffering full

        set IgnoreImports 0
        if {![ReadImportTables]} {
            puts stderr {ignoring import commands...}
            set IgnoreImports 1
        }

        ExpandHead [lindex $args 0] [lindex $args 2]
        ExpandFile [lindex $args 1]
    } else {
        PrintUsage
    }

    return
}


# PrintUsage --
#
#       Print brief description of command-line syntax and exit script.
#
# Arguments:
#       none
#
# Results:
#       The script is exited.

proc PrintUsage {} {
    global argv0

    puts stderr "usage: tclsh $argv0 \[-v\] \[-V versionfile\] headfile mainfile toolname"
    exit 1
}


# ExpandHead --
#
#       Expand header file by replacing "VERSION" with the current version
#       number and replacing "TOOLNAME" with the name of the tool.
#
# Arguments:
#       headFileName    Name of header file.
#       toolName        Name of tool.
#
# Results:
#       Prints expanded header file to standard output.

proc ExpandHead {headFileName toolName} {

    # Get current version number from VERSION file.

    global versionFilename
    if {[catch {set versionFile [open $versionFilename {RDONLY}]} errorMessage]} {
        puts stderr "*** error: $errorMessage"
        exit 1
    }
    gets $versionFile version
    close $versionFile

    if {[catch {set headFile [open $headFileName {RDONLY}]} errorMessage]} {
        puts stderr "*** error: $errorMessage"
        exit 1
    }
    while {[gets $headFile line] >= 0} {
        regsub {VERSION} $line $version line
        regsub {TOOLNAME} $line $toolName line
        puts $line
    }
    close $headFile

    return
}


# ExpandFile --
#
#       Expand file by replacing source and import commands with the contents
#       of the files being sourced or imported.
#
# Arguments:
#       fileName    Name of file.
#
# Results:
#       Prints expanded file to standard output.

proc ExpandFile {fileName} {
    global Verbose
    global Expanded

    if {[IsHttpUrl $fileName]} {

        # It's doubtful if we'll ever implement this feature.  --LL

        puts stderr "*** error: $fileName: remote file retrieval not supported\
                     yet"
        exit 1
    }

    # Skip the file if it's already been expanded.

    if {[info exists Expanded($fileName)]} {
        return
    }

    if {$Verbose} {
        puts stderr "expanding {$fileName}"
    }
    set Expanded($fileName) 1

    if {$fileName == {stdin}} {
        set file stdin
    } else {
        if {[catch {set file [open $fileName {RDONLY}]} errorMessage]} {
            puts stderr "*** error: $errorMessage"
            exit 1
        }
    }

    while {[gets $file line] >= 0} {

        # Trim trailing whitespace and trailing comments on all lines.

        regsub "\[ \t\]*;\[ \t\]*#.*" [string trimright $line] {} line

        # Trim leading whitespace and get first 6 characters for comparison.

        set trimmedLine [string trimleft $line]
        set keyword [string range $trimmedLine 0 5]
	set after [string index $trimmedLine 6]

        # Expansion can be avoided by putting a backslash before "source" or
        # "import".

        if {(($keyword == {source}) || ($keyword == {import})) && [string is space $after]} {

            # Collapse whitespace before splitting line, so that each extra
            # space does not produce a {} list element.

            regsub -all { +} $trimmedLine { } trimmedLine
            set words [split $trimmedLine]

            if {($keyword == {source}) && ([llength $words] == 2)} {
                ExpandFile [lindex $words 1]

            } elseif {$keyword == {import}} {
                set classNames [ReadImportCommand $file $words]
                ExpandClasses $classNames

            } else {
                # This wasn't an actual source or import command.

                PrintLine $line
            }
        } else {
            PrintLine $line
        }
    }

    if {$fileName != {stdin}} {
        close $file
    }
    return
}


# ReadImportCommand --
#
#       Read remainder of import command, if necessary.
#
# Arguments:
#       file    Source file open for reading.
#       words   List of words in first line of import command.
#
# Results:
#       Names of classes to import.

proc ReadImportCommand {file words} {
    set classNames [lrange $words 1 end]

    # If line ends with backslash, read next line.

    while {[lindex $classNames end] == "\\"} {
        set classNames [lrange $classNames 0 [expr [llength $classNames] - 2]]
        if {[gets $file line] < 0} {
            break
        }
        set classNames "$classNames $line"
    }

    return $classNames
}


# ExpandClasses --
#
#       Look up classes and expand files needed by classes.
#
# Arguments:
#       classNames  Names of classes to import.
#
# Results:
#       Prints expanded files to standard output.

proc ExpandClasses {classNames} {
    global Verbose
    global IgnoreImports
    global ImportTable

    # Ignore all import commands if an error occurred previously.

    if {$IgnoreImports} {
        return
    }

    foreach className $classNames {
        if {$Verbose} {
            puts stderr "importing $className"
        }

        if {[info exists ImportTable($className)]} {
            foreach file $ImportTable($className) {
                ExpandFile $file
            }
        } else {
            puts stderr "*** warning: $className not in importTable"
        }
    }
}


# PrintLine --
#
#       Print line, if it isn't a blank line or a comment line.
#
# Arguments:
#       line    Line of Tcl source code.
#
# Results:
#       Prints line to standard output.

proc PrintLine {line} {
    if {![regexp "^\[ \t\]*\$" $line] && ![regexp "^\[ \t\]*#" $line]} {
        puts $line
    }
    return
}


# ReadImportTables --
#
#       Read in all import tables.
#
# Arguments:
#       none
#
# Results:
#       Boolean indicating whether an import table was found.

proc ReadImportTables {} {
    set dirs [FindImportDirs]
    if {$dirs == {}} {
        puts stderr "*** error: no importTable in current directory or\
                     TCLCL_IMPORT_DIRS"
        return 0
    }
    foreach dir $dirs {
        ReadImportTable $dir
    }
    return 1
}


# ReadImportTable --
#
#       Read in import table.
#
# Arguments:
#       dir     Directory containing import table.
#
# Results:
#       Stores import table entries in ImportTable array.

proc ReadImportTable {dir} {
    global ImportTable

    set file [open [file join $dir importTable] {RDONLY}]
    while {[gets $file line] >= 0} {
        set className [lindex $line 0]
        set fileName [lindex $line 1]
        set fileName [file join $dir $fileName]

        # Don't add the file if it's already in the ImportTable array.

        if {[info exists ImportTable($className)] &&
            ([lsearch -exact $ImportTable($className) $fileName] != -1)} {
            continue
        }

        # Each element of the ImportTable array is a list of files.

        lappend ImportTable($className) $fileName
    }
    close $file

    return
}


# FindImportDirs --
#
#       Find the TclCL import directories.
#
# Arguments:
#       none
#
# Results:
#       List of TclCL import directories.

proc FindImportDirs {} {
    global env

    if {[file exists importLocation]} {
        set file [open importLocation {RDONLY}]
        while {[gets $file line] >= 0} {
            lappend list $line
        }
        close $file
    } elseif {[info exists env(TCLCL_IMPORT_DIRS)]} {
        set list [split $env(TCLCL_IMPORT_DIRS) :]
    } else {
        set list .
    }

    set dirs {}
    foreach dir $list {
        if {[file readable $dir/importTable]} {
            lappend dirs $dir
        }
    }

    return $dirs
}


# IsHttpUrl --
#
#       Determine whether string is a HTTP URL.
#
# Arguments:
#       str     Possible HTTP URL.
#
# Results:
#       Boolean indicating whether string is a HTTP URL.

proc IsHttpUrl {str} {

    # This regular expression is rather stupid.  It's too lax and it doesn't
    # work if the URL doesn't have a trailing slash.  --LL

    if {[regexp {([^:]+)://([^:/]+)(:([0-9]+))?(/.*)} $str \
            url protocol server dummy port path]} {
        if {[info exists protocol]} {
            return [string match http $protocol]
        } else {
            return 0
        }
    }
    return 0
}


# Start up this script.

main
