# mashlet-publisher.tcl --
#
#       Using an importTable, generate a mashlet per object, and publish each
#       mashlet on a web-server.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/common/mashlet-publisher.tcl,v 1.6 2002/02/03 04:25:43 lim Exp $


#
# Using an importTable, generate a mashlet per object,
# and publish each mashlet on a web-server.
#
Class MashletPublisher

#
# For each object in the importTable in the provided directory
# <i>import_dir</i>, generate and publish a mashlet (named
# <object>.mash) in the provided directory <i>publish_dir</i>.  The user
# is reponsible for cleaning out stale mashlets from this directory and
# making the directory readable & executable.  <i>publish_url</i> is
# expected to be the URL to which the <i>publish_dir</i> is mapped.
#
MashletPublisher public init { import_dir publish_dir publish_url } {
	$self instvar publishDir_  publishURL_
	set publishDir_ $publish_dir
	set publishURL_ $publish_url

	set objects [$self read_import_table $import_dir]
	foreach object $objects {
		$self publish_mashlet $object
	}
}

#
# Create an array of lists called table_.  The array will be indexed using
# the name of a Mash object.  The value of the array at that index will
# be assigned a list of files (paths relative to the provided directory
# <i>dir</i>) in which the Mash object and methods on it are defined
# according to the importTable in the provided directory <i>dir</i>.
# Return a list of all of the objects mapped in the importTable.
#
MashletPublisher instproc read_import_table { dir } {
	$self instvar table_ objects_ import_

	set import_ [new Import]
	$import_ read_dir $dir
	$import_ instvar {table_ importTable_}
	array set table_ [array get importTable_]

	set objects_ ""
	foreach object [array names table_] {
		# Add all the uncommented-out objects in the importTable to a list
		if {[string range $object 0 0] != "#" } {
			lappend objects_ $object
		}
	}

	return $objects_
}

#
# Return a unique pathname for the supplied filename taken from the table_.
#
MashletPublisher private get_uniq_path { filename } {
	return $filename
	$self instvar objects_ table_ magicNumTailEls_ import_

	# Determine the number of tail elements that need to be returned to provide a unique filename
	if { ![info exists magicNumTailEls_] } {
		# accumulate a listing of all files
		foreach object $objects_ {
			if [info exists files] {
				set files [concat $files $table_($object)]
			} else {
				set files $table_($object)
			}
		}

		# eliminate duplicates from the list
		set files [lsort_uniq $files]

		set numTailEls 1
		while 1 {
			foreach file $files {
				# break each element in the list into a prefix and suffix
				set parsedName [file split $file]
				set preStart 0
				set suffEnd [expr [llength $parsedName] - 1]
				set suffStart [expr $suffEnd - $numTailEls + 1]
				set preEnd [expr $suffStart - 1]
				set prefix ""
				foreach pathEl [lrange $parsedName $preStart $preEnd] {
					set prefix [$import_ file join $prefix $pathEl]
				}
				lappend prefixes $prefix
				set suffix ""
				foreach pathEl [lrange $parsedName $suffStart $suffEnd] {
					set suffix [$import_ file join $suffix $pathEl]
				}
				lappend suffixes $suffix
			}
			# when all of the suffixes are unique, we have found the magic number
			if { [llength [lsort_uniq $suffixes]] == [llength $suffixes] } {
				break
			} else {
				incr numTailEls
				unset prefixes
				unset suffixes
			}
		}
		set magicNumTailEls_ $numTailEls
	}

	set parsedName [file split $filename]
	set suffEnd [expr [llength $parsedName] - 1]
	set suffStart [expr $suffEnd - $magicNumTailEls_ + 1]
	set suffix ""
	foreach pathEl [lrange $parsedName $suffStart $suffEnd] {
		set suffix [$import_ file join $suffix $pathEl]
	}
	return $suffix
}

#
# Sorts the provided list, eliminating duplicate elements.
#
proc lsort_uniq { list } {
	foreach element $list {
		set dummy_array($element) blah
	}
	return [array names dummy_array]
}

#
# In the publishDir_, publish a mashlet for the provided <i>object</i>
# by publishing the tcl files in which methods or defns for this object
# are found and then sourcing these tcl files in a file called
# <object>.mash
#
MashletPublisher private publish_mashlet { object } {
	$self instvar table_ writtento_ publishDir_ publishURL_

	# error msg if object was not found in importTable
	if ![info exist table_($object)] {
		puts stderr "*** warning: can't find import entry for $object"
		return
	}

	# assign a filename for mashlet to be written to
	$self instvar import_
	set htmlable [$import_ class_to_file $object]
	set mashlet [$import_ file join $publishDir_ $htmlable.mash]
	puts stdout "Publishing $mashlet"

	# publish the files in which methods or defns of this object are found
	# and source them in a published mashlet
	foreach fileIn $table_($object) {
		# if this is the first time I am appending to this mashlet,
		# clean out any old stuff beforehand
		if {[file exists $mashlet] && ![info exists writtento_($mashlet)]} {
			file delete $mashlet
		}

		# copy $fileIn contents into a published file, $fileOut
		if [catch {open $fileIn r} fileInId] {
			puts stderr "Cannot open $fileIn: $fileInId"
			continue
		} else {
			set relativeFileName [$self get_uniq_path $fileIn]
			set fileOut [$import_ file join $publishDir_ $relativeFileName]
			set URL [$import_ file join $publishURL_ $relativeFileName]
			#puts "trying to copy $fileIn to $fileOut"

			# create the mirror directory within the publishDir_ if it doesn't already exist
			set fileOutDir [$import_ file dirname $fileOut]
			if { ![file exists $fileOutDir] } {
				#puts stdout "Creating directory: $fileOutDir"
				file mkdir $fileOutDir
			}
			# make sure the directory path from the publishDir_ to fileOut is executable
			set path $fileOut
			while { [set dir [$import_ file dirname $path]] != $publishDir_ } {
				#puts stdout "Making $dir executable"
				file attributes $dir -permissions 0775
				set path $dir
			}

			#puts "trying to open $fileOutId"
			if [catch {open $fileOut w 0664} fileOutId] {
				puts stderr "Cannot open $fileOut: $fileOutId"
				close $fileInId
				continue
			}
			#puts "done trying to open $fileOutId"
			file attributes $fileOut -permissions 0664
			# puts stdout "Copying $fileIn to $fileOut"
			puts -nonewline $fileOutId [read $fileInId]
			close $fileInId
			close $fileOutId
		}

		# add source command to mashlet
		if [catch {open $mashlet a 0664} mashletFileId] {
			puts stderr "Cannot open $mashlet: $mashletFileId"
			continue
		} else {
			file attributes $mashlet -permissions 0664
			# append a source command for the current file
			# puts stdout "Sourcing $URL in $mashlet"
			puts $mashletFileId "source $URL"
			# set flag to indicate that I have dumped to this mashlet
			set writtento_($mashlet) 1
			close $mashletFileId
		}
	}
}

