# application-dc.tcl --
#
#       Base of the DC app.  Contains a service manager for talking to the
#       SDS.  Creates UI and parses command line options.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import RTPApplication
import Observer   
import WidgetResourceInitializer
import FontInitializer
import DcBroadcastAgent
import DcStudioAgent
import CDcUI
import CServiceManager
import CService
import CDcSession
import DcApi

#-----------------------------------------------------------------------------
# Class:
#   DcObserver
#
# Description:
#   This is an observer that gets attached to the StudioAgent, and
#   get notified when events happen.   In particular, we are interested
#   in format change events (trigger_format{}).  
#
# Members:
#   ui_ -- the CDcUI object.
#   agent_ -- the DcStudioAgent we are attached to.
#-----------------------------------------------------------------------------
Class DcObserver -superclass Observer

#-----------------------------------------------------------------------------
# Method:
#   DcObserver init
#
# Arguments:
#   ui -- the CDcUI object.
#   agent -- the DcStudioAgent we are attached to.
#
# Description:
#   Initialize members.
#-----------------------------------------------------------------------------
DcObserver instproc init {ui agent} {
    $self next
    $self instvar agent_ ui_

    set agent_ $agent
    set ui_ $ui
}


#-----------------------------------------------------------------------------
# Method:
#   DcObserver trigger_format
#   DcObserver decoder_changed
#
# Arguments:
#   src -- the Source/RTP object whose format has changed.
#
# Description:
#   1. Tell the agent to "reactivate" the $src (removes old decoder, 
#   creates new decoder, set the source of the decoder). 2. Tell the
#   UI to reattach the source (set the targets of the decoder to 
#   various renderers).
#
#-----------------------------------------------------------------------------
DcObserver instproc trigger_format {src} {
    $self instvar agent_ ui_
    $agent_ reactivate $src
    $ui_ reattach $src 
}

DcObserver instproc decoder_changed {src} {
    $self instvar agent_ ui_
    $agent_ reactivate $src
    $ui_ reattach $src 
}


#-----------------------------------------------------------------------------
# Class:
#   DcApplication
#
# Description:
#   Main application class that is responsible for creating everything
#   else.
#
# Members:
#   sds_session_  -- The session for SDS protocol.
#   bcast_agent_  -- The VideoAgent responsible for broadcast session.
#   studio_agent_ -- The VideoAgent responsible for studio session.
#   service_mgr_  -- The service manager object for managing connections
#                    to services.
#   ui_           -- The DcUI object that encapsulates all the UI 
#                    components of this application.
#                
#-----------------------------------------------------------------------------
Class DcApplication -superclass RTPApplication

DcApplication instproc init { framePath argv } {
    $self next "dc"

    # Initialization of variables and resources.
    set options [$self options]
    $self init_arguments $options
    $self init_resources $options
    $options load_preferences "rtp dc"
    set argv [$options parse_args $argv]
    $self check_rtp_sdes

    # create the discovery service session
    $self instvar sds_session_
    set addr [$options get_option optServiceAddress]
    set port [$options get_option optServicePort]
    set ttl  [$options get_option optServiceTTL]
    set sds_session_ [new CDcSession $addr $port $port $ttl]
    $sds_session_ StartQuery

    # create the broadcast agent
    $self instvar bcast_agent_
    set addr [$options get_option optBroadcastAddress]
    set port [$options get_option optBroadcastPort]
    set ttl  [$options get_option optBroadcastTTL]
    set bcast_agent_ [new DcBroadcastAgent $self $addr $port $ttl]
    puts "BroadCast to $addr/$port/$ttl"

    # create the Studio agent
    $self instvar studio_agent_
    set addr [$options get_option optMStudioAddress]
    set port [$options get_option optMStudioPort]
    set ttl  [$options get_option optMStudioTTL]
    set studio_agent_ [new DcStudioAgent $self $addr $port $ttl \
	    [$bcast_agent_ get_switcher]]

    # create the service manager
    $self instvar service_mgr_
    set service_mgr_ [new CServiceManager "ClientApp" "DC" 0]
    $service_mgr_ Attach $self

    # the ui stuff here
    $self instvar ui_
    set ui_ [new CDcUI $self]
    $ui_ build_ui $framePath

    $studio_agent_ attach [new DcObserver $ui_ $studio_agent_]

    # by this time the UI has been built, so we can init the API object
    $self instvar dcapi_
    set apiPort [$options get_option apiPort]
    set dcapi_ [new DcApi $self $ui_ $apiPort]
}

DcApplication instproc init_arguments { options } {
    $options register_option -t defaultTTL
    $options register_option -r remoteHosts

    # for Media Studio Session
    $options register_option -ma optMStudioAddress
    $options register_option -mp optMStudioPort
    $options register_option -mt optMStudioTTL

    # for Broadcast Session
    $options register_option -ba optBroadcastAddress
    $options register_option -bp optBroadcastPort
    $options register_option -bt optBroadcastTTL

    # for the service discovery service
    $options register_option -sa optServiceAddress
    $options register_option -sp optServicePort
    $options register_option -st optServiceTTL

    # for remote communcation
    $options register_option -cp optComPort

    # for the RPC Api server
    $options register_option -apip apiPort
}

DcApplication instproc init_resources { options } {

    new WidgetResourceInitializer
    new FontInitializer [$self options]

    # for Media Studio Session
    $options add_default optMStudioAddress "224.1.2.3"
    $options add_default optMStudioPort "12344"
    $options add_default optMStudioTTL "4"

    # for Broadcast Session
    $options add_default optBroadcastAddress "224.1.2.4"
    $options add_default optBroadcastPort "12344"
    $options add_default optBroadcastTTL "8"

    # for the service discovery service
    $options add_default optServiceAddress "224.4.6.8"
    $options add_default optServicePort "12344"
    $options add_default optServiceTTL "16"

    # for remote communcation
    $options add_default optComPort "12344"

    # for the RPC Api server
    $options add_default apiPort "6907"
}


DcApplication instproc destroy { } {
    $self instvar service_mgr_
    delete $service_mgr_ 
}

#---------------------------------------------------------------------------
#
# Method:
#   DcApplication NewConnection 
#
# Arguments:
#   service -- the newly connected service object
#
# Description:
#   This is a callback function for the service manager.
#   It will be called whenever a new connection is made
#   The service will have attributes to it that will
#   indicate which agent object it should go to.
#
#---------------------------------------------------------------------------
DcApplication public NewConnection { service } {
    set szType [$service set m_szType]

    switch -exact -- $szType {
	VideoService {
	    $self instvar ui_
	    set uiThumbnailFrame [$ui_ set m_uiThumbnailFrame]
	    $uiThumbnailFrame NewConnection $service
	}
    }
}


#---------------------------------------------------------------------------
# Method:
#   DcApplication get_studio_agent
#   DcApplication get_bcast_agent
#   DcApplication get_sds_session
#
# Description:
#   Return the studio agent, broadcast agent and SDS session respectively. 
#---------------------------------------------------------------------------
DcApplication instproc get_studio_agent {} {
    $self instvar studio_agent_
    return $studio_agent_
}
DcApplication instproc get_bcast_agent {} {
    $self instvar bcast_agent_
    return $bcast_agent_
}
DcApplication instproc get_sds_session {} {
    $self instvar sds_session_
    return $sds_session_
}


#---------------------------------------------------------------------------
# Method:
#   DcApplication new_service
#
# Description:
#   Call the service manager to create a new service for $addr/$port.  The
#   new service is returned.
#---------------------------------------------------------------------------
DcApplication instproc new_service {addr port} {
    $self instvar service_mgr_
    return [$service_mgr_ NewService $addr $port]
}

#---------------------------------------------------------------------------
# Method:
#   DcApplication get_service_spec
#
# Description:
#   Ask the service manager to return textual information about
#   the current socket.
#---------------------------------------------------------------------------
DcApplication instproc get_service_spec {} {
    $self instvar service_mgr_
    return [$service_mgr_ GetSpec]
}
