# ui-combobox.tcl --
#
#       Defines a combobox widget.  Copies of similar code are present
#       in other parts of mash.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import TextEntry

# Implements a combobox widget that consists of an entry widget
# and a menu that can be used to quickly enter one of several
# defaults.
Class Combobox

Combobox set arrow [image create bitmap -data {
\#define arrow_width 14
\#define arrow_height 14
static char arrow_bits[] = {
   0x00, 0x00, 0xe0, 0x01, 0xe0, 0x01, 0xe0, 0x01, 0xe0, 0x01, 0xfc, 0x0f,
   0xf8, 0x07, 0xf0, 0x03, 0xe0, 0x01, 0xc0, 0x00, 0x00, 0x00, 0xfe, 0x1f,
   0xfe, 0x1f, 0x00, 0x00};
}]

# Creates a new Combobox widget in a Tk window <i>w</i>.
# The initial value in the Combobox is given by <i>value</i>.
# When a new value is entered, the tcl procedure <i>cmd</i> is
# called to validate the new value (as with a TextEntry object).
# All arguments after <i>cmd</i> are default values available in
# the menu.
Combobox public init {w value cmd args} {
    $self instvar win_ entry_
    $class instvar arrow

    set win_ $w
    frame $w
    set entry_ [new TextEntry $cmd $w.entry $value]
    pack $w.entry -side left -fill x -expand yes

    menubutton $w.b -image $arrow -menu $w.b.m -indicatoron no \
	-relief raised -bd 2
    pack $w.b
    menu $w.b.m
    foreach a $args {
	$w.b.m add command -label $a -command "$self set \"$a\""
    }
}

# Destroys this widget.
Combobox public destroy {} {
    $self instvar win_
    destroy $win_
}

# Sets the value in this widget to <i>value</i>
Combobox public set {value} {
#puts "Combobox setting value to $value"
#    $self instvar win_
#    set e $win_.entry
#    $e delete 0 end
#    $e insert 0 $value
    $self instvar entry_
    $entry_ set-value $value
}

# Gets the string that is currently entered in this widget.
Combobox public get {} {
    $self instvar entry_
#puts "Combobox returning [$entry_ entry-value]"
    return [$entry_ entry-value]
}

# Enables this widget so that the value can be changed and the
# menu is active.
Combobox public enable {} {
    $self instvar win_
    $win_.b configure -state normal
    $win_.entry configure -state normal
}

# Disables this widget so that the value cannot be changed either
# by typing or via the menu.
Combobox public disable {} {
    $self instvar win_
    $win_.b configure -state disabled
    $win_.entry configure -state disabled
}
