# dali-subprogram.tcl --
#
#       This file contains the Dali subprogram base class.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# The DaliSubprogram object responds to graph interface communication
# recieved through its recv_graph_comm method and
# sends graph interface communication through the send_graph_comm
# method.

# Object fields
#
#   input_id_list_    : list of input ids
#   output_id_list_   : list of output ids
#   input_info_       : array of info on input_ids. For each input id $i,
#         ($i,spec)       : multicast spec for input data
#         ($i,trigger)    : 0/1 indicating if input is execution trigger
#         ($i,buffertype) : indicates what type of buffer this Dali subprogram
#                           expects the input to be dumped into
#         ($i,buffername) : indicates the name of the buffer object of the
#                           type indicated above
#         ($i,decoder)    : decoder object for input
#   output_id_list_ of output ids
#   output_info_      : array of info on output_ids. For each output id $o,
#         ($o,spec)      : multicast spec of where output goes
#         ($o,buffertype): indicates type of above buffer
#         ($o,encoder)   : name of encoder object.
#         ($o,format)    : format of output
#         ($o,vagent)    : name of video agent used as a transmitter
#         ($o,bp)        : name of buffer pool
#   parameter_id_list_: list of parameter ids
#   parameter_info_   : array on info on parameters. For each parameter $p,
#         ($p,oname)     : name of parameter object.
#   vagent_array_        : array associated by "addr,port" indexes that holds
#                          the name of video agents used by this object.
#   tvector_             : list of pairs (id,ts) indicating that trigger is
#                          waiting for frame with timestamp ts to appear on
#                          on id.

# Object methods
#   init : initialization
#   recv_graph_comm : handles incoming graph interface communication
#   send_graph_comm : handles outgoing graph interface communication
#   set_input_spec : handles changes to input spec
#   set_output_spec : handles changes to output spec
#   set_output_format : handles changes to output format
#   trigger : executes Dali code to produce 1 frame of output
#   set_output_geometry : handles changes to output geometry

import PsvpVideoAgent
import GraphComm

Class DaliSubprogram

DaliSubprogram instproc init {id control_spec} {
    $self next

    $self instvar input_id_list_;
    $self instvar input_info_;
    $self instvar output_id_list_;
    $self instvar output_info_;
    $self instvar parameter_id_list_;
    $self instvar parameter_info_;
    $self instvar vagent_array_;
    $self instvar comm_obj_;
    $self instvar id_;

    set input_id_list_ "";
    set output_id_list_ "";
    set parameter_id_list_ "";

    set id_ $id;

    set control_spec [split $control_spec "/"];
    set ctrl_addr [lindex $control_spec 0];
    set ctrl_port [lindex $control_spec 1];
    set ctrl_ttl [lindex $control_spec 2];

    set comm_obj_ [new GraphComm/DaliSubprogram $self $id_ $ctrl_addr $ctrl_port $ctrl_ttl];
}


DaliSubprogram instproc send_debug_mesg {mesg} {
    $self instvar comm_obj_;

    $comm_obj_ send_debug $mesg
}

DaliSubprogram instproc send_completion_token {} {
    $self instvar comm_obj_;

    $comm_obj_ send_trigger_command "trigger_completion_token";
}

DaliSubprogram instproc set_input_spec {id spec} {
    $self instvar input_info_
    $self instvar vagent_array_;

    # If alread set properly, ignore.

    if {$input_info_($id,spec) == $spec} {
	return;
    }

    # If input already associated with a spec, delete
    # decoder no longer needed.

    if {$input_info_($id,spec) != ""} {
	set decoder $input_info_($id,decoder);
	if {$decoder != ""} {

	    # Redefine src's trigger_sr proc to do nothing

	    set src [$decoder set src_];
	    $src proc trigger_sr {args} {};

	    # Use video agent to delete decoder

	    set vagent [$decoder set agent_];
	    $vagent delete_decoder $decoder;
	    set input_info_($id,decoder) "";
	}
	set input_info_($id,spec) "";
    }

    set spec_split [split $spec "/"];

    set addr [lindex $spec_split 0];
    set port [lindex $spec_split 1];
    set srcid [lindex $spec_split 2];

    # Find video agent handling this addr/port creating
    # one if none exists.

    if {![info exists vagent_array_($addr,$port)]} {
	set vagent_array_($addr,$port) [new PsvpVideoAgent $addr/$port];
    }
    set vagent $vagent_array_($addr,$port);

    # Retrieve source associated with target srcid

    set src [$vagent get_source_by_id $srcid];

    # If no source yet, establish callback with video
    # agent for when source does appear

    if {$src == ""} {
	$vagent set_create_decoder_callback $srcid "$self set_input_spec_cb $id $spec";
    } else {

	# Create appropriate decoder for input source format and
	# target buffer type.

	set fmt_name [$vagent classmap [$src format_name]];

	set btype $input_info_($id,buffertype);

	set bname $input_info_($id,buffername);

	set decoder [new Module/VideoDecoder/${fmt_name}To${btype}];

	if {$decoder == ""} {
	    # No such decoder. Create a NULL decoder and associate that.
	    set decoder [new Module/VideoDecoder/Null];
	} else {
	    $decoder set_frame_buffer $bname;
	    if {$input_info_($id,trigger) != 0} {
		$decoder set_callback "$self trigger";
	    }
	}

	# Instruct video agent to set up decoder as target for source

	$vagent set_src_decoder $src $decoder;

	# Update input info array

	set input_info_($id,decoder) $decoder;

	# Remap sources trigger_sr to call ours with correct associated id
	$src proc trigger_sr {args} "$self trigger_sr $src $id";

    }
    set input_info_($id,spec) $spec;
}

DaliSubprogram instproc set_input_spec_cb {id spec src} {
    $self instvar input_info_
    $self instvar vagent_array_

    # Check to make sure spec is still good.

    if {$input_info_($id,spec) != $spec} {
	puts "$input_info_($id,spec) != $spec !!!";
	return;
    }

    if {$input_info_($id,decoder) != ""} {
	puts "$input_info_($id,decoder) != {} !!!";
	return;
    }

    # Find videoagent and source

    set spec_split [split $spec "/"];

    set addr [lindex $spec_split 0];
    set port [lindex $spec_split 1];
    set srcid [lindex $spec_split 2];

    set vagent $vagent_array_($addr,$port);

    # Setup decoder.

    set fmt_name [$vagent classmap [$src format_name]];

    set btype $input_info_($id,buffertype);

    set bname $input_info_($id,buffername);

    set decoder [new Module/VideoDecoder/${fmt_name}To${btype}];

    if {$decoder == ""} {
	# No such decoder. Create a NULL decoder and associate that.
	set decoder [new Module/VideoDecoder/Null];
    } else {
	$decoder set_frame_buffer $bname;
	if {$input_info_($id,trigger) != 0} {
	    $decoder set_callback "$self trigger";
	}
    }

    # Update input info array

    set input_info_($id,decoder) $decoder;

    # Remap sources trigger_sr to call ours with correct associated id
    $src proc trigger_sr {args} "$self trigger_sr $src $id";

    return $decoder;
}

DaliSubprogram instproc set_output_spec {id spec} {
    $self instvar output_info_
    $self instvar vagent_array_

    # If already set properly, ignore.

    if {$output_info_($id,spec) == $spec} {
	return;
    }

    # If out spec already set, unset it.

    if {$output_info_($id,spec) != ""} {
	set encoder $output_info_($id,encoder);
	if {$encoder != ""} {
	    delete $encoder;
	}
	set bp $output_info_($id,bp);
	if {$bp != ""} {
	    delete $bp;
	}
	set output_info_($id,encoder) "";
	set output_info_($id,spec) "";
	set output_info_($id,vagent) "";
	set output_info_($id,bp) "";
    }

    # Find video agent for addr/port

    set spec_split [split $spec "/"];

    set addr [lindex $spec_split 0];
    set port [lindex $spec_split 1];

    if {![info exists vagent_array_($addr,$port)]} {
	set vagent_array_($addr,$port) [new PsvpVideoAgent $addr/$port];
	[$vagent_array_($addr,$port) get_transmitter] set loopback_ 0;
    }
    set vagent $vagent_array_($addr,$port);

    # Create encoder for buffertype and target format

    set format $output_info_($id,format);
    set btype $output_info_($id,buffertype);

    set encoder [new Module/VideoEncoder/${btype}To${format}];

    # Set video agent transmitter as encoder target

    set bp "";

    if {$encoder != ""} {
	set ssrc [$vagent get_local_srcid];
	set bp [new BufferPool/RTP];
	$bp srcid $ssrc;

	$encoder buffer-pool $bp;
	$encoder mtu [$self get_option mtu];

	global kpatel_debug;

	if {[info exists kpatel_debug]} {
	    set logger [new Module/RTPPktLogger];
	    $logger target [$vagent get_transmitter];
	    $encoder target $logger;
	} else {
	    $encoder target [$vagent get_transmitter];
	}

	# Total hack FIXME
	$vagent local_bandwidth 30000000
    }

    # Update variables

    set output_info_($id,spec) $spec;
    set output_info_($id,encoder) $encoder;
    set output_info_($id,vagent) $vagent;
    set output_info_($id,bp) $bp;
}

DaliSubprogram instproc set_output_format {id format} {
    $self instvar output_info_

    if {$output_info_($id,format) == $format} {
	return;
    }

    set output_info_($id,format) $format;

    if {$output_info_($id,spec) == ""} {
	return;
    }

    if {$output_info_($id,encoder) != ""} {
	delete $output_info_($id,encoder);
    }
    set output_info_($id,encoder) "";

    if {$output_info_($id,bp) != ""} {
	delete $output_info_($id,bp);
    }
    set output_info_($id,bp) "";

    set btype $output_info_($id,buffertype);

    set encoder [new Module/VideoEncoder/${btype}To${format}];
    set bp "";

    if {$encoder != ""} {
	set ssrc [$output_info_($id,vagent) get_local_srcid];
	set bp [new BufferPool/RTP];
	$bp srcid $ssrc;

	$encoder buffer-pool $bp;
	$encoder mtu [$self get_option mtu];
	$encoder target [$output_info_($id,vagent) get_transmitter]

	# Total hack FIXME
	$vagent local_bandwidth 30000000
    }

    set output_info_($id,encoder) $encoder;
    set output_info_($id,bp) $bp;

}

DaliSubprogram instproc trigger_vector {vector} {
    $self instvar id_;
    $self instvar tvector_ input_info_ input_id_list_;
    $self instvar tvector_queue_;

    #    puts "$id_ :: $vector";

    if {[info exists tvector_]} {
	# Got called while already processing a trigger
	# vector. For now, send debug message and do
	# nothing.

#	$self send_debug_mesg "Trigger_vector queued";
#	lappend tvector_queue_ $vector;

	# Got called while already processing a trigger
	# vector. Save this one in a queue to process
	# when done. Currently, q length is just one (i.e.,
	# save most recent.)

	set tvector_queue_ [list $vector];

	return;
    }

    set tvector_ "";

    foreach entry $vector {
	set id [lindex $entry 0];
	set ts [lindex $entry 1];

	set cur_ts [$input_info_($id,buffername) set ts_];
	if {$cur_ts > $ts} {
	    # Missed
	    foreach id $input_id_list_ {
		if {$input_info_($id,decoder) != "" } {
		    $input_info_($id,decoder) set_callback "";
		    $input_info_($id,decoder) delay_off;
		}
	    }
	    unset tvector_;
	    $self send_debug_mesg "Missed timestamp for input $id";
	    $self check_tvector_queue
	    return;
	} elseif {$cur_ts == $ts} {
	    # Right on, set delay.
	    if {$input_info_($id,decoder) != "" } {
		$input_info_($id,decoder) delay_on;
	    }
	} else {
	    # Waiting.
	    if {$input_info_($id,decoder) == ""} {
		# Big problem!!!
		foreach id $input_id_list_ {
		    if {$input_info_($id,decoder) != "" } {
			$input_info_($id,decoder) set_callback "";
			$input_info_($id,decoder) delay_off;
		    }
		}
		$self send_debug_mesg "No decoder for input $id";
		unset tvector_;
		$self check_tvector_queue
		return;
	    } else {
		# Set callback
		$input_info_($id,decoder) set_callback "$self check_vector $id $ts";
		lappend tvector_ [list $id $ts];
	    }
	}
    }
    if {$tvector_ == ""} {
	# Ready to go.
	$self trigger;
	foreach id $input_id_list_ {
	    if {$input_info_($id,decoder) != "" } {
		$input_info_($id,decoder) set_callback "";
		$input_info_($id,decoder) delay_off;
	    }
	}
	unset tvector_;
	$self check_tvector_queue
    }
}

DaliSubprogram instproc check_tvector_queue {} {
    $self instvar tvector_queue_;

    if {[info exists tvector_queue_]} {
	if {[llength $tvector_queue_] == 0} {
	    unset tvector_queue_;
	} else {
	    set next_tvector [lindex $tvector_queue_ 0];
	    set tvector_queue_ [lrange $tvector_queue_ 1 end];
	    $self send_debug_mesg "Processing trigger vector from queue";
	    $self trigger_vector $next_tvector;
	    return;
	}
    }
}

DaliSubprogram instproc check_vector {id ts} {
    $self instvar input_info_ tvector_ input_id_list_

    set cur_ts [$input_info_($id,buffername) set ts_];

    if {$cur_ts > $ts} {
	# Missed
	foreach id $input_id_list_ {
	    if {$input_info_($id,decoder) != "" } {
		$input_info_($id,decoder) set_callback "";
		$input_info_($id,decoder) delay_off;
	    }
	}
	unset tvector_;
	$self send_debug_mesg "Check vector missed on input $id";
	$self check_tvector_queue;
	return;

    } elseif {$cur_ts == $ts} {
	# Right on
	set new_tvector "";
	foreach entry $tvector_ {
	    set eid [lindex $entry 0];
	    if {$eid != $id} {
		lappend new_tvector $entry;
	    }
	}
	set tvector_ $new_tvector;

	$input_info_($id,decoder) set_callback "";
	$input_info_($id,decoder) delay_on;

	if {$tvector_ == ""} {
	    # Ready to go.
	    $self trigger;
	    foreach id $input_id_list_ {
		if {$input_info_($id,decoder) != "" } {
		    $input_info_($id,decoder) set_callback "";
		    $input_info_($id,decoder) delay_off;
		}
	    }
	    unset tvector_;
	    $self check_tvector_queue;
	}
    }
}

DaliSubprogram instproc rated_trigger {in_id} {
    $self instvar input_info_;

    set period $input_info_($in_id,rate_trigger_period);
    set offset $input_info_($in_id,rate_trigger_offset);

    set bname $input_info_($in_id,buffername);

    set ts [expr [$bname set ts_]];

    # No unsigned ints in Tcl, so must be very careful here.

    if {$offset == 0} {
	# Offset has been reset since last time
#	puts "Offset reset, ts = $ts";

	$self trigger;
	if {$ts < 0} {
	    set input_info_($in_id,rate_trigger_offset) [expr int($ts / $period) * $period];
	} else {
	    set input_info_($in_id,rate_trigger_offset) [expr (int($ts / $period)+1) * $period];
	}
    } elseif {$ts > $offset} {
#	puts "$ts ([expr $ts]) > $offset";
	$self trigger;
	set input_info_($in_id,rate_trigger_offset) [expr (int(($ts - $offset) / $period) + 1) * $period + $offset];
    } else {
#	puts "$ts ([expr $ts]) < $offset";
    }
}


DaliSubprogram instproc trigger {} {
    puts "No default trigger action";
}

DaliSubprogram instproc set_output_geometry {out_id geometry} {
    puts "No default output geometry action";
}

DaliSubprogram instproc set_ntp_reference {in_id out_id} {
    $self instvar output_info_;

    set output_info_($out_id,synch_master) $in_id;
}

DaliSubprogram instproc trigger_sr {src in_id} {
    $self instvar output_info_ output_id_list_;

    foreach o $output_id_list_ {
	if {[info exists output_info_($o,synch_master)]} {
	    if {$output_info_($o,synch_master) == $in_id} {
		catch {
		    set in_layer [lindex [$src set layers_] 0];
		    set out_layer [lindex [[$output_info_($o,vagent) set local_] set layers_] 0];
		    $out_layer set ref_mts_ [$in_layer set mts_];
		    $out_layer set ref_ntp_sec_ [$in_layer set ntp_ts_sec_];
		    $out_layer set ref_ntp_fsec_ [$in_layer set ntp_ts_fsec_];
		}
	    }
	}
    }
}

Class GraphComm/DaliSubprogram -superclass GraphComm;

GraphComm/DaliSubprogram instproc init {subprog args} {
    eval $self next $args

    $self instvar subprog_;

    set subprog_ $subprog;
}

GraphComm/DaliSubprogram instproc update_input_attr_value {in_id attr_name attr_value} {
    $self instvar subprog_;

    $self next $in_id $attr_name $attr_value;

    switch -exact -- $attr_name {
	spec {
	    $subprog_ instvar input_id_list_;

	    if {[lsearch $input_id_list_ $in_id] != -1} {
		$subprog_ set_input_spec $in_id $attr_value;
	    }
	}
	trigger {
	    $subprog_ instvar input_id_list_ input_info_;

	    if {[lsearch $input_id_list_ $in_id] != -1} {
		switch -exact -- [lindex $attr_value 0] {
		    auto {
			set trigger_flag [lindex $attr_value 1];
			set input_info_($in_id,trigger) $trigger_flag;

			if {$input_info_($in_id,decoder) != ""} {
			    if {$input_info_($in_id,trigger) != 0} {
				$input_info_($in_id,decoder) set_callback "$subprog_ trigger";
			    } else {
				$input_info_($in_id,decoder) set_callback "";
			    }
			}
		    }
		    rate_limited {
			set trigger_flag [lindex $attr_value 1];
			if {$trigger_flag == 1} {
			    set trigger_offset [lindex $attr_value 2];
			    set trigger_period [lindex $attr_value 3];
			} else {
			    set trigger_offset 0;
			    set trigger_period 0;
			}

			if {$trigger_offset != ""} {
			    # Reset offset
			    set input_info_($in_id,rate_trigger_offset) $trigger_offset;
			}
			set input_info_($in_id,rate_trigger_period) $trigger_period;

			set input_info_($in_id,trigger) $trigger_flag;

			if {$input_info_($in_id,decoder) != ""} {
			    if {$input_info_($in_id,trigger) != 0} {
				$input_info_($in_id,decoder) set_callback "$subprog_ rated_trigger $in_id";
			    } else {
				$input_info_($in_id,decoder) set_callback "";
			    }
			}
		    }
		}
	    }
	}
    }
}

GraphComm/DaliSubprogram instproc update_output_attr_value {out_id attr_name attr_value} {
    $self instvar subprog_;

    $self next $out_id $attr_name $attr_value;

    $subprog_ instvar output_id_list_ output_info_;

    if {[lsearch $output_id_list_ $out_id] == -1} {
	return;
    }

    switch -exact -- $attr_name {
	spec {
	    $subprog_ set_output_spec $out_id $attr_value;
	}
	format {
	    $subprog_ set_output_format $out_id $attr_value;
	}
	geometry {
	    $subprog_ set_output_geometry $out_id $attr_value;
	}
    }
}

GraphComm/DaliSubprogram instproc recv_trigger_command {cmd} {
    $self instvar subprog_;

    set cmd_type [lindex $cmd 0];

    switch -exact -- $cmd_type {
	trigger {
	    $subprog_ trigger;
	}
	trigger_vector {
	    set vector [lindex $cmd 1];
	    $subprog_ trigger_vector $vector;
	}
    }
}

GraphComm/DaliSubprogram instproc update_parameter_attr_value {param_id attr_name attr_value} {

    $self instvar subprog_;

    $self next $param_id $attr_name $attr_value;

    $subprog_ instvar parameter_id_list_ parameter_info_;

    if {[lsearch $parameter_id_list_ $param_id] == -1} {
	return;
    }

    switch -exact -- $attr_name {
	value {
	    set pobj $parameter_info_($param_id,oname);
	    if {[$pobj get] != $attr_value} {
		$pobj set $attr_value;
	    }
	}
    }
}

GraphComm/DaliSubprogram instproc setup {} {
    $self instvar subprog_;

    $subprog_ instvar input_id_list_;

    foreach i $input_id_list_ {
	$subprog_ instvar input_info_

	$self create_input $i;

	if {[info exists input_info_($i,primary_trigger)]} {
	    $self create_input_attr $i "primary_trigger"
	}
    }

    $subprog_ instvar output_id_list_;

    foreach o $output_id_list_ {
	$self create_output $o;
    }

    $subprog_ instvar parameter_id_list_
    $subprog_ instvar parameter_info_;

    foreach p $parameter_id_list_ {
	$self create_parameter $p;

	set pobj $parameter_info_($p,oname);

	$self create_parameter_attr $p type
	$self create_parameter_attr $p domain
	$self create_parameter_attr $p value

	$self set_parameter_attr $p type [$pobj type];
	$self set_parameter_attr $p domain [$pobj domain];
    }
}

