/*
 * video-rgb.cc --
 *      
 *      Allows the capture and transmission of RGB images using the
 *      Datapath VisionRGB board (www.datapath.co.uk)
 *      
 *      The RGBSDK directory on the install disk included with the board
 *      needs to be copied into the mash-code directory in order to compile.
 *      Currently it is forced to capture at RGB888 or 24 bits/pixel
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = 
"@(#) $Header: /usr/mash/src/repository/mash/mash-1/video/video-rgb.cc,v 1.6 2002/02/03 04:20:17 lim Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fstream.h>
#include <sys/stat.h>
#include <windows.h>

#include "video-device.h"
#include "Tcl.h"
#include "module.h"
#include "rgb-converter.h"

#include <stddef.h>
#include <api.h>
#include <vdif.h>
#include <rgbcapt.h>
#define API(type,modifier,name,args) \
   extern "C" type (__stdcall *name) args = 0;
#include <rgbcapi.h>
#undef API

// Color subsampling options.
#define CF_422 0
#define CF_411 1
#define CF_CIF 2

static int PixelFormatToBytesPerPixel[] =
{
	0  /* RGBPIXELFORMAT_UNKNOWN */,
	2  /* RGBPIXELFORMAT_555 */,
	2  /* RGBPIXELFORMAT_565 */,
	4  /* RGBPIXELFORMAT_888 */,
	};

	
/* The Registry Strings *******************************************************/

const char RegistryKeyName[] =      { "SOFTWARE\\Datapath\\Vision RGB" };

const char RGBCaptureDLLName[] =    { "RGBCaptureDLLName" };

const char RGBExtensionDLLName[] =  { "RGBExtensionDLLName" };
const char DefaultSourceName[] =    { "DefaultSourceName" };
const char NumberOfDevices[] =      { "NumberOfDevices" };
const char BarAdjustmentsAdmin[] =  { "BarAdjustmentsAdministrator" };
const char BarAdjustmentsUser[] =   { "BarAdjustmentsUser" };
const char BarInput[] =             { "BarInput" };

const char InputSourceKeyFormat[] = { "%s\\Sources\\%s" };
const char SourcesKeyFormat[] =     { "%s\\Sources" };

const char SourceList[] =           { "SourceList" };

const char Device[] =               { "Device" };
const char Input[] =                { "Input" };
const char Format[] =               { "Format" };
const char Phase[] =                { "Phase" };
const char ModeName[] =             { "Mode Name" };
const char Brightness[] =           { "Brightness" };
const char Contrast[] =             { "Contrast" };
const char BlackLevel[] =           { "BlackLevel" };
const char SampleRate[] =           { "SampleRate" };
const char SyncEdge[] =             { "SyncEdge" };
const char HorScale[] =             { "HorizontalScale" };
const char HorOffset[] =            { "HorizontalOffset" };
const char VerOffset[] =            { "VerticalOffset" };

/* The Initialisation File Strings ********************************************/
const char VESAModesIni[] =         { "vdif.ini" };
const char VisionRGBIni[] =         { "vsnrgb.ini" };
/******************************************************************************/

//Log to get diagnostics
//FILE *timeLog;

void FrameCapturedFunction(LPVOID);	
int captureFlag = 0;
void FrameCapturedFunction(LPVOID fnArg) {
	captureFlag = 1;
	//timeLog = fopen("timelog.txt", "a");
	//fprintf(timeLog, "3. Capture complete %li where %li \n", clock(), CLOCKS_PER_SEC);
	//fclose(timeLog);	
	//printf("Frame has been captured!");
}

enum
{
	TYPE_USHORT,
	TYPE_SHORT,
	TYPE_ULONG,
};

typedef struct
{
	const char  *Name;
	int         Offset;
	int         Type;
	void        *PVoid;
}  REGISTRYENTRY, *PREGISTRYENTRY;

#define DECLARE_PARM(name,type) \
   { name, offsetof ( RGBCAPTUREPARMS, name ), type, NULL }

static REGISTRYENTRY  ParmsEntry[] =
{
	DECLARE_PARM ( Input,      TYPE_USHORT ),
	DECLARE_PARM ( Format,     TYPE_USHORT ),
	DECLARE_PARM ( Phase,      TYPE_USHORT ),
	DECLARE_PARM ( Brightness, TYPE_USHORT ),
	DECLARE_PARM ( Contrast,   TYPE_USHORT ),
	DECLARE_PARM ( BlackLevel, TYPE_USHORT ),
	DECLARE_PARM ( SampleRate, TYPE_USHORT ),
	DECLARE_PARM ( SyncEdge,   TYPE_USHORT ),
	DECLARE_PARM ( HorScale,   TYPE_ULONG ),
	DECLARE_PARM ( HorOffset,  TYPE_SHORT ),
	DECLARE_PARM ( VerOffset,  TYPE_SHORT ),
};

#define PARMSENTRYCOUNT  \
      ( sizeof ( ParmsEntry ) / sizeof ( REGISTRYENTRY ))

typedef struct
{
	BITMAPINFO  BitmapInfo;
	DWORD       dummy[2];
}  RGBBITMAP, *PRGBBITMAP;


//
//	VideoCaptureTest
//
// this is the C++ object that implements the virtual, video-capture device 
//	that generates test streams. As any other VideoCaptureXXX object, it 
//	has to inherit from VideoCapture. 
//
class VideoCaptureRGB : public VideoCapture {
public:
	VideoCaptureRGB(const char * cformat, const char *dev);
	virtual ~VideoCaptureRGB();

	virtual int command(int argc, const char*const* argv);
	virtual void start();
	virtual void stop();
	virtual void grab();

	
protected:
	//inline int status();
	int capture();
	void format();
	void setsize();
	int set_file(const char *filename);
	int set_bluepassion();
	unsigned short  DetectPixelFormat(DWORD *);
	int ReadCaptureParms (LPSTR, unsigned short *,PRGBCAPTUREPARMS);
	int SetUpCapture ( HRGBCAPTURE, 
					   unsigned long *, PRGBCAPTUREBUFFER, 
				       PRGBCAPTUREPARMS);  
	BOOL Detect ( HRGBCAPTURE, PRGBCAPTUREPARMS);
	int CountBits ( unsigned long, int start, int end);  
	void _Read (HKEY, PREGISTRYENTRY ,int count, PVOID);
	int ReadMode (LPSTR, VDIF  FAR *);
	void DefaultsFromVDIF (PRGBCAPTUREPARMS);
	BOOL FindMode (HRGBCAPTURE, PRGBCAPTUREPARMS);
	int fd_;
	int format_;
	int cformat_;
	int port_;
	int pixelsize_;
	unsigned short pixelFormat_;
	
	RGBBITMAP RGBBitmap;
	HRGBCAPTURE hRGBCapture;	
	RGBCAPTUREBUFFER captureBuffer;
	unsigned long bufferSize;
	RGBCAPTUREPARMS captureParms;
 
	unsigned char *tm_;

	// definition of the frame the encoder expects
	char *yuvdata_; 
	int width_;
	int height_;
	
	
	// definition of the frame obtained from the capture card.
	LPVOID rgbdata_; 
	int rgbwidth_;
	int rgbheight_;

	int decimate_;
};


//
//	RGBDevice
//
// A RGBDevice object creates an instance of the C++ version of the 
//	capture device (VideoCaptureTest) and links it to a tcl object 
//	(VideoCapture/Test), so that calls can be made from one world to 
//	the other. As any other XXXDevice object, it has to inherit from 
//	VideoDevice. 
//
// Creating a XXXDevice object means that the device XXX has been 
//	reserved by the creator. In order to create a new VideoCaptureXXX 
//	object, a call must be done to XXXDevice::create() with the 
//	specific parameters selected (color subsampling, size, port, ...)
//
class RGBDevice : public VideoDevice {
public:
	RGBDevice(const char* clsname, const char* nickname,
			const char* devname, const char* attr, int free);
  	TclObject* create(int argc, const char*const* argv) {
		if (argc != 5) {
			printf("invalid arguments passed to RGBDevice\n");
			return (0);
		}
		if (strcmp(argv[4], "422") == 0) {
			printf("422\n");
			return (new VideoCaptureRGB("422",name_));
		} else if (strcmp(argv[4], "411") == 0) {
			printf("411\n");
			return (new VideoCaptureRGB("411",name_));
		} else if (strcmp(argv[4], "cif") == 0) {
			printf("cif\n");
			return (new VideoCaptureRGB("cif",name_));
		}
		printf("VideoDevice Done\n");
		return (0);
	}
  
protected:
	const char* name_;
};

RGBDevice::RGBDevice(const char* clsname, const char* nickname,
				 const char *devname, const char* attr, int free) 
	: VideoDevice(clsname, nickname), name_(devname)
{
	if (free)
		attributes_ = attr; 
	else
		attributes_ = "disabled";
	printf("RGBDevice Done\n");
}



//
//	RGBScanner
//
// this object is used to allow the scanning of your box in order to 
//	check if a video-capture device is present. The idea is to create 
//	an object of the type XXXScanner for every video-capture device "XXX". 
//	With this, you ensure that the XXXScanner constructor is going to be 
//	called at least once during mash initialization. This constructor 
//	checks for the existence of the video-capture device it refers to 
//	and, should it find such device, it creates a new XXXDevice object. 
//
//	In our case, we want a VideoCapture/Test tcl object to exists always, 
//	so that the TestScanner constructor always creates a new RGBDevice
//
class RGBScanner {
public:
	RGBScanner();
};

static RGBScanner find_rgb_devices;

RGBScanner::RGBScanner()
{
	printf("RGBScanner\n");
	const char *rgbclassnames[] = {"VideoCapture/RGB"};
	char * nick = new char[512];
	strcpy(nick, "RGB");
	char * attr = new char[512];
	strcpy (attr, "format { 411 422 cif } size { small large cif } ");
	new RGBDevice (rgbclassnames[0], nick, "RGB", attr, 1);
	printf("RGBScanner Done\n");
}


//
//	VideoCaptureRGB::VideoCaptureRGB
//
VideoCaptureRGB::VideoCaptureRGB(const char *cformat,
						   const char * /* dev */)
{





	HINSTANCE   hDLL;
	if (!strcmp(cformat,"422")) {
		printf("422_");
		cformat_ = CF_422;
	}
	else if (!strcmp(cformat, "411")) {
		printf("411_");
		cformat_ = CF_411;
	}
	else if (!strcmp(cformat, "cif")) {
		printf("CIF");
		cformat_ = CF_CIF;
	}
	//not sure what this is for...
	decimate_  = 2;
	yuvdata_ = NULL; 
	rgbdata_ = NULL; 

	{
		/* List of functions to be imported from RGB Capture Device DLL. */
		static APIFNENTRY _FnList[] =
		{
			#define API(type,modifier,name,args) \
			{ (FARPROC *)&name, #name },
			#include "rgbcapi.h"
			#undef API 
			{ NULL, NULL }
		}; char  nameOfDLL[_MAX_PATH];
		
		/* The path to the DLL is obtained from the Registry */
		{
			LONG  lError;
			HKEY  hKey;
			lError = RegOpenKey ( HKEY_LOCAL_MACHINE, RegistryKeyName, &hKey );
			if ( lError == ERROR_SUCCESS )
				{
					DWORD dwType;
					DWORD dwLength;
					
					dwType = REG_SZ;
					dwLength = sizeof ( nameOfDLL );
					lError = RegQueryValueEx 
						( hKey, RGBCaptureDLLName, NULL, &dwType, (LPBYTE)nameOfDLL, &dwLength );
					RegCloseKey ( hKey );
				}
			if ( lError != ERROR_SUCCESS )
				{
					//FIXME: What happens in case of error?
				}
			}
		hDLL = LoadLibrary(nameOfDLL );
		if ( hDLL == NULL )
			{
			//FIXME: What happens in case of error?
			}
		if ( !APILoadFunctions ( hDLL, _FnList, "" ))
			{
			//FIXME: What happens in case of error?
			}
	}
	
	/* Initializing Important Attributes */	
	captureBuffer.Index = 0;
	captureBuffer.LpVoidBuffer = NULL;
	hRGBCapture = (HRGBCAPTURE) NULL;
	bufferSize = 0;
}


//
//	VideoCaptureRGB::VideoCaptureRGB
//
VideoCaptureRGB::~VideoCaptureRGB()
{
	if ( hRGBCapture )
         {
            RGBCaptureEnable ( hRGBCapture, FALSE );
            if ( captureBuffer.LpVoidBuffer )
            {
               RGBCaptureFreeBuffer ( &captureBuffer );
            }
            RGBCaptureClose ( hRGBCapture );
         }
	if (rgbdata_ != NULL) {
		delete rgbdata_;
	}
	if (yuvdata_ != NULL) {
		delete yuvdata_;
	}
}

//
//	VideoCaptureRGB::command
//
// This method is called from tcl to change the video-capture device 
//	settings
//

int VideoCaptureRGB::command(int argc, const char*const* argv) {
	return (VideoCapture::command(argc, argv));
}
	
/*
  int  VideoCaptureRGB::status() {
return (status_);
}
*/

//
//	VideoCaptureRGB::start
//
void VideoCaptureRGB::start()
{
	unsigned short    usDevice;
	RGBERROR          error;
	unsigned short    numberOfDevices;

	/* Fixing format of how board is captureing image. */
	pixelFormat_ = 2; //DetectPixelFormat ((DWORD *)RGBBitmap.BitmapInfo.bmiColors );
	
	/* Initialise the Capture device */
	if (error = RGBCaptureInitialise ( &numberOfDevices ))
		{
			//FIXME: Error Message?
		}

	usDevice = 0; /* Assume device zero should be captured */  

	if (error = RGBCaptureOpen ( usDevice, &hRGBCapture ))
		{
			//FIXME: Error Message?
			RGBCaptureClose ( hRGBCapture );
		}

	/* Initialise captureParms with default values. */
	captureParms.Size = sizeof ( captureParms );
	captureParms.Flags = RGBCAPTURE_PARM_ALL;

	error = RGBCaptureGetParameters ( hRGBCapture, &captureParms,
									RGBCAPTURE_FLAG_DEFAULT | RGBCAPTURE_FLAG_REAL_WORLD );
	if (error)
		{
			//FIXME: Error Message?
			RGBCaptureClose ( hRGBCapture );
		}
	captureParms.Flags =
						RGBCAPTURE_PARM_PIXELFORMAT|
						RGBCAPTURE_PARM_FUNCTION |
						RGBCAPTURE_PARM_ARGUMENTS;
	
	captureParms.PixelFormat = pixelFormat_;
	captureParms.HWnd = NULL;
	captureParms.FrameCapturedFn = FrameCapturedFunction;
	captureParms.FrameCapturedArg = NULL;

	error = RGBCaptureSetParameters ( hRGBCapture, &captureParms,
							    RGBCAPTURE_FLAG_CURRENT | RGBCAPTURE_FLAG_REAL_WORLD );
	if ( error )
		{
			//FIXME: Error Message
			RGBCaptureClose ( hRGBCapture );
		}

	captureParms.Flags = RGBCAPTURE_PARM_ALL &
						 ~(RGBCAPTURE_PARM_PIXELFORMAT | RGBCAPTURE_PARM_HWND |
	   					 RGBCAPTURE_PARM_FUNCTION | RGBCAPTURE_PARM_ARGUMENTS );
    error = RGBCaptureSetParameters ( hRGBCapture, &captureParms,
            RGBCAPTURE_FLAG_CURRENT | RGBCAPTURE_FLAG_REAL_WORLD );
    if ( error )
        {
			//FIXME: Error Message
            RGBCaptureClose ( hRGBCapture );
         }

	if ( !SetUpCapture (hRGBCapture, &bufferSize, &captureBuffer, &captureParms ))
		{
			//FIXEME: Error Message
			RGBCaptureClose ( hRGBCapture );
		}
	switch (pixelFormat_)
		{
		case RGBPIXELFORMAT_555:
			pixelsize_ = 16;
			break;
		case RGBPIXELFORMAT_565:
			pixelsize_ = 16; 
			break;
		case RGBPIXELFORMAT_888:
			pixelsize_ = 32;
		}
	//FIXME: Detects the input parameters? Comment out to get default to work...
	while (!Detect ( hRGBCapture, &captureParms )) {
	printf("Hello");
	}
	captureParms.Flags = RGBCAPTURE_PARM_ALL &
                        ~( RGBCAPTURE_PARM_PIXELFORMAT | RGBCAPTURE_PARM_HWND |
						RGBCAPTURE_PARM_FUNCTION | RGBCAPTURE_PARM_ARGUMENTS );
    error = RGBCaptureSetParameters ( hRGBCapture, &captureParms,
                        RGBCAPTURE_FLAG_CURRENT | RGBCAPTURE_FLAG_REAL_WORLD );
	//End Comments
	if ( !SetUpCapture (hRGBCapture, &bufferSize, &captureBuffer, &captureParms ))
		{
			//FIXEME: Error Message
			RGBCaptureClose ( hRGBCapture );
		}
	error = RGBCaptureEnable ( hRGBCapture, TRUE );
	if ( error != 0 )
		{
			//FIXME: Error Message?
		}
	VideoCapture::start();
}

//
//	VideoCaptureRGB::stop
//
void VideoCaptureRGB::stop()
{   
	VideoCapture::stop();
}


//
//	VideoCaptureRGB::capture
//
// This method is called every time a frame is needed. The timing of the 
//	frame capturing is decided by the user by selecting the fps he wants. 
//	Therefore every time capture is called the video-capture device has 
//	to provide a frame.
//
int VideoCaptureRGB::capture()
{
/*
	FILE *red;
	FILE *green;
	FILE *blue;
	FILE *rgb;
*/
/*
	RGBERROR error;
*/
/*
	width_ =  captureParms.VideoTimings.HorAddrTime; //640; captureParms.VideoTimings.HorAddrTime;
	height_ = captureParms.VideoTimings.VerAddrTime; //480; //captureParms.VideoTimings.VerAddrTime;
*/
	rgbdata_ = (&captureBuffer)-> LpVoidBuffer;
/*
	red = fopen("red.ppm", "w");
	green = fopen("green.ppm", "w");
	blue = fopen("blue.ppm", "w");
	rgb = fopen("rgb.ppm", "w");

	u_char* p = (u_char *) rgbdata_;

	for (int i =0; i<= width_*height_; i+=1) {
		
		u_char myPixel[3];
		u_char redpixel = p[2];
		u_char greenpixel = p[1];
		u_char bluepixel = p[0];

		myPixel[2] = bluepixel;
		myPixel[1] = greenpixel;
		myPixel[0] = redpixel;
		fwrite(myPixel, 3, 1, rgb);

		myPixel[2] = 0; 
		myPixel[1] = 0; 
		myPixel[0] = redpixel;
		fwrite(myPixel, 3, 1, red);
		
		myPixel[2] = 0;
		myPixel[1] = greenpixel;
		myPixel[0] = 0;
		fwrite(myPixel, 3, 1, green);
		
		myPixel[2] = bluepixel;
		myPixel[1] = 0;
		myPixel[0] = 0; 
		fwrite(myPixel, 3, 1, blue);
		
		p +=4;
			
	}
	
	fclose(red);
	fclose(green);
	fclose(blue);
	fclose(rgb);

*/	
	//Format setups the yuvdata_ (converted from rgbdata_)
	format();

	if (cformat_ == CF_422) {
		memcpy(frame_, (u_char *)yuvdata_, 2*width_*height_);
	} else {//CF_411
		memcpy(frame_, (u_char *)yuvdata_, (int)(1.5*width_*height_));
	}
	return TRUE;

}

void VideoCaptureRGB::grab()
{
	RGBERROR error;
	
	//timeLog = fopen("timelog.txt", "a");
	//fprintf(timeLog, "1. Capture Request %li where %li \n", clock(), CLOCKS_PER_SEC);
	//fclose(timeLog);
	/* This triggers the first RGBWM_FRAMECAPTURED message. */
	//FIXME: What to do with error?

	if (captureFlag == 0)
		{ 
		//timeLog = fopen("timelog.txt", "a");
		//fprintf(timeLog, "2. Capture starting %li where %li \n", clock(), CLOCKS_PER_SEC);
		//fclose(timeLog);
		error = RGBCaptureFrameBufferReady ( hRGBCapture,
		  (&captureBuffer)->Index, bufferSize );
		}
	if ((captureFlag == 0))
		return;
	else
		{ if (capture() == 0)
			return;
		}
	
	//timeLog = fopen("timelog.txt", "a");
	//fprintf(timeLog, "6. Entire Done %li where %li \n", clock(), CLOCKS_PER_SEC);
	//fclose(timeLog);
	captureFlag = 0;
	suppress(frame_);
	saveblks(frame_);

	YuvFrame f(media_ts(), frame_, crvec_, outw_, outh_);
	target_->recv(&f);
	//timeLog = fopen("timelog.txt", "a");
	//fprintf(timeLog, "7. Sending Done %li where %li \n", clock(), CLOCKS_PER_SEC);
	//fclose(timeLog);
}

void VideoCaptureRGB::format()
{
	//FIXME: This is incorrect... forcing
	//	RGBCaptureGetParameters ( hRGBCapture, &captureParms, RGBCAPTURE_FLAG_DETECT);
	width_ =  captureParms.VideoTimings.HorAddrTime; //640; captureParms.VideoTimings.HorAddrTime;
	height_ = captureParms.VideoTimings.VerAddrTime; //480; //captureParms.VideoTimings.VerAddrTime;

	//timeLog = fopen("timelog.txt", "a");
	//fprintf(timeLog, "4. Conversion start %li where %li \n", clock(), CLOCKS_PER_SEC);

	switch (cformat_) {
	case CF_411:
		set_size_411(width_, height_);
		break;
	case CF_422:
		set_size_422(width_, height_);
		break;
	}

	// after calling set_size_xxx, you must always allocate the 
	//	reference buffer ref_ 
	allocref();
	// create a place to store the YUV data.
	u_int8_t *yuvdata = new u_int8_t[2*width_*height_];
	
	// convert the RGB input into YUV
	RGB_Converter* converter;
	if (cformat_ == CF_422) {
		converter = new RGB_Converter_422 (pixelsize_, NULL, 0);
	} else if (cformat_ == CF_411) {
		converter = new RGB_Converter_411 (pixelsize_, NULL, 0);
	} else { // 4:2:0
		converter = new RGB_Converter_411 (pixelsize_, NULL, 0);
	}
	
	rgbwidth_ = width_;
	rgbheight_ = height_;
	converter->convert ((u_int8_t *)rgbdata_, rgbwidth_, rgbheight_, 
					yuvdata, width_, height_, 0);
	delete converter;
	
	// free the old YUV buffer
	if (yuvdata_ != NULL) {
		delete[] yuvdata_;
	}
	yuvdata_ = (char *)yuvdata;
	printf("Formate Done\n");
    //fprintf(timeLog, "5. Conversion finished! %li where %li \n", clock(), CLOCKS_PER_SEC);
    //fclose(timeLog);
	return;
}

int VideoCaptureRGB::SetUpCapture (
							HRGBCAPTURE       hRGBCapture,
							unsigned long     *pBufferSize,
							PRGBCAPTUREBUFFER pCaptureBuffer,
							PRGBCAPTUREPARMS  pCaptureParms )
{
	RGBERROR error;
	error = RGBCaptureEnable ( hRGBCapture, FALSE );
	if ( error != 0 )
		{
			//FIXME: Error Message?
			return FALSE;
		}
   
	if ( pCaptureBuffer->LpVoidBuffer )
		{
			RGBCaptureFreeBuffer ( pCaptureBuffer );
		}

	/* Allocate the frame buffer. */
	*pBufferSize =
		pCaptureParms->VideoTimings.HorAddrTime *
		pCaptureParms->VideoTimings.VerAddrTime *
		PixelFormatToBytesPerPixel[pixelFormat_];
	error = RGBCaptureAllocateBuffer ( pCaptureBuffer, *pBufferSize );
	if ( error )
		{
			//FIXME: Error Message?
			return FALSE;
		}

	error = RGBCaptureEnable ( hRGBCapture, TRUE );
	if ( error != 0 )
		{
			//Error Message?
			return FALSE;
		}
	return TRUE;
}



/******************************************************************************/

unsigned short VideoCaptureRGB::DetectPixelFormat (
										 DWORD *pColourMask )
	/*
 * Summary: Detects the pixel format of the graphics card frame buffer.
 *
 * Purpose: 
 *          pColourMask points to an array of three DWORDs. Each DWORD is
 *             loaded with a bit mask representing the bits for red, green
 *             and blue respectively.
 *
 * Return:  One of the following values:
 *             RGBPIXELFORMAT_555,
 *             RGBPIXELFORMAT_565,
 *             RGBPIXELFORMAT_888,
 *             RGBPIXELFORMAT_UNKNOWN.
 */
{
	printf("DetectPixelFormat\n");
	HWND           hWnd;
	HDC            hDC, hDCCompatible;
	HBITMAP        hBitmap, hBitmapOld;
	DWORD          dwPixel, dwMask;
	int            nBytes, end, red, i, green, blue;
	unsigned short format;

	hWnd = GetDesktopWindow ( );
	hDC = GetDC ( hWnd );
	hDCCompatible = CreateCompatibleDC ( hDC );

	hBitmap = CreateCompatibleBitmap ( hDC, 1, 1 );
	hBitmapOld = (HBITMAP)SelectObject ( hDCCompatible, hBitmap );

	SetPixel ( hDCCompatible, 0, 0, RGB ( 255, 0, 0 ));
	nBytes = GetBitmapBits ( hBitmap, sizeof ( dwPixel ), &dwPixel );
	end = ( nBytes * 8 ) - 1;
	red = CountBits ( dwPixel, 0, end );

	dwMask = 0;
	for ( i = 0; i < nBytes; i++ )
		{
			dwMask |= ( 0xff << ( i * 8 ));
		}
	pColourMask[0] = dwPixel & dwMask;

	SetPixel ( hDCCompatible, 0, 0, RGB ( 0, 255, 0 ));
	GetBitmapBits ( hBitmap, sizeof ( dwPixel ), &dwPixel );
	green = CountBits ( dwPixel, 0, end );
	pColourMask[1] = dwPixel & dwMask;

	SetPixel ( hDCCompatible, 0, 0, RGB ( 0, 0, 255 ));
	GetBitmapBits ( hBitmap, sizeof ( dwPixel ), &dwPixel );
	blue = CountBits ( dwPixel, 0, end );
	pColourMask[2] = dwPixel & dwMask;

	SelectObject ( hDCCompatible, hBitmapOld );
	DeleteDC ( hDCCompatible );
	DeleteObject ( hBitmap );

	ReleaseDC ( hWnd, hDC );

	if (( red == 5 ) && ( green == 5 ) && ( blue == 5 ))
		{
			format =  RGBPIXELFORMAT_555;
		}
	else if (( red == 5 ) && ( green == 6 ) && ( blue == 5 ))
		{
			format =  RGBPIXELFORMAT_565;
		}
	else if (( red == 8 ) && ( green == 8 ) && ( blue == 8 ))
		{
			format =  RGBPIXELFORMAT_888;
		}
	else
		{
			format =  RGBPIXELFORMAT_UNKNOWN;
		}
   	printf("DetectPixelFormat Done\n");
	return format;
}

/******************************************************************************/
int VideoCaptureRGB::ReadCaptureParms (
							    LPSTR             pName,
							    unsigned short    *pusDevice,
							    PRGBCAPTUREPARMS  pCaptureParms )
	/*
 * Summary: Reads the capture parameters for a given input.
 *
 * Purpose:
 *          If no value for a particular field is found in the registry, the
 *          corresponding field in the structure is not modified. This allows
 *          the structure can be initialised with default values.
 *
 *          The values read from the registry are not validated here.
 *
 *          pName
 *             points to a string containing the mode name. The values for a
 *             mode are stored in:
 *                 HKLM\SOFTWARE\Datapath\Vision RGB\Sources\<*pName>.
 *
 * Return:  If the paramters are read, TRUE is returned; otherwise FALSE is
 *          returned. The circumstances under which FALSE will be returned
 *          are:
 *          - The registry key for the mode is not found.
 */
{
	printf("ReadCaptureParms\n");
	char  keyName[_MAX_PATH];
	LONG  lError;
	HKEY  hKey;
	DWORD dwType, dwLength, dwValue;

	wsprintf ( keyName, InputSourceKeyFormat, RegistryKeyName, pName );
	lError = RegOpenKey ( HKEY_LOCAL_MACHINE, keyName, &hKey );
	if ( lError != ERROR_SUCCESS )
		{
			return FALSE;
		}

	dwType = REG_DWORD;
	dwLength = sizeof ( DWORD );
	lError = RegQueryValueEx ( hKey, Device, NULL,
						  &dwType, (LPBYTE)&dwValue, &dwLength );
	if ( lError == ERROR_SUCCESS )
		{
			*pusDevice = (unsigned short)dwValue;
		}
	else
		{
			/* If any devices are present, device zero will be present. */
			*pusDevice = 0;
		}

	_Read ( hKey, ParmsEntry, PARMSENTRYCOUNT, pCaptureParms );

	dwType = REG_SZ;
	dwLength = sizeof ( pCaptureParms->VideoTimings.Description );
	lError = RegQueryValueEx ( hKey, ModeName, NULL,
						  &dwType, (LPBYTE)pCaptureParms->VideoTimings.Description, &dwLength );

	RegCloseKey ( hKey );

	printf("ReadCaptureParms Done\n");
	return ReadMode ( pCaptureParms->VideoTimings.Description,
				   &pCaptureParms->VideoTimings );
}

/******************************************************************************/

int VideoCaptureRGB::CountBits (
						  unsigned long  ulValue,
						  int            start,
						  int            end )
	/*
 * Summary: Counts the number of bits set in a bit mask.
 *
 * Purpose:
 *
 * Return:  The number of bits set.
 */
{
	printf("CountBits\n");
	int            count = 0, i;
	unsigned long  ulMask;

	for ( i = start, ulMask = 1 << start; i <= end; i++, ulMask <<= 1 )
		{
			if ( ulMask & ulValue )
				{
					count++;
				}
		}
	printf("CountBits Done\n");
	return count;
}

/******************************************************************************/

void VideoCaptureRGB::_Read (
					    HKEY           hKey,
					    PREGISTRYENTRY pRegistryEntry,
					    int            count,
					    PVOID          pStructure )
{
	printf("_Read\n");
	int   i;

	for ( i = 0; i < count; i++ )
		{
			DWORD dwType, dwLength, dwValue;
			LONG  lError;

			dwType = REG_DWORD;
			dwLength = sizeof ( DWORD );
			lError = RegQueryValueEx ( hKey, pRegistryEntry[i].Name, NULL,
								  &dwType, (LPBYTE)&dwValue, &dwLength );
			if ( lError == ERROR_SUCCESS )
				{
					void  FAR *lpVoid;

					lpVoid = ( (char FAR *)pStructure ) + pRegistryEntry[i].Offset;

					switch ( pRegistryEntry[i].Type )
						{
						case TYPE_USHORT:
							*(unsigned short *)lpVoid = (unsigned short)dwValue;
							break;
						case TYPE_SHORT:
							*(signed short *)lpVoid = (signed short)dwValue;
							break;
						case TYPE_ULONG:
							*(unsigned long *)lpVoid = (unsigned long)dwValue;
							break;
						}
				}
			else
				{
					/* Do nothing. */
				}
		}
	printf("_Read Done\n");
}


/******************************************************************************/

int VideoCaptureRGB:: ReadMode (LPSTR lpModeName, VDIF  FAR *pVDIF )
{
	printf("ReadMode\n");
	if ( !VDIFReadMode ( lpModeName, pVDIF, VisionRGBIni ))
		{
			if ( !VDIFReadMode ( lpModeName, pVDIF, VESAModesIni ))
				{
					return FALSE;
				}
		}
	printf("ReadMode Done\n");
	return TRUE;
}
/******************************************************************************/

void VideoCaptureRGB:: DefaultsFromVDIF ( PRGBCAPTUREPARMS pCaptureParms )
{
   pCaptureParms->Flags =
         RGBCAPTURE_PARM_PHASE |
         RGBCAPTURE_PARM_BRIGHTNESS |
         RGBCAPTURE_PARM_CONTRAST |
         RGBCAPTURE_PARM_SAMPLERATE |
         RGBCAPTURE_PARM_SYNCEDGE;
   RGBCaptureGetParameters ( (HRGBCAPTURE)NULL, pCaptureParms,
         RGBCAPTURE_FLAG_DEFAULT | RGBCAPTURE_FLAG_REAL_WORLD );

   pCaptureParms->BlackLevel =
         ( pCaptureParms->VideoTimings.HorBackPorch / 2 );
}

/******************************************************************************/

typedef struct
{
   PRGBCAPTUREPARMS  PCapParms;
   PRGBCAPTURECAPS   PCapCaps;
}  FINDMODEENUMDATA, *PFINDMODEENUMDATA;

static int
FindModeEnumFn (
   VDIF  FAR *pVDIF,
   void  FAR *pVoid )
/*
 * Summary: Compares components of a video mode.
 *
 * Purpose:
 *
 * Return:  If no match is found, TRUE is returned to continue enumerating;
 *          otherwise, FALSE is returned.
 */
{
   PRGBCAPTUREPARMS  pCapParms = ((PFINDMODEENUMDATA)pVoid)->PCapParms;
   PRGBCAPTURECAPS   pCapCaps = ((PFINDMODEENUMDATA)pVoid)->PCapCaps;
   unsigned short    flagMask = 0;
   unsigned short    detectedflags = pCapParms->VideoTimings.Flags;

   /* We do not check for sync polarity because Sync-On-Green is always */
   /* negative going. We pass through the syncs as detected             */
  

   if ( pCapCaps->Flags & RGBCAPTURECAPS_FLAG_DETECT_INTERLACED )
   {
      flagMask |= VDIF_FLAG_INTERLACED;
   }

   if ( flagMask )
   {
      if (( pVDIF->Flags & flagMask ) !=
            ( pCapParms->VideoTimings.Flags & flagMask ))
      {
         return TRUE;
      }
   }

   /* Compare the total number of lines. */
   if ( pCapCaps->Flags & RGBCAPTURECAPS_FLAG_DETECT_VERTOTAL )
   {
      unsigned short vTotal;
      signed long    slDifference;

      vTotal =
         pVDIF->VerAddrTime +
         pVDIF->VerBottomBorder +
         pVDIF->VerFrontPorch +
         pVDIF->VerSyncTime +
         pVDIF->VerBackPorch +
         pVDIF->VerTopBorder;
      slDifference = vTotal - pCapParms->VideoTimings.VerAddrTime;
      if (( slDifference < -3 ) || ( 3 < slDifference ))
      {
         return TRUE;
      }
   }

   /* Compare the refresh rate.
    * The accuracy of the measurement will depend on the granularity of
    * the clock. The clock that we use will depend upon the environment
    * in which the driver executes. */
   if ( pCapCaps->Flags & RGBCAPTURECAPS_FLAG_DETECT_VERFREQUENCY )
   {
      signed long    slDifference;

      slDifference = pVDIF->VerFrequency - pCapParms->VideoTimings.VerFrequency;
      if (( slDifference < -1000 ) || ( 1000 < slDifference ))
      {
         return TRUE;
      }
   }

   memcpy ( &pCapParms->VideoTimings, pVDIF, sizeof ( VDIF ));
   pCapParms->VideoTimings.Flags = detectedflags;
   pCapParms->VideoTimings.Flags |= ( pVDIF->Flags & VDIF_FLAG_READ_ONLY );

   return FALSE;
}

/******************************************************************************/

BOOL VideoCaptureRGB:: FindMode ( HRGBCAPTURE hRGBCapture, PRGBCAPTUREPARMS pCaptureParms )
/*
 * Summary: Finds a mode that matches the partial mode supplied.
 *
 * Purpose:
 *
 * Return:  If a matching mode is found, TRUE is returned; otherwise, FALSE is
 *          returned.
 */
{
   RGBCAPTURECAPS    rgbCaps;
   FINDMODEENUMDATA  findModeEnumData;

   rgbCaps.Size = sizeof ( RGBCAPTURECAPS );
   RGBCaptureGetCapabilities ( hRGBCapture, &rgbCaps );

   findModeEnumData.PCapParms = pCaptureParms;
   findModeEnumData.PCapCaps = &rgbCaps;
   if ( VDIFEnumModes ( VisionRGBIni, FindModeEnumFn, &findModeEnumData ))
   {
      if ( VDIFEnumModes ( VESAModesIni, FindModeEnumFn, &findModeEnumData ))
      {
         /* We haven't found a matching mode. */
         return FALSE;
      }
   }
   return TRUE;
}
/******************************************************************************/

BOOL VideoCaptureRGB:: Detect (HRGBCAPTURE hRGBCapture,PRGBCAPTUREPARMS  pCaptureParms )
/*
 * Summary: Detects the incoming signal and finds the corresponding mode.
 *
 * Purpose:
 *
 * Return:  If a matching mode is found, TRUE is returned; otherwise, FALSE is
 *          returned.
 */
{
   RGBERROR error;

   pCaptureParms->Flags =
         RGBCAPTURE_PARM_FORMAT |
         RGBCAPTURE_PARM_VDIF_CLOCKS |
         RGBCAPTURE_PARM_VDIF_FLAGS |
         RGBCAPTURE_PARM_VDIF_HTIMINGS |
         RGBCAPTURE_PARM_VDIF_VTIMINGS;
   error = RGBCaptureGetParameters ( hRGBCapture, pCaptureParms,
         RGBCAPTURE_FLAG_DETECT | RGBCAPTURE_FLAG_REAL_WORLD );
   if ( error == 0 )
   {
      if ( FindMode ( hRGBCapture, pCaptureParms ))
      {
         DefaultsFromVDIF ( pCaptureParms );
         pCaptureParms->HorScale = 0;
         pCaptureParms->HorOffset = 0;
         pCaptureParms->VerOffset = 0;
         return TRUE;
      }
   }
   return FALSE;
}

