/*
 * video-vigra.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/video/video-vigra.cc,v 1.9 2002/02/03 04:20:17 lim Exp $";

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>

extern "C" {
#include <vigrapix.h>
}

#include "video-device.h"
#include "module.h"
#include "tclcl.h"
#include "transmitter.h"

class VideoCaptureVigraPix : public VideoCapture {
 public:
	VideoCaptureVigraPix(const char *name);
	virtual ~VideoCaptureVigraPix();
	virtual void start();
 protected:
	virtual int command(int argc, const char*const* argv);
	virtual int capture() = 0;
	virtual void grab();
	virtual void setsize() = 0;

	vigrapix_t handle_;
	int format_;		/* video format: NTSC or PAL */
	int port_;		/* VigraPix input port */
	u_int decimate_;
};

class VideoCaptureVigraPix422 : public VideoCaptureVigraPix {
 public:
	VideoCaptureVigraPix422(const char *name);
 protected:
	virtual void setsize();
	virtual int capture();
	void capfield(u_char *yp, u_char *up, int width, int height, int skip);
};

class VideoCaptureVigraPixCIF : public VideoCaptureVigraPix {
 public:
	VideoCaptureVigraPixCIF(const char *name);
 protected:
	virtual void setsize();
	virtual int capture();
};

class VigraPixDevice : public VideoDevice {
 public:
	VigraPixDevice(const char* clsname, const char* nickname,
		       const char* devname, int free);
	TclObject* create(int argc, const char*const* argv) {
		if (argc != 5)
                        abort();/*FIXME*/
		if (strcmp(argv[4], "422") == 0)
			return (new VideoCaptureVigraPix422(name_));
		else if (strcmp(argv[4], "cif") == 0)
			return (new VideoCaptureVigraPixCIF(name_));
		return (0);
	}
protected:
	const char* name_;
};

class VigraPixScanner {
    public:
	VigraPixScanner(const int n);
};

static VigraPixScanner find_vigrapix_devices(4);

VigraPixScanner::VigraPixScanner(const int n)
{
	char*	clsname_template  = "VideoCapture/VigraPix%d";
	char*	devname_template  = "/dev/vigrapix%d";
	char*	nickname_template = "VigraPix %d";

	for(int i = 0; i < n; i++) {
		char	*clsname  = new char[strlen(clsname_template)  + 3];
		char	*devname  = new char[strlen(devname_template)  + 3];
		char	*nickname = new char[strlen(nickname_template) + 3];

		sprintf(nickname, nickname_template, i + 1);
		sprintf(devname, devname_template, i);
		sprintf(clsname, clsname_template, i);
		if (access(devname, R_OK) == 0) {
			int fd = open(devname, O_RDONLY);
			if (fd < 0) {
				new VigraPixDevice(clsname, nickname, devname, 0);
			} else {
				(void)close(fd);
				new VigraPixDevice(clsname, nickname, devname, 1);
			}
		} else {
			delete nickname;
			delete devname;
		}
	}
}

VigraPixDevice::VigraPixDevice(const char* clsname, const char* nickname,
			       const char* devname, int free)
	: VideoDevice(clsname, nickname), name_(devname)
{
	if (free)
		attributes_ = "\
format { 411 422 } \
size { small large cif } \
port { Composite S-Video-1 S-Video-2}";
	else
		attributes_ = "disabled";
}

/*FIXME should we explicitly call VideoCapture()? */
VideoCaptureVigraPix::VideoCaptureVigraPix(const char* name) : VideoCapture()
{
	handle_ = vigrapix_open((char *)name);
	if (handle_ == 0) {
		status_ = -1;
		return;
	}
	port_ = VIGRAPIX_COMP;
	format_ = VIGRAPIX_NOLOCK;
	decimate_ = 2;
}

VideoCaptureVigraPix::~VideoCaptureVigraPix()
{
	if (handle_ != 0) {
		vigrapix_close(handle_);
		handle_ = 0;
	}
}

VideoCaptureVigraPix422::VideoCaptureVigraPix422(const char* name)
	: VideoCaptureVigraPix(name)
{
}

VideoCaptureVigraPixCIF::VideoCaptureVigraPixCIF(const char* name)
	: VideoCaptureVigraPix(name)
{
}

void VideoCaptureVigraPix422::setsize()
{
	if (format_ == VIGRAPIX_NOLOCK)
		format_ = vigrapix_get_format(handle_);

	int inw, inh, off;
	switch (format_) {

	default:
	case VIGRAPIX_NTSC:
		inw = 640;
		inh = 480;
		off = 15;
		break;

	case VIGRAPIX_PAL:
	case VIGRAPIX_SECAM:
		inw = 768;
		inh = 576;
		off = 19;
		break;
	}
	set_size_422(inw / decimate_, inh / decimate_);
	allocref();
	if (format_ != VIGRAPIX_NOLOCK)
		vigrapix_set_scaler(handle_, 0, off, inw, inh / 2,
				    outw_, outh_);
}

void VideoCaptureVigraPix::start()
{
	vigrapix_set_port(handle_, port_);
	setsize();
	VideoCapture::start();
}

int VideoCaptureVigraPix::command(int argc, const char*const* argv)
{
	if (argc == 3) {
		if (strcmp(argv[1], "decimate") == 0) {
			int dec = atoi(argv[2]);
			Tcl& tcl = Tcl::instance();
			if (dec <= 0) {
				tcl.resultf("%s: divide by zero", argv[0]);
				return (TCL_ERROR);
			}
			if (dec != decimate_) {
				decimate_ = dec;
				setsize();
			}
			return (TCL_OK);
		} else if (strcmp(argv[1], "port") == 0) {
			const char* name = argv[2];
			int p = VIGRAPIX_COMP;
			if (strcasecmp(name, "s-video-1") == 0)
				p = VIGRAPIX_SVIDEO1;
			else if (strcasecmp(name, "s-video-2") == 0)
				p = VIGRAPIX_SVIDEO2;
			if (p != port_) {
				port_ = p;
				vigrapix_set_port(handle_, p);
			}
			return (TCL_OK);
		} else if (strcmp(argv[1], "contrast") == 0) {
			contrast(atof(argv[2]));
			return (TCL_OK);
		}
	} else if (argc == 2) {
#ifdef notdef
		if (strcmp(argv[1], "normalize") == 0)
			normalize();
		else
#endif
		if (strcmp(argv[1], "format") == 0) {
			Tcl& tcl = Tcl::instance();
			switch (format_) {

			case VIGRAPIX_NOLOCK:
				tcl.result("no-lock");
				break;

			case VIGRAPIX_NTSC:
				tcl.result("ntsc");
				break;

			case VIGRAPIX_PAL:
				tcl.result("pal");
				break;

			case VIGRAPIX_SECAM:
				tcl.result("secam");
				break;

			default:
				tcl.result("");
				break;
			}
			return (TCL_OK);
		}
	}
	return (VideoCapture::command(argc, argv));
}

void VideoCaptureVigraPix422::capfield(u_char *yp, u_char *up,
				       int width, int height, int skip)
{
	volatile u_int *fifo = (u_int *)handle_->fifo;
	u_char *vp = up + (framesize_ >> 1);

	for (int h = height; h > 0; h--) {
		for (int w = width; w > 0; w -= 2) {
			/* Y0 U Y1 V */
			u_int v = *fifo;
			yp[0] = v >> 24;
			yp[1] = v >> 8;
			yp += 2;
			*up++ = (v >> 16) ^ 0x80;
			*vp++ = v ^ 0x80;
		}
		yp += skip;
		up += skip / 2;
		vp += skip / 2;
	}
}

/*
 * 4:2:2 transfer
 */
int VideoCaptureVigraPix422::capture()
{
	u_char* yp = frame_;
	u_char* up = yp + framesize_;

	if (handle_->interlaced) {
		int field;
		do {
			if ((field = vigrapix_next_field(handle_)) < 0)
				return -1;
		} while (field != 0);

		capfield(yp, up, outw_, outh_ / 2, outw_);

		do {
			if ((field = vigrapix_next_field(handle_)) < 0)
				return -1;
		} while (field != 1);

		yp = frame_ + outw_;
		up = frame_ + framesize_ + outw_ / 2;

		capfield(yp, up, outw_, outh_ / 2, outw_);
	} else {
		if (vigrapix_next_field(handle_) < 0)
			return (-1);

		capfield(yp, up, outw_, outh_, 0);
	}
	return (0);
}

/*
 * 4:1:1 transfer
 */
int VideoCaptureVigraPixCIF::capture()
{
	if (vigrapix_next_field(handle_) < 0)
		return (-1);

	volatile u_int *fifo = (u_int *)handle_->fifo;

	u_char* yp = frame_;
	u_char* up = yp + framesize_;
	u_char* vp = up + (framesize_ >> 2);

	if (format_ == VIGRAPIX_NTSC && decimate_ <= 2) {
		int off = vstart_ * outw_;
		yp += (off + hstart_) * 16;
		off = ((off >> 1) + hstart_) * 8;
		up += off;
		vp += off;
		int ypinc = outw_ - inw_;
		int h = (vstop_ - vstart_) * 8;
		while (--h >= 0) {
			int w;
			for (w = inw_ >> 1; --w >= 0; ) {
				/* Y0 U Y1 V */
				u_int v = *fifo;
				yp[0] = v >> 24;
				yp[1] = v >> 8;
				yp += 2;
				*up++ = (v >> 16) ^ 0x80;
				*vp++ = v ^ 0x80;
			}
			yp += ypinc;
			up += ypinc >> 1;
			vp += ypinc >> 1;
			for (w = inw_ >> 1; --w >= 0; ) {
				u_int v = *fifo;
				yp[0] = v >> 24;
				yp[1] = v >> 8;
				yp += 2;
			}
			yp += ypinc;
		}
	} else {
		int h;
		for (h = outh_ >> 1; --h >= 0; ) {
			int w;
			for (w = outw_ >> 1; --w >= 0; ) {
				/* Y0 U Y1 V */
				u_int v = *fifo;
				yp[0] = v >> 24;
				yp[1] = v >> 8;
				yp += 2;
				*up++ = (v >> 16) ^ 0x80;
				*vp++ = v ^ 0x80;
			}
			for (w = outw_ >> 1; --w >= 0; ) {
				u_int v = *fifo;
				yp[0] = v >> 24;
				yp[1] = v >> 8;
				yp += 2;
			}
		}
	}
	return (0);
}

void VideoCaptureVigraPix::grab()
{
	if (format_ == VIGRAPIX_NOLOCK) {
		int fmt = vigrapix_get_format(handle_);
		if (fmt != VIGRAPIX_NOLOCK)
			setsize();
	}
	if (format_ != VIGRAPIX_NOLOCK) {
		if (capture() < 0)
			return;
		suppress(frame_);
		saveblks(frame_);
	}
	YuvFrame f(media_ts(), frame_, crvec_, outw_, outh_);
	target_->recv(&f);
}

void VideoCaptureVigraPixCIF::setsize()
{
	if (format_ == VIGRAPIX_NOLOCK)
		format_ = vigrapix_get_format(handle_);

	int inw, inh, off;
	switch (format_) {

	default:
	case VIGRAPIX_NTSC:
		inw = 640;
		inh = 480;
		off = 15;
		break;

	case VIGRAPIX_PAL:
	case VIGRAPIX_SECAM:
		inw = 768;
		inh = 576;
		off = 19;
		break;
	}
	/* decimate is 2 or 4 */
	set_size_cif(inw / decimate_, inh / decimate_);
	allocref();

	if (format_ == VIGRAPIX_NTSC && decimate_ <= 2)
		vigrapix_set_scaler(handle_, 0, off, inw, inh / 2,
				    inw / decimate_, inh / decimate_);
	else if (format_ != VIGRAPIX_NOLOCK)
		vigrapix_set_scaler(handle_, 0, off, inw, inh / 2,
				    outw_, outh_);
}
