/*
** Copyright (C) 2000 Alan McIvor <alan@mcivor.gen.nz>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include "messenger.h"
#include "project.h"
#include "reporter.h"

char *longMonthNames[] =
{
  "January", "February", "March", "April", "May", "June",
  "July", "August", "September", "October", "November", "December"
};

FILE *Reporter::OpenOutputFile(char *filename)
{
    FILE *f;

    if ( filename == NULL )
	return NULL;

    if ( strcmp(filename, "-") == 0 )
	return stdout;

    f = fopen(filename, "w");
    if ( f == NULL )
	Error("Cannot open file [%s]", filename);

    return f;
}

void Project::LoadDays()
{
    int i;
    int dayNo = 1;
    time_t t = startTime;
    struct tm *tm;
    char buf[BUFLEN];
    const int fullDay = 24*60*60;	// 1 day in seconds

    if ( days[0].t != 0 )
	return;

    for ( i = 0 ; i < MAX_TIME ; ++i )
    {
	tm = localtime( &t );
	while ( tm->tm_wday == 0 || tm->tm_wday == 6 )
	{
	    t += fullDay;
	    ++ dayNo;
	    tm = localtime( &t );
	}
	days[i].t = t;
	days[i].num = dayNo;

	if ( date_format == NULL || strcasecmp(date_format, "calendar") == 0 )
	    sprintf(buf, "%04d.%s.%02d",
		    1900+tm->tm_year, monthNames[tm->tm_mon], tm->tm_mday);
	else if ( strcasecmp(date_format, "count") == 0 )
	    sprintf(buf, "%03d", dayNo);
	else if ( strcasecmp(date_format, "raw") == 0 )
	    sprintf(buf, "%03d", i);
	else if ( strcasecmp(date_format, "iso") == 0 )
	    sprintf(buf, "%04d-%02d-%02d",
		    1900+tm->tm_year, tm->tm_mon+1, tm->tm_mday);
	else
	    Error("Invalid date format: [%s]", date_format);

	days[i].s = strdup(buf);
	t += fullDay;
	++dayNo;
    }
}

/*
  Returns day index (in range 0 through MAX_TIME-1 inclusive)
  or -1 if not valid.
*/  
int Project::FindDay(char *name)
{
    int i;

    LoadDays();
    for ( i = 0 ; i < MAX_TIME ; ++i )
    {
	if ( strcasecmp(days[i].s, name) == 0 )
	{
	    return i;
	}
    }

    return -1;
}

int Reporter::Month(time_t t)
{
    struct tm *tm;

    tm = localtime( &t );
    return tm->tm_mon;
}

int Reporter::Year(time_t t)
{
    struct tm *tm;

    tm = localtime( &t );
    return tm->tm_year + 1900;
}

int Reporter::Mday(time_t t)
{
    struct tm *tm;

    tm = localtime( &t );
    return tm->tm_mday;
}

int Reporter::Wday(time_t t)
{
    struct tm *tm;

    tm = localtime( &t );
    return tm->tm_wday;
}


int UserTaskWorkedDays(TASK *t, RESOURCE *r)
{
    int tot = 0;
    for ( TaskTimeBlockIterator tb = t->begin_when(); tb != t->end_when(); tb++ )
    {
	if ( tb->type() == TimeBlock::WORK_DONE )
	{
	    tot += tb->finish() - tb->start() + 1;
	}
    }

    if ( tot == 0 && t->percent_complete() > 0 )
	tot = (int)(t->percent_complete() / 100.0 * (double)t->duration());

    return tot;
}

void Reporter::TextReport( Project *project, char *filename)
{
    MILESTONE *m;
    int j;
    int ndays;

    project->SortTasks( tg_sortbyresource );

    FILE * f = OpenOutputFile(filename);

    fprintf(f, "Time Frame:\n");
    fprintf(f, "==========:\n");
    fprintf(f, " Start Date: %s\n", project->sStartDay() );
    fprintf(f, "Finish Date: %s\n", project->sFinishDay() );
    fprintf(f,"\n\n\n");

    fprintf(f,"Task list:\n");
    fprintf(f,"=========:\n");
    for ( int i = 0; i < project->nTasks(); ++i )
    {
	TASK * t = project->sortedTask(i);
	if ( t->isVacation() )
	    continue;
	fprintf(f,"%s %s %-20s ",
		project->sDays(t->start()), project->sDays(t->finish()), t->assigned()->name());
	if ( task_ids )
	    fprintf(f,"%s %s\n", t->id(), t->name());
	else
	    fprintf(f,"%s\n", t->name());
	if ((t->bstart() != INVALIDDAYNO) && (t->bfinish() != INVALIDDAYNO)) 
	{
	    fprintf(f,"%s %s Baseline\n",
		    project->sDays(t->bstart()), project->sDays(t->bfinish()));
	}
	if ((t->astart() != INVALIDDAYNO) && (t->afinish() != INVALIDDAYNO))
	    // we may want to print even if afinish in still not defined
	{
	    fprintf(f,"%s %s Actual\n",
		    project->sDays(t->astart()), project->sDays(t->afinish()));
	}
    }
    fprintf(f,"\n\n\n");

    project->SortMilestones();
    fprintf(f, "Milestone List:\n");
    fprintf(f, "==============:\n");
    for ( int i = 0; i < project->nMilestones(); ++i )
    {
	m = project->sortedMilestone(i);
	
	fprintf(f, "%s ",
		project->sDays(m->day()));
	if ( milestone_ids )
	    fprintf(f, "%s ", m->id());
	fprintf(f, "%s\n", m->name());
	if ( m->bday() != INVALIDDAYNO )
	    fprintf(f, "%s Baseline\n", project->sDays(m->bday()));
    }
    fprintf(f,"\n\n\n");

    fprintf(f,"Resources and Task Assignment:\n");
    fprintf(f,"=============================:\n");
    for ( Project::RPLCI rl = project->beginResourceList(); 
	  rl != project->endResourceList() ; rl++ )
    {
	RESOURCE * r = *rl;
	if ( r->is_group )
	{
	    fprintf(f, "Group( ");
	    for ( Project::RPLCI rl2 = r->contains.begin(); rl2 != r->contains.end(); rl2++ )
		fprintf(f,"%s%s ", (*rl2)->name(), rl2 == r->contains.end()?"":",");
	    fprintf(f, ")\n");
	}
	else
	    fprintf(f,"%s\n",r->name());
	for ( int i = 0; i < project->nTasks(); ++i )
	{
	    TASK * t = project->sortedTask(i);
	    if( t->assigned() != r )
		continue;
	    if ( t->isVacation() )
		continue;
	    fprintf(f,"     %s %s ",
		    project->sDays(t->start()), project->sDays(t->finish()));
	    if ( task_ids )
		fprintf(f, "%s ", t->id());
	    fprintf(f, "%s\n", t->name());
	}
	fprintf(f,"\n");
    }
    fprintf(f,"\n\n\n");

    if ( dependencies )
    {
	fprintf(f,"Dependencies:\n");
	fprintf(f,"============:\n");
	for ( int i = 0; i < project->nTasks(); ++i )
	{
	    TASK * t = project->sortedTask(i);
	    if ( t->begin_depends() == t->end_depends() )
		continue;
	    fprintf(f, "%s %s ", project->sDays(t->start()), project->sDays(t->finish()));
	    if ( task_ids )
		fprintf(f, "%s ", t->id());
	    fprintf(f, " %s\n", t->name());
	    for ( TASK::PTRLIST::const_iterator pt = t->begin_depends(); 
		  pt != t->end_depends(); 
		  pt++ )
	    {
		TASK * tt = *pt; 
		fprintf(f, "     %s %s ",
			project->sDays(tt->start()), project->sDays(tt->finish()));
		if ( task_ids )
		    fprintf(f, "%s ", tt->id());
		fprintf(f, "%s\n", tt->name());
	    }
	    fprintf(f, "\n");
	}
    }
    fprintf(f,"\n\n\n");

    if ( vacations )
    {
	fprintf(f,"Vacations:\n");
	fprintf(f,"=========:\n");
	for ( Project::RPLCI rl = project->beginResourceList(); 
	      rl != project->endResourceList() ; rl++ )
	{
	    RESOURCE * r = *rl;
	    
	    if ( r->is_group )
	    {
		fprintf(f, "Group( ");
		for ( Project::RPLCI rl2 = r->contains.begin(); rl2 != r->contains.end(); rl2++ )
		    fprintf(f,"%s%s ", (*rl2)->name(), rl2 == r->contains.end() ? "" : "," );
		fprintf(f, ")\n");
	    }
	    else
		fprintf(f,"%s\n",r->name());
	    for ( int i = 0; i < project->nTasks(); ++i )
	    {
		TASK * t = project->sortedTask(i);
		if ( t->assigned() != r )
		    continue;
		if ( !t->isVacation() )
		    continue;
		ndays = 0;
		for ( j = t->start(); j <= t->finish(); ++j )
		{
		    if ( project->ResourceIsUsedForTask(r,t,j) )
			++ndays;
		}

		fprintf(f,"     %s %s %5d\n",
			project->sDays(t->start()), project->sDays(t->finish()), ndays);
	    }
	    fprintf(f,"\n");
	}
	fprintf(f,"\n\n\n");
    }

    if ( f != stdout )
	fclose(f);
}

void Reporter::XMLReport( Project *project, char *filename)
{
    MILESTONE *m;
    int j;
    int ndays;

    project->SortTasks( tg_sortbyresource );

    FILE * f = OpenOutputFile(filename);

    fprintf(f, "<?xml version=\"1.0\"?>\n");
    fprintf(f, "<project dateformat=\"%s\" startdate=\"%s\" enddate=\"%s\">\n",
        project->date_format, project->sStartDay(), project->sFinishDay() );

    for ( Project::IPLCI pi = project->begin_items() ; pi != project->end_items() ; pi++ )
        fprintf(f, "\t<item cost=\"%3.2f\">%s</item>\n", (*pi)->cost(), (*pi)->description().c_str() );

    fprintf(f,"\t<tasks>\n");
    for ( int i = 0; i < project->nTasks(); ++i )
    {
        TASK * t = project->sortedTask(i);

        if ( t->isVacation() )
            continue;

        double time_cost = t->timeCost();
        double item_cost = t->itemCost();

        fprintf(f, "\t\t<task start=\"%s\" finish=\"%s\" assigned=\"%s\" uid=\"%s\" name=\"%s\" ",
            project->sDays(t->start()), project->sDays(t->finish()), t->assigned()->id(),
            t->id(), t->name());
        if ((t->bstart() != INVALIDDAYNO) && (t->bfinish() != INVALIDDAYNO))
        {
            fprintf(f,"bstart=\"%s\" bfinish=\"%s\" ",
                project->sDays(t->bstart()), project->sDays(t->bfinish()));
        }

        if (t->astart() != INVALIDDAYNO)
            fprintf(f,"astart=\"%s\" ", project->sDays(t->astart()));

        if (t->afinish() != INVALIDDAYNO)
            fprintf(f,"afinish=\"%s\" ", project->sDays(t->afinish()));

        fprintf(f," completed=\"%3.2f%%\" timecost=\"%3.2f\" itemcost=\"%3.2f\">\n",
            t->percent_complete(), time_cost, item_cost);

	    for ( Project::IPLCI pi = t->begin_items() ; pi != t->end_items() ; pi++ )
            fprintf(f,"\t\t\t<item cost=\"%3.2f\">%s</item>\n",
                (*pi)->cost(),
                (*pi)->description().c_str());

        for ( TASK::PTRLIST::const_iterator pt = t->begin_depends();
          pt != t->end_depends();
          pt++ )
        {
            TASK * tt = *pt;
            fprintf(f, "\t\t\t<depends uid=\"%s\" name=\"%s\" />\n",
                tt->id(), tt->name());
        }

	    for ( NOTE_ITERATOR n = t->begin_notes(); n != t->end_notes(); n++ )
        {
            fprintf(f, "\t\t\t<note>%s</note>\n", n->c_str());
        }

        fprintf(f,"\t\t</task>\n");
    }
    fprintf(f,"\t</tasks>\n\n");

    if (project->nMilestones())
    {
        project->SortMilestones();
        fprintf(f, "\t<milestones>\n");
        for ( int i = 0; i < project->nMilestones(); ++i )
        {
            m = project->sortedMilestone(i);

            fprintf(f, "\t\t<milestone date=\"%s\" ", project->sDays(m->day()));
            if ( milestone_ids )
                fprintf(f, "uid=\"%s\" ", m->id());
            fprintf(f, "name=\"%s\" ", m->name());
            if ( m->bday() != INVALIDDAYNO )
                fprintf(f, "baseline=\"%s\" ", project->sDays(m->bday()));
            fprintf(f, "/>\n");
        }
        fprintf(f,"\t</milestones>\n");
    }

    fprintf(f,"\t<resources>\n");
    for ( Project::RPLCI rl = project->beginResourceList();
	  rl != project->endResourceList() ; rl++ )
    {
        RESOURCE * r = *rl;
        if ( r->is_group )
        {
            fprintf(f, "\t\t<group uid=\"%s\" name=\"%s\">\n", r->id(), r->name());
            for ( Project::RPLCI rl2 = r->contains.begin(); rl2 != r->contains.end(); rl2++ )
            fprintf(f,"\t\t\t<resource uid=\"%s\" name=\"%s\" efficiency=\"%.2f\" rate=\"%.2f\">\n",
                (*rl2)->id(), (*rl2)->name(), (*rl2)->efficiency(), (*rl2)->rate());
        }
        else
            fprintf(f,"\t\t<resource uid=\"%s\" name=\"%s\" efficiency=\"%.2f\" rate=\"%.2f\">\n",
                r->id(), r->name(), r->efficiency(), r->rate());
        for ( int i = 0; i < project->nTasks(); ++i )
        {
            TASK * t = project->sortedTask(i);
            if( t->assigned() != r )
            continue;
            if ( t->isVacation() )
            continue;
            fprintf(f,"\t\t\t<task uid=\"%s\" name=\"%s\" />\n", t->id(), t->name());
        }
	    for ( int i = 0; i < project->nTasks(); ++i )
	    {
            TASK * t = project->sortedTask(i);
            if ( t->assigned() != r )
                continue;
            if ( !t->isVacation() )
                continue;
            ndays = 0;
            for ( j = t->start(); j <= t->finish(); ++j )
            {
                if ( project->ResourceIsUsedForTask(r,t,j) )
                    ++ndays;
            }

            fprintf(f,"\t\t\t<vacation start=\"%s\" end=\"%s\" days=\"%d\" />\n",
                project->sDays(t->start()), project->sDays(t->finish()), ndays);
	    }
	    for ( NOTE_ITERATOR n = r->begin_notes(); n != r->end_notes(); n++ )
            fprintf(f, "\t\t\t<note>%s</note>\n", n->c_str());
        if ( r->is_group )
            fprintf(f,"\t\t</group>\n");
        else
            fprintf(f,"\t\t</resource>\n");
    }
	fprintf(f,"\t</resources>\n");

	fprintf(f,"</project>\n");
    if ( f != stdout )
        fclose(f);
}

void Reporter::HardSchedule( Project *project, char *filename)
{
    FILE *f;
    TASK *t;
    int i;

    project->SortTasks( tg_sortbyresource );

    f = OpenOutputFile(filename);

    for( i=0; i<project->nTasks(); ++i )
    {
	t = project->sortedTask(i);
	if ( t->isVacation() )
	    continue;
	fprintf(f,"start %s %s\n",    t->id(), project->sDays(t->start()));
	fprintf(f,"finish %s %s\n\n", t->id(), project->sDays(t->finish()));
    }

    if( f!=stdout )
	fclose(f);
}


char *Reporter::TeXFix(char *buf, const char *txt)
{
    char *ptr = buf;
    for( ; *txt!=0; ++txt )
    {
	if( *txt=='\\' )
	{
	    strcpy(ptr,"$\\backslash$");
	    ptr += strlen(ptr);
	}
	else if( *txt=='_' )
	{
	    strcpy(ptr,"\\_");
	    ptr += strlen(ptr);
	}
	else if( *txt=='%' )
	{
	    strcpy(ptr,"\\%");
	    ptr += strlen(ptr);
	}
	else if( *txt=='&' )
	{
	    strcpy(ptr,"\\&");
	    ptr += strlen(ptr);
	}
	else
	    *(ptr++) = *txt;
    }
    *ptr = 0;
    return buf;
}

void Reporter::TexReport( Project *project, char *filename)
{
    char buf1[BUFLEN], buf2[BUFLEN], buf3[BUFLEN];

    project->SortTasks( tg_sortbyresource );

    FILE * f = OpenOutputFile(filename);

    fprintf(f, "\\subsection{Time Frame}\n\\label{times}\n\n");
    fprintf(f, "\\begin{tabular}{rl}\n");
    fprintf(f, "Start Date: & %s \\\\\n", project->sStartDay());
    fprintf(f, "Finish Date: & %s \\\\\n", project->sFinishDay());
    fprintf(f, "\\end{tabular}\n\n");
    fprintf(f,"\\subsection{Task list}\n\n");
    fprintf(f,"\\begin{itemize}\n");
    for ( int i = 0 ; i < project->nTasks() ; ++i )
    {
	TASK * t = project->sortedTask(i);
	if ( t->isVacation() )
	    continue;
	if ( task_ids )
	{
	    fprintf(f, "\\item %s %s ",
		    TeXFix(buf3,t->id()),
		    TeXFix(buf1,t->name()));
	}
	else
	{
	    fprintf(f, "\\item %s ",
		    TeXFix(buf3,t->name()));
	}
	fprintf(f, "\\\\{\\em %s - %s; assigned to %s}\n",
		project->sDays(t->start()), project->sDays(t->finish()),
		TeXFix(buf2, t->assigned()->name()));
	if ( ( t->bstart() != INVALIDDAYNO ) 
	     && ( t->bfinish() != INVALIDDAYNO ) ) 
	{
	    fprintf(f, "\nBaseline\\\\{\\em %s - %s}\n", 
		    project->sDays(t->bstart()), 
		    project->sDays(t->bfinish()));
	}
	else if ( t->bstart() != INVALIDDAYNO )
	{
	    fprintf(f, "\nBaseline\\\\{\\em %s - }\n", 
		    project->sDays(t->bstart()) );
	}
	else if ( t->bfinish() != INVALIDDAYNO )
	{
	    fprintf(f, "\nBaseline\\\\{\\em - %s }\n", 
		    project->sDays(t->bfinish()) );
	}
	if ( ( t->astart() != INVALIDDAYNO ) 
	     && ( t->afinish() != INVALIDDAYNO ) ) 
	{
	    fprintf(f, "\nActual\\\\{\\em %s - %s}\n", 
		    project->sDays(t->astart()), 
		    project->sDays(t->afinish()));
	}
	else if ( t->astart() != INVALIDDAYNO )
	{
	    fprintf(f, "\nActual\\\\{\\em %s - }\n", 
		    project->sDays(t->astart()) );
	}
	else if ( t->afinish() != INVALIDDAYNO )
	{
	    fprintf(f, "\nActual\\\\{\\em - %s }\n", 
		    project->sDays(t->afinish()) );
	}
	if ( t->desc() != NULL && *t->desc() != 0 )
	    fprintf(f, "\n{\\em %s}\n", t->desc());
	if ( t->overrun() != NULL && *t->overrun() != 0 )
	{
	    char buf[4096];
	    fprintf(f, "\n\\framebox{\\em %s}\n", TeXFix(buf, t->overrun()));
	}
	if ( task_notes && ( t->begin_notes() != t->end_notes() ) )
	{
	    char buf[4096];
	    fprintf(f, "  \\begin{itemize}\n");
	    for ( NOTE_ITERATOR n = t->begin_notes(); n != t->end_notes(); n++ )
		fprintf(f, "  \\item %s\n", TeXFix(buf, n->c_str()));
	    fprintf(f, "  \\end{itemize}\n");
	}
    }
    fprintf(f, "\\end{itemize}\n\n\n");

    project->SortMilestones();

    fprintf(f, "\\subsection{Milestone List}\n\\label{milestones}\n\n");

    if ( project->nMilestones() > 0 )
    {
	fprintf(f, "\\begin{itemize}\n");

	for ( int i = 0; i < project->nMilestones(); ++i )
	{
	    MILESTONE * m = project->sortedMilestone(i);

	    fprintf(f, "\\item ");
	    if ( milestone_ids )
		fprintf(f, "%s %s\\\\",
			TeXFix(buf1,m->id()),
			TeXFix(buf2,m->name()));
	    else
		fprintf(f, "%s\\\\",
			TeXFix(buf2,m->name()));
	    fprintf(f, " \\emph{%s}\n\n",
		    project->sDays(m->day()));
	    if ( m->bday() != INVALIDDAYNO )
	    {
		fprintf(f, "Baseline\\\\\emph{%s}\n",
			project->sDays(m->bday()));
	    }
	    fprintf(f, "\n\n");
	}

	fprintf(f, "\\end{itemize}\n\n");
    }

    fprintf(f,"\\subsection{Resources And Task Assignment}\n\n");

    for( Project::RPLCI rl = project->beginResourceList() ; 
	 rl != project->endResourceList() ; rl++ )
    {
	int total_days = 0;
	int total_worked = 0;
	RESOURCE * r = *rl;

	fprintf(f, "\\subsubsection{");
	if ( r->is_group )
	{
	    for ( Project::RPLCI rl2 = r->contains.begin(); rl2 != r->contains.end(); rl2++ )
		fprintf(f,"%s%s ",
			TeXFix(buf1, (*rl2)->name()),
			rl2 == r->contains.end() ? "" : "," );
	}
	else
	    fprintf(f, "%s ", TeXFix(buf1, r->name()));
	fprintf(f,"}\n\n");

	fprintf(f,"\\begin{supertabular}{|p{20mm}|p{20mm}|r|r|p{75mm}|}\n");

	fprintf(f, "\\hline\n");
	fprintf(f, "{\\bf Start} & {\\bf Finish} & {\\bf Days} & {\\bf Done} & {\\bf Task} \\\\\n");
	fprintf(f, "\\hline\n");

	for ( int i = 0; i < project->nTasks(); ++i )
	{
	    int j = 0, ndays = 0, worked = 0;
	    TASK * t = project->sortedTask(i);
	    if( t->assigned() != r )
		continue;
	    if ( t->isVacation() )
		continue;

	    for ( j = t->start(); j <= t->finish(); ++j )
	    {
		if ( project->ResourceIsUsedForTask(r,t,j) )
		    ++ndays;
	    }

	    total_days += ndays;
	    worked = UserTaskWorkedDays(t, t->assigned()); 
	    total_worked += worked;

	    fprintf(f, " %s & %s & %d & %d & ",
		    project->sDays(t->start()), project->sDays(t->finish()), ndays, worked);
	    if ( task_ids )
		fprintf(f,"%s %s \\\\\n",
			TeXFix(buf2,t->id()),
			TeXFix(buf1,t->name()));
	    else
		fprintf(f,"%s \\\\\n",
			TeXFix(buf1,t->name()));
	    
	    /* QQQ */
	    if ( print_task_days )
	    {
		RESOURCE *r = t->assigned();
		fprintf(f,"&&&& ");
		for ( ResourceTimeBlockIterator tb = r->begin_booked() ; 
		      tb != r->end_booked() ; tb++ )
		{
		    if ( tb->task() == t )
		    {
			if ( tb->start() == tb->finish() )
			    fprintf(f, "%s, ", project->sDays(tb->start()));
			else
			    fprintf(f, "%s~-~%s, ", project->sDays(tb->start()), 
				    project->sDays(tb->finish()));
		    }
		}
		fprintf(f,"\\\\\n");
	    }
	}

	fprintf(f, "\\hline\n");
	fprintf(f, "\\multicolumn{5}{l}{\\bf Total: %d days scheduled, %d days completed.}\\\\\n",
		total_days, total_worked);
	fprintf(f, "\\end{supertabular}\n\n\n");

	if ( resource_notes && ( r->begin_notes() != r->end_notes() ) )
	{
	    char buf[4096];
	    fprintf(f, "  \\begin{itemize}\n");
	    for ( NOTE_ITERATOR n = r->begin_notes(); n != r->end_notes(); n++ )
		fprintf(f, "  \\item %s\n", TeXFix(buf, n->c_str()));
	    fprintf(f, "  \\end{itemize}\n");
	}
    }

    if ( dependencies )
    {
	fprintf(f, "\\subsection{Dependencies}\n\n");
	fprintf(f, "\\begin{itemize}\n");
	for ( int i = 0; i < project->nTasks(); ++i )
	{
	    TASK * t = project->sortedTask(i);
	    if ( t->begin_depends() == t->end_depends() )
		continue;
	    fprintf(f, "\\item (%s -- %s) %s\n",
		    project->sDays(t->start()), project->sDays(t->finish()), 
		    TeXFix(buf1,t->name()) );
	    fprintf(f,"   \\begin{itemize}\n");
	    for ( TASK::PTRLIST::const_iterator pt = t->begin_depends(); 
		  pt != t->end_depends(); 
		  pt++ )
	    {
		TASK * tt = *pt;
		fprintf(f, "   \\item (%s -- %s) %s\n",
			project->sDays(tt->start()), project->sDays(tt->finish()),
			TeXFix(buf1,tt->name()) );
	    }
	    fprintf(f, "   \\end{itemize}\n");
	    fprintf(f, "\n");
	}
	fprintf(f, "\\end{itemize}\n\n\n");
    }

    if ( vacations )
    {
	fprintf(f, "\\subsection{Vacations}\n\\label{vacations}\n\n");

	for ( Project::RPLCI rl = project->beginResourceList() ;
	      rl != project->endResourceList() ; rl++ )
	{
	    RESOURCE * r = *rl;

	    fprintf(f, "\\subsubsection{");
	    if ( r->is_group )
	    {
		for ( Project::RPLCI rl2 = r->contains.begin(); rl2 != r->contains.end(); rl2++ )
		    fprintf(f,"%s%s ",
			    TeXFix(buf1, (*rl2)->name()),
			    rl2 == r->contains.end() ? "" : "," );
	    }
	    else
		fprintf(f, "%s ", TeXFix(buf1, r->name()));
	    fprintf(f,"}\n\n");

	    fprintf(f,"\\begin{supertabular}{|p{0.8in}|p{0.8in}|r|}\n");
	    fprintf(f, "\\hline\n");
	    fprintf(f, "{\\bf Start} & {\\bf Finish} & {\\bf Days} \\\\\n");
	    fprintf(f, "\\hline\n");

	    for ( int i = 0; i < project->nTasks(); ++i )
	    {
		int j = 0, ndays = 0;
		TASK * t = project->sortedTask(i);
		if ( t->assigned() != r )
		    continue;
		if ( !t->isVacation() )
		    continue;

		for ( j = t->start(); j <= t->finish(); ++j )
		{
		    if ( project->ResourceIsUsedForTask(r,t,j) )
			++ndays;
		}

		fprintf(f, " %s &", project->sDays(t->start()));
		fprintf(f, " %s &", project->sDays(t->finish()));
		fprintf(f, " %d \\\\\n", ndays);
	    }

	    fprintf(f, "\\hline\n\\end{supertabular}\n\n");
	}
    }

    if ( f != stdout )
	fclose(f);
}

void Reporter::PrintResourcePeriod( Project *project,
				    FILE *f, RESOURCE *r, int dayNo, int lastDay,
				    REPORTTYPE rep_type)
{
    TASK *t;
    int nDays;
    char buf1[BUFLEN], buf2[BUFLEN];

    if ( MESSENGER::getGlobalDebug() )
	fprintf(f,"  %% PrintResourcePeriod(%s,%s / %d)\n",
		r->id(), project->sDays(dayNo), dayNo);

    if ( r->is_group==0 )
    {
	for ( ResourceTimeBlockIterator tb = r->begin_booked() ;
	      tb != r->end_booked() ; tb++ )
	{
	    if ( ! tb->overlap( dayNo, lastDay ) )
		continue;

	    if( MESSENGER::getGlobalDebug() )
		fprintf(f,"    %% PrintResourcePeriod(%s,%s) - *something* booked\n",r->id(),project->sDays(dayNo));

	    t = tb->task();
	    if ( t->isVacation() )
		continue;

	    nDays = 0;
	    for ( int i = dayNo; i <= lastDay; ++i )
	    {
		if( tb->start()<=i && tb->finish()>=i )
		{
		    ++nDays;
		    if( MESSENGER::getGlobalDebug() )
			fprintf(f,"      %% %s / %s / %s\n",r->id(),t->id(),project->sDays(i));
		}
	    }

	    if( nDays==0 )
	    {
		switch ( rep_type )
		{
		    case TXT_REPORT:
			fprintf(f,"  - Paused: %s - %s.\n", t->id(), t->name() );
			break;

		    case TEX_REPORT:
			fprintf(f,"      \\item Paused: {\\tt %s} - %s.\n",
				TeXFix(buf1,t->id()), TeXFix(buf2,t->name()) );
			break;

		    case HTML_REPORT:
			fprintf(f,"<li> Paused: <tt>%s</tt>  - %s.\n",
				HTMLFix(buf1,t->id()), HTMLFix(buf2,t->name()) );
			break;

		    default:
			fprintf(stderr, "Unhandled case 5\n");
			exit(1);
		}
	    }
	    else
	    {
		int nworked;
		switch ( rep_type )
		{
		    case TXT_REPORT:
			fprintf(f,"  - %d days - %s - %s.\n", nDays, t->id(), t->name() );
			break;

		    case TEX_REPORT:
			fprintf(f,"      \\item %d days - {\\tt %s} - %s.\n",
				nDays, TeXFix(buf1,t->id()), TeXFix(buf2,t->name()) );
			break;

		    case HTML_REPORT:
			fprintf(f,"<li> %d days - <tt>%s</tt> - %s.\n",
				nDays, HTMLFix(buf1,t->id()), HTMLFix(buf2,t->name()) );
			break;

		    default:
			fprintf(stderr, "Unhandled case 6\n");
			exit(1);
		}
		nworked = UserTaskWorkedDays(t,t->assigned()); 
		if( nworked>0 )
		{
		    switch ( rep_type )
		    {
			case TXT_REPORT:
			    fprintf(f,"   (%d days completed.)\n",nworked);
			    break;

			case TEX_REPORT:
			    fprintf(f,"         \\\\(%d days completed.)\n",nworked);
			    break;

			case HTML_REPORT:
			    fprintf(f,"<br>(%d days completed.)\n",nworked);
			    break;

			default:
			    fprintf(stderr, "Unhandled case 7\n");
			    exit(1);
		    }
		}
	    }
	}
    }

    for ( Project::RPLCI rl = r->belongs_to.begin() ; 
	  rl != r->belongs_to.end() ; rl++ )
	PrintResourcePeriod(project, f, *rl, dayNo, lastDay, rep_type);
}


void Reporter::PrintPeriod(Project *project,
			   FILE *f, int dayNo, int lastDay, char *weekName,
			   REPORTTYPE rep_type)
{
    bool something = false;
    char buf1[BUFLEN];

    Debug("PrintPeriod(%p,%d,%d,%s,%d)",f,dayNo,lastDay,weekName,rep_type);

    for ( Project::TPLCI pt = project->beginTaskList(); 
	 pt != project->endTaskList(); 
	 pt++ )
    {
	if ( ! (*pt)->isVacation() && (*pt)->isActiveDuring(dayNo, lastDay) )
	{
	    something = true;
	    break;
	}
    }
    if ( !something )
      {
	Debug("Nothing in %s",weekName);
	return;
      }

    switch ( rep_type )
    {
	case TXT_REPORT:
	    fprintf(f,"\n\n----------------------------------------\n\n");
	    fprintf(f,"%s\n",weekName);
	    fprintf(f,"\n");
	    break;

	case TEX_REPORT:
	    fprintf(f,"\\clearpage\n");
	    fprintf(f,"\\subsection{%s}\n",weekName);
	    fprintf(f,"\\begin{itemize}\n");
	    break;

	case HTML_REPORT:
	    fprintf(f,"\n");
	    fprintf(f,"<h1>%s</h1>\n",weekName);
	    fprintf(f,"<ul>\n");
	    break;

	default:
	    fprintf(stderr, "Unhandled case 1\n");
	    exit(1);
    }

    something = false;
    for ( Project::RPLCI rl = project->beginResourceList() ; 
	  rl != project->endResourceList() ; rl++ )
    {
	int r_used = 0;

	RESOURCE * r = *rl;

	if( r->is_group )
	    continue;

	for ( ResourceTimeBlockIterator tb = r->begin_booked() ; 
	      tb != r->end_booked() ; tb++ )
	{
	    if ( tb->overlap( dayNo, lastDay) && !tb->task()->isVacation() )
	    {
		r_used = 1;
		break;
	    }
	}
	if ( r_used==0 )
	    continue;

	something = true;

	switch ( rep_type )
	{
	    case TXT_REPORT:
		fprintf(f,"* %s:\n", r->name() );
		PrintResourcePeriod(project, f, r, dayNo, lastDay, rep_type);
		break;

	    case TEX_REPORT:
		fprintf(f,"\\item {\\bf %s}\n", TeXFix(buf1,r->name() ) );
		fprintf(f,"      \\begin{itemize}\n");
		PrintResourcePeriod(project, f, r, dayNo, lastDay, rep_type);
		fprintf(f,"      \\end{itemize}\n\n");
		break;

	    case HTML_REPORT:
		fprintf(f,"<li> <strong>%s</strong>\n", HTMLFix(buf1,r->name()) );
		fprintf(f,"<ul>\n");
		PrintResourcePeriod(project, f, r, dayNo, lastDay, rep_type);
		fprintf(f,"</ul>\n\n");
		break;

	    default:
		fprintf(stderr, "Unhandled case 2\n");
		exit(1);
	}
    
    }
  
    if ( ! something )
    {
	switch ( rep_type )
	{
	    case TXT_REPORT:
		fprintf(f,"* Nothing in %s\n",weekName);
		break;

	    case TEX_REPORT:
		fprintf(f,"\\item Nothing in %s\n",weekName);
		break;

	    case HTML_REPORT:
		fprintf(f,"<li> Nothing in %s\n",weekName);
		break;

	    default:
		fprintf(stderr, "Unhandled case 3\n");
		exit(1);
	}
    }
  
    switch ( rep_type )
    {
	case TXT_REPORT:
	    fprintf(f,"\n\n\n");
	    break;

	case TEX_REPORT:
	    fprintf(f,"\\end{itemize}\n\n\n");
	    break;

	case HTML_REPORT:
	    fprintf(f,"</ul>\n\n\n");
	    break;

	default:
	    fprintf(stderr, "Unhandled case \n");
	    exit(1);
    }
}
  
  
void Reporter::PeriodicReport( Project * project,
			       Pperfunc IsPeriod,
			       char *filename,
			       REPORTTYPE rep_type)
{
    int dayNo;
    int lastDay=0;
    char weekName[BUFLEN];

    project->SortTasks( tg_sortbyresource );

    FILE * f = OpenOutputFile(filename);
    switch ( rep_type )
    {
	case TXT_REPORT:
	    break;

	case TEX_REPORT:
	    break;

	case HTML_REPORT:
	    fprintf(f,"<html>\n<body>\n\n");
	    break;

	default:
	    fprintf(stderr, "Unhandled case 4\n");
	    exit(1);
    }

    if( MESSENGER::getGlobalDebug() )
	fprintf(f,"%% time period / people / tasks report\n");

    for( dayNo=0; dayNo<MAX_TIME; ++dayNo )
    {
	if( (lastDay=(this->*IsPeriod)(project, dayNo, weekName)) )
	{
	    PrintPeriod( project, f, dayNo, lastDay, weekName, rep_type );
	}
    }

    switch ( rep_type )
    {
	case TXT_REPORT:
	    break;

	case TEX_REPORT:
	    break;

	case HTML_REPORT:
	    fprintf(f,"\n\n</body>\n</html>\n");
	    break;

	default:
	    fprintf(stderr, "Unhandled case 4\n");
	    exit(1);
    }

    if( f!=stdout )
	fclose(f);
}


int Reporter::IsWeek(Project *project, int dayNo, char *name)
{
    if( dayNo==0 )
    {
	sprintf(name,"Week of %s",project->sDays(dayNo));
	while( Wday(project->tDays(dayNo))!=5 )
	    ++dayNo;
	return dayNo;
    }

    if( Wday(project->tDays(dayNo))==1 )
    {
	sprintf(name,"Week of %s",project->sDays(dayNo));
	return dayNo+4;
    }

    *name = 0;
    return 0;
}

void Reporter::WeeklyReport(Project *project, char *filename, REPORTTYPE rep_type)
{
    PeriodicReport(project, &Reporter::IsWeek, filename, rep_type);
}


int Reporter::IsMonth(Project *project, int dayNo, char *name)
{
    static int lastMonth = -1;
    int thisMonth;

    thisMonth = Month( project->tDays(dayNo) );

    if( lastMonth != thisMonth )
    {
	lastMonth = thisMonth;
	sprintf(name,"Month of %s, %d",longMonthNames[thisMonth],
		Year(project->tDays(dayNo)));
	while( Month(project->tDays(dayNo))==thisMonth )
	    ++dayNo;
	return dayNo-1;
    }

    *name = 0;
    return 0;
}

void Reporter::MonthlyReport(Project *project, char *filename, REPORTTYPE rep_type)
{
    PeriodicReport(project, &Reporter::IsMonth,filename,rep_type);
}



char *Reporter::HTMLFix(char *buf, const char *txt)
{
    char *ptr = buf;
    for( ; *txt!=0; ++txt )
    {
	if ( *txt == '&' )
	{
	    strcpy(ptr, "&amp;");
	    ptr += 5;
	}
	else if ( *txt == '<' )
	{
	    strcpy(ptr, "&lt;");
	    ptr += 4;
	}
	else if ( *txt == '>' )
	{
	    strcpy(ptr, "&gt;");
	    ptr += 4;
	}
	else
	{
	    *(ptr++) = *txt;
	}
    }
    *ptr = 0;
    return buf;
}

void Reporter::HTMLReport(Project *project, char *filename)
{
    char buf1[BUFLEN], buf2[BUFLEN], buf3[BUFLEN];

    project->SortTasks( tg_sortbyresource );

    FILE * f = OpenOutputFile(filename);

    fprintf(f, "<html>\n<body>\n\n");

    fprintf(f, "<h1><a name=\"times\">Time Frame</a></h1>\n");
    fprintf(f, "<table BORDER=0 CELLPADDING=5>\n");
    fprintf(f, "<tr>\n<td align=right>Start Date:</td><td>%s</td>\n</tr>\n",
	    project->sStartDay());
    fprintf(f, "<tr>\n<td align=right>Finish Date:</td><td>%s</td>\n</tr>\n",
	    project->sFinishDay());
    fprintf(f, "</table>\n\n");

    fprintf(f, "<h1><a name=\"tasks\">Task List</a></h1>\n");
    fprintf(f, "<table BORDER=1 CELLPADDING=2>\n");
    fprintf(f, "<tr><th>Task</th><th>Start</th><th>Finish</th>");
    fprintf(f, "<th>Resource</th><th>Description</th>");
    fprintf(f, "<th COLSPAN=2>Baseline</th><th COLSPAN=2>Actual</th></tr>\n");
    for ( int i = 0 ; i < project->nTasks(); ++i )
    {
	TASK * t = project->sortedTask(i);
	if ( t->isVacation() )
	    continue;

	fprintf(f, "<tr>\n");
	if ( task_ids )
	    fprintf(f, "<td>%s %s</td>\n",
		    HTMLFix(buf3,t->id()),
		    HTMLFix(buf1,t->name()));
	else
	    fprintf(f, "<td>%s</td>\n",
		    HTMLFix(buf1,t->name()));
	fprintf(f, "<td>%s</td><td>%s</td><td>%s</td>\n",
		project->sDays(t->start()), project->sDays(t->finish()),
		HTMLFix(buf2,t->assigned()->name()));
	if (( t->desc() != NULL ) && ( *t->desc() != 0 ))
	{
	    fprintf(f, "<td>%s</td>\n",t->desc());
	}
	else
	{
	    fprintf(f, "<td>&nbsp;</td>\n");
	}

	if ((t->bstart() != INVALIDDAYNO) && (t->bfinish() != INVALIDDAYNO)) 
	{
	    fprintf(f, "<td>%s</td><td>%s</td>\n",
		    project->sDays(t->bstart()), project->sDays(t->bfinish()));
	}
	else if (t->bstart() != INVALIDDAYNO)
	{
	    fprintf(f, "<td>%s</td><td>&nbsp;</td>\n",
		    project->sDays(t->bstart()));
	}
	else if (t->bfinish() != INVALIDDAYNO)
	{
	    fprintf(f, "<td>&nbsp;</td><td>%s</td>\n",
		    project->sDays(t->bfinish()));
	}
	else
	{
	    fprintf(f, "<td>&nbsp;</td><td>&nbsp;</td>\n");
	}
	if ((t->astart() != INVALIDDAYNO) && (t->afinish() != INVALIDDAYNO)) 
	{
	    fprintf(f, "<td>%s</td><td>%s</td>\n",
		    project->sDays(t->astart()), project->sDays(t->afinish()));
	}
	else if (t->astart() != INVALIDDAYNO)
	{
	    fprintf(f, "<td>%s</td><td>&nbsp;</td>\n",
		    project->sDays(t->astart()));
	}
	else if (t->afinish() != INVALIDDAYNO)
	{
	    fprintf(f, "<td>&nbsp;</td><td>%s</td>\n",
		    project->sDays(t->afinish()));
	}
	else
	{
	    fprintf(f, "<td>&nbsp;</td><td>&nbsp;</td>\n");
	}
	fprintf(f, "</tr>\n");
	if (( t->overrun()!=NULL ) && ( *t->overrun()!=0 ))
	{
	    char buf[4096];
	    fprintf(f,
		    "<tr><td COLSPAN=9><strong>%s</strong></td></tr>\n",
		    HTMLFix(buf,t->overrun()));
	}
	if ( task_notes && ( t->begin_notes() != t->end_notes() ) )
	{
	    char buf[4096];
	    fprintf(f, "<tr><td>&nbsp;</td><td COLSPAN=8>\n");
	    for ( NOTE_ITERATOR n = t->begin_notes(); n != t->end_notes(); n++ )
		fprintf(f, "%s<br>\n", HTMLFix(buf,n->c_str()));
	    fprintf(f, "</td></tr>\n");
	}
    }
    fprintf(f, "</table>\n\n");

    project->SortMilestones();

    fprintf(f, "<h1><a name=\"milestones\">Milestone List</a></h1>\n\n");

    fprintf(f, "<table BORDER=1 CELLPADDING=2>\n");
    fprintf(f, "<tr>");
    fprintf(f, "<th>Milestone</th><th>Date</th>");
    fprintf(f, "<th>Baseline</th>");
    fprintf(f, "</tr>\n");

    for ( int i = 0; i < project->nMilestones(); ++i )
    {
	MILESTONE * m = project->sortedMilestone(i);

	fprintf(f, "<tr>\n");
	if ( milestone_ids )
	    fprintf(f, "<td>%s %s</td>",
		    HTMLFix(buf1,m->id()),
		    HTMLFix(buf2,m->name()));
	else
	    fprintf(f, "<td>%s</td>",
		    HTMLFix(buf2,m->name()));
	fprintf(f, "<td>%s</td>",
		project->sDays(m->day()));
	if ( m->bday() != INVALIDDAYNO )
	{
	    fprintf(f, "<td>%s</td>",
		    project->sDays(m->bday()));
	}
	else
	{
	    fprintf(f, "<td>&nbsp;</td>");
	}
	fprintf(f, "\n</tr>\n");
    }

    fprintf(f, "</table>\n\n");

    fprintf(f, "<h1><a name=\"assignments\">Resources And Task Assignment</a></h1>\n\n");

    for ( Project::RPLCI rl = project->beginResourceList() ; 
	  rl != project->endResourceList() ; rl++ )
    {
	int total_days = 0;
	int total_worked = 0;
	RESOURCE * r = *rl;

	fprintf(f, "<h2>");
	if ( r->is_group )
	{
	    for ( Project::RPLCI rl2 = r->contains.begin(); rl2 != r->contains.end(); rl2++ )
		fprintf(f, "%s%s ",
			HTMLFix(buf1,(*rl2)->name()),
			 rl2 == r->contains.end()?"":"," );
	}
	else
	    fprintf(f, "%s ",HTMLFix(buf1,r->name()));
	fprintf(f, "</h2>\n\n");

	fprintf(f, "<table BORDER=1 CELLPADDING=2>\n");

	fprintf(f, "<tr><th>Start</th> <th>Finish</th> <th>Days</th> <th>Done</th> <th>Task</th>");
	if ( print_task_days )
	    fprintf(f, "<th>Dates</th>");
	fprintf(f, "</tr>\n");

	for ( int i = 0; i < project->nTasks(); ++i )
	{
	    int j = 0, ndays = 0, worked = 0;
	    TASK * t = project->sortedTask(i);
	    if ( t->assigned() != r )
		continue;
	    if ( t->isVacation() )
		continue;

	    for ( j = t->start(); j <= t->finish(); ++j )
	    {
		if ( project->ResourceIsUsedForTask(r,t,j) )
		    ++ndays;
	    }

	    total_days += ndays;
	    worked = UserTaskWorkedDays(t, t->assigned()); 
	    total_worked += worked;

	    fprintf(f, "<tr><td>%s</td> <td>%s</td> <td ALIGN=\"RIGHT\">%d</td> <td ALIGN=\"RIGHT\">%d</td> ",
		    project->sDays(t->start()), project->sDays(t->finish()), ndays, worked);
	    if ( task_ids )
		fprintf(f, "<td>%s %s</td>\n",
			HTMLFix(buf2,t->id()), HTMLFix(buf1,t->name()));
	    else
		fprintf(f, "<td>%s</td>\n",
			HTMLFix(buf1,t->name()));
	    
	    /* QQQ */
	    if ( print_task_days )
	    {
		RESOURCE *r = t->assigned();
		fprintf(f, "<td>");
		for ( ResourceTimeBlockIterator tb = r->begin_booked() ; 
		      tb != r->end_booked() ; 
		      tb++ )
		{
		    if( tb->task() == t )
		    {
			if ( tb->start() == tb->finish() )
			    fprintf(f, "%s, ", project->sDays(tb->start()));
			else
			    fprintf(f, "%s&nbsp;-&nbsp;%s, ",
				    project->sDays(tb->start()), 
				    project->sDays(tb->finish()));
		    }
		}
		fprintf(f, "</td></tr>\n");
	    }
	    fprintf(f, "</tr>\n");
	}

	if ( print_task_days )
	    fprintf(f, "<tr><td colspan=6>");
	else
	    fprintf(f, "<tr><td colspan=5>");
	fprintf(f,
		"<strong>Total: %d days scheduled, %d days completed.</strong></td></tr>\n",
		total_days, total_worked);
	fprintf(f, "</table>\n");

	if ( resource_notes && ( r->begin_notes() != r->end_notes() ) )
	{
	    char buf[4096];
	    fprintf(f, "  <ul>\n");
	    for ( NOTE_ITERATOR n = r->begin_notes(); n != r->end_notes(); n++ )
		fprintf(f, "  <li> %s\n", HTMLFix(buf,n->c_str()));
	    fprintf(f, "  </ul>\n");
	}
    }

    if ( dependencies )
    {
	fprintf(f, "<h1>Dependencies</h1>\n\n");
	fprintf(f, "<ul>\n");
	for ( int i = 0; i < project->nTasks(); ++i )
	{
	    TASK * t = project->sortedTask(i);
	    if ( t->begin_depends() == t->end_depends() )
		continue;
	    fprintf(f, "<li> (%s&nbsp;-&nbsp;%s) ",
		    project->sDays(t->start()), project->sDays(t->finish()));
	    if ( task_ids )
		fprintf(f, "%s ", HTMLFix(buf2,t->id()));
	    fprintf(f, "%s\n", HTMLFix(buf1,t->name()));
	    fprintf(f, "   <ul>\n");
	    for ( Project::TPLCI pt = t->begin_depends(); pt != t->end_depends(); pt++ )
	    {
		TASK * tt = *pt;
		fprintf(f, "   <li> (%s&nbsp;-&nbsp;%s) ",
			project->sDays(tt->start()), project->sDays(tt->finish()));
		if ( task_ids )
		    fprintf(f, "%s ", HTMLFix(buf2,tt->id()));
		fprintf(f, "%s\n", HTMLFix(buf1,tt->name()));
	    }
	    fprintf(f, "   </ul>\n");
	    fprintf(f, "\n");
	}
	fprintf(f, "</ul>\n\n\n");
    }

    if ( vacations )
    {
	fprintf(f, "<h1>Vacations</h1>\n\n");

	for ( Project::RPLCI rl = project->beginResourceList() ; 
	      rl != project->endResourceList() ; rl++ )
	{
	    RESOURCE * r = *rl;

	    fprintf(f, "<h2>");
	    if ( r->is_group )
	    {
		for ( Project::RPLCI rl2 = r->contains.begin(); rl2 != r->contains.end(); rl2++ )
		    fprintf(f, "%s%s ",
			    HTMLFix(buf1,(*rl2)->name()),
			     rl2 == r->contains.end()?"":"," );
	    }
	    else
		fprintf(f, "%s ",HTMLFix(buf1,r->name()));
	    fprintf(f, "</h2>\n\n");

	    fprintf(f, "<table BORDER=1 CELLPADDING=2>\n");

	    fprintf(f, "<tr><th>Start</th> <th>Finish</th> <th>Days</th>");
	    fprintf(f, "</tr>\n");

	    for ( int i = 0; i < project->nTasks(); ++i )
	    {
		TASK * t = project->sortedTask(i);
		if ( t->assigned() != r )
		    continue;
		if ( !t->isVacation() )
		    continue;

		int ndays = 0;
		for ( int j = t->start(); j <= t->finish(); ++j )
		{
		    if ( project->ResourceIsUsedForTask(r,t,j) )
			++ndays;
		}

		fprintf(f, "<tr><td>%s</td>", project->sDays(t->start()));
		fprintf(f, " <td>%s</td>", project->sDays(t->finish()));
		fprintf(f, "<td ALIGN=\"RIGHT\">%d</td></tr>\n", ndays);
	    }

	    fprintf(f, "</table>\n");
	}
    }

    fprintf(f, "\n\n</body>\n</html>\n");

    if ( f != stdout )
	fclose(f);
}

