#! /usr/bin/env python
# -*- coding: iso-8859-1 -*-

# build-documentation.py --- Build the documentation for PyXMMS
# Copyright (c) 2003 Florent Rougon
#
# This file is part of PyXMMS.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 dated June, 1991.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING. If not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
# MA 02111-1307, USA.

import sys, os, os.path, distutils.util

doc_dir = "doc"


class error(Exception):
    """Base class for exceptions in this module."""
    def __init__(self, message=None):
        self.message = message
    def __str__(self):
        return "<%s: %s>" % (self.__class__.__name__, self.message)
    def complete_message(self):
        if self.message:
            return "%s: %s" % (self.ExceptionShortDescription, self.message)
        else:
            return "%s" % self.ExceptionShortDescription
    ExceptionShortDescription = "PyXMMS build-documentation.py generic " \
                                "exception"

class ProgramDoesNotLookBuilt(error):
    """Exception raised when PyXMMS does not look correctly built for the running Python."""
    ExceptionShortDescription = "PyXMMS does not look correctly built for " \
                                "this Python installation"

class DocDirectoryCannotBeCleaned(error):
    """Exception raised when the documentation directory cannot be cleaned before generating the new documentation."""
    ExceptionShortDescription = "The documentation directory (%s) cannot " \
                                "be cleaned" % doc_dir


def check_if_program_looks_built(build_dir):
    package_dir = os.path.join(build_dir, "xmms")
    files = ["__init__.py", "common.py", "control.py", "_xmmscontrol.so"]
    if sys.hexversion >= 0x02020000:
        files.extend(("config.py", "_xmmsconfig.so"))

    if not os.path.isdir(package_dir):
        raise ProgramDoesNotLookBuilt("%s is not an existing directory"
                                      % package_dir)
    for f in files:
        file_path = os.path.join(package_dir, f)
        if not os.path.isfile(file_path):
            raise ProgramDoesNotLookBuilt("%s is not an existing regular file"
                                          % file_path)


def build_documentation(build_dir):
    # Modules that are available to all Python versions supported by PyXMMS.
    all_pyversions_modules = ["common", "control"]
    # Modules that are only available on Python >= 2.2
    recent_pyversions_modules = ["config"]
    # Generate the list of modules to feed to pydoc
    modules = all_pyversions_modules[:]
    if sys.hexversion >= 0x02020000:
        modules.extend(recent_pyversions_modules)
    else:
        sys.stderr.write("Warning: this Python is 2.1 or earlier, so only "
                         "part of the PyXMMS documentation will be built.\n")
        
    base_dir = os.getcwd()

    # First, erase the existing documentation (especially useful if there is a
    # xmms.config.html left and we are building the documentation with
    # Python2.1).
    os.chdir(doc_dir)
    for module in all_pyversions_modules + recent_pyversions_modules:
        doc_filename = "xmms.%s.html" % module
        if os.path.isfile(doc_filename):
            os.unlink(doc_filename)
            # I follow the message style of pydoc to have an homogeneous
            # output...
            print "removed %s" % os.path.join(doc_dir, doc_filename)
        elif os.path.exists(doc_filename):
            raise DocDirectoryCannotBeCleaned('contains a "%s" which is not '
                                              'a regular file (I dare not '
                                              'erase it myself)'
                                              % doc_filename)

    os.chdir(os.path.join(base_dir, build_dir))
    for module in modules:
        qualified_module_name = "xmms.%s" % module
        # I fear that pydoc messes with namespaces (or would be fooled by the
        # current namespaces and clutter the generated docs), so I execute it
        # in a separate scope. Also, note that spawning a "pydoc" process
        # would not work since it would not necessarily use the right Python
        # version.
        d = {}
        sys.argv = ["pydoc", "-w", qualified_module_name]
        exec "import pydoc; pydoc.cli()" in d
        doc_filename = "%s.html" % qualified_module_name
        os.rename(os.path.join(".", doc_filename),
                  os.path.join("..", "..", doc_dir, doc_filename))

    # Remove the .pyc files created by pydoc when importing the various
    # modules
    for module in modules + ["__init__"]:
        os.unlink(os.path.join("xmms", "%s.pyc" % module))


def main():
    pyversion_str = "%u.%u" % ((sys.hexversion & 0xff000000L) >> 24,
                               (sys.hexversion & 0x00ff0000L) >> 16)
    build_dir = os.path.join("build",
                             "lib.%s-%s" % (distutils.util.get_platform(),
                                            pyversion_str))
    try:
        check_if_program_looks_built(build_dir)
        build_documentation(build_dir)
    except OSError, v:
        sys.exit("Error: %s" % str(v))
    except error, v:
        sys.exit("Error: %s" % v.complete_message())
        
    sys.exit(0)

if __name__ == "__main__": main()
