// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_CERT_X509_CERTIFICATE_KNOWN_ROOTS_WIN_H_
#define NET_CERT_X509_CERTIFICATE_KNOWN_ROOTS_WIN_H_

#include <stdint.h>

// This is a union of Microsoft trust roots over time, from 01 January
// 2007 through 24 November 2015.
//
// Extracted from
// http://www.download.windowsupdate.com/msdownload/update/v3/
// static/trustedr/en/authrootstl.cab
//
// Note that these *are not* trust anchors for Chromium. They are only used to
// distinguish `real' root CAs from roots that were user-installed.
static uint8_t kKnownRootCertSHA256Hashes[][32] = {
    // C=US, O=Network Solutions L.L.C., CN=Network Solutions Certificate
    // Authority
    {0x00, 0x16, 0x86, 0xCD, 0x18, 0x1F, 0x83, 0xA1, 0xB1, 0x21, 0x7D, 0x30,
     0x5B, 0x36, 0x5C, 0x41, 0xE3, 0x47, 0x0A, 0x78, 0xA1, 0xD3, 0x7B, 0x13,
     0x4A, 0x98, 0xCD, 0x54, 0x7B, 0x92, 0xDA, 0xB3},
    // C=SI, O=POSTA, OU=POSTArCA
    {0x00, 0x7E, 0x45, 0x2F, 0xD5, 0xCF, 0x83, 0x89, 0x46, 0x69, 0x6D, 0xFE,
     0x37, 0xA2, 0xDB, 0x2E, 0xF3, 0x99, 0x14, 0x36, 0xD2, 0x7B, 0xCB, 0xAB,
     0x45, 0x92, 0x20, 0x53, 0xC1, 0x5A, 0x87, 0xA8},
    // C=ES, O=Agencia Notarial de Certificacion S.L. Unipersonal - CIF
    // B83395988, CN=ANCERT Certificados Notariales
    {0x00, 0xAB, 0x44, 0x4A, 0xBD, 0x6B, 0xDB, 0xA3, 0x3D, 0xA8, 0xDE, 0x56,
     0x9A, 0xC4, 0xEC, 0xDE, 0x32, 0x6D, 0x1B, 0xE1, 0xA6, 0x14, 0x42, 0xD5,
     0xEE, 0xC3, 0x97, 0x5A, 0x0C, 0x24, 0x3F, 0x04},
    // C=US, O=Entrust, Inc., OU=See www.entrust.net/legal-terms, OU=(c) 2012
    // Entrust, Inc. - for authorized use only, CN=Entrust Root Certification
    // Authority - EC1
    {0x02, 0xED, 0x0E, 0xB2, 0x8C, 0x14, 0xDA, 0x45, 0x16, 0x5C, 0x56, 0x67,
     0x91, 0x70, 0x0D, 0x64, 0x51, 0xD7, 0xFB, 0x56, 0xF0, 0xB2, 0xAB, 0x1D,
     0x3B, 0x8E, 0xB0, 0x70, 0xE5, 0x6E, 0xDF, 0xF5},
    // C=US, O=Wells Fargo, OU=Wells Fargo Certification Authority, CN=Wells
    // Fargo Root Certificate Authority
    {0x03, 0x45, 0x8B, 0x6A, 0xBE, 0xEC, 0xC2, 0x14, 0x95, 0x3D, 0x97, 0x14,
     0x9A, 0xF4, 0x53, 0x91, 0x69, 0x1D, 0xE9, 0xF9, 0xCD, 0xCC, 0x26, 0x47,
     0x86, 0x3A, 0x3D, 0x67, 0xC9, 0x5C, 0x24, 0x3B},
    // C=US, O=AffirmTrust, CN=AffirmTrust Commercial
    {0x03, 0x76, 0xAB, 0x1D, 0x54, 0xC5, 0xF9, 0x80, 0x3C, 0xE4, 0xB2, 0xE2,
     0x01, 0xA0, 0xEE, 0x7E, 0xEF, 0x7B, 0x57, 0xB6, 0x36, 0xE8, 0xA9, 0x3C,
     0x9B, 0x8D, 0x48, 0x60, 0xC9, 0x6F, 0x5F, 0xA7},
    // C=KR, O=Government of Korea, OU=GPKI, CN=Root CA
    {0x03, 0x78, 0xB2, 0x02, 0xCC, 0xAB, 0xBA, 0x99, 0xA1, 0x2E, 0x56, 0x9A,
     0x11, 0xA0, 0x77, 0xDB, 0x1E, 0xDB, 0x39, 0x48, 0x20, 0x61, 0xC7, 0x5D,
     0x00, 0x73, 0x05, 0x9D, 0x9A, 0xB5, 0xB5, 0x13},
    // CN=ACEDICOM Root, OU=PKI, O=EDICOM, C=ES
    {0x03, 0x95, 0x0F, 0xB4, 0x9A, 0x53, 0x1F, 0x3E, 0x19, 0x91, 0x94, 0x23,
     0x98, 0xDF, 0xA9, 0xE0, 0xEA, 0x32, 0xD7, 0xBA, 0x1C, 0xDD, 0x9B, 0xC8,
     0x5D, 0xB5, 0x7E, 0xD9, 0x40, 0x0B, 0x43, 0x4A},
    // C=ES, CN=Autoridad de Certificacion Firmaprofesional CIF A62634068
    {0x04, 0x04, 0x80, 0x28, 0xBF, 0x1F, 0x28, 0x64, 0xD4, 0x8F, 0x9A, 0xD4,
     0xD8, 0x32, 0x94, 0x36, 0x6A, 0x82, 0x88, 0x56, 0x55, 0x3F, 0x3B, 0x14,
     0x30, 0x3F, 0x90, 0x14, 0x7F, 0x5D, 0x40, 0xEF},
    // C=us, O=U.S. Government, OU=FBCA, CN=Common Policy
    {0x04, 0xAC, 0xFB, 0x3B, 0x24, 0x79, 0x3F, 0x30, 0x0F, 0x67, 0xEF, 0x87,
     0xE4, 0x4D, 0xD7, 0x2C, 0xB9, 0xB2, 0x8B, 0x20, 0x4F, 0x38, 0x9A, 0x7C,
     0xD5, 0xAE, 0x28, 0x78, 0x5C, 0x7D, 0x42, 0xCD},
    // C=LT, O=VI Registru Centras - I.k. 124110246, OU=Registru Centro
    // Sertifikavimo Centras, CN=VI Registru Centras RCSC (RootCA)
    {0x05, 0x36, 0x80, 0x1F, 0xBB, 0x44, 0x3B, 0x3E, 0x90, 0x5F, 0xD6, 0xD7,
     0x0D, 0x8C, 0x81, 0xEB, 0x88, 0x55, 0x1B, 0xE8, 0x06, 0x12, 0x99, 0x11,
     0x0D, 0x2B, 0x4F, 0x82, 0xE6, 0x4C, 0xAD, 0xE1},
    // C=BE, O=Certipost s.a./n.v., CN=Certipost E-Trust Primary Qualified CA
    {0x05, 0x8A, 0x40, 0x32, 0x3E, 0xC8, 0xC4, 0x62, 0x62, 0xC3, 0x05, 0x2A,
     0x5D, 0x35, 0x7B, 0x91, 0xAC, 0x24, 0xD3, 0xDA, 0x26, 0x35, 0x1B, 0x3F,
     0xF4, 0x40, 0x7E, 0x99, 0xF7, 0xA4, 0xE9, 0xB4},
    // C=EU, L=Madrid (see current address at
    // www.camerfirma.com/address)/serialNumber=A82743287, O=AC Camerfirma S.A.,
    // CN=Chambers of Commerce Root - 2008
    {0x06, 0x3E, 0x4A, 0xFA, 0xC4, 0x91, 0xDF, 0xD3, 0x32, 0xF3, 0x08, 0x9B,
     0x85, 0x42, 0xE9, 0x46, 0x17, 0xD8, 0x93, 0xD7, 0xFE, 0x94, 0x4E, 0x10,
     0xA7, 0x93, 0x7E, 0xE2, 0x9D, 0x96, 0x93, 0xC0},
    // O=Digital Signature Trust Co., CN=DST Root CA X3
    {0x06, 0x87, 0x26, 0x03, 0x31, 0xA7, 0x24, 0x03, 0xD9, 0x09, 0xF1, 0x05,
     0xE6, 0x9B, 0xCF, 0x0D, 0x32, 0xE1, 0xBD, 0x24, 0x93, 0xFF, 0xC6, 0xD9,
     0x20, 0x6D, 0x11, 0xBC, 0xD6, 0x77, 0x07, 0x39},
    // C=JP, O=LGPKI, OU=Application CA G2
    {0x06, 0xDB, 0x3A, 0xF2, 0xDB, 0x7B, 0xAE, 0xE0, 0x0C, 0x03, 0xB9, 0x57,
     0x82, 0x88, 0xBB, 0xDE, 0x54, 0x1D, 0x90, 0x6E, 0xB0, 0x06, 0x93, 0x27,
     0x41, 0x32, 0x95, 0xFF, 0xB4, 0x86, 0x00, 0x8E},
    // C=IE, O=An Post, OU=Post.Trust Ltd., CN=Post.Trust Root CA
    {0x07, 0x45, 0x3D, 0x53, 0x79, 0x3B, 0xF4, 0x18, 0x19, 0xA5, 0x25, 0x1C,
     0x69, 0xF8, 0x8E, 0x2B, 0xB3, 0x44, 0xB5, 0x9C, 0xA8, 0x28, 0xB5, 0xA5,
     0x43, 0x78, 0x15, 0x99, 0xEA, 0xF3, 0xD6, 0x02},
    // C=IL, O=PersonalID Ltd., OU=Certificate Services, CN=PersonalID
    // Trustworthy RootCA 2011
    {0x07, 0x5B, 0xFC, 0xCA, 0x2D, 0x55, 0xAE, 0x6E, 0x35, 0x74, 0x2C, 0x32,
     0xAF, 0xD0, 0xCA, 0x8E, 0xA4, 0xC9, 0x58, 0xFE, 0xEF, 0xC2, 0x32, 0x24,
     0x99, 0x95, 0x41, 0xC0, 0x33, 0xD6, 0x9C, 0x8D},
    // C=CN, O=CFCA GT CA
    {0x07, 0x71, 0x92, 0x0C, 0x8C, 0xB8, 0x74, 0xD5, 0xC5, 0xA4, 0xDC, 0x0D,
     0x6A, 0x51, 0xA2, 0xD4, 0x95, 0xD3, 0x8C, 0x4D, 0xE2, 0xCD, 0x5B, 0x83,
     0xD2, 0xA0, 0x6F, 0xAA, 0x05, 0x19, 0x35, 0xF6},
    // C=US, O=Equifax, OU=Equifax Secure Certificate Authority
    {0x08, 0x29, 0x7A, 0x40, 0x47, 0xDB, 0xA2, 0x36, 0x80, 0xC7, 0x31, 0xDB,
     0x6E, 0x31, 0x76, 0x53, 0xCA, 0x78, 0x48, 0xE1, 0xBE, 0xBD, 0x3A, 0x0B,
     0x01, 0x79, 0xA7, 0x07, 0xF9, 0x2C, 0xF1, 0x78},
    // C=US, O=AffirmTrust, CN=AffirmTrust Networking
    {0x0A, 0x81, 0xEC, 0x5A, 0x92, 0x97, 0x77, 0xF1, 0x45, 0x90, 0x4A, 0xF3,
     0x8D, 0x5D, 0x50, 0x9F, 0x66, 0xB5, 0xE2, 0xC5, 0x8F, 0xCD, 0xB5, 0x31,
     0x05, 0x8B, 0x0E, 0x17, 0xF3, 0xF0, 0xB4, 0x1B},
    // C=HU, L=Budapest, O=NetLock Halozatbiztonsagi Kft., OU=Tanusitvanykiadok,
    // CN=NetLock Expressz (Class C) Tanusitvanykiado
    {0x0B, 0x5E, 0xED, 0x4E, 0x84, 0x64, 0x03, 0xCF, 0x55, 0xE0, 0x65, 0x84,
     0x84, 0x40, 0xED, 0x2A, 0x82, 0x75, 0x8B, 0xF5, 0xB9, 0xAA, 0x1F, 0x25,
     0x3D, 0x46, 0x13, 0xCF, 0xA0, 0x80, 0xFF, 0x3F},
    // C=LT, O=Skaitmeninio sertifikavimo centras, OU=Certification Authority,
    // CN=SSC Root CA B
    {0x0B, 0x9F, 0x26, 0xDF, 0xCA, 0x68, 0x4C, 0x2C, 0xFC, 0xE2, 0x3E, 0x4E,
     0x4D, 0xD5, 0x67, 0xC8, 0x86, 0xBA, 0x25, 0x9E, 0x1D, 0xB2, 0x67, 0xF9,
     0x80, 0x6F, 0x0C, 0x5A, 0x09, 0x97, 0x11, 0xF2},
    // C=EU, O=AC Camerfirma SA CIF A82743287, OU=http://www.chambersign.org,
    // CN=Chambers of Commerce Root
    {0x0C, 0x25, 0x8A, 0x12, 0xA5, 0x67, 0x4A, 0xEF, 0x25, 0xF2, 0x8B, 0xA7,
     0xDC, 0xFA, 0xEC, 0xEE, 0xA3, 0x48, 0xE5, 0x41, 0xE6, 0xF5, 0xCC, 0x4E,
     0xE6, 0x3B, 0x71, 0xB3, 0x61, 0x60, 0x6A, 0xC3},
    // C=GB, ST=Greater Manchester, L=Salford, O=COMODO CA Limited, CN=COMODO
    // Certification Authority
    {0x0C, 0x2C, 0xD6, 0x3D, 0xF7, 0x80, 0x6F, 0xA3, 0x99, 0xED, 0xE8, 0x09,
     0x11, 0x6B, 0x57, 0x5B, 0xF8, 0x79, 0x89, 0xF0, 0x65, 0x18, 0xF9, 0x80,
     0x8C, 0x86, 0x05, 0x03, 0x17, 0x8B, 0xAF, 0x66},
    // C=NL, O=DigiNotar, CN=DigiNotar Root CA/emailAddress=info@diginotar.nl
    {0x0D, 0x13, 0x6E, 0x43, 0x9F, 0x0A, 0xB6, 0xE9, 0x7F, 0x3A, 0x02, 0xA5,
     0x40, 0xDA, 0x9F, 0x06, 0x41, 0xAA, 0x55, 0x4E, 0x1D, 0x66, 0xEA, 0x51,
     0xAE, 0x29, 0x20, 0xD5, 0x1B, 0x2F, 0x72, 0x17},
    // CN=Autoridad de Certificacion Raiz del Estado Venezolano, C=VE,
    // L=Caracas, ST=Distrito Capital, O=Sistema Nacional de Certificacion
    // Electronica, OU=Superintendencia de Servicios de Certificacion
    // Electronica/emailAddress=acraiz@suscerte.gob.ve
    {0x0E, 0x88, 0xEB, 0x6E, 0xA2, 0x56, 0xE1, 0x9E, 0xF8, 0xD3, 0xAB, 0xD6,
     0x1A, 0x24, 0xD3, 0x8D, 0xBA, 0xD6, 0x32, 0x81, 0x6D, 0xD9, 0x57, 0x29,
     0x44, 0x27, 0xE4, 0x72, 0x4D, 0x81, 0xA3, 0x86},
    // C=NO, O=Buypass AS-983163327, CN=Buypass Class 2 CA 1
    {0x0F, 0x4E, 0x9C, 0xDD, 0x26, 0x4B, 0x02, 0x55, 0x50, 0xD1, 0x70, 0x80,
     0x63, 0x40, 0x21, 0x4F, 0xE9, 0x44, 0x34, 0xC9, 0xB0, 0x2F, 0x69, 0x7E,
     0xC7, 0x10, 0xFC, 0x5F, 0xEA, 0xFB, 0x5E, 0x38},
    // C=FR, O=Certplus, CN=Class 2 Primary CA
    {0x0F, 0x99, 0x3C, 0x8A, 0xEF, 0x97, 0xBA, 0xAF, 0x56, 0x87, 0x14, 0x0E,
     0xD5, 0x9A, 0xD1, 0x82, 0x1B, 0xB4, 0xAF, 0xAC, 0xF0, 0xAA, 0x9A, 0x58,
     0xB5, 0xD5, 0x7A, 0x33, 0x8A, 0x3A, 0xFB, 0xCB},
    // C=JP, O=Japanese Government, OU=GPKI, CN=ApplicationCA2 Root
    {0x12, 0x6B, 0xF0, 0x1C, 0x10, 0x94, 0xD2, 0xF0, 0xCA, 0x2E, 0x35, 0x23,
     0x80, 0xB3, 0xC7, 0x24, 0x29, 0x45, 0x46, 0xCC, 0xC6, 0x55, 0x97, 0xBE,
     0xF7, 0xF1, 0x2D, 0x8A, 0x17, 0x1F, 0x19, 0x84},
    // C=si, O=state-institutions, OU=sigen-ca
    {0x12, 0xD4, 0x80, 0xC1, 0xA3, 0xC6, 0x64, 0x78, 0x1B, 0x99, 0xD9, 0xDF,
     0x0E, 0x9F, 0xAF, 0x3F, 0x1C, 0xAC, 0xEE, 0x1B, 0x3C, 0x30, 0xC3, 0x12,
     0x3A, 0x33, 0x7A, 0x4A, 0x45, 0x4F, 0xFE, 0xD2},
    // C=EU, L=Madrid (see current address at
    // www.camerfirma.com/address)/serialNumber=A82743287, O=AC Camerfirma S.A.,
    // CN=Global Chambersign Root - 2008
    {0x13, 0x63, 0x35, 0x43, 0x93, 0x34, 0xA7, 0x69, 0x80, 0x16, 0xA0, 0xD3,
     0x24, 0xDE, 0x72, 0x28, 0x4E, 0x07, 0x9D, 0x7B, 0x52, 0x20, 0xBB, 0x8F,
     0xBD, 0x74, 0x78, 0x16, 0xEE, 0xBE, 0xBA, 0xCA},
    // C=US, O=Starfield Technologies, Inc., OU=Starfield Class 2 Certification
    // Authority
    {0x14, 0x65, 0xFA, 0x20, 0x53, 0x97, 0xB8, 0x76, 0xFA, 0xA6, 0xF0, 0xA9,
     0x95, 0x8E, 0x55, 0x90, 0xE4, 0x0F, 0xCC, 0x7F, 0xAA, 0x4F, 0xB7, 0xC2,
     0xC8, 0x67, 0x75, 0x21, 0xFB, 0x5F, 0xB6, 0x58},
    // C=FR, O=Certplus, CN=Certplus Root CA G1
    {0x15, 0x2A, 0x40, 0x2B, 0xFC, 0xDF, 0x2C, 0xD5, 0x48, 0x05, 0x4D, 0x22,
     0x75, 0xB3, 0x9C, 0x7F, 0xCA, 0x3E, 0xC0, 0x97, 0x80, 0x78, 0xB0, 0xF0,
     0xEA, 0x76, 0xE5, 0x61, 0xA6, 0xC7, 0x43, 0x3E},
    // C=es, O=Servicio de Certificacion del Colegio de Registradores (SCR),
    // OU=Certificado Propio, OU=Certificado Raiz, CN=Certificado de la Clave
    // Principal/street=Principe de Vergara 72 28006
    // Madrid/emailAddress=scr@registradores.org
    {0x15, 0x94, 0xCB, 0x5B, 0x82, 0x6C, 0x31, 0x5D, 0xE3, 0xBC, 0x93, 0x2C,
     0x56, 0x89, 0x5F, 0xF2, 0x3A, 0x3A, 0x98, 0x8B, 0x5D, 0xC1, 0xF0, 0x34,
     0xD2, 0x14, 0xDF, 0xD8, 0x58, 0xD8, 0x9E, 0xE8},
    // C=US, O=Network Solutions L.L.C., CN=Network Solutions Certificate
    // Authority
    {0x15, 0xF0, 0xBA, 0x00, 0xA3, 0xAC, 0x7A, 0xF3, 0xAC, 0x88, 0x4C, 0x07,
     0x2B, 0x10, 0x11, 0xA0, 0x77, 0xBD, 0x77, 0xC0, 0x97, 0xF4, 0x01, 0x64,
     0xB2, 0xF8, 0x59, 0x8A, 0xBD, 0x83, 0x86, 0x0C},
    // C=IE, O=Baltimore, OU=CyberTrust, CN=Baltimore CyberTrust Root
    {0x16, 0xAF, 0x57, 0xA9, 0xF6, 0x76, 0xB0, 0xAB, 0x12, 0x60, 0x95, 0xAA,
     0x5E, 0xBA, 0xDE, 0xF2, 0x2A, 0xB3, 0x11, 0x19, 0xD6, 0x44, 0xAC, 0x95,
     0xCD, 0x4B, 0x93, 0xDB, 0xF3, 0xF2, 0x6A, 0xEB},
    // C=GB, ST=Greater Manchester, L=Salford, O=COMODO CA Limited, CN=COMODO
    // ECC Certification Authority
    {0x17, 0x93, 0x92, 0x7A, 0x06, 0x14, 0x54, 0x97, 0x89, 0xAD, 0xCE, 0x2F,
     0x8F, 0x34, 0xF7, 0xF0, 0xB6, 0x6D, 0x0F, 0x3A, 0xE3, 0xA3, 0xB8, 0x4D,
     0x21, 0xEC, 0x15, 0xDB, 0xBA, 0x4F, 0xAD, 0xC7},
    // OU=GlobalSign ECC Root CA - R5, O=GlobalSign, CN=GlobalSign
    {0x17, 0x9F, 0xBC, 0x14, 0x8A, 0x3D, 0xD0, 0x0F, 0xD2, 0x4E, 0xA1, 0x34,
     0x58, 0xCC, 0x43, 0xBF, 0xA7, 0xF5, 0x9C, 0x81, 0x82, 0xD7, 0x83, 0xA5,
     0x13, 0xF6, 0xEB, 0xEC, 0x10, 0x0C, 0x89, 0x24},
    // C=BM, O=QuoVadis Limited, CN=QuoVadis Root CA 3
    {0x18, 0xF1, 0xFC, 0x7F, 0x20, 0x5D, 0xF8, 0xAD, 0xDD, 0xEB, 0x7F, 0xE0,
     0x07, 0xDD, 0x57, 0xE3, 0xAF, 0x37, 0x5A, 0x9C, 0x4D, 0x8D, 0x73, 0x54,
     0x6B, 0xF4, 0xF1, 0xFE, 0xD1, 0xE1, 0x8D, 0x35},
    // C=GB, ST=Greater Manchester, L=Salford, O=COMODO CA Limited, CN=COMODO
    // Certification Authority
    {0x1A, 0x0D, 0x20, 0x44, 0x5D, 0xE5, 0xBA, 0x18, 0x62, 0xD1, 0x9E, 0xF8,
     0x80, 0x85, 0x8C, 0xBC, 0xE5, 0x01, 0x02, 0xB3, 0x6E, 0x8F, 0x0A, 0x04,
     0x0C, 0x3C, 0x69, 0xE7, 0x45, 0x22, 0xFE, 0x6E},
    // C=ZA, ST=Western Cape, L=Somerset West, O=South African Post Office
    // Limited, OU=SAPO Trust Centre, CN=SAPO Class 3 Root
    // CA/emailAddress=pkiadmin@trustcentre.co.za
    {0x1A, 0x25, 0x12, 0xCD, 0xA6, 0x74, 0x4A, 0xBE, 0xA1, 0x14, 0x32, 0xA2,
     0xFD, 0xC9, 0xF8, 0xC0, 0x88, 0xDB, 0x5A, 0x98, 0xC8, 0x9E, 0x13, 0x35,
     0x25, 0x74, 0xCD, 0xE4, 0xD9, 0xE8, 0x0C, 0xDD},
    // C=CN, O=China Internet Network Information Center, CN=China Internet
    // Network Information Center EV Certificates Root
    {0x1C, 0x01, 0xC6, 0xF4, 0xDB, 0xB2, 0xFE, 0xFC, 0x22, 0x55, 0x8B, 0x2B,
     0xCA, 0x32, 0x56, 0x3F, 0x49, 0x84, 0x4A, 0xCF, 0xC3, 0x2B, 0x7B, 0xE4,
     0xB0, 0xFF, 0x59, 0x9F, 0x9E, 0x8C, 0x7A, 0xF7},
    // C=TN, O=ANCE, OU=Certification & PKI, CN=Agence Nationale de
    // Certification Electronique/emailAddress=ance@certification.tn
    {0x1D, 0x4F, 0x05, 0x96, 0xFC, 0xA2, 0x61, 0x1D, 0x09, 0xF8, 0x4C, 0x78,
     0xF2, 0xEA, 0x56, 0x5E, 0xF2, 0xEA, 0xB9, 0xCF, 0xC2, 0x72, 0xA1, 0x71,
     0x8B, 0xD3, 0x36, 0xE6, 0xE0, 0xAE, 0x02, 0x1A},
    // C=FR, O=ANSSI, OU=0002 130007669, CN=IGC/A AC racine Etat francais
    {0x1E, 0x1A, 0x69, 0x84, 0xB4, 0xE7, 0x6B, 0xD7, 0x09, 0xAE, 0xE3, 0xE9,
     0xC9, 0xCF, 0x31, 0x18, 0xEA, 0xC0, 0x96, 0xDA, 0xB9, 0xCC, 0x20, 0xDC,
     0x25, 0xFA, 0xAB, 0x67, 0x29, 0x7E, 0x96, 0x5A},
    // C=CH, O=SwissSign AG, CN=SwissSign Silver Root CA - G3
    {0x1E, 0x49, 0xAC, 0x5D, 0xC6, 0x9E, 0x86, 0xD0, 0x56, 0x5D, 0xA2, 0xC1,
     0x30, 0x5C, 0x41, 0x93, 0x30, 0xB0, 0xB7, 0x81, 0xBF, 0xEC, 0x50, 0xE5,
     0x4A, 0x1B, 0x35, 0xAF, 0x7F, 0xDD, 0xD5, 0x01},
    // C=ch, O=Swisscom, OU=Digital Certificate Services, CN=Swisscom Root CA 1
    {0x21, 0xDB, 0x20, 0x12, 0x36, 0x60, 0xBB, 0x2E, 0xD4, 0x18, 0x20, 0x5D,
     0xA1, 0x1E, 0xE7, 0xA8, 0x5A, 0x65, 0xE2, 0xBC, 0x6E, 0x55, 0xB5, 0xAF,
     0x7E, 0x78, 0x99, 0xC8, 0xA2, 0x66, 0xD9, 0x2E},
    // C=US, O=Cisco Systems, CN=Cisco RXC-R2
    {0x22, 0x9C, 0xCC, 0x19, 0x6D, 0x32, 0xC9, 0x84, 0x21, 0xCC, 0x11, 0x9E,
     0x78, 0x48, 0x6E, 0xEB, 0xEF, 0x60, 0x3A, 0xEC, 0xD5, 0x25, 0xC6, 0xB8,
     0x8B, 0x47, 0xAB, 0xB7, 0x40, 0x69, 0x2B, 0x96},
    // C=US, O=Digital Signature Trust Co., OU=DST-Entrust GTI CA
    {0x22, 0xE0, 0xD1, 0x1D, 0xC9, 0x20, 0x7E, 0x16, 0xC9, 0x2B, 0x2E, 0xE1,
     0x8C, 0xFD, 0xB2, 0xC2, 0xE9, 0x40, 0x62, 0x68, 0x47, 0x92, 0x1F, 0xC5,
     0x28, 0xCE, 0xDD, 0x2F, 0x79, 0x32, 0xF7, 0x14},
    // C=ES, O=IZENPE S.A., CN=Izenpe.com
    {0x23, 0x80, 0x42, 0x03, 0xCA, 0x45, 0xD8, 0xCD, 0xE7, 0x16, 0xB8, 0xC1,
     0x3B, 0xF3, 0xB4, 0x48, 0x45, 0x7F, 0xA0, 0x6C, 0xC1, 0x02, 0x50, 0x99,
     0x7F, 0xA0, 0x14, 0x58, 0x31, 0x7C, 0x41, 0xE5},
    // C=US, O=VeriSign, Inc., OU=VeriSign Trust Network, OU=(c) 2008 VeriSign,
    // Inc. - For authorized use only, CN=VeriSign Universal Root Certification
    // Authority
    {0x23, 0x99, 0x56, 0x11, 0x27, 0xA5, 0x71, 0x25, 0xDE, 0x8C, 0xEF, 0xEA,
     0x61, 0x0D, 0xDF, 0x2F, 0xA0, 0x78, 0xB5, 0xC8, 0x06, 0x7F, 0x4E, 0x82,
     0x82, 0x90, 0xBF, 0xB8, 0x60, 0xE8, 0x4B, 0x3C},
    // C=US, ST=Utah, L=Salt Lake City, O=Xcert EZ by DST, CN=Xcert EZ by
    // DST/emailAddress=ca@digsigtrust.com
    {0x26, 0x02, 0xD2, 0x1E, 0x81, 0x27, 0x7A, 0x83, 0xF6, 0x04, 0x81, 0x28,
     0xF6, 0x1D, 0x79, 0x4A, 0x06, 0xF4, 0x74, 0xE1, 0xF7, 0x5E, 0x49, 0x74,
     0x0A, 0x81, 0x7C, 0x26, 0x66, 0xF6, 0x22, 0x11},
    // CN=ComSign Global Root CA, O=ComSign Ltd., C=IL
    {0x26, 0x05, 0x87, 0x5A, 0xFC, 0xC1, 0x76, 0xB2, 0xD6, 0x6D, 0xD6, 0x6A,
     0x99, 0x5D, 0x7F, 0x8D, 0x5E, 0xBB, 0x86, 0xCE, 0x12, 0x0D, 0x0E, 0x7E,
     0x9E, 0x7C, 0x6E, 0xF2, 0x94, 0xA2, 0x7D, 0x4C},
    // C=FR, O=OpenTrust, CN=OpenTrust Root CA G2
    {0x27, 0x99, 0x58, 0x29, 0xFE, 0x6A, 0x75, 0x15, 0xC1, 0xBF, 0xE8, 0x48,
     0xF9, 0xC4, 0x76, 0x1D, 0xB1, 0x6C, 0x22, 0x59, 0x29, 0x25, 0x7B, 0xF4,
     0x0D, 0x08, 0x94, 0xF2, 0x9E, 0xA8, 0xBA, 0xF2},
    // C=DE, O=TC TrustCenter GmbH, OU=TC TrustCenter Universal CA, CN=TC
    // TrustCenter Universal CA II
    {0x28, 0x34, 0x99, 0x1C, 0xF6, 0x77, 0x46, 0x6D, 0x22, 0xBA, 0xAC, 0x3B,
     0x00, 0x55, 0xE5, 0xB9, 0x11, 0xD9, 0xA9, 0xE5, 0x5F, 0x5B, 0x85, 0xBA,
     0x02, 0xDC, 0x56, 0x67, 0x82, 0xC3, 0x0E, 0x8A},
    // C=US, O=RSA Data Security, Inc., OU=Secure Server Certification Authority
    {0x29, 0x30, 0xBD, 0x09, 0xA0, 0x71, 0x26, 0xBD, 0xC1, 0x72, 0x88, 0xD4,
     0xF2, 0xAD, 0x84, 0x64, 0x5E, 0xC9, 0x48, 0x60, 0x79, 0x07, 0xA9, 0x7B,
     0x5E, 0xD0, 0xB0, 0xB0, 0x58, 0x79, 0xEF, 0x69},
    // C=NL, O=DigiNotar, CN=DigiNotar Root CA G2/emailAddress=info@diginotar.nl
    {0x29, 0x4F, 0x55, 0xEF, 0x3B, 0xD7, 0x24, 0x4C, 0x6F, 0xF8, 0xA6, 0x8A,
     0xB7, 0x97, 0xE9, 0x18, 0x6E, 0xC2, 0x75, 0x82, 0x75, 0x1A, 0x79, 0x15,
     0x15, 0xE3, 0x29, 0x2E, 0x48, 0x37, 0x2D, 0x61},
    // C=FR, O=Certinomis, OU=0002 433998903, CN=Certinomis - Root CA
    {0x2A, 0x99, 0xF5, 0xBC, 0x11, 0x74, 0xB7, 0x3C, 0xBB, 0x1D, 0x62, 0x08,
     0x84, 0xE0, 0x1C, 0x34, 0xE5, 0x1C, 0xCB, 0x39, 0x78, 0xDA, 0x12, 0x5F,
     0x0E, 0x33, 0x26, 0x88, 0x83, 0xBF, 0x41, 0x58},
    // C=US, ST=Arizona, L=Scottsdale, O=Starfield Technologies, Inc.,
    // CN=Starfield Root Certificate Authority - G2
    {0x2C, 0xE1, 0xCB, 0x0B, 0xF9, 0xD2, 0xF9, 0xE1, 0x02, 0x99, 0x3F, 0xBE,
     0x21, 0x51, 0x52, 0xC3, 0xB2, 0xDD, 0x0C, 0xAB, 0xDE, 0x1C, 0x68, 0xE5,
     0x31, 0x9B, 0x83, 0x91, 0x54, 0xDB, 0xB7, 0xF5},
    // C=JP, O=Japanese Government, OU=ApplicationCA
    {0x2D, 0x47, 0x43, 0x7D, 0xE1, 0x79, 0x51, 0x21, 0x5A, 0x12, 0xF3, 0xC5,
     0x8E, 0x51, 0xC7, 0x29, 0xA5, 0x80, 0x26, 0xEF, 0x1F, 0xCC, 0x0A, 0x5F,
     0xB3, 0xD9, 0xDC, 0x01, 0x2F, 0x60, 0x0D, 0x19},
    // C=IN, O=India PKI, CN=CCA India 2011
    {0x2D, 0x66, 0xA7, 0x02, 0xAE, 0x81, 0xBA, 0x03, 0xAF, 0x8C, 0xFF, 0x55,
     0xAB, 0x31, 0x8A, 0xFA, 0x91, 0x90, 0x39, 0xD9, 0xF3, 0x1B, 0x4D, 0x64,
     0x38, 0x86, 0x80, 0xF8, 0x13, 0x11, 0xB6, 0x5A},
    // C=AT, ST=Austria, L=Vienna, O=ARGE DATEN - Austrian Society for Data
    // Protection, OU=A-CERT Certification Service, CN=A-CERT
    // ADVANCED/emailAddress=info@a-cert.at
    {0x2D, 0xC6, 0x2C, 0x3F, 0x6C, 0x0C, 0xC9, 0x02, 0x0B, 0xBA, 0x77, 0xE1,
     0xC5, 0x11, 0x51, 0x10, 0x24, 0xB9, 0x43, 0xEE, 0x59, 0x88, 0x56, 0xDA,
     0x5A, 0x22, 0xE2, 0x22, 0xB7, 0x27, 0x7A, 0x20},
    // C=AT, O=A-Trust Ges. f. Sicherheitssysteme im elektr. Datenverkehr GmbH,
    // OU=A-Trust-Root-05, CN=A-Trust-Root-05
    {0x2D, 0xDE, 0x9D, 0x0C, 0x0A, 0x90, 0xE7, 0xB3, 0x2B, 0x5A, 0xBC, 0x01,
     0xF4, 0x17, 0x99, 0xD4, 0x2E, 0x95, 0xA1, 0xE3, 0xC3, 0x1C, 0x3B, 0x39,
     0x37, 0x3B, 0xB8, 0x14, 0x1E, 0xA5, 0x44, 0x71},
    // C=US, O=GTE Corporation, OU=GTE CyberTrust Solutions, Inc., CN=GTE
    // CyberTrust Root
    {0x2D, 0xFC, 0xBA, 0xCA, 0xDF, 0x22, 0xA6, 0xFF, 0x10, 0x7A, 0x51, 0xFD,
     0x3E, 0x8B, 0x9E, 0x17, 0x85, 0x80, 0x28, 0x87, 0x9B, 0x13, 0xF7, 0xC3,
     0xB5, 0x7B, 0x3E, 0x1B, 0xD2, 0x31, 0x58, 0x09},
    // C=US, O=Equifax Secure, OU=Equifax Secure eBusiness CA-2
    {0x2F, 0x27, 0x4E, 0x48, 0xAB, 0xA4, 0xAC, 0x7B, 0x76, 0x59, 0x33, 0x10,
     0x17, 0x75, 0x50, 0x6D, 0xC3, 0x0E, 0xE3, 0x8E, 0xF6, 0xAC, 0xD5, 0xC0,
     0x49, 0x32, 0xCF, 0xE0, 0x41, 0x23, 0x42, 0x20},
    // C=DE, O=TC TrustCenter GmbH, OU=TC TrustCenter Universal CA, CN=TC
    // TrustCenter Universal CA III
    {0x30, 0x9B, 0x4A, 0x87, 0xF6, 0xCA, 0x56, 0xC9, 0x31, 0x69, 0xAA, 0xA9,
     0x9C, 0x6D, 0x98, 0x88, 0x54, 0xD7, 0x89, 0x2B, 0xD5, 0x43, 0x7E, 0x2D,
     0x07, 0xB2, 0x9C, 0xBE, 0xDA, 0x55, 0xD3, 0x5D},
    // C=US, O=IdenTrust, CN=IdenTrust Public Sector Root CA 1
    {0x30, 0xD0, 0x89, 0x5A, 0x9A, 0x44, 0x8A, 0x26, 0x20, 0x91, 0x63, 0x55,
     0x22, 0xD1, 0xF5, 0x20, 0x10, 0xB5, 0x86, 0x7A, 0xCA, 0xE1, 0x2C, 0x78,
     0xEF, 0x95, 0x8F, 0xD4, 0xF4, 0x38, 0x9F, 0x2F},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert Global Root G3
    {0x31, 0xAD, 0x66, 0x48, 0xF8, 0x10, 0x41, 0x38, 0xC7, 0x38, 0xF3, 0x9E,
     0xA4, 0x32, 0x01, 0x33, 0x39, 0x3E, 0x3A, 0x18, 0xCC, 0x02, 0x29, 0x6E,
     0xF9, 0x7C, 0x2A, 0xC9, 0xEF, 0x67, 0x31, 0xD0},
    // C=BR, ST=Rio de Janeiro, L=Rio de Janeiro, O=Certisign Certificadora
    // Digital Ltda., OU=Certisign Autoridade Certificadora AC3S
    {0x31, 0xEA, 0xCE, 0x9B, 0x4C, 0x9C, 0x71, 0x73, 0x4A, 0x18, 0x56, 0x80,
     0xBC, 0x24, 0x86, 0x6C, 0xA6, 0xCB, 0xD8, 0x2B, 0x3C, 0xB6, 0x1B, 0xCC,
     0x87, 0x06, 0x26, 0x1B, 0x59, 0xCE, 0x10, 0x73},
    // C=DE, O=TC TrustCenter GmbH, OU=TC TrustCenter Class 4 CA, CN=TC
    // TrustCenter Class 4 CA II
    {0x32, 0x66, 0x96, 0x7E, 0x59, 0xCD, 0x68, 0x00, 0x8D, 0x9D, 0xD3, 0x20,
     0x81, 0x11, 0x85, 0xC7, 0x04, 0x20, 0x5E, 0x8D, 0x95, 0xFD, 0xD8, 0x4F,
     0x1C, 0x7B, 0x31, 0x1E, 0x67, 0x04, 0xFC, 0x32},
    // C=HU, L=Budapest, O=Microsec Ltd., OU=e-Szigno CA, CN=Microsec e-Szigno
    // Root CA
    {0x32, 0x7A, 0x3D, 0x76, 0x1A, 0xBA, 0xDE, 0xA0, 0x34, 0xEB, 0x99, 0x84,
     0x06, 0x27, 0x5C, 0xB1, 0xA4, 0x77, 0x6E, 0xFD, 0xAE, 0x2F, 0xDF, 0x6D,
     0x01, 0x68, 0xEA, 0x1C, 0x4F, 0x55, 0x67, 0xD0},
    // C=SA, O=National Center for Digital Certification, OU=Saudi National Root
    // CA
    {0x34, 0xBB, 0x34, 0xE1, 0x4F, 0xAE, 0xD0, 0xD3, 0x39, 0x2F, 0x2F, 0xC4,
     0x41, 0xC0, 0xEC, 0xD5, 0xFD, 0x88, 0xAD, 0x88, 0x11, 0x8D, 0xF2, 0xD1,
     0xBA, 0x76, 0xCD, 0xEC, 0x1E, 0xEA, 0x10, 0xB8},
    // CN=EBG Elektronik Sertifika Hizmet
    // Sa\xC4\x9Flay\xC4\xB1c\xC4\xB1s\xC4\xB1, O=EBG Bili\xC5\x9Fim
    // Teknolojileri ve Hizmetleri A.\xC5\x9E., C=TR
    {0x35, 0xAE, 0x5B, 0xDD, 0xD8, 0xF7, 0xAE, 0x63, 0x5C, 0xFF, 0xBA, 0x56,
     0x82, 0xA8, 0xF0, 0x0B, 0x95, 0xF4, 0x84, 0x62, 0xC7, 0x10, 0x8E, 0xE9,
     0xA0, 0xE5, 0x29, 0x2B, 0x07, 0x4A, 0xAF, 0xB2},
    // C=ES, O=Agencia Notarial de Certificacion S.L. Unipersonal - CIF
    // B83395988, CN=ANCERT Corporaciones de Derecho Publico
    {0x36, 0x54, 0x4D, 0x2F, 0xCE, 0x03, 0xC6, 0xC7, 0x2B, 0x70, 0xEB, 0x1A,
     0x80, 0x64, 0x26, 0x4D, 0xC1, 0x51, 0x17, 0x68, 0xC2, 0xD8, 0xF7, 0x6A,
     0x8B, 0x9F, 0x1F, 0x2B, 0xBD, 0x15, 0x3B, 0x7D},
    // C=US, O=GeoTrust Inc., CN=GeoTrust Primary Certification Authority
    {0x37, 0xD5, 0x10, 0x06, 0xC5, 0x12, 0xEA, 0xAB, 0x62, 0x64, 0x21, 0xF1,
     0xEC, 0x8C, 0x92, 0x01, 0x3F, 0xC5, 0xF8, 0x2A, 0xE9, 0x8E, 0xE5, 0x33,
     0xEB, 0x46, 0x19, 0xB8, 0xDE, 0xB4, 0xD0, 0x6C},
    // C=HU, L=Budapest, O=NetLock Halozatbiztonsagi Kft., OU=Tanusitvanykiadok,
    // CN=NetLock Uzleti (Class B) Tanusitvanykiado
    {0x39, 0xDF, 0x7B, 0x68, 0x2B, 0x7B, 0x93, 0x8F, 0x84, 0x71, 0x54, 0x81,
     0xCC, 0xDE, 0x8D, 0x60, 0xD8, 0xF2, 0x2E, 0xC5, 0x98, 0x87, 0x7D, 0x0A,
     0xAA, 0xC1, 0x2B, 0x59, 0x18, 0x2B, 0x03, 0x12},
    // C=IE, O=An Post, OU=Post.Trust Ltd., CN=Post.Trust Root CA
    {0x39, 0xF6, 0x03, 0x3C, 0xA6, 0x64, 0xCE, 0xDB, 0x21, 0x67, 0x44, 0xD3,
     0xC6, 0xF2, 0xB2, 0xE8, 0xE3, 0xCE, 0xBD, 0x68, 0x69, 0xF7, 0xDF, 0xF4,
     0x7E, 0x0D, 0xB9, 0x1A, 0x79, 0xC4, 0x16, 0xF8},
    // C=CH, O=SwissSign AG, CN=SwissSign Platinum CA - G2
    {0x3B, 0x22, 0x2E, 0x56, 0x67, 0x11, 0xE9, 0x92, 0x30, 0x0D, 0xC0, 0xB1,
     0x5A, 0xB9, 0x47, 0x3D, 0xAF, 0xDE, 0xF8, 0xC8, 0x4D, 0x0C, 0xEF, 0x7D,
     0x33, 0x17, 0xB4, 0xC1, 0x82, 0x1D, 0x14, 0x36},
    // C=NL, O=Staat der Nederlanden, CN=Staat der Nederlanden Root CA - G3
    {0x3C, 0x4F, 0xB0, 0xB9, 0x5A, 0xB8, 0xB3, 0x00, 0x32, 0xF4, 0x32, 0xB8,
     0x6F, 0x53, 0x5F, 0xE1, 0x72, 0xC1, 0x85, 0xD0, 0xFD, 0x39, 0x86, 0x58,
     0x37, 0xCF, 0x36, 0x18, 0x7F, 0xA6, 0xF4, 0x28},
    // C=HU, L=Budapest, O=Microsec Ltd., CN=Microsec e-Szigno Root CA
    // 2009/emailAddress=info@e-szigno.hu
    {0x3C, 0x5F, 0x81, 0xFE, 0xA5, 0xFA, 0xB8, 0x2C, 0x64, 0xBF, 0xA2, 0xEA,
     0xEC, 0xAF, 0xCD, 0xE8, 0xE0, 0x77, 0xFC, 0x86, 0x20, 0xA7, 0xCA, 0xE5,
     0x37, 0x16, 0x3D, 0xF3, 0x6E, 0xDB, 0xF3, 0x78},
    // CN=ComSign Advanced Security CA
    {0x3C, 0xCC, 0x3C, 0xCF, 0xE4, 0x54, 0x96, 0xD0, 0x7B, 0x62, 0x0D, 0xBF,
     0x13, 0x28, 0xE8, 0xA1, 0x49, 0x00, 0x18, 0xF4, 0x86, 0x33, 0xC8, 0xA2,
     0x8A, 0x99, 0x5C, 0xA6, 0x04, 0x08, 0xB0, 0xBE},
    // C=AT, O=A-Trust Ges. f\xFCr Sicherheitssysteme im elektr. Datenverkehr
    // GmbH, OU=A-Trust-Qual-01, CN=A-Trust-Qual-01
    {0x3D, 0xEA, 0xE6, 0xAE, 0x97, 0x52, 0x84, 0xE0, 0xE6, 0xFA, 0x2E, 0xB7,
     0x6C, 0xE4, 0x6E, 0x12, 0x44, 0x18, 0x69, 0xA2, 0xA7, 0xD4, 0xE6, 0x7D,
     0xC7, 0xAB, 0x86, 0x64, 0xFE, 0xFD, 0xBB, 0xB0},
    // C=EE, O=AS Sertifitseerimiskeskus, CN=EE Certification Centre Root
    // CA/emailAddress=pki@sk.ee
    {0x3E, 0x84, 0xBA, 0x43, 0x42, 0x90, 0x85, 0x16, 0xE7, 0x75, 0x73, 0xC0,
     0x99, 0x2F, 0x09, 0x79, 0xCA, 0x08, 0x4E, 0x46, 0x85, 0x68, 0x1F, 0xF1,
     0x95, 0xCC, 0xBA, 0x8A, 0x22, 0x9B, 0x8A, 0x76},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert Assured ID Root CA
    {0x3E, 0x90, 0x99, 0xB5, 0x01, 0x5E, 0x8F, 0x48, 0x6C, 0x00, 0xBC, 0xEA,
     0x9D, 0x11, 0x1E, 0xE7, 0x21, 0xFA, 0xBA, 0x35, 0x5A, 0x89, 0xBC, 0xF1,
     0xDF, 0x69, 0x56, 0x1E, 0x3D, 0xC6, 0x32, 0x5C},
    // C=GB, ST=Greater Manchester, L=Salford, O=Comodo CA Limited, CN=Trusted
    // Certificate Services
    {0x3F, 0x06, 0xE5, 0x56, 0x81, 0xD4, 0x96, 0xF5, 0xBE, 0x16, 0x9E, 0xB5,
     0x38, 0x9F, 0x9F, 0x2B, 0x8F, 0xF6, 0x1E, 0x17, 0x08, 0xDF, 0x68, 0x81,
     0x72, 0x48, 0x49, 0xCD, 0x5D, 0x27, 0xCB, 0x69},
    // C=GB, O=Trustis Limited, OU=Trustis EVS Root CA
    {0x3F, 0x9D, 0xA4, 0x74, 0x4E, 0xC9, 0x67, 0x6C, 0xD3, 0x8B, 0x53, 0x0E,
     0x50, 0x0A, 0x46, 0x3F, 0xBC, 0xB1, 0x81, 0x65, 0x97, 0x7F, 0xF0, 0xDA,
     0x6D, 0x59, 0x93, 0xC3, 0xFE, 0x5F, 0xAB, 0x7C},
    // C=ZA, ST=Western Cape, L=Cape Town, O=Thawte Consulting cc,
    // OU=Certification Services Division, CN=Thawte Premium Server
    // CA/emailAddress=premium-server@thawte.com
    {0x3F, 0x9F, 0x27, 0xD5, 0x83, 0x20, 0x4B, 0x9E, 0x09, 0xC8, 0xA3, 0xD2,
     0x06, 0x6C, 0x4B, 0x57, 0xD3, 0xA2, 0x47, 0x9C, 0x36, 0x93, 0x65, 0x08,
     0x80, 0x50, 0x56, 0x98, 0x10, 0x5D, 0xBC, 0xE9},
    // C=KR, O=Government of Korea, OU=GPKI, CN=GPKIRootCA1
    {0x40, 0x7C, 0x27, 0x6B, 0xEA, 0xD2, 0xE4, 0xAF, 0x06, 0x61, 0xEF, 0x66,
     0x97, 0x34, 0x1D, 0xEC, 0x0A, 0x1F, 0x94, 0x34, 0xE4, 0xEA, 0xFB, 0x2D,
     0x3D, 0x32, 0xA9, 0x05, 0x49, 0xD9, 0xDE, 0x4A},
    // C=NL, O=Digidentity B.V., CN=Digidentity L3 Root CA - G2
    {0x41, 0x7D, 0xCF, 0x31, 0x80, 0xF4, 0xED, 0x1A, 0x37, 0x47, 0xAC, 0xF1,
     0x17, 0x93, 0x16, 0xCD, 0x48, 0xCB, 0x05, 0xC5, 0x78, 0x84, 0x35, 0x16,
     0x8A, 0xED, 0x98, 0xC9, 0x8C, 0xDC, 0xB6, 0x15},
    // C=CH, O=WISeKey, OU=Copyright (c) 2005, OU=OISTE Foundation Endorsed,
    // CN=OISTE WISeKey Global Root GA CA
    {0x41, 0xC9, 0x23, 0x86, 0x6A, 0xB4, 0xCA, 0xD6, 0xB7, 0xAD, 0x57, 0x80,
     0x81, 0x58, 0x2E, 0x02, 0x07, 0x97, 0xA6, 0xCB, 0xDF, 0x4F, 0xFF, 0x78,
     0xCE, 0x83, 0x96, 0xB3, 0x89, 0x37, 0xD7, 0xF5},
    // C=US, O=SecureTrust Corporation, CN=Secure Global CA
    {0x42, 0x00, 0xF5, 0x04, 0x3A, 0xC8, 0x59, 0x0E, 0xBB, 0x52, 0x7D, 0x20,
     0x9E, 0xD1, 0x50, 0x30, 0x29, 0xFB, 0xCB, 0xD4, 0x1C, 0xA1, 0xB5, 0x06,
     0xEC, 0x27, 0xF1, 0x5A, 0xDE, 0x7D, 0xAC, 0x69},
    // C=FR, O=KEYNECTIS, OU=ROOT, CN=KEYNECTIS ROOT CA
    {0x42, 0x10, 0xF1, 0x99, 0x49, 0x9A, 0x9A, 0xC3, 0x3C, 0x8D, 0xE0, 0x2B,
     0xA6, 0xDB, 0xAA, 0x14, 0x40, 0x8B, 0xDD, 0x8A, 0x6E, 0x32, 0x46, 0x89,
     0xC1, 0x92, 0x2D, 0x06, 0x97, 0x15, 0xA3, 0x32},
    // C=FR, O=Certeurope, OU=0002 434202180, CN=Certeurope Root CA 2
    {0x42, 0x14, 0x3A, 0x51, 0x1A, 0x3A, 0xFC, 0xDD, 0x80, 0xD5, 0x55, 0xDE,
     0xBB, 0x41, 0x91, 0xEC, 0x6B, 0xB2, 0x85, 0xEE, 0x66, 0xE6, 0x2E, 0xC6,
     0x57, 0xED, 0x20, 0xAD, 0xF7, 0xD5, 0x5F, 0xAA},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert Global Root CA
    {0x43, 0x48, 0xA0, 0xE9, 0x44, 0x4C, 0x78, 0xCB, 0x26, 0x5E, 0x05, 0x8D,
     0x5E, 0x89, 0x44, 0xB4, 0xD8, 0x4F, 0x96, 0x62, 0xBD, 0x26, 0xDB, 0x25,
     0x7F, 0x89, 0x34, 0xA4, 0x43, 0xC7, 0x01, 0x61},
    // C=US, O=Entrust, Inc., OU=See www.entrust.net/legal-terms, OU=(c) 2009
    // Entrust, Inc. - for authorized use only, CN=Entrust Root Certification
    // Authority - G2
    {0x43, 0xDF, 0x57, 0x74, 0xB0, 0x3E, 0x7F, 0xEF, 0x5F, 0xE4, 0x0D, 0x93,
     0x1A, 0x7B, 0xED, 0xF1, 0xBB, 0x2E, 0x6B, 0x42, 0x73, 0x8C, 0x4E, 0x6D,
     0x38, 0x41, 0x10, 0x3D, 0x3A, 0xA7, 0xF3, 0x39},
    // CN=T\xC3\x9CRKTRUST Elektronik Sertifika Hizmet
    // Sa\xC4\x9Flay\xC4\xB1c\xC4\xB1s\xC4\xB1, C=TR, L=ANKARA, O=(c) 2005
    // T\xC3\x9CRKTRUST Bilgi \xC4\xB0leti\xC5\x9Fim ve Bili\xC5\x9Fim
    // G\xC3\xBCvenli\xC4\x9Fi Hizmetleri A.\xC5\x9E.
    {0x44, 0x04, 0xE3, 0x3B, 0x5E, 0x14, 0x0D, 0xCF, 0x99, 0x80, 0x51, 0xFD,
     0xFC, 0x80, 0x28, 0xC7, 0xC8, 0x16, 0x15, 0xC5, 0xEE, 0x73, 0x7B, 0x11,
     0x1B, 0x58, 0x82, 0x33, 0xA9, 0xB5, 0x35, 0xA0},
    // C=US, O=VeriSign, Inc., OU=Class 4 Public Primary Certification Authority
    // - G2, OU=(c) 1998 VeriSign, Inc. - For authorized use only, OU=VeriSign
    // Trust Network
    {0x44, 0x64, 0x0A, 0x0A, 0x0E, 0x4D, 0x00, 0x0F, 0xBD, 0x57, 0x4D, 0x2B,
     0x8A, 0x07, 0xBD, 0xB4, 0xD1, 0xDF, 0xED, 0x3B, 0x45, 0xBA, 0xAB, 0xA7,
     0x6F, 0x78, 0x57, 0x78, 0xC7, 0x01, 0x19, 0x61},
    // C=GR, L=Athens, O=Hellenic Academic and Research Institutions Cert.
    // Authority, CN=Hellenic Academic and Research Institutions ECC RootCA 2015
    {0x44, 0xB5, 0x45, 0xAA, 0x8A, 0x25, 0xE6, 0x5A, 0x73, 0xCA, 0x15, 0xDC,
     0x27, 0xFC, 0x36, 0xD2, 0x4C, 0x1C, 0xB9, 0x95, 0x3A, 0x06, 0x65, 0x39,
     0xB1, 0x15, 0x82, 0xDC, 0x48, 0x7B, 0x48, 0x33},
    // C=US, ST=Arizona, L=Scottsdale, O=GoDaddy.com, Inc., CN=Go Daddy Root
    // Certificate Authority - G2
    {0x45, 0x14, 0x0B, 0x32, 0x47, 0xEB, 0x9C, 0xC8, 0xC5, 0xB4, 0xF0, 0xD7,
     0xB5, 0x30, 0x91, 0xF7, 0x32, 0x92, 0x08, 0x9E, 0x6E, 0x5A, 0x63, 0xE2,
     0x74, 0x9D, 0xD3, 0xAC, 0xA9, 0x19, 0x8E, 0xDA},
    // C=UY, O=ADMINISTRACION NACIONAL DE CORREOS, OU=SERVICIOS ELECTRONICOS,
    // CN=Correo Uruguayo - Root CA
    {0x46, 0x27, 0x32, 0x85, 0x61, 0x5D, 0x96, 0xE5, 0x2D, 0xA9, 0xFC, 0x2E,
     0xD8, 0xC0, 0x36, 0xF1, 0x0A, 0xF3, 0xD9, 0xF6, 0x28, 0x0F, 0x8D, 0x28,
     0x87, 0x06, 0xC5, 0x2B, 0x20, 0x11, 0xB4, 0xDA},
    // C=PT, O=SCEE, CN=ECRaizEstado
    {0x48, 0x8E, 0x13, 0x4F, 0x30, 0xC5, 0xDB, 0x56, 0xB7, 0x64, 0x73, 0xE6,
     0x08, 0x08, 0x68, 0x42, 0xBF, 0x21, 0xAF, 0x8A, 0xB3, 0xCD, 0x7A, 0xC6,
     0x7E, 0xBD, 0xF1, 0x25, 0xD5, 0x31, 0x83, 0x4E},
    // C=PA, ST=Panama, L=Panama City, O=TrustCor Systems S. de R.L.,
    // OU=TrustCor Certificate Authority, CN=TrustCor RootCert CA-1
    {0x48, 0x8F, 0xCA, 0x18, 0x9E, 0xAA, 0xDF, 0x54, 0xA3, 0xF9, 0x20, 0xED,
     0x39, 0xE5, 0x87, 0x18, 0x3B, 0xA5, 0x12, 0x23, 0x29, 0x99, 0xFA, 0xE3,
     0xE4, 0xA2, 0x85, 0xFE, 0x98, 0xE2, 0x98, 0xD1},
    // C=DK, O=TDC Internet, OU=TDC Internet Root CA
    {0x48, 0x98, 0xC6, 0x88, 0x8C, 0x0C, 0xFF, 0xB0, 0xD3, 0xE3, 0x1A, 0xCA,
     0x8A, 0x37, 0xD4, 0xE3, 0x51, 0x5F, 0xF7, 0x46, 0xD0, 0x26, 0x35, 0xD8,
     0x66, 0x46, 0xCF, 0xA0, 0xA3, 0x18, 0x5A, 0xE7},
    // C=TR, L=Ankara, O=T\xC3\x9CRKTRUST Bilgi \xC4\xB0leti\xC5\x9Fim ve
    // Bili\xC5\x9Fim G\xC3\xBCvenli\xC4\x9Fi Hizmetleri A.\xC5\x9E.,
    // CN=T\xC3\x9CRKTRUST Elektronik Sertifika Hizmet
    // Sa\xC4\x9Flay\xC4\xB1c\xC4\xB1s\xC4\xB1 H5
    {0x49, 0x35, 0x1B, 0x90, 0x34, 0x44, 0xC1, 0x85, 0xCC, 0xDC, 0x5C, 0x69,
     0x3D, 0x24, 0xD8, 0x55, 0x5C, 0xB2, 0x08, 0xD6, 0xA8, 0x14, 0x13, 0x07,
     0x69, 0x9F, 0x4A, 0xF0, 0x63, 0x19, 0x9D, 0x78},
    // C=us, ST=Utah, L=Salt Lake City, O=Digital Signature Trust Co.,
    // OU=National Retail Federation, CN=DST (NRF)
    // RootCA/emailAddress=ca@digsigtrust.com
    {0x49, 0xC8, 0x17, 0x5A, 0x98, 0x15, 0xE0, 0x8B, 0xEF, 0x12, 0x9A, 0x92,
     0x9D, 0xE1, 0xBA, 0xCA, 0xD0, 0x4E, 0x4D, 0xB6, 0x7A, 0x8C, 0x83, 0x92,
     0x93, 0x95, 0x3E, 0x50, 0x31, 0xC8, 0x1C, 0xA0},
    // C=DE, O=D-Trust GmbH, CN=D-TRUST Root Class 3 CA 2 2009
    {0x49, 0xE7, 0xA4, 0x42, 0xAC, 0xF0, 0xEA, 0x62, 0x87, 0x05, 0x00, 0x54,
     0xB5, 0x25, 0x64, 0xB6, 0x50, 0xE4, 0xF4, 0x9E, 0x42, 0xE3, 0x48, 0xD6,
     0xAA, 0x38, 0xE0, 0x39, 0xE9, 0x57, 0xB1, 0xC1},
    // C=US, O=Digital Signature Trust Co., CN=Baltimore EZ by
    // DST/mail=ca@digsigtrust.com
    {0x49, 0xF7, 0x4F, 0x82, 0x4F, 0x2E, 0x05, 0x9F, 0xE9, 0x9C, 0x98, 0xAF,
     0x32, 0x19, 0xEC, 0x0D, 0x9A, 0x00, 0x4D, 0x1B, 0x64, 0xDD, 0x2F, 0xD1,
     0x45, 0x26, 0x16, 0x31, 0x8A, 0xB8, 0x06, 0xC0},
    // C=US, O=thawte, Inc., OU=Certification Services Division, OU=(c) 2008
    // thawte, Inc. - For authorized use only, CN=thawte Primary Root CA - G3
    {0x4B, 0x03, 0xF4, 0x58, 0x07, 0xAD, 0x70, 0xF2, 0x1B, 0xFC, 0x2C, 0xAE,
     0x71, 0xC9, 0xFD, 0xE4, 0x60, 0x4C, 0x06, 0x4C, 0xF5, 0xFF, 0xB6, 0x86,
     0xBA, 0xE5, 0xDB, 0xAA, 0xD7, 0xFD, 0xD3, 0x4C},
    // C=CN, O=WoSign CA Limited, CN=Certification Authority of WoSign
    {0x4B, 0x22, 0xD5, 0xA6, 0xAE, 0xC9, 0x9F, 0x3C, 0xDB, 0x79, 0xAA, 0x5E,
     0xC0, 0x68, 0x38, 0x47, 0x9C, 0xD5, 0xEC, 0xBA, 0x71, 0x64, 0xF7, 0xF2,
     0x2D, 0xC1, 0xD6, 0x5F, 0x63, 0xD8, 0x57, 0x08},
    // C=BR, O=Certisign Certificadora Digital Ltda., OU=Certisign - Autoridade
    // Certificadora - AC4
    {0x4B, 0xDB, 0x74, 0x18, 0xBD, 0xF7, 0xFF, 0xE3, 0x3B, 0xA0, 0x88, 0x4A,
     0xFA, 0x7C, 0x0C, 0x61, 0xFD, 0x85, 0xA1, 0x53, 0x97, 0x2F, 0x65, 0xF7,
     0xD0, 0x1C, 0xB3, 0xEC, 0x7E, 0xB4, 0x07, 0x3C},
    // C=ES, O=Agencia Notarial de Certificacion S.L.U. - CIF B83395988,
    // CN=ANCERT Certificados Notariales V2
    {0x4B, 0xE8, 0xB5, 0xA1, 0xC7, 0x6C, 0x6A, 0xEA, 0xD0, 0x61, 0x19, 0x18,
     0xFC, 0xCF, 0x9D, 0xBD, 0x39, 0x8B, 0x67, 0xFB, 0x12, 0x29, 0x47, 0x58,
     0xBD, 0xF9, 0x94, 0xD0, 0xF9, 0x68, 0x2F, 0x60},
    // C=NL, O=Staat der Nederlanden, CN=Staat der Nederlanden EV Root CA
    {0x4D, 0x24, 0x91, 0x41, 0x4C, 0xFE, 0x95, 0x67, 0x46, 0xEC, 0x4C, 0xEF,
     0xA6, 0xCF, 0x6F, 0x72, 0xE2, 0x8A, 0x13, 0x29, 0x43, 0x2F, 0x9D, 0x8A,
     0x90, 0x7A, 0xC4, 0xCB, 0x5D, 0xAD, 0xC1, 0x5A},
    // C=ES, O=FNMT-RCM, OU=AC RAIZ FNMT-RCM
    {0x4D, 0x9E, 0xBB, 0x28, 0x82, 0x5C, 0x96, 0x43, 0xAB, 0x15, 0xD5, 0x4E,
     0x5F, 0x96, 0x14, 0xF1, 0x3C, 0xB3, 0xE9, 0x5D, 0xE3, 0xCF, 0x4E, 0xAC,
     0x97, 0x13, 0x01, 0xF3, 0x20, 0xF9, 0x22, 0x6E},
    // C=AT, O=Telekom-Control-Kommission, CN=Telekom-Control-Kommission Top 1
    {0x4D, 0xBB, 0x01, 0x57, 0xA6, 0x91, 0xFA, 0x73, 0x82, 0x28, 0x9D, 0x65,
     0xC0, 0x33, 0x2D, 0xDB, 0x1D, 0xCB, 0x64, 0x0B, 0x40, 0xAD, 0x10, 0xF0,
     0x10, 0xA4, 0x3E, 0x20, 0xF3, 0xAF, 0xED, 0x1E},
    // C=AT, ST=Austria, L=Vienna, O=Arge Daten Oesterreichische Gesellschaft
    // fuer Datenschutz/emailAddress=a-cert@argedaten.at
    {0x4E, 0x74, 0x80, 0xAD, 0x70, 0x2A, 0x37, 0x9D, 0xC5, 0x89, 0xAD, 0xB4,
     0xFA, 0xA6, 0x25, 0xE6, 0xA5, 0x99, 0x3F, 0x87, 0xEF, 0x23, 0x75, 0xD5,
     0x43, 0x7F, 0xFE, 0x3B, 0x79, 0xBE, 0x4E, 0x96},
    // C=US, ST=New Jersey, L=Jersey City, O=The USERTRUST Network, CN=USERTrust
    // ECC Certification Authority
    {0x4F, 0xF4, 0x60, 0xD5, 0x4B, 0x9C, 0x86, 0xDA, 0xBF, 0xBC, 0xFC, 0x57,
     0x12, 0xE0, 0x40, 0x0D, 0x2B, 0xED, 0x3F, 0xBC, 0x4D, 0x4F, 0xBD, 0xAA,
     0x86, 0xE0, 0x6A, 0xDC, 0xD2, 0xA9, 0xAD, 0x7A},
    // CN=ComSign Secured CA, O=ComSign, C=IL
    {0x50, 0x79, 0x41, 0xC7, 0x44, 0x60, 0xA0, 0xB4, 0x70, 0x86, 0x22, 0x0D,
     0x4E, 0x99, 0x32, 0x57, 0x2A, 0xB5, 0xD1, 0xB5, 0xBB, 0xCB, 0x89, 0x80,
     0xAB, 0x1C, 0xB1, 0x76, 0x51, 0xA8, 0x44, 0xD2},
    // C=JP, O=SECOM Trust Systems CO.,LTD., OU=Security Communication RootCA2
    {0x51, 0x3B, 0x2C, 0xEC, 0xB8, 0x10, 0xD4, 0xCD, 0xE5, 0xDD, 0x85, 0x39,
     0x1A, 0xDF, 0xC6, 0xC2, 0xDD, 0x60, 0xD8, 0x7B, 0xB7, 0x36, 0xD2, 0xB5,
     0x21, 0x48, 0x4A, 0xA4, 0x7A, 0x0E, 0xBE, 0xF6},
    // O=eSign Australia, OU=Public Secure Services, CN=eSign Imperito Primary
    // Root CA
    {0x52, 0x7A, 0x42, 0x26, 0x7D, 0xAE, 0xA8, 0xD8, 0xF5, 0x4E, 0x91, 0xD2,
     0x82, 0xD5, 0xC2, 0x5B, 0x61, 0x5B, 0xC0, 0xDC, 0x73, 0xDC, 0x63, 0xA5,
     0x8F, 0x91, 0x6C, 0x4E, 0xD5, 0xBF, 0x59, 0xAC},
    // C=US, O=GTE Corporation, CN=GTE CyberTrust Root
    {0x52, 0x7B, 0x05, 0x05, 0x27, 0xDF, 0x52, 0x9C, 0x0F, 0x7A, 0xD0, 0x0C,
     0xEF, 0x1E, 0x7B, 0xA4, 0x21, 0x78, 0x81, 0x82, 0x61, 0x5C, 0x32, 0x6C,
     0x8B, 0x6D, 0x1A, 0x20, 0x61, 0xA0, 0xBD, 0x7C},
    // C=GB, ST=Greater Manchester, L=Salford, O=COMODO CA Limited, CN=COMODO
    // RSA Certification Authority
    {0x52, 0xF0, 0xE1, 0xC4, 0xE5, 0x8E, 0xC6, 0x29, 0x29, 0x1B, 0x60, 0x31,
     0x7F, 0x07, 0x46, 0x71, 0xB8, 0x5D, 0x7E, 0xA8, 0x0D, 0x5B, 0x07, 0x27,
     0x34, 0x63, 0x53, 0x4B, 0x32, 0xB4, 0x02, 0x34},
    // C=US, O=Symantec Corporation, OU=Symantec Trust Network, CN=Symantec
    // Class 3 Public Primary Certification Authority - G4
    {0x53, 0xDF, 0xDF, 0xA4, 0xE2, 0x97, 0xFC, 0xFE, 0x07, 0x59, 0x4E, 0x8C,
     0x62, 0xD5, 0xB8, 0xAB, 0x06, 0xB3, 0x2C, 0x75, 0x49, 0xF3, 0x8A, 0x16,
     0x30, 0x94, 0xFD, 0x64, 0x29, 0xD5, 0xDA, 0x43},
    // C=JP, O=LGPKI, CN=Application CA G3 Root
    {0x54, 0xB4, 0xFC, 0x43, 0xD4, 0x4A, 0xA4, 0xCA, 0x9F, 0xC0, 0x3C, 0xA7,
     0xE9, 0x94, 0x9F, 0xBA, 0xE2, 0x67, 0xA0, 0x64, 0xD0, 0x2D, 0xA2, 0x18,
     0x52, 0x41, 0x2A, 0x38, 0x1B, 0x5D, 0x15, 0x37},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert Trusted Root G4
    {0x55, 0x2F, 0x7B, 0xDC, 0xF1, 0xA7, 0xAF, 0x9E, 0x6C, 0xE6, 0x72, 0x01,
     0x7F, 0x4F, 0x12, 0xAB, 0xF7, 0x72, 0x40, 0xC7, 0x8E, 0x76, 0x1A, 0xC2,
     0x03, 0xD1, 0xD9, 0xD2, 0x0A, 0xC8, 0x99, 0x88},
    // CN=Autoridad Certificadora Ra\xC3\xADz Nacional de Uruguay, O=AGESIC,
    // C=UY
    {0x55, 0x33, 0xA0, 0x40, 0x1F, 0x61, 0x2C, 0x68, 0x8E, 0xBC, 0xE5, 0xBF,
     0x53, 0xF2, 0xEC, 0x14, 0xA7, 0x34, 0xEB, 0x17, 0x8B, 0xFA, 0xE0, 0x0E,
     0x50, 0xE8, 0x5D, 0xAE, 0x67, 0x23, 0x07, 0x8A},
    // C=IT, L=Milan, O=Actalis S.p.A./03358520967, CN=Actalis Authentication
    // Root CA
    {0x55, 0x92, 0x60, 0x84, 0xEC, 0x96, 0x3A, 0x64, 0xB9, 0x6E, 0x2A, 0xBE,
     0x01, 0xCE, 0x0B, 0xA8, 0x6A, 0x64, 0xFB, 0xFE, 0xBC, 0xC7, 0xAA, 0xB5,
     0xAF, 0xC1, 0x55, 0xB3, 0x7F, 0xD7, 0x60, 0x66},
    // C=ES, O=Consejo General de la Abogacia NIF:Q-2863006I, CN=Autoridad de
    // Certificacion de la Abogacia
    {0x56, 0x07, 0xE2, 0x60, 0x16, 0x3F, 0x49, 0xC8, 0xEA, 0x41, 0x75, 0xA1,
     0xC0, 0xA5, 0x3B, 0x13, 0x19, 0x5C, 0xB7, 0xD0, 0x78, 0x45, 0x61, 0x1E,
     0x94, 0x3A, 0x2F, 0xF5, 0x07, 0x03, 0x68, 0x34},
    // C=US, ST=Arizona, L=Scottsdale, O=Starfield Technologies, Inc.,
    // CN=Starfield Services Root Certificate Authority - G2
    {0x56, 0x8D, 0x69, 0x05, 0xA2, 0xC8, 0x87, 0x08, 0xA4, 0xB3, 0x02, 0x51,
     0x90, 0xED, 0xCF, 0xED, 0xB1, 0x97, 0x4A, 0x60, 0x6A, 0x13, 0xC6, 0xE5,
     0x29, 0x0F, 0xCB, 0x2A, 0xE6, 0x3E, 0xDA, 0xB5},
    // C=FR, O=OpenTrust, CN=OpenTrust Root CA G1
    {0x56, 0xC7, 0x71, 0x28, 0xD9, 0x8C, 0x18, 0xD9, 0x1B, 0x4C, 0xFD, 0xFF,
     0xBC, 0x25, 0xEE, 0x91, 0x03, 0xD4, 0x75, 0x8E, 0xA2, 0xAB, 0xAD, 0x82,
     0x6A, 0x90, 0xF3, 0x45, 0x7D, 0x46, 0x0E, 0xB4},
    // CN=T\xC3\x9CRKTRUST Elektronik \xC4\xB0\xC5\x9Flem Hizmetleri, C=TR,
    // L=Ankara, O=T\xC3\x9CRKTRUST Bilgi \xC4\xB0leti\xC5\x9Fim ve
    // Bili\xC5\x9Fim G\xC3\xBCvenli\xC4\x9Fi Hizmetleri A.\xC5\x9E. (c)
    // Kas\xC4\xB1m 2005
    {0x56, 0xCE, 0x34, 0x7C, 0xC6, 0xDF, 0x4C, 0x35, 0x94, 0x3D, 0xFD, 0xEA,
     0xEE, 0x02, 0x3F, 0x97, 0x39, 0xA3, 0xF1, 0xCE, 0xDE, 0xEE, 0x0C, 0xD8,
     0x8D, 0xC2, 0x38, 0x6B, 0xC8, 0xA9, 0x1E, 0xAF},
    // L=ValiCert Validation Network, O=ValiCert, Inc., OU=ValiCert Class 2
    // Policy Validation Authority,
    // CN=http://www.valicert.com//emailAddress=info@valicert.com
    {0x58, 0xD0, 0x17, 0x27, 0x9C, 0xD4, 0xDC, 0x63, 0xAB, 0xDD, 0xB1, 0x96,
     0xA6, 0xC9, 0x90, 0x6C, 0x30, 0xC4, 0xE0, 0x87, 0x83, 0xEA, 0xE8, 0xC1,
     0x60, 0x99, 0x54, 0xD6, 0x93, 0x55, 0x59, 0x6B},
    // C=TW, O=TAIWAN-CA, OU=Root CA, CN=TWCA Global Root CA
    {0x59, 0x76, 0x90, 0x07, 0xF7, 0x68, 0x5D, 0x0F, 0xCD, 0x50, 0x87, 0x2F,
     0x9F, 0x95, 0xD5, 0x75, 0x5A, 0x5B, 0x2B, 0x45, 0x7D, 0x81, 0xF3, 0x69,
     0x2B, 0x61, 0x0A, 0x98, 0x67, 0x2F, 0x0E, 0x1B},
    // C=CH, O=SwissSign AG, CN=SwissSign Platinum Root CA - G3
    {0x59, 0xB3, 0x82, 0x9F, 0x1F, 0xF4, 0x43, 0x34, 0x49, 0x58, 0xFA, 0xE8,
     0xBF, 0xF6, 0x21, 0xB6, 0x84, 0xC8, 0x48, 0xCF, 0xBF, 0x7E, 0xAD, 0x6B,
     0x63, 0xA6, 0xCA, 0x50, 0xF2, 0x79, 0x4F, 0x89},
    // C=SI, O=Halcom, CN=Halcom CA FO
    {0x5A, 0x1B, 0x5D, 0x6B, 0xC6, 0x55, 0x23, 0xB4, 0x0A, 0x6D, 0xEF, 0xFA,
     0x45, 0xB4, 0x8E, 0x42, 0x88, 0xAE, 0x8D, 0xD8, 0x6D, 0xD7, 0x0A, 0x5B,
     0x85, 0x8D, 0x4A, 0x5A, 0xFF, 0xC9, 0x4F, 0x71},
    // CN=AC1 RAIZ MTIN/serialNumber=S2819001E, OU=PRESTADOR DE SERVICIOS DE
    // CERTIFICACION MTIN, OU=SUBDIRECCION GENERAL DE PROCESO DE DATOS,
    // O=MINISTERIO DE TRABAJO E INMIGRACION, L=MADRID, C=ES
    {0x5B, 0x1D, 0x9D, 0x24, 0xDE, 0x0A, 0xFE, 0xA8, 0xB3, 0x5B, 0xA0, 0x4A,
     0x1C, 0x3E, 0x25, 0xD0, 0x81, 0x2C, 0xDF, 0x7C, 0x46, 0x25, 0xDE, 0x0A,
     0x89, 0xAF, 0x9F, 0xE4, 0xBB, 0xD1, 0xBB, 0x15},
    // C=PL, O=Unizeto Technologies S.A., OU=Certum Certification Authority,
    // CN=Certum Trusted Network CA
    {0x5C, 0x58, 0x46, 0x8D, 0x55, 0xF5, 0x8E, 0x49, 0x7E, 0x74, 0x39, 0x82,
     0xD2, 0xB5, 0x00, 0x10, 0xB6, 0xD1, 0x65, 0x37, 0x4A, 0xCF, 0x83, 0xA7,
     0xD4, 0xA3, 0x2D, 0xB7, 0x68, 0xC4, 0x40, 0x8E},
    // C=CN, O=China Financial Certification Authority, CN=CFCA EV ROOT
    {0x5C, 0xC3, 0xD7, 0x8E, 0x4E, 0x1D, 0x5E, 0x45, 0x54, 0x7A, 0x04, 0xE6,
     0x87, 0x3E, 0x64, 0xF9, 0x0C, 0xF9, 0x53, 0x6D, 0x1C, 0xCC, 0x2E, 0xF8,
     0x00, 0xF3, 0x55, 0xC4, 0xC5, 0xFD, 0x70, 0xFD},
    // C=US, O=IdenTrust, CN=IdenTrust Commercial Root CA 1
    {0x5D, 0x56, 0x49, 0x9B, 0xE4, 0xD2, 0xE0, 0x8B, 0xCF, 0xCA, 0xD0, 0x8A,
     0x3E, 0x38, 0x72, 0x3D, 0x50, 0x50, 0x3B, 0xDE, 0x70, 0x69, 0x48, 0xE4,
     0x2F, 0x55, 0x60, 0x30, 0x19, 0xE5, 0x28, 0xAE},
    // C=AT, L=Vienna, ST=Austria, O=ARGE DATEN - Austrian Society for Data
    // Protection, OU=GLOBALTRUST Certification Service,
    // CN=GLOBALTRUST/emailAddress=info@globaltrust.info
    {0x5E, 0x35, 0x71, 0xF3, 0x3F, 0x45, 0xA7, 0xDF, 0x15, 0x37, 0xA6, 0x8B,
     0x5F, 0xFB, 0x9E, 0x03, 0x6A, 0xF9, 0xD2, 0xF5, 0xBC, 0x4C, 0x97, 0x17,
     0x13, 0x0D, 0xC4, 0x3D, 0x71, 0x75, 0xAA, 0xC7},
    // C=US, O=GeoTrust Inc., OU=(c) 2007 GeoTrust Inc. - For authorized use
    // only, CN=GeoTrust Primary Certification Authority - G2
    {0x5E, 0xDB, 0x7A, 0xC4, 0x3B, 0x82, 0xA0, 0x6A, 0x87, 0x61, 0xE8, 0xD7,
     0xBE, 0x49, 0x79, 0xEB, 0xF2, 0x61, 0x1F, 0x7D, 0xD7, 0x9B, 0xF9, 0x1C,
     0x1C, 0x6B, 0x56, 0x6A, 0x21, 0x9E, 0xD7, 0x66},
    // C=US, O=Equifax Secure Inc., CN=Equifax Secure Global eBusiness CA-1
    {0x5F, 0x0B, 0x62, 0xEA, 0xB5, 0xE3, 0x53, 0xEA, 0x65, 0x21, 0x65, 0x16,
     0x58, 0xFB, 0xB6, 0x53, 0x59, 0xF4, 0x43, 0x28, 0x0A, 0x4A, 0xFB, 0xD1,
     0x04, 0xD7, 0x7D, 0x10, 0xF9, 0xF0, 0x4C, 0x07},
    // C=JP, O=Japan Certification Services, Inc., CN=SecureSign RootCA1
    {0x5F, 0x96, 0x0E, 0xEB, 0xD7, 0x16, 0xDB, 0xCB, 0x4D, 0x8A, 0x78, 0xB9,
     0x96, 0xE6, 0x80, 0xEC, 0x25, 0x47, 0x44, 0x1E, 0x69, 0xB4, 0xE4, 0x4E,
     0x98, 0xA5, 0x95, 0x50, 0x2E, 0x28, 0xA0, 0x02},
    // C=PT, O=MULTICERT - Servi\xC3\xA7os de Certifica\xC3\xA7\xC3\xA3o
    // Electr\xC3\xB3nica S.A., CN=MULTICERT Root Certification Authority 01
    {0x60, 0x4D, 0x32, 0xD0, 0x36, 0x89, 0x5A, 0xED, 0x3B, 0xFE, 0xFA, 0xEB,
     0x72, 0x7C, 0x00, 0x9E, 0xC0, 0xF2, 0xB3, 0xCD, 0xFA, 0x42, 0xA1, 0xC7,
     0x17, 0x30, 0xE6, 0xA7, 0x2C, 0x3B, 0xE9, 0xD4},
    // C=KR, O=Government of Korea, OU=GPKI, CN=GPKIRootCA
    {0x60, 0x62, 0x23, 0xD9, 0xDB, 0x80, 0xDF, 0x39, 0x39, 0x60, 0x1E, 0x74,
     0xB7, 0xE8, 0x28, 0xE2, 0x80, 0x0C, 0xCE, 0x42, 0x73, 0xF7, 0x6F, 0x27,
     0x6A, 0xA6, 0x2D, 0xB0, 0xA8, 0xE3, 0xB6, 0xC1},
    // C=DE, ST=Hamburg, L=Hamburg, O=TC TrustCenter for Security in Data
    // Networks GmbH, OU=TC TrustCenter Class 1
    // CA/emailAddress=certificate@trustcenter.de
    {0x61, 0x4F, 0xD1, 0x8D, 0xA1, 0x49, 0x05, 0x60, 0xCD, 0xAD, 0x11, 0x96,
     0xE2, 0x49, 0x2A, 0xB7, 0x06, 0x2E, 0xAB, 0x1A, 0x67, 0xB3, 0xA3, 0x0F,
     0x1D, 0x05, 0x85, 0xA7, 0xD6, 0xBA, 0x68, 0x24},
    // C=US, O=VeriSign, Inc., OU=Class 4 Public Primary Certification Authority
    // - G2, OU=(c) 1998 VeriSign, Inc. - For authorized use only, OU=VeriSign
    // Trust Network
    {0x61, 0xDC, 0x0C, 0x03, 0x91, 0x69, 0x4C, 0x65, 0x52, 0x00, 0xC1, 0x50,
     0x5E, 0xBC, 0xC9, 0xE4, 0xE2, 0x16, 0xBC, 0x31, 0xA5, 0xC5, 0x1A, 0x36,
     0x11, 0x28, 0x34, 0x23, 0xC1, 0xD8, 0x9E, 0x37},
    // C=ES, O=FNMT, OU=FNMT Clase 2 CA
    {0x62, 0xB9, 0x26, 0x72, 0x66, 0x21, 0x28, 0x32, 0xA8, 0xE2, 0x2D, 0xAB,
     0x93, 0x3D, 0x91, 0xC7, 0x01, 0x12, 0x74, 0xAC, 0xF7, 0x17, 0x03, 0xF9,
     0xCC, 0x97, 0x83, 0x37, 0x51, 0xA6, 0xE9, 0x4F},
    // C=CH, O=SwissSign AG, CN=SwissSign Gold CA - G2
    {0x62, 0xDD, 0x0B, 0xE9, 0xB9, 0xF5, 0x0A, 0x16, 0x3E, 0xA0, 0xF8, 0xE7,
     0x5C, 0x05, 0x3B, 0x1E, 0xCA, 0x57, 0xEA, 0x55, 0xC8, 0x68, 0x8F, 0x64,
     0x7C, 0x68, 0x81, 0xF2, 0xC8, 0x35, 0x7B, 0x95},
    // C=US, O=Entrust.net, OU=www.entrust.net/CPS incorp. by ref. (limits
    // liab.), OU=(c) 1999 Entrust.net Limited, CN=Entrust.net Secure Server
    // Certification Authority
    {0x62, 0xF2, 0x40, 0x27, 0x8C, 0x56, 0x4C, 0x4D, 0xD8, 0xBF, 0x7D, 0x9D,
     0x4F, 0x6F, 0x36, 0x6E, 0xA8, 0x94, 0xD2, 0x2F, 0x5F, 0x34, 0xD9, 0x89,
     0xA9, 0x83, 0xAC, 0xEC, 0x2F, 0xFF, 0xED, 0x50},
    // C=US, O=Digital Signature Trust Co., OU=DSTCA E1
    {0x63, 0x04, 0x19, 0xAE, 0xC4, 0x78, 0xCB, 0xB4, 0xBB, 0x80, 0x83, 0xDE,
     0x9D, 0x9C, 0xF2, 0x79, 0x75, 0x2F, 0x03, 0x9D, 0xEF, 0x16, 0xE4, 0x64,
     0x71, 0xB6, 0x79, 0xCA, 0x93, 0x00, 0x2D, 0xB0},
    // C=TW, O=TAIWAN-CA, OU=Root CA, CN=TWCA Root Certification Authority
    {0x63, 0x2D, 0x80, 0xBB, 0x09, 0x6D, 0x20, 0x96, 0x77, 0xD1, 0x73, 0x4E,
     0x5B, 0x35, 0xEA, 0x9D, 0x30, 0x19, 0xB9, 0xC4, 0x4F, 0x8F, 0xCB, 0x26,
     0x40, 0xC8, 0x79, 0x03, 0x9A, 0xC9, 0x4E, 0xE8},
    // C=NL, O=Staat der Nederlanden, CN=Staat der Nederlanden Root CA - G2
    {0x66, 0x8C, 0x83, 0x94, 0x7D, 0xA6, 0x3B, 0x72, 0x4B, 0xEC, 0xE1, 0x74,
     0x3C, 0x31, 0xA0, 0xE6, 0xAE, 0xD0, 0xDB, 0x8E, 0xC5, 0xB3, 0x1B, 0xE3,
     0x77, 0xBB, 0x78, 0x4F, 0x91, 0xB6, 0x71, 0x6F},
    // O=eSign Australia, OU=Gatekeeper PKI, CN=Gatekeeper Root CA
    {0x67, 0xEC, 0x20, 0x59, 0xFB, 0xF5, 0x2D, 0x2E, 0x6A, 0xB5, 0x1A, 0x5A,
     0x9B, 0x3F, 0xC2, 0xE1, 0xDC, 0xD6, 0x58, 0xA1, 0xEF, 0x3A, 0x8F, 0x31,
     0x10, 0x7B, 0xC9, 0x80, 0x28, 0xB4, 0x94, 0xA2},
    // C=US, O=VeriSign, Inc., OU=Class 3 Public Primary Certification Authority
    {0x68, 0x72, 0x58, 0x62, 0x19, 0xC3, 0x49, 0xD8, 0x5A, 0xAA, 0x45, 0x86,
     0xA1, 0x44, 0x51, 0xF2, 0x45, 0x1A, 0xE3, 0xB6, 0x09, 0x2D, 0xBB, 0x1E,
     0xFF, 0xB0, 0x14, 0x7C, 0x33, 0xBF, 0x0F, 0xD4},
    // C=BR, O=ICP-Brasil, OU=Instituto Nacional de Tecnologia da Informacao -
    // ITI, L=Brasilia, ST=DF, CN=Autoridade Certificadora Raiz Brasileira
    {0x68, 0x7E, 0xA8, 0x90, 0x89, 0x30, 0x9D, 0x2C, 0xFE, 0x10, 0x7D, 0xB0,
     0x59, 0xFB, 0x10, 0xD6, 0x76, 0xF4, 0x5D, 0x32, 0x83, 0xAE, 0xE0, 0x56,
     0x90, 0x3E, 0xEA, 0x0C, 0xF3, 0xC1, 0x88, 0xF8},
    // C=SE, O=AddTrust AB, OU=AddTrust External TTP Network, CN=AddTrust
    // External CA Root
    {0x68, 0x7F, 0xA4, 0x51, 0x38, 0x22, 0x78, 0xFF, 0xF0, 0xC8, 0xB1, 0x1F,
     0x8D, 0x43, 0xD5, 0x76, 0x67, 0x1C, 0x6E, 0xB2, 0xBC, 0xEA, 0xB4, 0x13,
     0xFB, 0x83, 0xD9, 0x65, 0xD0, 0x6D, 0x2F, 0xF2},
    // C=US, O=Verizon Business, OU=OmniRoot, CN=Verizon Global Root CA
    {0x68, 0xAD, 0x50, 0x90, 0x9B, 0x04, 0x36, 0x3C, 0x60, 0x5E, 0xF1, 0x35,
     0x81, 0xA9, 0x39, 0xFF, 0x2C, 0x96, 0x37, 0x2E, 0x3F, 0x12, 0x32, 0x5B,
     0x0A, 0x68, 0x61, 0xE1, 0xD5, 0x9F, 0x66, 0x03},
    // C=US, O=VeriSign, Inc., OU=VeriSign Trust Network, OU=(c) 2007 VeriSign,
    // Inc. - For authorized use only, CN=VeriSign Class 3 Public Primary
    // Certification Authority - G4
    {0x69, 0xDD, 0xD7, 0xEA, 0x90, 0xBB, 0x57, 0xC9, 0x3E, 0x13, 0x5D, 0xC8,
     0x5E, 0xA6, 0xFC, 0xD5, 0x48, 0x0B, 0x60, 0x32, 0x39, 0xBD, 0xC4, 0x54,
     0xFC, 0x75, 0x8B, 0x2A, 0x26, 0xCF, 0x7F, 0x79},
    // C=US, O=VISA, OU=Visa International Service Association, CN=Visa
    // eCommerce Root
    {0x69, 0xFA, 0xC9, 0xBD, 0x55, 0xFB, 0x0A, 0xC7, 0x8D, 0x53, 0xBB, 0xEE,
     0x5C, 0xF1, 0xD5, 0x97, 0x98, 0x9F, 0xD0, 0xAA, 0xAB, 0x20, 0xA2, 0x51,
     0x51, 0xBD, 0xF1, 0x73, 0x3E, 0xE7, 0xD1, 0x22},
    // C=AT, O=A-Trust Ges. f. Sicherheitssysteme im elektr. Datenverkehr GmbH,
    // OU=A-Trust-Qual-03, CN=A-Trust-Qual-03
    {0x6B, 0xAF, 0x50, 0xAE, 0x34, 0x67, 0xEF, 0xF3, 0xC3, 0x5F, 0xEF, 0xDC,
     0x76, 0xA0, 0x2A, 0x97, 0xFA, 0xB6, 0x26, 0x77, 0x23, 0xED, 0xA9, 0x1E,
     0x99, 0xF1, 0xB3, 0xDC, 0x2B, 0x28, 0xF8, 0x2E},
    // C=HU, L=Budapest, O=NetLock Kft.,
    // OU=Tan\xC3\xBAs\xC3\xADtv\xC3\xA1nykiad\xC3\xB3k (Certification
    // Services), CN=NetLock Arany (Class Gold)
    // F\xC5\x91tan\xC3\xBAs\xC3\xADtv\xC3\xA1ny
    {0x6C, 0x61, 0xDA, 0xC3, 0xA2, 0xDE, 0xF0, 0x31, 0x50, 0x6B, 0xE0, 0x36,
     0xD2, 0xA6, 0xFE, 0x40, 0x19, 0x94, 0xFB, 0xD1, 0x3D, 0xF9, 0xC8, 0xD4,
     0x66, 0x59, 0x92, 0x74, 0xC4, 0x46, 0xEC, 0x98},
    // C=FR, O=Certplus, CN=Certplus Root CA G2
    {0x6C, 0xC0, 0x50, 0x41, 0xE6, 0x44, 0x5E, 0x74, 0x69, 0x6C, 0x4C, 0xFB,
     0xC9, 0xF8, 0x0F, 0x54, 0x3B, 0x7E, 0xAB, 0xBB, 0x44, 0xB4, 0xCE, 0x6F,
     0x78, 0x7C, 0x6A, 0x99, 0x71, 0xC4, 0x2F, 0x17},
    // C=AU, O=GOV, OU=DoD, OU=PKI, OU=CAs, CN=ADOCA02
    {0x6C, 0xCF, 0xD3, 0x02, 0xFC, 0x44, 0xBF, 0x45, 0x99, 0x32, 0x9B, 0x97,
     0x50, 0x87, 0x8E, 0xA4, 0x4E, 0x7E, 0x85, 0x66, 0x56, 0x4B, 0xCB, 0xD5,
     0x86, 0x16, 0x97, 0x62, 0xDD, 0x10, 0xC7, 0x4E},
    // O=Entrust.net, OU=www.entrust.net/CPS_2048 incorp. by ref. (limits
    // liab.), OU=(c) 1999 Entrust.net Limited, CN=Entrust.net Certification
    // Authority (2048)
    {0x6D, 0xC4, 0x71, 0x72, 0xE0, 0x1C, 0xBC, 0xB0, 0xBF, 0x62, 0x58, 0x0D,
     0x89, 0x5F, 0xE2, 0xB8, 0xAC, 0x9A, 0xD4, 0xF8, 0x73, 0x80, 0x1E, 0x0C,
     0x10, 0xB9, 0xC8, 0x37, 0xD2, 0x1E, 0xB1, 0x77},
    // C=ES, ST=Madrid, L=Madrid, O=IPS Certification Authority s.l. ipsCA,
    // OU=ipsCA, CN=ipsCA Global CA Root/emailAddress=global01@ipsca.com
    {0x6D, 0xEA, 0x86, 0xA1, 0xE6, 0x66, 0x20, 0xA0, 0x40, 0xC3, 0xC5, 0x94,
     0x3C, 0xB2, 0x15, 0xD2, 0xCA, 0x87, 0xFB, 0x6A, 0xC0, 0x9B, 0x59, 0x70,
     0x7E, 0x29, 0xD2, 0xFA, 0xCB, 0xD6, 0x6B, 0x4E},
    // C=us, O=U.S. Government, OU=FBCA, CN=Common Policy
    {0x6E, 0x5E, 0x93, 0xAE, 0x86, 0x7F, 0xD3, 0xE3, 0xE7, 0x83, 0x04, 0xE0,
     0x54, 0xD1, 0xA6, 0xAE, 0xAE, 0xD0, 0x29, 0x5D, 0x58, 0xC0, 0xE3, 0xFC,
     0x4C, 0x9F, 0xFE, 0x31, 0x0A, 0x34, 0x88, 0xCC},
    // C=US, ST=UT, L=Salt Lake City, O=The USERTRUST Network,
    // OU=http://www.usertrust.com, CN=UTN-USERFirst-Hardware
    {0x6E, 0xA5, 0x47, 0x41, 0xD0, 0x04, 0x66, 0x7E, 0xED, 0x1B, 0x48, 0x16,
     0x63, 0x4A, 0xA3, 0xA7, 0x9E, 0x6E, 0x4B, 0x96, 0x95, 0x0F, 0x82, 0x79,
     0xDA, 0xFC, 0x8D, 0x9B, 0xD8, 0x81, 0x21, 0x37},
    // C=CH, O=The Federal Authorities of the Swiss Confederation, OU=Services,
    // OU=Certification Authorities, CN=Swiss Government Root CA I
    {0x6E, 0xC6, 0x61, 0x4E, 0x9A, 0x8E, 0xFD, 0x47, 0xD6, 0x31, 0x8F, 0xFD,
     0xFD, 0x0B, 0xF6, 0x5B, 0x49, 0x3A, 0x14, 0x1F, 0x77, 0xC3, 0x8D, 0x0B,
     0x31, 0x9B, 0xE1, 0xBB, 0xBC, 0x05, 0x3D, 0xD2},
    // C=KR, O=KISA, OU=Korea Certification Authority Central, CN=KISA RootCA 1
    {0x6F, 0xDB, 0x3F, 0x76, 0xC8, 0xB8, 0x01, 0xA7, 0x53, 0x38, 0xD8, 0xA5,
     0x0A, 0x7C, 0x02, 0x87, 0x9F, 0x61, 0x98, 0xB5, 0x7E, 0x59, 0x4D, 0x31,
     0x8D, 0x38, 0x32, 0x90, 0x0F, 0xED, 0xCD, 0x79},
    // C=US, O=AffirmTrust, CN=AffirmTrust Premium
    {0x70, 0xA7, 0x3F, 0x7F, 0x37, 0x6B, 0x60, 0x07, 0x42, 0x48, 0x90, 0x45,
     0x34, 0xB1, 0x14, 0x82, 0xD5, 0xBF, 0x0E, 0x69, 0x8E, 0xCC, 0x49, 0x8D,
     0xF5, 0x25, 0x77, 0xEB, 0xF2, 0xE9, 0x3B, 0x9A},
    // C=TW, O=Government Root Certification Authority
    {0x70, 0xB9, 0x22, 0xBF, 0xDA, 0x0E, 0x3F, 0x4A, 0x34, 0x2E, 0x4E, 0xE2,
     0x2D, 0x57, 0x9A, 0xE5, 0x98, 0xD0, 0x71, 0xCC, 0x5E, 0xC9, 0xC3, 0x0F,
     0x12, 0x36, 0x80, 0x34, 0x03, 0x88, 0xAE, 0xA5},
    // C=PL, O=Telekomunikacja Polska S.A., OU=Signet Certification Authority,
    // CN=Signet Root CA
    {0x72, 0x86, 0xCE, 0x24, 0x9F, 0xE9, 0xE3, 0x2B, 0xD4, 0x75, 0x22, 0x57,
     0xC1, 0x7C, 0xD8, 0xF6, 0x99, 0x1A, 0x9C, 0x1E, 0x6F, 0x1A, 0x3C, 0xC7,
     0x33, 0x04, 0xED, 0x02, 0x3E, 0x6A, 0xE4, 0xEB},
    // C=NL, O=PTT Post, OU=KeyMail, CN=PTT Post Root CA/mail=ca@ptt-post.nl
    {0x73, 0x0B, 0x61, 0x9E, 0xAA, 0x75, 0x98, 0x63, 0xC6, 0x53, 0x60, 0xB7,
     0x41, 0x2E, 0x14, 0x57, 0xEC, 0xA9, 0x68, 0x44, 0xEF, 0x2F, 0x16, 0xD9,
     0x1F, 0xCF, 0x2E, 0xFE, 0x46, 0xA6, 0x47, 0xE9},
    // C=TR, O=Elektronik Bilgi Guvenligi A.S., CN=E-GUVEN Kok Elektronik
    // Sertifika Hizmet Saglayicisi S2
    {0x73, 0x2F, 0xF5, 0xBC, 0x47, 0x3C, 0x33, 0xEF, 0xAA, 0xD9, 0xB1, 0x87,
     0xFB, 0x45, 0x9A, 0x73, 0x2F, 0xFB, 0xEE, 0xA5, 0x9D, 0xF0, 0xFD, 0xFB,
     0x8E, 0x2F, 0xB5, 0xDE, 0xF4, 0x03, 0x2A, 0xFE},
    // C=ES, O=DIRECCION GENERAL DE LA POLICIA, OU=DNIE, CN=AC RAIZ DNIE
    {0x73, 0x97, 0x10, 0xC5, 0x24, 0x5E, 0x33, 0xEC, 0x8A, 0x24, 0x3A, 0x1B,
     0x20, 0x04, 0x8F, 0xC9, 0xD5, 0xF4, 0x52, 0x85, 0x99, 0x21, 0x38, 0x45,
     0xC1, 0x64, 0xD0, 0x04, 0xB8, 0xB6, 0x67, 0xF9},
    // C=US, O=Entrust, Inc., OU=www.entrust.net/CPS is incorporated by
    // reference, OU=(c) 2006 Entrust, Inc., CN=Entrust Root Certification
    // Authority
    {0x73, 0xC1, 0x76, 0x43, 0x4F, 0x1B, 0xC6, 0xD5, 0xAD, 0xF4, 0x5B, 0x0E,
     0x76, 0xE7, 0x27, 0x28, 0x7C, 0x8D, 0xE5, 0x76, 0x16, 0xC1, 0xE6, 0xE6,
     0x14, 0x1A, 0x2B, 0x2C, 0xBC, 0x7D, 0x8E, 0x4C},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert High Assurance EV
    // Root CA
    {0x74, 0x31, 0xE5, 0xF4, 0xC3, 0xC1, 0xCE, 0x46, 0x90, 0x77, 0x4F, 0x0B,
     0x61, 0xE0, 0x54, 0x40, 0x88, 0x3B, 0xA9, 0xA0, 0x1E, 0xD0, 0x0B, 0xA6,
     0xAB, 0xD7, 0x80, 0x6E, 0xD3, 0xB1, 0x18, 0xCF},
    // C=PA, ST=Panama, L=Panama City, O=TrustCor Systems S. de R.L.,
    // OU=TrustCor Certificate Authority, CN=TrustCor ECA-1
    {0x74, 0x4B, 0x11, 0x47, 0xB4, 0xA9, 0xA6, 0x9C, 0x32, 0x78, 0x5E, 0x9E,
     0x37, 0xC3, 0x32, 0x32, 0x41, 0xEF, 0x29, 0xF6, 0x3E, 0x76, 0xF1, 0x60,
     0x3D, 0x67, 0x61, 0xA7, 0x83, 0xD8, 0xA0, 0xFE},
    // C=si, O=state-institutions, OU=sigov-ca
    {0x74, 0xCB, 0x3A, 0x4E, 0xA7, 0x91, 0xAF, 0xB0, 0xA2, 0xD1, 0xA0, 0xB1,
     0x33, 0x01, 0xB3, 0xBE, 0xE0, 0xE5, 0x0A, 0xD5, 0xC7, 0x9A, 0x1A, 0x6F,
     0x2C, 0x66, 0x3E, 0x6F, 0x4E, 0xE7, 0xA4, 0x84},
    // C=AT, O=A-Trust Ges. f. Sicherheitssysteme im elektr. Datenverkehr GmbH,
    // OU=A-Trust-Qual-02, CN=A-Trust-Qual-02
    {0x75, 0xC9, 0xD4, 0x36, 0x1C, 0xB9, 0x6E, 0x99, 0x3A, 0xBD, 0x96, 0x20,
     0xCF, 0x04, 0x3B, 0xE9, 0x40, 0x7A, 0x46, 0x33, 0xF2, 0x02, 0xF0, 0xF4,
     0xC0, 0xE1, 0x78, 0x51, 0xCC, 0x60, 0x89, 0xCD},
    // C=TW, O=Government Root Certification Authority
    {0x76, 0x00, 0x29, 0x5E, 0xEF, 0xE8, 0x5B, 0x9E, 0x1F, 0xD6, 0x24, 0xDB,
     0x76, 0x06, 0x2A, 0xAA, 0xAE, 0x59, 0x81, 0x8A, 0x54, 0xD2, 0x77, 0x4C,
     0xD4, 0xC0, 0xB2, 0xC0, 0x11, 0x31, 0xE1, 0xB3},
    // C=US, O=Digital Signature Trust, OU=DST ACES, CN=DST ACES CA X6
    {0x76, 0x7C, 0x95, 0x5A, 0x76, 0x41, 0x2C, 0x89, 0xAF, 0x68, 0x8E, 0x90,
     0xA1, 0xC7, 0x0F, 0x55, 0x6C, 0xFD, 0x6B, 0x60, 0x25, 0xDB, 0xEA, 0x10,
     0x41, 0x6D, 0x7E, 0xB6, 0x83, 0x1F, 0x8C, 0x40},
    // C=DE, ST=Hamburg, L=Hamburg, O=TC TrustCenter for Security in Data
    // Networks GmbH, OU=TC TrustCenter Class 3
    // CA/emailAddress=certificate@trustcenter.de
    {0x76, 0xEF, 0x47, 0x62, 0xE5, 0x73, 0x20, 0x60, 0x06, 0xCB, 0xC3, 0x38,
     0xB1, 0x7C, 0xA4, 0xBC, 0x20, 0x05, 0x74, 0xA1, 0x19, 0x28, 0xD9, 0x0C,
     0x3E, 0xF3, 0x1C, 0x5E, 0x80, 0x3E, 0x6C, 0x6F},
    // L=Alvaro Obregon, ST=Distrito Federal,
    // C=MX/postalCode=01030/street=Insurgentes Sur 1940, CN=Autoridad
    // Certificadora Raiz de la Secretaria de Economia, OU=Direccion General de
    // Normatividad Mercantil, O=Secretaria de
    // Economia/emailAddress=acrse@economia.gob.mx
    {0x77, 0xE0, 0x4C, 0x9A, 0x75, 0x1C, 0x73, 0xF2, 0x3E, 0x2A, 0x13, 0x36,
     0x11, 0x2E, 0xC8, 0xD5, 0x15, 0x3D, 0x38, 0x2A, 0x15, 0x2F, 0xED, 0x89,
     0xD7, 0x53, 0x2C, 0x31, 0x02, 0x77, 0x1F, 0x3C},
    // C=SG, O=Netrust Certificate Authority 1, OU=Netrust CA1
    {0x78, 0x1D, 0x64, 0xDF, 0xA7, 0x7B, 0x00, 0xF2, 0xC0, 0x06, 0x70, 0x0B,
     0x1F, 0xDA, 0x86, 0xBF, 0x68, 0xB8, 0x65, 0xA6, 0x03, 0xC7, 0xA6, 0x56,
     0xF9, 0x2E, 0x90, 0xC0, 0x42, 0xCA, 0x28, 0x73},
    // C=FI, O=Sonera, CN=Sonera Class2 CA
    {0x79, 0x08, 0xB4, 0x03, 0x14, 0xC1, 0x38, 0x10, 0x0B, 0x51, 0x8D, 0x07,
     0x35, 0x80, 0x7F, 0xFB, 0xFC, 0xF8, 0x51, 0x8A, 0x00, 0x95, 0x33, 0x71,
     0x05, 0xBA, 0x38, 0x6B, 0x15, 0x3D, 0xD9, 0x27},
    // C=AT, O=A-Trust Ges. f. Sicherheitssysteme im elektr. Datenverkehr GmbH,
    // OU=A-Trust-nQual-03, CN=A-Trust-nQual-03
    {0x79, 0x3C, 0xBF, 0x45, 0x59, 0xB9, 0xFD, 0xE3, 0x8A, 0xB2, 0x2D, 0xF1,
     0x68, 0x69, 0xF6, 0x98, 0x81, 0xAE, 0x14, 0xC4, 0xB0, 0x13, 0x9A, 0xC7,
     0x88, 0xA7, 0x8A, 0x1A, 0xFC, 0xCA, 0x02, 0xFB},
    // C=ZA, ST=Western Cape, L=Somerset West, O=South African Post Office
    // Limited, OU=SAPO Trust Centre, CN=SAPO Class 3 Root
    // CA/emailAddress=pkiadmin@trustcentre.co.za
    {0x79, 0x6B, 0x93, 0xD0, 0xA3, 0xBA, 0x22, 0xE1, 0x91, 0xF2, 0x49, 0x5F,
     0x15, 0x0A, 0x5F, 0x9B, 0xFE, 0xE2, 0xCE, 0x15, 0x03, 0xDA, 0x21, 0x7B,
     0x74, 0xB8, 0xA9, 0x84, 0x68, 0x51, 0x10, 0xF1},
    // C=US, O=VeriSign, Inc., OU=Class 3 Public Primary Certification Authority
    // - G2, OU=(c) 1998 VeriSign, Inc. - For authorized use only, OU=VeriSign
    // Trust Network
    {0x79, 0x7E, 0x51, 0xF8, 0x83, 0xE8, 0x55, 0xD0, 0x21, 0xE5, 0xC7, 0x70,
     0x56, 0x66, 0x92, 0x99, 0x94, 0x07, 0x89, 0x55, 0x93, 0x23, 0x5D, 0xEF,
     0x52, 0xA0, 0x11, 0xF7, 0x16, 0xF8, 0xB6, 0xBF},
    // C=US, ST=MN, L=Minneapolis, O=Open Access Technology International Inc,
    // CN=OATI WebCARES Root CA
    {0x7A, 0x77, 0xC6, 0xC6, 0x1E, 0xEE, 0xB9, 0xAA, 0x65, 0xC4, 0xEA, 0x41,
     0x0D, 0x65, 0xD8, 0x95, 0xB2, 0x6A, 0x81, 0x12, 0x32, 0x83, 0x00, 0x9D,
     0xB1, 0x04, 0xB4, 0x8D, 0xE8, 0x0B, 0x24, 0x79},
    // C=CH, O=SwissSign AG, CN=SwissSign Gold Root CA - G3
    {0x7A, 0xF6, 0xEA, 0x9F, 0x75, 0x3A, 0x1E, 0x70, 0x9B, 0xD6, 0x4D, 0x0B,
     0xEB, 0x86, 0x7C, 0x11, 0xE8, 0xC2, 0x95, 0xA5, 0x6E, 0x24, 0xA6, 0xE0,
     0x47, 0x14, 0x59, 0xDC, 0xCD, 0xAA, 0x15, 0x58},
    // C=us, ST=Utah, L=Salt Lake City, O=Digital Signature Trust Co., OU=DSTCA
    // X1, CN=DST RootCA X1/emailAddress=ca@digsigtrust.com
    {0x7B, 0x1A, 0x15, 0xD7, 0xE5, 0xE1, 0x30, 0xC5, 0x79, 0xE6, 0x8F, 0xCA,
     0x18, 0x92, 0x57, 0xF8, 0x37, 0xB5, 0xC1, 0x88, 0xF1, 0xB2, 0xB2, 0xA7,
     0x91, 0xE9, 0x67, 0xCC, 0x88, 0xCC, 0x65, 0x28},
    // C=AT, O=A-Trust, OU=A-Trust-nQual-01, CN=A-Trust-nQual-01
    {0x7B, 0x1F, 0x8D, 0x8E, 0xFF, 0x5D, 0x73, 0x49, 0xFE, 0xDB, 0x7E, 0xAE,
     0x89, 0xC2, 0x9A, 0xAC, 0xC4, 0x17, 0x04, 0xF1, 0x50, 0x3A, 0xE3, 0xC8,
     0xC2, 0xEB, 0xA1, 0x02, 0x25, 0xD0, 0xF5, 0x68},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert Assured ID Root G2
    {0x7D, 0x05, 0xEB, 0xB6, 0x82, 0x33, 0x9F, 0x8C, 0x94, 0x51, 0xEE, 0x09,
     0x4E, 0xEB, 0xFE, 0xFA, 0x79, 0x53, 0xA1, 0x14, 0xED, 0xB2, 0xF4, 0x49,
     0x49, 0x45, 0x2F, 0xAB, 0x7D, 0x2F, 0xC1, 0x85},
    // C=ES, O=Colegio de Registradores de la Propiedad y Mercantiles de
    // Espa\xC3\xB1a, OU=Certificado Propio, CN=Registradores de Espa\xC3\xB1a -
    // CA Ra\xC3\xADz
    {0x7D, 0x2B, 0xF3, 0x48, 0x9E, 0xBC, 0x9A, 0xD3, 0x44, 0x8B, 0x8B, 0x08,
     0x27, 0x71, 0x5A, 0x3C, 0xBF, 0xE3, 0xD5, 0x23, 0xE3, 0xB5, 0x6A, 0x9B,
     0x5F, 0xC1, 0xD2, 0xA2, 0xDA, 0x2F, 0x20, 0xFE},
    // C=US, O=America Online Inc., CN=America Online Root Certification
    // Authority 2
    {0x7D, 0x3B, 0x46, 0x5A, 0x60, 0x14, 0xE5, 0x26, 0xC0, 0xAF, 0xFC, 0xEE,
     0x21, 0x27, 0xD2, 0x31, 0x17, 0x27, 0xAD, 0x81, 0x1C, 0x26, 0x84, 0x2D,
     0x00, 0x6A, 0xF3, 0x73, 0x06, 0xCC, 0x80, 0xBD},
    // C=FR, O=Certplus, CN=Class 1 Primary CA
    {0x7D, 0x8C, 0xE8, 0x22, 0x22, 0x2B, 0x90, 0xC0, 0xB1, 0x43, 0x42, 0xC7,
     0xA8, 0x14, 0x5D, 0x1F, 0x24, 0x35, 0x1F, 0x4D, 0x1A, 0x1F, 0xE0, 0xED,
     0xFD, 0x31, 0x2E, 0xE7, 0x3F, 0xB0, 0x01, 0x49},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert Assured ID Root G3
    {0x7E, 0x37, 0xCB, 0x8B, 0x4C, 0x47, 0x09, 0x0C, 0xAB, 0x36, 0x55, 0x1B,
     0xA6, 0xF4, 0x5D, 0xB8, 0x40, 0x68, 0x0F, 0xBA, 0x16, 0x6A, 0x95, 0x2D,
     0xB1, 0x00, 0x71, 0x7F, 0x43, 0x05, 0x3F, 0xC2},
    // C=HU, ST=Hungary, L=Budapest, O=NetLock Halozatbiztonsagi Kft.,
    // OU=Tanusitvanykiadok, CN=NetLock Kozjegyzoi (Class A) Tanusitvanykiado
    {0x7F, 0x12, 0xCD, 0x5F, 0x7E, 0x5E, 0x29, 0x0E, 0xC7, 0xD8, 0x51, 0x79,
     0xD5, 0xB7, 0x2C, 0x20, 0xA5, 0xBE, 0x75, 0x08, 0xFF, 0xDB, 0x5B, 0xF8,
     0x1A, 0xB9, 0x68, 0x4A, 0x7F, 0xC9, 0xF6, 0x67},
    // C=ES, O=Agencia Notarial de Certificacion S.L. Unipersonal - CIF
    // B83395988, CN=ANCERT Certificados CGN
    {0x81, 0x77, 0xD1, 0xA8, 0x2B, 0xA5, 0x01, 0xAF, 0xDD, 0x1E, 0x94, 0x83,
     0xAD, 0x7D, 0xA9, 0x12, 0xEE, 0x1E, 0x9F, 0xCB, 0x2A, 0x5A, 0x06, 0x1F,
     0xA3, 0xC4, 0x79, 0xFA, 0x80, 0x4C, 0xE6, 0xBA},
    // C=TR, O=Elektronik Bilgi Guvenligi A.S., CN=E-GUVEN Kok Elektronik
    // Sertifika Hizmet Saglayicisi S3
    {0x82, 0xD4, 0x2D, 0xB3, 0xD6, 0x57, 0xF1, 0x94, 0x4E, 0x65, 0xC1, 0x92,
     0xB1, 0xDD, 0x58, 0xDB, 0x8D, 0xF8, 0x41, 0x7B, 0x89, 0x16, 0x5B, 0x04,
     0x5F, 0x5C, 0x6A, 0x70, 0xC5, 0xF8, 0x93, 0x9E},
    // O=Cisco Systems, CN=Cisco Root CA 2048
    {0x83, 0x27, 0xBC, 0x8C, 0x9D, 0x69, 0x94, 0x7B, 0x3D, 0xE3, 0xC2, 0x75,
     0x11, 0x53, 0x72, 0x67, 0xF5, 0x9C, 0x21, 0xB9, 0xFA, 0x7B, 0x61, 0x3F,
     0xAF, 0xBC, 0xCD, 0x53, 0xB7, 0x02, 0x40, 0x00},
    // C=US, O=VeriSign, Inc., OU=Class 3 Public Primary Certification Authority
    // - G2, OU=(c) 1998 VeriSign, Inc. - For authorized use only, OU=VeriSign
    // Trust Network
    {0x83, 0xCE, 0x3C, 0x12, 0x29, 0x68, 0x8A, 0x59, 0x3D, 0x48, 0x5F, 0x81,
     0x97, 0x3C, 0x0F, 0x91, 0x95, 0x43, 0x1E, 0xDA, 0x37, 0xCC, 0x5E, 0x36,
     0x43, 0x0E, 0x79, 0xC7, 0xA8, 0x88, 0x63, 0x8B},
    // C=US, ST=Washington, L=Redmond, O=Microsoft Corporation, CN=Microsoft
    // Root Certificate Authority 2011
    {0x84, 0x7D, 0xF6, 0xA7, 0x84, 0x97, 0x94, 0x3F, 0x27, 0xFC, 0x72, 0xEB,
     0x93, 0xF9, 0xA6, 0x37, 0x32, 0x0A, 0x02, 0xB5, 0x61, 0xD0, 0xA9, 0x1B,
     0x09, 0xE8, 0x7A, 0x78, 0x07, 0xED, 0x7C, 0x61},
    // C=BM, O=QuoVadis Limited, CN=QuoVadis Root CA 2
    {0x85, 0xA0, 0xDD, 0x7D, 0xD7, 0x20, 0xAD, 0xB7, 0xFF, 0x05, 0xF8, 0x3D,
     0x54, 0x2B, 0x20, 0x9D, 0xC7, 0xFF, 0x45, 0x28, 0xF7, 0xD6, 0x77, 0xB1,
     0x83, 0x89, 0xFE, 0xA5, 0xE5, 0xC4, 0x9E, 0x86},
    // C=BR, O=Certisign Certificadora Digital Ltda., OU=Certisign Autoridade
    // Certificadora AC1S
    {0x85, 0xE0, 0xDF, 0xAE, 0x3E, 0x55, 0xA8, 0x43, 0x19, 0x5F, 0x8B, 0x08,
     0xC8, 0x34, 0x90, 0x50, 0xE4, 0x68, 0x93, 0x72, 0xF6, 0xE1, 0x33, 0xAD,
     0x0D, 0x19, 0x9A, 0xF9, 0x6E, 0x95, 0xCC, 0x08},
    // C=US, ST=UT, L=Salt Lake City, O=The USERTRUST Network,
    // OU=http://www.usertrust.com, CN=UTN - DATACorp SGC
    {0x85, 0xFB, 0x2F, 0x91, 0xDD, 0x12, 0x27, 0x5A, 0x01, 0x45, 0xB6, 0x36,
     0x53, 0x4F, 0x84, 0x02, 0x4A, 0xD6, 0x8B, 0x69, 0xB8, 0xEE, 0x88, 0x68,
     0x4F, 0xF7, 0x11, 0x37, 0x58, 0x05, 0xB3, 0x48},
    // C=AT, O=A-Trust Ges. f. Sicherheitssysteme im elektr. Datenverkehr GmbH,
    // OU=A-Trust-nQual-03, CN=A-Trust-nQual-03
    {0x86, 0x88, 0xE5, 0x8F, 0x4C, 0x7A, 0x94, 0x5F, 0xAD, 0xCE, 0x7F, 0x62,
     0xBF, 0xEF, 0x52, 0x1B, 0x82, 0xDA, 0x7D, 0xC3, 0x8B, 0xFD, 0xB0, 0x16,
     0x34, 0x78, 0xA5, 0xFE, 0x42, 0xE5, 0x78, 0x70},
    // C=ZA, ST=Western Cape, L=Cape Town, O=Thawte Consulting cc,
    // OU=Certification Services Division, CN=Thawte Server
    // CA/emailAddress=server-certs@thawte.com
    {0x87, 0xC6, 0x78, 0xBF, 0xB8, 0xB2, 0x5F, 0x38, 0xF7, 0xE9, 0x7B, 0x33,
     0x69, 0x56, 0xBB, 0xCF, 0x14, 0x4B, 0xBA, 0xCA, 0xA5, 0x36, 0x47, 0xE6,
     0x1A, 0x23, 0x25, 0xBC, 0x10, 0x55, 0x31, 0x6B},
    // C=ES, O=Agencia Catalana de Certificacio (NIF Q-0801176-I), OU=Serveis
    // Publics de Certificacio, OU=Vegeu https://www.catcert.net/verarrel (c)03,
    // OU=Jerarquia Entitats de Certificacio Catalanes, CN=EC-ACC
    {0x88, 0x49, 0x7F, 0x01, 0x60, 0x2F, 0x31, 0x54, 0x24, 0x6A, 0xE2, 0x8C,
     0x4D, 0x5A, 0xEF, 0x10, 0xF1, 0xD8, 0x7E, 0xBB, 0x76, 0x62, 0x6F, 0x4A,
     0xE0, 0xB7, 0xF9, 0x5B, 0xA7, 0x96, 0x87, 0x99},
    // DC=com, DC=microsoft, CN=Microsoft Root Certificate Authority
    {0x88, 0x5D, 0xE6, 0x4C, 0x34, 0x0E, 0x3E, 0xA7, 0x06, 0x58, 0xF0, 0x1E,
     0x11, 0x45, 0xF9, 0x57, 0xFC, 0xDA, 0x27, 0xAA, 0xBE, 0xEA, 0x1A, 0xB9,
     0xFA, 0xA9, 0xFD, 0xB0, 0x10, 0x2D, 0x40, 0x77},
    // C=BM, O=QuoVadis Limited, CN=QuoVadis Root CA 3 G3
    {0x88, 0xEF, 0x81, 0xDE, 0x20, 0x2E, 0xB0, 0x18, 0x45, 0x2E, 0x43, 0xF8,
     0x64, 0x72, 0x5C, 0xEA, 0x5F, 0xBD, 0x1F, 0xC2, 0xD9, 0xD2, 0x05, 0x73,
     0x07, 0x09, 0xC5, 0xD8, 0xB8, 0x69, 0x0F, 0x46},
    // C=SI, O=ACNLB
    {0x89, 0x4C, 0xE6, 0xDD, 0xB0, 0x12, 0xCB, 0x3F, 0x73, 0x69, 0x54, 0x66,
     0x8D, 0xE6, 0x3F, 0x43, 0x60, 0x80, 0xE9, 0x5F, 0x17, 0xB7, 0xA8, 0x1B,
     0xD9, 0x24, 0xEB, 0x21, 0xBE, 0xE9, 0xE4, 0x40},
    // C=US, O=U.S. Government, OU=FPKI, CN=Federal Common Policy CA
    {0x89, 0x4E, 0xBC, 0x0B, 0x23, 0xDA, 0x2A, 0x50, 0xC0, 0x18, 0x6B, 0x7F,
     0x8F, 0x25, 0xEF, 0x1F, 0x6B, 0x29, 0x35, 0xAF, 0x32, 0xA9, 0x45, 0x84,
     0xEF, 0x80, 0xAA, 0xF8, 0x77, 0xA3, 0xA0, 0x6E},
    // C=BM, O=QuoVadis Limited, CN=QuoVadis Root CA 1 G3
    {0x8A, 0x86, 0x6F, 0xD1, 0xB2, 0x76, 0xB5, 0x7E, 0x57, 0x8E, 0x92, 0x1C,
     0x65, 0x82, 0x8A, 0x2B, 0xED, 0x58, 0xE9, 0xF2, 0xF2, 0x88, 0x05, 0x41,
     0x34, 0xB7, 0xF1, 0xF4, 0xBF, 0xC9, 0xCC, 0x74},
    // C=SI, O=Halcom, CN=Halcom Root CA
    {0x8B, 0x3F, 0xDB, 0x15, 0x1A, 0xF7, 0x59, 0xC5, 0x66, 0x14, 0x3E, 0x07,
     0xC9, 0x50, 0xED, 0xE4, 0xF9, 0xE8, 0xC7, 0xCF, 0x80, 0x84, 0x53, 0xD3,
     0x3B, 0xCB, 0x78, 0xE5, 0x2A, 0x40, 0x0A, 0xF9},
    // C=CN, O=WoSign CA Limited, CN=CA WoSign ECC Root
    {0x8B, 0x45, 0xDA, 0x1C, 0x06, 0xF7, 0x91, 0xEB, 0x0C, 0xAB, 0xF2, 0x6B,
     0xE5, 0x88, 0xF5, 0xFB, 0x23, 0x16, 0x5C, 0x2E, 0x61, 0x4B, 0xF8, 0x85,
     0x56, 0x2D, 0x0D, 0xCE, 0x50, 0xB2, 0x9B, 0x02},
    // C=ES, O=Generalitat Valenciana, OU=PKIGVA, CN=Root CA Generalitat
    // Valenciana
    {0x8C, 0x4E, 0xDF, 0xD0, 0x43, 0x48, 0xF3, 0x22, 0x96, 0x9E, 0x7E, 0x29,
     0xA4, 0xCD, 0x4D, 0xCA, 0x00, 0x46, 0x55, 0x06, 0x1C, 0x16, 0xE1, 0xB0,
     0x76, 0x42, 0x2E, 0xF3, 0x42, 0xAD, 0x63, 0x0E},
    // C=US, O=thawte, Inc., OU=Certification Services Division, OU=(c) 2006
    // thawte, Inc. - For authorized use only, CN=thawte Primary Root CA
    {0x8D, 0x72, 0x2F, 0x81, 0xA9, 0xC1, 0x13, 0xC0, 0x79, 0x1D, 0xF1, 0x36,
     0xA2, 0x96, 0x6D, 0xB2, 0x6C, 0x95, 0x0A, 0x97, 0x1D, 0xB4, 0x6B, 0x41,
     0x99, 0xF4, 0xEA, 0x54, 0xB7, 0x8B, 0xFB, 0x9F},
    // C=DE, O=D-Trust GmbH, CN=D-TRUST Root Class 2 CA 2007
    {0x8D, 0xA7, 0x5F, 0x13, 0x27, 0x21, 0x7C, 0x88, 0x06, 0x0F, 0xD2, 0x52,
     0x9E, 0xFF, 0x28, 0x16, 0xE5, 0x0B, 0x0C, 0x74, 0x54, 0x1E, 0xA4, 0xEA,
     0x3D, 0xFC, 0xEE, 0x66, 0xA7, 0x1E, 0xFE, 0x09},
    // C=DE, ST=Hamburg, L=Hamburg, O=TC TrustCenter for Security in Data
    // Networks GmbH, OU=TC TrustCenter Class 4
    // CA/emailAddress=certificate@trustcenter.de
    {0x8D, 0xBB, 0x5A, 0x7C, 0x06, 0xC2, 0x0E, 0xF6, 0x2D, 0xD9, 0x12, 0xA3,
     0x67, 0x40, 0x99, 0x2F, 0xF6, 0xE1, 0xE8, 0x58, 0x3D, 0x42, 0xED, 0xE2,
     0x57, 0xC3, 0xAF, 0xFD, 0x7C, 0x76, 0x93, 0x99},
    // C=TR, L=Ankara, O=T\xC3\x9CRKTRUST Bilgi \xC4\xB0leti\xC5\x9Fim ve
    // Bili\xC5\x9Fim G\xC3\xBCvenli\xC4\x9Fi Hizmetleri A.\xC5\x9E.,
    // CN=T\xC3\x9CRKTRUST Elektronik Sertifika Hizmet
    // Sa\xC4\x9Flay\xC4\xB1c\xC4\xB1s\xC4\xB1 H6
    {0x8D, 0xE7, 0x86, 0x55, 0xE1, 0xBE, 0x7F, 0x78, 0x47, 0x80, 0x0B, 0x93,
     0xF6, 0x94, 0xD2, 0x1D, 0x36, 0x8C, 0xC0, 0x6E, 0x03, 0x3E, 0x7F, 0xAB,
     0x04, 0xBB, 0x5E, 0xB9, 0x9D, 0xA6, 0xB7, 0x00},
    // C=HU, L=Budapest, O=Microsec Ltd., CN=Microsec e-Szigno Root CA
    // 2009/emailAddress=info@e-szigno.hu
    {0x8E, 0x8C, 0x6E, 0xBF, 0x77, 0xDC, 0x73, 0xDB, 0x3E, 0x38, 0xE9, 0x3F,
     0x48, 0x03, 0xE6, 0x2B, 0x6B, 0x59, 0x33, 0xBE, 0xB5, 0x1E, 0xE4, 0x15,
     0x2F, 0x68, 0xD7, 0xAA, 0x14, 0x42, 0x6B, 0x31},
    // CN=SG TRUST SERVICES RACINE, OU=0002 43525289500022, O=SG TRUST SERVICES,
    // C=FR
    {0x8F, 0x1E, 0xCD, 0xAF, 0x29, 0xBC, 0xD5, 0x6E, 0xDD, 0xD6, 0xB5, 0xD5,
     0x6A, 0x07, 0xFC, 0xAC, 0x2B, 0x74, 0xD4, 0xBC, 0xD1, 0x79, 0x17, 0x91,
     0x44, 0xA0, 0x36, 0x5C, 0x27, 0xDC, 0xF1, 0x4B},
    // C=US, O=Digital Signature Trust Co., OU=DSTCA E2
    {0x8F, 0x62, 0xD7, 0x73, 0x6F, 0x99, 0xDB, 0xD3, 0x3E, 0xE0, 0x0E, 0x10,
     0xC7, 0xE3, 0x29, 0x33, 0x9C, 0x98, 0x8A, 0x5B, 0x47, 0xEF, 0x25, 0xF4,
     0x08, 0x29, 0x3C, 0xF2, 0x42, 0x6B, 0x4D, 0x44},
    // C=US, O=First Data Digital Certificates Inc., CN=First Data Digital
    // Certificates Inc. Certification Authority
    {0x8F, 0x63, 0x3D, 0xF0, 0x19, 0x1A, 0x41, 0x7D, 0xCD, 0x4F, 0x63, 0x31,
     0xF0, 0xE9, 0x0E, 0xB9, 0x47, 0x15, 0x8C, 0x8A, 0x3E, 0xA5, 0x9B, 0xFE,
     0x81, 0x42, 0x00, 0x41, 0x31, 0x5E, 0xAC, 0x6C},
    // C=DE, ST=Hamburg, L=Hamburg, O=TC TrustCenter for Security in Data
    // Networks GmbH, OU=TC TrustCenter Class 2
    // CA/emailAddress=certificate@trustcenter.de
    {0x8F, 0x9E, 0x27, 0x51, 0xDC, 0xD5, 0x74, 0xE9, 0xBA, 0x90, 0xE7, 0x44,
     0xEA, 0x92, 0x58, 0x1F, 0xD0, 0xAF, 0x64, 0x0A, 0xE8, 0x6A, 0xC1, 0xCE,
     0x21, 0x98, 0xC9, 0x0F, 0x96, 0xB4, 0x48, 0x23},
    // C=KR, O=KISA, OU=Korea Certification Authority Central, CN=KISA RootCA 3
    {0x8F, 0xDD, 0x29, 0x8D, 0x1C, 0x93, 0xB2, 0x2B, 0xFC, 0x42, 0xAA, 0xB1,
     0xC3, 0xA1, 0x5F, 0x0D, 0x01, 0x83, 0x2C, 0xA0, 0xA1, 0xAE, 0xF2, 0x8D,
     0x56, 0x80, 0xF0, 0x6E, 0x6C, 0x7F, 0xD4, 0xEF},
    // C=BM, O=QuoVadis Limited, CN=QuoVadis Root CA 2 G3
    {0x8F, 0xE4, 0xFB, 0x0A, 0xF9, 0x3A, 0x4D, 0x0D, 0x67, 0xDB, 0x0B, 0xEB,
     0xB2, 0x3E, 0x37, 0xC7, 0x1B, 0xF3, 0x25, 0xDC, 0xBC, 0xDD, 0x24, 0x0E,
     0xA0, 0x4D, 0xAF, 0x58, 0xB4, 0x7E, 0x18, 0x40},
    // C=DE, O=D-Trust GmbH, CN=D-TRUST Root Class 3 CA 2007
    {0x90, 0xF3, 0xE0, 0x53, 0x96, 0x99, 0x5F, 0xF2, 0x09, 0x22, 0xC4, 0x45,
     0x92, 0xDB, 0x62, 0xD7, 0x84, 0x5E, 0x1B, 0xF6, 0x4A, 0xEF, 0x51, 0x2C,
     0xCA, 0x75, 0xBC, 0x66, 0x9C, 0xAA, 0x24, 0x79},
    // C=PA, ST=Panama, L=Panama City, O=TrustCor Systems S. de R.L.,
    // OU=TrustCor Certificate Authority, CN=TrustCor RootCert CA-2
    {0x91, 0x11, 0x24, 0x07, 0x47, 0xE1, 0xF6, 0x52, 0xF6, 0x6D, 0x1F, 0x71,
     0x2A, 0x11, 0xF6, 0x98, 0x96, 0x3B, 0x49, 0x17, 0x02, 0xE3, 0x12, 0xF7,
     0x51, 0x3D, 0xA3, 0xD0, 0xFC, 0x1E, 0x5A, 0x28},
    // C=DE, O=T-Systems Enterprise Services GmbH, OU=T-Systems Trust Center,
    // CN=T-TeleSec GlobalRoot Class 2
    {0x91, 0xE2, 0xF5, 0x78, 0x8D, 0x58, 0x10, 0xEB, 0xA7, 0xBA, 0x58, 0x73,
     0x7D, 0xE1, 0x54, 0x8A, 0x8E, 0xCA, 0xCD, 0x01, 0x45, 0x98, 0xBC, 0x0B,
     0x14, 0x3E, 0x04, 0x1B, 0x17, 0x05, 0x25, 0x52},
    // C=SK, L=Bratislava, O=Disig a.s., CN=CA Disig
    {0x92, 0xBF, 0x51, 0x19, 0xAB, 0xEC, 0xCA, 0xD0, 0xB1, 0x33, 0x2D, 0xC4,
     0xE1, 0xD0, 0x5F, 0xBA, 0x75, 0xB5, 0x67, 0x90, 0x44, 0xEE, 0x0C, 0xA2,
     0x6E, 0x93, 0x1F, 0x74, 0x4F, 0x2F, 0x33, 0xCF},
    // C=CN, O=UniTrust, CN=UCA Root
    {0x93, 0xE6, 0x5E, 0xC7, 0x62, 0xF0, 0x55, 0xDC, 0x71, 0x8A, 0x33, 0x25,
     0x82, 0xC4, 0x1A, 0x04, 0x43, 0x0D, 0x72, 0xE3, 0xCB, 0x87, 0xE8, 0xB8,
     0x97, 0xB6, 0x75, 0x16, 0xF0, 0xD1, 0xAA, 0x39},
    // C=KR, O=KISA, OU=ROOTCA, CN=CertRSA01
    {0x95, 0x60, 0x57, 0x51, 0x7F, 0xF3, 0xBB, 0x35, 0x04, 0x93, 0x42, 0x28,
     0x8C, 0x1C, 0x9D, 0xCE, 0x85, 0x2D, 0xAC, 0xA6, 0x52, 0xB4, 0x65, 0xE9,
     0x74, 0x72, 0x53, 0xB5, 0xF9, 0x3B, 0x1F, 0x5E},
    // O=Cybertrust, Inc, CN=Cybertrust Global Root
    {0x96, 0x0A, 0xDF, 0x00, 0x63, 0xE9, 0x63, 0x56, 0x75, 0x0C, 0x29, 0x65,
     0xDD, 0x0A, 0x08, 0x67, 0xDA, 0x0B, 0x9C, 0xBD, 0x6E, 0x77, 0x71, 0x4A,
     0xEA, 0xFB, 0x23, 0x49, 0xAB, 0x39, 0x3D, 0xA3},
    // CN=T\xC3\x9CRKTRUST Elektronik Sertifika Hizmet
    // Sa\xC4\x9Flay\xC4\xB1c\xC4\xB1s\xC4\xB1, C=TR, L=Ankara,
    // O=T\xC3\x9CRKTRUST Bilgi \xC4\xB0leti\xC5\x9Fim ve Bili\xC5\x9Fim
    // G\xC3\xBCvenli\xC4\x9Fi Hizmetleri A.\xC5\x9E. (c) Aral\xC4\xB1k 2007
    {0x97, 0x8C, 0xD9, 0x66, 0xF2, 0xFA, 0xA0, 0x7B, 0xA7, 0xAA, 0x95, 0x00,
     0xD9, 0xC0, 0x2E, 0x9D, 0x77, 0xF2, 0xCD, 0xAD, 0xA6, 0xAD, 0x6B, 0xA7,
     0x4A, 0xF4, 0xB9, 0x1C, 0x66, 0x59, 0x3C, 0x50},
    // C=DE, O=Deutsche Telekom AG, OU=T-TeleSec Trust Center, CN=Deutsche
    // Telekom Root CA 1
    {0x98, 0x06, 0xAB, 0x85, 0x09, 0xE2, 0xF3, 0x5E, 0x19, 0x2F, 0x27, 0x5F,
     0x0C, 0x30, 0x8B, 0x94, 0x09, 0xB4, 0x25, 0x12, 0xF9, 0x0C, 0x65, 0x95,
     0x98, 0xC2, 0x2B, 0xE6, 0x13, 0x96, 0x22, 0x72},
    // C=NO, O=Buypass AS-983163327, CN=Buypass Class 2 Root CA
    {0x9A, 0x11, 0x40, 0x25, 0x19, 0x7C, 0x5B, 0xB9, 0x5D, 0x94, 0xE6, 0x3D,
     0x55, 0xCD, 0x43, 0x79, 0x08, 0x47, 0xB6, 0x46, 0xB2, 0x3C, 0xDF, 0x11,
     0xAD, 0xA4, 0xA0, 0x0E, 0xFF, 0x15, 0xFB, 0x48},
    // CN=ACCVRAIZ1, OU=PKIACCV, O=ACCV, C=ES
    {0x9A, 0x6E, 0xC0, 0x12, 0xE1, 0xA7, 0xDA, 0x9D, 0xBE, 0x34, 0x19, 0x4D,
     0x47, 0x8A, 0xD7, 0xC0, 0xDB, 0x18, 0x22, 0xFB, 0x07, 0x1D, 0xF1, 0x29,
     0x81, 0x49, 0x6E, 0xD1, 0x04, 0x38, 0x41, 0x13},
    // C=US, O=VeriSign, Inc., OU=VeriSign Trust Network, OU=(c) 2006 VeriSign,
    // Inc. - For authorized use only, CN=VeriSign Class 3 Public Primary
    // Certification Authority - G5
    {0x9A, 0xCF, 0xAB, 0x7E, 0x43, 0xC8, 0xD8, 0x80, 0xD0, 0x6B, 0x26, 0x2A,
     0x94, 0xDE, 0xEE, 0xE4, 0xB4, 0x65, 0x99, 0x89, 0xC3, 0xD0, 0xCA, 0xF1,
     0x9B, 0xAF, 0x64, 0x05, 0xE4, 0x1A, 0xB7, 0xDF},
    // C=SE, O=Swedish Social Insurance Agency, CN=Swedish Government Root
    // Authority v2
    {0x9C, 0xEF, 0xB0, 0xCB, 0x7B, 0x74, 0xE6, 0x42, 0x93, 0x25, 0x32, 0x83,
     0x1E, 0x0D, 0xC8, 0xF4, 0xD6, 0x8A, 0xD4, 0x14, 0x26, 0x1F, 0xC3, 0xF4,
     0x74, 0xB7, 0x95, 0xE7, 0x2A, 0x16, 0x4E, 0x57},
    // C=EU, O=AC Camerfirma SA CIF A82743287, OU=http://www.chambersign.org,
    // CN=Public Notary Root
    {0x9D, 0xB9, 0x30, 0xA7, 0xBC, 0xED, 0x5A, 0x59, 0x9D, 0xA6, 0x73, 0xD0,
     0xBB, 0x12, 0xC4, 0xC6, 0xC7, 0xAB, 0x5B, 0x3F, 0x88, 0xF3, 0x9C, 0x24,
     0xEE, 0x20, 0xA2, 0x47, 0x16, 0xB3, 0x79, 0xDF},
    // C=US, O=Digital Signature Trust Co., OU=DST (ANX Network) CA
    {0x9D, 0xF0, 0xEC, 0x44, 0xF5, 0x5B, 0x36, 0xD7, 0x9D, 0x4B, 0x53, 0xC2,
     0x08, 0xBE, 0xF8, 0xCB, 0x63, 0xD7, 0x8D, 0xCC, 0x8F, 0xCA, 0xFD, 0xE1,
     0x66, 0x23, 0x12, 0xF2, 0x12, 0x20, 0x4A, 0x37},
    // C=PL, O=Unizeto Technologies S.A., OU=Certum Certification Authority,
    // CN=Certum Trusted Network CA 2
    {0x9F, 0x8B, 0x05, 0x13, 0x7F, 0x20, 0xAC, 0xDE, 0x9B, 0x99, 0x64, 0x10,
     0xF4, 0xD0, 0xBF, 0x79, 0x71, 0xA1, 0x00, 0x6D, 0xC9, 0x9E, 0x09, 0x4C,
     0x34, 0x6D, 0x27, 0x9B, 0x93, 0xCF, 0xF7, 0xAE},
    // C=US, O=GeoTrust Inc., CN=GeoTrust Universal CA 2
    {0xA0, 0x23, 0x4F, 0x3B, 0xC8, 0x52, 0x7C, 0xA5, 0x62, 0x8E, 0xEC, 0x81,
     0xAD, 0x5D, 0x69, 0x89, 0x5D, 0xA5, 0x68, 0x0D, 0xC9, 0x1D, 0x1C, 0xB8,
     0x47, 0x7F, 0x33, 0xF8, 0x78, 0xB9, 0x5B, 0x0B},
    // C=GR, L=Athens, O=Hellenic Academic and Research Institutions Cert.
    // Authority, CN=Hellenic Academic and Research Institutions RootCA 2015
    {0xA0, 0x40, 0x92, 0x9A, 0x02, 0xCE, 0x53, 0xB4, 0xAC, 0xF4, 0xF2, 0xFF,
     0xC6, 0x98, 0x1C, 0xE4, 0x49, 0x6F, 0x75, 0x5E, 0x6D, 0x45, 0xFE, 0x0B,
     0x2A, 0x69, 0x2B, 0xCD, 0x52, 0x52, 0x3F, 0x36},
    // C=US, O=GeoTrust Inc., CN=GeoTrust Universal CA
    {0xA0, 0x45, 0x9B, 0x9F, 0x63, 0xB2, 0x25, 0x59, 0xF5, 0xFA, 0x5D, 0x4C,
     0x6D, 0xB3, 0xF9, 0xF7, 0x2F, 0xF1, 0x93, 0x42, 0x03, 0x35, 0x78, 0xF0,
     0x73, 0xBF, 0x1D, 0x1B, 0x46, 0xCB, 0xB9, 0x12},
    // C=LU, O=LuxTrust s.a., CN=LuxTrust Global Root
    {0xA1, 0xB2, 0xDB, 0xEB, 0x64, 0xE7, 0x06, 0xC6, 0x16, 0x9E, 0x3C, 0x41,
     0x18, 0xB2, 0x3B, 0xAA, 0x09, 0x01, 0x8A, 0x84, 0x27, 0x66, 0x6D, 0x8B,
     0xF0, 0xE2, 0x88, 0x91, 0xEC, 0x05, 0x19, 0x50},
    // C=CN, O=UniTrust, CN=UCA Global Root
    {0xA1, 0xF0, 0x5C, 0xCB, 0x80, 0xC2, 0xD7, 0x10, 0xEC, 0x7D, 0x47, 0x9A,
     0xBD, 0xCB, 0xB8, 0x79, 0xE5, 0x8D, 0x7E, 0xDB, 0x71, 0x49, 0xFE, 0x78,
     0xA8, 0x78, 0x84, 0xE3, 0xD0, 0xBA, 0xD0, 0xF9},
    // C=JP, O=SECOM Trust Systems CO.,LTD., OU=Security Communication EV
    // RootCA1
    {0xA2, 0x2D, 0xBA, 0x68, 0x1E, 0x97, 0x37, 0x6E, 0x2D, 0x39, 0x7D, 0x72,
     0x8A, 0xAE, 0x3A, 0x9B, 0x62, 0x96, 0xB9, 0xFD, 0xBA, 0x60, 0xBC, 0x2E,
     0x11, 0xF6, 0x47, 0xF2, 0xC6, 0x75, 0xFB, 0x37},
    // C=ch, O=admin, OU=Services, OU=Certification Authorities,
    // CN=Admin-Root-CA
    {0xA3, 0x1F, 0x09, 0x30, 0x53, 0xBD, 0x12, 0xC1, 0xF5, 0xC3, 0xC6, 0xEF,
     0xD4, 0x98, 0x02, 0x3F, 0xD2, 0x91, 0x4D, 0x77, 0x58, 0xD0, 0x5D, 0x69,
     0x8C, 0xE0, 0x84, 0xB5, 0x06, 0x26, 0xE0, 0xE5},
    // C=CH, O=The Federal Authorities of the Swiss Confederation, OU=Services,
    // OU=Certification Authorities, CN=Swiss Government Root CA II
    {0xA3, 0xD7, 0x43, 0x5A, 0x18, 0xC4, 0x6B, 0x23, 0xB6, 0xA4, 0xF8, 0x92,
     0x9C, 0xD5, 0x90, 0x50, 0xC9, 0x16, 0x8B, 0x03, 0xA7, 0xFA, 0xD5, 0x32,
     0x62, 0x6F, 0x29, 0x7C, 0xAC, 0x53, 0x56, 0xE4},
    // C=US, O=thawte, Inc., OU=(c) 2007 thawte, Inc. - For authorized use only,
    // CN=thawte Primary Root CA - G2
    {0xA4, 0x31, 0x0D, 0x50, 0xAF, 0x18, 0xA6, 0x44, 0x71, 0x90, 0x37, 0x2A,
     0x86, 0xAF, 0xAF, 0x8B, 0x95, 0x1F, 0xFB, 0x43, 0x1D, 0x83, 0x7F, 0x1E,
     0x56, 0x88, 0xB4, 0x59, 0x71, 0xED, 0x15, 0x57},
    // C=BM, O=QuoVadis Limited, OU=Root Certification Authority, CN=QuoVadis
    // Root Certification Authority
    {0xA4, 0x5E, 0xDE, 0x3B, 0xBB, 0xF0, 0x9C, 0x8A, 0xE1, 0x5C, 0x72, 0xEF,
     0xC0, 0x72, 0x68, 0xD6, 0x93, 0xA2, 0x1C, 0x99, 0x6F, 0xD5, 0x1E, 0x67,
     0xCA, 0x07, 0x94, 0x60, 0xFD, 0x6D, 0x88, 0x73},
    // C=US, O=VeriSign, Inc., OU=Class 3 Public Primary Certification Authority
    {0xA4, 0xB6, 0xB3, 0x99, 0x6F, 0xC2, 0xF3, 0x06, 0xB3, 0xFD, 0x86, 0x81,
     0xBD, 0x63, 0x41, 0x3D, 0x8C, 0x50, 0x09, 0xCC, 0x4F, 0xA3, 0x29, 0xC2,
     0xCC, 0xF0, 0xE2, 0xFA, 0x1B, 0x14, 0x03, 0x05},
    // C=US, O=GTE Corporation, OU=GTE CyberTrust Solutions, Inc., CN=GTE
    // CyberTrust Global Root
    {0xA5, 0x31, 0x25, 0x18, 0x8D, 0x21, 0x10, 0xAA, 0x96, 0x4B, 0x02, 0xC7,
     0xB7, 0xC6, 0xDA, 0x32, 0x03, 0x17, 0x08, 0x94, 0xE5, 0xFB, 0x71, 0xFF,
     0xFB, 0x66, 0x67, 0xD5, 0xE6, 0x81, 0x0A, 0x36},
    // C=US, O=Wells Fargo WellsSecure, OU=Wells Fargo Bank NA, CN=WellsSecure
    // Public Root Certificate Authority
    {0xA7, 0x12, 0x72, 0xAE, 0xAA, 0xA3, 0xCF, 0xE8, 0x72, 0x7F, 0x7F, 0xB3,
     0x9F, 0x0F, 0xB3, 0xD1, 0xE5, 0x42, 0x6E, 0x90, 0x60, 0xB0, 0x6E, 0xE6,
     0xF1, 0x3E, 0x9A, 0x3C, 0x58, 0x33, 0xCD, 0x43},
    // C=GR, O=Athens Exchange S.A., CN=ATHEX Root CA
    {0xA7, 0x35, 0x63, 0xE8, 0x59, 0xCB, 0xCF, 0xA1, 0x73, 0xCF, 0x32, 0x85,
     0xDE, 0xBF, 0x25, 0x78, 0xED, 0xE1, 0x5D, 0x47, 0xA3, 0xBE, 0xE3, 0x85,
     0x86, 0x1A, 0xB7, 0xA4, 0xFB, 0x6D, 0x7B, 0x6E},
    // C=TN, O=ANCE, OU=ANCE WEB, CN=Agence Nationale de Certification
    // Electronique/emailAddress=ance@certification.tn
    {0xA7, 0x98, 0xA1, 0xC7, 0x0E, 0x9B, 0x6D, 0x50, 0xEA, 0xA5, 0x72, 0x4A,
     0x26, 0xFA, 0xC7, 0x99, 0x18, 0x48, 0xED, 0xC6, 0x1B, 0xF4, 0x8D, 0x79,
     0x81, 0x6B, 0xCA, 0xFB, 0x66, 0x97, 0x21, 0x28},
    // C=US, ST=DC, L=Washington, O=ABA.ECOM, INC., CN=ABA.ECOM Root
    // CA/emailAddress=admin@digsigtrust.com
    {0xA9, 0xA9, 0xCA, 0x22, 0x89, 0x0D, 0x69, 0x49, 0x19, 0xC0, 0x95, 0xBB,
     0x80, 0x64, 0xCB, 0xC4, 0xA5, 0x25, 0xA5, 0xDB, 0xFA, 0xF8, 0x01, 0xEC,
     0x93, 0x5D, 0xBC, 0xCC, 0x07, 0xC8, 0xBA, 0xEE},
    // C=my, O=TM, OU=TM Applied Business Certification Authority, CN=TM Applied
    // Business Root Certificate
    {0xA9, 0xC7, 0x7A, 0xF1, 0xBC, 0xDF, 0xAA, 0x37, 0x39, 0x44, 0x2B, 0x0B,
     0x27, 0x34, 0xC6, 0x8E, 0xAF, 0x2E, 0x98, 0x33, 0xF0, 0xD7, 0x66, 0xFB,
     0xCA, 0xA6, 0xF2, 0xAE, 0xB4, 0x2D, 0xEC, 0x02},
    // C=LV, O=VAS Latvijas Pasts - Vien.reg.Nr.40003052790, OU=Sertifikacijas
    // pakalpojumi, CN=VAS Latvijas Pasts SSI(RCA)
    {0xAA, 0xD9, 0xCE, 0xED, 0x5A, 0xA6, 0xB1, 0xCE, 0xA2, 0x85, 0x96, 0xA8,
     0xE4, 0xE1, 0xAB, 0xED, 0x93, 0x86, 0xD6, 0xEB, 0xC9, 0xD4, 0xAA, 0xD9,
     0xAC, 0xDE, 0x0F, 0xA3, 0x6B, 0xA0, 0x69, 0xD0},
    // C=ZA, ST=Western Cape, L=Cape Town, O=Thawte Consulting cc,
    // OU=Certification Services Division, CN=Thawte Premium Server
    // CA/emailAddress=premium-server@thawte.com
    {0xAB, 0x70, 0x36, 0x36, 0x5C, 0x71, 0x54, 0xAA, 0x29, 0xC2, 0xC2, 0x9F,
     0x5D, 0x41, 0x91, 0x16, 0x3B, 0x16, 0x2A, 0x22, 0x25, 0x01, 0x13, 0x57,
     0xD5, 0x6D, 0x07, 0xFF, 0xA7, 0xBC, 0x1F, 0x72},
    // C=CZ, O=\xC4\x8Cesk\xC3\xA1 po\xC5\xA1ta, s.p. [I\xC4\x8C 47114983],
    // CN=PostSignum Root QCA 2
    {0xAD, 0x01, 0x6F, 0x95, 0x80, 0x50, 0xE0, 0xE7, 0xE4, 0x6F, 0xAE, 0x7D,
     0xCC, 0x50, 0x19, 0x7E, 0xD8, 0xE3, 0xFF, 0x0A, 0x4B, 0x26, 0x2E, 0x5D,
     0xDC, 0xDB, 0x3E, 0xDD, 0xDC, 0x7D, 0x65, 0x78},
    // C=US, O=Wells Fargo WellsSecure, OU=Wells Fargo Bank NA, CN=WellsSecure
    // Public Root Certification Authority 01 G2
    {0xAD, 0x75, 0x39, 0xE5, 0xCD, 0xC9, 0x85, 0xFA, 0x95, 0x24, 0x40, 0x55,
     0xA9, 0x20, 0x2D, 0x63, 0x46, 0x0E, 0xC9, 0x21, 0x46, 0x7D, 0x03, 0x4C,
     0xFD, 0xBE, 0x87, 0xEC, 0x6D, 0x00, 0xFE, 0xDC},
    // C=AT, O=A-Trust, OU=A-Trust-nQual-01, CN=A-Trust-nQual-01
    {0xAE, 0x2F, 0xEC, 0x82, 0x91, 0x18, 0xA2, 0x45, 0x5A, 0xD6, 0xA4, 0x15,
     0xE7, 0x18, 0x23, 0xEB, 0x9B, 0x7B, 0x6E, 0x35, 0x78, 0xA5, 0x1A, 0xC8,
     0xA5, 0x14, 0x46, 0xEA, 0xDB, 0xB0, 0x97, 0x9C},
    // CN=ComSign CA, O=ComSign, C=IL
    {0xAE, 0x44, 0x57, 0xB4, 0x0D, 0x9E, 0xDA, 0x96, 0x67, 0x7B, 0x0D, 0x3C,
     0x92, 0xD5, 0x7B, 0x51, 0x77, 0xAB, 0xD7, 0xAC, 0x10, 0x37, 0x95, 0x83,
     0x56, 0xD1, 0xE0, 0x94, 0x51, 0x8B, 0xE5, 0xF2},
    // C=JP, O=Japan Certification Services, Inc., CN=SecureSign RootCA3
    {0xAE, 0x92, 0xE9, 0x00, 0x00, 0x54, 0x1A, 0x9E, 0xBC, 0x10, 0x1B, 0x70,
     0xB6, 0xC3, 0x3A, 0x62, 0xF5, 0xA5, 0x3A, 0x55, 0xBA, 0x81, 0x5E, 0x81,
     0xD3, 0x1A, 0xBD, 0xDF, 0x03, 0x50, 0x7F, 0x5D},
    // C=JP, O=Japan Certification Services, Inc., CN=SecureSign RootCA2
    {0xAF, 0x6D, 0x08, 0xEE, 0xF3, 0xCA, 0xC4, 0xE1, 0x58, 0x4A, 0xBC, 0x63,
     0xC8, 0xA9, 0x47, 0x2A, 0xC5, 0x29, 0xAF, 0x99, 0xF3, 0xF7, 0x91, 0x31,
     0x9A, 0x43, 0x77, 0x60, 0x63, 0xF5, 0x8D, 0xCA},
    // L=Bogota AV Calle 26 N 68D-35, C=CO, O=Entidad de Certificacion Digital
    // Abierta Certicamara S.A., CN=CERTICAMARA S.A.
    {0xAF, 0x71, 0xA3, 0xBC, 0xA3, 0x22, 0xE5, 0x22, 0x4D, 0xF5, 0x46, 0x89,
     0x56, 0x96, 0xCE, 0x44, 0x9A, 0x8B, 0xD2, 0xBD, 0x13, 0x0F, 0x7A, 0x7A,
     0xE4, 0x57, 0x76, 0x7F, 0x5C, 0x23, 0xD8, 0xF8},
    // O=RSA Security Inc, OU=RSA Security 2048 V3
    {0xAF, 0x8B, 0x67, 0x62, 0xA1, 0xE5, 0x28, 0x22, 0x81, 0x61, 0xA9, 0x5D,
     0x5C, 0x55, 0x9E, 0xE2, 0x66, 0x27, 0x8F, 0x75, 0xD7, 0x9E, 0x83, 0x01,
     0x89, 0xA5, 0x03, 0x50, 0x6A, 0xBD, 0x6B, 0x4C},
    // C=IT, L=Milano, O=Actalis S.p.A./03358520967, CN=Actalis Authentication
    // CA G1
    {0xAF, 0xE8, 0x0A, 0x97, 0xEA, 0x11, 0x15, 0x91, 0x90, 0xE9, 0x27, 0xE0,
     0x8E, 0xC2, 0xE6, 0x0C, 0x59, 0xA4, 0x05, 0x59, 0x48, 0x3A, 0x3F, 0xEE,
     0x83, 0x8A, 0x85, 0x32, 0x1B, 0x3A, 0xEA, 0xAD},
    // O=eSign Australia, OU=Public Secure Services, CN=Primary Utility Root CA
    {0xB0, 0x4D, 0x70, 0x8F, 0x1A, 0xE0, 0x45, 0x62, 0x65, 0xDD, 0x1B, 0x66,
     0x90, 0x7A, 0x26, 0x91, 0xA2, 0x86, 0x80, 0xB8, 0x53, 0xE0, 0x31, 0xDF,
     0x3D, 0xF9, 0x08, 0x3A, 0xF7, 0x16, 0x14, 0xD7},
    // C=ES, O=IZENPE S.A. - CIF A-01337260-RMerc.Vitoria-Gasteiz T1055 F62 S8,
    // L=Avda del Mediterraneo Etorbidea 3 - 01010 Vitoria-Gasteiz,
    // CN=Izenpe.com/emailAddress=Info@izenpe.com
    {0xB0, 0x87, 0x7A, 0xEE, 0x2D, 0x39, 0x27, 0x4D, 0xF8, 0x31, 0xF6, 0x6F,
     0xDE, 0xEB, 0x77, 0x17, 0x55, 0x7C, 0x25, 0x8F, 0xC9, 0xEB, 0x55, 0x23,
     0x1A, 0x9F, 0x8A, 0x64, 0x7A, 0x75, 0x43, 0x3F},
    // C=TR, L=Ankara, O=E-Tu\xC4\x9Fra EBG Bili\xC5\x9Fim Teknolojileri ve
    // Hizmetleri A.\xC5\x9E., OU=E-Tugra Sertifikasyon Merkezi, CN=E-Tugra
    // Certification Authority
    {0xB0, 0xBF, 0xD5, 0x2B, 0xB0, 0xD7, 0xD9, 0xBD, 0x92, 0xBF, 0x5D, 0x4D,
     0xC1, 0x3D, 0xA2, 0x55, 0xC0, 0x2C, 0x54, 0x2F, 0x37, 0x83, 0x65, 0xEA,
     0x89, 0x39, 0x11, 0xF5, 0x5E, 0x55, 0xF2, 0x3C},
    // C=FR, O=NATIXIS, OU=0002 542044524, CN=CESAM
    {0xB2, 0x25, 0x99, 0x96, 0xFF, 0xF7, 0x35, 0xAB, 0x35, 0x01, 0x4E, 0xF6,
     0x3F, 0x3D, 0x41, 0x31, 0x90, 0x07, 0x9D, 0xD0, 0x3A, 0x09, 0x62, 0x43,
     0x26, 0x35, 0xA8, 0x69, 0x5F, 0x99, 0x53, 0x05},
    // C=SE, O=Carelink, CN=SITHS CA v3
    {0xB2, 0xF3, 0xC4, 0x21, 0x6A, 0xF7, 0xAF, 0xF7, 0x24, 0x62, 0x46, 0x6D,
     0xC1, 0x3C, 0xD2, 0x81, 0x0D, 0xB8, 0xEE, 0xD8, 0x53, 0xEA, 0xBB, 0x9A,
     0x06, 0x3A, 0x60, 0x8E, 0xFC, 0x18, 0xFB, 0xE8},
    // C=US, O=Symantec Corporation, OU=Symantec Trust Network, CN=Symantec
    // Class 3 Public Primary Certification Authority - G6
    {0xB3, 0x23, 0x96, 0x74, 0x64, 0x53, 0x44, 0x2F, 0x35, 0x3E, 0x61, 0x62,
     0x92, 0xBB, 0x20, 0xBB, 0xAA, 0x5D, 0x23, 0xB5, 0x46, 0x45, 0x0F, 0xDB,
     0x9C, 0x54, 0xB8, 0x38, 0x61, 0x67, 0xD5, 0x29},
    // C=au, O=SecureNet CA Class B
    {0xB3, 0xC9, 0x62, 0xD3, 0x40, 0x19, 0xFB, 0x38, 0xAB, 0x9F, 0xE9, 0xC6,
     0x23, 0x99, 0x74, 0x2A, 0xB2, 0x6C, 0x43, 0xC2, 0xD1, 0x8C, 0xE3, 0xF2,
     0xB1, 0x3C, 0x14, 0x32, 0x1E, 0x52, 0x96, 0x4B},
    // x500UniqueIdentifier=SEC-830101-9V9, L=Alvaro Obregon, ST=Distrito
    // Federal, C=MX/postalCode=01030/street=Insurgentes Sur 1940, CN=Autoridad
    // Certificadora Raiz de la Secretaria de Economia, OU=Direccion General de
    // Normatividad Mercantil, O=Secretaria de
    // Economia/emailAddress=acrse@economia.gob.mx
    {0xB4, 0x1D, 0x51, 0x6A, 0x53, 0x51, 0xD4, 0x2D, 0xEE, 0xA1, 0x91, 0xFA,
     0x6E, 0xDF, 0x2A, 0x67, 0xDE, 0xE2, 0xF3, 0x6D, 0xC9, 0x69, 0x01, 0x2C,
     0x76, 0x66, 0x9E, 0x61, 0x6B, 0x90, 0x0D, 0xDF},
    // C=ZA, ST=Western Cape, L=Cape Town, O=Thawte Consulting cc,
    // OU=Certification Services Division, CN=Thawte Server
    // CA/emailAddress=server-certs@thawte.com
    {0xB4, 0x41, 0x0B, 0x73, 0xE2, 0xE6, 0xEA, 0xCA, 0x47, 0xFB, 0xC4, 0x2F,
     0x8F, 0xA4, 0x01, 0x8A, 0xF4, 0x38, 0x1D, 0xC5, 0x4C, 0xFA, 0xA8, 0x44,
     0x50, 0x46, 0x1E, 0xED, 0x09, 0x45, 0x4D, 0xE9},
    // C=US, O=GeoTrust Inc., OU=(c) 2008 GeoTrust Inc. - For authorized use
    // only, CN=GeoTrust Primary Certification Authority - G3
    {0xB4, 0x78, 0xB8, 0x12, 0x25, 0x0D, 0xF8, 0x78, 0x63, 0x5C, 0x2A, 0xA7,
     0xEC, 0x7D, 0x15, 0x5E, 0xAA, 0x62, 0x5E, 0xE8, 0x29, 0x16, 0xE2, 0xCD,
     0x29, 0x43, 0x61, 0x88, 0x6C, 0xD1, 0xFB, 0xD4},
    // C=US, ST=Arizona, L=Scottsdale, O=Starfield Technologies, Inc.,
    // OU=http://certificates.starfieldtech.com/repository/, CN=Starfield
    // Services Root Certificate Authority
    {0xB5, 0xBD, 0x2C, 0xB7, 0x9C, 0xBD, 0x19, 0x07, 0x29, 0x8D, 0x6B, 0xDF,
     0x48, 0x42, 0xE5, 0x16, 0xD8, 0xC7, 0x8F, 0xA6, 0xFC, 0x96, 0xD2, 0x5F,
     0x71, 0xAF, 0x81, 0x4E, 0x16, 0xCC, 0x24, 0x5E},
    // C=DE, O=Deutsche Telekom AG, OU=T-TeleSec Trust Center, CN=Deutsche
    // Telekom Root CA 2
    {0xB6, 0x19, 0x1A, 0x50, 0xD0, 0xC3, 0x97, 0x7F, 0x7D, 0xA9, 0x9B, 0xCD,
     0xAA, 0xC8, 0x6A, 0x22, 0x7D, 0xAE, 0xB9, 0x67, 0x9E, 0xC7, 0x0B, 0xA3,
     0xB0, 0xC9, 0xD9, 0x22, 0x71, 0xC1, 0x70, 0xD3},
    // C=ES, ST=Barcelona, L=Barcelona (see current address at
    // https://www.anf.es/address/), O=ANF Autoridad de Certificaci\xC3\xB3n,
    // OU=ANF Clase 1 CA/serialNumber=G63287510, CN=ANF Server CA
    {0xB6, 0x44, 0xD9, 0x55, 0xFF, 0xF2, 0x9B, 0x74, 0xE3, 0xB5, 0x68, 0x7E,
     0x90, 0x8E, 0xE7, 0xC3, 0xC9, 0x19, 0x7B, 0xA3, 0x33, 0x6C, 0xC6, 0x32,
     0x85, 0x31, 0xF6, 0xC0, 0x57, 0xD6, 0x77, 0xFD},
    // C=NO, O=Buypass AS-983163327, CN=Buypass Class 3 CA 1
    {0xB7, 0xB1, 0x2B, 0x17, 0x1F, 0x82, 0x1D, 0xAA, 0x99, 0x0C, 0xD0, 0xFE,
     0x50, 0x87, 0xB1, 0x28, 0x44, 0x8B, 0xA8, 0xE5, 0x18, 0x4F, 0x84, 0xC5,
     0x1E, 0x02, 0xB5, 0xC8, 0xFB, 0x96, 0x2B, 0x24},
    // C=FR, O=OpenTrust, CN=OpenTrust Root CA G3
    {0xB7, 0xC3, 0x62, 0x31, 0x70, 0x6E, 0x81, 0x07, 0x8C, 0x36, 0x7C, 0xB8,
     0x96, 0x19, 0x8F, 0x1E, 0x32, 0x08, 0xDD, 0x92, 0x69, 0x49, 0xDD, 0x8F,
     0x57, 0x09, 0xA4, 0x10, 0xF7, 0x5B, 0x62, 0x92},
    // C=MO, O=Macao Post, CN=Macao Post eSignTrust Root Certification Authority
    {0xB8, 0xBB, 0xE5, 0x23, 0xBF, 0xCA, 0x3B, 0x11, 0xD5, 0x0F, 0x73, 0xF7,
     0xF1, 0x0B, 0x3E, 0xC8, 0xEC, 0x95, 0x8A, 0xA1, 0xDC, 0x86, 0xF6, 0x6D,
     0x95, 0x41, 0x90, 0x7F, 0xF1, 0xA1, 0x10, 0xEF},
    // C=FR, ST=France, L=Paris, O=PM/SGDN, OU=DCSSI,
    // CN=IGC/A/emailAddress=igca@sgdn.pm.gouv.fr
    {0xB9, 0xBE, 0xA7, 0x86, 0x0A, 0x96, 0x2E, 0xA3, 0x61, 0x1D, 0xAB, 0x97,
     0xAB, 0x6D, 0xA3, 0xE2, 0x1C, 0x10, 0x68, 0xB9, 0x7D, 0x55, 0x57, 0x5E,
     0xD0, 0xE1, 0x12, 0x79, 0xC1, 0x1C, 0x89, 0x32},
    // C=GR, O=Hellenic Academic and Research Institutions Cert. Authority,
    // CN=Hellenic Academic and Research Institutions RootCA 2011
    {0xBC, 0x10, 0x4F, 0x15, 0xA4, 0x8B, 0xE7, 0x09, 0xDC, 0xA5, 0x42, 0xA7,
     0xE1, 0xD4, 0xB9, 0xDF, 0x6F, 0x05, 0x45, 0x27, 0xE8, 0x02, 0xEA, 0xA9,
     0x2D, 0x59, 0x54, 0x44, 0x25, 0x8A, 0xFE, 0x71},
    // L=ValiCert Validation Network, O=ValiCert, Inc., OU=ValiCert Class 3
    // Policy Validation Authority,
    // CN=http://www.valicert.com//emailAddress=info@valicert.com
    {0xBC, 0x23, 0xF9, 0x8A, 0x31, 0x3C, 0xB9, 0x2D, 0xE3, 0xBB, 0xFC, 0x3A,
     0x5A, 0x9F, 0x44, 0x61, 0xAC, 0x39, 0x49, 0x4C, 0x4A, 0xE1, 0x5A, 0x9E,
     0x9D, 0xF1, 0x31, 0xE9, 0x9B, 0x73, 0x01, 0x9A},
    // C=HK, O=Hongkong Post, CN=Hongkong Post Root CA
    {0xBC, 0xDD, 0x8D, 0xF4, 0x27, 0x63, 0x66, 0xD7, 0xFF, 0x4B, 0x68, 0x8D,
     0xC8, 0x15, 0x00, 0xD8, 0xE9, 0x82, 0x52, 0xC0, 0x49, 0xC8, 0xFF, 0x1E,
     0x8C, 0x82, 0xF2, 0xBA, 0xEC, 0x9D, 0x5C, 0x16},
    // C=US, O=AffirmTrust, CN=AffirmTrust Premium ECC
    {0xBD, 0x71, 0xFD, 0xF6, 0xDA, 0x97, 0xE4, 0xCF, 0x62, 0xD1, 0x64, 0x7A,
     0xDD, 0x25, 0x81, 0xB0, 0x7D, 0x79, 0xAD, 0xF8, 0x39, 0x7E, 0xB4, 0xEC,
     0xBA, 0x9C, 0x5E, 0x84, 0x88, 0x82, 0x14, 0x23},
    // C=CH, O=SwissSign AG, CN=SwissSign Silver CA - G2
    {0xBE, 0x6C, 0x4D, 0xA2, 0xBB, 0xB9, 0xBA, 0x59, 0xB6, 0xF3, 0x93, 0x97,
     0x68, 0x37, 0x42, 0x46, 0xC3, 0xC0, 0x05, 0x99, 0x3F, 0xA9, 0x8F, 0x02,
     0x0D, 0x1D, 0xED, 0xBE, 0xD4, 0x8A, 0x81, 0xD5},
    // OU=GlobalSign ECC Root CA - R4, O=GlobalSign, CN=GlobalSign
    {0xBE, 0xC9, 0x49, 0x11, 0xC2, 0x95, 0x56, 0x76, 0xDB, 0x6C, 0x0A, 0x55,
     0x09, 0x86, 0xD7, 0x6E, 0x3B, 0xA0, 0x05, 0x66, 0x7C, 0x44, 0x2C, 0x97,
     0x62, 0xB4, 0xFB, 0xB7, 0x73, 0xDE, 0x22, 0x8C},
    // C=JP, O=Japan Certification Services, Inc., CN=SecureSign RootCA11
    {0xBF, 0x0F, 0xEE, 0xFB, 0x9E, 0x3A, 0x58, 0x1A, 0xD5, 0xF9, 0xE9, 0xDB,
     0x75, 0x89, 0x98, 0x57, 0x43, 0xD2, 0x61, 0x08, 0x5C, 0x4D, 0x31, 0x4F,
     0x6F, 0x5D, 0x72, 0x59, 0xAA, 0x42, 0x16, 0x12},
    // C=TW, O=TAIWAN-CA, OU=Root CA, CN=TWCA Root Certification Authority
    {0xBF, 0xD8, 0x8F, 0xE1, 0x10, 0x1C, 0x41, 0xAE, 0x3E, 0x80, 0x1B, 0xF8,
     0xBE, 0x56, 0x35, 0x0E, 0xE9, 0xBA, 0xD1, 0xA6, 0xB9, 0xBD, 0x51, 0x5E,
     0xDC, 0x5C, 0x6D, 0x5B, 0x87, 0x11, 0xAC, 0x44},
    // C=CN, O=GUANG DONG CERTIFICATE AUTHORITY CO.,LTD., CN=GDCA TrustAUTH R5
    // ROOT
    {0xBF, 0xFF, 0x8F, 0xD0, 0x44, 0x33, 0x48, 0x7D, 0x6A, 0x8A, 0xA6, 0x0C,
     0x1A, 0x29, 0x76, 0x7A, 0x9F, 0xC2, 0xBB, 0xB0, 0x5E, 0x42, 0x0F, 0x71,
     0x3A, 0x13, 0xB9, 0x92, 0x89, 0x1D, 0x38, 0x93},
    // C=TW, O=Chunghwa Telecom Co., Ltd., OU=ePKI Root Certification Authority
    {0xC0, 0xA6, 0xF4, 0xDC, 0x63, 0xA2, 0x4B, 0xFD, 0xCF, 0x54, 0xEF, 0x2A,
     0x6A, 0x08, 0x2A, 0x0A, 0x72, 0xDE, 0x35, 0x80, 0x3E, 0x2F, 0xF5, 0xFF,
     0x52, 0x7A, 0xE5, 0xD8, 0x72, 0x06, 0xDF, 0xD5},
    // C=CL, ST=Region Metropolitana, L=Santiago, O=E-CERTCHILE, OU=Autoridad
    // Certificadora/emailAddress=sclientes@ccs.cl, CN=E-CERT ROOT CA
    {0xC1, 0x09, 0xE0, 0xD1, 0xB6, 0x40, 0x00, 0xB0, 0x1E, 0x89, 0x4B, 0xDA,
     0xA9, 0x55, 0xE1, 0xFF, 0x91, 0xB6, 0xD0, 0x84, 0xA8, 0x38, 0xD9, 0x0E,
     0x04, 0x4B, 0x9E, 0x3F, 0xCD, 0x2A, 0x8B, 0xFA},
    // C=FR, O=Certplus, CN=Class 3 Primary CA
    {0xC1, 0xB1, 0x2F, 0x48, 0x00, 0x20, 0x33, 0x6E, 0x5B, 0x04, 0xF5, 0x20,
     0xBC, 0x19, 0xC2, 0xE2, 0xE1, 0x0A, 0xB4, 0x2C, 0x9D, 0x92, 0x35, 0xF0,
     0x5C, 0xBE, 0xC3, 0x3F, 0xFA, 0x4D, 0x4D, 0xEA},
    // C=GB, O=Trustis Limited, OU=Trustis FPS Root CA
    {0xC1, 0xB4, 0x82, 0x99, 0xAB, 0xA5, 0x20, 0x8F, 0xE9, 0x63, 0x0A, 0xCE,
     0x55, 0xCA, 0x68, 0xA0, 0x3E, 0xDA, 0x5A, 0x51, 0x9C, 0x88, 0x02, 0xA0,
     0xD3, 0xA6, 0x73, 0xBE, 0x8F, 0x8E, 0x55, 0x7D},
    // C=ES, L=C/ Muntaner 244 Barcelona, CN=Autoridad de Certificacion
    // Firmaprofesional CIF A62634068/emailAddress=ca@firmaprofesional.com
    {0xC1, 0xCF, 0x0B, 0x52, 0x09, 0x64, 0x35, 0xE3, 0xF1, 0xB7, 0x1D, 0xAA,
     0xEC, 0x45, 0x5A, 0x23, 0x11, 0xC8, 0x40, 0x4F, 0x55, 0x83, 0xA9, 0xE2,
     0x13, 0xC6, 0x9D, 0x85, 0x7D, 0x94, 0x33, 0x05},
    // C=HU, L=Budapest, O=NISZ Nemzeti Infokommunik\xC3\xA1ci\xC3\xB3s
    // Szolg\xC3\xA1ltat\xC3\xB3 Zrt.,
    // CN=F\xC5\x91tan\xC3\xBAs\xC3\xADtv\xC3\xA1nykiad\xC3\xB3 -
    // Korm\xC3\xA1nyzati Hiteles\xC3\xADt\xC3\xA9s Szolg\xC3\xA1ltat\xC3\xB3
    {0xC2, 0x15, 0x73, 0x09, 0xD9, 0xAE, 0xE1, 0x7B, 0xF3, 0x4F, 0x4D, 0xF5,
     0xE8, 0x8D, 0xBA, 0xEB, 0xA5, 0x7E, 0x03, 0x61, 0xEB, 0x81, 0x4C, 0xBC,
     0x23, 0x9F, 0x4D, 0x54, 0xD3, 0x29, 0xA3, 0x8D},
    // C=CO, L=Carrera 9 16-21 Bogota, O=Certicamara S.A. Entidad de
    // Certificacion, CN=Certificado Empresarial Clase-A
    {0xC2, 0x95, 0x9D, 0xB8, 0x33, 0x9E, 0x8D, 0xBC, 0xF6, 0x40, 0x9C, 0xA9,
     0x2A, 0x66, 0xC4, 0x9F, 0xD2, 0xE3, 0x24, 0x94, 0x94, 0x0A, 0x90, 0x11,
     0x43, 0xBD, 0x7E, 0xB7, 0x28, 0x27, 0xDE, 0xC2},
    // C=LT, O=Skaitmeninio sertifikavimo centras, OU=CA ROOT Services, CN=SSC
    // GDL CA Root B
    {0xC3, 0x1E, 0xEF, 0x56, 0x82, 0xAB, 0xB5, 0x51, 0xEB, 0xC8, 0x28, 0xDE,
     0xD8, 0x40, 0x98, 0x51, 0x8A, 0x67, 0x68, 0x52, 0x6D, 0x15, 0x2E, 0xE1,
     0x64, 0xCF, 0xB9, 0x72, 0xA1, 0x42, 0x5D, 0x53},
    // C=IN, O=India PKI, CN=CCA India 2015 SPL
    {0xC3, 0x4C, 0x5D, 0xF5, 0x30, 0x80, 0x07, 0x8F, 0xFE, 0x45, 0xB2, 0x1A,
     0x7F, 0x60, 0x04, 0x69, 0x91, 0x72, 0x04, 0xF4, 0xF0, 0x29, 0x3F, 0x1D,
     0x72, 0x09, 0x39, 0x3E, 0x52, 0x65, 0xC0, 0x4F},
    // C=US, O=The Go Daddy Group, Inc., OU=Go Daddy Class 2 Certification
    // Authority
    {0xC3, 0x84, 0x6B, 0xF2, 0x4B, 0x9E, 0x93, 0xCA, 0x64, 0x27, 0x4C, 0x0E,
     0xC6, 0x7C, 0x1E, 0xCC, 0x5E, 0x02, 0x4F, 0xFC, 0xAC, 0xD2, 0xD7, 0x40,
     0x19, 0x35, 0x0E, 0x81, 0xFE, 0x54, 0x6A, 0xE4},
    // C=US, ST=UT, L=Salt Lake City, O=The USERTRUST Network,
    // OU=http://www.usertrust.com, CN=UTN-USERFirst-Network Applications
    {0xC3, 0x8D, 0xCB, 0x38, 0x95, 0x93, 0x93, 0x35, 0x86, 0x91, 0xEA, 0x4D,
     0x4F, 0x3C, 0xE4, 0x95, 0xCE, 0x74, 0x89, 0x96, 0xE6, 0x4E, 0xD1, 0x89,
     0x1D, 0x89, 0x7A, 0x0F, 0xC4, 0xDD, 0x55, 0xC6},
    // CN=T\xC3\x9CRKTRUST Elektronik Sertifika Hizmet
    // Sa\xC4\x9Flay\xC4\xB1c\xC4\xB1s\xC4\xB1, C=TR, L=Ankara,
    // O=T\xC3\x9CRKTRUST Bilgi \xC4\xB0leti\xC5\x9Fim ve Bili\xC5\x9Fim
    // G\xC3\xBCvenli\xC4\x9Fi Hizmetleri A.\xC5\x9E. (c) Kas\xC4\xB1m 2005
    {0xC4, 0x70, 0xCF, 0x54, 0x7E, 0x23, 0x02, 0xB9, 0x77, 0xFB, 0x29, 0xDD,
     0x71, 0xA8, 0x9A, 0x7B, 0x6C, 0x1F, 0x60, 0x77, 0x7B, 0x03, 0x29, 0xF5,
     0x60, 0x17, 0xF3, 0x28, 0xBF, 0x4F, 0x6B, 0xE6},
    // CN=T\xC3\x9CRKTRUST Elektronik \xC4\xB0\xC5\x9Flem Hizmetleri, C=TR,
    // L=ANKARA, O=(c) 2005 T\xC3\x9CRKTRUST Bilgi \xC4\xB0leti\xC5\x9Fim ve
    // Bili\xC5\x9Fim G\xC3\xBCvenli\xC4\x9Fi Hizmetleri A.\xC5\x9E.
    {0xC4, 0x99, 0xF6, 0xCE, 0xCC, 0x5D, 0xA4, 0xD6, 0x1F, 0x14, 0xED, 0x04,
     0x05, 0x27, 0x0C, 0x52, 0x49, 0xD0, 0xE7, 0x96, 0x15, 0xB0, 0xDA, 0x42,
     0x65, 0x9E, 0xD2, 0xD7, 0xFF, 0xEF, 0x8A, 0x40},
    // C=US, O=VISA, OU=Visa International Service Association, CN=Visa
    // Information Delivery Root CA
    {0xC5, 0x7A, 0x3A, 0xCB, 0xE8, 0xC0, 0x6B, 0xA1, 0x98, 0x8A, 0x83, 0x48,
     0x5B, 0xF3, 0x26, 0xF2, 0x44, 0x87, 0x75, 0x37, 0x98, 0x49, 0xDE, 0x01,
     0xCA, 0x43, 0x57, 0x1A, 0xF3, 0x57, 0xE7, 0x4B},
    // C=IL, O=StartCom Ltd., OU=Secure Digital Certificate Signing, CN=StartCom
    // Certification Authority
    {0xC7, 0x66, 0xA9, 0xBE, 0xF2, 0xD4, 0x07, 0x1C, 0x86, 0x3A, 0x31, 0xAA,
     0x49, 0x20, 0xE8, 0x13, 0xB2, 0xD1, 0x98, 0x60, 0x8C, 0xB7, 0xB7, 0xCF,
     0xE2, 0x11, 0x43, 0xB8, 0x36, 0xDF, 0x09, 0xEA},
    // C=TN, CN=Tunisian Root Certificate Authority - TunRootCA2, O=National
    // Digital Certification Agency
    {0xC7, 0x95, 0xFF, 0x8F, 0xF2, 0x0C, 0x96, 0x66, 0x88, 0xF0, 0x64, 0xA1,
     0xE0, 0x91, 0x42, 0x1D, 0x31, 0x10, 0xA3, 0x45, 0x6C, 0x17, 0xEC, 0x24,
     0x04, 0xB9, 0x98, 0x73, 0x87, 0x41, 0xF6, 0x22},
    // C=IL, O=StartCom Ltd., CN=StartCom Certification Authority G2
    {0xC7, 0xBA, 0x65, 0x67, 0xDE, 0x93, 0xA7, 0x98, 0xAE, 0x1F, 0xAA, 0x79,
     0x1E, 0x71, 0x2D, 0x37, 0x8F, 0xAE, 0x1F, 0x93, 0xC4, 0x39, 0x7F, 0xEA,
     0x44, 0x1B, 0xB7, 0xCB, 0xE6, 0xFD, 0x59, 0x95},
    // C=US, O=GeoTrust Inc., CN=GeoTrust Global CA 2
    {0xCA, 0x2D, 0x82, 0xA0, 0x86, 0x77, 0x07, 0x2F, 0x8A, 0xB6, 0x76, 0x4F,
     0xF0, 0x35, 0x67, 0x6C, 0xFE, 0x3E, 0x5E, 0x32, 0x5E, 0x01, 0x21, 0x72,
     0xDF, 0x3F, 0x92, 0x09, 0x6D, 0xB7, 0x9B, 0x85},
    // OU=GlobalSign Root CA - R2, O=GlobalSign, CN=GlobalSign
    {0xCA, 0x42, 0xDD, 0x41, 0x74, 0x5F, 0xD0, 0xB8, 0x1E, 0xB9, 0x02, 0x36,
     0x2C, 0xF9, 0xD8, 0xBF, 0x71, 0x9D, 0xA1, 0xBD, 0x1B, 0x1E, 0xFC, 0x94,
     0x6F, 0x5B, 0x4C, 0x99, 0xF4, 0x2C, 0x1B, 0x9E},
    // CN=Autoridad de Certificacion Raiz del Estado Venezolano, C=VE,
    // L=Caracas, ST=Distrito Capital, O=Sistema Nacional de Certificacion
    // Electronica, OU=Superintendencia de Servicios de Certificacion
    // Electronica/emailAddress=acraiz@suscerte.gob.ve
    {0xCA, 0x7A, 0x5E, 0x68, 0xC5, 0x3D, 0x2C, 0x51, 0xF7, 0x2F, 0x6B, 0x46,
     0x5D, 0x3E, 0xD7, 0x53, 0xF5, 0x90, 0x3E, 0xC7, 0x90, 0x1C, 0x8D, 0x0F,
     0x55, 0xD8, 0x68, 0x33, 0x7C, 0x81, 0x97, 0x5A},
    // C=US, O=DigiCert Inc, OU=www.digicert.com, CN=DigiCert Global Root G2
    {0xCB, 0x3C, 0xCB, 0xB7, 0x60, 0x31, 0xE5, 0xE0, 0x13, 0x8F, 0x8D, 0xD3,
     0x9A, 0x23, 0xF9, 0xDE, 0x47, 0xFF, 0xC3, 0x5E, 0x43, 0xC1, 0x14, 0x4C,
     0xEA, 0x27, 0xD4, 0x6A, 0x5A, 0xB1, 0xCB, 0x5F},
    // OU=GlobalSign Root CA - R3, O=GlobalSign, CN=GlobalSign
    {0xCB, 0xB5, 0x22, 0xD7, 0xB7, 0xF1, 0x27, 0xAD, 0x6A, 0x01, 0x13, 0x86,
     0x5B, 0xDF, 0x1C, 0xD4, 0x10, 0x2E, 0x7D, 0x07, 0x59, 0xAF, 0x63, 0x5A,
     0x7C, 0xF4, 0x72, 0x0D, 0xC9, 0x63, 0xC5, 0x3B},
    // C=BR, O=ICP-Brasil, OU=Instituto Nacional de Tecnologia da Informacao -
    // ITI, CN=Autoridade Certificadora Raiz Brasileira v1
    {0xCB, 0xD8, 0xED, 0x38, 0xD4, 0xA2, 0xD6, 0x77, 0xD4, 0x53, 0xD7, 0x0D,
     0xD8, 0x89, 0x0A, 0xF4, 0xF6, 0x37, 0x4C, 0xBA, 0x62, 0x99, 0x94, 0x3F,
     0x1A, 0xB3, 0xA6, 0x93, 0x6C, 0x6F, 0xD7, 0x95},
    // C=FR, O=Certplus, CN=Class 3P Primary CA
    {0xCC, 0xC8, 0x94, 0x89, 0x37, 0x1B, 0xAD, 0x11, 0x1C, 0x90, 0x61, 0x9B,
     0xEA, 0x24, 0x0A, 0x2E, 0x6D, 0xAD, 0xD9, 0x9F, 0x9F, 0x6E, 0x1D, 0x4D,
     0x41, 0xE5, 0x8E, 0xD6, 0xDE, 0x3D, 0x02, 0x85},
    // C=LV, OU=Sertifikacijas pakalpojumu dala, CN=E-ME SSI (RCA)
    {0xCD, 0x0B, 0x3B, 0x2A, 0xA1, 0x74, 0xB5, 0x5F, 0x18, 0xC7, 0x50, 0x2F,
     0x3C, 0x3A, 0x76, 0xF2, 0x19, 0x81, 0x75, 0xCE, 0x45, 0x63, 0x73, 0x70,
     0xCF, 0x4F, 0x48, 0xB9, 0xC2, 0xCE, 0x4F, 0xBF},
    // DC=rs, DC=posta, DC=ca, CN=Configuration, CN=Services, CN=Public Key
    // Services, CN=AIA, CN=Posta CA Root
    {0xCD, 0x20, 0x12, 0x56, 0xFE, 0x5C, 0xED, 0x0B, 0xFF, 0xF8, 0xDF, 0x59,
     0x5F, 0xFF, 0x36, 0xB1, 0x41, 0x6D, 0x53, 0x13, 0xA9, 0x99, 0xF5, 0x32,
     0xEF, 0x4A, 0x99, 0x15, 0xDF, 0x96, 0xDE, 0xE0},
    // C=FR, O=Certplus, CN=Class 3TS Primary CA
    {0xCE, 0x7D, 0xD0, 0x96, 0xC8, 0xFD, 0xE2, 0xBF, 0x5C, 0x43, 0x8E, 0xDB,
     0x57, 0x4B, 0xD6, 0x45, 0x43, 0x85, 0x33, 0x4E, 0xE8, 0xFF, 0x10, 0x6C,
     0x0F, 0x93, 0xD5, 0x05, 0x1B, 0xE6, 0xBA, 0xC3},
    // C=US, OU=www.xrampsecurity.com, O=XRamp Security Services Inc, CN=XRamp
    // Global Certification Authority
    {0xCE, 0xCD, 0xDC, 0x90, 0x50, 0x99, 0xD8, 0xDA, 0xDF, 0xC5, 0xB1, 0xD2,
     0x09, 0xB7, 0x37, 0xCB, 0xE2, 0xC1, 0x8C, 0xFB, 0x2C, 0x10, 0xC0, 0xFF,
     0x0B, 0xCF, 0x0D, 0x32, 0x86, 0xFC, 0x1A, 0xA2},
    // C=US, O=Equifax Secure Inc., CN=Equifax Secure eBusiness CA-1
    {0xCF, 0x56, 0xFF, 0x46, 0xA4, 0xA1, 0x86, 0x10, 0x9D, 0xD9, 0x65, 0x84,
     0xB5, 0xEE, 0xB5, 0x8A, 0x51, 0x0C, 0x42, 0x75, 0xB0, 0xE5, 0xF9, 0x4F,
     0x40, 0xBB, 0xAE, 0x86, 0x5E, 0x19, 0xF6, 0x73},
    // O=Entrust.net, OU=www.entrust.net/CPS_2048 incorp. by ref. (limits
    // liab.), OU=(c) 1999 Entrust.net Limited, CN=Entrust.net Certification
    // Authority (2048)
    {0xD1, 0xC3, 0x39, 0xEA, 0x27, 0x84, 0xEB, 0x87, 0x0F, 0x93, 0x4F, 0xC5,
     0x63, 0x4E, 0x4A, 0xA9, 0xAD, 0x55, 0x05, 0x01, 0x64, 0x01, 0xF2, 0x64,
     0x65, 0xD3, 0x7A, 0x57, 0x46, 0x63, 0x35, 0x9F},
    // C=CZ, O=Prvn\xC3\xAD certifika\xC4\x8Dn\xC3\xAD autorita, a.s., CN=I.CA
    // Root CA/RSA/serialNumber=NTRCZ-26439395
    {0xD3, 0xD6, 0x07, 0xA9, 0xFF, 0x24, 0xA1, 0x95, 0x23, 0xB6, 0xDA, 0x9D,
     0x2C, 0x64, 0x94, 0x46, 0xF8, 0x78, 0x8C, 0xB9, 0x6D, 0x9F, 0xD1, 0x30,
     0x97, 0x2E, 0x12, 0x0C, 0x13, 0x67, 0x77, 0x30},
    // C=NL, O=Staat der Nederlanden, CN=Staat der Nederlanden Root CA
    {0xD4, 0x1D, 0x82, 0x9E, 0x8C, 0x16, 0x59, 0x82, 0x2A, 0xF9, 0x3F, 0xCE,
     0x62, 0xBF, 0xFC, 0xDE, 0x26, 0x4F, 0xC8, 0x4E, 0x8B, 0x95, 0x0C, 0x5F,
     0xF2, 0x75, 0xD0, 0x52, 0x35, 0x46, 0x95, 0xA3},
    // C=CN, O=WoSign CA Limited, CN=Certification Authority of WoSign G2
    {0xD4, 0x87, 0xA5, 0x6F, 0x83, 0xB0, 0x74, 0x82, 0xE8, 0x5E, 0x96, 0x33,
     0x94, 0xC1, 0xEC, 0xC2, 0xC9, 0xE5, 0x1D, 0x09, 0x03, 0xEE, 0x94, 0x6B,
     0x02, 0xC3, 0x01, 0x58, 0x1E, 0xD9, 0x9E, 0x16},
    // C=CN, O=WoSign CA Limited, CN=CA
    // \xE6\xB2\x83\xE9\x80\x9A\xE6\xA0\xB9\xE8\xAF\x81\xE4\xB9\xA6
    {0xD6, 0xF0, 0x34, 0xBD, 0x94, 0xAA, 0x23, 0x3F, 0x02, 0x97, 0xEC, 0xA4,
     0x24, 0x5B, 0x28, 0x39, 0x73, 0xE4, 0x47, 0xAA, 0x59, 0x0F, 0x31, 0x0C,
     0x77, 0xF4, 0x8F, 0xDF, 0x83, 0x11, 0x22, 0x54},
    // C=GB, ST=Greater Manchester, L=Salford, O=Comodo CA Limited, CN=AAA
    // Certificate Services
    {0xD7, 0xA7, 0xA0, 0xFB, 0x5D, 0x7E, 0x27, 0x31, 0xD7, 0x71, 0xE9, 0x48,
     0x4E, 0xBC, 0xDE, 0xF7, 0x1D, 0x5F, 0x0C, 0x3E, 0x0A, 0x29, 0x48, 0x78,
     0x2B, 0xC8, 0x3E, 0xE0, 0xEA, 0x69, 0x9E, 0xF4},
    // C=DE, O=Deutscher Sparkassen Verlag GmbH, OU=S-TRUST Certification
    // Services, CN=S-TRUST Universal Root CA
    {0xD8, 0x0F, 0xEF, 0x91, 0x0A, 0xE3, 0xF1, 0x04, 0x72, 0x3B, 0x04, 0x5C,
     0xEC, 0x2D, 0x01, 0x9F, 0x44, 0x1C, 0xE6, 0x21, 0x3A, 0xDF, 0x15, 0x67,
     0x91, 0xE7, 0x0C, 0x17, 0x90, 0x11, 0x0A, 0x31},
    // C=PL, O=Unizeto Sp. z o.o., CN=Certum CA
    {0xD8, 0xE0, 0xFE, 0xBC, 0x1D, 0xB2, 0xE3, 0x8D, 0x00, 0x94, 0x0F, 0x37,
     0xD2, 0x7D, 0x41, 0x34, 0x4D, 0x99, 0x3E, 0x73, 0x4B, 0x99, 0xD5, 0x65,
     0x6D, 0x97, 0x78, 0xD4, 0xD8, 0x14, 0x36, 0x24},
    // C=ch, O=Swisscom, OU=Digital Certificate Services, CN=Swisscom Root EV CA
    // 2
    {0xD9, 0x5F, 0xEA, 0x3C, 0xA4, 0xEE, 0xDC, 0xE7, 0x4C, 0xD7, 0x6E, 0x75,
     0xFC, 0x6D, 0x1F, 0xF6, 0x2C, 0x44, 0x1F, 0x0F, 0xA8, 0xBC, 0x77, 0xF0,
     0x34, 0xB1, 0x9E, 0x5D, 0xB2, 0x58, 0x01, 0x5D},
    // O=TeliaSonera, CN=TeliaSonera Root CA v1
    {0xDD, 0x69, 0x36, 0xFE, 0x21, 0xF8, 0xF0, 0x77, 0xC1, 0x23, 0xA1, 0xA5,
     0x21, 0xC1, 0x22, 0x24, 0xF7, 0x22, 0x55, 0xB7, 0x3E, 0x03, 0xA7, 0x26,
     0x06, 0x93, 0xE8, 0xA2, 0x4B, 0x0F, 0xA3, 0x89},
    // C=BE, O=Certipost s.a./n.v., CN=Certipost E-Trust TOP Root CA
    {0xDD, 0xFF, 0x53, 0xEC, 0xD7, 0x74, 0x3B, 0x60, 0xBB, 0x7B, 0x27, 0x95,
     0xFF, 0x57, 0x32, 0xFA, 0x78, 0x5F, 0x9A, 0x14, 0xDF, 0x11, 0x20, 0xFB,
     0x40, 0xA3, 0x8C, 0xF8, 0x4C, 0xA2, 0xA5, 0x66},
    // C=US, ST=Washington, L=Redmond, O=Microsoft Corporation, CN=Microsoft
    // Root Certificate Authority 2010
    {0xDF, 0x54, 0x5B, 0xF9, 0x19, 0xA2, 0x43, 0x9C, 0x36, 0x98, 0x3B, 0x54,
     0xCD, 0xFC, 0x90, 0x3D, 0xFA, 0x4F, 0x37, 0xD3, 0x99, 0x6D, 0x8D, 0x84,
     0xB4, 0xC3, 0x1E, 0xEC, 0x6F, 0x3C, 0x16, 0x3E},
    // C=ES, ST=Barcelona, L=Barcelona (see current address at
    // http://www.anf.es/es/address-direccion.html ), O=ANF Autoridad de
    // Certificacion, OU=ANF Clase 1
    // CA/emailAddress=info@anf.es/serialNumber=G63287510, CN=ANF Global Root CA
    {0xE0, 0xE1, 0x7A, 0xEA, 0x06, 0xCF, 0x9C, 0xE1, 0x2A, 0xAE, 0x81, 0x90,
     0x34, 0x5A, 0x2C, 0x59, 0x72, 0x01, 0x30, 0xA7, 0xD8, 0xFF, 0x72, 0xF3,
     0x74, 0x5A, 0xD7, 0x5D, 0xBA, 0xA3, 0x65, 0xB6},
    // C=SK, L=Bratislava, O=Disig a.s., CN=CA Disig Root R2
    {0xE2, 0x3D, 0x4A, 0x03, 0x6D, 0x7B, 0x70, 0xE9, 0xF5, 0x95, 0xB1, 0x42,
     0x20, 0x79, 0xD2, 0xB9, 0x1E, 0xDF, 0xBB, 0x1F, 0xB6, 0x51, 0xA0, 0x63,
     0x3E, 0xAA, 0x8A, 0x9D, 0xC5, 0xF8, 0x07, 0x03},
    // C=CN, O=CNNIC, CN=CNNIC ROOT
    {0xE2, 0x83, 0x93, 0x77, 0x3D, 0xA8, 0x45, 0xA6, 0x79, 0xF2, 0x08, 0x0C,
     0xC7, 0xFB, 0x44, 0xA3, 0xB7, 0xA1, 0xC3, 0x79, 0x2C, 0xB7, 0xEB, 0x77,
     0x29, 0xFD, 0xCB, 0x6A, 0x8D, 0x99, 0xAE, 0xA7},
    // C=US, O=VeriSign, Inc., OU=VeriSign Trust Network, OU=(c) 1999 VeriSign,
    // Inc. - For authorized use only, CN=VeriSign Class 4 Public Primary
    // Certification Authority - G3
    {0xE3, 0x89, 0x36, 0x0D, 0x0F, 0xDB, 0xAE, 0xB3, 0xD2, 0x50, 0x58, 0x4B,
     0x47, 0x30, 0x31, 0x4E, 0x22, 0x2F, 0x39, 0xC1, 0x56, 0xA0, 0x20, 0x14,
     0x4E, 0x8D, 0x96, 0x05, 0x61, 0x79, 0x15, 0x06},
    // C=FR, O=Dhimyotis, CN=Certigna
    {0xE3, 0xB6, 0xA2, 0xDB, 0x2E, 0xD7, 0xCE, 0x48, 0x84, 0x2F, 0x7A, 0xC5,
     0x32, 0x41, 0xC7, 0xB7, 0x1D, 0x54, 0x14, 0x4B, 0xFB, 0x40, 0xC1, 0x1F,
     0x3F, 0x1D, 0x0B, 0x42, 0xF5, 0xEE, 0xA1, 0x2D},
    // C=TR, L=Gebze - Kocaeli, O=T\xC3\xBCrkiye Bilimsel ve Teknolojik
    // Ara\xC5\x9Ft\xC4\xB1rma Kurumu - T\xC3\x9CB\xC4\xB0TAK, OU=Ulusal
    // Elektronik ve Kriptoloji Ara\xC5\x9Ft\xC4\xB1rma Enstit\xC3\xBCs\xC3\xBC
    // - UEKAE, OU=Kamu Sertifikasyon Merkezi, CN=T\xC3\x9CB\xC4\xB0TAK UEKAE
    // K\xC3\xB6k Sertifika Hizmet Sa\xC4\x9Flay\xC4\xB1c\xC4\xB1s\xC4\xB1 -
    // S\xC3\xBCr\xC3\xBCm 3
    {0xE4, 0xC7, 0x34, 0x30, 0xD7, 0xA5, 0xB5, 0x09, 0x25, 0xDF, 0x43, 0x37,
     0x0A, 0x0D, 0x21, 0x6E, 0x9A, 0x79, 0xB9, 0xD6, 0xDB, 0x83, 0x73, 0xA0,
     0xC6, 0x9E, 0xB1, 0xCC, 0x31, 0xC7, 0xC5, 0x2A},
    // C=us, ST=Utah, L=Salt Lake City, O=Digital Signature Trust Co., OU=DSTCA
    // X2, CN=DST RootCA X2/emailAddress=ca@digsigtrust.com
    {0xE5, 0x72, 0x10, 0xAB, 0x81, 0x2C, 0x8D, 0xF3, 0x08, 0x26, 0x7C, 0xB4,
     0x29, 0x1B, 0x98, 0xE9, 0x56, 0x59, 0x7C, 0xA3, 0x6E, 0xC2, 0xB9, 0x51,
     0x89, 0xEF, 0x17, 0x23, 0x39, 0x6B, 0xCA, 0xC8},
    // C=BR, O=Serasa S.A., OU=Serasa CA III, CN=Serasa Certificate Authority
    // III
    {0xE5, 0xBD, 0xA8, 0x20, 0xE5, 0xCE, 0x15, 0xBF, 0xD0, 0x7B, 0xA1, 0x1F,
     0xFB, 0x1C, 0x7C, 0x8A, 0x59, 0x10, 0xCE, 0x1B, 0x90, 0x17, 0x5C, 0x34,
     0x30, 0x8B, 0xC2, 0x50, 0x04, 0x53, 0xCC, 0xDC},
    // C=KR, O=Government of Korea, OU=GPKI, CN=Root CA
    {0xE5, 0xC0, 0x1C, 0xB4, 0x09, 0x32, 0x79, 0xFA, 0xA1, 0x9F, 0xCF, 0xA2,
     0x4E, 0xA4, 0x3E, 0xB1, 0xB2, 0x6D, 0x07, 0xA6, 0x15, 0xAD, 0xF7, 0x24,
     0x01, 0x84, 0xA1, 0xE7, 0x16, 0xB7, 0x61, 0xC9},
    // C=TR, O=Elektronik Bilgi Guvenligi A.S., CN=e-Guven Kok Elektronik
    // Sertifika Hizmet Saglayicisi
    {0xE6, 0x09, 0x07, 0x84, 0x65, 0xA4, 0x19, 0x78, 0x0C, 0xB6, 0xAC, 0x4C,
     0x1C, 0x0B, 0xFB, 0x46, 0x53, 0xD9, 0xD9, 0xCC, 0x6E, 0xB3, 0x94, 0x6E,
     0xB7, 0xF3, 0xD6, 0x99, 0x97, 0xBA, 0xD5, 0x98},
    // C=DE, O=TC TrustCenter GmbH, OU=TC TrustCenter Class 2 CA, CN=TC
    // TrustCenter Class 2 CA II
    {0xE6, 0xB8, 0xF8, 0x76, 0x64, 0x85, 0xF8, 0x07, 0xAE, 0x7F, 0x8D, 0xAC,
     0x16, 0x70, 0x46, 0x1F, 0x07, 0xC0, 0xA1, 0x3E, 0xEF, 0x3A, 0x1F, 0xF7,
     0x17, 0x53, 0x8D, 0x7A, 0xBA, 0xD3, 0x91, 0xB4},
    // C=LT, O=Skaitmeninio sertifikavimo centras, OU=Certification Authority,
    // CN=SSC Root CA C
    {0xE6, 0xE4, 0xA9, 0x51, 0xEC, 0xBF, 0x7D, 0x8E, 0xDC, 0x01, 0xBC, 0x87,
     0x3F, 0x7B, 0x6F, 0xD3, 0x58, 0x68, 0xBD, 0xB1, 0x0E, 0xD7, 0x86, 0xF3,
     0xA1, 0xB1, 0xEE, 0x16, 0xD8, 0xCE, 0xC3, 0xE9},
    // C=JP, O=SECOM Trust.net, OU=Security Communication RootCA1
    {0xE7, 0x5E, 0x72, 0xED, 0x9F, 0x56, 0x0E, 0xEC, 0x6E, 0xB4, 0x80, 0x00,
     0x73, 0xA4, 0x3F, 0xC3, 0xAD, 0x19, 0x19, 0x5A, 0x39, 0x22, 0x82, 0x01,
     0x78, 0x95, 0x97, 0x4A, 0x99, 0x02, 0x6B, 0x6C},
    // C=US, O=VeriSign, Inc., OU=Class 3 Public Primary Certification Authority
    {0xE7, 0x68, 0x56, 0x34, 0xEF, 0xAC, 0xF6, 0x9A, 0xCE, 0x93, 0x9A, 0x6B,
     0x25, 0x5B, 0x7B, 0x4F, 0xAB, 0xEF, 0x42, 0x93, 0x5B, 0x50, 0xA2, 0x65,
     0xAC, 0xB5, 0xCB, 0x60, 0x27, 0xE4, 0x4E, 0x70},
    // C=US, ST=New Jersey, L=Jersey City, O=The USERTRUST Network, CN=USERTrust
    // RSA Certification Authority
    {0xE7, 0x93, 0xC9, 0xB0, 0x2F, 0xD8, 0xAA, 0x13, 0xE2, 0x1C, 0x31, 0x22,
     0x8A, 0xCC, 0xB0, 0x81, 0x19, 0x64, 0x3B, 0x74, 0x9C, 0x89, 0x89, 0x64,
     0xB1, 0x74, 0x6D, 0x46, 0xC3, 0xD4, 0xCB, 0xD2},
    // C=us, ST=Utah, L=Salt Lake City, O=Digital Signature Trust Co., OU=United
    // Parcel Service, CN=DST (UPS) RootCA/emailAddress=ca@digsigtrust.com
    {0xE8, 0x73, 0xD4, 0x08, 0x2A, 0x7B, 0x46, 0x32, 0x93, 0x4F, 0x48, 0xA5,
     0xCC, 0x1E, 0xE5, 0x00, 0x93, 0x2F, 0x66, 0x1E, 0x56, 0xC3, 0x46, 0x7C,
     0x5C, 0x84, 0xD3, 0x14, 0x47, 0x47, 0x6B, 0x0C},
    // C=AT, O=A-Trust Ges. f\xFCr Sicherheitssysteme im elektr. Datenverkehr
    // GmbH, OU=A-Trust-Qual-01, CN=A-Trust-Qual-01
    {0xE8, 0xA2, 0xF4, 0x41, 0x65, 0x76, 0x78, 0x97, 0x5F, 0x2B, 0x97, 0xD7,
     0x75, 0x71, 0x9C, 0x7D, 0x49, 0xD9, 0x22, 0x34, 0x55, 0x45, 0x40, 0xEC,
     0x14, 0xD9, 0x2E, 0x16, 0xFE, 0x27, 0xD2, 0xCB},
    // C=BR, O=Certisign Certificadora Digital Ltda., OU=Certisign - Autoridade
    // Certificadora - AC2
    {0xE8, 0xB2, 0x8D, 0x2A, 0x3D, 0x81, 0xF6, 0x3B, 0x4E, 0x44, 0x67, 0xC2,
     0x19, 0x0A, 0x63, 0x1F, 0xC0, 0x62, 0x35, 0x3B, 0x5F, 0x2D, 0x25, 0x85,
     0x1D, 0xDA, 0x6B, 0x64, 0x4A, 0xC7, 0x8B, 0x3F},
    // C=DE, O=D-Trust GmbH, CN=D-TRUST Qualified Root CA 1 2007:PN
    {0xE8, 0xC6, 0xAA, 0x6B, 0x5F, 0x58, 0xA8, 0xF2, 0xA6, 0x36, 0x5C, 0xF9,
     0x8E, 0x65, 0x69, 0x35, 0x63, 0xA3, 0x8B, 0x7B, 0x2F, 0x32, 0xCF, 0x1B,
     0xE0, 0x6F, 0x2D, 0x22, 0x29, 0xD4, 0xBF, 0x59},
    // C=BG, O=InfoNotary PLC, DC=root-ca, CN=InfoNotary CSP Root, OU=InfoNotary
    // CSP Root/emailAddress=csp@infonotary.com
    {0xEA, 0x7E, 0x31, 0x2E, 0xCE, 0x48, 0x7B, 0x4C, 0x0A, 0xA6, 0x3C, 0xC8,
     0x0A, 0xB9, 0xFC, 0xB3, 0x3C, 0x72, 0x05, 0x73, 0xF8, 0x94, 0x5F, 0x77,
     0x61, 0x74, 0x5F, 0xC6, 0x38, 0x63, 0xD3, 0x9D},
    // C=RO, O=certSIGN, OU=certSIGN ROOT CA
    {0xEA, 0xA9, 0x62, 0xC4, 0xFA, 0x4A, 0x6B, 0xAF, 0xEB, 0xE4, 0x15, 0x19,
     0x6D, 0x35, 0x1C, 0xCD, 0x88, 0x8D, 0x4F, 0x53, 0xF3, 0xFA, 0x8A, 0xE6,
     0xD7, 0xC4, 0x66, 0xA9, 0x4E, 0x60, 0x42, 0xBB},
    // C=CH, O=admin, OU=Services, OU=Certification Authorities,
    // CN=AdminCA-CD-T01
    {0xEA, 0xC0, 0x22, 0x0C, 0x5C, 0x9F, 0xEC, 0xC5, 0x12, 0x1D, 0x37, 0x20,
     0x87, 0x2D, 0x06, 0x70, 0x7B, 0x52, 0x66, 0xBE, 0x25, 0xD4, 0xEB, 0xB5,
     0x6A, 0xB8, 0x04, 0xBB, 0xBF, 0x85, 0xFE, 0x03},
    // C=US, O=VeriSign, Inc., OU=VeriSign Trust Network, OU=(c) 1999 VeriSign,
    // Inc. - For authorized use only, CN=VeriSign Class 3 Public Primary
    // Certification Authority - G3
    {0xEB, 0x04, 0xCF, 0x5E, 0xB1, 0xF3, 0x9A, 0xFA, 0x76, 0x2F, 0x2B, 0xB1,
     0x20, 0xF2, 0x96, 0xCB, 0xA5, 0x20, 0xC1, 0xB9, 0x7D, 0xB1, 0x58, 0x95,
     0x65, 0xB8, 0x1C, 0xB9, 0xA1, 0x7B, 0x72, 0x44},
    // C=HU, L=Budapest, O=NetLock Kft.,
    // OU=Tan\xC3\xBAs\xC3\xADtv\xC3\xA1nykiad\xC3\xB3k (Certification
    // Services), CN=NetLock Platina (Class Platinum)
    // F\xC5\x91tan\xC3\xBAs\xC3\xADtv\xC3\xA1ny
    {0xEB, 0x7E, 0x05, 0xAA, 0x58, 0xE7, 0xBD, 0x32, 0x8A, 0x28, 0x2B, 0xF8,
     0x86, 0x70, 0x33, 0xF3, 0xC0, 0x35, 0x34, 0x2B, 0x51, 0x6E, 0xE8, 0x5C,
     0x01, 0x67, 0x3D, 0xFF, 0xFF, 0xBB, 0xFE, 0x58},
    // C=BE, O=GlobalSign nv-sa, OU=Root CA, CN=GlobalSign Root CA
    {0xEB, 0xD4, 0x10, 0x40, 0xE4, 0xBB, 0x3E, 0xC7, 0x42, 0xC9, 0xE3, 0x81,
     0xD3, 0x1E, 0xF2, 0xA4, 0x1A, 0x48, 0xB6, 0x68, 0x5C, 0x96, 0xE7, 0xCE,
     0xF3, 0xC1, 0xDF, 0x6C, 0xD4, 0x33, 0x1C, 0x99},
    // C=DE, O=TC TrustCenter GmbH, OU=TC TrustCenter Universal CA, CN=TC
    // TrustCenter Universal CA I
    {0xEB, 0xF3, 0xC0, 0x2A, 0x87, 0x89, 0xB1, 0xFB, 0x7D, 0x51, 0x19, 0x95,
     0xD6, 0x63, 0xB7, 0x29, 0x06, 0xD9, 0x13, 0xCE, 0x0D, 0x5E, 0x10, 0x56,
     0x8A, 0x8A, 0x77, 0xE2, 0x58, 0x61, 0x67, 0xE7},
    // emailAddress=pki@sk.ee, C=EE, O=AS Sertifitseerimiskeskus, CN=Juur-SK
    {0xEC, 0xC3, 0xE9, 0xC3, 0x40, 0x75, 0x03, 0xBE, 0xE0, 0x91, 0xAA, 0x95,
     0x2F, 0x41, 0x34, 0x8F, 0xF8, 0x8B, 0xAA, 0x86, 0x3B, 0x22, 0x64, 0xBE,
     0xFA, 0xC8, 0x07, 0x90, 0x15, 0x74, 0xE9, 0x39},
    // C=NO, O=Buypass AS-983163327, CN=Buypass Class 3 Root CA
    {0xED, 0xF7, 0xEB, 0xBC, 0xA2, 0x7A, 0x2A, 0x38, 0x4D, 0x38, 0x7B, 0x7D,
     0x40, 0x10, 0xC6, 0x66, 0xE2, 0xED, 0xB4, 0x84, 0x3E, 0x4C, 0x29, 0xB4,
     0xAE, 0x1D, 0x5B, 0x93, 0x32, 0xE6, 0xB2, 0x4D},
    // C=DE, O=D-Trust GmbH, CN=D-TRUST Root Class 3 CA 2 EV 2009
    {0xEE, 0xC5, 0x49, 0x6B, 0x98, 0x8C, 0xE9, 0x86, 0x25, 0xB9, 0x34, 0x09,
     0x2E, 0xEC, 0x29, 0x08, 0xBE, 0xD0, 0xB0, 0xF3, 0x16, 0xC2, 0xD4, 0x73,
     0x0C, 0x84, 0xEA, 0xF1, 0xF3, 0xD3, 0x48, 0x81},
    // C=ES, ST=Madrid, L=Madrid, O=IPS Certification Authority s.l. ipsCA,
    // OU=ipsCA, CN=ipsCA Main CA Root/emailAddress=main01@ipsca.com
    {0xEE, 0xFC, 0xA8, 0x88, 0xDB, 0x44, 0x2C, 0xEA, 0x1F, 0x03, 0xFA, 0xC5,
     0xDE, 0x5B, 0x1A, 0xF2, 0x10, 0xAE, 0x03, 0xF5, 0xE1, 0x65, 0x8D, 0xDB,
     0x88, 0x0C, 0x64, 0x5E, 0x78, 0x62, 0x45, 0x46},
    // C=EU, O=AC Camerfirma SA CIF A82743287, OU=http://www.chambersign.org,
    // CN=Global Chambersign Root
    {0xEF, 0x3C, 0xB4, 0x17, 0xFC, 0x8E, 0xBF, 0x6F, 0x97, 0x87, 0x6C, 0x9E,
     0x4E, 0xCE, 0x39, 0xDE, 0x1E, 0xA5, 0xFE, 0x64, 0x91, 0x41, 0xD1, 0x02,
     0x8B, 0x7D, 0x11, 0xC0, 0xB2, 0x29, 0x8C, 0xED},
    // C=BE, O=Certipost s.a./n.v., CN=Certipost E-Trust Primary Normalised CA
    {0xF0, 0x03, 0x55, 0xEE, 0xF1, 0x01, 0xC7, 0xDF, 0x4E, 0x46, 0xCC, 0xE6,
     0x41, 0x7D, 0xFF, 0xCE, 0x3D, 0xB8, 0x2D, 0xBB, 0x13, 0x69, 0xC3, 0xB4,
     0x39, 0xC4, 0xE3, 0x3B, 0xEE, 0x44, 0x5C, 0x42},
    // C=FI, ST=Finland, O=Vaestorekisterikeskus CA, OU=Certification Authority
    // Services, OU=Varmennepalvelut, CN=VRK Gov. Root CA
    {0xF0, 0x08, 0x73, 0x3E, 0xC5, 0x00, 0xDC, 0x49, 0x87, 0x63, 0xCC, 0x92,
     0x64, 0xC6, 0xFC, 0xEA, 0x40, 0xEC, 0x22, 0x00, 0x0E, 0x92, 0x7D, 0x05,
     0x3C, 0xE9, 0xC9, 0x0B, 0xFA, 0x04, 0x6C, 0xB2},
    // C=ch, O=Swisscom, OU=Digital Certificate Services, CN=Swisscom Root CA 2
    {0xF0, 0x9B, 0x12, 0x2C, 0x71, 0x14, 0xF4, 0xA0, 0x9B, 0xD4, 0xEA, 0x4F,
     0x4A, 0x99, 0xD5, 0x58, 0xB4, 0x6E, 0x4C, 0x25, 0xCD, 0x81, 0x14, 0x0D,
     0x29, 0xC0, 0x56, 0x13, 0x91, 0x4C, 0x38, 0x41},
    // C=LT, O=Skaitmeninio sertifikavimo centras, OU=Certification Authority,
    // CN=SSC Root CA A
    {0xF1, 0xB1, 0x3F, 0x5C, 0x9A, 0x32, 0x64, 0x03, 0xB0, 0xF3, 0x1B, 0xBE,
     0x76, 0x99, 0xCD, 0x17, 0xC7, 0xD1, 0xC0, 0xB9, 0x81, 0x58, 0x6D, 0xD1,
     0xA7, 0xB2, 0x19, 0xC5, 0x25, 0x08, 0xFE, 0x99},
    // C=US, O=SecureTrust Corporation, CN=SecureTrust CA
    {0xF1, 0xC1, 0xB5, 0x0A, 0xE5, 0xA2, 0x0D, 0xD8, 0x03, 0x0E, 0xC9, 0xF6,
     0xBC, 0x24, 0x82, 0x3D, 0xD3, 0x67, 0xB5, 0x25, 0x57, 0x59, 0xB4, 0xE7,
     0x1B, 0x61, 0xFC, 0xE9, 0xF7, 0x37, 0x5D, 0x73},
    // C=ES, ST=BARCELONA, L=BARCELONA, O=IPS Seguridad CA, OU=Certificaciones,
    // CN=IPS SERVIDORES/emailAddress=ips@mail.ips.es
    {0xF1, 0xF3, 0xCC, 0x20, 0x7A, 0x6D, 0x47, 0x94, 0x7B, 0x8C, 0xB9, 0xC3,
     0x04, 0x22, 0x22, 0x9D, 0xE0, 0xD7, 0x1F, 0xB8, 0x67, 0xE0, 0xB9, 0xA3,
     0xED, 0xA0, 0x8E, 0x0E, 0x17, 0x36, 0xBC, 0x28},
    // C=AT, O=A-Trust Ges. f. Sicherheitssysteme im elektr. Datenverkehr GmbH,
    // OU=A-Trust-Qual-02, CN=A-Trust-Qual-02
    {0xF2, 0x86, 0x30, 0xBA, 0xBF, 0x25, 0x6E, 0x56, 0x7B, 0x58, 0x21, 0x06,
     0x9F, 0xCF, 0x13, 0x14, 0x8A, 0xB9, 0xA2, 0x3E, 0x28, 0xFC, 0x0D, 0x70,
     0x61, 0x5A, 0xAE, 0x6E, 0xD2, 0x84, 0xF4, 0xC8},
    // CN=Atos TrustedRoot 2011, O=Atos, C=DE
    {0xF3, 0x56, 0xBE, 0xA2, 0x44, 0xB7, 0xA9, 0x1E, 0xB3, 0x5D, 0x53, 0xCA,
     0x9A, 0xD7, 0x86, 0x4A, 0xCE, 0x01, 0x8E, 0x2D, 0x35, 0xD5, 0xF8, 0xF9,
     0x6D, 0xDF, 0x68, 0xA6, 0xF4, 0x1A, 0xA4, 0x74},
    // C=IN, O=India PKI, CN=CCA India 2007
    {0xF3, 0x75, 0xE2, 0xF7, 0x7A, 0x10, 0x8B, 0xAC, 0xC4, 0x23, 0x48, 0x94,
     0xA9, 0xAF, 0x30, 0x8E, 0xDE, 0xCA, 0x1A, 0xCD, 0x8F, 0xBD, 0xE0, 0xE7,
     0xAA, 0xA9, 0x63, 0x4E, 0x9D, 0xAF, 0x7E, 0x1C},
    // OU=Copyright (c) 1997 Microsoft Corp., OU=Microsoft Corporation,
    // CN=Microsoft Root Authority
    {0xF3, 0x84, 0x06, 0xE5, 0x40, 0xD7, 0xA9, 0xD9, 0x0C, 0xB4, 0xA9, 0x47,
     0x92, 0x99, 0x64, 0x0F, 0xFB, 0x6D, 0xF9, 0xE2, 0x24, 0xEC, 0xC7, 0xA0,
     0x1C, 0x0D, 0x95, 0x58, 0xD8, 0xDA, 0xD7, 0x7D},
    // C=ES, O=Agencia Notarial de Certificacion S.L.U. - CIF B83395988,
    // CN=ANCERT Certificados CGN V2
    {0xF4, 0x33, 0x6B, 0xC2, 0xAC, 0x75, 0x95, 0x0B, 0xEC, 0xCF, 0x1C, 0x1F,
     0x2F, 0x9D, 0xA6, 0xDD, 0xDA, 0xFD, 0x1F, 0x41, 0x16, 0x1C, 0xA7, 0x1F,
     0x59, 0xC7, 0x68, 0x89, 0xBD, 0x47, 0x40, 0x33},
    // L=ValiCert Validation Network, O=ValiCert, Inc., OU=ValiCert Class 1
    // Policy Validation Authority,
    // CN=http://www.valicert.com//emailAddress=info@valicert.com
    {0xF4, 0xC1, 0x49, 0x55, 0x1A, 0x30, 0x13, 0xA3, 0x5B, 0xC7, 0xBF, 0xFE,
     0x17, 0xA7, 0xF3, 0x44, 0x9B, 0xC1, 0xAB, 0x5B, 0x5A, 0x0A, 0xE7, 0x4B,
     0x06, 0xC2, 0x3B, 0x90, 0x00, 0x4C, 0x01, 0x04},
    // C=IT, O=SIA S.p.A., L=Milano, CN=SIA Secure Server CA
    {0xF5, 0x63, 0xC5, 0xC3, 0xE5, 0x12, 0xE6, 0x3B, 0x97, 0xB5, 0x43, 0x8F,
     0x2B, 0xD4, 0xA9, 0xAE, 0x78, 0xA4, 0xF9, 0xEA, 0xD9, 0x2B, 0xCC, 0x34,
     0xFE, 0x97, 0x3B, 0xDC, 0x7C, 0x6D, 0x21, 0x48},
    // C=SE, O=Swedish Social Insurance Agency, CN=Swedish Government Root
    // Authority v1
    {0xF6, 0x57, 0xA6, 0x33, 0xEE, 0xB9, 0xBC, 0x5D, 0x15, 0xA4, 0x61, 0x75,
     0x17, 0x49, 0xEA, 0x4B, 0x31, 0x67, 0x27, 0xDC, 0xF1, 0xA9, 0xF9, 0x86,
     0xB5, 0x45, 0x84, 0x45, 0xF6, 0x48, 0x5D, 0xDE},
    // C=UY, O=ADMINISTRACION NACIONAL DE CORREOS, OU=SERVICIOS ELECTRONICOS,
    // CN=SERVICIOS DE CERTIFICACION - A.N.C./mail=correo_cert@correo.com.uy
    {0xF7, 0x73, 0xBC, 0x65, 0x65, 0x9F, 0x1B, 0xC5, 0x90, 0x87, 0xBF, 0x21,
     0x4E, 0xEA, 0xD8, 0x64, 0x01, 0x0D, 0x58, 0x87, 0xCD, 0x2C, 0xD8, 0x4E,
     0x4F, 0x1B, 0xA7, 0x52, 0x3F, 0xE5, 0x56, 0x40},
    // C=SK, L=Bratislava, O=Disig a.s., CN=CA Disig Root R1
    {0xF9, 0x6F, 0x23, 0xF4, 0xC3, 0xE7, 0x9C, 0x07, 0x7A, 0x46, 0x98, 0x8D,
     0x5A, 0xF5, 0x90, 0x06, 0x76, 0xA0, 0xF0, 0x39, 0xCB, 0x64, 0x5D, 0xD1,
     0x75, 0x49, 0xB2, 0x16, 0xC8, 0x24, 0x40, 0xCE},
    // C=HK, O=Hongkong Post, CN=Hongkong Post Root CA 1
    {0xF9, 0xE6, 0x7D, 0x33, 0x6C, 0x51, 0x00, 0x2A, 0xC0, 0x54, 0xC6, 0x32,
     0x02, 0x2D, 0x66, 0xDD, 0xA2, 0xE7, 0xE3, 0xFF, 0xF1, 0x0A, 0xD0, 0x61,
     0xED, 0x31, 0xD8, 0xBB, 0xB4, 0x10, 0xCF, 0xB2},
    // C=PL, O=Krajowa Izba Rozliczeniowa S.A., CN=SZAFIR ROOT CA
    {0xFA, 0xBC, 0xF5, 0x19, 0x7C, 0xDD, 0x7F, 0x45, 0x8A, 0xC3, 0x38, 0x32,
     0xD3, 0x28, 0x40, 0x21, 0xDB, 0x24, 0x25, 0xFD, 0x6B, 0xEA, 0x7A, 0x2E,
     0x69, 0xB7, 0x48, 0x6E, 0x8F, 0x51, 0xF9, 0xCC},
    // C=BR, O=ICP-Brasil, OU=Instituto Nacional de Tecnologia da Informacao -
    // ITI, CN=Autoridade Certificadora Raiz Brasileira v2
    {0xFB, 0x47, 0xD9, 0x2A, 0x99, 0x09, 0xFD, 0x4F, 0xA9, 0xBE, 0xC0, 0x27,
     0x37, 0x54, 0x3E, 0x1F, 0x35, 0x14, 0xCE, 0xD7, 0x47, 0x40, 0x7A, 0x8D,
     0x9C, 0xFA, 0x39, 0x7B, 0x09, 0x15, 0x06, 0x7C},
    // C=IT, O=SIA S.p.A., L=Milano, CN=SIA Secure Client CA
    {0xFC, 0x0A, 0x0F, 0xE2, 0x7C, 0x9D, 0xC1, 0x3C, 0x81, 0x23, 0x8A, 0x59,
     0x13, 0xA1, 0xDA, 0xF8, 0x18, 0x41, 0x68, 0xBE, 0xB7, 0xE5, 0xA4, 0x51,
     0x2A, 0x77, 0x1F, 0xD4, 0xF4, 0x53, 0x65, 0x1D},
    // C=SE, O=Inera AB, CN=SITHS Root CA v1
    {0xFC, 0x50, 0xB2, 0x6B, 0xDC, 0x4A, 0x8F, 0xDF, 0x13, 0x44, 0xCC, 0x80,
     0x15, 0x7A, 0xE1, 0x3A, 0xC6, 0x71, 0xE2, 0x70, 0x6F, 0xAC, 0xFC, 0x06,
     0x05, 0xFE, 0x34, 0xE2, 0x49, 0xEB, 0x72, 0xD6},
    // C=FR, O=Certinomis, OU=0002 433998903, CN=Certinomis - Autorit\xC3\xA9
    // Racine
    {0xFC, 0xBF, 0xE2, 0x88, 0x62, 0x06, 0xF7, 0x2B, 0x27, 0x59, 0x3C, 0x8B,
     0x07, 0x02, 0x97, 0xE1, 0x2D, 0x76, 0x9E, 0xD1, 0x0E, 0xD7, 0x93, 0x07,
     0x05, 0xA8, 0x09, 0x8E, 0xFF, 0xC1, 0x4D, 0x17},
    // C=DE, O=T-Systems Enterprise Services GmbH, OU=T-Systems Trust Center,
    // CN=T-TeleSec GlobalRoot Class 3
    {0xFD, 0x73, 0xDA, 0xD3, 0x1C, 0x64, 0x4F, 0xF1, 0xB4, 0x3B, 0xEF, 0x0C,
     0xCD, 0xDA, 0x96, 0x71, 0x0B, 0x9C, 0xD9, 0x87, 0x5E, 0xCA, 0x7E, 0x31,
     0x70, 0x7A, 0xF3, 0xE9, 0x6D, 0x52, 0x2B, 0xBD},
    // C=SI, O=Halcom, CN=Halcom CA PO 2
    {0xFE, 0x71, 0x14, 0xD0, 0x7A, 0x14, 0x77, 0x59, 0x89, 0x1F, 0xF3, 0x7B,
     0x4F, 0x53, 0xEB, 0x43, 0x56, 0x82, 0x96, 0xBC, 0x3B, 0xF8, 0x9B, 0xC1,
     0x2C, 0xAF, 0xB1, 0x86, 0x98, 0x5E, 0xF2, 0x8D},
    // C=US, O=GeoTrust Inc., CN=GeoTrust Global CA
    {0xFF, 0x85, 0x6A, 0x2D, 0x25, 0x1D, 0xCD, 0x88, 0xD3, 0x66, 0x56, 0xF4,
     0x50, 0x12, 0x67, 0x98, 0xCF, 0xAB, 0xAA, 0xDE, 0x40, 0x79, 0x9C, 0x72,
     0x2D, 0xE4, 0xD2, 0xB5, 0xDB, 0x36, 0xA7, 0x3A},
    // C=BR, O=Serasa S.A., OU=Serasa CA II, CN=Serasa Certificate Authority II
    {0xFF, 0xCE, 0xF2, 0x22, 0x4E, 0x29, 0xB0, 0xB3, 0x6E, 0xC8, 0x31, 0x4E,
     0x68, 0x68, 0x22, 0xF3, 0xAC, 0x0F, 0x1C, 0x5E, 0x0C, 0x2D, 0x5C, 0x0E,
     0xB2, 0x48, 0x4C, 0xE7, 0xE2, 0x54, 0x0F, 0xD0},
};

#endif  // NET_CERT_X509_CERTIFICATE_KNOWN_ROOTS_WIN_H_
