
/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

// $Id: Calendars.i,v 1.19 2002/02/15 16:38:00 marmar Exp $

#ifndef quantlib_calendar_i
#define quantlib_calendar_i

%include Date.i
%include String.i

%{
using QuantLib::RollingConvention;
using QuantLib::Preceding;
using QuantLib::ModifiedPreceding;
using QuantLib::Following;
using QuantLib::ModifiedFollowing;
%}

// typemap rolling conventions to corresponding strings

%typemap(python,in) RollingConvention (RollingConvention temp),
                    const RollingConvention & (RollingConvention temp) {
    if (PyString_Check($source)) {
        std::string s(PyString_AsString($source));
        s = StringFormatter::toLowercase(s);
        if (s == "f" || s == "fol" || s == "following")
            temp = Following;
        else if (s == "mf" ||s == "modfol" || s == "modifiedfollowing")
            temp = ModifiedFollowing;
        else if (s == "p" || s == "pre" || s == "preceding")
            temp = Preceding;
        else if (s == "mp" ||s == "modpre" || s == "modifiedpreceding")
            temp = ModifiedPreceding;
        else {
            PyErr_SetString(PyExc_TypeError,(s+" is not "
                "a rolling convention").c_str());
            return NULL;
        }
    } else {
        PyErr_SetString(PyExc_TypeError,"rolling convention expected");
        return NULL;
    }
    $target = &temp;
};

%typemap(python,out) RollingConvention, const RollingConvention & {
    switch (*$source) {
      case Following:
        $target = PyString_FromString("Following");
        break;
      case ModifiedFollowing:
        $target = PyString_FromString("ModifiedFollowing");
        break;
      case Preceding:
        $target = PyString_FromString("Preceding");
        break;
      case ModifiedPreceding:
        $target = PyString_FromString("ModifiedPreceding");
        break;
      default:
        $target = PyString_FromString("Unknown");
    }
};

%typemap(python,ret) RollingConvention {
    delete $source;
}


%{
using QuantLib::Calendar;
using QuantLib::Calendars::Frankfurt;
using QuantLib::Calendars::Helsinki;
using QuantLib::Calendars::Johannesburg;
using QuantLib::Calendars::London;
using QuantLib::Calendars::Milan;
using QuantLib::Calendars::NewYork;
using QuantLib::Calendars::TARGET;
using QuantLib::Calendars::Tokyo;
using QuantLib::Calendars::Toronto;
using QuantLib::Calendars::Sydney;
using QuantLib::Calendars::Wellington;
using QuantLib::Calendars::Zurich;
%}

// export Calendar
class Calendar {
  public:
    // constructor redefined below as string-based factory
    ~Calendar();
    // inspectors
    bool isBusinessDay(const Date& d);
    bool isHoliday(const Date& d);
    Date roll(const Date& d, RollingConvention convention);
    Date advance(const Date& d, int n, TimeUnit unit,
      RollingConvention convention);
};

%addmethods Calendar {
    Calendar(const String& name) {
        String s = StringFormatter::toLowercase(name);
        if (s == "target" || s == "euro" || s == "eur")
            return new TARGET;
        else if (s == "newyork" || s == "ny" || s == "nyc")
            return new NewYork;
        else if (s == "london" || s == "lon")
            return new London;
        else if (s == "milan" || s == "mil")
            return new Milan;
        else if (s == "frankfurt" || s == "fft")
            return new Frankfurt;
        else if (s == "zurich" || s == "zur")
            return new Zurich;
        else if (s == "helsinki")
            return new Helsinki;
        else if (s == "johannesburg")
            return new Johannesburg;
        else if (s == "wellington")
            return new Wellington;
        else if (s == "tokyo")
            return new Tokyo;
        else if (s == "toronto")
            return new Toronto;
        else if (s == "sydney")
            return new Sydney;
        else
            throw Error("Unknown calendar: " + name);
        QL_DUMMY_RETURN((Calendar*)(0));
    }
    String __str__() {
        return self->name()+" calendar";
    }
    String __repr__() {
        return "Calendar('" + self->name() + "')";
    }
    int __cmp__(const Calendar& other) {
        return ((*self) == other ? 0 : 1);
    }
}


#endif

