#!/usr/bin/python

"""
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
"""

__version__ = "$Revision: 1.15 $"
# $Id: mcmultifactorpricers.py,v 1.15 2002/03/19 17:19:52 nando Exp $

import QuantLib
import unittest

class McMultiFactorPricersTest(unittest.TestCase):
    def runTest(self):
        "Testing Monte Carlo Multi Factor Pricers"
        cor = [[1.00, 0.50, 0.30, 0.10],
               [0.50, 1.00, 0.20, 0.40],
               [0.30, 0.20, 1.00, 0.60],
               [0.10, 0.40, 0.60, 1.00]]

        volatilities = [ 0.30,  0.35,  0.25,  0.20]
        covariance = QuantLib.getCovariance(volatilities, cor)

        dividendYields = [0.01, 0.05, 0.04, 0.03]
        riskFreeRate = 0.05
        resTime = 1.0

# degenerate portfolio
        perfectCorrelation = [[1.00, 1.00, 1.00, 1.00],
                              [1.00, 1.00, 1.00, 1.00],
                              [1.00, 1.00, 1.00, 1.00],
                              [1.00, 1.00, 1.00, 1.00]]
        sameAssetVols = [ 0.30,  0.30,  0.30,  0.30]
        sameAssetCovariance = QuantLib.getCovariance(sameAssetVols,
                                            perfectCorrelation)
        sameAssetDividend = [ 0.030,  0.030,  0.030,  0.030]



        seed = 86421
        fixedSamples = 100
        minimumTol = 0.01


# McEverest ###############
        p = QuantLib.McEverest(dividendYields, covariance,
                            riskFreeRate, resTime,
                            0, seed)
        storedValue = 0.743448106012
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-10):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())


        p = QuantLib.McEverest(dividendYields, covariance,
                            riskFreeRate, resTime,
                            1, seed)
        storedValue = 0.756978711699
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-10):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())

# McBasket ###############
        sameAssetValues = [ 25,  25,   25,  25]
        type = 'Call'
        strike = 100

        p = QuantLib.McBasket(type, sameAssetValues, strike,
                sameAssetDividend,
                sameAssetCovariance, riskFreeRate, resTime,
                0, seed)
        # european would be 12.4426495605
        storedValue = 10.4484452071
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-3):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())


        p = QuantLib.McBasket(type, sameAssetValues, strike,
                sameAssetDividend,
                sameAssetCovariance, riskFreeRate, resTime,
                1, seed)
        # european would be 12.4426495605
        storedValue = 12.2946770638
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-3):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())

# McMaxBasket ###############
        assetValues = [ 100,  110,   90,  105]

        p = QuantLib.McMaxBasket(assetValues, dividendYields,
                covariance, riskFreeRate, resTime,
                0, seed)
        storedValue = 120.733779723
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-9):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())


        p = QuantLib.McMaxBasket(assetValues, dividendYields,
                covariance, riskFreeRate, resTime,
                1, seed)
        storedValue = 123.520909467
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-9):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())



# McPagoda ###############
        portfolio   = [0.15, 0.20, 0.35, 0.30]
        fraction = 0.62
        roof = 0.20
        timeIncrements = [0.25, 0.5, 0.75, 1]

        p = QuantLib.McPagoda(portfolio,
                    fraction, roof,
                    dividendYields, covariance, riskFreeRate,
                    timeIncrements,
                    0, seed);
        storedValue =  0.0343897536199
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-10):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())


        p = QuantLib.McPagoda(portfolio,
                    fraction, roof,
                    dividendYields, covariance, riskFreeRate,
                    timeIncrements,
                    1, seed);
        storedValue = 0.0386095367584
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-10):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())



# McHimalaya ###############
        strike = 101
        p = QuantLib.McHimalaya(assetValues, dividendYields, covariance,
                       riskFreeRate, strike, timeIncrements,
                       0, seed)
        storedValue = 5.07684985849
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-10):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())


        p = QuantLib.McHimalaya(assetValues, dividendYields, covariance,
                       riskFreeRate, strike, timeIncrements,
                       1, seed)
        storedValue = 6.24780498724
        pvalue = p.valueWithSamples(fixedSamples)
        if not (abs(pvalue-storedValue) <= 1e-10):
            self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                      """ % locals())
        tol = p.errorEstimate()/pvalue
        tol = min(tol/2.0, minimumTol)
        pvalue = p.value(tol)
        accuracy = p.errorEstimate()/pvalue
        if not (accuracy <= tol):
            self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                      """ % locals())



def main():
    print 'testing QuantLib', QuantLib.__version__, QuantLib.QuantLibc.__file__, QuantLib.__file__
    import sys
    suite = unittest.TestSuite()
    suite.addTest(McMultiFactorPricersTest())
    if sys.hexversion >= 0x020100f0:
        unittest.TextTestRunner(verbosity=2).run(suite)
    else:
        unittest.TextTestRunner().run(suite)
    raw_input('press any key to continue')

if __name__ == '__main__':
    main()
