/***************************************************************************
 *            report.c
 *
 *  Sun Sep 24 12:13:36 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file report.c
	@brief Report support (GtkHTML) - incomplete.
	@author Copyright 2006, 2007  Neil Williams <linux@codehelp.co.uk>
    @author Copyright 1999  Robert Lissner, Jay MacDonald,
	@author Copyright 1999  Sam Phillips, Keith Wesolowski. 
	
	The intention is to migrate the entire report functionality
	to a combination of GtkHTML and QOF. For now, it remains borked.
*/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"

#include <math.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include <gtkhtml/gtkhtml.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-paper.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-preview.h>
#include <libgnomeprintui/gnome-print-job-preview.h>
#include "types.h"
#include "dim_list_menu.h"
#include "edit.h"
#include "filter.h"
#include "main.h"
#include "menus.h"
#include "field.h"
#include "sort.h"
#include "fileout.h"

static GtkWidget * select_dlg = NULL;
/* prototype GtkHTML output */
#define WRITETAG(x)	gtk_html_write (html, handle, (x), strlen((x)));

/** \todo migrate into the tab context 
@{
*/
static gint page_num, pages;
static GnomeFont *font;
static GtkHTML *html;
static gshort row = 0;
/** @} */
/** replaces QLFieldInfo fields[MAX_FIELDS + 1] */
static GList * thead_list = NULL;
static GtkWidget * entry1 = NULL;
/** \todo replace with dynamic allocation. */
gchar *newrow[2];
static gint widthx;
static gchar *spaces;
static gchar *pos_text;
static size_t local_len;
static GtkTextBuffer * buffer;
/** Here's a bunch of information stored for print_on_screen, for each
   column, to keep the speed up */
typedef struct
{
	gdouble group_total;
	gdouble grand_total;
	gint skip_over;
	gboolean do_group;
	gboolean error;
	gboolean do_total;
	gint16 field;
	gint16 list_col;
	gint16 justification;
	gint16 formatting;
	gint16 decimal_places;
	gint16 width;				/* when printed, put a space after */
	gint16 type;
} QlColInfo;

enum {
	REPORT_COL,
	FROM_COL,
	LAST_COL
};

/** \brief report header 

the problem here is that the old code works on columns when HTML
would need to run via rows in a <table>. However, the current GtkSheet
output is difficult to export. So, update the old column handlers 
to set the <thead> and store all the column data in a thead struct.
*/
typedef struct 
{
	QlCurrFormat curr_fmt;
	QlDateFormat date_fmt;
	QlTimeFormat time_fmt;
	/** \todo standardise and localise the format specifier for plain text. */
	gint txt_format;
	QlFieldInfo * field_info;
	/** \todo remove this placeholder when the stream is part of the tab context. */
	GtkHTMLStream * temp;
}QlTHead;

/** \brief checks to see if the file should be saved. */
static void
close_filetab (GtkWidget G_GNUC_UNUSED * w, gpointer data)
{
	gint tab_index;
	QlTabData * tab;

	tab = (QlTabData*) data;
	g_return_if_fail (tab);
	if (check_if_changed (tab))
	{
		/* ask_about_save will save or close the tab. */
		ask_about_save (tab);
		tab->file->changed = FALSE;
		dim_all_menus(tab->qlc);
		return;
	}
	tab_index = gtk_notebook_get_current_page (tab->qlc->notebook);
	gtk_notebook_remove_page (tab->qlc->notebook, tab_index);
	dim_all_menus (tab->qlc);
}

static void
close_report (GtkWidget G_GNUC_UNUSED * w, gpointer data)
{
	gint tab_index;
	QlTabData * tab;

	tab = (QlTabData*) data;
	g_return_if_fail (tab);
	tab_index = gtk_notebook_get_current_page (tab->qlc->notebook);
	gtk_notebook_remove_page (tab->qlc->notebook, tab_index);
	dim_all_menus (tab->qlc);
}

GtkWidget *
tab_label_box (QlTabData * tab, const gchar * label_text)
{
	GtkWidget * label_box, * label, * tab_close;
	GtkWidget * tab_image;
	GtkRcStyle *rcstyle;

	label_box = gtk_hbox_new (FALSE, 0);
	label = gtk_label_new (label_text);
	gtk_box_pack_start (GTK_BOX(label_box), label, FALSE, FALSE, 5);
	tab_close = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(tab_close), GTK_RELIEF_NONE);
	gtk_button_set_focus_on_click (GTK_BUTTON (tab_close), FALSE);
	/** \todo enable tooltip via context */
	rcstyle = gtk_rc_style_new ();
	rcstyle->xthickness = rcstyle->ythickness = 0; 
	gtk_widget_modify_style (tab_close, rcstyle);
	g_object_unref (rcstyle);
	tab_image = gtk_image_new_from_stock(GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
	gtk_container_add(GTK_CONTAINER(tab_close), tab_image);
	gtk_box_pack_start (GTK_BOX(label_box), tab_close, FALSE, FALSE, 0);
	gtk_widget_show_all (label_box);
	/* closing a report does not ask about save */
	if ((tab->file->file_path) && (!tab->file->report_ptr))
		g_signal_connect (tab_close, "clicked", G_CALLBACK (close_filetab), tab);
	else
		g_signal_connect (tab_close, "clicked", G_CALLBACK (close_report), tab);
	return label_box;
}

/** \brief Print each row according to the column specifiers

\param thead - the QlTHead information for this column
\param tab - the QlTabData context.

*/
static void
thead_cb (gpointer theadp, gpointer tabp)
{
	gchar * buf;
	QlTabData * tab = (QlTabData*) tabp;
	QlFieldInfo * field;
	QlTHead * thead = (QlTHead*)theadp;
	GtkHTMLStream * handle = thead->temp;

	field = ql_get_fieldinfo (tab, thead->field_info->sheet_column);
	buf = g_strconcat ("<td><b>", field->name, "</b></td>\n", NULL);
	WRITETAG(buf);
	g_free (buf);
}

static void
tbl_row_cb (gpointer theadp, gpointer tabp)
{
	gchar * buf, * tag;
	QlTabData * tab = (QlTabData*) tabp;
	QlTHead * thead = (QlTHead*)theadp;
	GtkHTMLStream * handle = thead->temp;

	buf = NULL;
	/* <table> needs data in rows */
	/** \bug buf is always empty */
	buf = gtk_sheet_cell_get_text (tab->view->sheet, row,
		thead->field_info->sheet_column);
	if (!buf)
		buf = "&nbsp;";
	tag = g_strconcat ("<td>", buf, "</td>\n", NULL);
	WRITETAG(tag);
	g_free (buf);
}

/** \brief Add a new HTML table 

 \param tab - the noteboook tab of the file view.

Creates a new tab for the report view.
*/
static void
html_on_screen (QlTabData * tab)
{
	GtkWidget *scrolled_window;
	GtkWidget *html_widget;
	gchar * buf;
	gshort c, field_num;
	GtkHTMLStream * handle = NULL;

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	html_widget = gtk_html_new ();
	html = GTK_HTML (html_widget);
	gtk_html_set_allow_frameset (html, FALSE);
	gtk_html_load_empty (html);
	gtk_html_set_editable (html, FALSE);
	gtk_container_add (GTK_CONTAINER (scrolled_window), html_widget);
	handle = gtk_html_begin_content (html, "text/html; charset=UTF-8");
	WRITETAG("<html><head>\n");
	WRITETAG("<meta name=\"Generator\" content=\"QUICKLIST\">\n");
	WRITETAG("</head><body>\n");
	buf = g_strconcat ("<h1>", tab->file->report_ptr->header, "</h1>\n", NULL);
	WRITETAG(buf);
	g_free (buf);
	buf = g_strconcat ("<h2>", tab->file->report_ptr->name, "</h2>\n", NULL);
	WRITETAG(buf);
	g_free (buf);
	for (c = 0; c < tab->file->report_ptr->last_column; c++)
	{
		QlFieldInfo * field;
		QlTHead * thead = g_new0 (QlTHead, 1);
		/** \bug QlReportColumn starts at 1, not zero. */
		field = g_new0(QlFieldInfo, 1);
		thead->field_info = field;
		thead->field_info->sheet_column = c + 1;
		field_num = tab->file->report_ptr->column[c+1].field;
		field = ql_get_fieldinfo (tab, field_num);
		thead->field_info->type = field->type;
		thead->temp = handle;
		/* not used yet - could support alternative formatting in report. */
		switch (thead->field_info->type)
		{
			case FIELD_TYPE_NUMERIC :
			{
				thead->curr_fmt = field->formatting;
				break;
			}
			case FIELD_TYPE_DATE :
			{
				thead->date_fmt = field->formatting;
				break;
			}
			case FIELD_TYPE_TIME :
			{
				thead->time_fmt = field->formatting;
				break;
			}
			case FIELD_TYPE_TEXT :
			{
				thead->txt_format = field->formatting;
				break;
			}
		}
		thead_list = g_list_append (thead_list, thead);
	}
	/* write the table headers */
	WRITETAG ("<table border=\"1\"><tr>\n");
	g_list_foreach (thead_list, thead_cb, tab);
	WRITETAG ("</tr>\n");
	/* <table> needs data in rows 
	so write HTML by column, one row at a time. */
	for (row = 0; row < tab->file->last_row; row++)
	{
		WRITETAG ("<tr>\n");
		g_list_foreach (thead_list, tbl_row_cb, tab);
		WRITETAG ("</tr>\n");
	}
	WRITETAG("</table>\n</body></html>\n");
	/** \bug data appears set but is not being displayed */
	gtk_notebook_append_page (tab->qlc->notebook, scrolled_window, 
		tab_label_box(tab, tab->file->report_ptr->name));
	gtk_widget_show_all (scrolled_window);
}

static void
print_footer (GtkHTML G_GNUC_UNUSED *fhtml, GnomePrintContext *context,
	      gdouble x, gdouble y, gdouble width, gdouble G_GNUC_UNUSED height, 
		  gpointer G_GNUC_UNUSED user_data)
{
	/* Translators: read as 'page 1 of 2' */
	gchar *text = g_strdup_printf (_("- %d of %d -"), page_num, pages);
	gdouble tw = gnome_font_get_width_utf8 (font, "text");

	if (font) {
		gnome_print_newpath     (context);
		gnome_print_setrgbcolor (context, .0, .0, .0);
		gnome_print_moveto      (context, x + (width - tw)/2, y - gnome_font_get_ascender (font));
		gnome_print_setfont     (context, font);
		gnome_print_show        (context, text);
	}
	g_free (text);
	page_num++;
}

void
print_cb (GtkWidget G_GNUC_UNUSED * widget, gpointer G_GNUC_UNUSED data)
{
	GnomePrintJob *print_master;
	GnomePrintContext *print_context;

	print_master = gnome_print_job_new (NULL);
	gtk_html_print_set_master (html, print_master);
	
	print_context = gnome_print_job_get_context (print_master);
	font = gnome_font_find_closest ("Helvetica", 12);
	page_num = 1;
	pages = gtk_html_print_get_pages_num (html, print_context,
		.0, gnome_font_get_ascender (font) - gnome_font_get_descender (font));
	gtk_html_print_with_header_footer (html, print_context,
		.0, gnome_font_get_ascender (font) - gnome_font_get_descender (font),
		NULL, print_footer, NULL);
	if (font)
		g_object_unref (font);
	gnome_print_job_render (print_master, print_context);
}

void
print_preview_cb (GtkWidget G_GNUC_UNUSED * widget, gpointer G_GNUC_UNUSED data)
{
	GnomePrintJob *print_master;
	GnomePrintContext *print_context;
	GtkWidget *preview;

	print_master = gnome_print_job_new (NULL);
	gtk_html_print_set_master (html, print_master);

	print_context = gnome_print_job_get_context (print_master);
	font = gnome_font_find_closest ("Helvetica", 12);

	page_num = 1;
	pages = gtk_html_print_get_pages_num (html, print_context,
		.0, gnome_font_get_ascender (font) - gnome_font_get_descender (font));
	gtk_html_print_with_header_footer (html, print_context,
		.0, gnome_font_get_ascender (font) - gnome_font_get_descender (font),
		NULL, print_footer, NULL);
	if (font)
		g_object_unref (font);
	
	gnome_print_job_close (print_master);
	preview = gnome_print_job_preview_new (print_master, "Print Preview");
	gtk_widget_show (preview);

	g_object_unref (print_master);
}

/** \todo remove old report_set_col_field code */

/* Sets current column and current field. */
static void
report_set_col_field (QlTabData * tab, gint col)
{
	tab->view->report_sel_col = col;
	tab->view->report_sel_field = tab->file->report_ptr->column[col].field;
}

static void
display_column (QlTabData * tab, gint colx)
{
	gint fieldx;
	gint rowx;
	gint list_mode_col;
	gchar *text;
	QlReportColumn *this_column;
	QlFieldInfo * field;

	this_column = &tab->file->report_ptr->column[colx];

	fieldx = this_column->field;
	field = ql_get_fieldinfo (tab, fieldx);
	list_mode_col = field->sheet_column;
	gtk_sheet_set_column_width (tab->view->report_sheet,
		colx, this_column->width * 8);
	gtk_sheet_column_button_add_label (tab->view->report_sheet,
		colx, field->name);
	gtk_sheet_column_set_justification (tab->view->report_sheet,
		colx, field->justification);

	/* now display the six rows that we are showing as examples */
	if (this_column->group)
		gtk_sheet_set_cell_text (tab->view->report_sheet,
			0, colx, _("Group"));
	else
		gtk_sheet_cell_clear (tab->view->report_sheet, 0, colx);

	if (this_column->total)
		gtk_sheet_set_cell_text (tab->view->report_sheet, 1, colx,
			_("Total"));
	else
		gtk_sheet_cell_clear (tab->view->report_sheet, 1, colx);

	/* insert the up to six rows of data */
	for (rowx = 2; rowx < 8; rowx++)
	{
		gtk_sheet_cell_clear (tab->view->report_sheet, rowx, colx);
		if (tab->view->adjust_rows[rowx] >= 0)
		{
			text = gtk_sheet_cell_get_text (tab->view->sheet,
				tab->view->adjust_rows[rowx], list_mode_col);
			if (text)
				gtk_sheet_set_cell_text (tab->view->report_sheet,
					rowx, colx, text);
		}
	}
}

static void
display_all_columns (QlTabData * tab)
{
	gint colx;
	big_draw_start (tab);
	for (colx = 0; colx <= tab->file->report_ptr->last_column; colx++)
		display_column (tab, colx);
	big_draw_end (tab);
}								/* end of display_all_columns */

static void
new_report_column_width (GtkSheet G_GNUC_UNUSED * sheet, gint col, 
						 gint width, gpointer data)
{
	QlTabData * tab;

	tab = (QlTabData*)data;
	front_is_changed (tab);
	/* gotta have some sort of limit.  Paper is finite */
	if (width > 640)
		tab->file->report_ptr->column[col].width = 80;
	else
		tab->file->report_ptr->column[col].width = width / 8;
}

/*  Adjust_report displays six rows of data.  They must match
	the current filter. Returns number of rows found */
static gint
pick6rows (QlTabData * tab)
{
	gint hits = 0;
	gint rowx, torow, fromrow;

	/* start out in case all rows are shown */
	for (rowx = 0; rowx < 6; rowx++)
	{
		if (rowx <= tab->file->last_row)
			tab->view->adjust_rows[rowx + 2] = rowx;
		else
			tab->view->adjust_rows[rowx + 2] = 0;
	}

	torow = fromrow = 0;
	do
	{
		if (row_is_visible (tab, fromrow))
		{
			hits++;
			tab->view->adjust_rows[torow + 2] = fromrow;
			torow++;
		}
	}
	while (++fromrow <= tab->file->last_row && torow < 6);
	for (; torow < 6; torow++)
		tab->view->adjust_rows[torow + 2] = -1;
	if (!hits)
		level1_error (tab, _("No rows match the current filter"));
	return hits;
}								/* End of pick6 rows */

/* Get rid of this report */
static void
pos_delete (GtkObject G_GNUC_UNUSED * object, gpointer data)
{
	gint reportx;
	QlTabData * tab = (QlTabData*) data;

	/* move all others down one spot.  There may be none to move */
	/** \todo move to a GList and do this properly! */
	for (reportx = tab->file->report_no; reportx < tab->file->report_ct - 1;
		reportx++)
		memcpy (&tab->file->reports[reportx], &tab->file->reports[reportx + 1],
			sizeof (QlReportInfo));
	tab->file->report_ct--;
}

/*  Save the dimensions of the print on screen window */
static void
pos_get_dim (QlTabData * tab)
{
	tab->file->report_ptr->width =
		GTK_WIDGET (tab->view->report_win)->allocation.width;
	tab->file->report_ptr->height =
		GTK_WIDGET (tab->view->report_win)->allocation.height;
}

/*  Clicked in print_on_screen, go to list mode */
static void
pos_list_mode (GtkObject G_GNUC_UNUSED * object, gpointer data)
{
	QlTabData * tab = (QlTabData*)data;
	pos_get_dim (tab);
	dim_all_menus (tab->qlc);
}

/* Just put one cell on the screen, justified properly */
void
pos_justify (gint jx)
{
	GtkTextIter justify_iter;

	gint half;
	if (jx == GTK_JUSTIFY_LEFT)
	{
		gtk_text_buffer_get_end_iter (buffer, &justify_iter);
		gtk_text_buffer_insert (buffer, &justify_iter, pos_text, local_len);
		gtk_text_buffer_insert (buffer, &justify_iter, spaces, 
			widthx - local_len);
		return;
	}

	if (jx == GTK_JUSTIFY_RIGHT)
	{
		gtk_text_buffer_get_end_iter (buffer, &justify_iter);
		gtk_text_buffer_insert (buffer, &justify_iter, spaces, 
			widthx - local_len);
		gtk_text_buffer_insert (buffer, &justify_iter, pos_text, 
			local_len);
		return;
	}

	/* oh fun, what's left is center */
	half = (widthx - local_len) / 2;
	gtk_text_buffer_get_end_iter (buffer, &justify_iter);
	gtk_text_buffer_insert (buffer, &justify_iter, spaces, half);
	gtk_text_buffer_insert (buffer, &justify_iter, pos_text, local_len);
	gtk_text_buffer_insert (buffer, &justify_iter, spaces, 
		widthx - local_len - half);
}								/* end of pos_justify */

static gboolean
report_activate (GtkSheet G_GNUC_UNUSED * sheet, gint rrow, gint col, gpointer data)
{
	QlTabData * tab;

	tab = (QlTabData*)data;
	g_return_val_if_fail (tab, FALSE);
	if (rrow < 0 || col < 0)
		return TRUE;			/* row or column buttons */
	report_set_col_field (tab, col);
	dim_all_menus (tab->qlc);
	return TRUE;
}

static void
select_report_range (GtkSheet G_GNUC_UNUSED * sheet, GtkSheetRange * range, 
					 gpointer data)
{
	QlTabData * tab;
	GtkSheetRange local_range;

	tab = (QlTabData*)data;
	local_range = *range;
	if (local_range.col0 == local_range.coli)
	{
		report_set_col_field (tab, local_range.col0);
		dim_all_menus (tab->qlc);
	}
/** \bug #456095 setting negative value in array is impossible.
 This cannot be a sane way of unsetting the selection.
	else
		report_set_col_field (tab, -1);	*//* no selection */
}

static void
sort_change (QlTabData * tab, GtkObject G_GNUC_UNUSED * object, gpointer entry)
{
	tab->file->report_ptr->sort = GPOINTER_TO_INT (entry) - 1;
	if (!entry)
		return;					/* -1  means none */
	sort_do_it (tab, tab->file->report_ptr->sort);
	pick6rows (tab);
	display_all_columns (tab);
}

static void
filter_change (QlTabData * tab, GtkObject G_GNUC_UNUSED * object, 
			   gpointer entry)
{
	tab->file->report_ptr->filter = GPOINTER_TO_INT (entry) - 1;
	filter_do_apply (tab, tab->file->report_ptr->filter);
	pick6rows (tab);
	display_all_columns (tab);
}

/* Build the sort menu button for the report window */
static GtkWidget *
sort_filter_menu (QlTabData * tab)
{
/*	GtkTreeView * treeview;
	GtkTreeModel * model;
	GtkTreeStore * treestore;
	GtkTreeIter parent_iter, child_iter;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSortable * sort;
	GtkTreeSelection * select;
*/
	/** \todo create a drop-down toolbar button + suitable menu? */
//	GtkWidget *hbox2;
	GtkWidget *menu, *menu_item, *name_label;
	gint sortx;
	gint filterx;
//  GtkWidget *sort_button;
//  GtkWidget *filter_button;
/*
	make_basic_dialog1 ();

	gtk_window_set_title (GTK_WINDOW (dialog1_win),
		_("New report"));
	top_vbox = GTK_DIALOG (dialog1_win)->vbox;
	gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
	hbox6 = GTK_DIALOG (dialog1_win)->action_area;
	hbox2 = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);
*/
	menu = gtk_menu_new ();
	menu_item = gtk_menu_item_new_with_label (_("-Do not sort-"));
	gtk_widget_show (menu_item);
	g_signal_connect (GTK_OBJECT (menu_item),
		"activate", G_CALLBACK (sort_change), GINT_TO_POINTER (0));
	gtk_menu_attach (GTK_MENU (menu), menu_item, 0, 1, 0, 1);
	/* add the individual sorts to the menu */
	for (sortx = 0; sortx < tab->file->sort_ct; sortx++)
	{
		menu_item =
			gtk_menu_item_new_with_label (tab->file->sorts[sortx].name);
		gtk_widget_show (menu_item);
		{
			/* workaround */
			gint t;
			t = sortx + 1;
			g_signal_connect (GTK_OBJECT (menu_item), "activate", 
				G_CALLBACK (sort_change), GINT_TO_POINTER (t));
		}
		gtk_menu_attach (GTK_MENU (menu), menu_item, 0, 1, 0, 1);
	}

/*	sort_button = gtk_option_menu_new ();
   gtk_option_menu_set_menu (GTK_OPTION_MENU (sort_button), menu);
   gtk_option_menu_set_history (GTK_OPTION_MENU (sort_button), 
				front->report_ptr->sort + 1);
   hbox3 = gtk_hbox_new (FALSE, 5);
*/
	/* Pack Widgets into boxes */
	name_label = gtk_label_new (_("Sort rule: "));
/*	gtk_box_pack_start (GTK_BOX (hbox3), name_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox3), sort_button, FALSE, FALSE, 0);
*/
	menu = gtk_menu_new ();
	menu_item = gtk_menu_item_new_with_label (_("-Print all rows-"));
	gtk_widget_show (menu_item);
	g_signal_connect (GTK_OBJECT (menu_item), "activate", 
		G_CALLBACK (filter_change), GINT_TO_POINTER (0));
	gtk_menu_attach (GTK_MENU (menu), menu_item, 0, 1, 0, 1);

	/* add the individual filters to the menu */
	for (filterx = 0; filterx < tab->file->filter_ct; filterx++)
	{
		menu_item = gtk_menu_item_new_with_label (tab->file->filters
			[filterx].name);
		gtk_widget_show (menu_item);
		{
			/* workaround */
			gint t;
			t = filterx + 1;
			g_signal_connect (GTK_OBJECT (menu_item), "activate", 
				G_CALLBACK (filter_change), GINT_TO_POINTER (t));
		}
		gtk_menu_attach (GTK_MENU (menu), menu_item, 0, 1, 0, 1);
	}

/*	filter_button = gtk_option_menu_new ();
   gtk_option_menu_set_menu (GTK_OPTION_MENU (filter_button), menu);
   gtk_option_menu_set_history (GTK_OPTION_MENU (filter_button), 
					front->report_ptr->filter + 1);
*//* Pack Widgets into boxes */
	name_label = gtk_label_new (_("Filter: "));
/*	gtk_box_pack_end (GTK_BOX (hbox3), filter_button, FALSE, FALSE, 0);
   gtk_box_pack_end (GTK_BOX (hbox3), name_label, FALSE, FALSE, 0);
   return (hbox3);
*/
	return NULL;
}								/* end of sort_filter_menu () */

/*  Preview clicked while adjusting report format  */
static void
adjust_preview (GtkObject G_GNUC_UNUSED * object, gpointer data)
{
	gchar *text;
	QlTabData * tab;

	tab = (QlTabData*) data;
	text = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry1)));
	if (check_entry (tab, text))
	{
		g_free (text);
		return;
	}
	tab->file->report_ptr->name = g_strdup(text);
	g_free (text);
	text = g_strdup (gtk_entry_get_text (GTK_ENTRY (tab->view->entry2)));
	if (check_entry (tab, text))
	{
		g_free (text);
		return;
	}
	tab->file->report_ptr->header = g_strdup(text);
	g_free (text);
	text = g_strdup (gtk_entry_get_text (GTK_ENTRY (tab->view->entry3)));
	if (check_entry (tab, text))
	{
		g_free (text);
		return;
	}
	tab->file->report_ptr->footer = g_strdup(text);
	g_free (text);
	html_on_screen (tab);
}

/*   Initialise the report mode window, fill it up and display it */
static void
adjust_report (QlTabData * tab)
{
	gchar *temp_file_name;
	gint rowx;
	GtkWidget *vbox;
	GtkWidget *menu_bar;
	GtkWidget *hbox1;
	GtkWidget *name_label;

	/* treeviews and header */
	GtkWidget *hbox2;
	/* for basic */
	GtkWidget *header_label;
	/* sort and filter menus */
	GtkWidget *hbox3;

	GtkWidget *scrolled_window;
	/* Footer */
	GtkWidget *hbox5;
	GtkWidget *footer_label;
	/* same as action area */
	GtkWidget *hbox6;
	GtkWidget *preview_button;
	GtkWidget *edit_list_button;
	GtkWidget *delete_button;

	pick6rows (tab);				/* find six rows to display */

	/* now make a window for editing this report format */
	tab->view->report_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_container_set_border_width (GTK_CONTAINER (tab->view->report_win), 5);
	tab->view->display_mode = DISPLAY_REPORT;
	tab->view->report_mode = DISPLAY_EDIT;
	temp_file_name = g_strconcat (tab->file->file_name, "  (", 
			_("Edit Report Layout"), ")", NULL);
	gtk_window_set_title (GTK_WINDOW (tab->view->report_win), temp_file_name);
	g_free (temp_file_name);

	gtk_window_set_resizable (GTK_WINDOW (tab->view->report_win), TRUE);
//	set_window_size_loc (tab->view->report_win);

	vbox = gtk_vbox_new (FALSE, 3);
	menu_bar = build_report_mb (tab->qlc);
	gtk_box_pack_start (GTK_BOX (vbox), menu_bar, FALSE, TRUE, 0);

	/* Show and allow changes to report name */
	hbox1 = gtk_hbox_new (FALSE, 5);
	name_label = gtk_label_new (_("Report Name"));
	gtk_box_pack_start (GTK_BOX (hbox1), name_label, FALSE, FALSE, 0);

	entry1 = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (hbox1), entry1, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox1, FALSE, FALSE, 0);
	gtk_entry_set_text (GTK_ENTRY (entry1),
		tab->file->report_ptr->name);

	/* Show and allow changes to report header */
	hbox2 = gtk_hbox_new (FALSE, 5);
	header_label = gtk_label_new (_("Print at top of page"));
	gtk_box_pack_start (GTK_BOX (hbox2), header_label, FALSE, FALSE, 0);

	tab->view->entry2 = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (tab->view->entry2), 80);
	gtk_box_pack_start (GTK_BOX (hbox2), tab->view->entry2, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);
	gtk_entry_set_text (GTK_ENTRY (tab->view->entry2),
		tab->file->report_ptr->header);

	/* build menus for sorts and filters */
	hbox3 = sort_filter_menu (tab);
	gtk_box_pack_start (GTK_BOX (vbox), hbox3, FALSE, FALSE, 0);

	/* make sheet and add to scroll_win */
	tab->view->report_sheet = GTK_SHEET (gtk_sheet_new_browser (8, 
		tab->file->report_ptr->last_column + 1, temp_file_name));
	report_set_col_field (tab, 0);	/* cursor starts here */
	gtk_sheet_row_button_add_label (tab->view->report_sheet,
		0, _("Group"));
	gtk_sheet_row_button_add_label (tab->view->report_sheet,
		1, _("Total"));
	for (rowx = 2; rowx < 8; rowx++)
		gtk_sheet_row_button_add_label (tab->view->report_sheet,
			rowx, _("Sample"));
	gtk_sheet_set_autoscroll (tab->view->report_sheet, TRUE);
	gtk_sheet_set_autoresize (tab->view->report_sheet, FALSE);
	gtk_sheet_set_row_titles_width (tab->view->report_sheet, 60);
	gtk_container_set_border_width (GTK_CONTAINER (tab->view->report_sheet),
		4);

	/* set sensitivity for all column and row buttons */
	gtk_sheet_show_row_titles (tab->view->report_sheet);
	gtk_sheet_show_column_titles (tab->view->report_sheet);
	gtk_sheet_columns_set_sensitivity (tab->view->report_sheet, TRUE);
	gtk_sheet_rows_set_sensitivity (tab->view->report_sheet, FALSE);
	display_all_columns (tab);

	/* create a new scrolled window. */
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);

	gtk_container_set_border_width (GTK_CONTAINER (scrolled_window), 3);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW
		(scrolled_window), GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);

	gtk_container_add (GTK_CONTAINER (scrolled_window),
		GTK_WIDGET (tab->view->report_sheet));
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);

	/* Show and allow changes to report footer */
	hbox5 = gtk_hbox_new (FALSE, 5);
	footer_label = gtk_label_new (_("Print at bottom of page"));
	gtk_box_pack_start (GTK_BOX (hbox5), footer_label, FALSE, FALSE, 0);

	tab->view->entry3 = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (tab->view->entry3), 80);
	gtk_box_pack_start (GTK_BOX (hbox5), tab->view->entry3, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox5, FALSE, FALSE, 0);
	gtk_entry_set_text (GTK_ENTRY (tab->view->entry3),
		tab->file->report_ptr->footer);

	/* Now do Preview,  OK  and Cancel buttons */
	hbox6 = gtk_hbox_new (TRUE, 5);
	preview_button = gtk_button_new_from_stock (GTK_STOCK_PRINT_PREVIEW);
	gtk_box_pack_start (GTK_BOX (hbox6), preview_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (preview_button), "clicked",
		G_CALLBACK (adjust_preview), tab);

	edit_list_button = gtk_button_new_from_stock (GTK_STOCK_EDIT);
	gtk_box_pack_start (GTK_BOX (hbox6), edit_list_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (edit_list_button), "clicked",
		G_CALLBACK (pos_list_mode), tab);

	delete_button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_box_pack_start (GTK_BOX (hbox6), delete_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (delete_button), "clicked",
		G_CALLBACK (pos_delete), tab);

	gtk_box_pack_start (GTK_BOX (vbox), hbox6, FALSE, FALSE, 0);

	gtk_container_add (GTK_CONTAINER (tab->view->report_win), vbox);
	dim_all_menus (tab->qlc);
	g_signal_connect (GTK_OBJECT (tab->view->report_win), "delete_event",
		G_CALLBACK (pos_list_mode), tab);
/*	g_signal_connect (GTK_OBJECT (front->report_win), "focus-in-event",
		G_CALLBACK (focus_in_event_cb), NULL);*/
	g_signal_connect (GTK_OBJECT (tab->view->report_sheet), 
		"new_column_width",	G_CALLBACK (new_report_column_width), tab);
	g_signal_connect (GTK_OBJECT (tab->view->report_sheet),
		"select_range", G_CALLBACK (select_report_range), tab);
	g_signal_connect (GTK_OBJECT (tab->view->report_sheet),
		"activate", G_CALLBACK (report_activate), tab);
	gtk_widget_show_all (tab->view->report_win);
}								/* end of build_basic_report_mode */

/*  OK clicked setting up a list of report columns */
static void
add_ok (GtkWidget G_GNUC_UNUSED * w, gpointer data)
{
	gchar *text;
	QlTabData * tab;

	tab = (QlTabData *)g_object_get_data (G_OBJECT(data), QLTABID);
	g_return_if_fail (tab);
	/** \bug tab->view is not being created. */
	if (strcmp(gtk_entry_get_text (GTK_ENTRY (entry1)), "") == 0)
		return;
	text = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry1)));
	if (check_entry (tab, text))
	{
		g_free (text);
		return;
	}
	front_is_changed (tab);
	strcpy (tab->file->report_ptr->name, text);
	adjust_report (tab);
	g_free (text);
	gtk_widget_destroy (GTK_WIDGET(data));
}

/* An item clicked while adding a column   */
static G_GNUC_UNUSED void
col_from_clicked (GtkWidget G_GNUC_UNUSED * widget, gint G_GNUC_UNUSED rrow,
	gint G_GNUC_UNUSED col, GdkEventButton G_GNUC_UNUSED * event, gpointer G_GNUC_UNUSED data)
{
/*
	QLReportColumn *this_column;
	gint colx, col_spot, col_last;
	gint fieldx;
	if (event->type != GDK_2BUTTON_PRESS)
	return;
	  
	col_spot = tab->view->report_sel_col;
	col_last = tab->file->report_ptr->last_column;
	if (col_last >= MAX_FIELDS - 1)
	{
		level2_error (_("You are limited to 30 columns on a report."));
	return;
  }

	for (colx = col_last; colx >= col_spot; colx--)
		tab->file->report_ptr->column[colx + 1] =
			tab->file->report_ptr->column[colx];

	this_column = &tab->file->report_ptr->column[col_spot];
	fieldx = tab->file->col_to_field[row];
	newrow[0] = tab->file->fields[fieldx].name;

	this_column->field = fieldx;
	this_column->width = tab->file->fields[fieldx].width;
	this_column->group = 0;
	this_column->total = 0;

	gtk_sheet_insert_columns (tab->view->report_sheet, col_spot, 1);
	display_column (col_spot);
	tab->file->report_ptr->last_column++;
*/
}

/*  From treeview selected.  This adds one column to a new report layout  */
static G_GNUC_UNUSED void
from_clicked (GtkWidget G_GNUC_UNUSED * widget, gint G_GNUC_UNUSED rrow,
	gint G_GNUC_UNUSED col, GdkEventButton G_GNUC_UNUSED * event, gpointer G_GNUC_UNUSED data)
{
/*	QLReportColumn *this_column;
	gint colx;
	gint fieldx;  
	if (event->type != GDK_2BUTTON_PRESS)
		return;
      
	if (tab->file->report_ptr->last_column >= MAX_FIELDS - 1)
	level2_error (_("You are limited to 30 columns on a report."));
	colx = ++tab->file->report_ptr->last_column;
	this_column = &tab->file->report_ptr->column[colx];
	fieldx = tab->file->col_to_field[row];
	newrow[0] = tab->file->fields[fieldx].name;

	this_column->field = fieldx;
	this_column->width = tab->file->fields[fieldx].width;
	gtk_widget_set_sensitive (ok_button, TRUE);*/
}

static void
report_dlg_cancel (GtkWidget G_GNUC_UNUSED * w, gpointer data)
{
	gtk_widget_destroy (GTK_WIDGET(data));
}

static void
report_add_cancel (GtkWidget G_GNUC_UNUSED * w, gpointer data)
{
	QlTabData * tab;

	g_return_if_fail (data);
	tab = g_object_get_data (G_OBJECT(data), QLTABID);
	tab->file->report_ct--;
	gtk_widget_destroy (data);
}

/* This creates a new report format */
void
report_add (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gint colx, fieldx;
	guint width;
	GtkWidget * report_dlg;
	GtkWidget *top_vbox;
	GtkWidget *hbox1;
	GtkWidget *name_label;
	GtkWidget *scrwin_to;
	GtkWidget *hbox2;			/* treeview and header */
	GtkTreeView * treeview;
	GtkTreeModel * model;
	GtkTreeStore * treestore;
	GtkTreeIter parent_iter, child_iter;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSortable * sort;
	GtkTreeSelection * rselect;
	GtkWidget *hbox6;			/* same as action area */
	GtkWidget *cancel_button, *ok_button;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	g_return_if_fail (tab);
	width = 0;
	CHECK_CHANGED(tab);
	tab->file->report_no = tab->file->report_ct++;
	tab->file->report_ptr = &tab->file->reports[tab->file->report_no];

	tab->file->report_ptr = g_new0 (QlReportInfo, 1);
	tab->file->report_ptr->last_column = -1;
	tab->file->report_ptr->sort = -1;
	tab->file->report_ptr->filter = -1;

	report_dlg = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (report_dlg), TRUE);
	gtk_window_set_position (GTK_WINDOW (report_dlg), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (report_dlg), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (report_dlg), 5);

	gtk_window_set_title (GTK_WINDOW (report_dlg),
		_("New report"));
	top_vbox = GTK_DIALOG (report_dlg)->vbox;
	gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
	hbox6 = GTK_DIALOG (report_dlg)->action_area;

	/* allow for report name */
	hbox1 = gtk_hbox_new (FALSE, 5);
	name_label = gtk_label_new (_("Please enter a report name"));
	gtk_box_pack_start (GTK_BOX (hbox1), name_label, FALSE, FALSE, 0);

	entry1 = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (hbox1), entry1, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (top_vbox), hbox1, FALSE, FALSE, 0);

	/* now the area containing treeview */
	/*  The horizontal box containing the from and to listboxes  */
	hbox2 = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);
	/* The from (column names) treeview */
	treeview = GTK_TREE_VIEW (gtk_tree_view_new ());
	model = gtk_tree_view_get_model (treeview);
	treestore = gtk_tree_store_new (SINGLE_COL, G_TYPE_STRING);

	sort = GTK_TREE_SORTABLE (treestore);
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), 
		GTK_TREE_MODEL (sort));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_reorderable (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_store_append (treestore, &parent_iter, NULL);

	gtk_tree_store_set (treestore, &parent_iter, REPORT_COL,
		_("Available columns"), -1);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Column Name"), renderer, "text", REPORT_COL, NULL);
	gtk_tree_view_column_set_sort_column_id (column, REPORT_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	rselect = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (rselect, GTK_SELECTION_SINGLE);
	scrwin_to = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_to),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_container_set_border_width (GTK_CONTAINER (scrwin_to), 5);
	gtk_box_pack_start (GTK_BOX (hbox2), scrwin_to, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (scrwin_to), 
		GTK_WIDGET(treeview));
	g_object_set_data (G_OBJECT(report_dlg), QLTABID, tab);

	ok_button = gtk_button_new_from_stock (GTK_STOCK_OK);
	gtk_box_pack_start (GTK_BOX (hbox6), ok_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (ok_button), "clicked",
		G_CALLBACK (add_ok), report_dlg);

	cancel_button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_box_pack_start (GTK_BOX (hbox6), cancel_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		G_CALLBACK (report_add_cancel), report_dlg);
	g_signal_connect (GTK_OBJECT (report_dlg),
		"delete_event", G_CALLBACK (report_add_cancel), report_dlg);
	/* now populate the tree of column names */
	newrow[1] = NULL;
	for (colx = 0; colx <= tab->file->last_field; colx++)
	{
		QlFieldInfo * field;
		fieldx = tab->file->col_to_field[colx];
		field = ql_get_fieldinfo (tab, fieldx);
		newrow[0] = field->name;
		gtk_tree_store_append (treestore, &child_iter, &parent_iter);
		gtk_tree_store_set (treestore, &child_iter, REPORT_COL, 
			newrow[0], -1);
		width = (strlen(newrow[0]) > width) ? 
			strlen(newrow[0]) : width;
	}
	gtk_widget_set_size_request (GTK_WIDGET(treeview), 
		width + 200, 200);
	gtk_tree_view_expand_all (GTK_TREE_VIEW(treeview));
	gtk_widget_show_all (report_dlg);
}								/* End of Report_add */

/* Make a treeview of fields names that can be added to report */
void
report_col_add (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint colx, fieldx;
	guint width;
	GtkWidget *top_vbox;
	GtkWidget *hbox2;			/* treeview and header */
	GtkTreeView * treeview;
	GtkTreeModel * model;
	GtkTreeStore * treestore;
	GtkTreeIter parent_iter, child_iter;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSortable * sort;
	GtkTreeSelection * rselect;
	GtkWidget *hbox6;			/* same as action area */
	GtkWidget *done_button;
	GtkWidget * report_dlg;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	width = 0;
	report_dlg = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (report_dlg), TRUE);
	gtk_window_set_position (GTK_WINDOW (report_dlg), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (report_dlg), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (report_dlg), 5);

	gtk_window_set_title (GTK_WINDOW (report_dlg),
		_("Add columns to report"));
	top_vbox = GTK_DIALOG (report_dlg)->vbox;
	gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
	hbox6 = GTK_DIALOG (report_dlg)->action_area;

	/* now the area containing tree of fields that can be added */
	/*  The horizontal box containing the from and to listboxes  */
	hbox2 = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);

	/* The from (column names) treeview */
	/** \todo same as below, abstract this routine */
	treeview = GTK_TREE_VIEW (gtk_tree_view_new ());
	model = gtk_tree_view_get_model (treeview);
	treestore = gtk_tree_store_new (SINGLE_COL, G_TYPE_STRING);
	sort = GTK_TREE_SORTABLE (treestore);
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), 
		GTK_TREE_MODEL (sort));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_reorderable (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_store_append (treestore, &parent_iter, NULL);
	gtk_tree_store_set (treestore, &parent_iter, FROM_COL,
		_("Available columns"), -1);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Column Name"), renderer, "text", FROM_COL, NULL);
	gtk_tree_view_column_set_sort_column_id (column, FROM_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	rselect = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (rselect, GTK_SELECTION_SINGLE);

	/* Now do OK, Done and Cancel buttons */
	done_button = gtk_button_new_from_stock (GTK_STOCK_APPLY);
	gtk_box_pack_start (GTK_BOX (hbox6), done_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (done_button), "clicked",
		G_CALLBACK (gtk_widget_destroy), report_dlg);

	g_signal_connect (GTK_OBJECT (report_dlg),
		"delete_event", G_CALLBACK (report_dlg_cancel), report_dlg);
	/* now populate the tree of column names */
	newrow[1] = NULL;
	for (colx = 0; colx <= tab->file->last_field; colx++)
	{
		QlFieldInfo * field;
		/** \todo use a callback here */
		fieldx = tab->file->col_to_field[colx];
		field = ql_get_fieldinfo (tab, fieldx);
		newrow[0] = field->name;
		gtk_tree_store_append (treestore, &child_iter, &parent_iter);
		gtk_tree_store_set (treestore, &child_iter, FROM_COL, 
			newrow[0], -1);
		width = (strlen(newrow[0]) > width) ? 
			strlen(newrow[0]) : width;
	}
	gtk_widget_set_size_request (GTK_WIDGET(treeview), 
		width + 200, 200);
	gtk_tree_view_expand_all (GTK_TREE_VIEW(treeview));

	gtk_widget_show_all (report_dlg);
}								/* End of Report_col_add */

static void
swap_columns (QlTabData * tab, gint from, gint to)
{
	QlReportColumn temp_column;

	temp_column = tab->file->report_ptr->column[from];
	tab->file->report_ptr->column[from] = tab->file->report_ptr->column[to];
	tab->file->report_ptr->column[to] = temp_column;

	display_column (tab, from);
	display_column (tab, to);
	report_set_col_field (tab, to);
	gtk_sheet_select_column (tab->view->report_sheet, to);
	dim_all_menus (tab->qlc);
}

void
report_col_left (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	swap_columns (tab, tab->view->report_sel_col, tab->view->report_sel_col - 1);
}

void
report_col_right (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	swap_columns (tab, tab->view->report_sel_col, tab->view->report_sel_col + 1);
}

/* Move the selection to the given point in report sheet */
void
report_go_to (QlTabData * tab, gint row0, gint rowi, gint col0, gint coli)
{
	GtkSheetRange range;
	range.col0 = col0;
	range.coli = coli;
	range.row0 = row0;
	range.rowi = rowi;
	gtk_sheet_select_range (tab->view->report_sheet, &range);
}

void
report_col_del (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint colx;
	gint this_col;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	this_col = tab->view->report_sel_col;
	/* someday we should have code to ask if the user really wants this */
	gtk_sheet_delete_columns (tab->view->report_sheet,
		this_col, 1);

	/** \todo use a GList instead. */
	/* remove entry from report->column by moving down all higher */
	for (colx = this_col; colx < tab->file->report_ptr->last_column; colx++)
		tab->file->report_ptr->column[colx] =
			tab->file->report_ptr->column[colx + 1];
	tab->file->report_ptr->last_column--;
	if (this_col > tab->file->report_ptr->last_column)
		this_col = tab->file->report_ptr->last_column;
	report_go_to (tab, 0, 0, this_col, this_col);
}								/* end of report_col_del */

/** \brief User clicked OK to select a specific report format, 
go print on screen */
static void
select_ok (GtkButton G_GNUC_UNUSED * button, gpointer data)
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar * report_name;
	gint reportx;
	QlTabData * tab;

	selection = GTK_TREE_SELECTION(data);
	tab = (QlTabData*)g_object_get_data (G_OBJECT(selection), QLTABID);
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		reportx = -1;
		gtk_tree_model_get (model, &iter, REPORT_COL, &report_name, -1);
		for (reportx = 0; reportx < tab->file->report_ct; reportx++)
		{
			if (0 == strcmp (tab->file->reports[reportx].name, report_name))
			{
				tab->file->report_no = reportx;
			}
		}
		if (tab->file->report_no < 0)
			return;
		tab->file->report_ptr = &tab->file->reports[tab->file->report_no];
		if (tab->file->report_ptr->sort >= 0)
			sort_do_it (tab, tab->file->report_ptr->sort);
		filter_do_apply (tab, tab->file->report_ptr->filter);
		get_window_size_loc (tab->qlc->parent);
		html_on_screen (tab);
		gtk_widget_destroy (select_dlg);
	}
}

/** \brief Offer the list of reports for the user to select
*/
void
report_select (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint reportx;
	guint width;
	GtkWidget *top_vbox;
	GtkWidget *hbox2;			/* treeview and header */
	GtkWidget *scrwin_from;
	GtkTreeStore * treestore;
	GtkTreeIter parent_iter, child_iter;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeView * treeview;
	GtkTreeModel * treemodel;
	GtkTreeSortable * sort;
	GtkTreeSelection * rselect;
	GtkWidget *hbox6;			/* same as action area */
	GtkWidget *cancel_button, * ok_button;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);

	g_return_if_fail (tab);
	CHECK_CHANGED(tab);
	select_dlg = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (select_dlg), TRUE);
	gtk_window_set_position (GTK_WINDOW (select_dlg), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (select_dlg), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (select_dlg), 5);

	width = 0;
	gtk_window_set_title (GTK_WINDOW (select_dlg), _("Select a report"));
	top_vbox = GTK_DIALOG (select_dlg)->vbox;
	gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
	hbox6 = GTK_DIALOG (select_dlg)->action_area;

	treeview = GTK_TREE_VIEW(gtk_tree_view_new ());
	treemodel = gtk_tree_view_get_model (treeview);
	/* one column, name of report */
	treestore = gtk_tree_store_new (SINGLE_COL, G_TYPE_STRING);
	sort = GTK_TREE_SORTABLE (treestore);
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), 
		GTK_TREE_MODEL (sort));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_reorderable (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_store_append (treestore, &parent_iter, NULL);
	gtk_tree_store_set (treestore, &parent_iter, REPORT_COL, 
		_("Available reports"), -1);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Report Name"), renderer, "text", REPORT_COL, NULL);
	gtk_tree_view_column_set_sort_column_id (column, REPORT_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	/*  The horizontal box containing the from and to listboxes  */
	hbox2 = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);

	/* The report names tree */
	scrwin_from = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_from),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_container_set_border_width (GTK_CONTAINER (scrwin_from), 5);
	gtk_box_pack_start (GTK_BOX (hbox2), scrwin_from, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (scrwin_from), 
		GTK_WIDGET(treeview));

	ok_button = gtk_button_new_from_stock (GTK_STOCK_OK);
	rselect = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (rselect, GTK_SELECTION_SINGLE);
	g_object_set_data (G_OBJECT(rselect), QLTABID, tab);
	g_signal_connect (G_OBJECT (ok_button), "clicked",
		G_CALLBACK (select_ok), rselect);
	gtk_window_set_resizable (GTK_WINDOW(select_dlg), TRUE);
	cancel_button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_box_pack_start (GTK_BOX (hbox6), ok_button, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox6), cancel_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		G_CALLBACK (report_dlg_cancel), select_dlg);
	g_signal_connect (GTK_OBJECT (select_dlg), "delete_event", 
		G_CALLBACK(report_dlg_cancel), select_dlg);
	/* now populate the model with report names */
	/** \todo Convert tab->file->reports to a GHashTable */
	newrow[1] = NULL;
	for (reportx = 0; reportx < tab->file->report_ct; reportx++)
	{
		newrow[0] = tab->file->reports[reportx].name;
		gtk_tree_store_append (treestore, &child_iter, &parent_iter);
		gtk_tree_store_set (treestore, &child_iter, REPORT_COL, 
			newrow[0], -1);
		width = (strlen(newrow[0]) > width) ? 
			strlen(newrow[0]) : width;
	}
	gtk_widget_set_size_request (GTK_WIDGET(treeview), 
		width + 200, 200);
	gtk_tree_view_expand_all (GTK_TREE_VIEW(treeview));
	gtk_widget_show_all (select_dlg);
}								/* End of Report_select */

/** \todo WTF? group = !group ?*/
void
report_totals_group (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	tab->file->report_ptr->column[tab->view->report_sel_col].group =
		!tab->file->report_ptr->column[tab->view->report_sel_col].group;
	display_column (tab, tab->view->report_sel_col);
}

/** \todo WTF? total = !total? */
void
report_totals_total (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	tab->file->report_ptr->column[tab->view->report_sel_col].total =
		!tab->file->report_ptr->column[tab->view->report_sel_col].total;
	display_column (tab, tab->view->report_sel_col);
}

/* To tree clicked, remove the clicked upon entry */
/** \todo revise as treeview selection handler. */
static G_GNUC_UNUSED void
to_clicked (GtkWidget G_GNUC_UNUSED * widget, gint G_GNUC_UNUSED rrow,
	gint G_GNUC_UNUSED col, GdkEventButton * event, gpointer G_GNUC_UNUSED data)
{
//	gint colx;
	if (event->type != GDK_2BUTTON_PRESS)
		return;
	
/*	for (colx = row; colx < front->report_ptr->last_column; colx++)
		front->report_ptr->column[colx] =
			front->report_ptr->column[colx + 1];
	front->report_ptr->last_column--;*/
}
