#lang typed/racket/base

#|
The constants used for argument reduction in flexp/error and flexpm1/error
|#

(require (only-in racket/math exact-floor)
         "../flonum-constants.rkt"
         "../flonum-functions.rkt")

(provide expm1-min expm1-max exp-min flexpm1-reduction)

;; Exponent base used for argument reduction
(define base (fl+ 1.0 #i1/32))
;; Bounds for which flexpm1-small/error is accurate
(define expm1-min 0.0005)
(define expm1-max (- (fllog +min.0)))
(define exp-min 0.00118)
;; Least and greatest integer exponents used in argument reduction
(define kmin (exact-floor (fl/ (fllog expm1-min) (fllog base))))
(define kmax (exact-floor (fl/ (fllog expm1-max) (fllog base))))

#|
Argument reduction for exponential:

  exp(x) = exp(base^k) * exp(x - base^k)

where base = 1 + ε is chosen to minimize *both* of the following:

 1. The number of reductions necessary to get arguments near zero
 2. The size of the table of exp(base^k)

These two goals are at odds: the closer `base' is to 1, the larger the table is, but fewer
reductions are necessary. Currently, the table has 462 entries, and it takes <= 4 reductions to get
`x' to where (+ 1 (flexpm1-small/error x)) is accurate.

The natural number `k' is chosen to reduce positive `x' as close to zero as possible without going
negative.
|#

;; Generated by (parameterize ([bf-precision 1070]) (bigfloat->fl2 (bfexpm1 (bf -650.0))))
(define-values (expm1-neg-650-hi expm1-neg-650-lo)
  (values -1.0 5.111951948651156e-283))

(: flexpm1-reduction (Flonum -> (Values Flonum Flonum Flonum)))
(define (flexpm1-reduction x)
  (cond [(x . fl> . expm1-max)  (values x 0.0 0.0)]
        [(x . fl>= . expm1-min)
         (define k (flfloor (fl/ (fllog x) (fllog base))))
         (define i (- (fl->exact-integer k) kmin))
         (define expm1-b^k (vector-ref positive-ds i))
         (define b^k (vector-ref base^ks i))
         (let*-values ([(d2 d1)  (values (car expm1-b^k) (cdr expm1-b^k))])
           (values (fl- x b^k) d2 d1))]
        [(x . fl> . (- expm1-min))  (values x 0.0 0.0)]
        [(x . fl> . -650.0)
         (define k (flfloor (fl/ (fllog (- x)) (fllog base))))
         (define i (- (fl->exact-integer k) kmin))
         (define expm1-b^k (vector-ref negative-ds i))
         (define b^k (vector-ref base^ks i))
         (let*-values ([(d2 d1)  (values (car expm1-b^k) (cdr expm1-b^k))])
           (values (fl+ x b^k) d2 d1))]
        [(x . fl> . (fllog +min.0))
         ;; This special case keeps flexpm1-reduction from returning subnormals
         (values (fl+ x 650.0) expm1-neg-650-hi expm1-neg-650-lo)]
        [else
         (values x 0.0 0.0)]))

;; The tables below were generated by the following code:
#|
(require "../../../bigfloat.rkt"
         "expansion-base.rkt")

(: base^ks (Vectorof Flonum))
;; base^k for each kmin <= k <= kmax
(define base^ks
  (build-vector (- (+ kmax 1) kmin) (λ: ([k : Index]) (flexpt base (fl (+ k kmin))))))

(: positive-ds (Vectorof (Pair Flonum Flonum)))
;; exp(base^k) - 1 for each kmin <= k <= kmax
(define positive-ds
  (build-vector (- (+ kmax 1) kmin)
                (λ: ([k : Index])
                  (parameterize ([bf-precision 256])
                    (define x (flexpt base (fl (+ k kmin))))
                    (define-values (y2 y1) (bigfloat->fl2 (bfexpm1 (bf x))))
                    (cons y2 y1)))))

(: negative-ds (Vectorof (Pair Flonum Flonum)))
;; exp(-base^k) - 1 for each kmin <= k <= kmax
(define negative-ds
  (build-vector (- (+ kmax 1) kmin)
                (λ: ([k : Index])
                  (parameterize ([bf-precision 1070])
                    (define x (flexpt base (fl (+ k kmin))))
                    (define-values (y2 y1) (bigfloat->fl2 (bfexpm1 (bf (- x)))))
                    (cons y2 y1)))))
|#

(: base^ks (Vectorof Flonum))
(define base^ks
  #(0.0004849953040935936
    0.0005001514073465185
    0.0005157811388260972
    0.0005318992994144127
    0.0005485211525211131
    0.0005656624385373979
    0.0005833393897416915
    0.0006015687456711194
    0.0006203677689733418
    0.0006397542617537588
    0.0006597465824335637
    0.0006803636631346127
    0.0007016250276075693
    0.0007235508097203059
    0.0007461617725240654
    0.0007694793279154424
    0.0007935255569128001
    0.000818323230566325
    0.0008438958315215226
    0.0008702675762565702
    0.0008974634380145881
    0.0009255091704525439
    0.000954431332029186
    0.000984257311155098
    0.0010150153521286949
    0.0010467345818827166
    0.0010794450375665515
    0.0011131776949905063
    0.0011479644979589595
    0.001183838388520177
    0.0012208333381614324
    0.0012589843799789772
    0.0012983276418533202
    0.0013389003806612365
    0.0013807410175569002
    0.0014238891743555532
    0.0014683857110541643
    0.001514272764524607
    0.001561593788416001
    0.001610393594304001
    0.001660718394126001
    0.0017126158439424385
    0.0017661350890656397
    0.0018213268105989411
    0.001878243273430158
    0.0019369383757248503
    0.001997467699966252
    0.0020598885655901972
    0.002124260083264891
    0.002190643210866919
    0.00225910081120651
    0.0023296977115567137
    0.0024025007650428608
    0.0024775789139504503
    0.002555003255011402
    0.002634847106730508
    0.0027171860788158366
    0.002802098143778831
    0.00288966371077192
    0.002979965701733542
    0.0030730896299127155
    0.003169123680847488
    0.003268158795873972
    0.0033702887582450335
    0.003475610281940191
    0.0035842231032508216
    0.00369623007522741
    0.0038117372650782666
    0.003930854054611963
    0.004053693243818586
    0.004180371157687917
    0.004311007756365664
    0.004445726748752091
    0.004584655709650594
    0.004727926200577175
    0.004875673894345212
    0.0050280387035435
    0.005185164913029235
    0.005347201316561398
    0.005514301357703941
    0.00568662327513219
    0.005864330252480071
    0.006047590572870073
    0.006236577778272263
    0.00643147083384327
    0.006632454297400873
    0.00683971849419465
    0.007053459697138233
    0.007273880312673803
    0.007501189072444859
    0.007735601230958761
    0.007977338769426223
    0.008226630605970791
    0.008483712812407378
    0.00874882883779511
    0.009022229738976206
    0.009304174418319213
    0.009594929868891688
    0.009894771427294554
    0.010203983034397509
    0.010522857504222431
    0.010851696801229382
    0.0111908123262678
    0.011540525211463669
    0.011901166624321908
    0.012273078081331969
    0.012656611771373592
    0.013052130889229017
    0.013460009979517424
    0.013880635291377343
    0.014314405144232885
    0.014761730304990162
    0.015223034377021105
    0.015698754201303015
    0.016189340270093736
    0.016695257153534163
    0.017216983939582105
    0.017755014687694047
    0.018309858896684485
    0.018882041987205875
    0.01947210579930606
    0.020080609105534372
    0.020708128140082323
    0.021355257144459894
    0.022022608930224268
    0.022710815459293775
    0.023420528442396706
    0.024152419956221603
    0.02490718307985353
    0.02568553255109895
    0.026488205443320794
    0.027315961863424568
    0.028169585671656585
    0.029049885223895854
    0.0299576941371426
    0.030893872078928304
    0.03185930558139482
    0.032854908880813403
    0.03388162478333882
    0.03494042555781816
    0.03603231385649998
    0.0371583236645156
    0.03831952127903172
    0.03951700631900146
    0.04075191276647025
    0.04202541004042245
    0.043338704104185645
    0.04469303860744145
    0.046089696063924
    0.04752999906592162
    0.04901531153673167
    0.05054704002225453
    0.05212663502294999
    0.05375559236741718
    0.05543545462889896
    0.05716781258605205
    0.058954306729366183
    0.060796628814658873
    0.06269652346511696
    0.06465578982340187
    0.06667628325538318
    0.0687599171071139
    0.07090866451671121
    0.07312456028285844
    0.07540970279169776
    0.07776625600393831
    0.08019645150406139
    0.0827025906135633
    0.08528704657023717
    0.08795226677555708
    0.09070077511229323
    0.0935351743345524
    0.09645814853250716
    0.09947246567414801
    0.10258098022646514
    0.10578663585854217
    0.10909246822912161
    0.11250160786128166
    0.11601728310694671
    0.1196428232040388
    0.12338166142916501
    0.12723733834882642
    0.13121350517222724
    0.13531392720885935
    0.1395424874341362
    0.14390319016645295
    0.1484001648591546
    0.15303767001100319
    0.15782009719884704
    0.16275197523631102
    0.16783797446244572
    0.17308291116439717
    0.17849175213828458
    0.18406961939260597
    0.1898217949986249
    0.19575372609233194
    0.2018710300327173
    0.20817949972123972
    0.21468510908752847
    0.22139401874651374
    0.22831258183234227
    0.23544735001460299
    0.24280507970255932
    0.2503927384432643
    0.2582175115196163
    0.2662868087546043
    0.2746082715281857
    0.2831897800134415
    0.29203946063886155
    0.30116569378382596
    0.3105771217145705
    0.3202826567681509
    0.33029148979215556
    0.34061309884816043
    0.35125725818716547
    0.36223404750551436
    0.3735538614900617
    0.3852274196616261
    0.39726577652605194
    0.40968033204249105
    0.4224828424188189
    0.435685431244407
    0.4493006009707947
    0.46334124475113203
    0.47782065864960493
    0.4927525542324051
    0.5081510715521678
    0.524030792538173
    0.5404067548049909
    0.5572944658926469
    0.574709917951792
    0.5926696028877856
    0.6111905279780289
    0.6302902319773422
    0.6499868017266343
    0.6702988892805916
    0.69124572957061
    0.7128471586196916
    0.735123632326557
    0.7580962458367618
    0.7817867535191607
    0.8062175895666345
    0.8314118892405917
    0.8573935107793602
    0.8841870579912152
    0.9118179035534407
    0.9403122130394858
    0.9696969696969697
    1.0
    1.03125
    1.0634765625
    1.096710205078125
    1.1309823989868164
    1.1663255989551544
    1.202773273922503
    1.2403599387325812
    1.2791211868179744
    1.319093723906036
    1.3603154027780997
    1.4028252591149153
    1.4466635484622565
    1.491871784351702
    1.5384927776126927
    1.5865706769130892
    1.6361510105666233
    1.6872807296468302
    1.7400082524482938
    1.794383510337303
    1.8504579950353437
    1.9082848073801981
    1.9679187076108293
    2.0294161672236677
    2.0928354224494075
    2.158236529400951
    2.225681420944731
    2.295233965349254
    2.366960026766418
    2.440927527602869
    2.5172065128404584
    2.5958692163667227
    2.676990129378183
    2.760646070921251
    2.8469162606375398
    2.935882393782463
    3.027628718588165
    3.1222421160440454
    3.2198121821704215
    3.3204313128632474
    3.424194791390224
    3.5312008786211684
    3.6415509060780797
    3.7553493718930198
    3.8727040397646766
    3.993726041007323
    4.118529979788802
    4.247234041657202
    4.37996010545899
    4.516833858754583
    4.657984916840664
    4.803546945491934
    4.953657787538557
    5.108459593399137
    5.26809895569286
    5.432727048058262
    5.602499768310082
    5.777577886069773
    5.958127195009453
    6.144318669853498
    6.33632862828642
    6.534338897920371
    6.738536988480383
    6.9491162693703945
    7.166276152788219
    7.390222282562851
    7.62116672889294
    7.859328189170845
    8.104932195082434
    8.35821132617876
    8.619405430121846
    8.888761849813154
    9.166535657619814
    9.452989896920434
    9.748395831199197
    10.053033200924173
    10.367190488453053
    10.69116519121721
    11.025264103442748
    11.369803606675335
    11.725109969383938
    12.091519655927186
    12.46937964517491
    12.859047759086627
    13.260893001558085
    13.675295907856775
    14.102648904977299
    14.543356683257839
    14.997836579609647
    15.466518972722447
    15.949847690620023
    16.4482804309519
    16.962289194419146
    17.492360731744746
    18.03899700461177
    18.602715661005888
    19.18405052541232
    19.783552104331456
    20.401788107591813
    21.03934398595406
    21.69682348551512
    22.37484921943747
    23.07406325754489
    23.79512773434317
    24.538725476041392
    25.305560647167685
    26.096359417391675
    26.911870649185165
    27.7528666069722
    28.620143688440084
    29.514523178703836
    30.43685202803833
    31.388003653914527
    32.36887876809936
    33.380406229602464
    34.42354392427754
    35.49927967191121
    36.60863216165844
    37.75265191671026
    38.93242228910746
    40.14906048564207
    41.403718625818385
    42.69758483287521
    44.03188435890256
    45.40788074511826
    46.82687701840321
    48.29021692522831
    49.799286204141694
    51.35551389802112
    52.960373707334284
    54.61538538568848
    56.322116178991244
    58.08218230958472
    59.89725050675924
    61.76903958509547
    63.6993220721297
    65.68992588688376
    67.74273607084888
    69.8596965730629
    72.04281209097111
    74.29414996881395
    76.6158421553394
    79.01008722269376
    81.47915244840293
    84.02537596241552
    86.65116896124101
    89.35901799127979
    92.15148730350728
    95.03122128174189
    98.00094694679632
    101.0634765388837
    104.22171018072382
    107.47863862387145
    110.83734608086742
    114.30101314589453
    117.87291980670373
    121.55644855066323
    125.35508756787145
    129.27243405436744
    133.31219761856642
    137.47820379414662
    141.7743976627137
    146.2048475896735
    150.7737490768508
    155.48542873550238
    160.34434838348685
    165.3551092704708
    170.522456435173
    175.85128319877217
    181.34663579873381
    187.01371816744424
    192.85789686017688
    198.8847061370574
    205.09985320384044
    211.50922361646045
    218.11888685447485
    224.93510206867717
    231.96432400832333
    239.21320913358343
    246.68862191900791
    254.39764135397692
    262.3475676462887
    270.5459291352352
    279.00048942071135
    287.71925471510855
    296.7104814249557
    305.98268396948555
    315.54464284353196
    325.4054129323924
    335.57433208652964
    346.0610299642337
    356.875437150616
    368.0277945615727
    379.52866314162185
    391.38893386479754
    403.6198380480725
    416.23295798707477
    429.24023792417086
    442.6539953593012
    456.4869327142793
    470.7521493616006
    485.4631540291506
    500.63387759256153
    516.2786862673291
    532.4123952131831
    549.050282563595
    566.2081038937074
    583.9021071403857
    602.1490479885229
    620.9662057381641
    640.3713996674818
    660.3830059070906
    681.0199748416873
    702.3018490554899
    724.248781838474))

(: positive-ds (Vectorof (Pair Flonum Flonum)))
(define positive-ds
  #((0.00048511293333186416 . 7.287053259455084e-21)
    (0.0005002765039165264 . 5.174812552711603e-20)
    (0.0005159141767895225 . 2.1220112778569672e-20)
    (0.0005320407829306539 . 2.7353612783220783e-20)
    (0.0005486716177583588 . 1.436172748406251e-20)
    (0.000565822455705062 . 2.1992133061039514e-20)
    (0.0005835095652519211 . -2.010504872218286e-21)
    (0.000601749724437574 . 4.982650386129834e-20)
    (0.000620560236855965 . -7.430039714114664e-21)
    (0.0006399589481588143 . 3.8122182693055644e-20)
    (0.000659964263078802 . 2.138888295545083e-20)
    (0.0006805951629900561 . -5.314495618174954e-20)
    (0.0007018712240230741 . 4.6179617103884916e-20)
    (0.0007238126357517673 . -4.1596964367213793e-20)
    (0.0007464402204708848 . -1.8880161157866444e-21)
    (0.0007697754530826812 . 4.73133055222851e-20)
    (0.0007938404816122941 . 2.04622269272477e-20)
    (0.0008186581483719438 . -4.693486485344378e-20)
    (0.0008442520117947219 . -1.5493768351251446e-20)
    (0.0008706463689594116 . 2.853861232957886e-22)
    (0.0008978662788284951 . -5.4068667279405054e-20)
    (0.0009259375862222202 . -4.194952457490763e-20)
    (0.0009548869465523614 . -3.386529900189101e-20)
    (0.0009847418513400793 . -5.669583993860675e-20)
    (0.0010155306545430959 . -4.166473178227564e-20)
    (0.001047282599718231 . 9.521065082803394e-20)
    (0.00108002784804621 . 6.154424315085578e-20)
    (0.0011137975072465416 . 8.454888707177185e-20)
    (0.0011486236614111957 . 8.953086830150095e-20)
    (0.0011845394017867567 . 7.217159457790775e-20)
    (0.0012215788585357235 . 7.477344191447046e-20)
    (0.0012597772335086511 . 6.566772750951499e-21)
    (0.0012991708340598853 . 4.3179037768796526e-20)
    (0.0013397971079407453 . -1.0797335953851804e-19)
    (0.0013816946793051388 . 1.0047591559543256e-19)
    (0.0014249033858637816 . -2.5695478448084455e-20)
    (0.0014694643172244001 . 8.06630571658122e-20)
    (0.0015154198544565735 . 8.668353541071613e-20)
    (0.0015628137109211689 . -5.774014185861802e-20)
    (0.0016116909744056864 . 7.387283203139758e-20)
    (0.0016620981506082358 . 3.6190462953392175e-20)
    (0.0017140832080143208 . -1.1589495995584767e-20)
    (0.0017676956242121183 . 7.435250652996099e-20)
    (0.0018229864336935118 . 1.6584560557414573e-20)
    (0.0018800082771897484 . -5.314279891721045e-20)
    (0.00193881545259229 . 7.057409849045952e-20)
    (0.001999463967511161 . -9.116330016191191e-20)
    (0.0020620115935249127 . -2.0864709589013526e-19)
    (0.002126517922178207 . -4.876090730004397e-20)
    (0.00219304442278497 . 2.137019248779804e-20)
    (0.0022616545020970828 . 5.096078611740868e-20)
    (0.0023324135659006976 . 2.917296418544145e-20)
    (0.002405389082604425 . 1.2440609575139753e-19)
    (0.002480650648885927 . 2.726808826095052e-20)
    (0.002558270057465778 . 7.769966232307108e-20)
    (0.0026383213670799297 . 8.849414494444815e-20)
    (0.002720880974724625 . -1.8166863250830977e-19)
    (0.0028060276902502623 . 4.170398395508256e-20)
    (0.002893842813383458 . 1.3342105820505055e-19)
    (0.0029844102132593774 . 9.195101569892898e-20)
    (0.0030778164105494097 . -1.6343723184574783e-19)
    (0.003174150662272302 . -1.2879846095787289e-20)
    (0.003273505049380108 . -1.9626022851944244e-19)
    (0.0033759745672136167 . 1.0910988961422856e-20)
    (0.0034816572189254108 . 5.0829634028060636e-20)
    (0.003590654111972309 . -1.9843487184941522e-19)
    (0.003703069557782712 . 1.873310694823313e-19)
    (0.0038190111747082828 . 1.3562826460964407e-19)
    (0.0039385899943734605 . 3.46645056046899e-19)
    (0.004061920571540579 . 2.210205307217469e-19)
    (0.004189121097612757 . 6.473176348368055e-20)
    (0.004320313517901348 . 3.9553759423126633e-19)
    (0.004455623652789569 . 1.3778848742222007e-19)
    (0.004595181322928888 . -1.0442938740181728e-19)
    (0.004739120478610037 . -3.316247561247527e-19)
    (0.004887579333455926 . -2.7453099482491593e-19)
    (0.00504070050258943 . 3.7024133899284865e-19)
    (0.005198631145435001 . -2.1647643968448829e-19)
    (0.005361523113319176 . 1.888029366421676e-19)
    (0.005529533102041619 . -1.372189645824532e-19)
    (0.005702822809595014 . 2.8864400392230173e-19)
    (0.005881559099219254 . -8.575078874790418e-20)
    (0.006065914167982702 . -3.382932389054231e-19)
    (0.0062560657210910635 . -1.4881102497593667e-20)
    (0.006452197152132442 . 3.798115355624615e-19)
    (0.006654497729475611 . 2.7200995607274075e-19)
    (0.006863162789047346 . -2.7878483350076027e-19)
    (0.0070783939337239355 . 3.7048964867346634e-20)
    (0.007300399239581616 . -9.480859418132428e-20)
    (0.0075293934692608295 . 1.6517761136833034e-19)
    (0.007765598292709817 . -3.9745416880549755e-19)
    (0.008009242515584148 . -2.4044687013794144e-19)
    (0.008260562315590457 . -6.587400191683076e-19)
    (0.008519801487074855 . 8.387020028532755e-19)
    (0.008787211694169282 . -8.196097050138885e-19)
    (0.009063052732822494 . -3.2301461454261893e-19)
    (0.009347592802056508 . -7.025623380613278e-19)
    (0.00964110878480403 . 8.655823851246648e-19)
    (0.009943886538698017 . -2.5356392849550307e-19)
    (0.010256221197200734 . -8.086026309433652e-19)
    (0.010578417481476882 . 5.640352871418658e-19)
    (0.01091079002343331 . 5.970706595372109e-19)
    (0.011253663700366743 . -3.695457946677901e-19)
    (0.01160737398168092 . 3.289555843193904e-19)
    (0.011972267288155378 . -8.730097316112283e-20)
    (0.012348701364270177 . 3.8872528489959077e-19)
    (0.012737045664114032 . -4.7729496603624285e-19)
    (0.013137681751427724 . 6.478204344064299e-19)
    (0.013551003714360316 . 1.3298832303228675e-20)
    (0.013977418595542815 . 2.0257438569694155e-19)
    (0.014417346838112431 . -7.13005949816736e-19)
    (0.014871222748350642 . 5.681340686254808e-19)
    (0.015339494975630027 . -6.612062080438212e-19)
    (0.015822627010398216 . 8.296530982748908e-19)
    (0.016321097700962718 . -1.659895381856688e-18)
    (0.016835401789877486 . -1.3733254647069333e-18)
    (0.01736605047077166 . -9.783029760164461e-19)
    (0.017913571966502208 . -1.6404265725425483e-18)
    (0.01847851212955637 . 7.250935299760642e-19)
    (0.019061435065676088 . -1.4091769087450886e-18)
    (0.019662923781725627 . -6.952696577288123e-20)
    (0.02028358085887576 . -7.566567727966786e-19)
    (0.020924029152232494 . -9.59612072439542e-19)
    (0.021584912518096692 . -6.41488169822601e-20)
    (0.02226689657010246 . -7.336613651912944e-19)
    (0.022970669465547214 . -3.657425923384842e-19)
    (0.023696942723295693 . 1.7301455690443072e-18)
    (0.02444645207471311 . 1.242011812585554e-18)
    (0.025219958349160486 . -1.8439819571806356e-19)
    (0.02601824839566741 . 7.648227504707697e-19)
    (0.02684213604248493 . -3.024364317498657e-20)
    (0.027692463096313864 . -1.4810746531228193e-18)
    (0.028570100383102384 . -5.174668144972875e-19)
    (0.029475948832411267 . 4.448286316523317e-20)
    (0.030410940607456194 . -5.81533876508634e-19)
    (0.031376040283054726 . -1.4158564427160428e-18)
    (0.032372246073831004 . 2.2334054291587723e-18)
    (0.033400591115164815 . -2.5221828976912458e-18)
    (0.0344621447995138 . -8.510898707950959e-20)
    (0.035558014170888795 . -2.2354500799952555e-18)
    (0.03668934538042334 . 1.1684089702155055e-18)
    (0.037857325206150046 . -1.0125447739512992e-18)
    (0.039063182640279324 . -2.0920943476429972e-18)
    (0.040308190547470785 . -7.985899169864652e-19)
    (0.041593667397795686 . 2.07644114717057e-18)
    (0.04292097907831042 . -2.9718456774634723e-18)
    (0.04429154078739765 . -6.340767355681303e-19)
    (0.04570681901628487 . 2.1704004890048954e-18)
    (0.04716833362241929 . 2.9906195295324253e-18)
    (0.04867765999966722 . 1.9020569655558005e-19)
    (0.050236431350613966 . 2.9557106476059737e-18)
    (0.05184634106656999 . -1.515543925210912e-18)
    (0.053509145221242116 . -2.31880578111979e-18)
    (0.05522666518440587 . -1.4087074223799722e-18)
    (0.05700079036231938 . 3.193530579389267e-18)
    (0.05883348107205249 . -2.145260640991126e-18)
    (0.06072677155736839 . 3.0535299407780733e-18)
    (0.06268277315429321 . -1.9158317755446028e-18)
    (0.06470367761504209 . -6.452572296512067e-18)
    (0.06679176059954327 . 5.706584124310573e-18)
    (0.06894938534441653 . -4.893311618639166e-18)
    (0.07117900651992193 . 1.3836556609129577e-18)
    (0.07348317428610542 . 1.0150560143811903e-18)
    (0.0758645385601292 . -1.2725691451908302e-19)
    (0.07832585350759605 . 1.5354906427541542e-18)
    (0.08086998227155874 . 4.412576875593593e-18)
    (0.08349990195385616 . -9.270840538531389e-19)
    (0.08621870886444084 . 2.4601285608902732e-18)
    (0.0890296240554657 . -3.6117807566340314e-18)
    (0.09193599915808591 . -6.5483276811005975e-19)
    (0.0949413225412153 . -1.2057826985840237e-18)
    (0.09804922581285948 . 3.613052734932594e-18)
    (0.10126349068614282 . 3.63332655592265e-18)
    (0.10458805623376095 . -5.446963683297344e-18)
    (0.10802702655633457 . 2.4068800861136225e-19)
    (0.11158467889202853 . 5.547091525555895e-18)
    (0.11526547219684118 . -6.242140070908552e-18)
    (0.11907405622718036 . 1.9047348542199263e-18)
    (0.12301528115873718 . 4.260716267093159e-18)
    (0.12709420777826422 . -6.998082090230668e-19)
    (0.13131611828768022 . -2.022280195305526e-18)
    (0.135686527762978 . 1.3778664949460334e-17)
    (0.1402111963137291 . -7.957031399343018e-18)
    (0.14489614199258216 . 6.079724335094065e-18)
    (0.1497476545080694 . -1.0403594901115149e-19)
    (0.15477230979829656 . 9.750626447544526e-18)
    (0.15997698552772965 . -7.415235340924643e-18)
    (0.1653688775743427 . 1.233525816475006e-17)
    (0.17095551757989683 . 5.162148117868752e-18)
    (0.17674479164212215 . -5.735071010010956e-18)
    (0.18274496023412792 . -7.236699239167251e-18)
    (0.18896467944351802 . 1.2126334623173915e-18)
    (0.1954130236315065 . 3.6454880826080164e-18)
    (0.2020995096208769 . 1.1709803074569361e-18)
    (0.20903412253098091 . 9.629903048444698e-18)
    (0.2162273433882158 . -1.3746556035664427e-17)
    (0.2236901786516419 . -1.1161430087256522e-17)
    (0.23143419180571173 . -1.1783422723682643e-17)
    (0.2394715371855868 . 1.3681347180525074e-17)
    (0.24781499621535416 . -5.629683257071119e-19)
    (0.25647801625575717 . 1.7799283665847043e-17)
    (0.26547475227599016 . 2.3746497285091563e-17)
    (0.27482011158384456 . -2.3576750418849283e-17)
    (0.28452980187024574 . 2.294261508454993e-17)
    (0.2946203828481961 . 7.704460403433695e-18)
    (0.30510932179260053 . -3.1810717609493604e-18)
    (0.31601505331667373 . -2.7504732121747857e-17)
    (0.32735704375292773 . -2.295387335596396e-18)
    (0.3391558605424717 . -1.0542633246007705e-17)
    (0.35143324707592166 . 6.046403646719416e-18)
    (0.36421220347306293 . 1.6616431797725433e-17)
    (0.37751707383704586 . -2.288733632106875e-17)
    (0.39137364057289015 . -7.223380329827373e-18)
    (0.40580922642007966 . -1.4242771949989002e-17)
    (0.42085280491576815 . 2.1928227014922294e-17)
    (0.4365351200794305 . 2.746723466096044e-17)
    (0.4528888161926027 . 3.333036921193442e-20)
    (0.46994857863973616 . 2.2313748280730584e-17)
    (0.4877512868793533 . 2.510850553983312e-17)
    (0.5063361807299942 . 5.034323929063707e-17)
    (0.5257450412844764 . 3.549939469031809e-17)
    (0.5460223879105075 . -2.600409107758072e-17)
    (0.5672156929577491 . 2.6931694472926522e-17)
    (0.5893756159733434 . 3.4958600071342e-17)
    (0.6125562594323366 . 5.5315848730447774e-17)
    (0.6368154482194206 . -1.0700031297321404e-17)
    (0.6622150353574432 . -5.2025197336558966e-17)
    (0.6888212367702189 . 3.554366060808207e-17)
    (0.7167049981969127 . 1.2570720319880541e-17)
    (0.745942397747961 . 3.5141078190499194e-17)
    (0.7766150880142586 . -4.2199351454803637e-17)
    (0.8088107821192024 . -4.7545049409632634e-17)
    (0.8426237886453156 . 4.009422968651717e-18)
    (0.8781556009829815 . -3.617235787577101e-17)
    (0.915515547349229 . 8.295737913212842e-18)
    (0.954821508522209 . 1.1311450967949616e-17)
    (0.9962007112467078 . -3.341340557703004e-17)
    (1.0397906063049112 . 5.075753532140173e-18)
    (1.085739841434657 . -7.880618461116728e-17)
    (1.1342093406379186 . -6.691136763078832e-17)
    (1.1853735029825458 . -1.3540227196001537e-17)
    (1.23942153579231 . -1.122688801019997e-17)
    (1.2965589391815442 . 1.589418190470739e-17)
    (1.3570091612651922 . -9.570749163265672e-17)
    (1.4210154461147195 . 4.658582253165645e-17)
    (1.488842899696241 . 7.129474621289647e-17)
    (1.5607808026915324 . -1.0459889566319876e-17)
    (1.637145203350768 . -8.586169319418946e-17)
    (1.7182818284590453 . -7.747991575210629e-17)
    (1.8045693562372267 . -8.399216598536489e-17)
    (1.8964231016843132 . -9.78065845802123e-17)
    (1.9942991726738615 . -8.553441869169175e-17)
    (2.098699164244332 . -2.1290728624508516e-16)
    (2.210175469216157 . 8.01759282156125e-17)
    (2.3293372958186187 . -2.4867346652824196e-17)
    (2.4568574977678734 . 2.363531379168368e-17)
    (2.5934803396257973 . 1.3530965334869875e-16)
    (2.7400303407967566 . 2.84271535920292e-17)
    (2.8974223658015266 . -1.3520108699441425e-16)
    (3.066673157250024 . -2.0856498340728085e-16)
    (3.24891454212279 . 8.01857337785979e-17)
    (3.4454085826672616 . -5.781818869290501e-19)
    (3.657564991763094 . 1.1214605127608465e-16)
    (3.886961190654824 . 1.1863185200788392e-16)
    (4.135365456504734 . -1.6111676390969494e-16)
    (4.404763690759263 . -1.826064832365271e-16)
    (4.697390439898002 . 1.379838386618194e-16)
    (5.015764921510652 . -2.1246597826089647e-16)
    (5.3627329554868375 . -3.3775191059186603e-16)
    (5.741515878194139 . -3.717431673583332e-16)
    (6.155767734066404 . 5.959787843501827e-17)
    (6.609642303020833 . 3.7622363738566276e-16)
    (7.107871844826896 . -3.836032048686347e-16)
    (7.655859837096132 . -3.9321989985822915e-16)
    (8.259790469744917 . -3.1964018652469046e-16)
    (8.926758258075042 . 6.9905306889558135e-16)
    (9.664921877487837 . 3.480231208392409e-16)
    (10.48368724147004 . -5.058240114382486e-16)
    (11.393925986977917 . -6.675676784501677e-17)
    (12.408236956676573 . 8.310887616559652e-17)
    (13.541260051342169 . -8.73018890312213e-16)
    (14.810054065547394 . 3.629323995570106e-16)
    (16.23455294142346 . 1.7266347574671478e-15)
    (17.8381184420714 . -2.528705665223229e-16)
    (19.64821177052238 . 1.427525560659987e-16)
    (21.69721241950954 . 1.3639691024425497e-15)
    (24.023419895539476 . 1.059263125823483e-15)
    (26.67228339672289 . 5.677986535423153e-16)
    (29.697916670361636 . 8.797766564780851e-16)
    (33.16497097303293 . 2.2643393731648284e-15)
    (37.15095942152534 . 3.2992302512248215e-15)
    (41.749152554212905 . -7.432852127721744e-16)
    (47.07219963059767 . 1.2565598221538282e-15)
    (53.25667579654665 . -2.989602113304488e-15)
    (60.46881541504593 . 4.2658376888972845e-16)
    (68.91177162107691 . 6.5639980171300165e-15)
    (78.83484836691989 . 3.283618366631356e-15)
    (90.54529331889034 . 5.365925866805094e-16)
    (104.42343099333327 . -4.924303049844688e-15)
    (120.9421736013677 . -6.027389047485177e-15)
    (140.6922975166525 . 2.194549790000635e-15)
    (164.41535162028603 . -1.283154535426018e-14)
    (193.0467201709333 . 7.39192747490866e-15)
    (227.77226850156077 . 4.041651467709608e-15)
    (270.1032564200185 . 1.5109735837066455e-14)
    (321.97595769445013 . -2.4305227779521264e-15)
    (385.8848853901661 . -9.673369616452878e-16)
    (465.06199877311366 . 2.589243254189485e-14)
    (563.7192066814514 . 2.7978709586136315e-15)
    (687.3785453728616 . 2.6211396926545634e-14)
    (843.3245752330527 . -2.749513333300684e-15)
    (1041.2282720337255 . 1.0726990009004963e-13)
    (1294.013181424767 . 3.902520136312285e-14)
    (1619.066185376973 . 2.1522648714933098e-14)
    (2039.9419672006518 . 1.7004799414141945e-14)
    (2588.779948644629 . -1.1152610240152095e-13)
    (3309.757172003916 . -5.473819345344809e-14)
    (4264.059139978984 . -4.1250862575658904e-13)
    (5537.092631527714 . -4.1770682564106657e-13)
    (7249.0369893365005 . -6.983673906909162e-15)
    (9570.408550911743 . 8.406510611058146e-13)
    (12745.21822672292 . -7.065701374874163e-15)
    (17125.732590817657 . 3.7713879548521975e-13)
    (23225.129627419305 . -1.340299096808553e-12)
    (31798.011661335175 . 6.747331571251255e-13)
    (43964.705617713975 . 1.798201787821037e-12)
    (61405.07824431698 . 2.3192412808684584e-12)
    (86663.8454157708 . -2.3526174091132417e-12)
    (123636.59808609908 . -3.0399042496665012e-12)
    (178351.93658192107 . -6.339856905865759e-14)
    (260244.22671984948 . -5.410202944966889e-12)
    (384248.6532859191 . 5.73990477091046e-12)
    (574290.8534432624 . 6.293454072918912e-12)
    (869170.6475863855 . 2.5628917849990087e-11)
    (1332607.5631375103 . 5.752777064120257e-11)
    (2070614.9539178647 . -3.1249221708431105e-11)
    (3261951.7582527394 . 1.9615216664993022e-10)
    (5212232.270862971 . 2.331221928415278e-10)
    (8451441.464476893 . -7.598267899257087e-10)
    (13912250.75293453 . -7.275005102634024e-11)
    (23261010.584611285 . 4.1496939448420006e-10)
    (39521710.28864972 . -3.095599924876826e-09)
    (68271092.3676407 . 4.954815266536196e-09)
    (119965607.16041763 . -5.0301550654932666e-09)
    (214549388.82292324 . -1.122660075850228e-08)
    (390739674.37478554 . -2.5660991429857667e-09)
    (725076776.775437 . 9.076770263427603e-10)
    (1371737450.3579445 . 1.6447268313752807e-08)
    (2647346051.5503793 . 1.0246451628127077e-07)
    (5215230997.525738 . -2.2934934972615406e-07)
    (10493934467.679167 . -6.352421890093103e-07)
    (21582048848.945744 . -5.84894932458084e-07)
    (45397618107.24928 . -1.7327545064306486e-06)
    (97738428194.02795 . -3.865763973784669e-06)
    (215528592474.9177 . -9.286963901441946e-06)
    (487165943884.08777 . -1.2737444427048404e-05)
    (1129579558136.7493 . 0.00010483243681732672)
    (2688874144903.8423 . 2.5247520705553776e-05)
    (6576495882115.553 . 0.00025003085283704853)
    (16540814160421.832 . 0.00039813022933945206)
    (42819026062657.85 . -0.00045919653514175875)
    (114189306235880.12 . -0.0011724802860866989)
    (313997528128146.56 . 0.013936075444993389)
    (891158951538994.4 . -0.03848075420868936)
    (2613011211001386.0 . -0.17105080091174218)
    (7923680177264742.0 . -0.28291272396024725)
    (2.487530444089166e+16 . 1.4203043682557206)
    (8.093496202618488e+16 . 0.709099724309675)
    (2.7322184677454848e+17 . -1.2668239140658226)
    (9.580904901869568e+17 . -40.61664863422302)
    (3.494020779824991e+18 . -139.5623508973493)
    (1.326796767819535e+19 . 617.2907294451816)
    (5.252814237556169e+19 . -1774.6728954278933)
    (2.170972358192325e+20 . 1042.9639641826302)
    (9.379393155479388e+20 . -29506.92230986667)
    (4.241848966805818e+21 . 251611.61569680815)
    (2.011019589973599e+22 . -268237.093288903)
    (1.0009169886484786e+23 . -3548470.624775735)
    (5.237939767221943e+23 . -3307357.444456701)
    (2.886584436022493e+24 . 171249912.60047522)
    (1.6779200899862084e+25 . 695418037.9432284)
    (1.030493783595181e+26 . -4372769307.022909)
    (6.69812185700711e+26 . 45952129210.82991)
    (4.615981684139182e+27 . 41155653435.27635)
    (3.378876585354472e+28 . -1744261341683.3298)
    (2.632064117770894e+29 . 17316783105981.531)
    (2.1861538855792003e+30 . -5136456416769.427)
    (1.939973296239877e+31 . -695689534862793.2)
    (1.8430595562481285e+32 . 5470877559248909.0)
    (1.8786136536347794e+33 . 4.394420226189102e+16)
    (2.0589457107550113e+34 . 1.0912841158473578e+18)
    (2.43190328008533e+35 . 4.6849202048064115e+17)
    (3.10282384509504e+36 . -7.468629217962736e+19)
    (4.286714207435176e+37 . -3.2260863393585525e+21)
    (6.428777896972236e+38 . -1.4334692317849637e+22)
    (1.049258352602445e+40 . -3.386043891431298e+23)
    (1.8686803171612082e+41 . 2.3827454147506918e+24)
    (3.6414175441569734e+42 . -1.5352351717107655e+26)
    (7.785924405715174e+43 . -4.847798189948165e+26)
    (1.83195004156455e+45 . -1.181343290649753e+28)
    (4.757509638501038e+46 . 2.3174658337892313e+30)
    (1.3678793449206142e+48 . -6.402735276766394e+31)
    (4.3681675192606486e+49 . 2.484103073726363e+33)
    (1.554385147664416e+51 . -7.15823207740273e+34)
    (6.184361178951956e+52 . 2.2564932541358492e+36)
    (2.7607228460808756e+54 . -1.2066931474805295e+37)
    (1.387729345039176e+56 . -1.0512565090219215e+40)
    (7.884094045764237e+57 . 5.814378629016813e+41)
    (5.081890344836013e+59 . 1.3610757266697308e+42)
    (3.731114527928971e+61 . 2.1822352604466645e+45)
    (3.1329865879018768e+63 . -6.292921774847357e+46)
    (3.0213931478059766e+65 . -1.5061242076069467e+49)
    (3.360962881853185e+67 . -1.626360949945326e+51)
    (4.331773172710414e+69 . -2.7993324957165646e+53)
    (6.498475870165788e+71 . 3.733949158560767e+55)
    (1.1401496818019767e+74 . 1.3656955888023106e+57)
    (2.3509420759921335e+76 . -4.791685622166505e+59)
    (5.725893145375883e+78 . 7.446926832904932e+61)
    (1.6558684839861452e+81 . 1.0296317776844197e+65)
    (5.716370875276776e+83 . -1.0021500957970777e+67)
    (2.3688109954366353e+86 . -3.7581882450375625e+69)
    (1.1850440268660405e+89 . 4.778157332560249e+72)
    (7.199273758087368e+91 . 1.837747429137931e+75)
    (5.343538708730692e+94 . -7.365706356630585e+77)
    (4.876111352707183e+97 . 1.3148108653599204e+81)
    (5.505869521450131e+100 . 1.944404306955852e+84)
    (7.744202935901645e+103 . 6.17849039544379e+87)
    (1.3661778750202e+107 . 5.006695302453677e+90)
    (3.044330854000249e+110 . 1.539474389406713e+94)
    (8.631792057674117e+113 . -5.041548497925841e+97)
    (3.137647310570189e+117 . 4.245702082430937e+100)
    (1.473577574797504e+121 . 1.0029465158093984e+105)
    (9.013317301694191e+124 . -5.43643842366429e+108)
    (7.239770606552406e+128 . -1.687650686719372e+110)
    (7.7017817197777e+132 . -2.0151109448204943e+116)
    (1.094705200077448e+137 . 6.712298179422747e+120)
    (2.0978537960154526e+141 . 7.391845166298471e+124)
    (5.471176099614323e+145 . 4.715571837807654e+129)
    (1.9606303234256925e+150 . 1.1974880544002449e+134)
    (9.750639080045098e+154 . 5.375534992176601e+138)
    (6.798926795391929e+159 . 2.5659008792304313e+143)
    (6.717445112142007e+164 . -4.9777237030042644e+148)
    (9.50723558093052e+169 . -4.241955675103341e+153)
    (1.9492540757045903e+175 . 2.457155476948572e+158)
    (5.857021794165392e+180 . -3.3143415128822057e+164)
    (2.6101576642750866e+186 . -1.3194322779996063e+170)
    (1.7465763572163952e+192 . 3.986263435107535e+175)
    (1.7772803479239517e+198 . -1.0878848502109484e+182)
    (2.7865113209812e+204 . -1.4593352360634193e+188)
    (6.822899310782423e+210 . 3.2812165844793366e+194)
    (2.6456384412921967e+217 . 3.8066766815192514e+200)
    (1.6481038358004873e+224 . -6.769938218924482e+207)
    (1.6740380998219687e+231 . 8.466685826074968e+214)
    (2.8151919034596256e+238 . -2.123973250628041e+222)
    (7.962604223977985e+245 . 3.9217709122676487e+229)
    (3.850021038738558e+253 . -7.1045613508698715e+236)
    (3.236002323062726e+261 . -2.104819443379521e+245)
    (4.810570685749818e+269 . -1.4167319742866789e+253)
    (1.2875533828926397e+278 . 7.626059974847463e+260)
    (6.319684656519154e+286 . -3.83875210857012e+270)
    (5.797184730644923e+295 . -4.1974821203145185e+278)
    (1.0134858165681662e+305 . -1.1960732643227e+288)
    (+inf.0 . -inf.0)))

(: negative-ds (Vectorof (Pair Flonum Flonum)))
(define negative-ds
  #((-0.0004848777128822606 . -2.106959040416988e-20)
    (-0.0005000263524810411 . 1.7625505823099235e-20)
    (-0.0005156481466004561 . 2.027643368070832e-20)
    (-0.0005317578660592662 . -4.8512463234164716e-20)
    (-0.000548370742296052 . 3.579583327230676e-20)
    (-0.000565502481702158 . 2.6250708424808036e-20)
    (-0.0005831692803986474 . -2.1161599078553446e-20)
    (-0.0006013878394708923 . -1.9981896427908373e-20)
    (-0.0006201753806748412 . 2.227288708361765e-20)
    (-0.0006395496626294227 . -1.1302604898028686e-20)
    (-0.0006595289975099812 . 3.7183329388334764e-21)
    (-0.0006801322682580862 . 1.4343382871702033e-20)
    (-0.0007013789463235141 . 4.19236554910797e-20)
    (-0.0007232891099546805 . 3.428062753271238e-20)
    (-0.000745883463054277 . 1.1283170331370492e-20)
    (-0.000769183354617377 . 4.255757126500432e-20)
    (-0.0007932107987697786 . 2.814972083732834e-20)
    (-0.0008179884954248897 . -4.979052687052559e-20)
    (-0.0008435398515779981 . 4.1020418015530164e-20)
    (-0.0008698890032573279 . -2.0284963214805618e-20)
    (-0.0008970608381518608 . -4.288020202961567e-20)
    (-0.0009250810189364862 . 3.73119220198268e-20)
    (-0.0009539760073156515 . -1.8634585995285095e-20)
    (-0.0009837730888073086 . 1.2199363570121225e-21)
    (-0.0010145003982895866 . 4.495559977190981e-20)
    (-0.001046186946333284 . 4.3474490341103084e-20)
    (-0.0010788626463439418 . 3.604379947920555e-20)
    (-0.0011125583425379566 . 5.56391640355605e-20)
    (-0.0011473058387779002 . -7.037849041712792e-20)
    (-0.0011831379282929455 . 9.69524773821592e-21)
    (-0.001220088424311041 . -5.470493004640914e-20)
    (-0.001258192191630257 . 3.5437812670146733e-20)
    (-0.0012974851791574989 . -6.765506925362685e-20)
    (-0.0013380044534436095 . -7.414679426638717e-20)
    (-0.0013797882332446968 . 1.0149191885812544e-20)
    (-0.001422875925140385 . -2.7262070498829763e-20)
    (-0.0014673081602405546 . 6.623304433785929e-20)
    (-0.0015131268320130301 . 1.0106498893687096e-19)
    (-0.0015603751352655953 . 4.0820161827105014e-20)
    (-0.001609097606316648 . -1.0669093726536096e-19)
    (-0.0016593401643897737 . 9.218550532039208e-20)
    (-0.001711150154268498 . -2.3236047013700504e-20)
    (-0.0017645763902484882 . 4.4785808041305754e-20)
    (-0.0018196692014255033 . 7.644737360967079e-20)
    (-0.0018764804783584495 . -6.641327861281367e-20)
    (-0.0019350637211479776 . -5.812752395640082e-20)
    (-0.0019954740889721588 . -1.2997916755622335e-19)
    (-0.002057768451121909 . -1.924260643119962e-19)
    (-0.0021220054395799806 . 3.1240315322899305e-20)
    (-0.0021882455031885183 . 4.2440809687249976e-20)
    (-0.0022565509634513816 . 9.101604562571392e-20)
    (-0.0023269860720186593 . 1.6128931642996482e-20)
    (-0.0023996170699020513 . -1.935399175287333e-19)
    (-0.002474512248471081 . -2.855759081333477e-20)
    (-0.002551742012281381 . -1.743237522189642e-19)
    (-0.0026313789437876507 . 2.2491723561939065e-20)
    (-0.002713497869995199 . 1.304494983910357e-19)
    (-0.002798175931105389 . 9.895003953358041e-20)
    (-0.0028854926512116782 . -1.9814715270209227e-19)
    (-0.002975530011104378 . 9.235275936603056e-20)
    (-0.003068372523243691 . -2.5554744043043932e-20)
    (-0.0031641073089620593 . 1.2133904469057152e-20)
    (-0.003262824177958322 . 7.791026035643279e-20)
    (-0.0033646157101477107 . 8.069325092220694e-21)
    (-0.0034695773399332124 . 6.066527208813204e-20)
    (-0.00357780744296538 . 1.726578866486843e-19)
    (-0.003689407425459237 . 8.755195129513237e-20)
    (-0.0038044818161384756 . -3.1544240795769355e-20)
    (-0.003923138360878761 . 3.7520119240160075e-19)
    (-0.0040454881201235264 . 1.278316467235735e-19)
    (-0.0041716455691472795 . 1.6981916830393132e-19)
    (-0.004301728701243024 . 3.1010114127382133e-19)
    (-0.0044358591339120675 . 2.000684148507886e-20)
    (-0.004574162218136062 . -1.4209647258652529e-19)
    (-0.004716767150812785 . 1.724297340303725e-19)
    (-0.00486380709043878 . -2.4665487524436087e-19)
    (-0.005015419276123577 . 5.1615663988880967e-20)
    (-0.005171745150021845 . -1.5285919966070898e-19)
    (-0.005332930483271392 . -1.5172228554575442e-19)
    (-0.005499125505526528 . -2.598313022886681e-20)
    (-0.005670485038177827 . 6.549990421578792e-20)
    (-0.005847168631350863 . -4.778854559887401e-20)
    (-0.006029340704777994 . 6.799527292234638e-20)
    (-0.006217170692638675 . 2.0085641109616846e-19)
    (-0.006410833192465222 . 4.5600645000456827e-20)
    (-0.006610508118212287 . 1.8791009186949312e-19)
    (-0.006816380857589562 . 2.1325222339957728e-19)
    (-0.007028642433758505 . -2.205966102543369e-20)
    (-0.007247489671494958 . -3.6357095189256593e-19)
    (-0.007473125367920641 . 1.533065983050011e-19)
    (-0.0077057584679074 . -5.2859089248663636e-20)
    (-0.007945604244259023 . 3.52418730915407e-19)
    (-0.008192884482776051 . -2.5245826598779236e-20)
    (-0.00844782767230976 . 8.110388823566882e-19)
    (-0.008710669199911777 . 4.247218097579681e-19)
    (-0.008981651551186255 . -7.726214439377394e-19)
    (-0.00926102451595153 . -5.843325417155317e-19)
    (-0.009549045399318171 . -3.0763492576095402e-19)
    (-0.009845979238290085 . 7.67519079563552e-19)
    (-0.010152099023994757 . -5.144489171785002e-19)
    (-0.010467685929648083 . -6.743894052511647e-19)
    (-0.010793029544358106 . -6.769368153187454e-20)
    (-0.011128428112870787 . -8.623795245782583e-19)
    (-0.011474188781359276 . 5.262395896247483e-19)
    (-0.011830627849356191 . 3.0303240849562576e-19)
    (-0.012198071027926161 . 2.1854305731782524e-19)
    (-0.012576853704173097 . 3.8112097413428696e-19)
    (-0.012967321212173649 . -2.87469018122663e-19)
    (-0.01336982911042459 . -7.489622800860591e-19)
    (-0.013784743465887926 . -3.9765310768588313e-20)
    (-0.014212441144712845 . -5.600773306602792e-19)
    (-0.014653310109708509 . 1.5943725405796341e-19)
    (-0.015107749724635898 . 4.540939127519002e-19)
    (-0.015576171065380543 . -5.589530927446483e-19)
    (-0.016058997238060837 . 1.7270579802793842e-18)
    (-0.016556663704118763 . -1.632993027352529e-18)
    (-0.01706961861243135 . -1.2270936729807808e-18)
    (-0.017598323138471438 . 1.1357749040933149e-18)
    (-0.018143251830536212 . -1.2030943817637002e-18)
    (-0.018704892963050477 . -1.0961186225376268e-18)
    (-0.019283748896939176 . -1.1780304229996325e-18)
    (-0.01988033644705036 . 8.827053506821462e-19)
    (-0.020495187256595035 . 8.515770379828979e-20)
    (-0.021128848178554448 . 3.425551953877502e-19)
    (-0.02178188166398822 . -3.7596333494025924e-19)
    (-0.022454866157158034 . -1.01288558297386e-18)
    (-0.023148396497361577 . -1.3459627647431748e-18)
    (-0.02386308432734972 . 4.8074020104275425e-20)
    (-0.024599558508176536 . -8.89541304505486e-19)
    (-0.025358465540306737 . 2.6362752413192174e-22)
    (-0.026140469990777973 . -1.0267799766765561e-18)
    (-0.026946254926186575 . -1.1581997036308915e-19)
    (-0.027776522351234137 . -4.0126581987619013e-19)
    (-0.0286319936525391 . 4.718771000396679e-19)
    (-0.029513410047381766 . -2.8335007352392444e-20)
    (-0.030421533037013122 . 1.2213826654497336e-18)
    (-0.031357144864117045 . 1.9330185531580772e-18)
    (-0.03232104897397196 . -1.6854710165805219e-18)
    (-0.03331407047881178 . 4.257378252205867e-19)
    (-0.03433705662483626 . 2.4723651429168603e-18)
    (-0.03539087726126849 . -5.421228227777854e-19)
    (-0.036476425310801204 . 3.179254059731123e-18)
    (-0.03759461724071392 . -2.7686588607727445e-18)
    (-0.03874639353387986 . -3.459622515126237e-18)
    (-0.03993271915881438 . -2.5034136530020386e-18)
    (-0.041154584037845485 . 2.7385810715378915e-18)
    (-0.04241300351241163 . 5.802644828900916e-19)
    (-0.04370901880441221 . -1.5561608482265385e-19)
    (-0.045043697472451384 . 3.0516396314831922e-18)
    (-0.04641813386172703 . -3.3547129864823247e-18)
    (-0.04783344954622212 . -1.5380904291870428e-18)
    (-0.049290793761756026 . 6.828028060603044e-19)
    (-0.05079134382834895 . -2.5745954131680757e-18)
    (-0.052336305560241644 . 3.187064224530237e-18)
    (-0.05392691366179642 . -3.358661718928209e-18)
    (-0.0555644321073834 . -2.3447499649171016e-18)
    (-0.05725015450322689 . -3.1214574459971403e-18)
    (-0.05898540442905266 . -2.521278275006181e-18)
    (-0.06077153575723495 . 9.13095461328589e-19)
    (-0.06260993294699418 . 3.2420863062163555e-18)
    (-0.0645020113110416 . -6.5708930371043775e-18)
    (-0.06644921725190488 . -6.103049538012378e-18)
    (-0.06845302846500008 . -1.2328510524166196e-18)
    (-0.07051495410533898 . 5.769458369011139e-18)
    (-0.07263653491457747 . 1.7691740313590935e-18)
    (-0.07481934330492018 . 3.039080270477699e-19)
    (-0.07706498339619806 . -5.340222251701746e-18)
    (-0.07937509100223099 . -5.619617147959712e-18)
    (-0.08175133356237452 . -3.076181991297649e-18)
    (-0.0841954100139305 . -3.119012840229868e-18)
    (-0.08670905060087507 . 3.601870935128459e-18)
    (-0.089294016614124 . -4.739740012930359e-18)
    (-0.09195210005831625 . 5.206792300023025e-18)
    (-0.09468512323985084 . 8.42204215564623e-19)
    (-0.09749493827066161 . -4.671350698143233e-18)
    (-0.10038342648195772 . -4.744879802072286e-18)
    (-0.10335249774189831 . 6.800862460492163e-18)
    (-0.10640408967090506 . -5.749718563746328e-18)
    (-0.10954016674805077 . 2.7476876652079134e-18)
    (-0.1127627193016928 . 4.33716548465386e-18)
    (-0.11607376237725281 . 4.312212954115727e-18)
    (-0.11947533447477532 . 1.2605340450492377e-18)
    (-0.1229694961486328 . 4.7683414012950735e-18)
    (-0.12655832846148324 . -8.274642425290404e-19)
    (-0.1302439312843307 . -7.544462567212541e-18)
    (-0.13402842143429172 . -4.835633997741849e-19)
    (-0.13791393064143284 . 6.015289387262617e-18)
    (-0.14190260333582086 . 1.2163795292001367e-17)
    (-0.14599659424571793 . -8.969875857035464e-18)
    (-0.15019806579766423 . 7.294989543591792e-18)
    (-0.154509185309023 . -1.0739013545956815e-17)
    (-0.1589321219634219 . 1.1063200181016539e-17)
    (-0.16346904355941147 . 1.3950176656595435e-18)
    (-0.1681221130225865 . -8.647084263078306e-18)
    (-0.1728934846713762 . -8.855706197072986e-18)
    (-0.17778530022671651 . -8.50118134351519e-18)
    (-0.18279968455587453 . 1.0790546879106912e-18)
    (-0.18793874114080633 . -9.100130552281101e-18)
    (-0.19320454726160494 . 6.2853817379441374e-18)
    (-0.19859914888583774 . 1.3007838561458206e-17)
    (-0.20412455525489343 . 4.043712127012572e-19)
    (-0.2097827331588613 . -7.782733745473872e-18)
    (-0.21557560089196137 . -4.5961349288957284e-18)
    (-0.22150502188114043 . 1.2231543329352422e-17)
    (-0.22757279798115346 . -1.1787015047465016e-17)
    (-0.23378066243027457 . -1.118495064497693e-18)
    (-0.24013027246173207 . 1.1240192085279803e-17)
    (-0.2466232015670544 . -2.4173623167463854e-18)
    (-0.2532609314087494 . -7.788664599260712e-18)
    (-0.26004484338113865 . 2.1732695352126535e-17)
    (-0.2669762098197317 . 4.529222286361873e-18)
    (-0.2740561848612734 . 2.0575847649803967e-17)
    (-0.2812857949585305 . -2.562961006182989e-17)
    (-0.28866592905602184 . 7.6895092526933e-18)
    (-0.29619732843524027 . -1.2336564511105047e-17)
    (-0.3038805762404842 . -1.6172286700388382e-17)
    (-0.31171608669920775 . -2.1874975777510195e-17)
    (-0.3197040940538329 . -8.199761174223195e-18)
    (-0.32784464122524176 . -1.0258586762660572e-17)
    (-0.33613756823169166 . 1.1065872227439662e-17)
    (-0.3445825003906736 . -3.509750290406745e-18)
    (-0.35317883633526936 . -1.9495584200750616e-17)
    (-0.3619257358808497 . 1.3385237077918724e-17)
    (-0.37082210778249936 . 7.503495774649699e-18)
    (-0.3798665974283421 . -1.6155435473600083e-17)
    (-0.38905757451896517 . 3.581996412453827e-18)
    (-0.3983931207883944 . -2.186742617697992e-17)
    (-0.407871017827531 . -1.620643325584559e-17)
    (-0.41748873507660395 . 2.1990083596590378e-17)
    (-0.42724341805899774 . -7.30038018146262e-18)
    (-0.43713187693474415 . -2.744142914670074e-17)
    (-0.4471505754579812 . 1.032221696621471e-17)
    (-0.4572956204287403 . -9.401746630272272e-18)
    (-0.4675627517354664 . 1.568464578260928e-18)
    (-0.4779473330906439 . 1.3777543920072e-17)
    (-0.4884443435677295 . 2.2433236991789505e-17)
    (-0.49904837005319985 . -1.1756227866063207e-17)
    (-0.5097536007328203 . -3.979700667545985e-17)
    (-0.5205538197361378 . 2.992316493227244e-17)
    (-0.5314424030675929 . -2.5953365570114604e-17)
    (-0.5424123159564149 . 4.263923762167425e-17)
    (-0.5534561117604869 . 6.792197915735056e-18)
    (-0.564565932561529 . 8.85792762861878e-18)
    (-0.5757335115900775 . 5.2638232186759144e-17)
    (-0.5869501776187284 . -1.3263688854694558e-17)
    (-0.5982068614607824 . 4.431643083882866e-17)
    (-0.6094941047086339 . -3.889357793196616e-17)
    (-0.6208020708418347 . 1.7874991379533952e-17)
    (-0.6321205588285577 . -1.2428753672788363e-17)
    (-0.643439019336053 . -5.1389049528919193e-17)
    (-0.6547465736554562 . -3.9092853614269235e-17)
    (-0.6660320354338488 . 1.4405811497785132e-17)
    (-0.6772839352916448 . 4.667728400652459e-18)
    (-0.6884905483860748 . -1.921487972240059e-17)
    (-0.6996399249616673 . -2.4088184148953082e-17)
    (-0.7107199239061172 . 3.6526794545109734e-17)
    (-0.7217182493047579 . 1.4129801662985832e-18)
    (-0.7326224899590079 . 4.5901053136667967e-17)
    (-0.7434201618036991 . -1.4159755448943397e-18)
    (-0.7540987531252149 . 4.455822229604516e-17)
    (-0.7646457724470043 . 5.427256105217267e-17)
    (-0.7750487989115286 . 4.0213058679437405e-18)
    (-0.7852955349483044 . -4.039666687131752e-17)
    (-0.795373860976783 . -4.07751871965384e-17)
    (-0.8052718918507843 . -4.8301054248702166e-17)
    (-0.8149780347085777 . -1.0406009309415639e-17)
    (-0.824481047850057 . 2.3814638364661886e-17)
    (-0.8337701002204579 . -4.6086942307388796e-17)
    (-0.8428348310394419 . 6.987367940997186e-18)
    (-0.8516654090759375 . 2.82580324784011e-17)
    (-0.8602525910337603 . -3.7602668586884535e-17)
    (-0.8685877784816475 . -7.355679499376496e-18)
    (-0.8766630727349206 . 2.01084833539142e-17)
    (-0.8844713270755226 . 4.731238866507822e-17)
    (-0.8920061956836538 . -1.3025370452067454e-18)
    (-0.8992621786486502 . 1.6298635484044654e-17)
    (-0.9062346624300305 . -1.504289551609632e-17)
    (-0.9129199551526632 . -2.0989088200491108e-17)
    (-0.9193153161435139 . -4.2147809532680856e-17)
    (-0.9254189791520612 . 4.686671952331961e-17)
    (-0.9312301687426531 . -5.419294796925712e-17)
    (-0.9367491094050615 . -4.808539975754049e-17)
    (-0.9419770269992622 . 4.0991793084493095e-17)
    (-0.9469161422317695 . 4.915666149603816e-17)
    (-0.951569655953083 . -4.3806599310472934e-17)
    (-0.9559417261680803 . -3.923738462103911e-17)
    (-0.9600374367622607 . 1.679229255061313e-17)
    (-0.9638627580650455 . 2.6144817761556684e-17)
    (-0.9674244994949288 . 4.2640486031288585e-17)
    (-0.9707302546579267 . 2.3475832444048937e-17)
    (-0.9737883393979396 . 3.6833816984937015e-18)
    (-0.9766077234225442 . 2.0979771513422152e-17)
    (-0.9791979562473878 . -4.974085983287543e-18)
    (-0.9815690883136698 . 3.6232375134888925e-17)
    (-0.9837315882330274 . 2.119845953945831e-17)
    (-0.985696257199431 . -5.186377179270369e-17)
    (-0.9874741416755248 . 4.5042414260721607e-17)
    (-0.9890764455086011 . 3.8290555638111146e-17)
    (-0.9905144426568394 . 3.831724791843613e-17)
    (-0.991799391707835 . 4.409184517378617e-17)
    (-0.9929424533476672 . 1.7878546358495637e-17)
    (-0.9939546118893758 . 4.7119386774000124e-17)
    (-0.9948466018950534 . 8.814048062835117e-18)
    (-0.9956288408269502 . 5.1654135731138195e-18)
    (-0.9963113685419894 . 4.8641606507366167e-17)
    (-0.9969037943036427 . -1.6791831975825047e-18)
    (-0.9974152518287409 . -3.5544029307038557e-17)
    (-0.9978543627186245 . 4.106505067185739e-17)
    (-0.9982292084487856 . -2.3699092478631005e-17)
    (-0.9985473109138543 . -2.122461256205012e-17)
    (-0.9988156213506826 . -3.139208227148587e-17)
    (-0.9990405172966104 . -2.667851776257334e-17)
    (-0.9992278070877242 . 3.114294486006144e-17)
    (-0.9993827412675937 . -1.8003741649909573e-17)
    (-0.9995100301644678 . -1.1351588621731435e-17)
    (-0.9996138668072848 . 2.86709228121328e-17)
    (-0.999697954290198 . 3.655284977454683e-17)
    (-0.9997655366626393 . -1.1274704894998538e-17)
    (-0.9998194324171634 . -3.2318508196837294e-18)
    (-0.9998620696692346 . -4.625633465510879e-17)
    (-0.9998955221695238 . -4.3085570938583103e-17)
    (-0.9999215453570454 . 4.1926814388898224e-17)
    (-0.9999416117467417 . -1.42988109080781e-17)
    (-0.9999569450435333 . 3.994008776201599e-17)
    (-0.9999685524817359 . -2.1315811752046272e-17)
    (-0.9999772549994149 . 3.9091222055449664e-17)
    (-0.9999837149671728 . -1.8233879155653072e-17)
    (-0.9999884612959822 . -3.2841618774310393e-17)
    (-0.9999919118454622 . -1.1104512988790167e-17)
    (-0.9999943931396973 . 4.0361849614562793e-17)
    (-0.9999961574703498 . -2.134042792321693e-17)
    (-0.999997397525303 . 1.3533076109041662e-17)
    (-0.999998258725082 . -5.407110478742794e-17)
    (-0.9999988494792682 . -4.3390217644143825e-17)
    (-0.9999992495920951 . 3.730548029465148e-17)
    (-0.9999995170519197 . 3.3188757935647772e-18)
    (-0.9999996934351678 . 2.0224874192213467e-17)
    (-0.9999998081436597 . 1.438749764251143e-17)
    (-0.9999998816770032 . -6.987630739479885e-18)
    (-0.9999999281209097 . -4.8283410460647105e-17)
    (-0.9999999570096083 . 3.6506793474258766e-17)
    (-0.9999999746974519 . 2.107938803420783e-17)
    (-0.9999999853525123 . 2.7390032102353362e-17)
    (-0.9999999916642777 . 5.605733083081871e-18)
    (-0.9999999953390686 . 3.790820747547506e-17)
    (-0.9999999974407513 . -1.7823127048251318e-17)
    (-0.9999999986208357 . -4.748354681027246e-17)
    (-0.9999999992709975 . -4.7891545352614994e-17)
    (-0.9999999996222632 . -5.177147691724898e-17)
    (-0.9999999998082539 . -2.6963028933845586e-18)
    (-0.9999999999047069 . 3.645589784367344e-17)
    (-0.9999999999536652 . -2.102246580402669e-17)
    (-0.9999999999779724 . -1.4578586000249962e-17)
    (-0.9999999999897686 . 1.8931117745502415e-17)
    (-0.9999999999953603 . 2.2586090325480838e-17)
    (-0.9999999999979473 . -2.707782230578898e-18)
    (-0.9999999999991147 . -6.697416717863371e-18)
    (-0.9999999999996281 . -2.1847548217979098e-17)
    (-0.9999999999998479 . -4.389412098860936e-17)
    (-0.9999999999999395 . -5.063859677654805e-17)
    (-0.9999999999999767 . 3.942076671643759e-17)
    (-0.9999999999999912 . -1.337442136942254e-17)
    (-0.9999999999999968 . -3.490832470810129e-17)
    (-0.9999999999999989 . 1.1911245891949965e-17)
    (-0.9999999999999997 . 4.963332588729193e-17)
    (-0.9999999999999999 . 1.5181680740804442e-17)
    (-1.0 . 4.0200513017888305e-17)
    (-1.0 . 1.2355599792293348e-17)
    (-1.0 . 3.660029429583496e-18)
    (-1.0 . 1.0437427468932139e-18)
    (-1.0 . 2.8620322059163255e-19)
    (-1.0 . 7.536949322264369e-20)
    (-1.0 . 1.9037414132224144e-20)
    (-1.0 . 4.606230918723704e-21)
    (-1.0 . 1.0661670573173551e-21)
    (-1.0 . 2.357462530668593e-22)
    (-1.0 . 4.972601982525333e-23)
    (-1.0 . 9.990838514493427e-24)
    (-1.0 . 1.909147574124114e-24)
    (-1.0 . 3.464301918629924e-25)
    (-1.0 . 5.959759382869178e-26)
    (-1.0 . 9.704085710359217e-27)
    (-1.0 . 1.4929558185834875e-27)
    (-1.0 . 2.16638641231196e-28)
    (-1.0 . 2.9595635553379995e-29)
    (-1.0 . 3.7992995430783965e-30)
    (-1.0 . 4.574243408007208e-31)
    (-1.0 . 5.154710128939581e-32)
    (-1.0 . 5.425760641374366e-33)
    (-1.0 . 5.323074268438217e-34)
    (-1.0 . 4.856854626017807e-35)
    (-1.0 . 4.112005638501019e-36)
    (-1.0 . 3.2228706814304173e-37)
    (-1.0 . 2.3327890584950366e-38)
    (-1.0 . 1.5555055969050827e-39)
    (-1.0 . 9.530541239149814e-41)
    (-1.0 . 5.351370112995799e-42)
    (-1.0 . 2.7461832867933594e-43)
    (-1.0 . 1.284369007315253e-44)
    (-1.0 . 5.4586641410044394e-46)
    (-1.0 . 2.1019400400312647e-47)
    (-1.0 . 7.310586300709408e-49)
    (-1.0 . 2.2892894917392245e-50)
    (-1.0 . 6.4334119603663054e-52)
    (-1.0 . 1.6169818855396586e-53)
    (-1.0 . 3.6222397384786406e-55)
    (-1.0 . 7.206016097986095e-57)
    (-1.0 . 1.2683765492843837e-58)
    (-1.0 . 1.9677716993955896e-60)
    (-1.0 . 2.680164311533663e-62)
    (-1.0 . 3.191842581968051e-64)
    (-1.0 . 3.3097314751182344e-66)
    (-1.0 . 2.9753378277376716e-68)
    (-1.0 . 2.3085234617081637e-70)
    (-1.0 . 1.5388223638575858e-72)
    (-1.0 . 8.770778222904261e-75)
    (-1.0 . 4.253613945711465e-77)
    (-1.0 . 1.746452430408311e-79)
    (-1.0 . 6.039126957671881e-82)
    (-1.0 . 1.749361652381559e-84)
    (-1.0 . 4.221527179358914e-87)
    (-1.0 . 8.438505045627658e-90)
    (-1.0 . 1.3890289959826033e-92)
    (-1.0 . 1.8714190249358942e-95)
    (-1.0 . 2.0508145275329015e-98)
    (-1.0 . 1.8162435490055362e-101)
    (-1.0 . 1.2912884751044706e-104)
    (-1.0 . 7.319691075989751e-108)
    (-1.0 . 3.2847940909116385e-111)
    (-1.0 . 1.1585079822572272e-114)
    (-1.0 . 3.187101356583876e-118)
    (-1.0 . 6.786205335253e-122)
    (-1.0 . 1.109469428988187e-125)
    (-1.0 . 1.3812592336764689e-129)
    (-1.0 . 1.298400858897444e-133)
    (-1.0 . 9.134879417118437e-138)
    (-1.0 . 4.766776416446869e-142)
    (-1.0 . 1.827760579796531e-146)
    (-1.0 . 5.1004005602277926e-151)
    (-1.0 . 1.0255738026920948e-155)
    (-1.0 . 1.4708203663521784e-160)
    (-1.0 . 1.4886612146520806e-165)
    (-1.0 . 1.0518304626907384e-170)
    (-1.0 . 5.130167546980931e-176)
    (-1.0 . 1.707352362929865e-181)
    (-1.0 . 3.831186191113585e-187)
    (-1.0 . 5.725486869602135e-193)
    (-1.0 . 5.626574339653866e-199)
    (-1.0 . 3.588716803231487e-205)
    (-1.0 . 1.465652583234918e-211)
    (-1.0 . 3.7798059795032864e-218)
    (-1.0 . 6.067578864133267e-225)
    (-1.0 . 5.973579693952893e-232)
    (-1.0 . 3.5521557119111035e-239)
    (-1.0 . 1.2558705316392284e-246)
    (-1.0 . 2.59738840369466e-254)
    (-1.0 . 3.0902326394300804e-262)
    (-1.0 . 2.0787554436363743e-270)
    (-1.0 . 7.766668266238272e-279)
    (-1.0 . 1.582357434509706e-287)
    (-1.0 . 1.7249752189434757e-296)
    (-1.0 . 9.866936306875695e-306)
    (-1.0 . 2.90235496098e-315)))

