/* 
 *  rocks/iop.c
 *
 *  Interoperability of rocks with ordinary sockets.
 *
 *  Copyright (C) 2001 Victor Zandy
 *  See COPYING for distribution terms.
 */
#include <assert.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>

#include "rs.h"
#include "log.h"

#define ROCKIDLEN 16
static char rockid[ROCKIDLEN] = "IROCKYOUROCKMAN";

struct iop {
	iop_type_t type;
	rs_t rs;		     /* owning rock */

	/* Type IOP_LISTEN only */
	rocklist_t known;            /* list of known rocks */
	
	/* Type IOP_SERVER only */
	rs_t parent;                 /* server from which this rock was created */
	int dontclose;               /* keep around until reconnection */
};

/* FIXME: Update associated IOP data.
   Define semantics of RS; whether it must be closed, etc. */
void
rs_free_iop(rs_t rs)
{
	iop_t iop;

	if (!rs->iop)
		return;
	iop = rs->iop;
	rs->iop = NULL;
	rs_rocklist_free(&iop->known);
	free(iop);
}

int
rs_iop_listener(rs_t rs)
{
	iop_t iop;
	iop = (iop_t) malloc(sizeof(struct iop));
	if (!iop)
		return -1;
	iop->rs = rs;
	rs->iop = iop;
	iop->type = IOP_LISTEN;
	iop->known = NULL;
	iop->parent = NULL;
	iop->dontclose = 0;
	return 0;
}

int
iop_new_server(rs_t listener, rs_t rs)
{
	iop_t iop;
	iop = (iop_t) malloc(sizeof(struct iop));
	if (!iop)
		return -1;
	iop->rs = rs;
	rs->iop = iop;
	iop->type = IOP_SERVER;
	iop->parent = listener;
	iop->known = NULL;
	iop->dontclose = 0;
	return 0;
}


static int
readid(rs_t rs)
{
	char *p, buf[ROCKIDLEN];
	int rv, n;
	ring_t ring;
	fd_set fds;
	struct timeval tv;

	/* FIXME: Free this ring. */
	ring = rs_new_ring(ROCKIDLEN);
	if (!ring)
		return -1;

	while (1) {
		FD_ZERO(&fds);
		FD_SET(rs->sd, &fds);
		tv.tv_sec = 0;
		tv.tv_usec = 500000;
		n = select(rs->sd+1, &fds, NULL, NULL, &tv);
		if (0 > n && errno == EINTR)
			continue;
		if (0 > n) {
			rs_log("IOP: client got select error waiting for announcement");
			return -1;
		}
		if (0 == n) {
			rs_log("IOP: client timed out waiting for announcement");
			return -1; /* timeout */
		}
		rv = read(rs->sd, buf, sizeof(buf));
		if (0 == rv) {
			rs_log("IOP: client got EOF from server but no announcement");
			return -1;
		}
		if (0 > rv && errno == EINTR)
			continue;
		if (0 > rv) {
			assert(errno != EWOULDBLOCK);
			return -1;
		}
		rs_push_ring(ring, buf, rv);
		p = rs_ring_data(ring);
		n = rs_ring_nbytes(ring);
		if (n < ROCKIDLEN)
			continue;
		p += n - ROCKIDLEN;
		if (!strncmp(p, rockid, ROCKIDLEN))
			return 0;
	}
}

int
rs_iop_connect(rs_t rs)
{
	int td;
	int isrock;

	shutdown(rs->sd, 1);
	isrock = 0 > readid(rs) ? 0 : 1;
	td = socket(AF_INET, SOCK_STREAM, 0);
	if (0 > td) {
		rserrno = errno;
		return -1;
	}
	if (!isrock) {
		rs_log("rock <%p> peer is not a rock", rs);
		rs_log("rock <%p> reconnecting as ordinary socket", rs);
		if (0 > connect(td, (struct sockaddr *)&rs->sa_peer,
				sizeof(struct sockaddr_in)))
			return -1;
		if (td != rs->sd) {
			close(rs->sd);
			if (0 > dup2(td, rs->sd)) {
				rserrno = errno;
				return -1;
			}
			close(td);
		}
		rs_fallback(rs);
		return 0;
	}

	rs_log("client: iop server is a rock");
	if (0 > rs_reset_on_close(rs->sd, 1)) {
		rserrno = errno;
		return -1;
	}
	close(rs->sd);
	assert(td != rs->sd);
	if (0 > dup2(td, rs->sd)) {
		rserrno = errno;
		return -1;
	}
	close(td);

	/* Reuse the local address (set in rs_connect) */
	if (0 > bind(rs->sd, (struct sockaddr *)&rs->sa_locl, sizeof(struct sockaddr_in))) {
		rserrno = errno;
		return -1;
	}
	if (0 > connect(rs->sd, (struct sockaddr *)&rs->sa_peer, sizeof(struct sockaddr_in))) {
		rserrno = errno;
		return -1;
	}
	rs_log("IOP reconnect as rock");
	if (0 > rs_init_connection(rs))
		return -1;
	return 0;
}

/* PARENT is a listener that is being closed.  If PARENT is RS's IOP
   parent, clean up RS's IOP state. */
int
rs_iop_chk_parent_closed(rs_t parent, rs_t rs)
{
	assert(rs->iop);
	assert(rs->iop->parent);

	if (parent != rs->iop->parent)
		return 0;

	if (rs->iop->dontclose) {
		/* already scheduled for closing but held for
		   checking new connections on the listener.
		   now do the close. */
		rs->iop->dontclose = 0;
		rs_free_iop(rs);
		return rs_close(rs->sd);
	}
	rs_fallback(rs);
	return 0;
}

/* Handle EOF on a server rock with pending interoperability check.
   FIXME:
    - This may be called several times for the same rock if the server
      is dumb and calls read multiple times on a EOFed socket.
 */
int
rs_iop_eof(rs_t rs)
{
	rs_t srv_rs;
	iop_t iop = rs->iop;

	assert(rs->role == RS_ROLE_SERVER);
	assert(iop);
	srv_rs = iop->parent;
	assert(srv_rs);
	assert(srv_rs->iop);
	assert(srv_rs->role == RS_ROLE_LISTEN);
	assert(!rs_rocklist_findsa(srv_rs->iop->known, &rs->sa_peer));
	if (0 > rs_rocklist_insert(&srv_rs->iop->known, rs))
		return -1;
	if (0 > rs_xwrite(rs->sd, rockid, ROCKIDLEN)) {
		rs_log("IOP: Error writing ID string to known rock");
		return -1;
	}
	rs_log("Wrote announcement to client");
	rs->iop->dontclose = 1;
	if (0 > rs_reset_on_close(rs->sd, 1))
		return -1;
	rs_log("IOP: Server <%p> detected rock client", rs);
	return 0;
}

int
rs_iop_accept(rs_t srv_rs, rs_t rs)
{
	unsigned char byte;
	rs_t old;
	rs_log("IOP: <%p> accept connection <%p> from %s",
	       srv_rs, rs, rs_ipstr(&rs->sa_peer));
	old = rs_rocklist_findsa(srv_rs->iop->known, &rs->sa_peer);
	if (old) {
		rs_log("IOP: known rock");
		rs_rocklist_remove(&srv_rs->iop->known, old);
		old->iop->dontclose = 0;
		rs_close(old->sd);
		rs_free_iop(rs);
		return rs_init_connection(rs);
	} else {
		rs_log("IOP: unknown to <%p>", srv_rs);
		if (0 > iop_new_server(srv_rs, rs)) {
			rserrno = errno = ENOMEM;
			return -1;
		}

		/* FIXME: a hack to deal with servers that fork. */
		/* Give the client a few moments to be a rock. */
		if (0 > rs_waitread(rs->sd, 100))
			return 0;
		if (0 == recv(rs->sd, &byte, 1, MSG_PEEK)) {
			rs_log("Got EOF fork hack during accept...moving into IOP");
			return rs_iop_eof(rs);
		}

		return 0;
	}
}

int
rs_iop_isdontclose(rs_t rs)
{
	return rs->iop && rs->iop->dontclose;
}
