/*				       	-*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: cache.c,v 1.17 2000/08/16 10:08:58 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool <mbp@linuxcare.com>
 * Copyright (C) 1999 by Andrew Tridgell <tridge@linuxcare.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"

#include <sys/file.h>
#include <sys/param.h>

#include "sysheaders.h"

#include "rproxy.h"
#include "cache.h"
#include "util.h"
#include "urllib.h"
#include "proto.h"
#include "trace.h"
#include "error.h"

/* Cache handling code.

   Each request URL maps directly into a body file containing the contents of 
   the request, and a signature file containing the server-generated
   signature.  Remember that the cached value is only ever a basis for doing
   a differential transfer from upstream: we *never* return it verbatim to
   the client.

   We don't need to cache response headers because we always send the request 
   upstream and therefore can always return up-to-date response headers.

   As data is received it is written into temporary files.

   Because there are now two files we can't do an atomic-rename anymore.
   Instead, we grab a lock on the cache directory, rename both files, and
   then release the lock.  I'm sure there are better ways but this will do.
   Said lock is an fcntl lock over the entire extent of the directory, i.e.
   extent (0, 0).  We must also hold this lock while opening files, to make
   sure that we get corresponding signature and body files.  However, once
   we've opened the files we can release the lock, because we never modify
   files and so the file handle will remain valid.

   The cache file name is just the base64-encoded MD4 hash of the URL,
   without query parameters. */

/* 
   TODO: Don't just crash if an error occurs. :-/ */

/* FIXME: permissions on cache file seem very random
 *
 * FIXME: Don't use stdio */

static char const *cache_root = "/tmp/rproxy-cache";
static char     cache_tname[1024], sig_tname[PATH_MAX],
    cache_real_name[PATH_MAX], sig_real_name[PATH_MAX];

static int      root_fd;


/* Prepare to use a given cache directory. */
int
cache_init(char const *root_dir)
{
    cache_root = root_dir;

    trace(LOGAREA_CACHE, "initialize cache directory %s", root_dir);

    /* If this fails, the most likely cause is that the directory already
       exists, which is fine.  We'll trap errors later if anything else is
       wrong. */
    mkdir(cache_root, 0775);

    root_fd = open(cache_root, O_RDONLY);
    if (root_fd < 0) {
	rp_log(LOGAREA_PROCESS, LOG_ERR, "can't open cache root %s: %s",
               cache_root, strerror(errno));
        exit(1);
    }

    return 0;
}



/* Convert a URL into a cache filename; return the length of the string. */
static int
cache_url_to_filename(char const *url, char *buf)
{
    unsigned char   md4buf[16];
    char            b64[23];
    int             good_part;

    /* Count characters up to the end of the string or the first query
       character. */
    good_part = strcspn(url, "?");

    hs_mdfour(md4buf, (unsigned char const *) url, good_part);
    base64_encode(md4buf, 16, b64);

    string_sub(b64, "/", "_");

    return slprintf(buf, PATH_MAX, "%s/%s", cache_root, b64);
}


/* we use the complete request line hashed then base64 encoded to give the
   name of the cache file */
static void
cache_mangle_name(char const *request)
{
    cache_url_to_filename(request, cache_real_name);
    sprintf(sig_real_name, "%s.sig", cache_real_name);
}


int
cache_open_sig_tmp(char const *url)
{
    int             len;
    int fd;

    len = cache_url_to_filename(url, sig_tname);
    slprintf(sig_tname + len, sizeof sig_tname, ".sigtmp.%d", getpid());

    fd = open(sig_tname, O_CREAT | O_WRONLY | O_TRUNC, 0600);
    if (fd == -1) {
        rp_request_failed(NULL, HTTP_BAD_GATEWAY,
                       "couldn't open %s: %s\n", sig_tname, strerror(errno));
    }
    trace(LOGAREA_CACHE, "open temporary sig file %s", sig_tname);
    return fd;
}


FILE *
cache_open_tmp(char const *url)
{
    cache_mangle_name(url);
    slprintf(cache_tname, sizeof(cache_tname), "%s.tmp%u",
             cache_real_name, (unsigned) getpid());
    trace(LOGAREA_CACHE, "open temporary body file %s", cache_tname);
    return fopen(cache_tname, "wb");
}


static int
cache_lock(void)
{
    struct flock    l = {
	F_WRLCK, SEEK_SET, 0, 0, 0
    };

    return fcntl(root_fd, F_SETLKW, &l);
}


static int
cache_unlock(void)
{
    struct flock    l = {
	F_UNLCK, SEEK_SET, 0, 0, 0
    };

    return fcntl(root_fd, F_SETLKW, &l);
}


int
cache_open(request_t *req, char const *url, FILE ** body_fp, FILE ** sig_fp)
{
    int             ok;

    assert(url && body_fp && sig_fp);

    cache_mangle_name(url);
    cache_lock();
    trace(LOGAREA_CACHE, "cache_open file=%s", cache_real_name);
    /* TODO: Rather than this, construct an in-memory hs_io object that acts
       like /dev/null. */
    *body_fp = fopen(cache_real_name, "rb");
    if (!*body_fp && errno == ENOENT)
	*body_fp = fopen("/dev/null", "rb");
    *sig_fp = fopen(sig_real_name, "rb");
    if (!*sig_fp && errno == ENOENT)
	*sig_fp = fopen("/dev/null", "rb");
    cache_unlock();
    ok = *body_fp && *sig_fp;
    if (!ok) {
	rp_request_failed(req, HTTP_BAD_GATEWAY,
                          "couldn't open old cache %s: %s",
                          cache_real_name,
                          strerror(errno));
    }
    return ok;
}


void
cache_tmp_commit(void)
{
    cache_lock();

    trace(LOGAREA_CACHE, "commit %s to %s", cache_tname, cache_real_name);
    if (rename(cache_tname, cache_real_name) != 0) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "can't commit %s to %s: %s",
               cache_tname, cache_real_name, strerror(errno));
    }

    trace(LOGAREA_CACHE, "commit %s to %s", sig_tname, sig_real_name);
    if (rename(sig_tname, sig_real_name) != 0) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "can't commit %s to %s: %s",
               sig_tname, sig_real_name, strerror(errno));
    }

    cache_unlock();
}



void
cache_tmp_rollback(void)
{
    unlink(cache_tname);
    unlink(sig_tname);
}
