# Samizdat engine deployment functions
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'


# todo: document this class
# 
class ConfigHash < Hash

  # pre-load with supplied hash
  #
  def initialize(hash)
    super()
    self.merge!(hash)
  end

  # translate to option or ConfigHash of suboptions
  #
  def method_missing(name)
    value = self[name.to_s]
    case value
    when nil then nil
    when Hash then ConfigHash.new(value)
    else value
    end
  end

  # traverse the tree of sub-hashes and only update modified values
  #
  def deep_update!(hash)
    hash.each do |key, value|
      if value.kind_of? Hash then
        case self[key]
        when ConfigHash then self[key].deep_update!(value)
        when Hash then self[key] = ConfigHash.new(self[key]).deep_update!(value)
        else self[key] = value   # scalar or nil is discarded in favor of hash
        end
      else
        self[key] = value
      end
    end
    self
  end
end


# helper method to load YAML data from file
#
def load_yaml_file(filename, trusted = false)
  File.open(filename) {|f| YAML.load(trusted ? f.read.untaint : f) }
end

# load and operate mapping of server prefixes to Samizdat site names
#
class SamizdatSites
  include Singleton

  SITES_MAP = '/etc/samizdat/sites.yaml'

  # loads Samizdat sites mapping from /etc/samizdat/sites.yaml
  #
  def initialize
    return unless File.readable?(SITES_MAP)

    @sites = {}
    sites_raw = load_yaml_file(SITES_MAP, true)
    sites_raw.each do |server_name, map|
      @sites[server_name] = sites_raw[server_name].keys.sort_by {|p|
        -p.size
      }.collect {|prefix|
        site = map[prefix]

        # normalize prefix
        if '/' == prefix
          prefix = ''
        elsif '/' == prefix[-1, 1]
          prefix.sub!(%r{/+\z}, '')
        end

        [ prefix,
          Regexp.new(/\A#{Regexp.escape(prefix)}(.*)\z/).freeze,
          site
        ]
      }
    end
  end

  # determine site name, URI prefix, and URI tail (used to determine route to
  # dispatch) from CGI variables
  #
  # prefixes are sorted by descending length so that more specific prefixes are
  # tried before shorter ones
  #
  def find(server_name, request_uri)
    # fall back to config.yaml in current directory
    return unless @sites.kind_of? Hash and @sites[server_name].kind_of? Array

    @sites[server_name].each do |prefix, prefix_pattern, site|
      match = prefix_pattern.match(request_uri)
      if match.kind_of? MatchData
        return site, prefix, match[1]
      end
    end
  end

  # return list of all site names
  #
  def all
    sites = {}
    @sites.each {|server_name, map|
      map.each {|prefix, prefix_pattern, site| sites[site] = true }
    }
    sites.keys
  end
end

# name of site being accessed by current request
#
# can be overridden by SAMIZDAT_SITE environment variable
#
# can be amended to fall back to config.yaml in the current directory on the
# server to identify the site (see method source)
#
# using unsafe file paths has security implications: make sure Samizdat CGI
# scripts can only be reached through pre-determined locations
#
def site_name
  ENV['SAMIZDAT_SITE'] or (
    $samizdat_current_request.kind_of?(Request) and
    $samizdat_current_request.site_name)
  # or (Dir.pwd + '/config.yaml').untaint
end

# load, merge in, and cache site configuration from rdf.yaml, defaults.yaml,
# and site.yaml (in that order), cache xhtml.yaml
#
class SiteConfig < SimpleDelegator

  # look up config files in one of given directories, by default look in /etc,
  # /usr/share, /usr/local/share, and current directory
  DIRS = [ '/etc/samizdat/',
           Config::CONFIG['datadir'] + '/samizdat/',
           '/usr/local/share/samizdat/',
           '' ]

  # global RDF-relational mapping
  RDF = 'rdf.yaml'

  # default settings common for all sites
  DEFAULTS = 'defaults.yaml'

  # XHTML options for Samizdat::Sanitize
  XHTML = 'xhtml.yaml'

  # location of site-specific configs
  SITES_DIR = '/etc/samizdat/sites/'

  def initialize(site)
    @@rdf ||= load_yaml_file(find_file(RDF, DIRS), true)
    @@defaults ||= load_yaml_file(find_file(DEFAULTS, DIRS), true)

    @config = ConfigHash.new(@@rdf)
    @config.deep_update!(@@defaults)

    site_config =
      if site =~ %r{/.*/config\.yaml\z} and File.readable? site
        site   # config.yaml in current directory
      else
        File.join(SITES_DIR, site + '.yaml')
      end

    @config.deep_update!(load_yaml_file(site_config, true))

    if @config.cache and @config.cache =~ /\Adruby:/ then
      @drb = @config.cache
    end

    if @config.plugins
      @plugins = Plugins.new(@config.plugins)
    end

    super @config
  end

  attr_reader :drb, :plugins

  # pre-loaded Samizdat::Sanitize object for XHTML validation
  #
  def xhtml
    @@xhtml ||= Samizdat::Sanitize.new(load_yaml_file(find_file(XHTML, DIRS), true))
  end

  # check if all parameters necessary to send out emails are set
  #
  def email_enabled?
    @config['email'] and @config['email']['address'] and @config['email']['sendmail']
  end

  # content directory
  #
  # keep its parts (document root, site content config) under control:
  # Samizdat will write files under this directory!
  #
  def content_dir
    if @content_dir.nil?
      content = @config['site']['content']
      content.kind_of?(String) or return @content_dir = ''

      request = $samizdat_current_request
      @content_dir = request.filename(request.content_location).untaint
    end

    @content_dir
  end

  # check if multimedia content upload is possible
  #
  def upload_enabled?
    @config['site']['content'].kind_of? String and
      File.directory?(content_dir) and
      File.writable?(content_dir)
  end
end

class SiteConfigSingleton
  include Singleton

  def initialize
    @config = {}
  end

  # cache parsed SiteConfig by site_name
  #
  def config
    @config[site_name] ||= SiteConfig.new(site_name)
  end
end

# shortcut access to SiteConfig
#
def config
  SiteConfigSingleton.instance.config
end


# wraps Cache methods to prepend cache key with site_name
#
class SiteCache
  def initialize(cache)
    @cache = cache
  end

  def flush(base = Regexp.new('\A' + Regexp.escape(site_name) + '/'))
    @cache.flush(base)
  end

  def delete(key)
    @cache.delete(site_name + '/' + key)
  end

  def []=(key, value)
    @cache[site_name + '/' + key] = value
  end

  def [](key)
    @cache[site_name + '/' + key]
  end

  def fetch_or_add(key, &p)
    @cache.fetch_or_add(site_name + '/' + key, &p)
  end
end

# puts a SiteCache wrapper around cache objects
#
class CacheSingleton
  include Singleton

  # size limit for in-process cache
  LOCAL_SIZE = 2000

  def initialize
    @drb = {}
    @cache = {}
    @persistent = {}
  end

  # explicitly request an in-process cache (e.g. for singleton classes)
  #
  def local
    @local ||= SiteCache.new(Samizdat::Cache.new(nil, LOCAL_SIZE))
  end

  # cache DRb connections by per-site DRb URI, keep same SiteCache-wrapped
  # in-process local cache
  #
  def cache
    @cache[site_name] ||=
      if config.drb then
        @drb[config.drb] ||= SiteCache.new(DRbObject.new(nil, config.drb))
      else
        local
      end
  end

  # persistent cross-site cache that is not flushed on site changes (e.g. for
  # RSS feeds)
  #
  # DRb connections are cached the same as in #cache, but SiteCache wrapper is
  # not used
  #
  def persistent
    if config.drb then
      @persistent[config.drb] ||= DRbObject.new(nil, config.drb)
    else
      local
    end
  end
end

# shortcut access to Cache object
#
def cache
  CacheSingleton.instance.cache
end

# shortcut to explicitly request an in-process cache
#
def local_cache
  CacheSingleton.instance.local
end

# shortcut to explicitly request persistent cache
#
def persistent_cache
  CacheSingleton.instance.persistent
end


# database connection management
#
# permanently keeps DB connections in in-process cache
#
def db
  # todo: verify concurrency in db and storage
  # todo: check connection timeouts
  # optimize: generate connection pool
  local_cache.fetch_or_add('database/' + site_name) do
    db = DBI.connect(config['db']['dsn'],
      (ENV['USER'] or config['db']['user']),
      ENV['USER']? nil : config['db']['password'])
    begin
      db['AutoCommit'] = false
      db['quote_boolean'] = Proc.new {|value| value ? "'true'" : "'false'" }
      db['detect_boolean'] = Proc.new do |column_info, value|
        ::DBI::SQL_VARCHAR == column_info['sql_type'] and
          5 == column_info['precision'] and
          ['true', 'false'].include?(value.downcase)
      end
    rescue DBI::NotSupportedError
      # no need to disable if it's not there
    end
    db
  end
end

# rdf storage access shortcut
#
def rdf
  local_cache.fetch_or_add('rdf:' + site_name) do
    Samizdat::RDF.new(db, config)
  end
end
