# Samizdat session management
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

# session management
#
class Session

  # cookie is a hex digest between 128 and 2048 bits long
  COOKIE_PATTERN = Regexp.new(/\A[[:alnum:]]{32,512}\z/).freeze

  # start new session on login
  #
  # returns cookie value on success, nil on failure
  #
  def Session.start(login, password)
    db.transaction do
      id, p = db.select_one(
        'SELECT id, password FROM Member WHERE login = ?', login)

      if id and p.nil?
        raise AccountBlockedError
      elsif id.nil? or not Password.check(password, p)
        return nil
      end

      cookie = random_digest(id)
      db.do(
        'UPDATE Member
        SET login_time = current_timestamp,
            last_time = current_timestamp,
            session = ?
        WHERE id = ?', cookie, id)
      db.commit
      return cookie
    end
  end

  def Session.cache_key(cookie)
    'session/' + cookie
  end

  # set _@login_ and _@full_name_ to 'guest'
  #
  def reset_to_guest
    @cookie = nil
    @member = nil
    copy_from_member_object(Member.new(nil))
  end

  # Usage:
  #
  #   Session.new(request.cookie('session'))
  #
  def initialize(cookie)
    reset_to_guest

    if cookie and cookie =~ COOKIE_PATTERN
      @cookie = cookie

      db.transaction do |db|
        @member, @login_time = db.select_one(
          'SELECT id, login_time
             FROM Member
            WHERE session = ?', @cookie)

        @member and fresh!
      end

      if @member
        copy_from_member_object(Member.cached(@member))
      end
    end
  end

  attr_reader :member, :login, :full_name

  # true if moderator priviledges are available to this member
  #
  def moderator?
    @moderator
  end

  # erase session from database, remove the session cookie
  #
  def clear!
    if @member
      db.do "UPDATE Member SET session=NULL WHERE id = ?", @member
      db.commit
    end
    if @cookie
      cache.delete(Session.cache_key(@cookie))
    end
    reset_to_guest
  end

  # check if session is still fresh, clear it if it's not
  #
  def fresh!
    if @login_time and @login_time.to_time < Time.now - config['timeout']['login']
      clear!   # remove stale session
      false
    else
      true
    end
  end

  private

  def copy_from_member_object(member)
    @login = member.login
    @full_name = member.full_name
    @moderator = member.allowed_to?('moderate')
  end
end
