/*
 * Copyright 1995,96,97 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: samain.c,v 4.8 1997/04/18 15:54:04 bousch Exp $
 *
 * Main module for the "samuel" desk calculator
 */

#include <errno.h>
#include <getopt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/times.h>
#include "saml.h"
#include "saml-parse.h"
#include "saml-util.h"
#include "samuel.h"
#include "tokens.h"

#ifdef USE_READLINE
#define VERSION "1.21r"
#else
#define VERSION "1.21"
#endif

mref_t mr_model;	/* Will contain the model */
unsigned int modulo;
int explicit_variables = 0;
int call_induce = 1;
int use_tensors = 0;
int use_rat_fns = 0;
int quiet;
const char *post_processor = NULL;
int poly_type;

static int max_stack_depth = 0;
static int stack_depth = 0;
static mref_t *sam_stack = NULL;
static char *condition = NULL;

void panic (const char *msg)
{
#define ERR_MSG(x) write(2,x,strlen(x))
	/*
	 * Don't rely on stdio, it might allocate memory, and memory
	 * could be completely exhausted at this point
	 */
	ERR_MSG("panic: "); ERR_MSG(msg); ERR_MSG("\n");
	abort();
}

void push_mref (mref_t mr)
{
	if (stack_depth >= max_stack_depth) {
		max_stack_depth += (max_stack_depth + 16);
		sam_stack = realloc(sam_stack,
			max_stack_depth * sizeof(mref_t));
		if (sam_stack == NULL)
			panic("stack overflow (out of memory)");
	}
	sam_stack[stack_depth++] = mr;
}

mref_t pop_mref (void)
{
	if (stack_depth <= 0)
		panic("stack underflow");
	return sam_stack[--stack_depth];
}

void clean_stack (void)
{
	if (stack_depth == 0)
		return;
	fprintf(stderr, "clean_stack: collect %d item(s)\n", stack_depth);
	while (stack_depth)
		mref_free(pop_mref());
}

static void print_mnode_statistics (void)
{
	if (!quiet) fprintf(stderr,
		"Allocated-reserved-freed mnodes = %ld-%ld-%ld\n",
		nb_mnodes_allocated,
		nb_mnodes_reserved,
		nb_mnodes_freed);
}

static void import_environment (void)
{
	char *variable;

	variable = getenv("MODULUS");
	if (variable)
		modulo = strtoul(variable, NULL, 10);

	variable = getenv("LITERALS");
	if (variable) {
		if (!strcmp(variable, "dense"))
			poly_type = ST_APOLY;
		if (!strcmp(variable, "sparse"))
			poly_type = ST_POLY;
		/* Other values are invalid, and silently ignored. */
	}
}

static void export_environment (void)
{
	char buff[32];

	sprintf(buff, "%u", modulo);
	setenv("MODULUS", buff, 1);

	sprintf(buff, "%u", getpid());
	setenv("SPID", buff, 1);
	
	switch(poly_type) {
	    case ST_APOLY:
	    	setenv("LITERALS", "dense", 1);
	    	break;
	    case ST_POLY:
	    	setenv("LITERALS", "sparse", 1);
	    	break;
	}
}

static void parse_condition (void)
{
	mref_t cond, pol, coeff;
	int i, n;
	char *p, *literal;

	if (!condition) {
		mref_cast(mr_model, poly_type);
		if (use_rat_fns)
			mref_cast(mr_model, ST_RATIONAL);
		if (use_tensors)
			mref_cast(mr_model, ST_TENSOR);
		return;
	}
	for (n = -1, p = condition; *p; p++)
		if (*p == ',') ++n;
	cond = mref_new(); pol = mref_new(); coeff = mref_new();
	mref_array(cond, ST_UPOLY, n+1);
	literal = strtok(condition, ",");
	for (i = 0; (p = strtok(NULL,",")); i++) {
		mref_build(coeff, ST_INTEGER, p);
		mref_promote(coeff, mr_model);
		mref_setitem(cond, i, coeff);
	}
	mref_array(pol, ST_UPOLY, 2);
	mref_zero(coeff, mr_model);
	mref_setitem(pol, 0, coeff);
	mref_one(coeff, mr_model);
	mref_setitem(pol, 1, coeff);
	mref_copy(mr_model, cond);
	mref_cast(mr_model, ST_ALGEXT);
	mref_cast(mr_model, poly_type);
	if (use_rat_fns)
		mref_cast(mr_model, ST_RATIONAL);
	if (use_tensors)
		mref_cast(mr_model, ST_TENSOR);
	mref_promote(pol, mr_model);
	assign_to_variable(literal, pol);
	mref_free(cond); mref_free(pol); mref_free(coeff);
}

void print_cpu_usage (void)
{
	struct tms t;

	if (quiet) return;
	print_mnode_statistics();
	times(&t);
	fprintf(stderr, (t.tms_cutime || t.tms_cstime) ?
	  "CPU usage: %.3fu + %.3fs (self), %.3fu + %.3fs (children)\n" :
	  "CPU usage: %.3fu + %.3fs\n",
	  t.tms_utime/(double)CLK_TCK, t.tms_stime/(double)CLK_TCK,
	  t.tms_cutime/(double)CLK_TCK, t.tms_cstime/(double)CLK_TCK);
}

static const char sh_opt[] = "bc:eiqvhVZQm:rtn:";
static const struct option lg_opt[] = {
	{ "batch",		0, NULL, 'b' },
	{ "escape-variables",	0, NULL, 'e' },
	{ "escape",		0, NULL, 'e' },
	{ "no-induce",		0, NULL, 'i' },
	{ "quiet",		0, NULL, 'q' },
	{ "silent",		0, NULL, 'q' },
	{ "verbose",		0, NULL, 'v' },
	{ "integers",		0, NULL, 'Z' },
	{ "rationals",		0, NULL, 'Q' },
	{ "modulo",		1, NULL, 'm' },
	{ "condition",		1, NULL, 'c' },
	{ "tensors",		0, NULL, 't' },
	{ "rational-functions",	0, NULL, 'r' },
	{ "nice",		1, NULL, 'n' },
	{ "sparse-literals",	0, NULL, 160 },
	{ "sparse",		0, NULL, 160 },
	{ "dense-literals",	0, NULL, 161 },
	{ "dense",		0, NULL, 161 },
	{ "version",		0, NULL, 162 },
	{ "copyright",		0, NULL, 162 },
	{ "help",		0, NULL, 163 },
	{ "usage",		0, NULL, 163 },
	{ "post-processor",	1, NULL, 164 },
	{ "post",		1, NULL, 164 },
	{ NULL, 0, NULL, 0 }
};
static const char Copyright[] = "\
Samuel "VERSION" (compiled "__DATE__") Copyright (C) 1995,96,97 Thierry Bousch
This program is FREE SOFTWARE, with absolutely NO WARRANTY.
You are welcome to redistribute it under certain conditions; read
the file \"COPYING\" for details.
";
static const char Usage[] = "\
Usage: %s [options]
Options:
  -b, --batch			Equivalent to -eiq
  -e, --escape			Variable names must be escaped
  -i, --no-induce		Don't call induce for subscripted names
  -q, --quiet			Don't print debugging messages, only errors
  -v, --verbose			Print debugging messages too
      --post PROGRAM		Post-process the output
      --help			Display this menu, and exit
      --version			Display the version number, and exit
  -Z, --integers		Use polynomials with integer coefficients
  -Q, --rationals		or with rational coefficients
  -m, --modulo NUMBER		or with coefficients in Z/nZ
  -c, --condition X,A0,A1...An	Compute modulo A0+A1.X+...+An.X^n
  -r, --rational-functions	Use rational functions, not polynomials
  -t, --tensors			Compute on tensors
  -n, --nice NUMBER		Be nice to other processes
      --sparse			Optimize for a sparse set of literals
      --dense			or for a dense set (the default)
";

int main (int argc, char **argv)
{
	int c, errs;
	char buff[32];

	quiet = !isatty(1);
	modulo = 0;
	poly_type = ST_APOLY;
	import_environment();

	/* Parse arguments */
	while ((c = getopt_long(argc,argv,sh_opt,lg_opt,NULL)) != EOF)
	    switch(c) {
	    case 'b':
	    	/* Batch mode */
	    	explicit_variables = 1;
	    	call_induce = 0;
	    	quiet = 1;
	    	break;
	    case 'e':
	    	/* Variable names must be escaped */
	    	explicit_variables = 1;
	    	break;
	    case 'i':
	    	/* Disable interpolation of subscripted variables */
	    	call_induce = 0;
	    	break;
	    case 'm':
	    	/* Modulo what? */
		modulo = strtoul(optarg, NULL, 10);
		break;
	    case 'Z':
	    	/* Coefficients are integers */
	    	modulo = 1;
	    	break;
	    case 'Q':
	    	/* Coefficients are rational numbers */
	    	modulo = 0;
	    	break;
	    case 't':
	    	/* Coefficients are tensors */
	    	use_tensors = 1;
	    	break;
	    case 'c':
	    	/* Coefficients are algebraic numbers */
	    	condition = optarg;
	    	break;
	    case 'r':
	    	/* Use rational functions instead of polynomials */
	    	use_rat_fns = 1;
	    	break;
	    case 160:
	    	/* Optimize for sparse literals */
		poly_type = ST_POLY;
	    	break;
	    case 161:
	    	/* Optimize for dense literals */
	    	poly_type = ST_APOLY;
	    	break;
	    case 'n':
	    	/* Modify niceness */
	    	if (nice(atoi(optarg)) < 0)
	    		perror("samuel: nice");
	    	break;
	    case 'q':
	    	/* Display only error messages */
	    	quiet = 1;
	    	break;
	    case 'v':
	    	/* Display also informational messages */
	    	quiet = 0;
	    	break;
	    case 162:
	    	/* Print the version number and the copyright notice */
		fputs(Copyright, stdout);
	    	return 0;
	    case 163:
	    	/* Display some help on standard output */
	    	printf(Usage, argv[0]);
	    	return 0;
	    case 164:
	    	/* Define a post-processor */
	    	post_processor = optarg;
	    	break;
	    default:
	        /* Usage error */
		fprintf(stderr, Usage, argv[0]);
		return 2;
	    }
	/* Non-option arguments are ignored */
	saml_init();
	mr_model = mref_new();
	print_mnode_statistics();
	switch(modulo) {
	    case 0:
	    	/* Arithmetics in Q */
	    	mref_build(mr_model, ST_RATIONAL, "0");
	    	break;
	    case 1:
	    	/* Arithmetics in Z */
	    	mref_build(mr_model, ST_INTEGER, "0");
	    	break;
	    default:
		/* Arithmetics in Z/nZ */
		sprintf(buff, "0:%u", modulo);
		mref_build(mr_model, ST_CYCLIC, buff);
		break;
	}
	parse_condition();
	export_environment();
	if (!quiet)
		fputs(Copyright, stderr);
	init_lexer();
	errs = yyparse();
	if (!quiet) {
		/*
		 * Free all mrefs, and verify that there are no mnodes left,
		 * except the reserved ones
		 */
		mref_free(mr_model);
		if (id_table) {
			sam_id *p;
			for (p = id_table; p; p = p->next)
				mref_free(p->mref);
		}
		print_mnode_statistics();
	}
	return errs;
}

static sam_id *id_table = NULL;

void list_variables (void)
{
	sam_id *p;

	printf("Vars:");
	for (p = id_table; p; p = p->next)
		printf(" %s(%d)", p->name, p->mref);
	printf(".\n");
}

int fn_variables (mref_t result, int argc, mref_t *argv)
{
	sam_id *p;
	char buffer[24];
	int nvars = 0;
	
	fprintf(stderr, "Vars:");
	for (p = id_table; p; p = p->next) {
		fprintf(stderr, " %s(%d)", p->name, p->mref);
		++nvars;
	}
	fprintf(stderr, ".\n");

	sprintf(buffer, "%d", nvars);
	mref_build(result, ST_INTEGER, buffer);
	mref_promote(result, mr_model);
	return 0;
}

int read_variable (mref_t mr, const char* var)
{
	sam_id *p;

	for (p = id_table; p; p = p->next)
		if (!strcmp(p->name,var)) {
			mref_copy(mr, p->mref);
			return 0;
		}
	/* No such variable */
	return -1;
}

int destroy_variable (const char* var)
{
	sam_id *p, **prev;

	for (prev = &id_table; (p = *prev) != NULL; prev = &(p->next))
		if (!strcmp(p->name,var)) {
			*prev = p->next;
			mref_free(p->mref);
			free(p);
			return 0;
		}
	/* No such variable */
	return -1;
}

void assign_to_variable (const char* var, mref_t mr)
{
	sam_id *p;
	for (p = id_table; p; p = p->next)
		if (!strcmp(p->name,var)) {
			mref_copy(p->mref, mr);
			return;
		}
	/* Create a new variable */
	p = malloc(sizeof(sam_id) + strlen(var) + 1);
	if (!p) panic("assign_to_variable: out of memory");
	strcpy(p->name, var);
	p->mref = mref_new();
	mref_copy(p->mref, mr);
	p->next = id_table;
	id_table = p;
}

void read_identifier (mref_t mr, const char *name)
{
	if (explicit_variables)
		goto ret_literal;

	if (call_induce) {
		/*
		 * See if the variable name contains brackets (not as the
		 * first character; it would be a FORM identifier) and
		 * perform interpolation in this case.
		 */
		if (name[0] != '[' && strchr(name,'[')) {
			/* Call induce */
			mref_t tmp;
			char *cmd;
			
			cmd = alloca(11 + strlen(name));
			strcpy(cmd, "induce -q ");
			strcat(cmd, name);
			execute_function(cmd, 0);
			tmp = pop_mref();
			mref_copy(mr, tmp); mref_free(tmp);
			return;
		}
	}
	/* Is there a variable with this name? */
	if (read_variable(mr,name) == 0)
		return;

	/* Return a literal */
ret_literal:
	mref_build(mr, ST_LITERAL, name);
	mref_promote(mr, mr_model);
}

void write_mref_to_file (const char* file, mref_t mr)
{
	FILE *fp;

	if (file[0] == '|') {
		fp = popen(file+1, "w");
		if (!fp) { perror("samuel: popen"); return; }
		signal(SIGPIPE, SIG_IGN);
		fputs(mref_string(mr), fp);
		fputc('\n', fp);
		pclose(fp);
		signal(SIGPIPE, SIG_DFL);
	}
	else {
		fp = fopen(file, "w");
		if (!fp) { perror("samuel: fopen"); return; }
		fputs(mref_string(mr), fp);
		fputc('\n', fp);
		fclose(fp);
	}
}

void read_mref_from_file (mref_t mr, const char* file)
{
	FILE *fp;

	if (file[0] == '|') {
		/* Pipe */
		if ((fp = popen(file+1, "r")) == NULL) {
			perror("samuel: popen");
			return;
		}
		saml_init_lexer_fd(fp);
		saml_parse(mr, mr_model);
		if (pclose(fp) < 0)
			perror("samuel: pclose");
	} else {
		/* File */
		if ((fp = fopen(file, "r")) == NULL) {
			perror("samuel: fopen");
			return;
		}
		saml_init_lexer_fd(fp);
		saml_parse(mr, mr_model);
		if (fclose(fp) < 0)
			perror("samuel: fclose");
	}
}

mref_t make_square_matrix (int n2)
{
	int n, i, j;
	mref_t melt, line, matrix;

	matrix = mref_new();
	/* Check that n2 is a square */
	for (n = 0; n*n < n2; n++)
		;
	if (n*n > n2) {
		/* Return a void object */
		fprintf(stderr, "make_square_matrix: not square\n");
		while (n2--)
			mref_free(pop_mref());
		return matrix;
	}
	mref_array(matrix, ST_MATRIX, n);
	line = mref_new();
	for (i = n-1; i >= 0; i--) {
		mref_array(line, ST_LINE, n);
		for (j = n-1; j >= 0; j--) {
			melt = pop_mref();
			mref_setitem(line, j, melt);
			mref_free(melt);
		}
		mref_setitem(matrix, i, line);
	}
	mref_free(line);
	return matrix;
}
