 /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@foni.net)

   This file was initially copied from the hp3300 testools and adjusted to
   suit. Original copyright notice follows:

   Copyright (C) 2001 Bertrik Sikken (bertrik@zonnet.nl)

   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RT8891 chipset,
    such as HP4400C, HP4470C.
    Parts of this source were inspired by other backends.

    History:


    Version 0.17g 18.04.04 10.alpha, little fixes
    Version 0.17d 05.04.04 10.alpha, little fixes
    Version 0.17c 31.03.04 10.alpha, little fixes
    Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
    Version 0.12 13/10/03 third alpha, Backend name changed to HP_RTS
    Version 0.11 30/08/03 second alpha
    Version 0.10 19/07/03 first alpha
*/

/*
    Provides a simple interface to read and write data from the scanner,
    without any knowledge whether it's a parallel or USB scanner
*/

#ifdef LIBUSB_SUPPORT

#include <errno.h>
#include "hp_rts_xfer.h"

#include <stdio.h>     /* printf */

#include "/usr/include/usb.h"     /*   libusb include file */
/*#include <usb.h>     *   libusb include file */



struct usb_dev_handle {
  int fd;

  struct usb_bus *bus;
  struct usb_device *device;

  int config;
  int interface;
  int altsetting;

  /* Added by RMT so implementations can store other per-open-device data */
  void *impl_info;
};
typedef struct
{
  SANE_Bool open;
  int fd;
  SANE_String devname;
  SANE_Int vendor;
  SANE_Int product;
/*  SANE_Int bulk_in_ep;
  SANE_Int bulk_out_ep;
  SANE_Int int_in_ep;
  SANE_Int int_out_ep;*/
  SANE_Int interface_nr;
  usb_dev_handle *libusb_handle;
  struct usb_device *libusb_device;
}
device_list_type;

/* total number of devices that can be opened at the same time */
#define MAX_DEVICES 2
/* per-device information, using the functions' parameters dn as index */
static device_list_type devices[MAX_DEVICES];

/************************************************************************
  USB transfer routines implemented with libusb calls
************************************************************************/
#define LIBUSB_TIMEOUT  5000L
#define HP_RTS_BULK_IN  0x81
#define HP_RTS_BULK_OUT 0x02

#define LIBUSB_FORMAT   "libusb(%s,%s)"

/****************************************************************************/
static SANE_Int
LibUsbInit(TFnReportDevice *pfnReportDevice)
/****************************************************************************/
{
	struct usb_bus *pbus;
	struct usb_device *pdev;
	SANE_Char szDeviceName[32];
	TScannerModel *pModel;

	devices[1].open = SANE_FALSE;

	usb_init();
	usb_find_busses();
	usb_find_devices();
	for (pbus = usb_busses; pbus; pbus = pbus->next) {
		for (pdev = pbus->devices; pdev; pdev = pdev->next) {
#ifdef DEBUG
			DBG(DBG_MSG, "Bus %s Dev %s : 0x%04X-0x%04X\n", pbus->dirname, pdev->filename,
					pdev->descriptor.idVendor, pdev->descriptor.idProduct);
#endif
			if (Hp_rts_MatchUsbDevice(pdev->descriptor.idVendor,
				pdev->descriptor.idProduct, &pModel)) {

				sprintf(szDeviceName, LIBUSB_FORMAT, pbus->dirname, pdev->filename);
				pfnReportDevice(pModel, szDeviceName);
				return(SANE_TRUE);
			}
		}
	}
	return(SANE_FALSE);
}


/****************************************************************************/
static SANE_Int
LibUsbOpen(SANE_Char *pszName)
/****************************************************************************/
{
	struct usb_bus *pbus;
	struct usb_device *pdev;
	usb_dev_handle *devLibUsb;
	SANE_Char szDeviceName[32];
	SANE_Int dn=0;
	int result,interface = 0;

	if (devices[1].open){
		DBG (DBG_ERR, "LibUsbOpen : Device already open\n");
		return -1;
	}
	usb_find_busses();
	usb_find_devices();
	for (pbus = usb_busses; pbus; pbus = pbus->next) {
		for (pdev = pbus->devices; pdev; pdev = pdev->next) {
			sprintf(szDeviceName, LIBUSB_FORMAT, pbus->dirname, pdev->filename);

			if (strcmp(pszName, szDeviceName) == 0) {
				/* try to open */
				devLibUsb = usb_open(pdev);
				if (!devLibUsb){
					if (errno == EPERM)
					{
						DBG (DBG_ERR, "LibUsbOpen : Make sure you run as root or set appropriate "
								   "permissions\n");
						return -1;
					}
					else if (errno == EBUSY)
					{
						DBG (DBG_ERR, "LibUsbOpen : Maybe the kernel scanner driver claims the "
							   "scanner's interface?\n");
						return -1;
					}
					return -1;
				}
				pdev = usb_device (devLibUsb);
				if (!pdev->config){
					DBG (DBG_ERR, "LibUsbOpen : device `%s' not configured?\n", szDeviceName);
				}

				/* Claim the interface */
				result = usb_claim_interface (devLibUsb,interface);
				if (result < 0){
					DBG (DBG_ERR, "LibUsbOpen : libusb complained: %s\n", usb_strerror ());
					if (errno == EPERM){
						DBG (DBG_ERR, "Make sure you run as root or set appropriate "
												   "permissions\n");
					}
					else if (errno == EBUSY){
						DBG (DBG_ERR, "Maybe the kernel scanner driver claims the "
												   "scanner's interface?\n");
					}
					usb_close (devLibUsb);
					return -1;
				}

				/* Set the configuration */
				if (devLibUsb)
					dn++;
					devices[dn].libusb_device = pdev;
					devices[dn].libusb_handle = devLibUsb;
					devices[dn].devname = strdup (szDeviceName);
					devices[dn].open = SANE_TRUE;
					devices[dn].vendor = pdev->descriptor.idVendor;
					devices[dn].product = pdev->descriptor.idProduct;
					printf("LibUsbOpen: Return with iHandle= %d,devLibUsb=%d\n",dn,(int)devLibUsb);
					return dn;
			}
		}
	}
	return -1;
}


/****************************************************************************/
static void
LibUsbExit(SANE_Int iHandle)
/****************************************************************************/
{
	if (iHandle < 0) {
#ifdef DEBUG
		DBG(DBG_ERR, "ERROR: LibUsbExit - nothing to close\n");
#endif
		return;
	}

	if (usb_release_interface(devices[iHandle].libusb_handle, 0) != 0) {
#ifdef DEBUG
		DBG(DBG_ERR, "ERROR: LibUsbExit - could not release interface (%d)\n",
				iHandle);
#endif
	}
	usb_close(devices[iHandle].libusb_handle);
	devices[iHandle].open = SANE_FALSE;
}


/****************************************************************************/
static SANE_Int
LibUsbWriteReg(SANE_Int iHandle, SANE_Byte reg, SANE_Byte bData)
/****************************************************************************/
{ /* writes a single register */
	SANE_Byte data[]={0x88,0,0x00,0x01,0};
	ssize_t write_size = 0;

#ifdef USBDEBUG
	SANE_Byte *tmp=data;
	SANE_Int tmp_size=5;
	fprintf(stderr,"WR transfer type=bulk size=%d dir=OUT\n",5);
#endif
	if (iHandle < 0) {
		return(-1);
	}
	data[1]=reg;
	data[4]=bData;
	write_size = usb_bulk_write(devices[iHandle].libusb_handle,0x02,(char *)data,5,TIMEOUT);
	if (write_size >= 0){
#ifdef USBDEBUG
	while(tmp_size--) fprintf(stderr,"%02hhx ",*tmp++);
		fprintf(stderr,"\n");
#endif
		return(0);
	}
#ifdef DEBUG
	DBG(DBG_ERR, "ERROR: LibUsbWriteReg - could not write to USB interface\n");
#endif
	return(-1);
}


/****************************************************************************/
static SANE_Int
LibUsbReadReg(SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pabData)
/****************************************************************************/
{ /* Reads a single register and returns it */
	SANE_Byte data[]={0x80,0,0x00,0x01};
	ssize_t read_size = 0;
	ssize_t write_size = 0;

	data[1] = reg;
#ifdef USBDEBUG
	SANE_Byte *tmpr=pabData;
	SANE_Byte *tmpw=data;
	SANE_Int tmp_size=4;
	fprintf(stderr,"RR transfer type=bulk size=%d dir=OUT\n",tmp_size);
	while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpw++);
	fprintf(stderr,"\n");
#endif

	if (iHandle < 0) {
#ifdef DEBUG
		DBG(DBG_ERR, "ERROR: LibUsbRead - unvalid iHandle %d\n",iHandle);
#endif
		return(-1);
  }
	write_size = usb_bulk_write(devices[iHandle].libusb_handle,0x02,(char *)data,4,TIMEOUT);
/*	printf("LibUsbReadReg: write_size = %d\n",write_size);*/
	if (write_size >=0) {
#ifdef USBDEBUG
		fprintf(stderr,"RR transfer type=bulk size=%d dir=IN\n",1);
#endif
		read_size = usb_bulk_read(devices[iHandle].libusb_handle,0x81,(char *)pabData,1,TIMEOUT);
		if (read_size >= 0) {
#ifdef USBDEBUG
			tmp_size=1;
			while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpr++);
			fprintf(stderr,"\n");
#endif
			return(0);}
	}else{
#ifdef DEBUG
		DBG(DBG_ERR, "ERROR: LibUsbReadReg - could not write to USB interface\n");
#endif
		return -1;
	};
#ifdef DEBUG
	DBG(DBG_ERR, "ERROR: LibUsbReadReg - could not read from USB interface\n");
#endif
	return(-1);
}

/****************************************************************************/
static SANE_Int
LibUsbWriteBulk(SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pabData, SANE_Int iSize,SANE_Int wHeader)
/****************************************************************************/
{ /* writes count registers into data - return error code. */
	SANE_Byte init[]={0x88,0,0x00,0x01};
	SANE_Int returncode;
	SANE_Byte *request;     /* Compose a request string. */
	ssize_t write_size = 0;
#ifdef USBDEBUG
	SANE_Byte *tmpp=pabData;
	SANE_Byte *tmpw;
	SANE_Int tmp_size=iSize;
#endif
	returncode = -1;
	if (iHandle < 0) {
		return(returncode);
	}
	if (wHeader){
		request = malloc(iSize + 4);     /* Compose a request string. */
		init[1] = reg;
		init[2] = (iSize>>8); /* split count over these bytes.*/
		init[3] = (SANE_Byte) iSize;
		memcpy(request,init,4);        /* stick them together */
		memcpy(request+4,pabData,iSize);
#ifdef USBDEBUG
		tmp_size= iSize+4;
		tmpw = request;
		fprintf(stderr,"WB1 transfer type=bulk size=%d dir=OUT\n",tmp_size);
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpw++);
		fprintf(stderr,"\n");
#endif
		write_size = usb_bulk_write(devices[iHandle].libusb_handle,0x02,(char *)request,iSize+4,TIMEOUT);
		if (write_size >= 0){
			returncode = 0;
		}else{
#ifdef DEBUG
			DBG(DBG_ERR, "ERROR: LibUsbWriteBulk - could not write to USB interface\n");
#endif
			returncode = -1;
		}
		free(request);                  /* clean up */
	}else
	{
#ifdef USBDEBUG
		tmp_size= iSize;
		fprintf(stderr,"WB2 transfer type=bulk size=%d dir=OUT\n",tmp_size);
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpp++);
		fprintf(stderr,"\n");
#endif
		write_size = usb_bulk_write(devices[iHandle].libusb_handle,0x02,(char *)pabData,iSize,TIMEOUT);
		if (write_size >=0){
			returncode = 0;
		}else{
#ifdef DEBUG
			DBG(DBG_ERR, "ERROR: LibUsbWriteBulk - could not write to USB interface\n");
#endif
		}
	}
	return(returncode);
}


/****************************************************************************/
static SANE_Int
LibUsbReadBulk(SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pabData, SANE_Int iSize,SANE_Int wWrite)
/****************************************************************************/
{ /* Reads count registers into data - no return error code. */
	SANE_Byte init[]={0x80,0,0x00,0x01};
	SANE_Int returncode;
	ssize_t write_size = 0;
	ssize_t read_size = 0;
#ifdef USBDEBUG
	SANE_Byte *tmpr=pabData;
	SANE_Byte *tmpw=init;
	SANE_Int tmp_size=iSize;
#endif

	returncode = -1;
	if (iHandle < 0) {
		return(returncode);
	}

	if (wWrite){
		init[1] = reg;
		init[2] = (iSize>>8) & 0xFF; /*split count over these bytes.*/
		init[3] = (iSize) & 0xFF;
#ifdef USBDEBUG
		tmp_size= 4;
		fprintf(stderr,"RB transfer type=bulk size=%d dir=OUT\n",tmp_size);
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpw++);
		fprintf(stderr,"\n");
#endif
		write_size = usb_bulk_write(devices[iHandle].libusb_handle,0x02,(char *)init,4,TIMEOUT);
		if (write_size <=0){
#ifdef DEBUG
			DBG(DBG_ERR, "ERROR: LibUsbReadBulk - could not write to USB interface\n");
#endif
			return -1;
		}
	}
#ifdef USBDEBUG
	fprintf(stderr,"RB transfer type=bulk size=%d dir=IN\n",iSize);
#endif
	read_size = usb_bulk_read(devices[iHandle].libusb_handle,0x81,(char *)pabData,iSize,TIMEOUT);
	if (read_size >=0){
#ifdef USBDEBUG
		tmp_size=iSize;
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpr++);
		fprintf(stderr,"\n");
#endif
		return 0;
	}else
	{
		DBG(DBG_ERR, "ERROR: LibUsbReadBulk - could not read from USB interface\n");
		return -1;
	}
  return(returncode);
}

/****************************************************************************/
static SANE_Int
LibUsbReadBulk_all_regs(SANE_Int iHandle, SANE_Byte *pabData)
/****************************************************************************/
{
	SANE_Byte init[]={0x80,0,0x00,0xf4};
	ssize_t write_size = 0;
	ssize_t read_size = 0;
#ifdef USBDEBUG
	SANE_Byte *tmpr=pabData;
	SANE_Byte *tmpw=init;
	SANE_Int tmp_size=4;
#endif

	if (iHandle < 0) {
		return -1;
	}

#ifdef USBDEBUG
	fprintf(stderr,"AR transfer type=bulk size=%d dir=OUT\n",tmp_size);
	while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpw++);
	fprintf(stderr,"\n");
#endif
	write_size = usb_bulk_write(devices[iHandle].libusb_handle,0x02,(char *)init,4,TIMEOUT);
	if (write_size <=0){
#ifdef DEBUG
		DBG(DBG_ERR, "ERROR: LibUsbReadBulk_all_regs - could not write to USB interface\n");
#endif
		return -1;
	}

#ifdef USBDEBUG
	fprintf(stderr,"AR transfer type=bulk size=%d dir=IN\n",1);
#endif
	read_size = usb_bulk_read(devices[iHandle].libusb_handle,0x81,(char *)pabData,192,TIMEOUT);
	if (read_size >=0){
#ifdef USBDEBUG
		tmp_size=192;
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpr++);
		fprintf(stderr,"\n");
#endif
	}else
	{
		DBG(DBG_ERR, "ERROR: LibUsbReadBulk - could not read from USB interface\n");
		return -1;
	}

	read_size = usb_bulk_read(devices[iHandle].libusb_handle,0x81,(char *)pabData+192,52,TIMEOUT);
	if (read_size >=0){
#ifdef USBDEBUG
		tmp_size=52;
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpr++);
		fprintf(stderr,"\n");
#endif
	}else
	{
		DBG(DBG_ERR, "ERROR: LibUsbReadBulk - could not read from USB interface\n");
		return -1;
	}
 return 0;
}

/****************************************************************************/
XferModule LibUsbDev = {
/****************************************************************************/
  "LibUSB",
  LibUsbInit,
  LibUsbOpen,
  LibUsbExit,
  LibUsbWriteReg,
  LibUsbReadReg,
  LibUsbWriteBulk,
  LibUsbReadBulk,
  LibUsbReadBulk_all_regs
};

#endif /* LIBUSB_SUPPORT */
