/*
 * SARG Squid Analysis Report Generator      http://sarg.sourceforge.net
 *                                                            1998, 2011
 *
 * SARG donations:
 *      please look at http://sarg.sourceforge.net/donations.php
 * Support:
 *     http://sourceforge.net/projects/sarg/forums/forum/363374
 * ---------------------------------------------------------------------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 */

#include "include/conf.h"
#include "include/defs.h"
#ifdef HAVE_WS2TCPIP_H
#include <ws2tcpip.h> //define getnameinfo on windows
#endif

void ip2name(char *ip,int ip_len)
{
#ifdef HAVE_GETNAMEINFO
	struct sockaddr_storage sa;
	int sockaddr_size;
	char host[NI_MAXHOST];
	int n1,n2,n3,n4,next=0;
	int error;

	memset(&sa,0,sizeof(sa));
	if (sscanf(ip,"%d.%d.%d.%d%n",&n1,&n2,&n3,&n4,&next)==4 && ip[next]=='\0') {
		struct sockaddr_in *s4=(struct sockaddr_in *)&sa;
		if (inet_pton(AF_INET,ip,&s4->sin_addr)!=1) return;
		sa.ss_family=AF_INET;
		sockaddr_size=sizeof(*s4);
	} else {
		struct sockaddr_in6 *s6=(struct sockaddr_in6 *)&sa;
		if (inet_pton(AF_INET6,ip,&s6->sin6_addr)!=1) return;
		sa.ss_family=AF_INET6;
		sockaddr_size=sizeof(*s6);
	}
#ifdef HAVE_SOCKADDR_SA_LEN
	sa.ss_len=sockaddr_size;
#endif
	error=getnameinfo((struct sockaddr *)&sa,sockaddr_size,host,sizeof(host),NULL,0,0);
	if (error==EAI_AGAIN) {
		/*
		This is a temporary failure. According to the man page we should try again but
		it doesn't say if the program should wait before trying again nor how many attempts
		before it becomes a fatal error. I could find no clues on internet so I try once and
		leave it at that. Considering the number of IP addresses to resolve and the absence
		of serious consequences should some IP fail to be resolved properly, it is best
		not waste too much time on this.
		*/
		error=getnameinfo((struct sockaddr *)&sa,sizeof(sa),host,sizeof(host),NULL,0,0);
	}
	if (error==0) {
		safe_strcpy(ip,host,ip_len);
	} else {
		debuga(_("IP to name resolution (getnameinfo) on IP address %s failed with error %d - %s\n"),ip,error,gai_strerror(error));
	}
#else
	struct in_addr addr;
	struct hostent *hp;
	char **p;

#ifdef HAVE_INET_ATON
	if (inet_aton(ip,&addr) == 0)
		return;
#else
	addr.s_addr=inet_addr(ip);
	if (addr.s_addr==-1) return;
#endif

	hp = gethostbyaddr((void *)&addr, sizeof (addr), AF_INET);
	if (hp == NULL)
		return;

	for (p = hp->h_addr_list; *p != 0; p++) {
		struct in_addr in;

		(void) memcpy(&in.s_addr, *p, sizeof (in.s_addr));
		safe_strcpy(ip,hp->h_name,ip_len);
	}
#endif
	return;
}

void name2ip(char *name,int name_size)
{
#ifdef HAVE_GETADDRINFO
	int error;
	char *port;
	struct addrinfo *res;
	char *addr;

	addr=name;
	if (name[0]=='[') { //IPv6 address
		port=strchr(name,']');
		if (port) { //confirmed IPv6 address
			*port='\0';
			addr++;
		}
	} else { //IPv4 address
		port=strchr(name,':');
		if (port) *port='\0';
	}

	error=getaddrinfo(addr,NULL,NULL,&res);
	if (error) {
		freeaddrinfo(res);
		debuga(_("Cannot resolve host name %s: %s\n"),name,gai_strerror(error));
		exit(EXIT_FAILURE);
	}
	if (res->ai_family==AF_INET) {
		struct sockaddr_in *s4=(struct sockaddr_in *)res->ai_addr;
		struct in_addr *sa=&s4->sin_addr;
		if (res->ai_addrlen<sizeof(*s4)) {
			debuga(_("Short structure returned by getaddrinfo for an IPv4 address: %d bytes instead of %d\n"),res->ai_addrlen,(int)sizeof(*s4));
			exit(EXIT_FAILURE);
		}
		inet_ntop(res->ai_family,sa,name,name_size);
	} else if (res->ai_family==AF_INET6) {
		struct sockaddr_in6 *s6=(struct sockaddr_in6 *)res->ai_addr;
		struct in6_addr *sa6=&s6->sin6_addr;
		if (res->ai_addrlen<sizeof(*s6)) {
			debuga(_("Short structure returned by getaddrinfo for an IPv6 address: %d bytes instead of %d\n"),res->ai_addrlen,(int)sizeof(*s6));
			exit(EXIT_FAILURE);
		}
		inet_ntop(res->ai_family,sa6,name,name_size);
	} else {
		debuga(_("Invalid address type %d returned when resolving host name \"%s\"\n"),res->ai_family,name);
	}
	freeaddrinfo(res);
#else
	struct in_addr ia;
	struct hostent *hp;
	char *port;
	char n1[4];
	char n2[4];
	char n3[4];
	char n4[4];
	struct getwordstruct gwarea;

	port=strchr(name,':');
	if (port) *port='\0';

	if((hp=gethostbyname(name))==NULL)
		return;

	memcpy(&ia.s_addr,hp->h_addr_list[0],sizeof(ia.s_addr));
	ia.s_addr=ntohl(ia.s_addr);
	getword_start(&gwarea,inet_ntoa(ia));
	if (getword(n4,sizeof(n4),&gwarea,'.')<0 || getword(n3,sizeof(n3),&gwarea,'.')<0 ||
	    getword(n2,sizeof(n2),&gwarea,'.')<0 || getword(n1,sizeof(n1),&gwarea,0)<0) {
		printf("SARG: Maybe you have a broken record or garbage in your %s ip address.\n",gwarea.beginning);
		exit(EXIT_FAILURE);
	}
	snprintf(name,name_size,"%s.%s.%s.%s",n1,n2,n3,n4);
#endif

	return;
}
