# book.tcl: Book editing windows for Scid.
# Copyright (C) 2000  Shane Hudson

set maxBooks [sc_info limit books]

proc storeBookText {id} {
  sc_book set $id [.book$id.text get 1.0 "end-1c"]
}

proc storeBookTexts {} {
  global maxBooks
  for {set i 1} {$i <= $maxBooks} {incr i} {
    if {[winfo exists .book$i]} { storeBookText $i }
  }
}

proc updateBookWin {id} {
  set w .book$id
  $w.text delete 1.0 end
  $w.text insert end [sc_book get $id]

  # Update the book window status bar:
  set str "  --  "
  if {[sc_book readonly $id]} {
    set str "  %%  "
  } elseif {[sc_book altered $id]} {
    set str "  XX  "
  }
  append str "[file tail [sc_book name $id]]  [sc_book size $id] positions"
  set moves [lsort -ascii [sc_book moves $id]]
  set len [llength $moves]
  if {$len} {
    append str "  \[[llength $moves]: [join $moves " "]\]"
  } else {
    append str {  [No moves from this position]}
  }
  $w.status configure -text $str
  unset str
}

proc updateBookWins {} {
  global maxBooks
  for {set i 1} {$i <= $maxBooks} {incr i} {
    if {[winfo exists .book$i]} { updateBookWin $i }
  }
}

proc closeBookWin {id} {
  catch {sc_book close $id}
}

proc confirmCloseBook {id} {
  if {! [winfo exists .book$id]} { return }
  storeBookText $id
  if {[sc_book altered $id]  &&  ! [sc_book readonly $id]} {
    set result [tk_dialog .dialog "Save changes?" \
      "This file has been altered; do you want to save it?" \
      "" 0 "Save changes" "Close without saving" "Cancel"]
    if {$result == 2} { return }
    if {$result == 0} { sc_book write $id }
  }
  sc_book close $id
  focus .
  destroy .book$id
  return
}

proc saveBookWin {id} {
  set w .book$id
  busyCursor . 1
  set temp_oldcursor [$w.text cget -cursor]
  $w.text configure -cursor watch
  update idletasks
  storeBookText $id
  sc_book write $id
  updateBookWin $id
  $w.text configure -cursor $temp_oldcursor
  busyCursor . 0
}

proc book_MoveToDeepestMatch {id} {
  if {! [winfo exists .book$id]} { return }
  sc_move ply [sc_book deepest $id]
  updateBoard .board
  return
}

proc newBookWin {cmd {fname ""}} {
  global maxBooks
  set showErrors 1
  if {$cmd == "openSilent"} { set showErrors 0 }
  if {$fname == ""} { set showErrors 1 }
  if {[sc_book available] < 1} {
    if {$showErrors} {
      tk_messageBox -type ok -icon info -title "Too many books open" \
        -message "You already have $maxBooks books open; close one first."
    }
    return 0
  }
  set new_types { {"Scid Book files" {".epd"} } }
  set open_types $new_types
  if {[sc_info gzip]} {
    set open_types { {"Scid Book files" {".epd" ".epd.gz"} } }
  }
  if {$fname == ""} {
    if {$cmd == "create"} {
      set fname [tk_getSaveFile -initialdir $::initialDir(book) -filetypes $new_types -title "Create a Book file"]
    } elseif {$cmd == "open"} {
      set fname [tk_getOpenFile -initialdir $::initialDir(book) -filetypes $open_types -title "Open a Book file"]
    } else { return 0 }
  }
  if {$fname == ""} { return 0 }

  busyCursor . 1
  if {[catch {sc_book $cmd $fname} result]} {
    if {$showErrors} {
      busyCursor . 0
      tk_messageBox -type ok -icon error -title "Scid: Book file error" \
        -message $result
    }
    return 0
  }
  busyCursor . 0
  set id $result
  set w .book$id
  toplevel $w
  wm title $w "Scid Book: [file tail $fname]"
  wm minsize $w 40 1
  bind $w <Destroy> "closeBookWin $id"
  bind $w <F1> { helpWindow Book }

  frame $w.grid
  text $w.text -background white -font font_Regular -width 60 -height 7 \
    -wrap none -setgrid 1 -yscrollcommand "$w.ybar set" \
    -xscrollcommand "$w.xbar set"
  scrollbar $w.ybar -takefocus 0 -command "$w.text yview"
  scrollbar $w.xbar -orient horizontal -takefocus 0 -command "$w.text xview"
  label $w.status -width 1 -anchor w -font font_Small -relief sunken

  frame $w.menu -borderwidth 3 -relief raised
  pack $w.menu  -side top -fill x
  menubutton $w.menu.file -text File -menu $w.menu.file.m -underline 0
  menubutton $w.menu.edit -text Edit -menu $w.menu.edit.m -underline 0
  menubutton $w.menu.tools -text Tools -menu $w.menu.tools.m -underline 0
  menubutton $w.menu.help -text Help -menu $w.menu.help.m -underline 0

  foreach i {file edit tools help} {
    menu $w.menu.$i.m -tearoff 0
    pack $w.menu.$i -side left
  }

  set m $w.menu.file.m
  $m add command -label "New" -acc "Ctrl+N" -underline 0 \
    -command {newBookWin create}
  bind $w <Control-n> {newBookWin create}
  $m add command -label "Open" -acc "Ctrl+O" -underline 0 \
    -command {newBookWin open}
  bind $w <Control-o> {newBookWin open}
  $m add command -label "Save" -acc "Ctrl+S" -underline 0 \
    -command "saveBookWin $id"
  if {[sc_book readonly $id]} {
    $m entryconfig "Save" -state disabled
  } else {
    bind $w <Control-s> "saveBookWin $id; break"
  }
  $m add command -label "Close" -acc "Ctrl+Q" -underline 0 \
    -command "confirmCloseBook $id"
  bind $w <Control-q> "confirmCloseBook $id"

  set m $w.menu.edit.m
  $m add command -label "Cut" -acc "Ctrl+X" -underline 2 -command "tk_textCut $w.text"
  bind $w <Control-x> "tk_textCut $w.text; break"
  $m add command -label "Copy" -acc "Ctrl+C" -underline 0 -command "tk_textCopy $w.text"
  bind $w <Control-c> "tk_textCopy $w.text; break"
  $m add command -label "Paste" -acc "Ctrl+V" -underline 0 -command "tk_textPaste $w.text"
  bind $w <Control-v> "tk_textPaste $w.text; break"
  $m add command -label "Select All" -acc "Ctrl+A" -underline 7 \
    -command "$w.text tag add sel 1.0 end"
  bind $w <Control-a> "$w.text tag add sel 1.0 end; break"
  $m add separator
  $m add command -label "Revert" -acc "Ctrl+R" -underline 0 \
    -command "updateBookWin $id"
  bind $w <Control-r> "updateBookWin $id; break"
  $m add command -label "Sort lines" -accel "Ctrl+Shift+S" \
    -underline 0 -command "book_sortLines $w.text"
  bind $w <Control-S> "book_sortLines $w.text; break"

  set m $w.menu.tools.m
  $m add command -label "Find Deepest game position" \
    -underline 5 -command "book_MoveToDeepestMatch $id"
  $m add separator
  $m add command -label "Next position in book file" \
    -accelerator "Ctrl+DownArrow" -underline 0 \
    -command "sc_book next $id; updateBoardAndPgn .board"
  bind $w <Control-Down> "sc_book next $id; updateBoardAndPgn .board; break"
  $m add command -label "Previous position in book file" \
    -accelerator "Ctrl+UpArrow" -underline 0 \
    -command "sc_book prev $id; updateBoardAndPgn .board"
  bind $w <Control-Up> "sc_book prev $id; updateBoardAndPgn .board; break"
  $m add separator
  $m add command -label "Paste analysis" -accelerator "Ctrl+Shift+A" \
    -underline 6 -command "book_pasteAnalysis $w.text"
  bind $w <Control-A> "book_pasteAnalysis $w.text; break"
  $m add separator
  $m add command -label "Strip out EPD field" -underline 0 \
    -command "book_chooseStripField $id"

  $w.menu.help.m add command -label "Book files help" -underline 0 \
    -acc "F1" -command "helpWindow Book"
  $w.menu.help.m add command -label "General index" -underline 0 \
    -command "helpWindow Index"

  pack $w.status -side bottom -fill x
  pack $w.grid -fill both -expand yes
  grid $w.text -in $w.grid -row 0 -column 0 -sticky news
  grid $w.ybar -in $w.grid -row 0 -column 1 -sticky news
  grid $w.xbar -in $w.grid -row 1 -column 0 -sticky news

  grid rowconfig $w.grid 0 -weight 1 -minsize 0
  grid columnconfig $w.grid 0 -weight 1 -minsize 0

  # Right-mouse button cut/copy/paste menu:
  menu $w.text.edit -tearoff 0
  $w.text.edit add command -label "Cut"  -command "tk_textCut $w.text"
  $w.text.edit add command -label "Copy" -command "tk_textCopy $w.text"
  $w.text.edit add command -label "Paste" -command "tk_textPaste $w.text"
  bind $w.text <ButtonPress-3> "tk_popup $w.text.edit %X %Y"

  updateBookWin $id
  return 1
}


proc book_sortLines {textwidget} {
  if {! [winfo exists $textwidget]} { return }
  set text [$textwidget get 1.0 "end-1c"]
  set fieldlist [split $text "\n"]
  set sortedlist [lsort $fieldlist]
  while {[lindex $sortedlist 0] == ""} {
    set sortedlist [lrange $sortedlist 1 end]
  }
  set newtext [join $sortedlist "\n"]
  append newtext "\n"
  if {! [string compare $text $newtext]} { return }
  $textwidget delete 1.0 end
  $textwidget insert end "$newtext"
}

# book_pasteAnalysis:
#    Pastes current chess engine analysis into this book file position.
proc book_pasteAnalysis {textwidget} {
  global analysis
  if {! [winfo exists $textwidget]} { return }
  if {! [winfo exists .analysisWin1]} { return }
  $textwidget insert insert "acd $analysis(depth1)\n"
  $textwidget insert insert "acn $analysis(nodes1)\n"
  set ce [expr int($analysis(score1) * 100)]
  if {[sc_pos side] == "black"} { set ce [expr 0 - $ce] }
  $textwidget insert insert "ce $ce\n"
  $textwidget insert insert "pv $analysis(moves1)\n"
}


set book_stripField ""

proc book_chooseStripField {id} {
  global book_stripField
  if {! [winfo exists .book$id]} { return }
  set w [toplevel .bookStrip]
  wm title $w "Scid: Strip EPD field"
  wm resizable $w false false
  label $w.label -text "Enter the name of the EPD field you want\n\
removed from all positions in this book:"
  entry $w.e -width 10 -background white -textvariable book_stripField
  pack $w.label $w.e -side top -pady 5 -padx 5
  addHorizontalRule $w
  set b [frame $w.buttons]
  pack $b -side bottom -pady 5
  button $b.ok -text "Strip EPD field" \
    -command "book_stripEpdField $id \$book_stripField"
  button $b.cancel -text "Cancel" -command "focus .book$id; destroy $w"
  pack $b.ok $b.cancel -side left -padx 5
  bind $w <Return> "$b.ok invoke"
  bind $w <Escape> "$b.cancel invoke"
  focus .bookStrip.e
  grab .bookStrip
}

proc book_stripEpdField {id field} {
  if {! [winfo exists .bookStrip]} { return }
  if {! [string compare $field ""]} { beep; return }
  set result [sc_book strip $id $field]
  updateBookWin $id
  tk_messageBox -type ok -icon info -title "Scid: EPD field stripped" \
    -message "Scid found and stripped an EPD field named \"$field\" from\
$result book positions."
  focus .book$id
  destroy .bookStrip
}
