/* vim:set noet ts=4: */
/** 
 * scim-python
 * 
 * Copyright (c) 2007-2008 Huang Peng <shawn.p.huang@gmail.com>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: $
 */
#include "scim-python.h"
#if Py_UNICODE_SIZE == 2
#  include <glib.h>
#endif

extern PyTypeObject PyIMEngineType;

struct PyIMEngineObject {
	PyListObject list;
	/* Type-specific fields go here. */
	PyIMEngine engine;
};


PyIMEngine::PyIMEngine (
		PyObject			*self,
		PyObject 			*factory,
		PyObject			*config,
		const String		&encoding,
		int				  	id)
  : IMEngineInstanceBase (PyIMEngineFactory::from_pyobject (factory), 
	encoding, id), 
	self (self), 
	factory (factory),
	config (config)
{
	Py_INCREF (self);
	Py_INCREF (factory);
	Py_INCREF (config);

	reload_signal_connection = 
		PyConfig_from_pyobject (config)->signal_connect_reload (slot (this, &PyIMEngine::reload_config));
}

PyIMEngine::~PyIMEngine ()
{
	reload_signal_connection.disconnect ();

	Py_XDECREF (config);
	Py_XDECREF (factory);
	Py_XDECREF (self);
}

void
PyIMEngine::operator delete (void *p)
{
}

#define PY_CALL(fun) ({								\
	PyObject *pValue = NULL;						\
	PyObject *pFunc = NULL;							\
	pFunc = PyObject_GetAttrString (this->self, fun);\
	if (pFunc != NULL) {							\
		pValue = PyObject_CallObject (pFunc, NULL);	\
		Py_DECREF (pFunc);							\
	}												\
	pValue;})

#define PY_CALL_1(fun, arg1) ({						\
	PyObject *pValue = NULL;						\
	PyObject *pFunc = NULL;							\
	PyObject *pArgs = NULL;							\
	pFunc = PyObject_GetAttrString (this->self, fun);\
	if (pFunc != NULL) {							\
		pArgs = Py_BuildValue ("(O)", arg1);		\
		pValue = PyObject_CallObject (pFunc, pArgs);\
		Py_DECREF (pFunc);							\
		Py_DECREF (pArgs);							\
	}												\
	pValue;})

#define PY_CHECK_RET(v)							\
	if (v == NULL) {							\
		PyErr_Print ();							\
		return;									\
	}

#define PY_CHECK_RET_VAL(v, val)				\
	if (v == NULL) {							\
		PyErr_Print ();							\
		return val;								\
	}

bool 
PyIMEngine::process_key_event (const KeyEvent &key)
{
	PyObject *pValue;

	bool result = false;

	pValue = PY_CALL_1("process_key_event", PyKeyEvent_New (key));

	PY_CHECK_RET_VAL (pValue, false);

	result = (pValue == Py_True); 
	Py_XDECREF (pValue);

	return result;
}

PyDoc_STRVAR (IMEngine_process_key_event__doc__,
"process_key_event (key_event)\n\n"
"This function will be called, when user press keyboard in client application.\n"
"IM developer must implement it in sub class.\n");
PyObject *
PyIMEngine::py_process_key_event (PyIMEngineObject *self, PyObject *args)
{
	PyErr_SetString (PyExc_NotImplementedError, "process_key_event is not implemented!");
	return NULL;
}


void
PyIMEngine::move_preedit_caret (unsigned int pos)
{
	PyObject *pValue = NULL;

	pValue = PY_CALL_1 ("move_preedit_caret", PyInt_FromLong ((unsigned long)pos));

	PY_CHECK_RET (pValue);
	Py_XDECREF (pValue);
}

PyDoc_STRVAR (IMEngine_move_preedit_caret__doc__,
"move_preedit_caret (pos)\n\n"
"This function will be called, when client move preedit caret.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_move_preedit_caret (PyIMEngineObject *self, PyObject *args)
{
	unsigned int pos;

	if (!PyArg_ParseTuple (args, "I:move_preedit_caret", &pos))
		return NULL;

	self->engine.IMEngineInstanceBase::move_preedit_caret (pos);

	Py_INCREF (Py_None);
	return Py_None;

}

void
PyIMEngine::select_candidate (unsigned int index)
{
	PyObject *pValue = NULL;

	pValue = PY_CALL_1 ("select_candidate", PyInt_FromLong ((unsigned long)index));
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}


PyDoc_STRVAR (IMEngine_select_candidate__doc__,
"select_candidate (index)\n\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_select_candidate (PyIMEngineObject *self, PyObject *args)
{
	unsigned int index;

	if (!PyArg_ParseTuple (args, "I:select_candidate", &index))
		return NULL;

	self->engine.IMEngineInstanceBase::select_candidate (index);

	Py_INCREF (Py_None);
	return Py_None;

}

void
PyIMEngine::update_lookup_table_page_size (unsigned int page_size)
{
	PyObject *pValue = NULL;

	pValue = PY_CALL_1 ("update_lookup_table_page_size", PyInt_FromLong ((unsigned long)page_size));
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}


PyDoc_STRVAR (IMEngine_update_lookup_table_page_size__doc__,
"update_lookup_table_page_size (page_size)\n\n"
"This function will be called, when page size of lookup table is changed.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_update_lookup_table_page_size (PyIMEngineObject *self, PyObject *args)
{
	unsigned int page_size;

	if (!PyArg_ParseTuple (args, "I:update_lookup_table_page_size", &page_size))
		return NULL;

	self->engine.IMEngineInstanceBase::move_preedit_caret (page_size);

	Py_INCREF (Py_None);
	return Py_None;

}

void
PyIMEngine::lookup_table_page_up ()
{
	PyObject *pValue = NULL;
	pValue = PY_CALL ("lookup_table_page_up");
	PY_CHECK_RET (pValue);
	Py_XDECREF (pValue);
}

PyDoc_STRVAR (IMEngine_lookup_table_page_up__doc__,
"lookup_table_page_up ()\n\n"
"This function will be called, when user press page up button in candidate window.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_lookup_table_page_up (PyIMEngineObject *self)
{
	self->engine.IMEngineInstanceBase::lookup_table_page_up ();

	Py_INCREF (Py_None);
	return Py_None;
}

void
PyIMEngine::lookup_table_page_down ()
{
	PyObject *pValue = NULL;

	pValue = PY_CALL ("lookup_table_page_down");
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}

PyDoc_STRVAR (IMEngine_lookup_table_page_down__doc__,
"lookup_table_page_down ()\n\n"
"This function will be called, when user press page down button in candidate window.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_lookup_table_page_down (PyIMEngineObject *self)
{
	self->engine.IMEngineInstanceBase::lookup_table_page_down ();

	Py_INCREF (Py_None);
	return Py_None;
}

void
PyIMEngine::reset ()
{
	PyObject *pValue = NULL;

	pValue = PY_CALL ("reset");
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}

PyDoc_STRVAR (IMEngine_reset__doc__,
"reset ()\n\n"
"This function will be called, when im client request reseting engine.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_reset (PyIMEngineObject *self)
{
	self->engine.IMEngineInstanceBase::reset ();

	Py_INCREF (Py_None);
	return Py_None;
}

void
PyIMEngine::focus_in ()
{
	PyObject *pValue = NULL;

	pValue = PY_CALL ("focus_in");
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}

PyDoc_STRVAR (IMEngine_focus_in__doc__,
"focus_in ()\n\n"
"This function will be called, when engine is focused in.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_focus_in (PyIMEngineObject *self)
{
	self->engine.IMEngineInstanceBase::focus_in ();

	Py_INCREF (Py_None);
	return Py_None;
}

void
PyIMEngine::focus_out ()
{
	PyObject *pValue = NULL;

	pValue = PY_CALL ("focus_out");
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}

PyDoc_STRVAR (IMEngine_focus_out__doc__,
"focus_out ()\n\n"
"This function will be called, when engine is focused out.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_focus_out (PyIMEngineObject *self)
{
	self->engine.IMEngineInstanceBase::focus_out ();

	Py_INCREF (Py_None);
	return Py_None;
}

void
PyIMEngine::trigger_property (const String &property)
{
	PyObject *pValue = NULL;

	pValue = PY_CALL_1 ("trigger_property", PyString_FromString (property.c_str ()));
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}


PyDoc_STRVAR (IMEngine_trigger_property__doc__,
"trigger_property (prop_name)\n\n"
"This function will be called, when user presses property button on panel.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_trigger_property (PyIMEngineObject *self, PyObject *args)
{
	char *property;

	if (!PyArg_ParseTuple (args, "s:trigger_property", &property))
		return NULL;

	self->engine.IMEngineInstanceBase::trigger_property (String (property));

	Py_INCREF (Py_None);
	return Py_None;

}

void
PyIMEngine::process_helper_event (const String &helper_uuid, const Transaction &trans)
{
	//IMEngineInstanceBase::process_helper_event (helper_uuid, trans);
}

PyDoc_STRVAR (IMEngine_process_helper_event__doc__,
"process_helper (uuid, trans)\n\n"
"This function will be called, when user ....\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_process_helper_event (PyIMEngineObject *self, PyObject *args)
{
	// TODO: 
	char *helper_uuid;
	PyObject *trans;

	if (!PyArg_ParseTuple (args, "sO:process_helper_event", &helper_uuid, &trans))
		return NULL;

	//self->engine.IMEngineInstanceBase::update_client_capabilities (cap);

	Py_INCREF (Py_None);
	return Py_None;

}

void
PyIMEngine::update_client_capabilities (unsigned int cap)
{
	PyObject *pValue = NULL;

	pValue = PY_CALL_1 ("update_client_capabilities", PyInt_FromLong ((unsigned long)cap));
	PY_CHECK_RET (pValue);

	Py_XDECREF (pValue);
}

PyDoc_STRVAR (IMEngine_update_client_capabilities__doc__,
"update_client_capabilities (cap)\n\n"
"This function will be called, when im client request update client capabilities.\n"
"IM developer may override it in sub class.\n");
PyObject *
PyIMEngine::py_update_client_capabilities (PyIMEngineObject *self, PyObject *args)
{
	unsigned int cap;

	if (!PyArg_ParseTuple (args, "I:update_client_capabilities", &cap))
		return NULL;
	self->engine.IMEngineInstanceBase::update_client_capabilities (cap);

	Py_INCREF (Py_None);
	return Py_None;

}


void 
PyIMEngine::reload_config (const ConfigPointer &config)
{
	PyObject *pFunc = NULL;
	PyObject *pValue = NULL;
	PyObject *pArgs = NULL;

	pFunc = PyObject_GetAttrString (this->self, "reload_config");
	if (pFunc == NULL)
		goto _failed_out;

	pArgs = Py_BuildValue ("(O)", this->config);
	if (pArgs == NULL)
		goto _failed_out;

	pValue = PyObject_CallObject (pFunc, pArgs);
	if (pValue == NULL)
		goto _failed_out;

	goto _success_out;

_failed_out:
	PyErr_Print ();
_success_out:
	Py_XDECREF (pArgs);
	Py_XDECREF (pFunc);
	Py_XDECREF (pValue);
}

IMEngineInstanceBase *
PyIMEngine::from_pyobject (PyObject *object)
{
	PyIMEngineObject *self = (PyIMEngineObject *)object;
	return (IMEngineInstanceBase *) &self->engine;
}

PyObject *
PyIMEngine::py_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{
	PyIMEngine *self;
	self = (PyIMEngine *)type->tp_alloc (type, 0);
	return (PyObject *)self;
}


int
PyIMEngine::py_init (PyIMEngineObject *self, PyObject *args, PyObject *kwds)
{
	PyObject *factory;
	PyObject *config;
	char *encoding;
	int id;

	if (!PyArg_ParseTuple (args, "OOsi:__init__", &factory, &config, &encoding, &id)) {
		PyErr_Print ();
		return -1;
	}

	new (&self->engine) PyIMEngine ((PyObject *)self, 
								factory,
								config,
								String (encoding), id);
	return 0;
}

void
PyIMEngine::py_dealloc (PyIMEngineObject *self)
{
	((PyObject *)self)->ob_type->tp_free (self);
}

PyDoc_STRVAR (IMEngine_show_preedit_string__doc__,
"show_preedit_string ()\n\n"
"Hide the preedit string.\n");
PyObject *
PyIMEngine::py_show_preedit_string (PyIMEngineObject *self)
{
	self->engine.show_preedit_string ();

	Py_INCREF (Py_None);
	return Py_None;
}


PyDoc_STRVAR (IMEngine_show_aux_string__doc__,
"show_aux_string ()\n\n"
"Hide the aux string.\n");
PyObject *
PyIMEngine::py_show_aux_string (PyIMEngineObject *self)
{
	self->engine.show_aux_string ();

	Py_INCREF (Py_None);
	return Py_None;
}

PyDoc_STRVAR (IMEngine_show_lookup_table__doc__,
"show_lookup_table ()\n\n"
"Hide the lookup table.\n");
PyObject *
PyIMEngine::py_show_lookup_table (PyIMEngineObject *self)
{
	self->engine.show_lookup_table ();

	Py_INCREF (Py_None);
	return Py_None;
}

PyDoc_STRVAR (IMEngine_hide_preedit_string__doc__,
"hide_preedit_string ()\n\n"
"Hide the preedit string.\n");
PyObject *
PyIMEngine::py_hide_preedit_string (PyIMEngineObject *self)
{
	self->engine.hide_preedit_string ();

	Py_INCREF (Py_None);
	return Py_None;
}

PyDoc_STRVAR (IMEngine_hide_aux_string__doc__,
"hide_aux_string ()\n\n"
"Hide the aux string.\n");
PyObject *
PyIMEngine::py_hide_aux_string (PyIMEngineObject *self)
{
	self->engine.hide_aux_string ();

	Py_INCREF (Py_None);
	return Py_None;
}

PyDoc_STRVAR (IMEngine_hide_lookup_table__doc__,
"hide_lookup_table ()\n\n"
"Hide the lookup table.\n");
PyObject *
PyIMEngine::py_hide_lookup_table (PyIMEngineObject *self)
{
	self->engine.hide_lookup_table ();

	Py_INCREF (Py_None);
	return Py_None;
}

PyDoc_STRVAR (IMEngine_update_preedit_caret__doc__,
"update_preedit_caret (pos)\n\n"
"Update the caret position of preedit string.\n");
PyObject *
PyIMEngine::py_update_preedit_caret (PyIMEngineObject *self, PyObject *args)
{
	int caret;
	if (!PyArg_ParseTuple (args, "i:update_preedit_caret", &caret))
			return NULL;
	self->engine.update_preedit_caret (caret);

	Py_INCREF (Py_None);
	return Py_None;
}

/*
void update_preedit_string (const WideString	&str,
							const AttributeList &attrs = AttributeList ());
*/
PyDoc_STRVAR (IMEngine_update_preedit_string__doc__,
"update_preedit_string (text, attrs = None)\n\n"
"Update preedit string with attrs.\n");
PyObject *
PyIMEngine::py_update_preedit_string (PyIMEngineObject *self, PyObject *args)
{
	Py_UNICODE *str = NULL;
	PyObject *pAttrs = NULL;
	AttributeList attrs;

#if Py_UNICODE_SIZE == 4


	if (!PyArg_ParseTuple (args, "u|O:update_preedit_string", &str, &pAttrs))
		return NULL;

	self->engine.update_preedit_string (WideString ((wchar_t *)str), 
								Attributes_FromTupleOrList (pAttrs));
#else
	gunichar *unistr = NULL;
	int size = 0;

	if (!PyArg_ParseTuple (args, "u#|O:update_preedit_string", &str, &size, &pAttrs))
		return NULL;

	unistr = g_utf16_to_ucs4 (str, size, NULL, NULL, NULL);

	self->engine.update_preedit_string (WideString ((wchar_t *)unistr), 
								Attributes_FromTupleOrList (pAttrs));
	g_free (unistr);

#endif

	Py_INCREF (Py_None);
	return Py_None;

}

/*
 * void update_aux_string (const WideString	&str,
 * 						   const AttributeList &attrs = AttributeList ());
 */
PyDoc_STRVAR (IMEngine_update_aux_string__doc__,
"update_aux_string (text, attrs = None)\n\n"
"Update aux string with attrs.\n");
PyObject *
PyIMEngine::py_update_aux_string (PyIMEngineObject *self, PyObject *args)
{
	Py_UNICODE *str = NULL;
	PyObject *pAttrs = NULL;
	AttributeList attrs;

#if Py_UNICODE_SIZE == 4
	if (!PyArg_ParseTuple (args, "u|O:update_aux_string", &str, &pAttrs))
		return NULL;

	self->engine.update_aux_string (WideString ((wchar_t *)str),
								Attributes_FromTupleOrList (pAttrs));
#else
	int size = 0;
	gunichar *unistr = NULL;
	if (!PyArg_ParseTuple (args, "u#|O:update_aux_string", &str, &size, &pAttrs))
		return NULL;

	unistr = g_utf16_to_ucs4 (str, size, NULL, NULL, NULL);
	self->engine.update_aux_string (WideString ((wchar_t *)unistr),
								Attributes_FromTupleOrList (pAttrs));
	g_free (unistr);
#endif

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void update_lookup_table (const LookupTable &table);
 */
PyDoc_STRVAR (IMEngine_update_lookup_table__doc__,
"update_lookup_table (lookup_table)\n\n"
"Update the lookup table that contains candidates.\n");
PyObject *
PyIMEngine::py_update_lookup_table (PyIMEngineObject *self, PyObject *args)
{

	PyObject *lookup_table = NULL;

	if (!PyArg_ParseTuple (args, "O:update_lookup_table", &lookup_table))
		return NULL;

	self->engine.update_lookup_table (PyLookupTable::from_pyobject (lookup_table));

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void commit_string (const WideString &str);
 */
PyDoc_STRVAR (IMEngine_commit_string__doc__,
"commit_string (text)\n\n"
"Commit text to im client.\n");
PyObject *
PyIMEngine::py_commit_string (PyIMEngineObject *self, PyObject *args)
{
	Py_UNICODE *str = NULL;

#if Py_UNICODE_SIZE == 4
	if (!PyArg_ParseTuple (args, "u:commit_string", &str))
		return NULL;

	self->engine.commit_string (WideString ((wchar_t *)str));
#else
	int size = 0;
	gunichar *unistr = NULL;

	if (!PyArg_ParseTuple (args, "u#:commit_string", &str, &size))
		return NULL;

	unistr = g_utf16_to_ucs4 (str, size, NULL, NULL, NULL);
	self->engine.commit_string (WideString ((wchar_t *)unistr));
	g_free (unistr);
#endif

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void forward_key_event (const KeyEvent &key);
 */
PyDoc_STRVAR (IMEngine_forward_key_event__doc__,
"forward_key_event (key_event)\n\n"
"Forward a key_event (scim.KeyEvent) to im client.\n");
PyObject *
PyIMEngine::py_forward_key_event (PyIMEngineObject *self, PyObject *args)
{
	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void register_properties (const PropertyList &properties);
 */
PyDoc_STRVAR (IMEngine_register_properties__doc__,
"register_properties (props)\n\n"
"Regitser props (a list of scim.Property) with IMEngine.\n");
PyObject *
PyIMEngine::py_register_properties (PyIMEngineObject *self, PyObject *args)
{
	PyObject *props = NULL;
	PropertyList proplist;
	int i;

	if (!PyArg_ParseTuple (args, "O:register_properties", &props))
		return NULL;

	if (PyList_Check (props)) {
		for (i = 0; i < PyList_Size (props); i++) {
			PyObject *prop = PyList_GetItem (props, i);
			proplist.push_back (PyProperty_AsProperty (prop));
		}
	}
	else if (PyTuple_Check (props)) {
		for (i = 0; i < PyTuple_Size (props); i++) {
			PyObject *prop = PyTuple_GetItem (props, i);
			proplist.push_back (PyProperty_AsProperty (prop));
		}

	}
	else {
		PyErr_SetString (PyExc_TypeError, "the argument must be a list or a tuple that contains propertys");
		return NULL;
	}

	self->engine.register_properties (proplist);

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void update_property (const Property &property);
 */
PyDoc_STRVAR (IMEngine_update_property__doc__,
"update_property (property)\n\n"
"Update property (scim.Property) with IMEngine.\n");
PyObject *
PyIMEngine::py_update_property (PyIMEngineObject *self, PyObject *args)
{
	PyObject *prop = NULL;

	if (!PyArg_ParseTuple (args, "O:update_property", &prop))
		return NULL;


	self->engine.update_property (PyProperty_AsProperty (prop));

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void beep ();
 */
PyDoc_STRVAR (IMEngine_beep__doc__,
"beep ()\n\n"
"Beep.\n");
PyObject *
PyIMEngine::py_beep (PyIMEngineObject *self)
{
	self->engine.beep ();

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void start_helper (const String &helper_uuid);
 */
PyDoc_STRVAR (IMEngine_start_helper__doc__,
"start_helper (helper_uuid)\n\n"
"Start a helper specified with the uuid.\n");
PyObject *
PyIMEngine::py_start_helper (PyIMEngineObject *self, PyObject *args)
{
	char *str = NULL;

	if (!PyArg_ParseTuple (args, "s:start_helper", &str))
		return NULL;

	self->engine.start_helper (String (str));

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void stop_helper (const String &helper_uuid);
 */
PyDoc_STRVAR (IMEngine_stop_helper__doc__,
"stop_helper (helper_uuid)\n\n"
"Stop a helper specified with the uuid.\n");
PyObject *
PyIMEngine::py_stop_helper (PyIMEngineObject *self, PyObject *args)
{
	char *str = NULL;

	if (!PyArg_ParseTuple (args, "s:stop_helper", &str))
		return NULL;

	self->engine.stop_helper (String (str));

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * void send_helper_event (const String &helper_uuid, const Transaction &trans);
 */
PyDoc_STRVAR (IMEngine_send_helper_event__doc__,
"send_helper_event ()\n\n"
"Do nothing.\n");
PyObject *
PyIMEngine::py_send_helper_event (PyIMEngineObject *self, PyObject *args)
{
	char *str = NULL;

	if (!PyArg_ParseTuple (args, "s:send_helper_event", &str))
		return NULL;

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * bool get_surrounding_text (WideString &text, 
 * 							int &cursor, i
 * 							nt maxlen_before = -1, 
 * 							int maxlen_after = -1);
 */
PyDoc_STRVAR (IMEngine_get_surrounding_text__doc__,
"get_surrounding_text ()\n\n"
"Do nothing.\n");
PyObject *
PyIMEngine::py_get_surrounding_text (PyIMEngineObject *self, PyObject *args)
{
	char *str = NULL;

	if (!PyArg_ParseTuple (args, "s:get_surrounding_text", &str))
		return NULL;

	Py_INCREF (Py_None);
	return Py_None;
}

/*
 * bool delete_surrounding_text (int offset, int len);
 */
PyDoc_STRVAR (IMEngine_delete_surrounding_text__doc__,
"delete_surrounding_text ()\n\n"
"Do nothing.\n");
PyObject *
PyIMEngine::py_delete_surrounding_text (PyIMEngineObject *self, PyObject *args)
{
	int offset;
	int len;

	if (!PyArg_ParseTuple (args, "ii:delete_surrounding_text", &offset, &len))
		return NULL;

	self->engine.delete_surrounding_text (offset, len);

	Py_INCREF (Py_None);
	return Py_None;
}

PyMethodDef
PyIMEngine::py_methods[] = {

#define ENTRY(name, type) \
	{ \
		#name, \
		(PyCFunction)PyIMEngine::py_##name, \
		type, \
		IMEngine_##name##__doc__ \
	}

	ENTRY (show_preedit_string, METH_NOARGS),
	ENTRY (show_aux_string, METH_NOARGS),
	ENTRY (show_lookup_table, METH_NOARGS),
	ENTRY (hide_preedit_string, METH_NOARGS),
	ENTRY (hide_aux_string, METH_NOARGS),
	ENTRY (hide_lookup_table, METH_NOARGS),

	ENTRY (update_preedit_caret, METH_VARARGS),
	ENTRY (update_preedit_string, METH_VARARGS),
	ENTRY (update_aux_string, METH_VARARGS),
	ENTRY (update_lookup_table, METH_VARARGS),

	ENTRY (commit_string, METH_VARARGS),
	ENTRY (forward_key_event, METH_VARARGS),

	ENTRY (register_properties, METH_VARARGS),
	ENTRY (update_property, METH_VARARGS),

	ENTRY (beep, METH_NOARGS),

	ENTRY (start_helper, METH_VARARGS),
	ENTRY (stop_helper, METH_VARARGS),
	ENTRY (send_helper_event, METH_VARARGS),

	ENTRY (get_surrounding_text, METH_VARARGS),
	ENTRY (delete_surrounding_text, METH_VARARGS),

	ENTRY (process_key_event, METH_VARARGS),
	ENTRY (move_preedit_caret, METH_VARARGS),
	ENTRY (select_candidate, METH_VARARGS),
	ENTRY (update_lookup_table_page_size, METH_VARARGS),

	ENTRY (lookup_table_page_up, METH_NOARGS),
	ENTRY (lookup_table_page_down, METH_NOARGS),
	ENTRY (reset, METH_NOARGS),
	ENTRY (focus_in, METH_NOARGS),
	ENTRY (focus_out, METH_NOARGS),

	ENTRY (trigger_property, METH_VARARGS),
	ENTRY (process_helper_event, METH_VARARGS),
	ENTRY (update_client_capabilities, METH_VARARGS),
#undef ENTRY
		{ NULL }
};

PyTypeObject PyIMEngineType = {
	PyObject_HEAD_INIT (NULL)
	0,						 		/*ob_size*/
	"scim.IMEngine", 				/*tp_name*/
	sizeof (PyIMEngineObject), 		/*tp_basicsize*/
	0,						 		/*tp_itemsize*/
	(destructor)PyIMEngine::py_dealloc,	/*tp_dealloc*/
	0,			  					/*tp_print*/
	0,						 		/*tp_getattr*/
	0,								/*tp_setattr*/
	0,								/*tp_compare*/
	0,			  					/*tp_repr*/
	0,								/*tp_as_number*/
	0,			  					/*tp_as_sequence*/
	0,								/*tp_as_mapping*/
	0,			  					/*tp_hash */
	0,								/*tp_call*/
	0,		  						/*tp_str*/
	0,					   			/*tp_getattro*/
	0,								/*tp_setattro*/
	0,					 			/*tp_as_buffer*/
	Py_TPFLAGS_DEFAULT | 
	Py_TPFLAGS_BASETYPE,			/*tp_flags*/
	"IMEngineInstanceBase objects",	/* tp_doc */
	0,					  	/* tp_traverse */
	0,					  	/* tp_clear */
	0,					  	/* tp_richcompare */
	0,					  	/* tp_weaklistoffset */
	0,					  	/* tp_iter */
	0,					  	/* tp_iternext */
	PyIMEngine::py_methods,	/* tp_methods */
	0,			 			/* tp_members */
	0,						/* tp_getset */
	0,						/* tp_base */
	0,						/* tp_dict */
	0,						/* tp_descr_get */
	0,						/* tp_descr_set */
	0,						/* tp_dictoffset */
	(initproc)PyIMEngine::py_init,	/* tp_init */
	0,						 		/* tp_alloc */
	PyIMEngine::py_new,				/* tp_new */
	PyObject_Del,					/* tp_free */
};


void init_engine (PyObject *module)
{
	if (PyType_Ready (&PyIMEngineType) < 0)
		return;

	Py_INCREF (&PyIMEngineType);
	PyModule_AddObject (module, "IMEngine", (PyObject *)&PyIMEngineType);
}

