/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_event.cpp,v 1.21 2004/08/04 14:58:03 suzhe Exp $
 */

#define Uses_SCIM_EVENT
#include "scim_private.h"
#include "scim.h"

namespace scim {

struct __KeyUnicode
{
    unsigned short code;
    unsigned short ucs;
};

struct __KeyName
{
    int  value;
    char *name;
};

class __KeyUnicodeLessByCode
{
public:
    bool operator ()(const __KeyUnicode &lhs, const __KeyUnicode &rhs) const {
        return lhs.code < rhs.code;
    }

    bool operator ()(const __KeyUnicode &lhs, const int &rhs) const {
        return ((int) lhs.code) < rhs;
    }

    bool operator ()(const int &lhs, const __KeyUnicode &rhs) const {
        return lhs < ((int) rhs.code);
    }
};

class __KeyNameLessByCode
{
public:
    bool operator ()(const __KeyName &lhs, const __KeyName &rhs) const {
        return lhs.value < rhs.value;
    }

    bool operator ()(const __KeyName &lhs, const int &rhs) const {
        return lhs.value < rhs;
    }

    bool operator ()(const int &lhs, const __KeyName &rhs) const {
        return lhs < rhs.value;
    }
};

class __KeyNameLessByName
{
public:
    bool operator ()(const __KeyName &lhs, const __KeyName &rhs) const {
        return strcmp (lhs.name, rhs.name) < 0;
    }

    bool operator ()(const __KeyName &lhs, const char *rhs) const {
        return strcmp (lhs.name, rhs) < 0;
    }

    bool operator ()(const char *lhs, const __KeyName &rhs) const {
        return strcmp (lhs, rhs.name) < 0;
    }
};

/* Thanks to Markus G. Kuhn <mkuhn@acm.org> for the ksysym<->Unicode
 * mapping functions, from the xterm sources.
 */

/* These tables could be compressed by contiguous ranges, but the benefit of doing so
 * is smallish. It would save about ~1000 bytes total.
 */
static __KeyUnicode __scim_key_to_unicode_tab [] = {
  { 0x01a1, 0x0104 }, /*                     Aogonek Ą LATIN CAPITAL LETTER A WITH OGONEK */
  { 0x01a2, 0x02d8 }, /*                       breve ˘ BREVE */
  { 0x01a3, 0x0141 }, /*                     Lstroke Ł LATIN CAPITAL LETTER L WITH STROKE */
  { 0x01a5, 0x013d }, /*                      Lcaron Ľ LATIN CAPITAL LETTER L WITH CARON */
  { 0x01a6, 0x015a }, /*                      Sacute Ś LATIN CAPITAL LETTER S WITH ACUTE */
  { 0x01a9, 0x0160 }, /*                      Scaron Š LATIN CAPITAL LETTER S WITH CARON */
  { 0x01aa, 0x015e }, /*                    Scedilla Ş LATIN CAPITAL LETTER S WITH CEDILLA */
  { 0x01ab, 0x0164 }, /*                      Tcaron Ť LATIN CAPITAL LETTER T WITH CARON */
  { 0x01ac, 0x0179 }, /*                      Zacute Ź LATIN CAPITAL LETTER Z WITH ACUTE */
  { 0x01ae, 0x017d }, /*                      Zcaron Ž LATIN CAPITAL LETTER Z WITH CARON */
  { 0x01af, 0x017b }, /*                   Zabovedot Ż LATIN CAPITAL LETTER Z WITH DOT ABOVE */
  { 0x01b1, 0x0105 }, /*                     aogonek ą LATIN SMALL LETTER A WITH OGONEK */
  { 0x01b2, 0x02db }, /*                      ogonek ˛ OGONEK */
  { 0x01b3, 0x0142 }, /*                     lstroke ł LATIN SMALL LETTER L WITH STROKE */
  { 0x01b5, 0x013e }, /*                      lcaron ľ LATIN SMALL LETTER L WITH CARON */
  { 0x01b6, 0x015b }, /*                      sacute ś LATIN SMALL LETTER S WITH ACUTE */
  { 0x01b7, 0x02c7 }, /*                       caron ˇ CARON */
  { 0x01b9, 0x0161 }, /*                      scaron š LATIN SMALL LETTER S WITH CARON */
  { 0x01ba, 0x015f }, /*                    scedilla ş LATIN SMALL LETTER S WITH CEDILLA */
  { 0x01bb, 0x0165 }, /*                      tcaron ť LATIN SMALL LETTER T WITH CARON */
  { 0x01bc, 0x017a }, /*                      zacute ź LATIN SMALL LETTER Z WITH ACUTE */
  { 0x01bd, 0x02dd }, /*                 doubleacute ˝ DOUBLE ACUTE ACCENT */
  { 0x01be, 0x017e }, /*                      zcaron ž LATIN SMALL LETTER Z WITH CARON */
  { 0x01bf, 0x017c }, /*                   zabovedot ż LATIN SMALL LETTER Z WITH DOT ABOVE */
  { 0x01c0, 0x0154 }, /*                      Racute Ŕ LATIN CAPITAL LETTER R WITH ACUTE */
  { 0x01c3, 0x0102 }, /*                      Abreve Ă LATIN CAPITAL LETTER A WITH BREVE */
  { 0x01c5, 0x0139 }, /*                      Lacute Ĺ LATIN CAPITAL LETTER L WITH ACUTE */
  { 0x01c6, 0x0106 }, /*                      Cacute Ć LATIN CAPITAL LETTER C WITH ACUTE */
  { 0x01c8, 0x010c }, /*                      Ccaron Č LATIN CAPITAL LETTER C WITH CARON */
  { 0x01ca, 0x0118 }, /*                     Eogonek Ę LATIN CAPITAL LETTER E WITH OGONEK */
  { 0x01cc, 0x011a }, /*                      Ecaron Ě LATIN CAPITAL LETTER E WITH CARON */
  { 0x01cf, 0x010e }, /*                      Dcaron Ď LATIN CAPITAL LETTER D WITH CARON */
  { 0x01d0, 0x0110 }, /*                     Dstroke Đ LATIN CAPITAL LETTER D WITH STROKE */
  { 0x01d1, 0x0143 }, /*                      Nacute Ń LATIN CAPITAL LETTER N WITH ACUTE */
  { 0x01d2, 0x0147 }, /*                      Ncaron Ň LATIN CAPITAL LETTER N WITH CARON */
  { 0x01d5, 0x0150 }, /*                Odoubleacute Ő LATIN CAPITAL LETTER O WITH DOUBLE ACUTE */
  { 0x01d8, 0x0158 }, /*                      Rcaron Ř LATIN CAPITAL LETTER R WITH CARON */
  { 0x01d9, 0x016e }, /*                       Uring Ů LATIN CAPITAL LETTER U WITH RING ABOVE */
  { 0x01db, 0x0170 }, /*                Udoubleacute Ű LATIN CAPITAL LETTER U WITH DOUBLE ACUTE */
  { 0x01de, 0x0162 }, /*                    Tcedilla Ţ LATIN CAPITAL LETTER T WITH CEDILLA */
  { 0x01e0, 0x0155 }, /*                      racute ŕ LATIN SMALL LETTER R WITH ACUTE */
  { 0x01e3, 0x0103 }, /*                      abreve ă LATIN SMALL LETTER A WITH BREVE */
  { 0x01e5, 0x013a }, /*                      lacute ĺ LATIN SMALL LETTER L WITH ACUTE */
  { 0x01e6, 0x0107 }, /*                      cacute ć LATIN SMALL LETTER C WITH ACUTE */
  { 0x01e8, 0x010d }, /*                      ccaron č LATIN SMALL LETTER C WITH CARON */
  { 0x01ea, 0x0119 }, /*                     eogonek ę LATIN SMALL LETTER E WITH OGONEK */
  { 0x01ec, 0x011b }, /*                      ecaron ě LATIN SMALL LETTER E WITH CARON */
  { 0x01ef, 0x010f }, /*                      dcaron ď LATIN SMALL LETTER D WITH CARON */
  { 0x01f0, 0x0111 }, /*                     dstroke đ LATIN SMALL LETTER D WITH STROKE */
  { 0x01f1, 0x0144 }, /*                      nacute ń LATIN SMALL LETTER N WITH ACUTE */
  { 0x01f2, 0x0148 }, /*                      ncaron ň LATIN SMALL LETTER N WITH CARON */
  { 0x01f5, 0x0151 }, /*                odoubleacute ő LATIN SMALL LETTER O WITH DOUBLE ACUTE */
  { 0x01f8, 0x0159 }, /*                      rcaron ř LATIN SMALL LETTER R WITH CARON */
  { 0x01f9, 0x016f }, /*                       uring ů LATIN SMALL LETTER U WITH RING ABOVE */
  { 0x01fb, 0x0171 }, /*                udoubleacute ű LATIN SMALL LETTER U WITH DOUBLE ACUTE */
  { 0x01fe, 0x0163 }, /*                    tcedilla ţ LATIN SMALL LETTER T WITH CEDILLA */
  { 0x01ff, 0x02d9 }, /*                    abovedot ˙ DOT ABOVE */
  { 0x02a1, 0x0126 }, /*                     Hstroke Ħ LATIN CAPITAL LETTER H WITH STROKE */
  { 0x02a6, 0x0124 }, /*                 Hcircumflex Ĥ LATIN CAPITAL LETTER H WITH CIRCUMFLEX */
  { 0x02a9, 0x0130 }, /*                   Iabovedot İ LATIN CAPITAL LETTER I WITH DOT ABOVE */
  { 0x02ab, 0x011e }, /*                      Gbreve Ğ LATIN CAPITAL LETTER G WITH BREVE */
  { 0x02ac, 0x0134 }, /*                 Jcircumflex Ĵ LATIN CAPITAL LETTER J WITH CIRCUMFLEX */
  { 0x02b1, 0x0127 }, /*                     hstroke ħ LATIN SMALL LETTER H WITH STROKE */
  { 0x02b6, 0x0125 }, /*                 hcircumflex ĥ LATIN SMALL LETTER H WITH CIRCUMFLEX */
  { 0x02b9, 0x0131 }, /*                    idotless ı LATIN SMALL LETTER DOTLESS I */
  { 0x02bb, 0x011f }, /*                      gbreve ğ LATIN SMALL LETTER G WITH BREVE */
  { 0x02bc, 0x0135 }, /*                 jcircumflex ĵ LATIN SMALL LETTER J WITH CIRCUMFLEX */
  { 0x02c5, 0x010a }, /*                   Cabovedot Ċ LATIN CAPITAL LETTER C WITH DOT ABOVE */
  { 0x02c6, 0x0108 }, /*                 Ccircumflex Ĉ LATIN CAPITAL LETTER C WITH CIRCUMFLEX */
  { 0x02d5, 0x0120 }, /*                   Gabovedot Ġ LATIN CAPITAL LETTER G WITH DOT ABOVE */
  { 0x02d8, 0x011c }, /*                 Gcircumflex Ĝ LATIN CAPITAL LETTER G WITH CIRCUMFLEX */
  { 0x02dd, 0x016c }, /*                      Ubreve Ŭ LATIN CAPITAL LETTER U WITH BREVE */
  { 0x02de, 0x015c }, /*                 Scircumflex Ŝ LATIN CAPITAL LETTER S WITH CIRCUMFLEX */
  { 0x02e5, 0x010b }, /*                   cabovedot ċ LATIN SMALL LETTER C WITH DOT ABOVE */
  { 0x02e6, 0x0109 }, /*                 ccircumflex ĉ LATIN SMALL LETTER C WITH CIRCUMFLEX */
  { 0x02f5, 0x0121 }, /*                   gabovedot ġ LATIN SMALL LETTER G WITH DOT ABOVE */
  { 0x02f8, 0x011d }, /*                 gcircumflex ĝ LATIN SMALL LETTER G WITH CIRCUMFLEX */
  { 0x02fd, 0x016d }, /*                      ubreve ŭ LATIN SMALL LETTER U WITH BREVE */
  { 0x02fe, 0x015d }, /*                 scircumflex ŝ LATIN SMALL LETTER S WITH CIRCUMFLEX */
  { 0x03a2, 0x0138 }, /*                         kra ĸ LATIN SMALL LETTER KRA */
  { 0x03a3, 0x0156 }, /*                    Rcedilla Ŗ LATIN CAPITAL LETTER R WITH CEDILLA */
  { 0x03a5, 0x0128 }, /*                      Itilde Ĩ LATIN CAPITAL LETTER I WITH TILDE */
  { 0x03a6, 0x013b }, /*                    Lcedilla Ļ LATIN CAPITAL LETTER L WITH CEDILLA */
  { 0x03aa, 0x0112 }, /*                     Emacron Ē LATIN CAPITAL LETTER E WITH MACRON */
  { 0x03ab, 0x0122 }, /*                    Gcedilla Ģ LATIN CAPITAL LETTER G WITH CEDILLA */
  { 0x03ac, 0x0166 }, /*                      Tslash Ŧ LATIN CAPITAL LETTER T WITH STROKE */
  { 0x03b3, 0x0157 }, /*                    rcedilla ŗ LATIN SMALL LETTER R WITH CEDILLA */
  { 0x03b5, 0x0129 }, /*                      itilde ĩ LATIN SMALL LETTER I WITH TILDE */
  { 0x03b6, 0x013c }, /*                    lcedilla ļ LATIN SMALL LETTER L WITH CEDILLA */
  { 0x03ba, 0x0113 }, /*                     emacron ē LATIN SMALL LETTER E WITH MACRON */
  { 0x03bb, 0x0123 }, /*                    gcedilla ģ LATIN SMALL LETTER G WITH CEDILLA */
  { 0x03bc, 0x0167 }, /*                      tslash ŧ LATIN SMALL LETTER T WITH STROKE */
  { 0x03bd, 0x014a }, /*                         ENG Ŋ LATIN CAPITAL LETTER ENG */
  { 0x03bf, 0x014b }, /*                         eng ŋ LATIN SMALL LETTER ENG */
  { 0x03c0, 0x0100 }, /*                     Amacron Ā LATIN CAPITAL LETTER A WITH MACRON */
  { 0x03c7, 0x012e }, /*                     Iogonek Į LATIN CAPITAL LETTER I WITH OGONEK */
  { 0x03cc, 0x0116 }, /*                   Eabovedot Ė LATIN CAPITAL LETTER E WITH DOT ABOVE */
  { 0x03cf, 0x012a }, /*                     Imacron Ī LATIN CAPITAL LETTER I WITH MACRON */
  { 0x03d1, 0x0145 }, /*                    Ncedilla Ņ LATIN CAPITAL LETTER N WITH CEDILLA */
  { 0x03d2, 0x014c }, /*                     Omacron Ō LATIN CAPITAL LETTER O WITH MACRON */
  { 0x03d3, 0x0136 }, /*                    Kcedilla Ķ LATIN CAPITAL LETTER K WITH CEDILLA */
  { 0x03d9, 0x0172 }, /*                     Uogonek Ų LATIN CAPITAL LETTER U WITH OGONEK */
  { 0x03dd, 0x0168 }, /*                      Utilde Ũ LATIN CAPITAL LETTER U WITH TILDE */
  { 0x03de, 0x016a }, /*                     Umacron Ū LATIN CAPITAL LETTER U WITH MACRON */
  { 0x03e0, 0x0101 }, /*                     amacron ā LATIN SMALL LETTER A WITH MACRON */
  { 0x03e7, 0x012f }, /*                     iogonek į LATIN SMALL LETTER I WITH OGONEK */
  { 0x03ec, 0x0117 }, /*                   eabovedot ė LATIN SMALL LETTER E WITH DOT ABOVE */
  { 0x03ef, 0x012b }, /*                     imacron ī LATIN SMALL LETTER I WITH MACRON */
  { 0x03f1, 0x0146 }, /*                    ncedilla ņ LATIN SMALL LETTER N WITH CEDILLA */
  { 0x03f2, 0x014d }, /*                     omacron ō LATIN SMALL LETTER O WITH MACRON */
  { 0x03f3, 0x0137 }, /*                    kcedilla ķ LATIN SMALL LETTER K WITH CEDILLA */
  { 0x03f9, 0x0173 }, /*                     uogonek ų LATIN SMALL LETTER U WITH OGONEK */
  { 0x03fd, 0x0169 }, /*                      utilde ũ LATIN SMALL LETTER U WITH TILDE */
  { 0x03fe, 0x016b }, /*                     umacron ū LATIN SMALL LETTER U WITH MACRON */
  { 0x047e, 0x203e }, /*                    overline ‾ OVERLINE */
  { 0x04a1, 0x3002 }, /*               kana_fullstop 。 IDEOGRAPHIC FULL STOP */
  { 0x04a2, 0x300c }, /*         kana_openingbracket 「 LEFT CORNER BRACKET */
  { 0x04a3, 0x300d }, /*         kana_closingbracket 」 RIGHT CORNER BRACKET */
  { 0x04a4, 0x3001 }, /*                  kana_comma 、 IDEOGRAPHIC COMMA */
  { 0x04a5, 0x30fb }, /*            kana_conjunctive ・ KATAKANA MIDDLE DOT */
  { 0x04a6, 0x30f2 }, /*                     kana_WO ヲ KATAKANA LETTER WO */
  { 0x04a7, 0x30a1 }, /*                      kana_a ァ KATAKANA LETTER SMALL A */
  { 0x04a8, 0x30a3 }, /*                      kana_i ィ KATAKANA LETTER SMALL I */
  { 0x04a9, 0x30a5 }, /*                      kana_u ゥ KATAKANA LETTER SMALL U */
  { 0x04aa, 0x30a7 }, /*                      kana_e ェ KATAKANA LETTER SMALL E */
  { 0x04ab, 0x30a9 }, /*                      kana_o ォ KATAKANA LETTER SMALL O */
  { 0x04ac, 0x30e3 }, /*                     kana_ya ャ KATAKANA LETTER SMALL YA */
  { 0x04ad, 0x30e5 }, /*                     kana_yu ュ KATAKANA LETTER SMALL YU */
  { 0x04ae, 0x30e7 }, /*                     kana_yo ョ KATAKANA LETTER SMALL YO */
  { 0x04af, 0x30c3 }, /*                    kana_tsu ッ KATAKANA LETTER SMALL TU */
  { 0x04b0, 0x30fc }, /*              prolongedsound ー KATAKANA-HIRAGANA PROLONGED SOUND MARK */
  { 0x04b1, 0x30a2 }, /*                      kana_A ア KATAKANA LETTER A */
  { 0x04b2, 0x30a4 }, /*                      kana_I イ KATAKANA LETTER I */
  { 0x04b3, 0x30a6 }, /*                      kana_U ウ KATAKANA LETTER U */
  { 0x04b4, 0x30a8 }, /*                      kana_E エ KATAKANA LETTER E */
  { 0x04b5, 0x30aa }, /*                      kana_O オ KATAKANA LETTER O */
  { 0x04b6, 0x30ab }, /*                     kana_KA カ KATAKANA LETTER KA */
  { 0x04b7, 0x30ad }, /*                     kana_KI キ KATAKANA LETTER KI */
  { 0x04b8, 0x30af }, /*                     kana_KU ク KATAKANA LETTER KU */
  { 0x04b9, 0x30b1 }, /*                     kana_KE ケ KATAKANA LETTER KE */
  { 0x04ba, 0x30b3 }, /*                     kana_KO コ KATAKANA LETTER KO */
  { 0x04bb, 0x30b5 }, /*                     kana_SA サ KATAKANA LETTER SA */
  { 0x04bc, 0x30b7 }, /*                    kana_SHI シ KATAKANA LETTER SI */
  { 0x04bd, 0x30b9 }, /*                     kana_SU ス KATAKANA LETTER SU */
  { 0x04be, 0x30bb }, /*                     kana_SE セ KATAKANA LETTER SE */
  { 0x04bf, 0x30bd }, /*                     kana_SO ソ KATAKANA LETTER SO */
  { 0x04c0, 0x30bf }, /*                     kana_TA タ KATAKANA LETTER TA */
  { 0x04c1, 0x30c1 }, /*                    kana_CHI チ KATAKANA LETTER TI */
  { 0x04c2, 0x30c4 }, /*                    kana_TSU ツ KATAKANA LETTER TU */
  { 0x04c3, 0x30c6 }, /*                     kana_TE テ KATAKANA LETTER TE */
  { 0x04c4, 0x30c8 }, /*                     kana_TO ト KATAKANA LETTER TO */
  { 0x04c5, 0x30ca }, /*                     kana_NA ナ KATAKANA LETTER NA */
  { 0x04c6, 0x30cb }, /*                     kana_NI ニ KATAKANA LETTER NI */
  { 0x04c7, 0x30cc }, /*                     kana_NU ヌ KATAKANA LETTER NU */
  { 0x04c8, 0x30cd }, /*                     kana_NE ネ KATAKANA LETTER NE */
  { 0x04c9, 0x30ce }, /*                     kana_NO ノ KATAKANA LETTER NO */
  { 0x04ca, 0x30cf }, /*                     kana_HA ハ KATAKANA LETTER HA */
  { 0x04cb, 0x30d2 }, /*                     kana_HI ヒ KATAKANA LETTER HI */
  { 0x04cc, 0x30d5 }, /*                     kana_FU フ KATAKANA LETTER HU */
  { 0x04cd, 0x30d8 }, /*                     kana_HE ヘ KATAKANA LETTER HE */
  { 0x04ce, 0x30db }, /*                     kana_HO ホ KATAKANA LETTER HO */
  { 0x04cf, 0x30de }, /*                     kana_MA マ KATAKANA LETTER MA */
  { 0x04d0, 0x30df }, /*                     kana_MI ミ KATAKANA LETTER MI */
  { 0x04d1, 0x30e0 }, /*                     kana_MU ム KATAKANA LETTER MU */
  { 0x04d2, 0x30e1 }, /*                     kana_ME メ KATAKANA LETTER ME */
  { 0x04d3, 0x30e2 }, /*                     kana_MO モ KATAKANA LETTER MO */
  { 0x04d4, 0x30e4 }, /*                     kana_YA ヤ KATAKANA LETTER YA */
  { 0x04d5, 0x30e6 }, /*                     kana_YU ユ KATAKANA LETTER YU */
  { 0x04d6, 0x30e8 }, /*                     kana_YO ヨ KATAKANA LETTER YO */
  { 0x04d7, 0x30e9 }, /*                     kana_RA ラ KATAKANA LETTER RA */
  { 0x04d8, 0x30ea }, /*                     kana_RI リ KATAKANA LETTER RI */
  { 0x04d9, 0x30eb }, /*                     kana_RU ル KATAKANA LETTER RU */
  { 0x04da, 0x30ec }, /*                     kana_RE レ KATAKANA LETTER RE */
  { 0x04db, 0x30ed }, /*                     kana_RO ロ KATAKANA LETTER RO */
  { 0x04dc, 0x30ef }, /*                     kana_WA ワ KATAKANA LETTER WA */
  { 0x04dd, 0x30f3 }, /*                      kana_N ン KATAKANA LETTER N */
  { 0x04de, 0x309b }, /*                 voicedsound ゛ KATAKANA-HIRAGANA VOICED SOUND MARK */
  { 0x04df, 0x309c }, /*             semivoicedsound ゜ KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK */
  { 0x05ac, 0x060c }, /*                Arabic_comma ، ARABIC COMMA */
  { 0x05bb, 0x061b }, /*            Arabic_semicolon ؛ ARABIC SEMICOLON */
  { 0x05bf, 0x061f }, /*        Arabic_question_mark ؟ ARABIC QUESTION MARK */
  { 0x05c1, 0x0621 }, /*                Arabic_hamza ء ARABIC LETTER HAMZA */
  { 0x05c2, 0x0622 }, /*          Arabic_maddaonalef آ ARABIC LETTER ALEF WITH MADDA ABOVE */
  { 0x05c3, 0x0623 }, /*          Arabic_hamzaonalef أ ARABIC LETTER ALEF WITH HAMZA ABOVE */
  { 0x05c4, 0x0624 }, /*           Arabic_hamzaonwaw ؤ ARABIC LETTER WAW WITH HAMZA ABOVE */
  { 0x05c5, 0x0625 }, /*       Arabic_hamzaunderalef إ ARABIC LETTER ALEF WITH HAMZA BELOW */
  { 0x05c6, 0x0626 }, /*           Arabic_hamzaonyeh ئ ARABIC LETTER YEH WITH HAMZA ABOVE */
  { 0x05c7, 0x0627 }, /*                 Arabic_alef ا ARABIC LETTER ALEF */
  { 0x05c8, 0x0628 }, /*                  Arabic_beh ب ARABIC LETTER BEH */
  { 0x05c9, 0x0629 }, /*           Arabic_tehmarbuta ة ARABIC LETTER TEH MARBUTA */
  { 0x05ca, 0x062a }, /*                  Arabic_teh ت ARABIC LETTER TEH */
  { 0x05cb, 0x062b }, /*                 Arabic_theh ث ARABIC LETTER THEH */
  { 0x05cc, 0x062c }, /*                 Arabic_jeem ج ARABIC LETTER JEEM */
  { 0x05cd, 0x062d }, /*                  Arabic_hah ح ARABIC LETTER HAH */
  { 0x05ce, 0x062e }, /*                 Arabic_khah خ ARABIC LETTER KHAH */
  { 0x05cf, 0x062f }, /*                  Arabic_dal د ARABIC LETTER DAL */
  { 0x05d0, 0x0630 }, /*                 Arabic_thal ذ ARABIC LETTER THAL */
  { 0x05d1, 0x0631 }, /*                   Arabic_ra ر ARABIC LETTER REH */
  { 0x05d2, 0x0632 }, /*                 Arabic_zain ز ARABIC LETTER ZAIN */
  { 0x05d3, 0x0633 }, /*                 Arabic_seen س ARABIC LETTER SEEN */
  { 0x05d4, 0x0634 }, /*                Arabic_sheen ش ARABIC LETTER SHEEN */
  { 0x05d5, 0x0635 }, /*                  Arabic_sad ص ARABIC LETTER SAD */
  { 0x05d6, 0x0636 }, /*                  Arabic_dad ض ARABIC LETTER DAD */
  { 0x05d7, 0x0637 }, /*                  Arabic_tah ط ARABIC LETTER TAH */
  { 0x05d8, 0x0638 }, /*                  Arabic_zah ظ ARABIC LETTER ZAH */
  { 0x05d9, 0x0639 }, /*                  Arabic_ain ع ARABIC LETTER AIN */
  { 0x05da, 0x063a }, /*                Arabic_ghain غ ARABIC LETTER GHAIN */
  { 0x05e0, 0x0640 }, /*              Arabic_tatweel ـ ARABIC TATWEEL */
  { 0x05e1, 0x0641 }, /*                  Arabic_feh ف ARABIC LETTER FEH */
  { 0x05e2, 0x0642 }, /*                  Arabic_qaf ق ARABIC LETTER QAF */
  { 0x05e3, 0x0643 }, /*                  Arabic_kaf ك ARABIC LETTER KAF */
  { 0x05e4, 0x0644 }, /*                  Arabic_lam ل ARABIC LETTER LAM */
  { 0x05e5, 0x0645 }, /*                 Arabic_meem م ARABIC LETTER MEEM */
  { 0x05e6, 0x0646 }, /*                 Arabic_noon ن ARABIC LETTER NOON */
  { 0x05e7, 0x0647 }, /*                   Arabic_ha ه ARABIC LETTER HEH */
  { 0x05e8, 0x0648 }, /*                  Arabic_waw و ARABIC LETTER WAW */
  { 0x05e9, 0x0649 }, /*          Arabic_alefmaksura ى ARABIC LETTER ALEF MAKSURA */
  { 0x05ea, 0x064a }, /*                  Arabic_yeh ي ARABIC LETTER YEH */
  { 0x05eb, 0x064b }, /*             Arabic_fathatan ً ARABIC FATHATAN */
  { 0x05ec, 0x064c }, /*             Arabic_dammatan ٌ ARABIC DAMMATAN */
  { 0x05ed, 0x064d }, /*             Arabic_kasratan ٍ ARABIC KASRATAN */
  { 0x05ee, 0x064e }, /*                Arabic_fatha َ ARABIC FATHA */
  { 0x05ef, 0x064f }, /*                Arabic_damma ُ ARABIC DAMMA */
  { 0x05f0, 0x0650 }, /*                Arabic_kasra ِ ARABIC KASRA */
  { 0x05f1, 0x0651 }, /*               Arabic_shadda ّ ARABIC SHADDA */
  { 0x05f2, 0x0652 }, /*                Arabic_sukun ْ ARABIC SUKUN */
  { 0x06a1, 0x0452 }, /*                 Serbian_dje ђ CYRILLIC SMALL LETTER DJE */
  { 0x06a2, 0x0453 }, /*               Macedonia_gje ѓ CYRILLIC SMALL LETTER GJE */
  { 0x06a3, 0x0451 }, /*                 Cyrillic_io ё CYRILLIC SMALL LETTER IO */
  { 0x06a4, 0x0454 }, /*                Ukrainian_ie є CYRILLIC SMALL LETTER UKRAINIAN IE */
  { 0x06a5, 0x0455 }, /*               Macedonia_dse ѕ CYRILLIC SMALL LETTER DZE */
  { 0x06a6, 0x0456 }, /*                 Ukrainian_i і CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I */
  { 0x06a7, 0x0457 }, /*                Ukrainian_yi ї CYRILLIC SMALL LETTER YI */
  { 0x06a8, 0x0458 }, /*                 Cyrillic_je ј CYRILLIC SMALL LETTER JE */
  { 0x06a9, 0x0459 }, /*                Cyrillic_lje љ CYRILLIC SMALL LETTER LJE */
  { 0x06aa, 0x045a }, /*                Cyrillic_nje њ CYRILLIC SMALL LETTER NJE */
  { 0x06ab, 0x045b }, /*                Serbian_tshe ћ CYRILLIC SMALL LETTER TSHE */
  { 0x06ac, 0x045c }, /*               Macedonia_kje ќ CYRILLIC SMALL LETTER KJE */
  { 0x06ae, 0x045e }, /*         Byelorussian_shortu ў CYRILLIC SMALL LETTER SHORT U */
  { 0x06af, 0x045f }, /*               Cyrillic_dzhe џ CYRILLIC SMALL LETTER DZHE */
  { 0x06b0, 0x2116 }, /*                  numerosign № NUMERO SIGN */
  { 0x06b1, 0x0402 }, /*                 Serbian_DJE Ђ CYRILLIC CAPITAL LETTER DJE */
  { 0x06b2, 0x0403 }, /*               Macedonia_GJE Ѓ CYRILLIC CAPITAL LETTER GJE */
  { 0x06b3, 0x0401 }, /*                 Cyrillic_IO Ё CYRILLIC CAPITAL LETTER IO */
  { 0x06b4, 0x0404 }, /*                Ukrainian_IE Є CYRILLIC CAPITAL LETTER UKRAINIAN IE */
  { 0x06b5, 0x0405 }, /*               Macedonia_DSE Ѕ CYRILLIC CAPITAL LETTER DZE */
  { 0x06b6, 0x0406 }, /*                 Ukrainian_I І CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I */
  { 0x06b7, 0x0407 }, /*                Ukrainian_YI Ї CYRILLIC CAPITAL LETTER YI */
  { 0x06b8, 0x0408 }, /*                 Cyrillic_JE Ј CYRILLIC CAPITAL LETTER JE */
  { 0x06b9, 0x0409 }, /*                Cyrillic_LJE Љ CYRILLIC CAPITAL LETTER LJE */
  { 0x06ba, 0x040a }, /*                Cyrillic_NJE Њ CYRILLIC CAPITAL LETTER NJE */
  { 0x06bb, 0x040b }, /*                Serbian_TSHE Ћ CYRILLIC CAPITAL LETTER TSHE */
  { 0x06bc, 0x040c }, /*               Macedonia_KJE Ќ CYRILLIC CAPITAL LETTER KJE */
  { 0x06be, 0x040e }, /*         Byelorussian_SHORTU Ў CYRILLIC CAPITAL LETTER SHORT U */
  { 0x06bf, 0x040f }, /*               Cyrillic_DZHE Џ CYRILLIC CAPITAL LETTER DZHE */
  { 0x06c0, 0x044e }, /*                 Cyrillic_yu ю CYRILLIC SMALL LETTER YU */
  { 0x06c1, 0x0430 }, /*                  Cyrillic_a а CYRILLIC SMALL LETTER A */
  { 0x06c2, 0x0431 }, /*                 Cyrillic_be б CYRILLIC SMALL LETTER BE */
  { 0x06c3, 0x0446 }, /*                Cyrillic_tse ц CYRILLIC SMALL LETTER TSE */
  { 0x06c4, 0x0434 }, /*                 Cyrillic_de д CYRILLIC SMALL LETTER DE */
  { 0x06c5, 0x0435 }, /*                 Cyrillic_ie е CYRILLIC SMALL LETTER IE */
  { 0x06c6, 0x0444 }, /*                 Cyrillic_ef ф CYRILLIC SMALL LETTER EF */
  { 0x06c7, 0x0433 }, /*                Cyrillic_ghe г CYRILLIC SMALL LETTER GHE */
  { 0x06c8, 0x0445 }, /*                 Cyrillic_ha х CYRILLIC SMALL LETTER HA */
  { 0x06c9, 0x0438 }, /*                  Cyrillic_i и CYRILLIC SMALL LETTER I */
  { 0x06ca, 0x0439 }, /*             Cyrillic_shorti й CYRILLIC SMALL LETTER SHORT I */
  { 0x06cb, 0x043a }, /*                 Cyrillic_ka к CYRILLIC SMALL LETTER KA */
  { 0x06cc, 0x043b }, /*                 Cyrillic_el л CYRILLIC SMALL LETTER EL */
  { 0x06cd, 0x043c }, /*                 Cyrillic_em м CYRILLIC SMALL LETTER EM */
  { 0x06ce, 0x043d }, /*                 Cyrillic_en н CYRILLIC SMALL LETTER EN */
  { 0x06cf, 0x043e }, /*                  Cyrillic_o о CYRILLIC SMALL LETTER O */
  { 0x06d0, 0x043f }, /*                 Cyrillic_pe п CYRILLIC SMALL LETTER PE */
  { 0x06d1, 0x044f }, /*                 Cyrillic_ya я CYRILLIC SMALL LETTER YA */
  { 0x06d2, 0x0440 }, /*                 Cyrillic_er р CYRILLIC SMALL LETTER ER */
  { 0x06d3, 0x0441 }, /*                 Cyrillic_es с CYRILLIC SMALL LETTER ES */
  { 0x06d4, 0x0442 }, /*                 Cyrillic_te т CYRILLIC SMALL LETTER TE */
  { 0x06d5, 0x0443 }, /*                  Cyrillic_u у CYRILLIC SMALL LETTER U */
  { 0x06d6, 0x0436 }, /*                Cyrillic_zhe ж CYRILLIC SMALL LETTER ZHE */
  { 0x06d7, 0x0432 }, /*                 Cyrillic_ve в CYRILLIC SMALL LETTER VE */
  { 0x06d8, 0x044c }, /*           Cyrillic_softsign ь CYRILLIC SMALL LETTER SOFT SIGN */
  { 0x06d9, 0x044b }, /*               Cyrillic_yeru ы CYRILLIC SMALL LETTER YERU */
  { 0x06da, 0x0437 }, /*                 Cyrillic_ze з CYRILLIC SMALL LETTER ZE */
  { 0x06db, 0x0448 }, /*                Cyrillic_sha ш CYRILLIC SMALL LETTER SHA */
  { 0x06dc, 0x044d }, /*                  Cyrillic_e э CYRILLIC SMALL LETTER E */
  { 0x06dd, 0x0449 }, /*              Cyrillic_shcha щ CYRILLIC SMALL LETTER SHCHA */
  { 0x06de, 0x0447 }, /*                Cyrillic_che ч CYRILLIC SMALL LETTER CHE */
  { 0x06df, 0x044a }, /*           Cyrillic_hardsign ъ CYRILLIC SMALL LETTER HARD SIGN */
  { 0x06e0, 0x042e }, /*                 Cyrillic_YU Ю CYRILLIC CAPITAL LETTER YU */
  { 0x06e1, 0x0410 }, /*                  Cyrillic_A А CYRILLIC CAPITAL LETTER A */
  { 0x06e2, 0x0411 }, /*                 Cyrillic_BE Б CYRILLIC CAPITAL LETTER BE */
  { 0x06e3, 0x0426 }, /*                Cyrillic_TSE Ц CYRILLIC CAPITAL LETTER TSE */
  { 0x06e4, 0x0414 }, /*                 Cyrillic_DE Д CYRILLIC CAPITAL LETTER DE */
  { 0x06e5, 0x0415 }, /*                 Cyrillic_IE Е CYRILLIC CAPITAL LETTER IE */
  { 0x06e6, 0x0424 }, /*                 Cyrillic_EF Ф CYRILLIC CAPITAL LETTER EF */
  { 0x06e7, 0x0413 }, /*                Cyrillic_GHE Г CYRILLIC CAPITAL LETTER GHE */
  { 0x06e8, 0x0425 }, /*                 Cyrillic_HA Х CYRILLIC CAPITAL LETTER HA */
  { 0x06e9, 0x0418 }, /*                  Cyrillic_I И CYRILLIC CAPITAL LETTER I */
  { 0x06ea, 0x0419 }, /*             Cyrillic_SHORTI Й CYRILLIC CAPITAL LETTER SHORT I */
  { 0x06eb, 0x041a }, /*                 Cyrillic_KA К CYRILLIC CAPITAL LETTER KA */
  { 0x06ec, 0x041b }, /*                 Cyrillic_EL Л CYRILLIC CAPITAL LETTER EL */
  { 0x06ed, 0x041c }, /*                 Cyrillic_EM М CYRILLIC CAPITAL LETTER EM */
  { 0x06ee, 0x041d }, /*                 Cyrillic_EN Н CYRILLIC CAPITAL LETTER EN */
  { 0x06ef, 0x041e }, /*                  Cyrillic_O О CYRILLIC CAPITAL LETTER O */
  { 0x06f0, 0x041f }, /*                 Cyrillic_PE П CYRILLIC CAPITAL LETTER PE */
  { 0x06f1, 0x042f }, /*                 Cyrillic_YA Я CYRILLIC CAPITAL LETTER YA */
  { 0x06f2, 0x0420 }, /*                 Cyrillic_ER Р CYRILLIC CAPITAL LETTER ER */
  { 0x06f3, 0x0421 }, /*                 Cyrillic_ES С CYRILLIC CAPITAL LETTER ES */
  { 0x06f4, 0x0422 }, /*                 Cyrillic_TE Т CYRILLIC CAPITAL LETTER TE */
  { 0x06f5, 0x0423 }, /*                  Cyrillic_U У CYRILLIC CAPITAL LETTER U */
  { 0x06f6, 0x0416 }, /*                Cyrillic_ZHE Ж CYRILLIC CAPITAL LETTER ZHE */
  { 0x06f7, 0x0412 }, /*                 Cyrillic_VE В CYRILLIC CAPITAL LETTER VE */
  { 0x06f8, 0x042c }, /*           Cyrillic_SOFTSIGN Ь CYRILLIC CAPITAL LETTER SOFT SIGN */
  { 0x06f9, 0x042b }, /*               Cyrillic_YERU Ы CYRILLIC CAPITAL LETTER YERU */
  { 0x06fa, 0x0417 }, /*                 Cyrillic_ZE З CYRILLIC CAPITAL LETTER ZE */
  { 0x06fb, 0x0428 }, /*                Cyrillic_SHA Ш CYRILLIC CAPITAL LETTER SHA */
  { 0x06fc, 0x042d }, /*                  Cyrillic_E Э CYRILLIC CAPITAL LETTER E */
  { 0x06fd, 0x0429 }, /*              Cyrillic_SHCHA Щ CYRILLIC CAPITAL LETTER SHCHA */
  { 0x06fe, 0x0427 }, /*                Cyrillic_CHE Ч CYRILLIC CAPITAL LETTER CHE */
  { 0x06ff, 0x042a }, /*           Cyrillic_HARDSIGN Ъ CYRILLIC CAPITAL LETTER HARD SIGN */
  { 0x07a1, 0x0386 }, /*           Greek_ALPHAaccent Ά GREEK CAPITAL LETTER ALPHA WITH TONOS */
  { 0x07a2, 0x0388 }, /*         Greek_EPSILONaccent Έ GREEK CAPITAL LETTER EPSILON WITH TONOS */
  { 0x07a3, 0x0389 }, /*             Greek_ETAaccent Ή GREEK CAPITAL LETTER ETA WITH TONOS */
  { 0x07a4, 0x038a }, /*            Greek_IOTAaccent Ί GREEK CAPITAL LETTER IOTA WITH TONOS */
  { 0x07a5, 0x03aa }, /*         Greek_IOTAdiaeresis Ϊ GREEK CAPITAL LETTER IOTA WITH DIALYTIKA */
  { 0x07a7, 0x038c }, /*         Greek_OMICRONaccent Ό GREEK CAPITAL LETTER OMICRON WITH TONOS */
  { 0x07a8, 0x038e }, /*         Greek_UPSILONaccent Ύ GREEK CAPITAL LETTER UPSILON WITH TONOS */
  { 0x07a9, 0x03ab }, /*       Greek_UPSILONdieresis Ϋ GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA */
  { 0x07ab, 0x038f }, /*           Greek_OMEGAaccent Ώ GREEK CAPITAL LETTER OMEGA WITH TONOS */
  { 0x07ae, 0x0385 }, /*        Greek_accentdieresis ΅ GREEK DIALYTIKA TONOS */
  { 0x07af, 0x2015 }, /*              Greek_horizbar ― HORIZONTAL BAR */
  { 0x07b1, 0x03ac }, /*           Greek_alphaaccent ά GREEK SMALL LETTER ALPHA WITH TONOS */
  { 0x07b2, 0x03ad }, /*         Greek_epsilonaccent έ GREEK SMALL LETTER EPSILON WITH TONOS */
  { 0x07b3, 0x03ae }, /*             Greek_etaaccent ή GREEK SMALL LETTER ETA WITH TONOS */
  { 0x07b4, 0x03af }, /*            Greek_iotaaccent ί GREEK SMALL LETTER IOTA WITH TONOS */
  { 0x07b5, 0x03ca }, /*          Greek_iotadieresis ϊ GREEK SMALL LETTER IOTA WITH DIALYTIKA */
  { 0x07b6, 0x0390 }, /*    Greek_iotaaccentdieresis ΐ GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS */
  { 0x07b7, 0x03cc }, /*         Greek_omicronaccent ό GREEK SMALL LETTER OMICRON WITH TONOS */
  { 0x07b8, 0x03cd }, /*         Greek_upsilonaccent ύ GREEK SMALL LETTER UPSILON WITH TONOS */
  { 0x07b9, 0x03cb }, /*       Greek_upsilondieresis ϋ GREEK SMALL LETTER UPSILON WITH DIALYTIKA */
  { 0x07ba, 0x03b0 }, /* Greek_upsilonaccentdieresis ΰ GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS */
  { 0x07bb, 0x03ce }, /*           Greek_omegaaccent ώ GREEK SMALL LETTER OMEGA WITH TONOS */
  { 0x07c1, 0x0391 }, /*                 Greek_ALPHA Α GREEK CAPITAL LETTER ALPHA */
  { 0x07c2, 0x0392 }, /*                  Greek_BETA Β GREEK CAPITAL LETTER BETA */
  { 0x07c3, 0x0393 }, /*                 Greek_GAMMA Γ GREEK CAPITAL LETTER GAMMA */
  { 0x07c4, 0x0394 }, /*                 Greek_DELTA Δ GREEK CAPITAL LETTER DELTA */
  { 0x07c5, 0x0395 }, /*               Greek_EPSILON Ε GREEK CAPITAL LETTER EPSILON */
  { 0x07c6, 0x0396 }, /*                  Greek_ZETA Ζ GREEK CAPITAL LETTER ZETA */
  { 0x07c7, 0x0397 }, /*                   Greek_ETA Η GREEK CAPITAL LETTER ETA */
  { 0x07c8, 0x0398 }, /*                 Greek_THETA Θ GREEK CAPITAL LETTER THETA */
  { 0x07c9, 0x0399 }, /*                  Greek_IOTA Ι GREEK CAPITAL LETTER IOTA */
  { 0x07ca, 0x039a }, /*                 Greek_KAPPA Κ GREEK CAPITAL LETTER KAPPA */
  { 0x07cb, 0x039b }, /*                Greek_LAMBDA Λ GREEK CAPITAL LETTER LAMDA */
  { 0x07cc, 0x039c }, /*                    Greek_MU Μ GREEK CAPITAL LETTER MU */
  { 0x07cd, 0x039d }, /*                    Greek_NU Ν GREEK CAPITAL LETTER NU */
  { 0x07ce, 0x039e }, /*                    Greek_XI Ξ GREEK CAPITAL LETTER XI */
  { 0x07cf, 0x039f }, /*               Greek_OMICRON Ο GREEK CAPITAL LETTER OMICRON */
  { 0x07d0, 0x03a0 }, /*                    Greek_PI Π GREEK CAPITAL LETTER PI */
  { 0x07d1, 0x03a1 }, /*                   Greek_RHO Ρ GREEK CAPITAL LETTER RHO */
  { 0x07d2, 0x03a3 }, /*                 Greek_SIGMA Σ GREEK CAPITAL LETTER SIGMA */
  { 0x07d4, 0x03a4 }, /*                   Greek_TAU Τ GREEK CAPITAL LETTER TAU */
  { 0x07d5, 0x03a5 }, /*               Greek_UPSILON Υ GREEK CAPITAL LETTER UPSILON */
  { 0x07d6, 0x03a6 }, /*                   Greek_PHI Φ GREEK CAPITAL LETTER PHI */
  { 0x07d7, 0x03a7 }, /*                   Greek_CHI Χ GREEK CAPITAL LETTER CHI */
  { 0x07d8, 0x03a8 }, /*                   Greek_PSI Ψ GREEK CAPITAL LETTER PSI */
  { 0x07d9, 0x03a9 }, /*                 Greek_OMEGA Ω GREEK CAPITAL LETTER OMEGA */
  { 0x07e1, 0x03b1 }, /*                 Greek_alpha α GREEK SMALL LETTER ALPHA */
  { 0x07e2, 0x03b2 }, /*                  Greek_beta β GREEK SMALL LETTER BETA */
  { 0x07e3, 0x03b3 }, /*                 Greek_gamma γ GREEK SMALL LETTER GAMMA */
  { 0x07e4, 0x03b4 }, /*                 Greek_delta δ GREEK SMALL LETTER DELTA */
  { 0x07e5, 0x03b5 }, /*               Greek_epsilon ε GREEK SMALL LETTER EPSILON */
  { 0x07e6, 0x03b6 }, /*                  Greek_zeta ζ GREEK SMALL LETTER ZETA */
  { 0x07e7, 0x03b7 }, /*                   Greek_eta η GREEK SMALL LETTER ETA */
  { 0x07e8, 0x03b8 }, /*                 Greek_theta θ GREEK SMALL LETTER THETA */
  { 0x07e9, 0x03b9 }, /*                  Greek_iota ι GREEK SMALL LETTER IOTA */
  { 0x07ea, 0x03ba }, /*                 Greek_kappa κ GREEK SMALL LETTER KAPPA */
  { 0x07eb, 0x03bb }, /*                Greek_lambda λ GREEK SMALL LETTER LAMDA */
  { 0x07ec, 0x03bc }, /*                    Greek_mu μ GREEK SMALL LETTER MU */
  { 0x07ed, 0x03bd }, /*                    Greek_nu ν GREEK SMALL LETTER NU */
  { 0x07ee, 0x03be }, /*                    Greek_xi ξ GREEK SMALL LETTER XI */
  { 0x07ef, 0x03bf }, /*               Greek_omicron ο GREEK SMALL LETTER OMICRON */
  { 0x07f0, 0x03c0 }, /*                    Greek_pi π GREEK SMALL LETTER PI */
  { 0x07f1, 0x03c1 }, /*                   Greek_rho ρ GREEK SMALL LETTER RHO */
  { 0x07f2, 0x03c3 }, /*                 Greek_sigma σ GREEK SMALL LETTER SIGMA */
  { 0x07f3, 0x03c2 }, /*       Greek_finalsmallsigma ς GREEK SMALL LETTER FINAL SIGMA */
  { 0x07f4, 0x03c4 }, /*                   Greek_tau τ GREEK SMALL LETTER TAU */
  { 0x07f5, 0x03c5 }, /*               Greek_upsilon υ GREEK SMALL LETTER UPSILON */
  { 0x07f6, 0x03c6 }, /*                   Greek_phi φ GREEK SMALL LETTER PHI */
  { 0x07f7, 0x03c7 }, /*                   Greek_chi χ GREEK SMALL LETTER CHI */
  { 0x07f8, 0x03c8 }, /*                   Greek_psi ψ GREEK SMALL LETTER PSI */
  { 0x07f9, 0x03c9 }, /*                 Greek_omega ω GREEK SMALL LETTER OMEGA */
/*  0x08a1                               leftradical ? ??? */
/*  0x08a2                            topleftradical ? ??? */
/*  0x08a3                            horizconnector ? ??? */
  { 0x08a4, 0x2320 }, /*                 topintegral ⌠ TOP HALF INTEGRAL */
  { 0x08a5, 0x2321 }, /*                 botintegral ⌡ BOTTOM HALF INTEGRAL */
  { 0x08a6, 0x2502 }, /*               vertconnector │ BOX DRAWINGS LIGHT VERTICAL */
/*  0x08a7                          topleftsqbracket ? ??? */
/*  0x08a8                          botleftsqbracket ? ??? */
/*  0x08a9                         toprightsqbracket ? ??? */
/*  0x08aa                         botrightsqbracket ? ??? */
/*  0x08ab                             topleftparens ? ??? */
/*  0x08ac                             botleftparens ? ??? */
/*  0x08ad                            toprightparens ? ??? */
/*  0x08ae                            botrightparens ? ??? */
/*  0x08af                      leftmiddlecurlybrace ? ??? */
/*  0x08b0                     rightmiddlecurlybrace ? ??? */
/*  0x08b1                          topleftsummation ? ??? */
/*  0x08b2                          botleftsummation ? ??? */
/*  0x08b3                 topvertsummationconnector ? ??? */
/*  0x08b4                 botvertsummationconnector ? ??? */
/*  0x08b5                         toprightsummation ? ??? */
/*  0x08b6                         botrightsummation ? ??? */
/*  0x08b7                      rightmiddlesummation ? ??? */
  { 0x08bc, 0x2264 }, /*               lessthanequal ≤ LESS-THAN OR EQUAL TO */
  { 0x08bd, 0x2260 }, /*                    notequal ≠ NOT EQUAL TO */
  { 0x08be, 0x2265 }, /*            greaterthanequal ≥ GREATER-THAN OR EQUAL TO */
  { 0x08bf, 0x222b }, /*                    integral ∫ INTEGRAL */
  { 0x08c0, 0x2234 }, /*                   therefore ∴ THEREFORE */
  { 0x08c1, 0x221d }, /*                   variation ∝ PROPORTIONAL TO */
  { 0x08c2, 0x221e }, /*                    infinity ∞ INFINITY */
  { 0x08c5, 0x2207 }, /*                       nabla ∇ NABLA */
  { 0x08c8, 0x2245 }, /*                 approximate ≅ APPROXIMATELY EQUAL TO */
/*  0x08c9                              similarequal ? ??? */
  { 0x08cd, 0x21d4 }, /*                    ifonlyif ⇔ LEFT RIGHT DOUBLE ARROW */
  { 0x08ce, 0x21d2 }, /*                     implies ⇒ RIGHTWARDS DOUBLE ARROW */
  { 0x08cf, 0x2261 }, /*                   identical ≡ IDENTICAL TO */
  { 0x08d6, 0x221a }, /*                     radical √ SQUARE ROOT */
  { 0x08da, 0x2282 }, /*                  includedin ⊂ SUBSET OF */
  { 0x08db, 0x2283 }, /*                    includes ⊃ SUPERSET OF */
  { 0x08dc, 0x2229 }, /*                intersection ∩ INTERSECTION */
  { 0x08dd, 0x222a }, /*                       union ∪ UNION */
  { 0x08de, 0x2227 }, /*                  logicaland ∧ LOGICAL AND */
  { 0x08df, 0x2228 }, /*                   logicalor ∨ LOGICAL OR */
  { 0x08ef, 0x2202 }, /*           partialderivative ∂ PARTIAL DIFFERENTIAL */
  { 0x08f6, 0x0192 }, /*                    function ƒ LATIN SMALL LETTER F WITH HOOK */
  { 0x08fb, 0x2190 }, /*                   leftarrow ← LEFTWARDS ARROW */
  { 0x08fc, 0x2191 }, /*                     uparrow ↑ UPWARDS ARROW */
  { 0x08fd, 0x2192 }, /*                  rightarrow → RIGHTWARDS ARROW */
  { 0x08fe, 0x2193 }, /*                   downarrow ↓ DOWNWARDS ARROW */
  { 0x09df, 0x2422 }, /*                       blank ␢ BLANK SYMBOL */
  { 0x09e0, 0x25c6 }, /*                soliddiamond ◆ BLACK DIAMOND */
  { 0x09e1, 0x2592 }, /*                checkerboard ▒ MEDIUM SHADE */
  { 0x09e2, 0x2409 }, /*                          ht ␉ SYMBOL FOR HORIZONTAL TABULATION */
  { 0x09e3, 0x240c }, /*                          ff ␌ SYMBOL FOR FORM FEED */
  { 0x09e4, 0x240d }, /*                          cr ␍ SYMBOL FOR CARRIAGE RETURN */
  { 0x09e5, 0x240a }, /*                          lf ␊ SYMBOL FOR LINE FEED */
  { 0x09e8, 0x2424 }, /*                          nl ␤ SYMBOL FOR NEWLINE */
  { 0x09e9, 0x240b }, /*                          vt ␋ SYMBOL FOR VERTICAL TABULATION */
  { 0x09ea, 0x2518 }, /*              lowrightcorner ┘ BOX DRAWINGS LIGHT UP AND LEFT */
  { 0x09eb, 0x2510 }, /*               uprightcorner ┐ BOX DRAWINGS LIGHT DOWN AND LEFT */
  { 0x09ec, 0x250c }, /*                upleftcorner ┌ BOX DRAWINGS LIGHT DOWN AND RIGHT */
  { 0x09ed, 0x2514 }, /*               lowleftcorner └ BOX DRAWINGS LIGHT UP AND RIGHT */
  { 0x09ee, 0x253c }, /*               crossinglines ┼ BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL */
/*  0x09ef                            horizlinescan1 ? ??? */
/*  0x09f0                            horizlinescan3 ? ??? */
  { 0x09f1, 0x2500 }, /*              horizlinescan5 ─ BOX DRAWINGS LIGHT HORIZONTAL */
/*  0x09f2                            horizlinescan7 ? ??? */
/*  0x09f3                            horizlinescan9 ? ??? */
  { 0x09f4, 0x251c }, /*                       leftt ├ BOX DRAWINGS LIGHT VERTICAL AND RIGHT */
  { 0x09f5, 0x2524 }, /*                      rightt ┤ BOX DRAWINGS LIGHT VERTICAL AND LEFT */
  { 0x09f6, 0x2534 }, /*                        bott ┴ BOX DRAWINGS LIGHT UP AND HORIZONTAL */
  { 0x09f7, 0x252c }, /*                        topt ┬ BOX DRAWINGS LIGHT DOWN AND HORIZONTAL */
  { 0x09f8, 0x2502 }, /*                     vertbar │ BOX DRAWINGS LIGHT VERTICAL */
  { 0x0aa1, 0x2003 }, /*                     emspace   EM SPACE */
  { 0x0aa2, 0x2002 }, /*                     enspace   EN SPACE */
  { 0x0aa3, 0x2004 }, /*                    em3space   THREE-PER-EM SPACE */
  { 0x0aa4, 0x2005 }, /*                    em4space   FOUR-PER-EM SPACE */
  { 0x0aa5, 0x2007 }, /*                  digitspace   FIGURE SPACE */
  { 0x0aa6, 0x2008 }, /*                  punctspace   PUNCTUATION SPACE */
  { 0x0aa7, 0x2009 }, /*                   thinspace   THIN SPACE */
  { 0x0aa8, 0x200a }, /*                   hairspace   HAIR SPACE */
  { 0x0aa9, 0x2014 }, /*                      emdash — EM DASH */
  { 0x0aaa, 0x2013 }, /*                      endash – EN DASH */
/*  0x0aac                               signifblank ? ??? */
  { 0x0aae, 0x2026 }, /*                    ellipsis … HORIZONTAL ELLIPSIS */
/*  0x0aaf                           doubbaselinedot ? ??? */
  { 0x0ab0, 0x2153 }, /*                    onethird ⅓ VULGAR FRACTION ONE THIRD */
  { 0x0ab1, 0x2154 }, /*                   twothirds ⅔ VULGAR FRACTION TWO THIRDS */
  { 0x0ab2, 0x2155 }, /*                    onefifth ⅕ VULGAR FRACTION ONE FIFTH */
  { 0x0ab3, 0x2156 }, /*                   twofifths ⅖ VULGAR FRACTION TWO FIFTHS */
  { 0x0ab4, 0x2157 }, /*                 threefifths ⅗ VULGAR FRACTION THREE FIFTHS */
  { 0x0ab5, 0x2158 }, /*                  fourfifths ⅘ VULGAR FRACTION FOUR FIFTHS */
  { 0x0ab6, 0x2159 }, /*                    onesixth ⅙ VULGAR FRACTION ONE SIXTH */
  { 0x0ab7, 0x215a }, /*                  fivesixths ⅚ VULGAR FRACTION FIVE SIXTHS */
  { 0x0ab8, 0x2105 }, /*                      careof ℅ CARE OF */
  { 0x0abb, 0x2012 }, /*                     figdash ‒ FIGURE DASH */
  { 0x0abc, 0x2329 }, /*            leftanglebracket 〈 LEFT-POINTING ANGLE BRACKET */
  { 0x0abd, 0x002e }, /*                decimalpoint . FULL STOP */
  { 0x0abe, 0x232a }, /*           rightanglebracket 〉 RIGHT-POINTING ANGLE BRACKET */
/*  0x0abf                                    marker ? ??? */
  { 0x0ac3, 0x215b }, /*                   oneeighth ⅛ VULGAR FRACTION ONE EIGHTH */
  { 0x0ac4, 0x215c }, /*                threeeighths ⅜ VULGAR FRACTION THREE EIGHTHS */
  { 0x0ac5, 0x215d }, /*                 fiveeighths ⅝ VULGAR FRACTION FIVE EIGHTHS */
  { 0x0ac6, 0x215e }, /*                seveneighths ⅞ VULGAR FRACTION SEVEN EIGHTHS */
  { 0x0ac9, 0x2122 }, /*                   trademark ™ TRADE MARK SIGN */
  { 0x0aca, 0x2613 }, /*               signaturemark ☓ SALTIRE */
/*  0x0acb                         trademarkincircle ? ??? */
  { 0x0acc, 0x25c1 }, /*            leftopentriangle ◁ WHITE LEFT-POINTING TRIANGLE */
  { 0x0acd, 0x25b7 }, /*           rightopentriangle ▷ WHITE RIGHT-POINTING TRIANGLE */
  { 0x0ace, 0x25cb }, /*                emopencircle ○ WHITE CIRCLE */
  { 0x0acf, 0x25a1 }, /*             emopenrectangle □ WHITE SQUARE */
  { 0x0ad0, 0x2018 }, /*         leftsinglequotemark ‘ LEFT SINGLE QUOTATION MARK */
  { 0x0ad1, 0x2019 }, /*        rightsinglequotemark ’ RIGHT SINGLE QUOTATION MARK */
  { 0x0ad2, 0x201c }, /*         leftdoublequotemark “ LEFT DOUBLE QUOTATION MARK */
  { 0x0ad3, 0x201d }, /*        rightdoublequotemark ” RIGHT DOUBLE QUOTATION MARK */
  { 0x0ad4, 0x211e }, /*                prescription ℞ PRESCRIPTION TAKE */
  { 0x0ad6, 0x2032 }, /*                     minutes ′ PRIME */
  { 0x0ad7, 0x2033 }, /*                     seconds ″ DOUBLE PRIME */
  { 0x0ad9, 0x271d }, /*                  latincross ✝ LATIN CROSS */
/*  0x0ada                                  hexagram ? ??? */
  { 0x0adb, 0x25ac }, /*            filledrectbullet ▬ BLACK RECTANGLE */
  { 0x0adc, 0x25c0 }, /*         filledlefttribullet ◀ BLACK LEFT-POINTING TRIANGLE */
  { 0x0add, 0x25b6 }, /*        filledrighttribullet ▶ BLACK RIGHT-POINTING TRIANGLE */
  { 0x0ade, 0x25cf }, /*              emfilledcircle ● BLACK CIRCLE */
  { 0x0adf, 0x25a0 }, /*                emfilledrect ■ BLACK SQUARE */
  { 0x0ae0, 0x25e6 }, /*            enopencircbullet ◦ WHITE BULLET */
  { 0x0ae1, 0x25ab }, /*          enopensquarebullet ▫ WHITE SMALL SQUARE */
  { 0x0ae2, 0x25ad }, /*              openrectbullet ▭ WHITE RECTANGLE */
  { 0x0ae3, 0x25b3 }, /*             opentribulletup △ WHITE UP-POINTING TRIANGLE */
  { 0x0ae4, 0x25bd }, /*           opentribulletdown ▽ WHITE DOWN-POINTING TRIANGLE */
  { 0x0ae5, 0x2606 }, /*                    openstar ☆ WHITE STAR */
  { 0x0ae6, 0x2022 }, /*          enfilledcircbullet • BULLET */
  { 0x0ae7, 0x25aa }, /*            enfilledsqbullet ▪ BLACK SMALL SQUARE */
  { 0x0ae8, 0x25b2 }, /*           filledtribulletup ▲ BLACK UP-POINTING TRIANGLE */
  { 0x0ae9, 0x25bc }, /*         filledtribulletdown ▼ BLACK DOWN-POINTING TRIANGLE */
  { 0x0aea, 0x261c }, /*                 leftpointer ☜ WHITE LEFT POINTING INDEX */
  { 0x0aeb, 0x261e }, /*                rightpointer ☞ WHITE RIGHT POINTING INDEX */
  { 0x0aec, 0x2663 }, /*                        club ♣ BLACK CLUB SUIT */
  { 0x0aed, 0x2666 }, /*                     diamond ♦ BLACK DIAMOND SUIT */
  { 0x0aee, 0x2665 }, /*                       heart ♥ BLACK HEART SUIT */
  { 0x0af0, 0x2720 }, /*                maltesecross ✠ MALTESE CROSS */
  { 0x0af1, 0x2020 }, /*                      dagger † DAGGER */
  { 0x0af2, 0x2021 }, /*                doubledagger ‡ DOUBLE DAGGER */
  { 0x0af3, 0x2713 }, /*                   checkmark ✓ CHECK MARK */
  { 0x0af4, 0x2717 }, /*                 ballotcross ✗ BALLOT X */
  { 0x0af5, 0x266f }, /*                musicalsharp ♯ MUSIC SHARP SIGN */
  { 0x0af6, 0x266d }, /*                 musicalflat ♭ MUSIC FLAT SIGN */
  { 0x0af7, 0x2642 }, /*                  malesymbol ♂ MALE SIGN */
  { 0x0af8, 0x2640 }, /*                femalesymbol ♀ FEMALE SIGN */
  { 0x0af9, 0x260e }, /*                   telephone ☎ BLACK TELEPHONE */
  { 0x0afa, 0x2315 }, /*           telephonerecorder ⌕ TELEPHONE RECORDER */
  { 0x0afb, 0x2117 }, /*         phonographcopyright ℗ SOUND RECORDING COPYRIGHT */
  { 0x0afc, 0x2038 }, /*                       caret ‸ CARET */
  { 0x0afd, 0x201a }, /*          singlelowquotemark ‚ SINGLE LOW-9 QUOTATION MARK */
  { 0x0afe, 0x201e }, /*          doublelowquotemark „ DOUBLE LOW-9 QUOTATION MARK */
/*  0x0aff                                    cursor ? ??? */
  { 0x0ba3, 0x003c }, /*                   leftcaret < LESS-THAN SIGN */
  { 0x0ba6, 0x003e }, /*                  rightcaret > GREATER-THAN SIGN */
  { 0x0ba8, 0x2228 }, /*                   downcaret ∨ LOGICAL OR */
  { 0x0ba9, 0x2227 }, /*                     upcaret ∧ LOGICAL AND */
  { 0x0bc0, 0x00af }, /*                     overbar ¯ MACRON */
  { 0x0bc2, 0x22a4 }, /*                    downtack ⊤ DOWN TACK */
  { 0x0bc3, 0x2229 }, /*                      upshoe ∩ INTERSECTION */
  { 0x0bc4, 0x230a }, /*                   downstile ⌊ LEFT FLOOR */
  { 0x0bc6, 0x005f }, /*                    underbar _ LOW LINE */
  { 0x0bca, 0x2218 }, /*                         jot ∘ RING OPERATOR */
  { 0x0bcc, 0x2395 }, /*                        quad ⎕ APL FUNCTIONAL SYMBOL QUAD (Unicode 3.0) */
  { 0x0bce, 0x22a5 }, /*                      uptack ⊥ UP TACK */
  { 0x0bcf, 0x25cb }, /*                      circle ○ WHITE CIRCLE */
  { 0x0bd3, 0x2308 }, /*                     upstile ⌈ LEFT CEILING */
  { 0x0bd6, 0x222a }, /*                    downshoe ∪ UNION */
  { 0x0bd8, 0x2283 }, /*                   rightshoe ⊃ SUPERSET OF */
  { 0x0bda, 0x2282 }, /*                    leftshoe ⊂ SUBSET OF */
  { 0x0bdc, 0x22a3 }, /*                    lefttack ⊣ LEFT TACK */
  { 0x0bfc, 0x22a2 }, /*                   righttack ⊢ RIGHT TACK */
  { 0x0cdf, 0x2017 }, /*        hebrew_doublelowline ‗ DOUBLE LOW LINE */
  { 0x0ce0, 0x05d0 }, /*                hebrew_aleph א HEBREW LETTER ALEF */
  { 0x0ce1, 0x05d1 }, /*                  hebrew_bet ב HEBREW LETTER BET */
  { 0x0ce2, 0x05d2 }, /*                hebrew_gimel ג HEBREW LETTER GIMEL */
  { 0x0ce3, 0x05d3 }, /*                hebrew_dalet ד HEBREW LETTER DALET */
  { 0x0ce4, 0x05d4 }, /*                   hebrew_he ה HEBREW LETTER HE */
  { 0x0ce5, 0x05d5 }, /*                  hebrew_waw ו HEBREW LETTER VAV */
  { 0x0ce6, 0x05d6 }, /*                 hebrew_zain ז HEBREW LETTER ZAYIN */
  { 0x0ce7, 0x05d7 }, /*                 hebrew_chet ח HEBREW LETTER HET */
  { 0x0ce8, 0x05d8 }, /*                  hebrew_tet ט HEBREW LETTER TET */
  { 0x0ce9, 0x05d9 }, /*                  hebrew_yod י HEBREW LETTER YOD */
  { 0x0cea, 0x05da }, /*            hebrew_finalkaph ך HEBREW LETTER FINAL KAF */
  { 0x0ceb, 0x05db }, /*                 hebrew_kaph כ HEBREW LETTER KAF */
  { 0x0cec, 0x05dc }, /*                hebrew_lamed ל HEBREW LETTER LAMED */
  { 0x0ced, 0x05dd }, /*             hebrew_finalmem ם HEBREW LETTER FINAL MEM */
  { 0x0cee, 0x05de }, /*                  hebrew_mem מ HEBREW LETTER MEM */
  { 0x0cef, 0x05df }, /*             hebrew_finalnun ן HEBREW LETTER FINAL NUN */
  { 0x0cf0, 0x05e0 }, /*                  hebrew_nun נ HEBREW LETTER NUN */
  { 0x0cf1, 0x05e1 }, /*               hebrew_samech ס HEBREW LETTER SAMEKH */
  { 0x0cf2, 0x05e2 }, /*                 hebrew_ayin ע HEBREW LETTER AYIN */
  { 0x0cf3, 0x05e3 }, /*              hebrew_finalpe ף HEBREW LETTER FINAL PE */
  { 0x0cf4, 0x05e4 }, /*                   hebrew_pe פ HEBREW LETTER PE */
  { 0x0cf5, 0x05e5 }, /*            hebrew_finalzade ץ HEBREW LETTER FINAL TSADI */
  { 0x0cf6, 0x05e6 }, /*                 hebrew_zade צ HEBREW LETTER TSADI */
  { 0x0cf7, 0x05e7 }, /*                 hebrew_qoph ק HEBREW LETTER QOF */
  { 0x0cf8, 0x05e8 }, /*                 hebrew_resh ר HEBREW LETTER RESH */
  { 0x0cf9, 0x05e9 }, /*                 hebrew_shin ש HEBREW LETTER SHIN */
  { 0x0cfa, 0x05ea }, /*                  hebrew_taw ת HEBREW LETTER TAV */
  { 0x0da1, 0x0e01 }, /*                  Thai_kokai ก THAI CHARACTER KO KAI */
  { 0x0da2, 0x0e02 }, /*                Thai_khokhai ข THAI CHARACTER KHO KHAI */
  { 0x0da3, 0x0e03 }, /*               Thai_khokhuat ฃ THAI CHARACTER KHO KHUAT */
  { 0x0da4, 0x0e04 }, /*               Thai_khokhwai ค THAI CHARACTER KHO KHWAI */
  { 0x0da5, 0x0e05 }, /*                Thai_khokhon ฅ THAI CHARACTER KHO KHON */
  { 0x0da6, 0x0e06 }, /*             Thai_khorakhang ฆ THAI CHARACTER KHO RAKHANG */
  { 0x0da7, 0x0e07 }, /*                 Thai_ngongu ง THAI CHARACTER NGO NGU */
  { 0x0da8, 0x0e08 }, /*                Thai_chochan จ THAI CHARACTER CHO CHAN */
  { 0x0da9, 0x0e09 }, /*               Thai_choching ฉ THAI CHARACTER CHO CHING */
  { 0x0daa, 0x0e0a }, /*               Thai_chochang ช THAI CHARACTER CHO CHANG */
  { 0x0dab, 0x0e0b }, /*                   Thai_soso ซ THAI CHARACTER SO SO */
  { 0x0dac, 0x0e0c }, /*                Thai_chochoe ฌ THAI CHARACTER CHO CHOE */
  { 0x0dad, 0x0e0d }, /*                 Thai_yoying ญ THAI CHARACTER YO YING */
  { 0x0dae, 0x0e0e }, /*                Thai_dochada ฎ THAI CHARACTER DO CHADA */
  { 0x0daf, 0x0e0f }, /*                Thai_topatak ฏ THAI CHARACTER TO PATAK */
  { 0x0db0, 0x0e10 }, /*                Thai_thothan ฐ THAI CHARACTER THO THAN */
  { 0x0db1, 0x0e11 }, /*          Thai_thonangmontho ฑ THAI CHARACTER THO NANGMONTHO */
  { 0x0db2, 0x0e12 }, /*             Thai_thophuthao ฒ THAI CHARACTER THO PHUTHAO */
  { 0x0db3, 0x0e13 }, /*                  Thai_nonen ณ THAI CHARACTER NO NEN */
  { 0x0db4, 0x0e14 }, /*                  Thai_dodek ด THAI CHARACTER DO DEK */
  { 0x0db5, 0x0e15 }, /*                  Thai_totao ต THAI CHARACTER TO TAO */
  { 0x0db6, 0x0e16 }, /*               Thai_thothung ถ THAI CHARACTER THO THUNG */
  { 0x0db7, 0x0e17 }, /*              Thai_thothahan ท THAI CHARACTER THO THAHAN */
  { 0x0db8, 0x0e18 }, /*               Thai_thothong ธ THAI CHARACTER THO THONG */
  { 0x0db9, 0x0e19 }, /*                   Thai_nonu น THAI CHARACTER NO NU */
  { 0x0dba, 0x0e1a }, /*               Thai_bobaimai บ THAI CHARACTER BO BAIMAI */
  { 0x0dbb, 0x0e1b }, /*                  Thai_popla ป THAI CHARACTER PO PLA */
  { 0x0dbc, 0x0e1c }, /*               Thai_phophung ผ THAI CHARACTER PHO PHUNG */
  { 0x0dbd, 0x0e1d }, /*                   Thai_fofa ฝ THAI CHARACTER FO FA */
  { 0x0dbe, 0x0e1e }, /*                Thai_phophan พ THAI CHARACTER PHO PHAN */
  { 0x0dbf, 0x0e1f }, /*                  Thai_fofan ฟ THAI CHARACTER FO FAN */
  { 0x0dc0, 0x0e20 }, /*             Thai_phosamphao ภ THAI CHARACTER PHO SAMPHAO */
  { 0x0dc1, 0x0e21 }, /*                   Thai_moma ม THAI CHARACTER MO MA */
  { 0x0dc2, 0x0e22 }, /*                  Thai_yoyak ย THAI CHARACTER YO YAK */
  { 0x0dc3, 0x0e23 }, /*                  Thai_rorua ร THAI CHARACTER RO RUA */
  { 0x0dc4, 0x0e24 }, /*                     Thai_ru ฤ THAI CHARACTER RU */
  { 0x0dc5, 0x0e25 }, /*                 Thai_loling ล THAI CHARACTER LO LING */
  { 0x0dc6, 0x0e26 }, /*                     Thai_lu ฦ THAI CHARACTER LU */
  { 0x0dc7, 0x0e27 }, /*                 Thai_wowaen ว THAI CHARACTER WO WAEN */
  { 0x0dc8, 0x0e28 }, /*                 Thai_sosala ศ THAI CHARACTER SO SALA */
  { 0x0dc9, 0x0e29 }, /*                 Thai_sorusi ษ THAI CHARACTER SO RUSI */
  { 0x0dca, 0x0e2a }, /*                  Thai_sosua ส THAI CHARACTER SO SUA */
  { 0x0dcb, 0x0e2b }, /*                  Thai_hohip ห THAI CHARACTER HO HIP */
  { 0x0dcc, 0x0e2c }, /*                Thai_lochula ฬ THAI CHARACTER LO CHULA */
  { 0x0dcd, 0x0e2d }, /*                   Thai_oang อ THAI CHARACTER O ANG */
  { 0x0dce, 0x0e2e }, /*               Thai_honokhuk ฮ THAI CHARACTER HO NOKHUK */
  { 0x0dcf, 0x0e2f }, /*              Thai_paiyannoi ฯ THAI CHARACTER PAIYANNOI */
  { 0x0dd0, 0x0e30 }, /*                  Thai_saraa ะ THAI CHARACTER SARA A */
  { 0x0dd1, 0x0e31 }, /*             Thai_maihanakat ั THAI CHARACTER MAI HAN-AKAT */
  { 0x0dd2, 0x0e32 }, /*                 Thai_saraaa า THAI CHARACTER SARA AA */
  { 0x0dd3, 0x0e33 }, /*                 Thai_saraam ำ THAI CHARACTER SARA AM */
  { 0x0dd4, 0x0e34 }, /*                  Thai_sarai ิ THAI CHARACTER SARA I */
  { 0x0dd5, 0x0e35 }, /*                 Thai_saraii ี THAI CHARACTER SARA II */
  { 0x0dd6, 0x0e36 }, /*                 Thai_saraue ึ THAI CHARACTER SARA UE */
  { 0x0dd7, 0x0e37 }, /*                Thai_sarauee ื THAI CHARACTER SARA UEE */
  { 0x0dd8, 0x0e38 }, /*                  Thai_sarau ุ THAI CHARACTER SARA U */
  { 0x0dd9, 0x0e39 }, /*                 Thai_sarauu ู THAI CHARACTER SARA UU */
  { 0x0dda, 0x0e3a }, /*                Thai_phinthu ฺ THAI CHARACTER PHINTHU */
  { 0x0dde, 0x0e3e }, /*      Thai_maihanakat_maitho ฾ ??? */
  { 0x0ddf, 0x0e3f }, /*                   Thai_baht ฿ THAI CURRENCY SYMBOL BAHT */
  { 0x0de0, 0x0e40 }, /*                  Thai_sarae เ THAI CHARACTER SARA E */
  { 0x0de1, 0x0e41 }, /*                 Thai_saraae แ THAI CHARACTER SARA AE */
  { 0x0de2, 0x0e42 }, /*                  Thai_sarao โ THAI CHARACTER SARA O */
  { 0x0de3, 0x0e43 }, /*          Thai_saraaimaimuan ใ THAI CHARACTER SARA AI MAIMUAN */
  { 0x0de4, 0x0e44 }, /*         Thai_saraaimaimalai ไ THAI CHARACTER SARA AI MAIMALAI */
  { 0x0de5, 0x0e45 }, /*            Thai_lakkhangyao ๅ THAI CHARACTER LAKKHANGYAO */
  { 0x0de6, 0x0e46 }, /*               Thai_maiyamok ๆ THAI CHARACTER MAIYAMOK */
  { 0x0de7, 0x0e47 }, /*              Thai_maitaikhu ็ THAI CHARACTER MAITAIKHU */
  { 0x0de8, 0x0e48 }, /*                  Thai_maiek ่ THAI CHARACTER MAI EK */
  { 0x0de9, 0x0e49 }, /*                 Thai_maitho ้ THAI CHARACTER MAI THO */
  { 0x0dea, 0x0e4a }, /*                 Thai_maitri ๊ THAI CHARACTER MAI TRI */
  { 0x0deb, 0x0e4b }, /*            Thai_maichattawa ๋ THAI CHARACTER MAI CHATTAWA */
  { 0x0dec, 0x0e4c }, /*            Thai_thanthakhat ์ THAI CHARACTER THANTHAKHAT */
  { 0x0ded, 0x0e4d }, /*               Thai_nikhahit ํ THAI CHARACTER NIKHAHIT */
  { 0x0df0, 0x0e50 }, /*                 Thai_leksun ๐ THAI DIGIT ZERO */
  { 0x0df1, 0x0e51 }, /*                Thai_leknung ๑ THAI DIGIT ONE */
  { 0x0df2, 0x0e52 }, /*                Thai_leksong ๒ THAI DIGIT TWO */
  { 0x0df3, 0x0e53 }, /*                 Thai_leksam ๓ THAI DIGIT THREE */
  { 0x0df4, 0x0e54 }, /*                  Thai_leksi ๔ THAI DIGIT FOUR */
  { 0x0df5, 0x0e55 }, /*                  Thai_lekha ๕ THAI DIGIT FIVE */
  { 0x0df6, 0x0e56 }, /*                 Thai_lekhok ๖ THAI DIGIT SIX */
  { 0x0df7, 0x0e57 }, /*                Thai_lekchet ๗ THAI DIGIT SEVEN */
  { 0x0df8, 0x0e58 }, /*                Thai_lekpaet ๘ THAI DIGIT EIGHT */
  { 0x0df9, 0x0e59 }, /*                 Thai_lekkao ๙ THAI DIGIT NINE */
  { 0x0ea1, 0x3131 }, /*               Hangul_Kiyeog ㄱ HANGUL LETTER KIYEOK */
  { 0x0ea2, 0x3132 }, /*          Hangul_SsangKiyeog ㄲ HANGUL LETTER SSANGKIYEOK */
  { 0x0ea3, 0x3133 }, /*           Hangul_KiyeogSios ㄳ HANGUL LETTER KIYEOK-SIOS */
  { 0x0ea4, 0x3134 }, /*                Hangul_Nieun ㄴ HANGUL LETTER NIEUN */
  { 0x0ea5, 0x3135 }, /*           Hangul_NieunJieuj ㄵ HANGUL LETTER NIEUN-CIEUC */
  { 0x0ea6, 0x3136 }, /*           Hangul_NieunHieuh ㄶ HANGUL LETTER NIEUN-HIEUH */
  { 0x0ea7, 0x3137 }, /*               Hangul_Dikeud ㄷ HANGUL LETTER TIKEUT */
  { 0x0ea8, 0x3138 }, /*          Hangul_SsangDikeud ㄸ HANGUL LETTER SSANGTIKEUT */
  { 0x0ea9, 0x3139 }, /*                Hangul_Rieul ㄹ HANGUL LETTER RIEUL */
  { 0x0eaa, 0x313a }, /*          Hangul_RieulKiyeog ㄺ HANGUL LETTER RIEUL-KIYEOK */
  { 0x0eab, 0x313b }, /*           Hangul_RieulMieum ㄻ HANGUL LETTER RIEUL-MIEUM */
  { 0x0eac, 0x313c }, /*           Hangul_RieulPieub ㄼ HANGUL LETTER RIEUL-PIEUP */
  { 0x0ead, 0x313d }, /*            Hangul_RieulSios ㄽ HANGUL LETTER RIEUL-SIOS */
  { 0x0eae, 0x313e }, /*           Hangul_RieulTieut ㄾ HANGUL LETTER RIEUL-THIEUTH */
  { 0x0eaf, 0x313f }, /*          Hangul_RieulPhieuf ㄿ HANGUL LETTER RIEUL-PHIEUPH */
  { 0x0eb0, 0x3140 }, /*           Hangul_RieulHieuh ㅀ HANGUL LETTER RIEUL-HIEUH */
  { 0x0eb1, 0x3141 }, /*                Hangul_Mieum ㅁ HANGUL LETTER MIEUM */
  { 0x0eb2, 0x3142 }, /*                Hangul_Pieub ㅂ HANGUL LETTER PIEUP */
  { 0x0eb3, 0x3143 }, /*           Hangul_SsangPieub ㅃ HANGUL LETTER SSANGPIEUP */
  { 0x0eb4, 0x3144 }, /*            Hangul_PieubSios ㅄ HANGUL LETTER PIEUP-SIOS */
  { 0x0eb5, 0x3145 }, /*                 Hangul_Sios ㅅ HANGUL LETTER SIOS */
  { 0x0eb6, 0x3146 }, /*            Hangul_SsangSios ㅆ HANGUL LETTER SSANGSIOS */
  { 0x0eb7, 0x3147 }, /*                Hangul_Ieung ㅇ HANGUL LETTER IEUNG */
  { 0x0eb8, 0x3148 }, /*                Hangul_Jieuj ㅈ HANGUL LETTER CIEUC */
  { 0x0eb9, 0x3149 }, /*           Hangul_SsangJieuj ㅉ HANGUL LETTER SSANGCIEUC */
  { 0x0eba, 0x314a }, /*                Hangul_Cieuc ㅊ HANGUL LETTER CHIEUCH */
  { 0x0ebb, 0x314b }, /*               Hangul_Khieuq ㅋ HANGUL LETTER KHIEUKH */
  { 0x0ebc, 0x314c }, /*                Hangul_Tieut ㅌ HANGUL LETTER THIEUTH */
  { 0x0ebd, 0x314d }, /*               Hangul_Phieuf ㅍ HANGUL LETTER PHIEUPH */
  { 0x0ebe, 0x314e }, /*                Hangul_Hieuh ㅎ HANGUL LETTER HIEUH */
  { 0x0ebf, 0x314f }, /*                    Hangul_A ㅏ HANGUL LETTER A */
  { 0x0ec0, 0x3150 }, /*                   Hangul_AE ㅐ HANGUL LETTER AE */
  { 0x0ec1, 0x3151 }, /*                   Hangul_YA ㅑ HANGUL LETTER YA */
  { 0x0ec2, 0x3152 }, /*                  Hangul_YAE ㅒ HANGUL LETTER YAE */
  { 0x0ec3, 0x3153 }, /*                   Hangul_EO ㅓ HANGUL LETTER EO */
  { 0x0ec4, 0x3154 }, /*                    Hangul_E ㅔ HANGUL LETTER E */
  { 0x0ec5, 0x3155 }, /*                  Hangul_YEO ㅕ HANGUL LETTER YEO */
  { 0x0ec6, 0x3156 }, /*                   Hangul_YE ㅖ HANGUL LETTER YE */
  { 0x0ec7, 0x3157 }, /*                    Hangul_O ㅗ HANGUL LETTER O */
  { 0x0ec8, 0x3158 }, /*                   Hangul_WA ㅘ HANGUL LETTER WA */
  { 0x0ec9, 0x3159 }, /*                  Hangul_WAE ㅙ HANGUL LETTER WAE */
  { 0x0eca, 0x315a }, /*                   Hangul_OE ㅚ HANGUL LETTER OE */
  { 0x0ecb, 0x315b }, /*                   Hangul_YO ㅛ HANGUL LETTER YO */
  { 0x0ecc, 0x315c }, /*                    Hangul_U ㅜ HANGUL LETTER U */
  { 0x0ecd, 0x315d }, /*                  Hangul_WEO ㅝ HANGUL LETTER WEO */
  { 0x0ece, 0x315e }, /*                   Hangul_WE ㅞ HANGUL LETTER WE */
  { 0x0ecf, 0x315f }, /*                   Hangul_WI ㅟ HANGUL LETTER WI */
  { 0x0ed0, 0x3160 }, /*                   Hangul_YU ㅠ HANGUL LETTER YU */
  { 0x0ed1, 0x3161 }, /*                   Hangul_EU ㅡ HANGUL LETTER EU */
  { 0x0ed2, 0x3162 }, /*                   Hangul_YI ㅢ HANGUL LETTER YI */
  { 0x0ed3, 0x3163 }, /*                    Hangul_I ㅣ HANGUL LETTER I */
  { 0x0ed4, 0x11a8 }, /*             Hangul_J_Kiyeog ᆨ HANGUL JONGSEONG KIYEOK */
  { 0x0ed5, 0x11a9 }, /*        Hangul_J_SsangKiyeog ᆩ HANGUL JONGSEONG SSANGKIYEOK */
  { 0x0ed6, 0x11aa }, /*         Hangul_J_KiyeogSios ᆪ HANGUL JONGSEONG KIYEOK-SIOS */
  { 0x0ed7, 0x11ab }, /*              Hangul_J_Nieun ᆫ HANGUL JONGSEONG NIEUN */
  { 0x0ed8, 0x11ac }, /*         Hangul_J_NieunJieuj ᆬ HANGUL JONGSEONG NIEUN-CIEUC */
  { 0x0ed9, 0x11ad }, /*         Hangul_J_NieunHieuh ᆭ HANGUL JONGSEONG NIEUN-HIEUH */
  { 0x0eda, 0x11ae }, /*             Hangul_J_Dikeud ᆮ HANGUL JONGSEONG TIKEUT */
  { 0x0edb, 0x11af }, /*              Hangul_J_Rieul ᆯ HANGUL JONGSEONG RIEUL */
  { 0x0edc, 0x11b0 }, /*        Hangul_J_RieulKiyeog ᆰ HANGUL JONGSEONG RIEUL-KIYEOK */
  { 0x0edd, 0x11b1 }, /*         Hangul_J_RieulMieum ᆱ HANGUL JONGSEONG RIEUL-MIEUM */
  { 0x0ede, 0x11b2 }, /*         Hangul_J_RieulPieub ᆲ HANGUL JONGSEONG RIEUL-PIEUP */
  { 0x0edf, 0x11b3 }, /*          Hangul_J_RieulSios ᆳ HANGUL JONGSEONG RIEUL-SIOS */
  { 0x0ee0, 0x11b4 }, /*         Hangul_J_RieulTieut ᆴ HANGUL JONGSEONG RIEUL-THIEUTH */
  { 0x0ee1, 0x11b5 }, /*        Hangul_J_RieulPhieuf ᆵ HANGUL JONGSEONG RIEUL-PHIEUPH */
  { 0x0ee2, 0x11b6 }, /*         Hangul_J_RieulHieuh ᆶ HANGUL JONGSEONG RIEUL-HIEUH */
  { 0x0ee3, 0x11b7 }, /*              Hangul_J_Mieum ᆷ HANGUL JONGSEONG MIEUM */
  { 0x0ee4, 0x11b8 }, /*              Hangul_J_Pieub ᆸ HANGUL JONGSEONG PIEUP */
  { 0x0ee5, 0x11b9 }, /*          Hangul_J_PieubSios ᆹ HANGUL JONGSEONG PIEUP-SIOS */
  { 0x0ee6, 0x11ba }, /*               Hangul_J_Sios ᆺ HANGUL JONGSEONG SIOS */
  { 0x0ee7, 0x11bb }, /*          Hangul_J_SsangSios ᆻ HANGUL JONGSEONG SSANGSIOS */
  { 0x0ee8, 0x11bc }, /*              Hangul_J_Ieung ᆼ HANGUL JONGSEONG IEUNG */
  { 0x0ee9, 0x11bd }, /*              Hangul_J_Jieuj ᆽ HANGUL JONGSEONG CIEUC */
  { 0x0eea, 0x11be }, /*              Hangul_J_Cieuc ᆾ HANGUL JONGSEONG CHIEUCH */
  { 0x0eeb, 0x11bf }, /*             Hangul_J_Khieuq ᆿ HANGUL JONGSEONG KHIEUKH */
  { 0x0eec, 0x11c0 }, /*              Hangul_J_Tieut ᇀ HANGUL JONGSEONG THIEUTH */
  { 0x0eed, 0x11c1 }, /*             Hangul_J_Phieuf ᇁ HANGUL JONGSEONG PHIEUPH */
  { 0x0eee, 0x11c2 }, /*              Hangul_J_Hieuh ᇂ HANGUL JONGSEONG HIEUH */
  { 0x0eef, 0x316d }, /*     Hangul_RieulYeorinHieuh ㅭ HANGUL LETTER RIEUL-YEORINHIEUH */
  { 0x0ef0, 0x3171 }, /*    Hangul_SunkyeongeumMieum ㅱ HANGUL LETTER KAPYEOUNMIEUM */
  { 0x0ef1, 0x3178 }, /*    Hangul_SunkyeongeumPieub ㅸ HANGUL LETTER KAPYEOUNPIEUP */
  { 0x0ef2, 0x317f }, /*              Hangul_PanSios ㅿ HANGUL LETTER PANSIOS */
/*  0x0ef3                  Hangul_KkogjiDalrinIeung ? ??? */
  { 0x0ef4, 0x3184 }, /*   Hangul_SunkyeongeumPhieuf ㆄ HANGUL LETTER KAPYEOUNPHIEUPH */
  { 0x0ef5, 0x3186 }, /*          Hangul_YeorinHieuh ㆆ HANGUL LETTER YEORINHIEUH */
  { 0x0ef6, 0x318d }, /*                Hangul_AraeA ㆍ HANGUL LETTER ARAEA */
  { 0x0ef7, 0x318e }, /*               Hangul_AraeAE ㆎ HANGUL LETTER ARAEAE */
  { 0x0ef8, 0x11eb }, /*            Hangul_J_PanSios ᇫ HANGUL JONGSEONG PANSIOS */
/*  0x0ef9                Hangul_J_KkogjiDalrinIeung ? ??? */
  { 0x0efa, 0x11f9 }, /*        Hangul_J_YeorinHieuh ᇹ HANGUL JONGSEONG YEORINHIEUH */
  { 0x0eff, 0x20a9 }, /*                  Korean_Won ₩ WON SIGN */
  { 0x13bc, 0x0152 }, /*                          OE Œ LATIN CAPITAL LIGATURE OE */
  { 0x13bd, 0x0153 }, /*                          oe œ LATIN SMALL LIGATURE OE */
  { 0x13be, 0x0178 }, /*                  Ydiaeresis Ÿ LATIN CAPITAL LETTER Y WITH DIAERESIS */
  { 0x20a0, 0x20a0 }, /*                     EcuSign ₠ EURO-CURRENCY SIGN */
  { 0x20a1, 0x20a1 }, /*                   ColonSign ₡ COLON SIGN */
  { 0x20a2, 0x20a2 }, /*                CruzeiroSign ₢ CRUZEIRO SIGN */
  { 0x20a3, 0x20a3 }, /*                  FFrancSign ₣ FRENCH FRANC SIGN */
  { 0x20a4, 0x20a4 }, /*                    LiraSign ₤ LIRA SIGN */
  { 0x20a5, 0x20a5 }, /*                    MillSign ₥ MILL SIGN */
  { 0x20a6, 0x20a6 }, /*                   NairaSign ₦ NAIRA SIGN */
  { 0x20a7, 0x20a7 }, /*                  PesetaSign ₧ PESETA SIGN */
  { 0x20a8, 0x20a8 }, /*                   RupeeSign ₨ RUPEE SIGN */
  { 0x20a9, 0x20a9 }, /*                     WonSign ₩ WON SIGN */
  { 0x20aa, 0x20aa }, /*               NewSheqelSign ₪ NEW SHEQEL SIGN */
  { 0x20ab, 0x20ab }, /*                    DongSign ₫ DONG SIGN */
  { 0x20ac, 0x20ac }, /*                    EuroSign € EURO SIGN */

  /* Following items added to SCIM, not in the xterm table */

  /* Function keys */
  { 0xFF09, 0x0009 },
  { 0xFF0A, 0x000a },
  { 0xFF0D, 0x000d },

  /* Numeric keypad */
  { 0xFF80 /* Space */, ' ' },
  { 0xFF89 /* Space */, 0x09 },
  { 0xFF8D /* Enter */, 0x0d },
  { 0xFFAA /* Multiply */, '*' },
  { 0xFFAB /* Add */, '+' },
  { 0xFFAD /* Subtract */, '-' },
  { 0xFFAE /* Decimal */, '.' },
  { 0xFFAF /* Divide */, '/' },
  { 0xFFB0 /* 0 */, '0' },
  { 0xFFB1 /* 1 */, '1' },
  { 0xFFB2 /* 2 */, '2' },
  { 0xFFB3 /* 3 */, '3' },
  { 0xFFB4 /* 4 */, '4' },
  { 0xFFB5 /* 5 */, '5' },
  { 0xFFB6 /* 6 */, '6' },
  { 0xFFB7 /* 7 */, '7' },
  { 0xFFB8 /* 8 */, '8' },
  { 0xFFB9 /* 9 */, '9' },
  { 0xFFBD /* Equal */, '=' },  

  /* End numeric keypad */
};

#define SCIM_NUM_KEY_UNICODES (sizeof (__scim_key_to_unicode_tab) / sizeof (__scim_key_to_unicode_tab[0]))

static __KeyName __scim_keys_by_code [] = {
  { 0x000020, "space" },
  { 0x000021, "exclam" },
  { 0x000022, "quotedbl" },
  { 0x000023, "numbersign" },
  { 0x000024, "dollar" },
  { 0x000025, "percent" },
  { 0x000026, "ampersand" },
  { 0x000027, "apostrophe" },
  { 0x000027, "quoteright" },
  { 0x000028, "parenleft" },
  { 0x000029, "parenright" },
  { 0x00002a, "asterisk" },
  { 0x00002b, "plus" },
  { 0x00002c, "comma" },
  { 0x00002d, "minus" },
  { 0x00002e, "period" },
  { 0x00002f, "slash" },
  { 0x000030, "0" },
  { 0x000031, "1" },
  { 0x000032, "2" },
  { 0x000033, "3" },
  { 0x000034, "4" },
  { 0x000035, "5" },
  { 0x000036, "6" },
  { 0x000037, "7" },
  { 0x000038, "8" },
  { 0x000039, "9" },
  { 0x00003a, "colon" },
  { 0x00003b, "semicolon" },
  { 0x00003c, "less" },
  { 0x00003d, "equal" },
  { 0x00003e, "greater" },
  { 0x00003f, "question" },
  { 0x000040, "at" },
  { 0x000041, "A" },
  { 0x000042, "B" },
  { 0x000043, "C" },
  { 0x000044, "D" },
  { 0x000045, "E" },
  { 0x000046, "F" },
  { 0x000047, "G" },
  { 0x000048, "H" },
  { 0x000049, "I" },
  { 0x00004a, "J" },
  { 0x00004b, "K" },
  { 0x00004c, "L" },
  { 0x00004d, "M" },
  { 0x00004e, "N" },
  { 0x00004f, "O" },
  { 0x000050, "P" },
  { 0x000051, "Q" },
  { 0x000052, "R" },
  { 0x000053, "S" },
  { 0x000054, "T" },
  { 0x000055, "U" },
  { 0x000056, "V" },
  { 0x000057, "W" },
  { 0x000058, "X" },
  { 0x000059, "Y" },
  { 0x00005a, "Z" },
  { 0x00005b, "bracketleft" },
  { 0x00005c, "backslash" },
  { 0x00005d, "bracketright" },
  { 0x00005e, "asciicircum" },
  { 0x00005f, "underscore" },
  { 0x000060, "grave" },
  { 0x000060, "quoteleft" },
  { 0x000061, "a" },
  { 0x000062, "b" },
  { 0x000063, "c" },
  { 0x000064, "d" },
  { 0x000065, "e" },
  { 0x000066, "f" },
  { 0x000067, "g" },
  { 0x000068, "h" },
  { 0x000069, "i" },
  { 0x00006a, "j" },
  { 0x00006b, "k" },
  { 0x00006c, "l" },
  { 0x00006d, "m" },
  { 0x00006e, "n" },
  { 0x00006f, "o" },
  { 0x000070, "p" },
  { 0x000071, "q" },
  { 0x000072, "r" },
  { 0x000073, "s" },
  { 0x000074, "t" },
  { 0x000075, "u" },
  { 0x000076, "v" },
  { 0x000077, "w" },
  { 0x000078, "x" },
  { 0x000079, "y" },
  { 0x00007a, "z" },
  { 0x00007b, "braceleft" },
  { 0x00007c, "bar" },
  { 0x00007d, "braceright" },
  { 0x00007e, "asciitilde" },
  { 0x0000a0, "nobreakspace" },
  { 0x0000a1, "exclamdown" },
  { 0x0000a2, "cent" },
  { 0x0000a3, "sterling" },
  { 0x0000a4, "currency" },
  { 0x0000a5, "yen" },
  { 0x0000a6, "brokenbar" },
  { 0x0000a7, "section" },
  { 0x0000a8, "diaeresis" },
  { 0x0000a9, "copyright" },
  { 0x0000aa, "ordfeminine" },
  { 0x0000ab, "guillemotleft" },
  { 0x0000ac, "notsign" },
  { 0x0000ad, "hyphen" },
  { 0x0000ae, "registered" },
  { 0x0000af, "macron" },
  { 0x0000b0, "degree" },
  { 0x0000b1, "plusminus" },
  { 0x0000b2, "twosuperior" },
  { 0x0000b3, "threesuperior" },
  { 0x0000b4, "acute" },
  { 0x0000b5, "mu" },
  { 0x0000b6, "paragraph" },
  { 0x0000b7, "periodcentered" },
  { 0x0000b8, "cedilla" },
  { 0x0000b9, "onesuperior" },
  { 0x0000ba, "masculine" },
  { 0x0000bb, "guillemotright" },
  { 0x0000bc, "onequarter" },
  { 0x0000bd, "onehalf" },
  { 0x0000be, "threequarters" },
  { 0x0000bf, "questiondown" },
  { 0x0000c0, "Agrave" },
  { 0x0000c1, "Aacute" },
  { 0x0000c2, "Acircumflex" },
  { 0x0000c3, "Atilde" },
  { 0x0000c4, "Adiaeresis" },
  { 0x0000c5, "Aring" },
  { 0x0000c6, "AE" },
  { 0x0000c7, "Ccedilla" },
  { 0x0000c8, "Egrave" },
  { 0x0000c9, "Eacute" },
  { 0x0000ca, "Ecircumflex" },
  { 0x0000cb, "Ediaeresis" },
  { 0x0000cc, "Igrave" },
  { 0x0000cd, "Iacute" },
  { 0x0000ce, "Icircumflex" },
  { 0x0000cf, "Idiaeresis" },
  { 0x0000d0, "ETH" },
  { 0x0000d0, "Eth" },
  { 0x0000d1, "Ntilde" },
  { 0x0000d2, "Ograve" },
  { 0x0000d3, "Oacute" },
  { 0x0000d4, "Ocircumflex" },
  { 0x0000d5, "Otilde" },
  { 0x0000d6, "Odiaeresis" },
  { 0x0000d7, "multiply" },
  { 0x0000d8, "Ooblique" },
  { 0x0000d9, "Ugrave" },
  { 0x0000da, "Uacute" },
  { 0x0000db, "Ucircumflex" },
  { 0x0000dc, "Udiaeresis" },
  { 0x0000dd, "Yacute" },
  { 0x0000de, "THORN" },
  { 0x0000de, "Thorn" },
  { 0x0000df, "ssharp" },
  { 0x0000e0, "agrave" },
  { 0x0000e1, "aacute" },
  { 0x0000e2, "acircumflex" },
  { 0x0000e3, "atilde" },
  { 0x0000e4, "adiaeresis" },
  { 0x0000e5, "aring" },
  { 0x0000e6, "ae" },
  { 0x0000e7, "ccedilla" },
  { 0x0000e8, "egrave" },
  { 0x0000e9, "eacute" },
  { 0x0000ea, "ecircumflex" },
  { 0x0000eb, "ediaeresis" },
  { 0x0000ec, "igrave" },
  { 0x0000ed, "iacute" },
  { 0x0000ee, "icircumflex" },
  { 0x0000ef, "idiaeresis" },
  { 0x0000f0, "eth" },
  { 0x0000f1, "ntilde" },
  { 0x0000f2, "ograve" },
  { 0x0000f3, "oacute" },
  { 0x0000f4, "ocircumflex" },
  { 0x0000f5, "otilde" },
  { 0x0000f6, "odiaeresis" },
  { 0x0000f7, "division" },
  { 0x0000f8, "oslash" },
  { 0x0000f9, "ugrave" },
  { 0x0000fa, "uacute" },
  { 0x0000fb, "ucircumflex" },
  { 0x0000fc, "udiaeresis" },
  { 0x0000fd, "yacute" },
  { 0x0000fe, "thorn" },
  { 0x0000ff, "ydiaeresis" },
  { 0x0001a1, "Aogonek" },
  { 0x0001a2, "breve" },
  { 0x0001a3, "Lstroke" },
  { 0x0001a5, "Lcaron" },
  { 0x0001a6, "Sacute" },
  { 0x0001a9, "Scaron" },
  { 0x0001aa, "Scedilla" },
  { 0x0001ab, "Tcaron" },
  { 0x0001ac, "Zacute" },
  { 0x0001ae, "Zcaron" },
  { 0x0001af, "Zabovedot" },
  { 0x0001b1, "aogonek" },
  { 0x0001b2, "ogonek" },
  { 0x0001b3, "lstroke" },
  { 0x0001b5, "lcaron" },
  { 0x0001b6, "sacute" },
  { 0x0001b7, "caron" },
  { 0x0001b9, "scaron" },
  { 0x0001ba, "scedilla" },
  { 0x0001bb, "tcaron" },
  { 0x0001bc, "zacute" },
  { 0x0001bd, "doubleacute" },
  { 0x0001be, "zcaron" },
  { 0x0001bf, "zabovedot" },
  { 0x0001c0, "Racute" },
  { 0x0001c3, "Abreve" },
  { 0x0001c5, "Lacute" },
  { 0x0001c6, "Cacute" },
  { 0x0001c8, "Ccaron" },
  { 0x0001ca, "Eogonek" },
  { 0x0001cc, "Ecaron" },
  { 0x0001cf, "Dcaron" },
  { 0x0001d0, "Dstroke" },
  { 0x0001d1, "Nacute" },
  { 0x0001d2, "Ncaron" },
  { 0x0001d5, "Odoubleacute" },
  { 0x0001d8, "Rcaron" },
  { 0x0001d9, "Uring" },
  { 0x0001db, "Udoubleacute" },
  { 0x0001de, "Tcedilla" },
  { 0x0001e0, "racute" },
  { 0x0001e3, "abreve" },
  { 0x0001e5, "lacute" },
  { 0x0001e6, "cacute" },
  { 0x0001e8, "ccaron" },
  { 0x0001ea, "eogonek" },
  { 0x0001ec, "ecaron" },
  { 0x0001ef, "dcaron" },
  { 0x0001f0, "dstroke" },
  { 0x0001f1, "nacute" },
  { 0x0001f2, "ncaron" },
  { 0x0001f5, "odoubleacute" },
  { 0x0001f8, "rcaron" },
  { 0x0001f9, "uring" },
  { 0x0001fb, "udoubleacute" },
  { 0x0001fe, "tcedilla" },
  { 0x0001ff, "abovedot" },
  { 0x0002a1, "Hstroke" },
  { 0x0002a6, "Hcircumflex" },
  { 0x0002a9, "Iabovedot" },
  { 0x0002ab, "Gbreve" },
  { 0x0002ac, "Jcircumflex" },
  { 0x0002b1, "hstroke" },
  { 0x0002b6, "hcircumflex" },
  { 0x0002b9, "idotless" },
  { 0x0002bb, "gbreve" },
  { 0x0002bc, "jcircumflex" },
  { 0x0002c5, "Cabovedot" },
  { 0x0002c6, "Ccircumflex" },
  { 0x0002d5, "Gabovedot" },
  { 0x0002d8, "Gcircumflex" },
  { 0x0002dd, "Ubreve" },
  { 0x0002de, "Scircumflex" },
  { 0x0002e5, "cabovedot" },
  { 0x0002e6, "ccircumflex" },
  { 0x0002f5, "gabovedot" },
  { 0x0002f8, "gcircumflex" },
  { 0x0002fd, "ubreve" },
  { 0x0002fe, "scircumflex" },
  { 0x0003a2, "kappa" },
  { 0x0003a2, "kra" },
  { 0x0003a3, "Rcedilla" },
  { 0x0003a5, "Itilde" },
  { 0x0003a6, "Lcedilla" },
  { 0x0003aa, "Emacron" },
  { 0x0003ab, "Gcedilla" },
  { 0x0003ac, "Tslash" },
  { 0x0003b3, "rcedilla" },
  { 0x0003b5, "itilde" },
  { 0x0003b6, "lcedilla" },
  { 0x0003ba, "emacron" },
  { 0x0003bb, "gcedilla" },
  { 0x0003bc, "tslash" },
  { 0x0003bd, "ENG" },
  { 0x0003bf, "eng" },
  { 0x0003c0, "Amacron" },
  { 0x0003c7, "Iogonek" },
  { 0x0003cc, "Eabovedot" },
  { 0x0003cf, "Imacron" },
  { 0x0003d1, "Ncedilla" },
  { 0x0003d2, "Omacron" },
  { 0x0003d3, "Kcedilla" },
  { 0x0003d9, "Uogonek" },
  { 0x0003dd, "Utilde" },
  { 0x0003de, "Umacron" },
  { 0x0003e0, "amacron" },
  { 0x0003e7, "iogonek" },
  { 0x0003ec, "eabovedot" },
  { 0x0003ef, "imacron" },
  { 0x0003f1, "ncedilla" },
  { 0x0003f2, "omacron" },
  { 0x0003f3, "kcedilla" },
  { 0x0003f9, "uogonek" },
  { 0x0003fd, "utilde" },
  { 0x0003fe, "umacron" },
  { 0x00047e, "overline" },
  { 0x0004a1, "kana_fullstop" },
  { 0x0004a2, "kana_openingbracket" },
  { 0x0004a3, "kana_closingbracket" },
  { 0x0004a4, "kana_comma" },
  { 0x0004a5, "kana_conjunctive" },
  { 0x0004a5, "kana_middledot" },
  { 0x0004a6, "kana_WO" },
  { 0x0004a7, "kana_a" },
  { 0x0004a8, "kana_i" },
  { 0x0004a9, "kana_u" },
  { 0x0004aa, "kana_e" },
  { 0x0004ab, "kana_o" },
  { 0x0004ac, "kana_ya" },
  { 0x0004ad, "kana_yu" },
  { 0x0004ae, "kana_yo" },
  { 0x0004af, "kana_tsu" },
  { 0x0004af, "kana_tu" },
  { 0x0004b0, "prolongedsound" },
  { 0x0004b1, "kana_A" },
  { 0x0004b2, "kana_I" },
  { 0x0004b3, "kana_U" },
  { 0x0004b4, "kana_E" },
  { 0x0004b5, "kana_O" },
  { 0x0004b6, "kana_KA" },
  { 0x0004b7, "kana_KI" },
  { 0x0004b8, "kana_KU" },
  { 0x0004b9, "kana_KE" },
  { 0x0004ba, "kana_KO" },
  { 0x0004bb, "kana_SA" },
  { 0x0004bc, "kana_SHI" },
  { 0x0004bd, "kana_SU" },
  { 0x0004be, "kana_SE" },
  { 0x0004bf, "kana_SO" },
  { 0x0004c0, "kana_TA" },
  { 0x0004c1, "kana_CHI" },
  { 0x0004c1, "kana_TI" },
  { 0x0004c2, "kana_TSU" },
  { 0x0004c2, "kana_TU" },
  { 0x0004c3, "kana_TE" },
  { 0x0004c4, "kana_TO" },
  { 0x0004c5, "kana_NA" },
  { 0x0004c6, "kana_NI" },
  { 0x0004c7, "kana_NU" },
  { 0x0004c8, "kana_NE" },
  { 0x0004c9, "kana_NO" },
  { 0x0004ca, "kana_HA" },
  { 0x0004cb, "kana_HI" },
  { 0x0004cc, "kana_FU" },
  { 0x0004cc, "kana_HU" },
  { 0x0004cd, "kana_HE" },
  { 0x0004ce, "kana_HO" },
  { 0x0004cf, "kana_MA" },
  { 0x0004d0, "kana_MI" },
  { 0x0004d1, "kana_MU" },
  { 0x0004d2, "kana_ME" },
  { 0x0004d3, "kana_MO" },
  { 0x0004d4, "kana_YA" },
  { 0x0004d5, "kana_YU" },
  { 0x0004d6, "kana_YO" },
  { 0x0004d7, "kana_RA" },
  { 0x0004d8, "kana_RI" },
  { 0x0004d9, "kana_RU" },
  { 0x0004da, "kana_RE" },
  { 0x0004db, "kana_RO" },
  { 0x0004dc, "kana_WA" },
  { 0x0004dd, "kana_N" },
  { 0x0004de, "voicedsound" },
  { 0x0004df, "semivoicedsound" },
  { 0x0005ac, "Arabic_comma" },
  { 0x0005bb, "Arabic_semicolon" },
  { 0x0005bf, "Arabic_question_mark" },
  { 0x0005c1, "Arabic_hamza" },
  { 0x0005c2, "Arabic_maddaonalef" },
  { 0x0005c3, "Arabic_hamzaonalef" },
  { 0x0005c4, "Arabic_hamzaonwaw" },
  { 0x0005c5, "Arabic_hamzaunderalef" },
  { 0x0005c6, "Arabic_hamzaonyeh" },
  { 0x0005c7, "Arabic_alef" },
  { 0x0005c8, "Arabic_beh" },
  { 0x0005c9, "Arabic_tehmarbuta" },
  { 0x0005ca, "Arabic_teh" },
  { 0x0005cb, "Arabic_theh" },
  { 0x0005cc, "Arabic_jeem" },
  { 0x0005cd, "Arabic_hah" },
  { 0x0005ce, "Arabic_khah" },
  { 0x0005cf, "Arabic_dal" },
  { 0x0005d0, "Arabic_thal" },
  { 0x0005d1, "Arabic_ra" },
  { 0x0005d2, "Arabic_zain" },
  { 0x0005d3, "Arabic_seen" },
  { 0x0005d4, "Arabic_sheen" },
  { 0x0005d5, "Arabic_sad" },
  { 0x0005d6, "Arabic_dad" },
  { 0x0005d7, "Arabic_tah" },
  { 0x0005d8, "Arabic_zah" },
  { 0x0005d9, "Arabic_ain" },
  { 0x0005da, "Arabic_ghain" },
  { 0x0005e0, "Arabic_tatweel" },
  { 0x0005e1, "Arabic_feh" },
  { 0x0005e2, "Arabic_qaf" },
  { 0x0005e3, "Arabic_kaf" },
  { 0x0005e4, "Arabic_lam" },
  { 0x0005e5, "Arabic_meem" },
  { 0x0005e6, "Arabic_noon" },
  { 0x0005e7, "Arabic_ha" },
  { 0x0005e7, "Arabic_heh" },
  { 0x0005e8, "Arabic_waw" },
  { 0x0005e9, "Arabic_alefmaksura" },
  { 0x0005ea, "Arabic_yeh" },
  { 0x0005eb, "Arabic_fathatan" },
  { 0x0005ec, "Arabic_dammatan" },
  { 0x0005ed, "Arabic_kasratan" },
  { 0x0005ee, "Arabic_fatha" },
  { 0x0005ef, "Arabic_damma" },
  { 0x0005f0, "Arabic_kasra" },
  { 0x0005f1, "Arabic_shadda" },
  { 0x0005f2, "Arabic_sukun" },
  { 0x0006a1, "Serbian_dje" },
  { 0x0006a2, "Macedonia_gje" },
  { 0x0006a3, "Cyrillic_io" },
  { 0x0006a4, "Ukrainian_ie" },
  { 0x0006a4, "Ukranian_je" },
  { 0x0006a5, "Macedonia_dse" },
  { 0x0006a6, "Ukrainian_i" },
  { 0x0006a6, "Ukranian_i" },
  { 0x0006a7, "Ukrainian_yi" },
  { 0x0006a7, "Ukranian_yi" },
  { 0x0006a8, "Cyrillic_je" },
  { 0x0006a8, "Serbian_je" },
  { 0x0006a9, "Cyrillic_lje" },
  { 0x0006a9, "Serbian_lje" },
  { 0x0006aa, "Cyrillic_nje" },
  { 0x0006aa, "Serbian_nje" },
  { 0x0006ab, "Serbian_tshe" },
  { 0x0006ac, "Macedonia_kje" },
  { 0x0006ae, "Byelorussian_shortu" },
  { 0x0006af, "Cyrillic_dzhe" },
  { 0x0006af, "Serbian_dze" },
  { 0x0006b0, "numerosign" },
  { 0x0006b1, "Serbian_DJE" },
  { 0x0006b2, "Macedonia_GJE" },
  { 0x0006b3, "Cyrillic_IO" },
  { 0x0006b4, "Ukrainian_IE" },
  { 0x0006b4, "Ukranian_JE" },
  { 0x0006b5, "Macedonia_DSE" },
  { 0x0006b6, "Ukrainian_I" },
  { 0x0006b6, "Ukranian_I" },
  { 0x0006b7, "Ukrainian_YI" },
  { 0x0006b7, "Ukranian_YI" },
  { 0x0006b8, "Cyrillic_JE" },
  { 0x0006b8, "Serbian_JE" },
  { 0x0006b9, "Cyrillic_LJE" },
  { 0x0006b9, "Serbian_LJE" },
  { 0x0006ba, "Cyrillic_NJE" },
  { 0x0006ba, "Serbian_NJE" },
  { 0x0006bb, "Serbian_TSHE" },
  { 0x0006bc, "Macedonia_KJE" },
  { 0x0006be, "Byelorussian_SHORTU" },
  { 0x0006bf, "Cyrillic_DZHE" },
  { 0x0006bf, "Serbian_DZE" },
  { 0x0006c0, "Cyrillic_yu" },
  { 0x0006c1, "Cyrillic_a" },
  { 0x0006c2, "Cyrillic_be" },
  { 0x0006c3, "Cyrillic_tse" },
  { 0x0006c4, "Cyrillic_de" },
  { 0x0006c5, "Cyrillic_ie" },
  { 0x0006c6, "Cyrillic_ef" },
  { 0x0006c7, "Cyrillic_ghe" },
  { 0x0006c8, "Cyrillic_ha" },
  { 0x0006c9, "Cyrillic_i" },
  { 0x0006ca, "Cyrillic_shorti" },
  { 0x0006cb, "Cyrillic_ka" },
  { 0x0006cc, "Cyrillic_el" },
  { 0x0006cd, "Cyrillic_em" },
  { 0x0006ce, "Cyrillic_en" },
  { 0x0006cf, "Cyrillic_o" },
  { 0x0006d0, "Cyrillic_pe" },
  { 0x0006d1, "Cyrillic_ya" },
  { 0x0006d2, "Cyrillic_er" },
  { 0x0006d3, "Cyrillic_es" },
  { 0x0006d4, "Cyrillic_te" },
  { 0x0006d5, "Cyrillic_u" },
  { 0x0006d6, "Cyrillic_zhe" },
  { 0x0006d7, "Cyrillic_ve" },
  { 0x0006d8, "Cyrillic_softsign" },
  { 0x0006d9, "Cyrillic_yeru" },
  { 0x0006da, "Cyrillic_ze" },
  { 0x0006db, "Cyrillic_sha" },
  { 0x0006dc, "Cyrillic_e" },
  { 0x0006dd, "Cyrillic_shcha" },
  { 0x0006de, "Cyrillic_che" },
  { 0x0006df, "Cyrillic_hardsign" },
  { 0x0006e0, "Cyrillic_YU" },
  { 0x0006e1, "Cyrillic_A" },
  { 0x0006e2, "Cyrillic_BE" },
  { 0x0006e3, "Cyrillic_TSE" },
  { 0x0006e4, "Cyrillic_DE" },
  { 0x0006e5, "Cyrillic_IE" },
  { 0x0006e6, "Cyrillic_EF" },
  { 0x0006e7, "Cyrillic_GHE" },
  { 0x0006e8, "Cyrillic_HA" },
  { 0x0006e9, "Cyrillic_I" },
  { 0x0006ea, "Cyrillic_SHORTI" },
  { 0x0006eb, "Cyrillic_KA" },
  { 0x0006ec, "Cyrillic_EL" },
  { 0x0006ed, "Cyrillic_EM" },
  { 0x0006ee, "Cyrillic_EN" },
  { 0x0006ef, "Cyrillic_O" },
  { 0x0006f0, "Cyrillic_PE" },
  { 0x0006f1, "Cyrillic_YA" },
  { 0x0006f2, "Cyrillic_ER" },
  { 0x0006f3, "Cyrillic_ES" },
  { 0x0006f4, "Cyrillic_TE" },
  { 0x0006f5, "Cyrillic_U" },
  { 0x0006f6, "Cyrillic_ZHE" },
  { 0x0006f7, "Cyrillic_VE" },
  { 0x0006f8, "Cyrillic_SOFTSIGN" },
  { 0x0006f9, "Cyrillic_YERU" },
  { 0x0006fa, "Cyrillic_ZE" },
  { 0x0006fb, "Cyrillic_SHA" },
  { 0x0006fc, "Cyrillic_E" },
  { 0x0006fd, "Cyrillic_SHCHA" },
  { 0x0006fe, "Cyrillic_CHE" },
  { 0x0006ff, "Cyrillic_HARDSIGN" },
  { 0x0007a1, "Greek_ALPHAaccent" },
  { 0x0007a2, "Greek_EPSILONaccent" },
  { 0x0007a3, "Greek_ETAaccent" },
  { 0x0007a4, "Greek_IOTAaccent" },
  { 0x0007a5, "Greek_IOTAdiaeresis" },
  { 0x0007a7, "Greek_OMICRONaccent" },
  { 0x0007a8, "Greek_UPSILONaccent" },
  { 0x0007a9, "Greek_UPSILONdieresis" },
  { 0x0007ab, "Greek_OMEGAaccent" },
  { 0x0007ae, "Greek_accentdieresis" },
  { 0x0007af, "Greek_horizbar" },
  { 0x0007b1, "Greek_alphaaccent" },
  { 0x0007b2, "Greek_epsilonaccent" },
  { 0x0007b3, "Greek_etaaccent" },
  { 0x0007b4, "Greek_iotaaccent" },
  { 0x0007b5, "Greek_iotadieresis" },
  { 0x0007b6, "Greek_iotaaccentdieresis" },
  { 0x0007b7, "Greek_omicronaccent" },
  { 0x0007b8, "Greek_upsilonaccent" },
  { 0x0007b9, "Greek_upsilondieresis" },
  { 0x0007ba, "Greek_upsilonaccentdieresis" },
  { 0x0007bb, "Greek_omegaaccent" },
  { 0x0007c1, "Greek_ALPHA" },
  { 0x0007c2, "Greek_BETA" },
  { 0x0007c3, "Greek_GAMMA" },
  { 0x0007c4, "Greek_DELTA" },
  { 0x0007c5, "Greek_EPSILON" },
  { 0x0007c6, "Greek_ZETA" },
  { 0x0007c7, "Greek_ETA" },
  { 0x0007c8, "Greek_THETA" },
  { 0x0007c9, "Greek_IOTA" },
  { 0x0007ca, "Greek_KAPPA" },
  { 0x0007cb, "Greek_LAMBDA" },
  { 0x0007cb, "Greek_LAMDA" },
  { 0x0007cc, "Greek_MU" },
  { 0x0007cd, "Greek_NU" },
  { 0x0007ce, "Greek_XI" },
  { 0x0007cf, "Greek_OMICRON" },
  { 0x0007d0, "Greek_PI" },
  { 0x0007d1, "Greek_RHO" },
  { 0x0007d2, "Greek_SIGMA" },
  { 0x0007d4, "Greek_TAU" },
  { 0x0007d5, "Greek_UPSILON" },
  { 0x0007d6, "Greek_PHI" },
  { 0x0007d7, "Greek_CHI" },
  { 0x0007d8, "Greek_PSI" },
  { 0x0007d9, "Greek_OMEGA" },
  { 0x0007e1, "Greek_alpha" },
  { 0x0007e2, "Greek_beta" },
  { 0x0007e3, "Greek_gamma" },
  { 0x0007e4, "Greek_delta" },
  { 0x0007e5, "Greek_epsilon" },
  { 0x0007e6, "Greek_zeta" },
  { 0x0007e7, "Greek_eta" },
  { 0x0007e8, "Greek_theta" },
  { 0x0007e9, "Greek_iota" },
  { 0x0007ea, "Greek_kappa" },
  { 0x0007eb, "Greek_lambda" },
  { 0x0007eb, "Greek_lamda" },
  { 0x0007ec, "Greek_mu" },
  { 0x0007ed, "Greek_nu" },
  { 0x0007ee, "Greek_xi" },
  { 0x0007ef, "Greek_omicron" },
  { 0x0007f0, "Greek_pi" },
  { 0x0007f1, "Greek_rho" },
  { 0x0007f2, "Greek_sigma" },
  { 0x0007f3, "Greek_finalsmallsigma" },
  { 0x0007f4, "Greek_tau" },
  { 0x0007f5, "Greek_upsilon" },
  { 0x0007f6, "Greek_phi" },
  { 0x0007f7, "Greek_chi" },
  { 0x0007f8, "Greek_psi" },
  { 0x0007f9, "Greek_omega" },
  { 0x0008a1, "leftradical" },
  { 0x0008a2, "topleftradical" },
  { 0x0008a3, "horizconnector" },
  { 0x0008a4, "topintegral" },
  { 0x0008a5, "botintegral" },
  { 0x0008a6, "vertconnector" },
  { 0x0008a7, "topleftsqbracket" },
  { 0x0008a8, "botleftsqbracket" },
  { 0x0008a9, "toprightsqbracket" },
  { 0x0008aa, "botrightsqbracket" },
  { 0x0008ab, "topleftparens" },
  { 0x0008ac, "botleftparens" },
  { 0x0008ad, "toprightparens" },
  { 0x0008ae, "botrightparens" },
  { 0x0008af, "leftmiddlecurlybrace" },
  { 0x0008b0, "rightmiddlecurlybrace" },
  { 0x0008b1, "topleftsummation" },
  { 0x0008b2, "botleftsummation" },
  { 0x0008b3, "topvertsummationconnector" },
  { 0x0008b4, "botvertsummationconnector" },
  { 0x0008b5, "toprightsummation" },
  { 0x0008b6, "botrightsummation" },
  { 0x0008b7, "rightmiddlesummation" },
  { 0x0008bc, "lessthanequal" },
  { 0x0008bd, "notequal" },
  { 0x0008be, "greaterthanequal" },
  { 0x0008bf, "integral" },
  { 0x0008c0, "therefore" },
  { 0x0008c1, "variation" },
  { 0x0008c2, "infinity" },
  { 0x0008c5, "nabla" },
  { 0x0008c8, "approximate" },
  { 0x0008c9, "similarequal" },
  { 0x0008cd, "ifonlyif" },
  { 0x0008ce, "implies" },
  { 0x0008cf, "identical" },
  { 0x0008d6, "radical" },
  { 0x0008da, "includedin" },
  { 0x0008db, "includes" },
  { 0x0008dc, "intersection" },
  { 0x0008dd, "union" },
  { 0x0008de, "logicaland" },
  { 0x0008df, "logicalor" },
  { 0x0008ef, "partialderivative" },
  { 0x0008f6, "function" },
  { 0x0008fb, "leftarrow" },
  { 0x0008fc, "uparrow" },
  { 0x0008fd, "rightarrow" },
  { 0x0008fe, "downarrow" },
  { 0x0009df, "blank" },
  { 0x0009e0, "soliddiamond" },
  { 0x0009e1, "checkerboard" },
  { 0x0009e2, "ht" },
  { 0x0009e3, "ff" },
  { 0x0009e4, "cr" },
  { 0x0009e5, "lf" },
  { 0x0009e8, "nl" },
  { 0x0009e9, "vt" },
  { 0x0009ea, "lowrightcorner" },
  { 0x0009eb, "uprightcorner" },
  { 0x0009ec, "upleftcorner" },
  { 0x0009ed, "lowleftcorner" },
  { 0x0009ee, "crossinglines" },
  { 0x0009ef, "horizlinescan1" },
  { 0x0009f0, "horizlinescan3" },
  { 0x0009f1, "horizlinescan5" },
  { 0x0009f2, "horizlinescan7" },
  { 0x0009f3, "horizlinescan9" },
  { 0x0009f4, "leftt" },
  { 0x0009f5, "rightt" },
  { 0x0009f6, "bott" },
  { 0x0009f7, "topt" },
  { 0x0009f8, "vertbar" },
  { 0x000aa1, "emspace" },
  { 0x000aa2, "enspace" },
  { 0x000aa3, "em3space" },
  { 0x000aa4, "em4space" },
  { 0x000aa5, "digitspace" },
  { 0x000aa6, "punctspace" },
  { 0x000aa7, "thinspace" },
  { 0x000aa8, "hairspace" },
  { 0x000aa9, "emdash" },
  { 0x000aaa, "endash" },
  { 0x000aac, "signifblank" },
  { 0x000aae, "ellipsis" },
  { 0x000aaf, "doubbaselinedot" },
  { 0x000ab0, "onethird" },
  { 0x000ab1, "twothirds" },
  { 0x000ab2, "onefifth" },
  { 0x000ab3, "twofifths" },
  { 0x000ab4, "threefifths" },
  { 0x000ab5, "fourfifths" },
  { 0x000ab6, "onesixth" },
  { 0x000ab7, "fivesixths" },
  { 0x000ab8, "careof" },
  { 0x000abb, "figdash" },
  { 0x000abc, "leftanglebracket" },
  { 0x000abd, "decimalpoint" },
  { 0x000abe, "rightanglebracket" },
  { 0x000abf, "marker" },
  { 0x000ac3, "oneeighth" },
  { 0x000ac4, "threeeighths" },
  { 0x000ac5, "fiveeighths" },
  { 0x000ac6, "seveneighths" },
  { 0x000ac9, "trademark" },
  { 0x000aca, "signaturemark" },
  { 0x000acb, "trademarkincircle" },
  { 0x000acc, "leftopentriangle" },
  { 0x000acd, "rightopentriangle" },
  { 0x000ace, "emopencircle" },
  { 0x000acf, "emopenrectangle" },
  { 0x000ad0, "leftsinglequotemark" },
  { 0x000ad1, "rightsinglequotemark" },
  { 0x000ad2, "leftdoublequotemark" },
  { 0x000ad3, "rightdoublequotemark" },
  { 0x000ad4, "prescription" },
  { 0x000ad6, "minutes" },
  { 0x000ad7, "seconds" },
  { 0x000ad9, "latincross" },
  { 0x000ada, "hexagram" },
  { 0x000adb, "filledrectbullet" },
  { 0x000adc, "filledlefttribullet" },
  { 0x000add, "filledrighttribullet" },
  { 0x000ade, "emfilledcircle" },
  { 0x000adf, "emfilledrect" },
  { 0x000ae0, "enopencircbullet" },
  { 0x000ae1, "enopensquarebullet" },
  { 0x000ae2, "openrectbullet" },
  { 0x000ae3, "opentribulletup" },
  { 0x000ae4, "opentribulletdown" },
  { 0x000ae5, "openstar" },
  { 0x000ae6, "enfilledcircbullet" },
  { 0x000ae7, "enfilledsqbullet" },
  { 0x000ae8, "filledtribulletup" },
  { 0x000ae9, "filledtribulletdown" },
  { 0x000aea, "leftpointer" },
  { 0x000aeb, "rightpointer" },
  { 0x000aec, "club" },
  { 0x000aed, "diamond" },
  { 0x000aee, "heart" },
  { 0x000af0, "maltesecross" },
  { 0x000af1, "dagger" },
  { 0x000af2, "doubledagger" },
  { 0x000af3, "checkmark" },
  { 0x000af4, "ballotcross" },
  { 0x000af5, "musicalsharp" },
  { 0x000af6, "musicalflat" },
  { 0x000af7, "malesymbol" },
  { 0x000af8, "femalesymbol" },
  { 0x000af9, "telephone" },
  { 0x000afa, "telephonerecorder" },
  { 0x000afb, "phonographcopyright" },
  { 0x000afc, "caret" },
  { 0x000afd, "singlelowquotemark" },
  { 0x000afe, "doublelowquotemark" },
  { 0x000aff, "cursor" },
  { 0x000ba3, "leftcaret" },
  { 0x000ba6, "rightcaret" },
  { 0x000ba8, "downcaret" },
  { 0x000ba9, "upcaret" },
  { 0x000bc0, "overbar" },
  { 0x000bc2, "downtack" },
  { 0x000bc3, "upshoe" },
  { 0x000bc4, "downstile" },
  { 0x000bc6, "underbar" },
  { 0x000bca, "jot" },
  { 0x000bcc, "quad" },
  { 0x000bce, "uptack" },
  { 0x000bcf, "circle" },
  { 0x000bd3, "upstile" },
  { 0x000bd6, "downshoe" },
  { 0x000bd8, "rightshoe" },
  { 0x000bda, "leftshoe" },
  { 0x000bdc, "lefttack" },
  { 0x000bfc, "righttack" },
  { 0x000cdf, "hebrew_doublelowline" },
  { 0x000ce0, "hebrew_aleph" },
  { 0x000ce1, "hebrew_bet" },
  { 0x000ce1, "hebrew_beth" },
  { 0x000ce2, "hebrew_gimel" },
  { 0x000ce2, "hebrew_gimmel" },
  { 0x000ce3, "hebrew_dalet" },
  { 0x000ce3, "hebrew_daleth" },
  { 0x000ce4, "hebrew_he" },
  { 0x000ce5, "hebrew_waw" },
  { 0x000ce6, "hebrew_zain" },
  { 0x000ce6, "hebrew_zayin" },
  { 0x000ce7, "hebrew_chet" },
  { 0x000ce7, "hebrew_het" },
  { 0x000ce8, "hebrew_tet" },
  { 0x000ce8, "hebrew_teth" },
  { 0x000ce9, "hebrew_yod" },
  { 0x000cea, "hebrew_finalkaph" },
  { 0x000ceb, "hebrew_kaph" },
  { 0x000cec, "hebrew_lamed" },
  { 0x000ced, "hebrew_finalmem" },
  { 0x000cee, "hebrew_mem" },
  { 0x000cef, "hebrew_finalnun" },
  { 0x000cf0, "hebrew_nun" },
  { 0x000cf1, "hebrew_samech" },
  { 0x000cf1, "hebrew_samekh" },
  { 0x000cf2, "hebrew_ayin" },
  { 0x000cf3, "hebrew_finalpe" },
  { 0x000cf4, "hebrew_pe" },
  { 0x000cf5, "hebrew_finalzade" },
  { 0x000cf5, "hebrew_finalzadi" },
  { 0x000cf6, "hebrew_zade" },
  { 0x000cf6, "hebrew_zadi" },
  { 0x000cf7, "hebrew_kuf" },
  { 0x000cf7, "hebrew_qoph" },
  { 0x000cf8, "hebrew_resh" },
  { 0x000cf9, "hebrew_shin" },
  { 0x000cfa, "hebrew_taf" },
  { 0x000cfa, "hebrew_taw" },
  { 0x000da1, "Thai_kokai" },
  { 0x000da2, "Thai_khokhai" },
  { 0x000da3, "Thai_khokhuat" },
  { 0x000da4, "Thai_khokhwai" },
  { 0x000da5, "Thai_khokhon" },
  { 0x000da6, "Thai_khorakhang" },
  { 0x000da7, "Thai_ngongu" },
  { 0x000da8, "Thai_chochan" },
  { 0x000da9, "Thai_choching" },
  { 0x000daa, "Thai_chochang" },
  { 0x000dab, "Thai_soso" },
  { 0x000dac, "Thai_chochoe" },
  { 0x000dad, "Thai_yoying" },
  { 0x000dae, "Thai_dochada" },
  { 0x000daf, "Thai_topatak" },
  { 0x000db0, "Thai_thothan" },
  { 0x000db1, "Thai_thonangmontho" },
  { 0x000db2, "Thai_thophuthao" },
  { 0x000db3, "Thai_nonen" },
  { 0x000db4, "Thai_dodek" },
  { 0x000db5, "Thai_totao" },
  { 0x000db6, "Thai_thothung" },
  { 0x000db7, "Thai_thothahan" },
  { 0x000db8, "Thai_thothong" },
  { 0x000db9, "Thai_nonu" },
  { 0x000dba, "Thai_bobaimai" },
  { 0x000dbb, "Thai_popla" },
  { 0x000dbc, "Thai_phophung" },
  { 0x000dbd, "Thai_fofa" },
  { 0x000dbe, "Thai_phophan" },
  { 0x000dbf, "Thai_fofan" },
  { 0x000dc0, "Thai_phosamphao" },
  { 0x000dc1, "Thai_moma" },
  { 0x000dc2, "Thai_yoyak" },
  { 0x000dc3, "Thai_rorua" },
  { 0x000dc4, "Thai_ru" },
  { 0x000dc5, "Thai_loling" },
  { 0x000dc6, "Thai_lu" },
  { 0x000dc7, "Thai_wowaen" },
  { 0x000dc8, "Thai_sosala" },
  { 0x000dc9, "Thai_sorusi" },
  { 0x000dca, "Thai_sosua" },
  { 0x000dcb, "Thai_hohip" },
  { 0x000dcc, "Thai_lochula" },
  { 0x000dcd, "Thai_oang" },
  { 0x000dce, "Thai_honokhuk" },
  { 0x000dcf, "Thai_paiyannoi" },
  { 0x000dd0, "Thai_saraa" },
  { 0x000dd1, "Thai_maihanakat" },
  { 0x000dd2, "Thai_saraaa" },
  { 0x000dd3, "Thai_saraam" },
  { 0x000dd4, "Thai_sarai" },
  { 0x000dd5, "Thai_saraii" },
  { 0x000dd6, "Thai_saraue" },
  { 0x000dd7, "Thai_sarauee" },
  { 0x000dd8, "Thai_sarau" },
  { 0x000dd9, "Thai_sarauu" },
  { 0x000dda, "Thai_phinthu" },
  { 0x000dde, "Thai_maihanakat_maitho" },
  { 0x000ddf, "Thai_baht" },
  { 0x000de0, "Thai_sarae" },
  { 0x000de1, "Thai_saraae" },
  { 0x000de2, "Thai_sarao" },
  { 0x000de3, "Thai_saraaimaimuan" },
  { 0x000de4, "Thai_saraaimaimalai" },
  { 0x000de5, "Thai_lakkhangyao" },
  { 0x000de6, "Thai_maiyamok" },
  { 0x000de7, "Thai_maitaikhu" },
  { 0x000de8, "Thai_maiek" },
  { 0x000de9, "Thai_maitho" },
  { 0x000dea, "Thai_maitri" },
  { 0x000deb, "Thai_maichattawa" },
  { 0x000dec, "Thai_thanthakhat" },
  { 0x000ded, "Thai_nikhahit" },
  { 0x000df0, "Thai_leksun" },
  { 0x000df1, "Thai_leknung" },
  { 0x000df2, "Thai_leksong" },
  { 0x000df3, "Thai_leksam" },
  { 0x000df4, "Thai_leksi" },
  { 0x000df5, "Thai_lekha" },
  { 0x000df6, "Thai_lekhok" },
  { 0x000df7, "Thai_lekchet" },
  { 0x000df8, "Thai_lekpaet" },
  { 0x000df9, "Thai_lekkao" },
  { 0x000ea1, "Hangul_Kiyeog" },
  { 0x000ea2, "Hangul_SsangKiyeog" },
  { 0x000ea3, "Hangul_KiyeogSios" },
  { 0x000ea4, "Hangul_Nieun" },
  { 0x000ea5, "Hangul_NieunJieuj" },
  { 0x000ea6, "Hangul_NieunHieuh" },
  { 0x000ea7, "Hangul_Dikeud" },
  { 0x000ea8, "Hangul_SsangDikeud" },
  { 0x000ea9, "Hangul_Rieul" },
  { 0x000eaa, "Hangul_RieulKiyeog" },
  { 0x000eab, "Hangul_RieulMieum" },
  { 0x000eac, "Hangul_RieulPieub" },
  { 0x000ead, "Hangul_RieulSios" },
  { 0x000eae, "Hangul_RieulTieut" },
  { 0x000eaf, "Hangul_RieulPhieuf" },
  { 0x000eb0, "Hangul_RieulHieuh" },
  { 0x000eb1, "Hangul_Mieum" },
  { 0x000eb2, "Hangul_Pieub" },
  { 0x000eb3, "Hangul_SsangPieub" },
  { 0x000eb4, "Hangul_PieubSios" },
  { 0x000eb5, "Hangul_Sios" },
  { 0x000eb6, "Hangul_SsangSios" },
  { 0x000eb7, "Hangul_Ieung" },
  { 0x000eb8, "Hangul_Jieuj" },
  { 0x000eb9, "Hangul_SsangJieuj" },
  { 0x000eba, "Hangul_Cieuc" },
  { 0x000ebb, "Hangul_Khieuq" },
  { 0x000ebc, "Hangul_Tieut" },
  { 0x000ebd, "Hangul_Phieuf" },
  { 0x000ebe, "Hangul_Hieuh" },
  { 0x000ebf, "Hangul_A" },
  { 0x000ec0, "Hangul_AE" },
  { 0x000ec1, "Hangul_YA" },
  { 0x000ec2, "Hangul_YAE" },
  { 0x000ec3, "Hangul_EO" },
  { 0x000ec4, "Hangul_E" },
  { 0x000ec5, "Hangul_YEO" },
  { 0x000ec6, "Hangul_YE" },
  { 0x000ec7, "Hangul_O" },
  { 0x000ec8, "Hangul_WA" },
  { 0x000ec9, "Hangul_WAE" },
  { 0x000eca, "Hangul_OE" },
  { 0x000ecb, "Hangul_YO" },
  { 0x000ecc, "Hangul_U" },
  { 0x000ecd, "Hangul_WEO" },
  { 0x000ece, "Hangul_WE" },
  { 0x000ecf, "Hangul_WI" },
  { 0x000ed0, "Hangul_YU" },
  { 0x000ed1, "Hangul_EU" },
  { 0x000ed2, "Hangul_YI" },
  { 0x000ed3, "Hangul_I" },
  { 0x000ed4, "Hangul_J_Kiyeog" },
  { 0x000ed5, "Hangul_J_SsangKiyeog" },
  { 0x000ed6, "Hangul_J_KiyeogSios" },
  { 0x000ed7, "Hangul_J_Nieun" },
  { 0x000ed8, "Hangul_J_NieunJieuj" },
  { 0x000ed9, "Hangul_J_NieunHieuh" },
  { 0x000eda, "Hangul_J_Dikeud" },
  { 0x000edb, "Hangul_J_Rieul" },
  { 0x000edc, "Hangul_J_RieulKiyeog" },
  { 0x000edd, "Hangul_J_RieulMieum" },
  { 0x000ede, "Hangul_J_RieulPieub" },
  { 0x000edf, "Hangul_J_RieulSios" },
  { 0x000ee0, "Hangul_J_RieulTieut" },
  { 0x000ee1, "Hangul_J_RieulPhieuf" },
  { 0x000ee2, "Hangul_J_RieulHieuh" },
  { 0x000ee3, "Hangul_J_Mieum" },
  { 0x000ee4, "Hangul_J_Pieub" },
  { 0x000ee5, "Hangul_J_PieubSios" },
  { 0x000ee6, "Hangul_J_Sios" },
  { 0x000ee7, "Hangul_J_SsangSios" },
  { 0x000ee8, "Hangul_J_Ieung" },
  { 0x000ee9, "Hangul_J_Jieuj" },
  { 0x000eea, "Hangul_J_Cieuc" },
  { 0x000eeb, "Hangul_J_Khieuq" },
  { 0x000eec, "Hangul_J_Tieut" },
  { 0x000eed, "Hangul_J_Phieuf" },
  { 0x000eee, "Hangul_J_Hieuh" },
  { 0x000eef, "Hangul_RieulYeorinHieuh" },
  { 0x000ef0, "Hangul_SunkyeongeumMieum" },
  { 0x000ef1, "Hangul_SunkyeongeumPieub" },
  { 0x000ef2, "Hangul_PanSios" },
  { 0x000ef3, "Hangul_KkogjiDalrinIeung" },
  { 0x000ef4, "Hangul_SunkyeongeumPhieuf" },
  { 0x000ef5, "Hangul_YeorinHieuh" },
  { 0x000ef6, "Hangul_AraeA" },
  { 0x000ef7, "Hangul_AraeAE" },
  { 0x000ef8, "Hangul_J_PanSios" },
  { 0x000ef9, "Hangul_J_KkogjiDalrinIeung" },
  { 0x000efa, "Hangul_J_YeorinHieuh" },
  { 0x000eff, "Korean_Won" },
  { 0x0013bc, "OE" },
  { 0x0013bd, "oe" },
  { 0x0013be, "Ydiaeresis" },

  { 0x001e9f, "combining_tilde" },
  { 0x001ef2, "combining_grave" },
  { 0x001ef3, "combining_acute" },
  { 0x001efa, "Ohorn" },
  { 0x001efb, "ohorn" },
  { 0x001efc, "Uhorn" },
  { 0x001efd, "uhorn" },
  { 0x001efe, "combining_hook" },
  { 0x001eff, "combining_belowdot" },

  { 0x0020a0, "EcuSign" },
  { 0x0020a1, "ColonSign" },
  { 0x0020a2, "CruzeiroSign" },
  { 0x0020a3, "FFrancSign" },
  { 0x0020a4, "LiraSign" },
  { 0x0020a5, "MillSign" },
  { 0x0020a6, "NairaSign" },
  { 0x0020a7, "PesetaSign" },
  { 0x0020a8, "RupeeSign" },
  { 0x0020a9, "WonSign" },
  { 0x0020aa, "NewSheqelSign" },
  { 0x0020ab, "DongSign" },
  { 0x0020ac, "EuroSign" },
  { 0x00fd01, "3270_Duplicate" },
  { 0x00fd02, "3270_FieldMark" },
  { 0x00fd03, "3270_Right2" },
  { 0x00fd04, "3270_Left2" },
  { 0x00fd05, "3270_BackTab" },
  { 0x00fd06, "3270_EraseEOF" },
  { 0x00fd07, "3270_EraseInput" },
  { 0x00fd08, "3270_Reset" },
  { 0x00fd09, "3270_Quit" },
  { 0x00fd0a, "3270_PA1" },
  { 0x00fd0b, "3270_PA2" },
  { 0x00fd0c, "3270_PA3" },
  { 0x00fd0d, "3270_Test" },
  { 0x00fd0e, "3270_Attn" },
  { 0x00fd0f, "3270_CursorBlink" },
  { 0x00fd10, "3270_AltCursor" },
  { 0x00fd11, "3270_KeyClick" },
  { 0x00fd12, "3270_Jump" },
  { 0x00fd13, "3270_Ident" },
  { 0x00fd14, "3270_Rule" },
  { 0x00fd15, "3270_Copy" },
  { 0x00fd16, "3270_Play" },
  { 0x00fd17, "3270_Setup" },
  { 0x00fd18, "3270_Record" },
  { 0x00fd19, "3270_ChangeScreen" },
  { 0x00fd1a, "3270_DeleteWord" },
  { 0x00fd1b, "3270_ExSelect" },
  { 0x00fd1c, "3270_CursorSelect" },
  { 0x00fd1d, "3270_PrintScreen" },
  { 0x00fd1e, "3270_Enter" },
  { 0x00fe01, "ISO_Lock" },
  { 0x00fe02, "ISO_Level2_Latch" },
  { 0x00fe03, "ISO_Level3_Shift" },
  { 0x00fe04, "ISO_Level3_Latch" },
  { 0x00fe05, "ISO_Level3_Lock" },
  { 0x00fe06, "ISO_Group_Latch" },
  { 0x00fe07, "ISO_Group_Lock" },
  { 0x00fe08, "ISO_Next_Group" },
  { 0x00fe09, "ISO_Next_Group_Lock" },
  { 0x00fe0a, "ISO_Prev_Group" },
  { 0x00fe0b, "ISO_Prev_Group_Lock" },
  { 0x00fe0c, "ISO_First_Group" },
  { 0x00fe0d, "ISO_First_Group_Lock" },
  { 0x00fe0e, "ISO_Last_Group" },
  { 0x00fe0f, "ISO_Last_Group_Lock" },
  { 0x00fe20, "ISO_Left_Tab" },
  { 0x00fe21, "ISO_Move_Line_Up" },
  { 0x00fe22, "ISO_Move_Line_Down" },
  { 0x00fe23, "ISO_Partial_Line_Up" },
  { 0x00fe24, "ISO_Partial_Line_Down" },
  { 0x00fe25, "ISO_Partial_Space_Left" },
  { 0x00fe26, "ISO_Partial_Space_Right" },
  { 0x00fe27, "ISO_Set_Margin_Left" },
  { 0x00fe28, "ISO_Set_Margin_Right" },
  { 0x00fe29, "ISO_Release_Margin_Left" },
  { 0x00fe2a, "ISO_Release_Margin_Right" },
  { 0x00fe2b, "ISO_Release_Both_Margins" },
  { 0x00fe2c, "ISO_Fast_Cursor_Left" },
  { 0x00fe2d, "ISO_Fast_Cursor_Right" },
  { 0x00fe2e, "ISO_Fast_Cursor_Up" },
  { 0x00fe2f, "ISO_Fast_Cursor_Down" },
  { 0x00fe30, "ISO_Continuous_Underline" },
  { 0x00fe31, "ISO_Discontinuous_Underline" },
  { 0x00fe32, "ISO_Emphasize" },
  { 0x00fe33, "ISO_Center_Object" },
  { 0x00fe34, "ISO_Enter" },
  { 0x00fe50, "dead_grave" },
  { 0x00fe51, "dead_acute" },
  { 0x00fe52, "dead_circumflex" },
  { 0x00fe53, "dead_tilde" },
  { 0x00fe54, "dead_macron" },
  { 0x00fe55, "dead_breve" },
  { 0x00fe56, "dead_abovedot" },
  { 0x00fe57, "dead_diaeresis" },
  { 0x00fe58, "dead_abovering" },
  { 0x00fe59, "dead_doubleacute" },
  { 0x00fe5a, "dead_caron" },
  { 0x00fe5b, "dead_cedilla" },
  { 0x00fe5c, "dead_ogonek" },
  { 0x00fe5d, "dead_iota" },
  { 0x00fe5e, "dead_voiced_sound" },
  { 0x00fe5f, "dead_semivoiced_sound" },
  { 0x00fe60, "dead_belowdot" },
  { 0x00fe61, "dead_hook" },
  { 0x00fe62, "dead_horn" },
  { 0x00fe70, "AccessX_Enable" },
  { 0x00fe71, "AccessX_Feedback_Enable" },
  { 0x00fe72, "RepeatKeys_Enable" },
  { 0x00fe73, "SlowKeys_Enable" },
  { 0x00fe74, "BounceKeys_Enable" },
  { 0x00fe75, "StickyKeys_Enable" },
  { 0x00fe76, "MouseKeys_Enable" },
  { 0x00fe77, "MouseKeys_Accel_Enable" },
  { 0x00fe78, "Overlay1_Enable" },
  { 0x00fe79, "Overlay2_Enable" },
  { 0x00fe7a, "AudibleBell_Enable" },
  { 0x00fed0, "First_Virtual_Screen" },
  { 0x00fed1, "Prev_Virtual_Screen" },
  { 0x00fed2, "Next_Virtual_Screen" },
  { 0x00fed4, "Last_Virtual_Screen" },
  { 0x00fed5, "Terminate_Server" },
  { 0x00fee0, "Pointer_Left" },
  { 0x00fee1, "Pointer_Right" },
  { 0x00fee2, "Pointer_Up" },
  { 0x00fee3, "Pointer_Down" },
  { 0x00fee4, "Pointer_UpLeft" },
  { 0x00fee5, "Pointer_UpRight" },
  { 0x00fee6, "Pointer_DownLeft" },
  { 0x00fee7, "Pointer_DownRight" },
  { 0x00fee8, "Pointer_Button_Dflt" },
  { 0x00fee9, "Pointer_Button1" },
  { 0x00feea, "Pointer_Button2" },
  { 0x00feeb, "Pointer_Button3" },
  { 0x00feec, "Pointer_Button4" },
  { 0x00feed, "Pointer_Button5" },
  { 0x00feee, "Pointer_DblClick_Dflt" },
  { 0x00feef, "Pointer_DblClick1" },
  { 0x00fef0, "Pointer_DblClick2" },
  { 0x00fef1, "Pointer_DblClick3" },
  { 0x00fef2, "Pointer_DblClick4" },
  { 0x00fef3, "Pointer_DblClick5" },
  { 0x00fef4, "Pointer_Drag_Dflt" },
  { 0x00fef5, "Pointer_Drag1" },
  { 0x00fef6, "Pointer_Drag2" },
  { 0x00fef7, "Pointer_Drag3" },
  { 0x00fef8, "Pointer_Drag4" },
  { 0x00fef9, "Pointer_EnableKeys" },
  { 0x00fefa, "Pointer_Accelerate" },
  { 0x00fefb, "Pointer_DfltBtnNext" },
  { 0x00fefc, "Pointer_DfltBtnPrev" },
  { 0x00fefd, "Pointer_Drag5" },
  { 0x00ff08, "BackSpace" },
  { 0x00ff09, "Tab" },
  { 0x00ff0a, "Linefeed" },
  { 0x00ff0b, "Clear" },
  { 0x00ff0d, "Return" },
  { 0x00ff13, "Pause" },
  { 0x00ff14, "Scroll_Lock" },
  { 0x00ff15, "Sys_Req" },
  { 0x00ff1b, "Escape" },
  { 0x00ff20, "Multi_key" },
  { 0x00ff21, "Kanji" },
  { 0x00ff22, "Muhenkan" },
  { 0x00ff23, "Henkan" },
  { 0x00ff23, "Henkan_Mode" },
  { 0x00ff24, "Romaji" },
  { 0x00ff25, "Hiragana" },
  { 0x00ff26, "Katakana" },
  { 0x00ff27, "Hiragana_Katakana" },
  { 0x00ff28, "Zenkaku" },
  { 0x00ff29, "Hankaku" },
  { 0x00ff2a, "Zenkaku_Hankaku" },
  { 0x00ff2b, "Touroku" },
  { 0x00ff2c, "Massyo" },
  { 0x00ff2d, "Kana_Lock" },
  { 0x00ff2e, "Kana_Shift" },
  { 0x00ff2f, "Eisu_Shift" },
  { 0x00ff30, "Eisu_toggle" },
  { 0x00ff31, "Hangul" },
  { 0x00ff32, "Hangul_Start" },
  { 0x00ff33, "Hangul_End" },
  { 0x00ff34, "Hangul_Hanja" },
  { 0x00ff35, "Hangul_Jamo" },
  { 0x00ff36, "Hangul_Romaja" },
  { 0x00ff37, "Codeinput" },
  { 0x00ff38, "Hangul_Jeonja" },
  { 0x00ff39, "Hangul_Banja" },
  { 0x00ff3a, "Hangul_PreHanja" },
  { 0x00ff3b, "Hangul_PostHanja" },
  { 0x00ff3c, "SingleCandidate" },
  { 0x00ff3d, "MultipleCandidate" },
  { 0x00ff3e, "PreviousCandidate" },
  { 0x00ff3f, "Hangul_Special" },
  { 0x00ff50, "Home" },
  { 0x00ff51, "Left" },
  { 0x00ff52, "Up" },
  { 0x00ff53, "Right" },
  { 0x00ff54, "Down" },
  { 0x00ff55, "Page_Up" },
  { 0x00ff55, "Prior" },
  { 0x00ff56, "Page_Down" },
  { 0x00ff56, "Next" },
  { 0x00ff57, "End" },
  { 0x00ff58, "Begin" },
  { 0x00ff60, "Select" },
  { 0x00ff61, "Print" },
  { 0x00ff62, "Execute" },
  { 0x00ff63, "Insert" },
  { 0x00ff65, "Undo" },
  { 0x00ff66, "Redo" },
  { 0x00ff67, "Menu" },
  { 0x00ff68, "Find" },
  { 0x00ff69, "Cancel" },
  { 0x00ff6a, "Help" },
  { 0x00ff6b, "Break" },
  { 0x00ff7e, "Arabic_switch" },
  { 0x00ff7e, "Greek_switch" },
  { 0x00ff7e, "Hangul_switch" },
  { 0x00ff7e, "Hebrew_switch" },
  { 0x00ff7e, "ISO_Group_Shift" },
  { 0x00ff7e, "Mode_switch" },
  { 0x00ff7e, "kana_switch" },
  { 0x00ff7e, "script_switch" },
  { 0x00ff7f, "Num_Lock" },
  { 0x00ff80, "KP_Space" },
  { 0x00ff89, "KP_Tab" },
  { 0x00ff8d, "KP_Enter" },
  { 0x00ff91, "KP_F1" },
  { 0x00ff92, "KP_F2" },
  { 0x00ff93, "KP_F3" },
  { 0x00ff94, "KP_F4" },
  { 0x00ff95, "KP_Home" },
  { 0x00ff96, "KP_Left" },
  { 0x00ff97, "KP_Up" },
  { 0x00ff98, "KP_Right" },
  { 0x00ff99, "KP_Down" },
  { 0x00ff9a, "KP_Page_Up" },
  { 0x00ff9a, "KP_Prior" },
  { 0x00ff9b, "KP_Page_Down" },
  { 0x00ff9b, "KP_Next" },
  { 0x00ff9c, "KP_End" },
  { 0x00ff9d, "KP_Begin" },
  { 0x00ff9e, "KP_Insert" },
  { 0x00ff9f, "KP_Delete" },
  { 0x00ffaa, "KP_Multiply" },
  { 0x00ffab, "KP_Add" },
  { 0x00ffac, "KP_Separator" },
  { 0x00ffad, "KP_Subtract" },
  { 0x00ffae, "KP_Decimal" },
  { 0x00ffaf, "KP_Divide" },
  { 0x00ffb0, "KP_0" },
  { 0x00ffb1, "KP_1" },
  { 0x00ffb2, "KP_2" },
  { 0x00ffb3, "KP_3" },
  { 0x00ffb4, "KP_4" },
  { 0x00ffb5, "KP_5" },
  { 0x00ffb6, "KP_6" },
  { 0x00ffb7, "KP_7" },
  { 0x00ffb8, "KP_8" },
  { 0x00ffb9, "KP_9" },
  { 0x00ffbd, "KP_Equal" },
  { 0x00ffbe, "F1" },
  { 0x00ffbf, "F2" },
  { 0x00ffc0, "F3" },
  { 0x00ffc1, "F4" },
  { 0x00ffc2, "F5" },
  { 0x00ffc3, "F6" },
  { 0x00ffc4, "F7" },
  { 0x00ffc5, "F8" },
  { 0x00ffc6, "F9" },
  { 0x00ffc7, "F10" },
  { 0x00ffc8, "F11" },
  { 0x00ffc9, "F12" },
  { 0x00ffca, "F13" },
  { 0x00ffcb, "F14" },
  { 0x00ffcc, "F15" },
  { 0x00ffcd, "F16" },
  { 0x00ffce, "F17" },
  { 0x00ffcf, "F18" },
  { 0x00ffd0, "F19" },
  { 0x00ffd1, "F20" },
  { 0x00ffd2, "F21" },
  { 0x00ffd3, "F22" },
  { 0x00ffd4, "F23" },
  { 0x00ffd5, "F24" },
  { 0x00ffd6, "F25" },
  { 0x00ffd7, "F26" },
  { 0x00ffd8, "F27" },
  { 0x00ffd9, "F28" },
  { 0x00ffda, "F29" },
  { 0x00ffdb, "F30" },
  { 0x00ffdc, "F31" },
  { 0x00ffdd, "F32" },
  { 0x00ffde, "F33" },
  { 0x00ffdf, "F34" },
  { 0x00ffe0, "F35" },
  { 0x00ffe1, "Shift_L" },
  { 0x00ffe2, "Shift_R" },
  { 0x00ffe3, "Control_L" },
  { 0x00ffe4, "Control_R" },
  { 0x00ffe5, "Caps_Lock" },
  { 0x00ffe6, "Shift_Lock" },
  { 0x00ffe7, "Meta_L" },
  { 0x00ffe8, "Meta_R" },
  { 0x00ffe9, "Alt_L" },
  { 0x00ffea, "Alt_R" },
  { 0x00ffeb, "Super_L" },
  { 0x00ffec, "Super_R" },
  { 0x00ffed, "Hyper_L" },
  { 0x00ffee, "Hyper_R" },
  { 0x00ffff, "Delete" },
  { 0xffffff, "VoidSymbol" },
};

#define SCIM_NUM_KEYS (sizeof (__scim_keys_by_code) / sizeof (__scim_keys_by_code [0]))

static __KeyName *__scim_keys_by_name = NULL;

static __KeyName __scim_key_mask_names [] =
{
    {SCIM_KEY_ShiftMask,        "Shift"},
    {SCIM_KEY_CapsLockMask,     "CapsLock"},
    {SCIM_KEY_ControlMask,      "Control"},
    {SCIM_KEY_Mod1Mask,         "Alt"},
    {SCIM_KEY_NumLockMask,      "NumLock"},
    {SCIM_KEY_ScrollLockMask,   "ScrollLock"},
    {SCIM_KEY_Mod2Mask,         "Mod1"},
    {SCIM_KEY_Mod2Mask,         "Mod2"},
    {SCIM_KEY_Mod3Mask,         "Mod3"},
    {SCIM_KEY_Mod4Mask,         "Mod4"},
    {SCIM_KEY_Mod5Mask,         "Mod5"},
    {SCIM_KEY_LockMask,         "Lock"},
    {SCIM_KEY_ReleaseMask,      "KeyRelease"}
};

#define SCIM_NUM_KEY_MASKS (sizeof (__scim_key_mask_names) / sizeof (__scim_key_mask_names[0]))

char
KeyEvent::get_ascii_code () const
{
    if (code >= SCIM_KEY_space && code <= SCIM_KEY_asciitilde)
        return (char) code;

    if (code >= SCIM_KEY_KP_0 && code <= SCIM_KEY_KP_9)
        return (char) (code - SCIM_KEY_KP_0 + SCIM_KEY_0);

    if (code == SCIM_KEY_Return)
        return 0x0d;
    if (code == SCIM_KEY_Linefeed)
        return 0x0a;
    if (code == SCIM_KEY_Tab)
        return 0x09;

    return 0;
}

ucs4_t
KeyEvent::get_unicode_code () const
{
    /* First check for Latin-1 characters (1:1 mapping) */
    if ((code >= 0x0020 && code <= 0x007e) ||
        (code >= 0x00a0 && code <= 0x00ff))
        return code;

    /* Also check for directly encoded 24-bit UCS characters:
     */
    if ((code & 0xff000000) == 0x01000000)
        return code & 0x00ffffff;

    /* Translation of KP_Decimal depends on locale.
    if (code == SCIM_KP_Decimal)
        return get_decimal_char ();
    */

    /* binary search in table */
    __KeyUnicode * it = std::lower_bound (__scim_key_to_unicode_tab,
                                          __scim_key_to_unicode_tab + SCIM_NUM_KEY_UNICODES,
                                          code,
                                          __KeyUnicodeLessByCode ());

    if (it != __scim_key_to_unicode_tab + SCIM_NUM_KEY_UNICODES && ((int) it->code) == code)
        return it->ucs;

    /* No matching Unicode value found */
    return 0;
}

String
KeyEvent::get_key_string () const
{
    size_t i;
    String str;
    int mask_skip = 0;

    for (i=0; i < SCIM_NUM_KEY_MASKS; ++i) {
        if ((__scim_key_mask_names [i].value & mask) && !(__scim_key_mask_names [i].value & mask_skip)) {
            if (str.length ())
                str += (String ("+") + String (__scim_key_mask_names [i].name));
            else
                str += String (__scim_key_mask_names [i].name);
        }
        mask_skip |= __scim_key_mask_names [i].value;
    }

    __KeyName *it = std::lower_bound (__scim_keys_by_code,
                                      __scim_keys_by_code + SCIM_NUM_KEYS,
                                      code,
                                      __KeyNameLessByCode ());

    if (it != __scim_keys_by_code + SCIM_NUM_KEYS && it->value == code) {
        if (str.length ())
            str += (String ("+") + String (it->name));
        else
            str += String (it->name);
    } else if (code) {
        char buf [20];
        snprintf (buf, 20, "0x%04x", code);
        if (str.length ())
            str += (String ("+") + String (buf));
        else
            str += String (buf);
    }

    return str;
}

bool
scim_key_to_string (String &str, const KeyEvent & key)
{
    str = key.get_key_string ();
    return str.length () != 0;
}

bool
scim_string_to_key (KeyEvent &key, const String & str)
{
    std::vector <String> list;
    bool skip;
    size_t i;

    if (!__scim_keys_by_name) {
        __scim_keys_by_name = new __KeyName [SCIM_NUM_KEYS];
        memcpy (__scim_keys_by_name, __scim_keys_by_code, SCIM_NUM_KEYS * sizeof (__KeyName));
        std::sort (__scim_keys_by_name, __scim_keys_by_name + SCIM_NUM_KEYS, __KeyNameLessByName ());
    }

    scim_split_string_list (list, str, '+');

    for (std::vector <String>::iterator it=list.begin (); it!=list.end (); ++it) {
        skip = false;
        for (i = 0; i < SCIM_NUM_KEY_MASKS; ++i) {
            if (*it == String (__scim_key_mask_names [i].name)) {
                key.mask |= __scim_key_mask_names [i].value;
                skip = true;
                break;
            }
        }

        if (skip) continue;

        __KeyName *p = std::lower_bound (__scim_keys_by_name,
                                         __scim_keys_by_name + SCIM_NUM_KEYS,
                                         it->c_str (),
                                         __KeyNameLessByName ());

        if (p != __scim_keys_by_name + SCIM_NUM_KEYS && strcmp (p->name, it->c_str ()) == 0) {
            key.code = p->value;
        } else if (it->length () > 2 && (*it)[0] == '0' && ((*it)[1] == 'x' || (*it)[1] == 'X')){
            key.code = strtol (it->c_str () + 2, NULL, 16);
        } else if (it->length () > 4 && ((*it)[0] == 'U' || (*it)[0] == 'u')) {
            key.code = strtol (it->c_str () + 1, NULL, 16);
            key.code |= 0x01000000;
        }
    }

    return key.code != 0;
}

bool
scim_key_list_to_string (String &str, const std::vector<KeyEvent> & keylist)
{
    std::vector<String> strlist;

    for (std::vector<KeyEvent>::const_iterator it = keylist.begin (); it != keylist.end (); ++it) {
        if (scim_key_to_string (str, *it))
            strlist.push_back (str);
    }

    str = scim_combine_string_list (strlist, ',');

    return str.length () != 0;
}

bool
scim_string_to_key_list (std::vector<KeyEvent> &keylist, const String &str)
{
    std::vector <String> strlist;
    scim_split_string_list (strlist, str, ',');

    keylist.clear ();

    for (std::vector <String>::iterator it = strlist.begin (); it != strlist.end (); ++it) {
        KeyEvent key;
        if (scim_string_to_key (key, *it))
            keylist.push_back (key);
    }
    return keylist.size () > 0;
}

} // namespace scim

/*
vi:ts=4:nowrap:ai:expandtab
*/
