#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sg_include.h"
#include "sg_err.h"

/* This code is an example of runtime selection to obtain the best
   features available from the Linux SCSI generic (sg) driver

*  Copyright (C) 1999 D. Gilbert
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   This program is meant as an example to application writers who wish
   to use the Linux sg driver. Linux changed its sg driver in kernel
   2.2.6 . The new version has extra features that application writers
   may wish to use but not at the expense of backward compatibility with
   the original driver. Also if an application is distributed in binary
   code, runtime selection is needed in the application code interfacing
   to sg in order to cope with the various combinations:

   App compiled with    App binary run on     Notes
   ----------------------------------------------------------------------
   new sg.h             new driver
   new sg.h             original driver       backward compatibility mode
   original sg.h        new driver            "forward" compatibility mode
   original sg.h        original driver


   Noteworthy features:
        - forward + backward compatible from 2.0 to 2.3 series of
          kernels (tested on: 2.0.37, 2.2.10, 2.3.8). Extra features
          are used when available. This is done via run time selection.
        - handles /usr/include/scsi bug in Redhat 6.0 + other distros
        - allows device argument to be a non-sg SCSI device (eg /dev/sda)
          and shows mapping to equivalent sg device
        - internal scan (used for previous item when non-sg device) does
          not hang on O_EXCL flag on device but steps over it.
        - allows app to request reserved buffer size and shows the amount
          actually reserved **
        - outputs as much error information as is available
        - uses categorization and sense buffer decode routines in sg_err.c
        - sets SCSI command length explicitly (when available)

   ** The original reserved buffer (ie SG_BIG_BUFF) is allocated one for
      all users of the driver. From 2.2.6 onwards, the reserved buffer is
      per file descriptor.

   To keep it simple this example program does a SCSI INQUIRY command
   which is mandatory for all device types to support. An option is
   provided ("-e") to force an erroneous opcode in the SCSI command.
   Another option is provided that requests a new reserved buffer size
   ("-b=<req_res_size_KB>").
   
   One assumption made here is that ioctl command numbers do not change.

   Version 0.57 (20010115)

6 byte INQUIRY command:
[0x12][   |lu][pg cde][res   ][al len][cntrl ]

*/

#ifndef SG_GET_RESERVED_SIZE
#define SG_GET_RESERVED_SIZE 0x2272
#endif

#ifndef SG_SET_RESERVED_SIZE
#define SG_SET_RESERVED_SIZE 0x2275
#endif

#ifndef SG_GET_VERSION_NUM
#define SG_GET_VERSION_NUM 0x2282
#endif

#ifndef SG_NEXT_CMD_LEN
#define SG_NEXT_CMD_LEN 0x2283
#endif

#ifndef SG_MAX_SENSE
#define SG_MAX_SENSE 16
#endif


#define SG_RT_UNKN (-1)
#define SG_RT_ORIG 0
#define SG_RT_NEW32 1  /* new driver version 2.1.31 + 2.1.32 */
#define SG_RT_NEW34 2  /* new driver version 2.1.34 and after */
        
typedef struct sg_h_n  /* for "forward" compatibility case */
{
    int pack_len;    /* [o] reply_len (ie useless), ignored as input */
    int reply_len;   /* [i] max length of expected reply (inc. sg_header) */
    int pack_id;     /* [io] id number of packet (use ints >= 0) */
    int result;      /* [o] 0==ok, else (+ve) Unix errno (best ignored) */
    unsigned int twelve_byte:1; 
        /* [i] Force 12 byte command length for group 6 & 7 commands  */
    unsigned int target_status:5;   /* [o] scsi status from target */
    unsigned int host_status:8;     /* [o] host status (see "DID" codes) */
    unsigned int driver_status:8;   /* [o] driver status+suggestion */
    unsigned int other_flags:10;    /* unused */
    unsigned char sense_buffer[SG_MAX_SENSE]; /* [o] Output in 3 cases:
           when target_status is CHECK_CONDITION or 
           when target_status is COMMAND_TERMINATED or
           when (driver_status & DRIVER_SENSE) is true. */
} Sg_h_n;      /* This structure is 36 bytes long on i386 */


static int open_scsi_dev_as_sg(char * devname);
static void adjust_reserved_buffer(int sg_fd, int sg_which, int req_res_size,
                                   int * reserved_sizep);

#define OFF sizeof(struct sg_header)
#define INQ_REPLY_LEN 96        /* logic assumes >= sizeof(inqCmdBlk) */
#define INQ_CMD_LEN 6
#define MY_PACK_ID 1234
#define BAD_OPCODE 0x1f


static void usage()
{
    printf("Usage: 'sg_runt_ex [-b=<reserved_KB>] [-e] <scsi_device>'\n");
    printf("    where: -b=<req_res_size_KB>  requested reserved buffer size\n");
    printf("                                 in Kilobytes\n");
    printf("           -e     use bad opcode on 'INQUIRY' to test errors\n");
}


int main(int argc, char * argv[])
{
    int sg_fd, k, cmd_len, num, ok;
    unsigned char inqCmdBlk [INQ_CMD_LEN] =
                                {0x12, 0, 0, 0, INQ_REPLY_LEN, 0};
    unsigned char inqBuff[OFF + INQ_REPLY_LEN];
    int inqInLen = OFF + sizeof(inqCmdBlk);
    int inqOutLen = OFF + INQ_REPLY_LEN;
    unsigned char * buffp = inqBuff + OFF;
    struct sg_header * sghp = (struct sg_header *)inqBuff;
    Sg_h_n * n_sghp = (Sg_h_n *)inqBuff;
    char * file_name = 0;
    int reserved_size;
    int req_res_size = 0;
    int do_error = 0;
    int sg_which = SG_RT_UNKN;
    int sg_version = 0;

    for (k = 1; k < argc; ++k) {
        if (0 == strncmp("-b=", argv[k], 3)) {
            num = sscanf(argv[k] + 3, "%d", &req_res_size);
            if ((1 != num) || (req_res_size < 0)) {
                printf("Couldn't decode number after '-b' switch\n");
                file_name = 0;
                break;
            }
            req_res_size *= 1024;
        }
        else if (0 == strcmp("-e", argv[k]))
            do_error = 1;
        else if (*argv[k] == '-') {
            printf("Unrecognized switch: %s\n", argv[k]);
            file_name = 0;
            break;
        }
        else
            file_name = argv[k];
    }
    if (0 == file_name) {
        usage();
        return 1;
    }
    printf("Linux SG device driver run time selection example program.\n");
    
    sg_fd = open_scsi_dev_as_sg(file_name);
    if (sg_fd < 0) {
        if (-9999 == sg_fd)
            printf("Failed trying to open SCSI device as an sg device\n");
        else
            perror("sg_runt_ex: open error");
        return 1;
    }

    /* Run time selection code follows */
    if (ioctl(sg_fd, SG_GET_RESERVED_SIZE, &reserved_size) < 0) {
        reserved_size = SG_BIG_BUFF;
        sg_which = SG_RT_ORIG;
    }
    else if (ioctl(sg_fd, SG_GET_VERSION_NUM, &sg_version) < 0)
        sg_which = SG_RT_NEW32;
    else {
        sg_which = SG_RT_NEW34;
        printf("  SG driver version: %d.%d.%d\n", sg_version / 10000,
               (sg_version / 100) % 100, sg_version % 100);
    }

    if (req_res_size > 0) /* this is case when "-b" swaitch given */
        adjust_reserved_buffer(sg_fd, sg_which, req_res_size, &reserved_size);
    
    sghp->reply_len = inqOutLen;
    sghp->pack_id = MY_PACK_ID;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;     /* some apps assume this is done ?!? */

    switch (sg_which) {
    case SG_RT_ORIG:
        sghp->sense_buffer[0] = 0;
        break;
    case SG_RT_NEW32:
        break;
    case SG_RT_NEW34: /* this is optional, explicitly setting cmd length */
        cmd_len = INQ_CMD_LEN; 
        if (ioctl(sg_fd, SG_NEXT_CMD_LEN, &cmd_len) < 0) {
            perror("sg_runt_ex: SG_NEXT_CMD_LEN error");
            close(sg_fd);
            return 1;
        }
        break;
    default:
        printf("Illegal state for sg_which=%d\n", sg_which);
        return 1;
    }
    memcpy(inqBuff + OFF, inqCmdBlk, INQ_CMD_LEN);

    if (do_error) /* force bad cmd opcode when "-e" given */
        *(inqBuff + OFF) = (unsigned char)BAD_OPCODE;

    if (write(sg_fd, inqBuff, inqInLen) < 0) {
        perror("sg_runt_ex: write error");
        close(sg_fd);
        return 1;
    }
    
    if (read(sg_fd, inqBuff, inqOutLen) < 0) {
        perror("sg_runt_ex: read error");
        close(sg_fd);
        return 1;
    }

    /* now for the error processing */
    ok = 0;
    switch (sg_which) {
    case SG_RT_ORIG:
        if ((0 == sghp->result) && (0 == sghp->sense_buffer[0]))
            ok = 1;
        else if (sghp->sense_buffer[0])
            sg_print_sense("'INQUIRY' command error", sghp->sense_buffer, 
                           SG_MAX_SENSE);
        else /* sghp->result is != 0 */
            printf("INQUIRY failed, sghp->result=%d\n", sghp->result);
        break;
    case SG_RT_NEW32:
    case SG_RT_NEW34: 
        switch (sg_err_category(n_sghp->target_status, n_sghp->host_status,
                n_sghp->driver_status, n_sghp->sense_buffer, SG_MAX_SENSE)) {
        case SG_ERR_CAT_CLEAN:
            ok = 1;
            break;
        case SG_ERR_CAT_RECOVERED:
            printf("Recovered error on INQUIRY, continuing\n");
            ok = 1;
            break;
        default: /* won't bother decoding other categories */
            sg_chk_n_print("'INQUIRY' command error", n_sghp->target_status, 
                           n_sghp->host_status, n_sghp->driver_status, 
                           n_sghp->sense_buffer, SG_MAX_SENSE);
            break;
        }
        break;
    default:
        break;
    }

    if (ok) { /* output result if it is available */
        int f = (int)*(buffp + 7);
        printf("Some of the INQUIRY command's results:\n");
        printf("    %.8s  %.16s  %.4s  ", buffp + 8, buffp + 16, buffp + 32);
        printf("[wide=%d sync=%d cmdque=%d sftre=%d]\n",
               !!(f & 0x20), !!(f & 0x10), !!(f & 2), !!(f & 1));
    }

    close(sg_fd);
    return 0;
}
    

#define MAX_SG_DEVS 26
#define MAX_FILENAME_LEN 128

#define SCAN_ALPHA 0
#define SCAN_NUMERIC 1
#define DEF_SCAN SCAN_ALPHA

static void make_dev_name(char * fname, int k, int do_numeric)
{
    char buff[MAX_FILENAME_LEN];

    strcpy(fname, "/dev/sg");
    if (do_numeric) {
        sprintf(buff, "%d", k);
        strcat(fname, buff);
    }
    else {
        if (k <= 26) {
            buff[0] = 'a' + (char)k;
            buff[1] = '\0';
            strcat(fname, buff);
        }
        else
            strcat(fname, "xxxx");
    }
}

typedef struct my_scsi_idlun
{
    int mux4;
    int host_unique_id;

} My_scsi_idlun;

static int open_scsi_dev_as_sg(char * devname)
{
    int fd, bus, bbus, k;
    My_scsi_idlun m_idlun, mm_idlun;
    int do_numeric = DEF_SCAN;
    char name[MAX_FILENAME_LEN];

    strcpy(name, devname);
    if ((fd = open(name, O_RDONLY | O_NONBLOCK)) < 0) {
        if (EACCES == errno) {
            if ((fd = open(name, O_RDWR | O_NONBLOCK)) < 0)
                return fd;
        }
    }
    if (ioctl(fd, SG_GET_TIMEOUT, 0) < 0) { /* not sg device ? */
        if (ioctl(fd, SCSI_IOCTL_GET_BUS_NUMBER, &bus) < 0) {
            printf("Need a filename that resolves to a SCSI device\n");
            close(fd);
            return -9999;
        }
        if (ioctl(fd, SCSI_IOCTL_GET_IDLUN, &m_idlun) < 0) {
            printf("Need a filename that resolves to a SCSI device (2)\n");
            close(fd);
            return -9999;
        }
        close(fd);
    
        for (k = 0; k < MAX_SG_DEVS; k++) {
            make_dev_name(name, k, do_numeric);
            if ((fd = open(name, O_RDONLY | O_NONBLOCK)) < 0) {
                if (EACCES == errno) 
                    fd = open(name, O_RDWR | O_NONBLOCK);
                if (fd < 0) {
                    if ((ENOENT == errno) && (0 == k) && 
                        (do_numeric == DEF_SCAN)) {
                        do_numeric = ! DEF_SCAN;
                        make_dev_name(name, k, do_numeric);
                        if ((fd = open(name, O_RDONLY | O_NONBLOCK)) < 0) {
                            if (EACCES == errno) 
                                fd = open(name, O_RDWR | O_NONBLOCK);
                        }
                    }
                    if (fd < 0) {
                        if (EBUSY == errno)
                            continue;  /* step over if O_EXCL already on it */
                        else
                            break;
                    }
                }
            }
            if (ioctl(fd, SCSI_IOCTL_GET_BUS_NUMBER, &bbus) < 0) {
                perror("sg ioctl failed");
                close(fd);
                fd = -9999;
            }
            if (ioctl(fd, SCSI_IOCTL_GET_IDLUN, &mm_idlun) < 0) {
                perror("sg ioctl failed (2)");
                close(fd);
                fd = -9999;
            }
            if ((bus == bbus) && 
                ((m_idlun.mux4 & 0xff) == (mm_idlun.mux4 & 0xff)) &&
                (((m_idlun.mux4 >> 8) & 0xff) == 
                                        ((mm_idlun.mux4 >> 8) & 0xff)) &&
                (((m_idlun.mux4 >> 16) & 0xff) == 
                                        ((mm_idlun.mux4 >> 16) & 0xff))) {
                printf("  >>> Mapping %s to sg device: %s\n", devname, name);
                break;
            }
            else {
                close(fd);
                fd = -9999;
            }
        }
    }
    if (fd >= 0) { /* everything ok, close and re-open read-write */
        close(fd);
        return open(name, O_RDWR);
    }
    else
        return fd;
}

static void adjust_reserved_buffer(int sg_fd, int sg_which, int req_res_size,
                                   int * reserved_sizep)
{
    switch (sg_which) {
    case SG_RT_ORIG:
        break; /* reserved_size has already been set to a constant */
    case SG_RT_NEW32: /* SG_SET_RESERVED_SIZE exists but does nothing in */
                      /* version 2.1.32 and 2.1.33, so harmless to try */
    case SG_RT_NEW34:
        if (ioctl(sg_fd, SG_SET_RESERVED_SIZE, &req_res_size) < 0) {
            perror("SG_SET_RESERVED_SIZE ioctl failed");
            return;
        }
        if (ioctl(sg_fd, SG_GET_RESERVED_SIZE, reserved_sizep) < 0) {
            perror("SG_GET_RESERVED_SIZE ioctl failed");
            return;
        }
        break;
    default:
        printf("adjust_reserved_buffer: bad sg_which=%d value\n", sg_which);
        return;
    }
    printf("  >>> Requested reserve buffer size=%dk, granted size=%dk\n",
           req_res_size / 1024, (*reserved_sizep) / 1024);
}
