/* -*- c++ -*-
 * __BEGIN_COPYRIGHT
 * SimpleDB API
 * 
 * Copyright (C) 2005 Eminence Technology Pty Ltd
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You can view the GNU Lesser General Public Licence at
 * http://www.gnu.org/licenses/lgpl.html or you can write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Eminence Technology Pty Ltd can be contacted by writing to
 * Eminence Technology, PO Box 118, Moorooka QLD 4105, Australia.
 * Alternatively, you may email opensource [at] eminence [dot] com [dot] au
 * __END_COPYRIGHT
 */

#ifndef __SIMPLEDB_COLUMN_H_
#define __SIMPLEDB_COLUMN_H_ 1

#include <sql.h>
#include <sqlext.h>
#include <iostream>
#include <string>
#include "Exception.h" // for Database::Exception

namespace SimpleDB {

  /** The abstract base class for the SimpleDB types.
   * This type contains the actual data as well as meta-data.
   */
  class Column {
    
  public:

    /** Exception class.
     * Thrown when a column's value is requested but the column hasn't been
     * bound
     */
    class UnboundException : public Exception {
    public:

      /** Constructor used when creating a new UnboundException object.
       */
      UnboundException() : Exception("Column is not bound.") {};
    };

    
    /** Tests if the current value is NULL
     * 
     * @return True if the value is NULL.
     */
    virtual bool isNull();
  
    /** This abstract function is called by Query when the Query.bind function
     * is invoked.
     * This function has an implementation that handles making sure columns are
     * bound. This function should be called by any derived objects.
     *
     * @param statementHandle The statementHandle columns are being bound to 
     * @param columnNumber The column number to bind this column to..
     */
    virtual void bind(SQLHSTMT statementHandle, int columnNumber) = 0;

    /** Function called to write a column value to the output stream.
     * This function is implemented by each column type to write the value
     * in a suitable form to the output stream.
     * 
     * @param oStream Output stream object.
     * @return Output stream object to which data was written.
     */
    virtual std::ostream& oStream(std::ostream& oStream) = 0;
  
    /** A virtual destructor that can be overridden if required.
     * This can be used, for example, to deallocate memory.
     */
    virtual ~Column() {};

    /** Constructor used to set initial value of columnBound variable.
     */
    Column();


    /** This method is to be called by all derived classes to make sure
     * the column was bound. It will cause an exception to be thrown if the
     * column wasn't bound.
     */
    void value();

  protected:
  
    /** The lengthOrIndex variable to be sent to the SQLBindCol function.
     */
    SQLINTEGER lengthOrIndex;

    /** Indicates weather this column has been bound to a query.
     */
    bool bound;

    /** Name that can be used 
     *
     */
    std::string name;
  };

  /** Class representing string values.
   */
  class StringColumn : public Column {
  
  public:
  
    /** Constructor to allocate memory for this column.
     * This function will throw an Exception if the memory cannot be allocated.
     *
     * @param bufferSize The maximum number of bytes the value will hold.
     */
    StringColumn(const int bufferSize);
  
    /** Destructor frees allocated memory.
     */
    ~StringColumn();

    std::ostream& oStream(std::ostream& oStream);

    void bind(SQLHSTMT statementHandle, int columnNumber);

    /** Returns the string value of this object.
     *
     * @return The string value.
     */
    std::string value();
    
  private:
    long bufferSize;
    char * buffer;
  };

  /** Class representing long values.
   */
  class LongColumn : public Column {
  
  public:
    std::ostream& oStream(std::ostream& oStream);

    void bind(SQLHSTMT statementHandle, int columnNumber);

    /** Used to retrieve the value of this column.
     *
     * @return The long value of this column.
     */
    long value() {Column::value(); if(isNull()) return 0; else return colValue;};

  private:
    /** The long column value
     */
    long colValue;
  
  };
  
  /** Class representing bool values.
   */
  class BoolColumn : public Column {
  
  public:
    std::ostream& oStream(std::ostream& oStream);

    void bind(SQLHSTMT statementHandle, int columnNumber);

    /** Used to retrieve the value of this column.
     *
     * @return The long value of this column.
     */
    bool value() ;

  private:
    /** The long column value
     */
    unsigned char colValue;
  
  };

}

inline
std::ostream& operator<<(std::ostream& strm, SimpleDB::Column& col) {
  return col.oStream(strm);
}

#endif
