-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
function NextSymbol (Previous : Iterator) return Iterator is
   Next : Iterator;

   --------------------------------------------------------------------------------

   function NextDeclarativeItem (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Found       : Boolean;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         exit when Declaration = NullSymbol;
         Item := RawDict.GetDeclarationItem (Declaration);
         case RawDict.GetSymbolDiscriminant (Item) is
            when PackageSymbol =>
               Found := Declaration /= RawDict.GetPackageBody (Item);
            when SubprogramSymbol =>
               Found := Declaration /= RawDict.GetSubprogramBody (Item);
            when others =>
               Found := True;
         end case;
         exit when Found;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      if Declaration = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(DeclarativeItemIterator, IsAbstract, RawDict.GetDeclarationItem (Declaration), Declaration);
      end if;

      return Next;

   end NextDeclarativeItem;

   --------------------------------------------------------------------------------

   function NextDeferredConstant (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop

         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;

         Item := RawDict.GetDeclarationItem (Declaration);

         if IsDeferredConstant (Item) then
            Next := Iterator'(DeferredConstantIterator, IsAbstract, RawDict.GetDeclarationItem (Declaration), Declaration);
            exit;
         end if;

         Declaration := RawDict.GetNextDeclaration (Declaration);

      end loop;

      return Next;

   end NextDeferredConstant;

   --------------------------------------------------------------------------------

   function NextArrayIndex (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Next   : Symbol;
      Result : Iterator;
   begin

      Next := RawDict.GetNextArrayIndex (Previous.Context);

      if Next = NullSymbol then
         Result := NullIterator;
      else
         Result := Iterator'(ArrayIndexIterator, IsAbstract, RawDict.GetArrayIndexType (Next), Next);
      end if;

      return Result;

   end NextArrayIndex;

   --------------------------------------------------------------------------------

   function NextLoop (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      TheLoop : Symbol;
      Next    : Iterator;
   begin

      TheLoop := RawDict.GetNextLoop (CurrentSymbol (Previous));

      if TheLoop = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(LoopIterator, IsAbstract, TheLoop, NullSymbol);
      end if;

      return Next;

   end NextLoop;

   --------------------------------------------------------------------------------

   function NextUndeclaredType (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop

         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;

         Item := RawDict.GetDeclarationItem (Declaration);

         if IsType (Item) and then RawDict.GetTypeDiscriminant (Item) = UnknownType then
            Next := Iterator'(UndeclaredTypeIterator, IsAbstract, RawDict.GetDeclarationItem (Declaration), Declaration);
            exit;
         end if;

         Declaration := RawDict.GetNextDeclaration (Declaration);

      end loop;

      return Next;

   end NextUndeclaredType;

   --------------------------------------------------------------------------------

   function NextPrivateType (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop

         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;

         Item := RawDict.GetDeclarationItem (Declaration);

         if IsType (Item)
           and then TypeIsPrivate (Item)
           and then RawDict.GetDeclarationContext (Declaration) = ProgramContext then
            Next := Iterator'(PrivateTypeIterator, IsAbstract, RawDict.GetDeclarationItem (Declaration), Declaration);
            exit;
         end if;

         Declaration := RawDict.GetNextDeclaration (Declaration);

      end loop;

      return Next;

   end NextPrivateType;

   --------------------------------------------------------------------------------

   function NextEnumerationLiteral (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Literal : Symbol;
      Next    : Iterator;
   begin

      Literal := RawDict.GetNextEnumerationLiteral (CurrentSymbol (Previous));

      if Literal = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(EnumerationLiteralIterator, IsAbstract, Literal, NullSymbol);
      end if;

      return Next;

   end NextEnumerationLiteral;

   --------------------------------------------------------------------------------

   function NextRecordComponent (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Component : Symbol;
      Next      : Iterator;
   begin

      Component := RawDict.GetNextRecordComponent (CurrentSymbol (Previous));

      if Component = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(RecordComponentIterator, IsAbstract, Component, NullSymbol);
      end if;

      return Next;

   end NextRecordComponent;

   --------------------------------------------------------------------------------

   function NextExtendedRecordComponent (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Component     : Symbol;
      Next          : Iterator;
      CurrentRecord : Symbol;
   begin
      Component := RawDict.GetNextRecordComponent (CurrentSymbol (Previous));
      if Component = NullSymbol then
         -- current record has no more fields so see if earlier extension has
         CurrentRecord :=
           BackTrackUpInheritChain (Previous.Context, RawDict.GetRecordComponentRecordType (CurrentSymbol (Previous)));
         loop
            if CurrentRecord = NullSymbol then
               -- searched entire extended record structure
               Next := NullIterator;
               exit;
            end if;

            -- We have a new extended record to search, we need the _second_ field (to skip Inherit)
            Component := RawDict.GetNextRecordComponent (RawDict.GetTypeFirstRecordComponent (CurrentRecord));
            if Component /= NullSymbol then
               -- found a field
               Next := Iterator'(ExtendedRecordComponentIterator, IsAbstract, Component, Previous.Context);
               exit;
            end if;

            -- if get here then we must have encountered a null extension - we must backtrack again
            CurrentRecord := BackTrackUpInheritChain (Previous.Context, CurrentRecord);
         end loop;

      else -- record has further fields
         Next := Iterator'(ExtendedRecordComponentIterator, IsAbstract, Component, Previous.Context);
      end if;

      return Next;

   end NextExtendedRecordComponent;

   --------------------------------------------------------------------------------

   function NextLibraryUnit (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Found       : Boolean;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         case RawDict.GetSymbolDiscriminant (Item) is
            when PackageSymbol =>
               Found := Declaration /= RawDict.GetPackageBody (Item);
            when others =>
               Found := Declaration /= RawDict.GetSubprogramBody (Item);
         end case;
         if Found then
            Next := Iterator'(LibraryUnitIterator, IsAbstract, Item, Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return Next;

   end NextLibraryUnit;

   --------------------------------------------------------------------------------

   function NextWithedPackage (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      ContextClause : Symbol;
      Next          : Iterator;
   begin

      ContextClause := RawDict.GetNextContextClause (Previous.Context);

      if ContextClause = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(WithedPackageIterator, IsAbstract, RawDict.GetContextClausePackage (ContextClause), ContextClause);
      end if;

      return Next;

   end NextWithedPackage;

   --------------------------------------------------------------------------------

   function NextInheritedPackage (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      ContextClause : Symbol;
      Next          : Iterator;
   begin

      ContextClause := RawDict.GetNextContextClause (Previous.Context);

      if ContextClause = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(InheritedPackageIterator, IsAbstract, RawDict.GetContextClausePackage (ContextClause), ContextClause);
      end if;

      return Next;

   end NextInheritedPackage;

   --------------------------------------------------------------------------------

   function NextVisibleOrPrivateSubprogram (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if RawDict.GetSymbolDiscriminant (Item) = SubprogramSymbol
           and then Declaration /= RawDict.GetSubprogramBody (Item) then
            Next := Iterator'(Previous.Discriminant, IsAbstract, Item, Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return Next;

   end NextVisibleOrPrivateSubprogram;

   --------------------------------------------------------------------------------

   function NextTaskType (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsTaskType (Item) then
            Next := Iterator'(TaskTypeIterator, IsAbstract, Item, Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;
      return Next;

   end NextTaskType;

   --------------------------------------------------------------------------------

   function NextOwnTask (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      OwnTask : Symbol;
      Next    : Iterator;
   begin

      OwnTask := RawDict.GetNextOwnTask (Previous.Context);

      if OwnTask = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(OwnTaskIterator, IsAbstract, RawDict.GetOwnTaskVariable (OwnTask), OwnTask);
      end if;
      return Next;
   end NextOwnTask;

   --------------------------------------------------------------------------------

   function NextProtectedType (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         -- filter items to leave just protected types that are actually declared (not just announced)
         if IsProtectedType (Item) and then IsVisibleScope (GetScope (Item)) then
            Next := Iterator'(ProtectedTypeIterator, IsAbstract, Item, Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return Next;

   end NextProtectedType;

   --------------------------------------------------------------------------------

   function NextSubprogramParameter (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Next : Iterator;

      --------------------------------------------------------------------------------

      function NextFormalParameter (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         SubprogramParameter : Symbol;
         ProofFunction       : Symbol;
         AdaFunction         : Symbol;
         GlobalVariables     : Iterator;
         Next                : Iterator;
      begin

         SubprogramParameter := RawDict.GetNextSubprogramParameter (CurrentSymbol (Previous));
         ProofFunction       := Previous.Context;

         if SubprogramParameter = NullSymbol then
            if ProofFunction = NullSymbol then
               Next := NullIterator;
            else
               AdaFunction     := GetAdaFunction (ProofFunction);
               GlobalVariables := FirstGlobalVariable (Previous.Abstraction, AdaFunction);
               if GlobalVariables = NullIterator then
                  Next := NullIterator;
               else
                  Next :=
                    Iterator'
                    (ImplicitProofFunctionGlobalIterator,
                     Previous.Abstraction,
                     CurrentSymbol (GlobalVariables),
                     GlobalVariables.Context);
               end if;
            end if;
         else
            Next := Iterator'(Previous.Discriminant, Previous.Abstraction, SubprogramParameter, ProofFunction);
         end if;

         return Next;

      end NextFormalParameter;

      --------------------------------------------------------------------------------

      function NextGlobalVariable (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         GlobalVariable : Symbol;
         Next           : Iterator;
      begin

         GlobalVariable := RawDict.GetNextGlobalVariable (Previous.Context);

         if GlobalVariable = NullSymbol then
            Next := NullIterator;
         else
            Next :=
              Iterator'
              (Previous.Discriminant,
               Previous.Abstraction,
               RawDict.GetGlobalVariableVariable (GlobalVariable),
               GlobalVariable);
         end if;

         return Next;

      end NextGlobalVariable;

      --------------------------------------------------------------------------------

   begin
      case Previous.Discriminant is
         when SubprogramParameterIterator | ImplicitProofFunctionParameterIterator =>
            Next := NextFormalParameter (Previous);

         when ImplicitProofFunctionGlobalIterator =>
            Next := NextGlobalVariable (Previous);

         when others =>
            Next := NullIterator; --782--this removes DF error from non-exec path

      end case;

      return Next;
   end NextSubprogramParameter;

   --------------------------------------------------------------------------------

   function NextGlobalVariable (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      GlobalVariable : Symbol;
      Next           : Iterator;
   begin

      GlobalVariable := RawDict.GetNextGlobalVariable (Previous.Context);

      if GlobalVariable = NullSymbol then
         Next := NullIterator;
      else
         Next :=
           Iterator'(GlobalVariableIterator, IsAbstract, RawDict.GetGlobalVariableVariable (GlobalVariable), GlobalVariable);
      end if;

      return Next;

   end NextGlobalVariable;

   --------------------------------------------------------------------------------

   function NextLocalVariable (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsVariable (Item) then
            Next := Iterator'(LocalVariableIterator, IsAbstract, Item, Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return Next;

   end NextLocalVariable;

   --------------------------------------------------------------------------------

   function NextProtectedElement (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);
      if Declaration = NullSymbol then
         Next := NullIterator;
      else
         Item := RawDict.GetDeclarationItem (Declaration);
         Next := Iterator'(LocalVariableIterator, IsAbstract, Item, Declaration);
      end if;
      return Next;
   end NextProtectedElement;

   --------------------------------------------------------------------------------

   function NextInitializedVariable (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsVariable (Item) and then VariableIsInitialized (Item) then
            Next := Iterator'(InitializedVariableIterator, IsAbstract, Item, Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return Next;

   end NextInitializedVariable;

   --------------------------------------------------------------------------------

   function NextImportExport (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Next : Iterator;

      --------------------------------------------------------------------------------

      function NextSubprogramParameterImportExport (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         SubprogramParameter : Symbol;
         TheProcedure        : Symbol;
         GlobalVariable      : Symbol;
         Variable            : Symbol;
         Next                : Iterator;
      begin

         SubprogramParameter := RawDict.GetNextSubprogramParameter (CurrentSymbol (Previous));
         TheProcedure        := RawDict.GetSubprogramParameterSubprogram (CurrentSymbol (Previous));

         loop
            exit when SubprogramParameter = NullSymbol;
            exit when IsImportExport (Previous.Abstraction, TheProcedure, SubprogramParameter);
            SubprogramParameter := RawDict.GetNextSubprogramParameter (SubprogramParameter);
         end loop;

         if SubprogramParameter = NullSymbol then

            GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (Previous.Abstraction, TheProcedure);

            loop
               exit when GlobalVariable = NullSymbol;
               Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);
               exit when IsImportExport (Previous.Abstraction, TheProcedure, Variable);
               GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);
            end loop;

            if GlobalVariable = NullSymbol then
               Next := NullIterator;
            else
               --# accept Flow, 501, Variable, "Non-executable path";
               Next := Iterator'(ImportExportIterator, -- 782 - Expect flow error here for non-exec path
                                 Previous.Abstraction, Variable, GlobalVariable);
               --# end accept;
            end if;

         else
            Next := Iterator'(ImportExportIterator, Previous.Abstraction, SubprogramParameter, NullSymbol);
         end if;

         --# accept Flow, 602, Variable, "Non-executable path";
         return Next;

      end NextSubprogramParameterImportExport;

      --------------------------------------------------------------------------------

      function NextGlobalVariableImportExport (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         TheProcedure   : Symbol;
         GlobalVariable : Symbol;
         Variable       : Symbol;
         Next           : Iterator;
      begin

         GlobalVariable := Previous.Context;
         TheProcedure   := RawDict.GetGlobalVariableSubprogram (GlobalVariable);

         loop
            GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);
            exit when GlobalVariable = NullSymbol;
            Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);
            exit when IsImportExport (Previous.Abstraction, TheProcedure, Variable);
         end loop;

         if GlobalVariable = NullSymbol then
            Next := NullIterator;
         else
            --# accept Flow, 501, Variable, "Non-executable path";
            Next := Iterator'(ImportExportIterator, -- 782 - Expect flow error here for non-exec path
                              Previous.Abstraction, Variable, GlobalVariable);
            --# end accept;
         end if;

         --# accept Flow, 602, Variable, "Non-executable path";
         return Next;

      end NextGlobalVariableImportExport;

      --------------------------------------------------------------------------------

   begin

      if Previous.Context = NullSymbol then
         Next := NextSubprogramParameterImportExport (Previous);
      else
         Next := NextGlobalVariableImportExport (Previous);
      end if;

      return Next;

   end NextImportExport;

   --------------------------------------------------------------------------------

   function NextExport (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Next : Iterator;

      --------------------------------------------------------------------------------

      function NextSubprogramParameterExport (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         Subprogram          : Symbol;
         SubprogramParameter : Symbol;
         GlobalVariable      : Symbol;
         Variable            : Symbol;
         Next                : Iterator;
      begin

         SubprogramParameter := CurrentSymbol (Previous);
         Subprogram          := GetSubprogram (SubprogramParameter);

         loop

            SubprogramParameter := RawDict.GetNextSubprogramParameter (SubprogramParameter);

            if SubprogramParameter = NullSymbol then
               Next := NullIterator;
               exit;
            end if;

            if IsExport (Previous.Abstraction, Subprogram, SubprogramParameter) then
               Next := Iterator'(ExportIterator, Previous.Abstraction, SubprogramParameter, NullSymbol);
               exit;
            end if;

         end loop;

         if Next = NullIterator then

            GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (Previous.Abstraction, Subprogram);

            loop

               exit when GlobalVariable = NullSymbol;

               Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);

               if IsExport (Previous.Abstraction, Subprogram, Variable) then
                  Next := Iterator'(ExportIterator, Previous.Abstraction, Variable, GlobalVariable);
                  exit;
               end if;

               GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);

            end loop;

         end if;

         return Next;

      end NextSubprogramParameterExport;

      --------------------------------------------------------------------------------

      function NextGlobalVariableExport (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         GlobalVariable : Symbol;
         Subprogram     : Symbol;
         Variable       : Symbol;
         Next           : Iterator;
      begin

         GlobalVariable := Previous.Context;
         Subprogram     := RawDict.GetGlobalVariableSubprogram (GlobalVariable);

         loop

            GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);

            if GlobalVariable = NullSymbol then
               Next := NullIterator;
               exit;
            end if;

            Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);

            if IsExport (Previous.Abstraction, Subprogram, Variable) then
               Next := Iterator'(ExportIterator, Previous.Abstraction, Variable, GlobalVariable);
               exit;
            end if;

         end loop;

         return Next;

      end NextGlobalVariableExport;

      --------------------------------------------------------------------------------

   begin

      if Previous.Context = NullSymbol then
         Next := NextSubprogramParameterExport (Previous);
      else
         Next := NextGlobalVariableExport (Previous);
      end if;

      return Next;

   end NextExport;

   --------------------------------------------------------------------------------

   function NextImport (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Next : Iterator;

      --------------------------------------------------------------------------------

      function NextSubprogramParameterImport (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         Subprogram          : Symbol;
         SubprogramParameter : Symbol;
         GlobalVariable      : Symbol;
         Variable            : Symbol;
         Next                : Iterator;
      begin

         SubprogramParameter := CurrentSymbol (Previous);
         Subprogram          := GetSubprogram (SubprogramParameter);

         loop

            SubprogramParameter := RawDict.GetNextSubprogramParameter (SubprogramParameter);

            if SubprogramParameter = NullSymbol then
               Next := NullIterator;
               exit;
            end if;

            if IsImport (Previous.Abstraction, Subprogram, SubprogramParameter) then
               Next := Iterator'(ImportIterator, Previous.Abstraction, SubprogramParameter, NullSymbol);
               exit;
            end if;

         end loop;

         if Next = NullIterator then

            GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (Previous.Abstraction, Subprogram);

            loop

               exit when GlobalVariable = NullSymbol;

               Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);

               if IsImport (Previous.Abstraction, Subprogram, Variable) then
                  Next := Iterator'(ImportIterator, Previous.Abstraction, Variable, GlobalVariable);
                  exit;
               end if;

               GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);

            end loop;

         end if;

         return Next;

      end NextSubprogramParameterImport;

      --------------------------------------------------------------------------------

      function NextGlobalVariableImport (Previous : Iterator) return Iterator
      --# global in Dict;
      is
         GlobalVariable : Symbol;
         Subprogram     : Symbol;
         Variable       : Symbol;
         Next           : Iterator;
      begin

         GlobalVariable := Previous.Context;
         Subprogram     := RawDict.GetGlobalVariableSubprogram (GlobalVariable);

         loop

            GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);

            if GlobalVariable = NullSymbol then
               Next := NullIterator;
               exit;
            end if;

            Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);

            if IsImport (Previous.Abstraction, Subprogram, Variable) then
               Next := Iterator'(ImportIterator, Previous.Abstraction, Variable, GlobalVariable);
               exit;
            end if;

         end loop;

         return Next;

      end NextGlobalVariableImport;

      --------------------------------------------------------------------------------

   begin

      if Previous.Context = NullSymbol then
         Next := NextSubprogramParameterImport (Previous);
      else
         Next := NextGlobalVariableImport (Previous);
      end if;

      return Next;

   end NextImport;

   --------------------------------------------------------------------------------

   function NextDependency (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Dependency : Symbol;
      Next       : Iterator;
   begin

      Dependency := RawDict.GetNextDependency (Previous.Context);

      if Dependency = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(DependencyIterator, IsAbstract, RawDict.GetDependencyImport (Dependency), Dependency);
      end if;

      return Next;

   end NextDependency;

   --------------------------------------------------------------------------------

   function NextInterruptStreamMapping (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Mapping : Symbol;
      Next    : Iterator;
   begin

      Mapping := RawDict.GetNextInterruptStreamMapping (Previous.Context);

      if Mapping = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(InterruptStreamMappingIterator, IsAbstract, Mapping, Mapping);
      end if;
      return Next;
   end NextInterruptStreamMapping;

   --------------------------------------------------------------------------------

   function NextSuspendsListItem (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      SuspendsListItem : Symbol;
      Next             : Iterator;
   begin

      SuspendsListItem := RawDict.GetNextSuspendsListItem (Previous.Context);

      if SuspendsListItem = NullSymbol then
         Next := NullIterator;
      else
         Next :=
           Iterator'(SuspendsListItemIterator, IsAbstract, RawDict.GetSuspendsListItem (SuspendsListItem), SuspendsListItem);
      end if;

      return Next;

   end NextSuspendsListItem;

   --------------------------------------------------------------------------------

   function NextVirtualElement (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      VirtualElement : Symbol;
      Next           : Iterator;
   begin
      VirtualElement := RawDict.GetNextVirtualElement (Previous.Context);
      if VirtualElement = NullSymbol then
         Next := NullIterator;
      else
         Next :=
           Iterator'(VirtualElementIterator, IsAbstract, RawDict.GetVirtualElementVariable (VirtualElement), VirtualElement);
      end if;
      return Next;
   end NextVirtualElement;

   --------------------------------------------------------------------------------

   function NextOwnedPackage (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      PrevSym    : Symbol;
      Item       : Symbol;
      Descendent : Symbol;
      Next       : Iterator;
   begin

      PrevSym := CurrentSymbol (Previous);
      Item    := RawDict.GetPackageSibling (PrevSym);

      if Item /= NullSymbol then
         loop
            Descendent := RawDict.GetPackageFirstChild (Item, Public);
            exit when Descendent = NullSymbol;
            Item := Descendent;
         end loop;
      elsif not IsPrivatePackage (PrevSym) then
         Item := RawDict.GetPackageParent (PrevSym);
      end if;

      if Item = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(OwnedPackageIterator, IsAbstract, Item, Previous.Context);
      end if;

      return Next;

   end NextOwnedPackage;

   --------------------------------------------------------------------------------

   function NextEmbeddedPackage (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Declaration : Symbol;
      Item        : Symbol;
      Next        : Iterator;
   begin

      Declaration := RawDict.GetNextDeclaration (Previous.Context);

      loop
         if Declaration = NullSymbol then
            Next := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsPackage (Item) and then Declaration /= RawDict.GetPackageBody (Item) then
            Next := Iterator'(EmbeddedPackageIterator, IsAbstract, Item, Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return Next;

   end NextEmbeddedPackage;

   --------------------------------------------------------------------------------

   function NextOwnVariable (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      OwnVariable  : Symbol;
      OwnVariables : Iterator;
   begin

      OwnVariable := RawDict.GetNextOwnVariable (Previous.Current);

      if OwnVariable = NullSymbol then
         OwnVariables := NullIterator;
      else
         OwnVariables := Iterator'(OwnVariableIterator, IsAbstract, OwnVariable, NullSymbol);
      end if;

      return OwnVariables;

   end NextOwnVariable;

   --------------------------------------------------------------------------------

   function NextInitializedOwnVariable (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      OwnVariable             : Symbol;
      InitializedOwnVariables : Iterator;
   begin

      OwnVariable := RawDict.GetNextOwnVariable (Previous.Current);

      loop

         if OwnVariable = NullSymbol then
            InitializedOwnVariables := NullIterator;
            exit;
         end if;

         if RawDict.GetOwnVariableInitialized (OwnVariable) then
            InitializedOwnVariables := Iterator'(InitializedOwnVariableIterator, IsAbstract, OwnVariable, NullSymbol);
            exit;
         end if;

         OwnVariable := RawDict.GetNextOwnVariable (OwnVariable);

      end loop;

      return InitializedOwnVariables;

   end NextInitializedOwnVariable;

   --------------------------------------------------------------------------------

   function NextAbstractOwnVariable (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      OwnVariable          : Symbol;
      AbstractOwnVariables : Iterator;
   begin

      OwnVariable := RawDict.GetNextOwnVariable (Previous.Current);

      loop

         if OwnVariable = NullSymbol then
            AbstractOwnVariables := NullIterator;
            exit;
         end if;

         if RawDict.GetOwnVariableConstituents (OwnVariable) /= NullSymbol then
            AbstractOwnVariables := Iterator'(AbstractOwnVariableIterator, IsAbstract, OwnVariable, NullSymbol);
            exit;
         end if;

         OwnVariable := RawDict.GetNextOwnVariable (OwnVariable);

      end loop;

      return AbstractOwnVariables;

   end NextAbstractOwnVariable;

   --------------------------------------------------------------------------------

   function NextConstituent (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Constituent  : Symbol;
      Constituents : Iterator;
   begin

      Constituent := RawDict.GetNextConstituent (Previous.Current);

      if Constituent = NullSymbol then
         Constituents := NullIterator;
      else
         Constituents := Iterator'(ConstituentIterator, IsAbstract, Constituent, NullSymbol);
      end if;

      return Constituents;

   end NextConstituent;

   --------------------------------------------------------------------------------

   function NextKnownDiscriminant (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Discriminant : Symbol;
      Next         : Iterator;
   begin

      Discriminant := RawDict.GetNextDiscriminant (Previous.Current);

      if Discriminant = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(KnownDiscriminantIterator, IsAbstract, Discriminant, NullSymbol);
      end if;

      return Next;

   end NextKnownDiscriminant;

   --------------------------------------------------------------------------------

   function NextDiscriminantConstraint (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Discriminant : Symbol;
      Next         : Iterator;
   begin
      Discriminant := RawDict.GetNextDiscriminantConstraint (Previous.Current);

      if Discriminant = NullSymbol then
         Next := NullIterator;
      else
         Next := Iterator'(DiscriminantConstraintIterator, IsAbstract, Discriminant, NullSymbol);
      end if;

      return Next;

   end NextDiscriminantConstraint;

   --------------------------------------------------------------------------------

   function NextGenericFormalParameter (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Next   : Symbol;
      Result : Iterator;
   begin

      Next := RawDict.GetGenericParameterNext (Previous.Current);

      if Next = NullSymbol then
         Result := NullIterator;
      else
         Result := Iterator'(GenericFormalParameterIterator, IsAbstract, Next, NullSymbol);
      end if;

      return Result;

   end NextGenericFormalParameter;

   --------------------------------------------------------------------------------

   function NextLoopOnEntryVar (Previous : Iterator) return Iterator
   --# global in Dict;
   is
      Next   : Symbol;
      Result : Iterator;
   begin

      Next := RawDict.GetLoopEntryVariableNext (Previous.Current);

      if Next = NullSymbol then
         Result := NullIterator;
      else
         Result := Iterator'(LoopOnEntryVarIterator, IsAbstract, Next, NullSymbol);
      end if;

      return Result;

   end NextLoopOnEntryVar;

   --------------------------------------------------------------------------------

begin -- NextSymbol

   case Previous.Discriminant is
      when NullSymIterator | DeclarativeItemIterator =>
         Next := NextDeclarativeItem (Previous);
      when DeferredConstantIterator =>
         Next := NextDeferredConstant (Previous);
      when ArrayIndexIterator =>
         Next := NextArrayIndex (Previous);
      when LoopIterator =>
         Next := NextLoop (Previous);
      when UndeclaredTypeIterator =>
         Next := NextUndeclaredType (Previous);
      when PrivateTypeIterator =>
         Next := NextPrivateType (Previous);
      when EnumerationLiteralIterator =>
         Next := NextEnumerationLiteral (Previous);
      when RecordComponentIterator =>
         Next := NextRecordComponent (Previous);
      when ExtendedRecordComponentIterator =>
         Next := NextExtendedRecordComponent (Previous);
      when LibraryUnitIterator =>
         Next := NextLibraryUnit (Previous);
      when WithedPackageIterator =>
         Next := NextWithedPackage (Previous);
      when InheritedPackageIterator =>
         Next := NextInheritedPackage (Previous);
      when VisibleSubprogramIterator | PrivateSubprogramIterator =>
         Next := NextVisibleOrPrivateSubprogram (Previous);
      when TaskTypeIterator =>
         Next := NextTaskType (Previous);
      when OwnTaskIterator =>
         Next := NextOwnTask (Previous);
      when ProtectedTypeIterator =>
         Next := NextProtectedType (Previous);
      when SubprogramParameterIterator | ImplicitProofFunctionGlobalIterator | ImplicitProofFunctionParameterIterator =>
         Next := NextSubprogramParameter (Previous);
      when KnownDiscriminantIterator =>
         Next := NextKnownDiscriminant (Previous);
      when DiscriminantConstraintIterator =>
         Next := NextDiscriminantConstraint (Previous);
      when GlobalVariableIterator =>
         Next := NextGlobalVariable (Previous);
      when LocalVariableIterator =>
         Next := NextLocalVariable (Previous);
      when ProtectedElementIterator =>
         Next := NextProtectedElement (Previous);
      when InitializedVariableIterator =>
         Next := NextInitializedVariable (Previous);
      when ImportExportIterator =>
         Next := NextImportExport (Previous);
      when ExportIterator =>
         Next := NextExport (Previous);
      when ImportIterator =>
         Next := NextImport (Previous);
      when DependencyIterator =>
         Next := NextDependency (Previous);
      when InterruptStreamMappingIterator =>
         Next := NextInterruptStreamMapping (Previous);
      when SuspendsListItemIterator =>
         Next := NextSuspendsListItem (Previous);
      when VirtualElementIterator =>
         Next := NextVirtualElement (Previous);
      when OwnedPackageIterator =>
         Next := NextOwnedPackage (Previous);
      when EmbeddedPackageIterator =>
         Next := NextEmbeddedPackage (Previous);
      when OwnVariableIterator =>
         Next := NextOwnVariable (Previous);
      when InitializedOwnVariableIterator =>
         Next := NextInitializedOwnVariable (Previous);
      when AbstractOwnVariableIterator =>
         Next := NextAbstractOwnVariable (Previous);
      when ConstituentIterator =>
         Next := NextConstituent (Previous);
      when GenericFormalParameterIterator =>
         Next := NextGenericFormalParameter (Previous);
      when LoopOnEntryVarIterator =>
         Next := NextLoopOnEntryVar (Previous);
   end case;

   return Next;

end NextSymbol;
