// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  chartdialogs.cc
  This file contains code for the dialog boxes in the StarPlot Chart menu.
*/

#include <gtk/gtk.h>
#include "starplot.h"


/* automagnitude(): Tries to estimate a good dim magnitude cutoff for a given 
 *  chart radius. */
double automagnitude(double radius)
{
  if (radius < 10) return 25;
  else if (radius <   12.5) return 25 - (radius -   10) * 4.4;
  else if (radius <   15) return 14   - (radius - 12.5) * 1.0;
  else if (radius <   20) return 11.5 - (radius -   15) * 0.8;
  else if (radius <   30) return  7.5 - (radius -   20) * 0.26;
  else if (radius <   40) return  4.9 - (radius -   30) * 0.14;
  else if (radius <   50) return  3.5 - (radius -   40) * 0.10;
  else if (radius <   75) return  2.5 - (radius -   50) * 0.068;
  else if (radius <  150) return  0.8 - (radius -   75) * 0.0173;
  else if (radius <  200) return -0.5 - (radius -  150) * 0.002;
  else if (radius <  600) return -0.6 - (radius -  200) * 0.005;
  else if (radius < 1000) return -2.6 - (radius -  600) * 0.0025;
  else if (radius < 2000) return -3.6 - (radius - 1000) * 0.001;
  else if (radius < 4000) return -4.6 - (radius - 2000) * 0.00045;
  else if (radius <10000) return -5.5 - (radius - 4000) * 0.000133;
  else if (radius <12500) return -6.3;
  else return -6.4;
}

/* my_gtk_position_dialog(): Sets up a box containing entry boxes for
 *  RA and Declination */
void my_gtk_position_dialog( GtkWidget **entrybox, GtkWidget *insertionbox,
			     SolidAngle posn, bool celestial_coords )
{
  GtkWidget *table, *labels[8];
  char labeltext[8][20], boxentries[6][5];

  sprintf(labeltext[5], DEGREE_SYMBOL);
  sprintf(labeltext[6], "'");
  sprintf(labeltext[7], "\"");

  if (celestial_coords) {
    sprintf(labeltext[1], "h"); sprintf(labeltext[2], "m");
    sprintf(labeltext[3], "s");
    sprintf(labeltext[0], "Right Ascension:");
    sprintf(labeltext[4], "Declination:");
  }
  else {
    sprintf(labeltext[1], DEGREE_SYMBOL); sprintf(labeltext[2], "'");
    sprintf(labeltext[3], "\"");
    sprintf(labeltext[0], "Galactic Longitude:");
    sprintf(labeltext[4], "Galactic Latitude:");
  }

  RadiansToRAStrings(posn.getPhi(), boxentries[0], boxentries[1], 
  		     boxentries[2], celestial_coords);
  RadiansToDecStrings(posn.getTheta(),
		      boxentries[3], boxentries[4], boxentries[5]);

  for (int i = 0; i < 6; i++) {
    stripspace(boxentries[i]);
    entrybox[i] = gtk_entry_new_with_max_length (5);
    gtk_entry_set_text (GTK_ENTRY (entrybox[i]), boxentries[i]);
    gtk_widget_set_usize (entrybox[i], 40, 20);
    gtk_widget_show (entrybox[i]);
  }
  for (int i = 0; i < 8; i++) {
    labels[i] = gtk_label_new (labeltext[i]);
    gtk_misc_set_alignment (GTK_MISC (labels[i]), (float)0.0, (float)0.0);
    gtk_widget_show (labels[i]);
  }

  table = gtk_table_new (2, 7, FALSE);
  gtk_widget_show (table);
  for (int i = 0; i < 2; i++) {
    gtk_table_attach (GTK_TABLE (table), labels[i*4], 0, 1, i, i + 1,
		      (GtkAttachOptions)GTK_FILL, (GtkAttachOptions)GTK_FILL,
		      5, 5);
    for (int j = 0; j < 3; j++) {
      gtk_table_attach (GTK_TABLE (table), entrybox[i*3 + j],
			j*2 + 1, j*2 + 2, i, i + 1,
			(GtkAttachOptions)0, (GtkAttachOptions)0, 5, 5);
      gtk_table_attach (GTK_TABLE (table), labels[i*4 + 1 + j],
			j*2 + 2, j*2 + 3, i, i + 1,
			(GtkAttachOptions)0, (GtkAttachOptions)0, 5, 5);
    }
  }
  gtk_box_pack_start (GTK_BOX (insertionbox), table, FALSE, FALSE, 0);

  return;
}


/* Handler for the Chart->Define Chart menu item */

/* Callback functions */

/* chart_define_callback(): Set the chart parameters to what was specified
 * in the dialog box and update the chart. */
void chart_define_callback(GtkWidget *widget, gpointer chart_define_data)
{
  GtkWidget **data = (GtkWidget **)chart_define_data;
  double oldchartradius = starrules.ChartRadius;
  double newchartradius = myatof(gtk_entry_get_text (GTK_ENTRY (data[7])));

  starrules.ChartLocation =
    SolidAngle( RAStringsToRadians( gtk_entry_get_text (GTK_ENTRY (data[0])),
				    gtk_entry_get_text (GTK_ENTRY (data[1])),
				    gtk_entry_get_text (GTK_ENTRY (data[2])),
				    starrules.CelestialCoords),
		DecStringsToRadians( gtk_entry_get_text (GTK_ENTRY (data[3])),
				     gtk_entry_get_text (GTK_ENTRY (data[4])),
				     gtk_entry_get_text (GTK_ENTRY (data[5]))
				     )
		).toCartesian(myatof(gtk_entry_get_text(GTK_ENTRY(data[6]))));
  starrules.ChartRadius = newchartradius;

  /* Even if the magnitude autoset feature is on, the user may have set
   *  the dim magnitude limit manually.  We should discard the user's
   *  changes only if the new chart radius differs from the old radius.
   *  In this test, we compare the ratio of new and old radii to 1.0 in order
   *  to get around floating point imprecision. */

  if (starrules.ChartAutosetDimmestMagnitude
      && (oldchartradius == 0.0
	  || fabs(newchartradius / oldchartradius - 1.0) > 0.01))
    starrules.ChartDimmestMagnitude = automagnitude(starrules.ChartRadius);

  redraw_all(LOCATION_CHANGE);
  return;
}

/* chart_define_defaults(): Restore the values handled by this dialog box
 * to their default settings and update the chart. */
void chart_define_defaults(GtkWidget *widget, gpointer data /* not used */)
{
  double oldchartradius = starrules.ChartRadius;
  double newchartradius = initrules.ChartRadius;

  starrules.ChartLocation = initrules.ChartLocation;
  starrules.ChartRadius = newchartradius;

  /* Even if the magnitude autoset feature is on, the user may have set
   *  the dim magnitude limit manually.  We should discard the user's
   *  changes only if the new chart radius differs from the old radius.
   *  In this test, we compare the ratio of new and old radii to 1.0 in order
   *  to get around floating point imprecision. */

  if (starrules.ChartAutosetDimmestMagnitude
      && (oldchartradius == 0.0
	  || fabs(newchartradius / oldchartradius - 1.0) > 0.01))
    starrules.ChartDimmestMagnitude = automagnitude(starrules.ChartRadius);

  redraw_all(LOCATION_CHANGE);
  return;
}

/* chart_define_search(): Look up the star in the star name entry box, and if
 *  found, insert its coordinates into all the other entry boxes.  Note: this
 *  function does not do anything else, it is still up to the user to hit
 *  OK to set these coordinates! */
void chart_define_search(GtkWidget *widget, gpointer chart_define_data)
{
  GtkWidget **data = (GtkWidget **)chart_define_data;
  const char *starname = gtk_entry_get_text (GTK_ENTRY (data[8]));
  StarArray searchresults;

  if (isempty(starname)) return;

  // Try several different kinds of searches, order of most to least specific
  searchresults.Search(starname, starrules.ChartFileNames, starrules,
		       /* not case-sensitive */ false,
		       /* exact match        */ true,
		       /* exit on match      */ true);
  if (! searchresults.size())
    searchresults.Search(starname, starrules.ChartFileNames, starrules,
			 /* case-sensitive  */ true,
			 /* substring match */ false,
			 /* exit on match   */ true);
  if (! searchresults.size())
    searchresults.Search(starname, starrules.ChartFileNames, starrules,
			 /* not case-sensitive */ false,
			 /* substring match    */ false,
			 /* exit on match      */ true);
  
  // if we don't find anything, leave a message in the star name entry
  //  box and return
  if (! searchresults.size()) {
    gtk_entry_set_text (GTK_ENTRY (data[8]), "Sorry, not found");
    return;
  }

  // otherwise, put the name which matched into the star name entry box,
  //  and put the star coordinates into the coordinate entry boxes.

  StringList infolist = searchresults[0].GetInfo(starrules,
						 false /*no dms punctuation*/,
						 SUBFIELD_DELIMITER);
  // where is the star name which matches the search string in the
  //  list of star characteristics?
  unsigned int nameplace = myatoi(infolist[0]);
  unsigned int infolistplace = nameplace ? nameplace + 9 : 1;
  gtk_entry_set_text (GTK_ENTRY (data[8]), infolist[infolistplace]);

  StringList ra = StringList(infolist[2], SUBFIELD_DELIMITER);
  StringList dec = StringList(infolist[3], SUBFIELD_DELIMITER);

  if (strcmp(ra[0], "N/A") == 0)
    ra = dec = StringList("0,0,0", ',');

  for (unsigned int i = 0; i < 3; i++) {
    gtk_entry_set_text (GTK_ENTRY (data[i]), ra[i]);
    gtk_entry_set_text (GTK_ENTRY (data[i + 3]), dec[i]);
  }
  gtk_entry_set_text (GTK_ENTRY (data[6]), infolist[4]);

  return;
}

/* This function has lots of GTK gui boilerplate because it is the most 
 *  complicated of the dialog boxes.  Sorry about that. */
void chart_define()
{
  GtkWidget *definedialog, *main_vbox, *query;
  GtkWidget *lookup_frame, *lookup_hbox, *lookup_entry, *search;
  GtkWidget *coord_frame, *coord_vbox;
  GtkWidget *coord_entry[6], *distance_hbox, *distance_label, *distance_entry;
  GtkWidget *chartrad_hbox, *chartrad_label, *chartrad_entry;
  GtkWidget *OK, *defaults, *cancel;
  static GtkWidget *chart_define_data[9];
  char chartrad_text[13], distance_text[13];

  // The window
  definedialog = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_policy (GTK_WINDOW (definedialog), FALSE, FALSE, TRUE);
  gtk_window_set_title (GTK_WINDOW (definedialog), "StarPlot: Define Chart");
  gtk_window_set_modal (GTK_WINDOW (definedialog), TRUE);
  gtk_signal_connect (GTK_OBJECT (definedialog), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);

  // Pack it vertically
  main_vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 10);
  gtk_container_add (GTK_CONTAINER (definedialog), main_vbox);
  gtk_widget_show (main_vbox);

  // Query text
  query = gtk_label_new (" Where should the chart origin be located?");
  gtk_box_pack_start (GTK_BOX (main_vbox), query, FALSE, FALSE, 0);
  gtk_misc_set_alignment (GTK_MISC (query), (float)0.0, (float)0.0);
  gtk_widget_show (query);

  // First element: a frame containing an hbox containing the star search
  //  entry box
  lookup_frame = gtk_frame_new ("Enter a star name and press \"Search\"");
  gtk_box_pack_start (GTK_BOX (main_vbox), lookup_frame, FALSE, FALSE, 0);
  gtk_widget_show (lookup_frame);

  lookup_hbox = gtk_hbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (lookup_frame), lookup_hbox);
  gtk_container_set_border_width (GTK_CONTAINER (lookup_hbox), 5);
  gtk_widget_show (lookup_hbox);
  
  lookup_entry = gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (lookup_hbox), lookup_entry, TRUE, TRUE, 5);
  gtk_widget_show (lookup_entry);

  search = gtk_button_new_with_label ("Search");
  gtk_box_pack_start (GTK_BOX (lookup_hbox), search, FALSE, FALSE, 5);
  gtk_widget_set_usize (search, BUTTONWIDTH, BUTTONHEIGHT);
  gtk_widget_show (search);

  // Second element: a frame containing a vbox containing the coordinate
  //  entry boxes
  coord_frame = gtk_frame_new ("Or, enter coordinates manually");
  gtk_box_pack_start (GTK_BOX (main_vbox), coord_frame, FALSE, FALSE, 0);
  gtk_widget_show (coord_frame);

  coord_vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (coord_frame), coord_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (coord_vbox), 5);
  gtk_widget_show (coord_vbox);

  my_gtk_position_dialog (coord_entry, coord_vbox,
			  starrules.ChartLocation.toSpherical(),
			  starrules.CelestialCoords);

  distance_hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (coord_vbox), distance_hbox, FALSE, FALSE, 5);
  gtk_widget_show (distance_hbox);

  distance_label = gtk_label_new ("Distance of chart origin from Sun (LY):");
  gtk_box_pack_start (GTK_BOX(distance_hbox), distance_label, FALSE, FALSE, 5);
  gtk_widget_show (distance_label);

  snprintf(distance_text, 13, "%.6g", starrules.ChartLocation.magnitude());
  distance_text[12] = 0;
  distance_entry = gtk_entry_new_with_max_length (12);
  gtk_box_pack_start (GTK_BOX(distance_hbox), distance_entry, FALSE, FALSE, 5);
  gtk_entry_set_text (GTK_ENTRY (distance_entry), distance_text);
  gtk_widget_set_usize (distance_entry, 80, 20);
  gtk_widget_show (distance_entry);

  // Third element: an hbox containing the Chart Radius entry box
  chartrad_hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (main_vbox), chartrad_hbox, FALSE, FALSE, 10);
  gtk_widget_show (chartrad_hbox);

  chartrad_label = gtk_label_new ("Radius of chart (LY):");
  gtk_box_pack_start (GTK_BOX(chartrad_hbox), chartrad_label, FALSE, FALSE, 5);
  gtk_misc_set_alignment (GTK_MISC (chartrad_label), (float)0.0, (float)0.0);
  gtk_widget_show (chartrad_label);

  snprintf(chartrad_text, 13, "%.6g", starrules.ChartRadius);
  chartrad_text[12] = 0;
  chartrad_entry = gtk_entry_new_with_max_length (12);
  gtk_box_pack_start(GTK_BOX(chartrad_hbox), chartrad_entry, FALSE, FALSE, 10);
  gtk_entry_set_text (GTK_ENTRY (chartrad_entry), chartrad_text);
  gtk_widget_set_usize (chartrad_entry, 80, 20);
  gtk_widget_show (chartrad_entry);

  // Last element: the OK/Defaults/Cancel buttons
  my_gtk_button_bar (&OK, &defaults, &cancel, main_vbox);

  // Set up chart_define_data
  for (int i = 0; i < 6; i++)
    chart_define_data[i] = coord_entry[i];
  chart_define_data[6] = distance_entry;
  chart_define_data[7] = chartrad_entry;
  chart_define_data[8] = lookup_entry;

  // Connect the buttons to signals
  gtk_signal_connect (GTK_OBJECT (search), "clicked",
		      GTK_SIGNAL_FUNC (chart_define_search),
		      chart_define_data);
  gtk_signal_connect (GTK_OBJECT (lookup_entry), "activate",
		      GTK_SIGNAL_FUNC (chart_define_search),
		      chart_define_data);
  gtk_signal_connect (GTK_OBJECT (OK), "clicked", 
		      GTK_SIGNAL_FUNC (chart_define_callback),
		      chart_define_data);
  gtk_signal_connect (GTK_OBJECT (defaults), "clicked", 
		      GTK_SIGNAL_FUNC (chart_define_defaults), NULL);
  my_gtk_buttons_connect_destroy (OK, defaults, cancel, definedialog);

  // Finally, set the window policy and show it
  gtk_window_set_focus (GTK_WINDOW (definedialog), OK);
  gtk_widget_show (definedialog);

  return;
}


/* Handler for the Chart->Orientation menu item */

/* Callback functions */

/* chart_orient_callback(): Set the chart parameters to what was specified
 * in the dialog box and update the chart. */
void chart_orient_callback(GtkWidget *widget, gpointer chart_orient_data)
{
  GtkWidget **data = (GtkWidget **)chart_orient_data;

  starrules.ChartOrientation =
    SolidAngle( RAStringsToRadians( gtk_entry_get_text (GTK_ENTRY (data[0])),
				    gtk_entry_get_text (GTK_ENTRY (data[1])),
				    gtk_entry_get_text (GTK_ENTRY (data[2])),
				    starrules.CelestialCoords),
		DecStringsToRadians( gtk_entry_get_text (GTK_ENTRY (data[3])),
				     gtk_entry_get_text (GTK_ENTRY (data[4])),
				     gtk_entry_get_text (GTK_ENTRY (data[5]))
				     )
		); // <sarcasm>what is this, LISP?</sarcasm>

  redraw_all(ORIENTATION_CHANGE);
  return;
}

/* chart_orient_defaults(): Restore the values handled by this dialog box
 * to their default settings and update the chart. */
void chart_orient_defaults(GtkWidget *widget, gpointer data)
{
  starrules.ChartOrientation = initrules.ChartOrientation;
  redraw_all(ORIENTATION_CHANGE);
  return;
}

void chart_orient()
{
  GtkWidget *orientdialog, *vbox, *entrybox[6], *explanation;
  GtkWidget *OK, *defaults, *cancel;
  static GtkWidget *chart_orient_data[6];

  // The window
  orientdialog = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_policy (GTK_WINDOW (orientdialog), FALSE, FALSE, TRUE);
  gtk_window_set_title (GTK_WINDOW (orientdialog), "StarPlot: Orientation");
  gtk_window_set_modal (GTK_WINDOW (orientdialog), TRUE);
  gtk_signal_connect (GTK_OBJECT (orientdialog), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);

  // Pack it vertically
  vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
  gtk_container_add (GTK_CONTAINER (orientdialog), vbox);
  gtk_widget_show (vbox);

  explanation = 
    gtk_label_new ("Set the orientation of the front of the chart:");
  gtk_misc_set_alignment (GTK_MISC (explanation), (float)0.0, (float)0.0);
  gtk_box_pack_start (GTK_BOX (vbox), explanation, FALSE, FALSE, 0);
  gtk_widget_show (explanation);

  // The entry boxes
  my_gtk_position_dialog (entrybox, vbox, starrules.ChartOrientation,
			  starrules.CelestialCoords);
  for (int i = 0; i < 6; i++)
    chart_orient_data[i] = entrybox[i];

  // set up the buttons
  my_gtk_button_bar (&OK, &defaults, &cancel, vbox);
  gtk_signal_connect (GTK_OBJECT (OK), "clicked", 
		      GTK_SIGNAL_FUNC (chart_orient_callback),
		      chart_orient_data);
  gtk_signal_connect (GTK_OBJECT (defaults), "clicked", 
		      GTK_SIGNAL_FUNC (chart_orient_defaults), NULL);
  my_gtk_buttons_connect_destroy (OK, defaults, cancel, orientdialog);

  gtk_window_set_focus (GTK_WINDOW (orientdialog), OK);
  gtk_widget_show (orientdialog);

  return;
}


/* Handler for the Chart->Star Filter menu item */

/* Callback functions */

/* chart_filter_callback(): Set the chart parameters to what was specified
 * in the dialog box and update the chart. */
void chart_filter_callback(GtkWidget *widget, gpointer chart_filter_data)
{
  double origdimmag = starrules.ChartDimmestMagnitude;
  bool origautoset = starrules.ChartAutosetDimmestMagnitude;
  GtkWidget **data = (GtkWidget **)chart_filter_data;

  for (int i = 0; i < 10; i++)
    starrules.StarClasses[i] = GTK_TOGGLE_BUTTON (data[i])->active;

  starrules.ChartDimmestMagnitude =
    myatof(gtk_entry_get_text (GTK_ENTRY (data[11])));
  starrules.ChartBrightestMagnitude =
    myatof(gtk_entry_get_text (GTK_ENTRY (data[10])));
  starrules.ChartAutosetDimmestMagnitude = 
    GTK_TOGGLE_BUTTON (data[12])->active;

  /* If the user has just turned on the magnitude autoset feature, we should
   *  autoset the dim magnitude limit, UNLESS the user has also just manually
   *  edited the dim magnitude limit. */
  if (!origautoset && starrules.ChartAutosetDimmestMagnitude
      && fabs(origdimmag - starrules.ChartDimmestMagnitude) < 0.1)
    starrules.ChartDimmestMagnitude = automagnitude(starrules.ChartRadius);

  redraw_all(FILTER_CHANGE);
  return;
}

/* chart_filter_defaults(): Restore the values handled by this dialog box
 * to their default settings and update the chart. */
void chart_filter_defaults(GtkWidget *widget, gpointer data)
{
  for (int i = 0; i < 10; i++)
    starrules.StarClasses[i] = initrules.StarClasses[i];
  starrules.ChartDimmestMagnitude = initrules.ChartDimmestMagnitude;
  starrules.ChartBrightestMagnitude = initrules.ChartBrightestMagnitude;
  starrules.ChartAutosetDimmestMagnitude
    = initrules.ChartAutosetDimmestMagnitude;

  if (starrules.ChartAutosetDimmestMagnitude)
    starrules.ChartDimmestMagnitude = automagnitude(starrules.ChartRadius);

  redraw_all(FILTER_CHANGE);
  return;
}

/* chart_filter(): This function creates the dialog box for turning stars on
 *  or off by spectral class and magnitude. */

void chart_filter()
{ 
  GtkWidget *filterdialog, *mainbox, *classframe, *magframe;
  GtkWidget *classtable, *classboxes[10], *clabel;
  GtkWidget *magtable, *magentrybox[2], *maglabel[2], *magautoset;
  GtkWidget *OK, *defaults, *cancel;
  static GtkWidget *chart_filter_data[13];
  char maglimits[2][7];

  // The window
  filterdialog = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_policy (GTK_WINDOW (filterdialog), FALSE, FALSE, TRUE);
  gtk_window_set_title (GTK_WINDOW (filterdialog), "StarPlot: Star Filter");
  gtk_window_set_modal (GTK_WINDOW (filterdialog), TRUE);
  gtk_signal_connect (GTK_OBJECT (filterdialog), "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);

  // Pack it vertically
  mainbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (mainbox), 10);
  gtk_container_add (GTK_CONTAINER (filterdialog), mainbox);
  gtk_widget_show (mainbox);

  // Frames in the window
  classframe = gtk_frame_new ("By Spectral Class");
  magframe = gtk_frame_new ("By Magnitude");
  gtk_box_pack_start (GTK_BOX (mainbox), classframe, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (mainbox), magframe, TRUE, FALSE, 0);
  gtk_widget_show (classframe);
  gtk_widget_show (magframe);

  // Create the spectral class checkboxes
  classtable = gtk_table_new (5, 4, FALSE);
  gtk_container_add (GTK_CONTAINER (classframe), classtable);
  gtk_widget_show (classtable);

  clabel = gtk_label_new ("Indicate which stellar classes to include:");
  gtk_widget_show (clabel);
  gtk_misc_set_alignment (GTK_MISC (clabel), (float)0.0, (float)0.0);
  gtk_table_attach (GTK_TABLE (classtable), clabel, 0, 4, 0, 1,
		    (GtkAttachOptions)(GTK_FILL | GTK_EXPAND),
		    (GtkAttachOptions)(GTK_FILL | GTK_EXPAND), 5, 5);

  classboxes[0] = 
    gtk_check_button_new_with_label ("Class O / Wolf-Rayet");
  classboxes[1] = gtk_check_button_new_with_label ("Class B");
  classboxes[2] = gtk_check_button_new_with_label ("Class A");
  classboxes[3] = gtk_check_button_new_with_label ("Class F");
  classboxes[4] = gtk_check_button_new_with_label ("Class G");
  classboxes[5] = gtk_check_button_new_with_label ("Class K");
  classboxes[6] = gtk_check_button_new_with_label ("Class M");
  classboxes[7] = gtk_check_button_new_with_label ("Dwarf stars");
  classboxes[9] =
    gtk_check_button_new_with_label ("Show Non-Stellar Objects");
  classboxes[8] = 
    gtk_check_button_new_with_label ("Show Unclassified Objects");

  for (int i = 0; i < 10; i++) {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (classboxes[i]),
				  starrules.StarClasses[i]);
    gtk_widget_show (classboxes[i]);
    chart_filter_data[i] = classboxes[i];
  }

  for (int i = 0; i < 4; i++) {
    gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[2*i],
			       i, i + 1, 1, 2);
    gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[2*i + 1],
			       i, i + 1, 2, 3);
  }
  gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[9], 0,2,3,4);
  gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[8], 0,2,4,5);

  // Create the magnitude text boxes
  snprintf(maglimits[0], 6, "%+2.1f", starrules.ChartBrightestMagnitude);
  snprintf(maglimits[1], 6, "%+2.1f", starrules.ChartDimmestMagnitude);
  maglimits[0][6] = maglimits[1][6] = 0;
  stripspace(maglimits[0]); stripspace(maglimits[1]);

  maglabel[0] = gtk_label_new ("Smallest (brightest) allowed magnitude: ");
  maglabel[1] = gtk_label_new ("Largest (dimmest) allowed magnitude: ");

  magtable = gtk_table_new (3, 2, FALSE);
  gtk_container_add (GTK_CONTAINER (magframe), magtable);
  gtk_widget_show (magtable);

  for (int i = 0; i < 2; i++) {
    gtk_widget_show (maglabel[i]);
    gtk_misc_set_alignment (GTK_MISC (maglabel[i]), (float)0.0, (float)0.0);

    magentrybox[i] = gtk_entry_new_with_max_length (6);
    gtk_entry_set_text (GTK_ENTRY (magentrybox[i]), maglimits[i]);
    gtk_widget_set_usize (magentrybox[i], 50, 20);
    gtk_widget_show (magentrybox[i]);

    gtk_table_attach (GTK_TABLE (magtable), maglabel[i], 0, 1, i, i + 1,
		      (GtkAttachOptions)GTK_FILL, (GtkAttachOptions)GTK_FILL,
		      5, 5);
    gtk_table_attach (GTK_TABLE (magtable), magentrybox[i], 1, 2, i, i + 1,
		      (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 5);

    chart_filter_data[10 + i] = magentrybox[i];
  }

  magautoset = gtk_check_button_new_with_label
    ("Autoset dim magnitude limit based on chart radius");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (magautoset),
				starrules.ChartAutosetDimmestMagnitude);
  gtk_table_attach (GTK_TABLE (magtable), magautoset, 0, 2, 2, 3,
		    (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 5);
  gtk_widget_show (magautoset);
  chart_filter_data[12] = magautoset;

  // set up the buttons
  my_gtk_button_bar (&OK, &defaults, &cancel, mainbox);
  gtk_signal_connect (GTK_OBJECT (OK), "clicked", 
		      GTK_SIGNAL_FUNC (chart_filter_callback),
		      chart_filter_data);
  gtk_signal_connect (GTK_OBJECT (defaults), "clicked", 
		      GTK_SIGNAL_FUNC (chart_filter_defaults), NULL);
  my_gtk_buttons_connect_destroy (OK, defaults, cancel, filterdialog);

  gtk_window_set_focus (GTK_WINDOW (filterdialog), OK);
  gtk_widget_show (filterdialog);

  return;
}
