__copyright__ = """
Copyright (C) 2005, Catalin Marinas <catalin.marinas@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License version 2 as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
"""

import sys, os, re, time, datetime, smtplib, email.Utils
from optparse import OptionParser, make_option

from stgit.commands.common import *
from stgit.utils import *
from stgit import stack, git, version, templates
from stgit.config import config


help = 'send a patch or series of patches by e-mail'
usage = """%prog [options] [<patch1>] [<patch2>] [<patch3>..<patch4>]

Send a patch or a range of patches by e-mail using the 'smtpserver'
configuration option. The From address and the e-mail format are
generated from the template file passed as argument to '--template'
(defaulting to '.git/patchmail.tmpl' or
'~/.stgit/templates/patchmail.tmpl' or or
'/usr/share/stgit/templates/patchmail.tmpl'). The To/Cc/Bcc addresses
can either be added to the template file or passed via the
corresponding command line options.

A preamble e-mail can be sent using the '--cover' and/or '--edit'
options. The first allows the user to specify a file to be used as a
template. The latter option will invoke the editor on the specified
file (defaulting to '.git/covermail.tmpl' or
'~/.stgit/templates/covermail.tmpl' or
'/usr/share/stgit/templates/covermail.tmpl').

All the subsequent e-mails appear as replies to the first e-mail sent
(either the preamble or the first patch). E-mails can be seen as
replies to a different e-mail by using the '--refid' option.

SMTP authentication is also possible with '--smtp-user' and
'--smtp-password' options, also available as configuration settings:
'smtpuser' and 'smtppassword'.

The template e-mail headers and body must be separated by
'%(endofheaders)s' variable, which is replaced by StGIT with
additional headers and a blank line. The patch e-mail template accepts
the following variables:

  %(patch)s        - patch name
  %(maintainer)s   - 'authname <authemail>' as read from the config file
  %(shortdescr)s   - the first line of the patch description
  %(longdescr)s    - the rest of the patch description, after the first line
  %(endofheaders)s - delimiter between e-mail headers and body
  %(diff)s         - unified diff of the patch
  %(diffstat)s     - diff statistics
  %(date)s         - current date/time
  %(version)s      - ' version' string passed on the command line (or empty)
  %(patchnr)s      - patch number
  %(totalnr)s      - total number of patches to be sent
  %(number)s       - empty if only one patch is sent or ' patchnr/totalnr'
  %(authname)s     - author's name
  %(authemail)s    - author's email
  %(authdate)s     - patch creation date
  %(commname)s     - committer's name
  %(commemail)s    - committer's e-mail

For the preamble e-mail template, only the %(maintainer)s, %(date)s,
%(endofheaders)s, %(version)s, %(patchnr)s, %(totalnr)s and %(number)s
variables are supported."""

options = [make_option('-a', '--all',
                       help = 'e-mail all the applied patches',
                       action = 'store_true'),
           make_option('--to',
                       help = 'add TO to the To: list',
                       action = 'append'),
           make_option('--cc',
                       help = 'add CC to the Cc: list',
                       action = 'append'),
           make_option('--bcc',
                       help = 'add BCC to the Bcc: list',
                       action = 'append'),
           make_option('--auto',
                       help = 'automatically cc the patch signers',
                       action = 'store_true'),
           make_option('--noreply',
                       help = 'do not send subsequent messages as replies',
                       action = 'store_true'),
           make_option('-v', '--version', metavar = 'VERSION',
                       help = 'add VERSION to the [PATCH ...] prefix'),
           make_option('-t', '--template', metavar = 'FILE',
                       help = 'use FILE as the message template'),
           make_option('-c', '--cover', metavar = 'FILE',
                       help = 'send FILE as the cover message'),
           make_option('-e', '--edit',
                       help = 'edit the cover message before sending',
                       action = 'store_true'),
           make_option('-s', '--sleep', type = 'int', metavar = 'SECONDS',
                       help = 'sleep for SECONDS between e-mails sending'),
           make_option('--refid',
                       help = 'use REFID as the reference id'),
           make_option('-u', '--smtp-user', metavar = 'USER',
                       help = 'username for SMTP authentication'),
           make_option('-p', '--smtp-password', metavar = 'PASSWORD',
                       help = 'username for SMTP authentication'),
           make_option('-b', '--branch',
                       help = 'use BRANCH instead of the default one'),
           make_option('-m', '--mbox',
                       help = 'generate an mbox file instead of sending',
                       action = 'store_true')]


def __get_maintainer():
    """Return the 'authname <authemail>' string as read from the
    configuration file
    """
    if config.has_option('stgit', 'authname') \
           and config.has_option('stgit', 'authemail'):
        return '%s <%s>' % (config.get('stgit', 'authname'),
                            config.get('stgit', 'authemail'))
    else:
        return None

def __parse_addresses(addresses):
    """Return a two elements tuple: (from, [to])
    """
    def __addr_list(addrs):
        m = re.search('[^@\s<,]+@[^>\s,]+', addrs);
        if (m == None):
            return []
        return [ m.group() ] + __addr_list(addrs[m.end():])

    from_addr_list = []
    to_addr_list = []
    for line in addresses.split('\n'):
        if re.match('from:\s+', line, re.I):
            from_addr_list += __addr_list(line)
        elif re.match('(to|cc|bcc):\s+', line, re.I):
            to_addr_list += __addr_list(line)

    if len(from_addr_list) == 0:
        raise CmdException, 'No "From" address'
    if len(to_addr_list) == 0:
        raise CmdException, 'No "To/Cc/Bcc" addresses'

    return (from_addr_list[0], to_addr_list)

def __send_message(smtpserver, from_addr, to_addr_list, msg, sleep,
                   smtpuser, smtppassword):
    """Send the message using the given SMTP server
    """
    try:
        s = smtplib.SMTP(smtpserver)
    except Exception, err:
        raise CmdException, str(err)

    s.set_debuglevel(0)
    try:
        if smtpuser and smtppassword:
            s.ehlo()
            s.login(smtpuser, smtppassword)

        s.sendmail(from_addr, to_addr_list, msg)
        # give recipients a chance of receiving patches in the correct order
        time.sleep(sleep)
    except Exception, err:
        raise CmdException, str(err)

    s.quit()

def __write_mbox(from_addr, msg):
    """Write an mbox like file to the standard output
    """
    r = re.compile('^From ', re.M)
    msg = r.sub('>\g<0>', msg)

    print 'From %s %s' % (from_addr, datetime.datetime.today().ctime())
    print msg
    print

def __build_address_headers(tmpl, options, extra_cc = []):
    """Build the address headers and check existing headers in the
    template.
    """
    def csv(lst):
        return reduce(lambda x, y: x + ', ' + y, lst)

    def replace_header(header, addr, tmpl):
        r = re.compile('^' + header + ':\s+.+$', re.I | re.M)
        if r.search(tmpl):
            tmpl = r.sub('\g<0>, ' + addr, tmpl, 1)
            h = ''
        else:
            h = header + ': ' + addr

        return tmpl, h

    headers = ''
    to_addr = ''
    cc_addr = ''
    bcc_addr = ''

    if options.to:
        to_addr = csv(options.to)
    if options.cc:
        cc_addr = csv(options.cc + extra_cc)
    elif extra_cc:
        cc_addr = csv(extra_cc)
    if options.bcc:
        bcc_addr = csv(options.bcc)

    # replace existing headers
    if to_addr:
        tmpl, h = replace_header('To', to_addr, tmpl)
        if h:
            headers += h + '\n'
    if cc_addr:
        tmpl, h = replace_header('Cc', cc_addr, tmpl)
        if h:
            headers += h + '\n'
    if bcc_addr:
        tmpl, h = replace_header('Bcc', bcc_addr, tmpl)
        if h:
            headers += h + '\n'

    return tmpl, headers

def __get_signers_list(msg):
    """Return the address list generated from signed-off-by and
    acked-by lines in the message.
    """
    addr_list = []

    r = re.compile('^(signed-off-by|acked-by):\s+(.+)$', re.I)
    for line in msg.split('\n'):
        m = r.match(line)
        if m:
            addr_list.append(m.expand('\g<2>'))

    return addr_list

def __build_extra_headers():
    """Build extra headers like content-type etc.
    """
    headers  = 'Content-Type: text/plain; charset=utf-8; format=fixed\n'
    headers += 'Content-Transfer-Encoding: 8bit\n'
    headers += 'User-Agent: StGIT/%s\n' % version.version

    return headers

def __build_cover(tmpl, total_nr, msg_id, options):
    """Build the cover message (series description) to be sent via SMTP
    """
    maintainer = __get_maintainer()
    if not maintainer:
        maintainer = ''

    tmpl, headers_end = __build_address_headers(tmpl, options)
    headers_end += 'Message-Id: %s\n' % msg_id
    if options.refid:
        headers_end += "In-Reply-To: %s\n" % options.refid
        headers_end += "References: %s\n" % options.refid
    headers_end += __build_extra_headers()

    if options.version:
        version_str = ' %s' % options.version
    else:
        version_str = ''

    total_nr_str = str(total_nr)
    patch_nr_str = '0'.zfill(len(total_nr_str))
    if total_nr > 1:
        number_str = ' %s/%s' % (patch_nr_str, total_nr_str)
    else:
        number_str = ''

    tmpl_dict = {'maintainer':   maintainer,
                 'endofheaders': headers_end,
                 'date':         email.Utils.formatdate(localtime = True),
                 'version':      version_str,
                 'patchnr':      patch_nr_str,
                 'totalnr':      total_nr_str,
                 'number':       number_str}

    try:
        msg = tmpl % tmpl_dict
    except KeyError, err:
        raise CmdException, 'Unknown patch template variable: %s' \
              % err
    except TypeError:
        raise CmdException, 'Only "%(name)s" variables are ' \
              'supported in the patch template'

    if options.edit:
        fname = '.stgitmail.txt'

        # create the initial file
        f = file(fname, 'w+')
        f.write(msg)
        f.close()

        # the editor
        if config.has_option('stgit', 'editor'):
            editor = config.get('stgit', 'editor')
        elif 'EDITOR' in os.environ:
            editor = os.environ['EDITOR']
        else:
            editor = 'vi'
        editor += ' %s' % fname

        print 'Invoking the editor: "%s"...' % editor,
        sys.stdout.flush()
        print 'done (exit code: %d)' % os.system(editor)

        # read the message back
        f = file(fname)
        msg = f.read()
        f.close()

    return msg.strip('\n')

def __build_message(tmpl, patch, patch_nr, total_nr, msg_id, ref_id, options):
    """Build the message to be sent via SMTP
    """
    p = crt_series.get_patch(patch)

    descr = p.get_description().strip()
    descr_lines = descr.split('\n')

    short_descr = descr_lines[0].rstrip()
    long_descr = reduce(lambda x, y: x + '\n' + y,
                        descr_lines[1:], '').lstrip()

    maintainer = __get_maintainer()
    if not maintainer:
        maintainer = '%s <%s>' % (p.get_commname(), p.get_commemail())

    if options.auto:
        extra_cc = __get_signers_list(descr)
    else:
        extra_cc = []

    tmpl, headers_end = __build_address_headers(tmpl, options, extra_cc)
    headers_end += 'Message-Id: %s\n' % msg_id
    if ref_id:
        headers_end += "In-Reply-To: %s\n" % ref_id
        headers_end += "References: %s\n" % ref_id
    headers_end += __build_extra_headers()

    if options.version:
        version_str = ' %s' % options.version
    else:
        version_str = ''

    total_nr_str = str(total_nr)
    patch_nr_str = str(patch_nr).zfill(len(total_nr_str))
    if total_nr > 1:
        number_str = ' %s/%s' % (patch_nr_str, total_nr_str)
    else:
        number_str = ''

    tmpl_dict = {'patch':        patch,
                 'maintainer':   maintainer,
                 'shortdescr':   short_descr,
                 'longdescr':    long_descr,
                 'endofheaders': headers_end,
                 'diff':         git.diff(rev1 = git_id('%s//bottom' % patch),
                                          rev2 = git_id('%s//top' % patch)),
                 'diffstat':     git.diffstat(rev1 = git_id('%s//bottom'%patch),
                                              rev2 = git_id('%s//top' % patch)),
                 'date':         email.Utils.formatdate(localtime = True),
                 'version':      version_str,
                 'patchnr':      patch_nr_str,
                 'totalnr':      total_nr_str,
                 'number':       number_str,
                 'authname':     p.get_authname(),
                 'authemail':    p.get_authemail(),
                 'authdate':     p.get_authdate(),
                 'commname':     p.get_commname(),
                 'commemail':    p.get_commemail()}
    for key in tmpl_dict:
        if not tmpl_dict[key]:
            tmpl_dict[key] = ''

    try:
        msg = tmpl % tmpl_dict
    except KeyError, err:
        raise CmdException, 'Unknown patch template variable: %s' \
              % err
    except TypeError:
        raise CmdException, 'Only "%(name)s" variables are ' \
              'supported in the patch template'

    return msg.strip('\n')

def func(parser, options, args):
    """Send the patches by e-mail using the patchmail.tmpl file as
    a template
    """
    smtpserver = config.get('stgit', 'smtpserver')

    smtpuser = None
    smtppassword = None
    if config.has_option('stgit', 'smtpuser'):
        smtpuser = config.get('stgit', 'smtpuser')
    if config.has_option('stgit', 'smtppassword'):
        smtppassword = config.get('stgit', 'smtppassword')

    applied = crt_series.get_applied()

    if options.all:
        patches = applied
    elif len(args) >= 1:
        patches = parse_patches(args, applied)
    else:
        raise CmdException, 'Incorrect options. Unknown patches to send'

    if options.smtp_password:
        smtppassword = options.smtp_password

    if options.smtp_user:
        smtpuser = options.smtp_user

    if (smtppassword and not smtpuser):
        raise CmdException, 'SMTP password supplied, username needed'
    if (smtpuser and not smtppassword):
        raise CmdException, 'SMTP username supplied, password needed'

    total_nr = len(patches)
    if total_nr == 0:
        raise CmdException, 'No patches to send'

    if options.noreply:
        ref_id = None
    else:
        ref_id = options.refid

    if options.sleep != None:
        sleep = options.sleep
    else:
        sleep = config.getint('stgit', 'smtpdelay')

    # send the cover message (if any)
    if options.cover or options.edit:
        # find the template file
        if options.cover:
            tmpl = file(options.cover).read()
        else:
            tmpl = templates.get_template('covermail.tmpl')
            if not tmpl:
                raise CmdException, 'No cover message template file found'

        msg_id = email.Utils.make_msgid('stgit')
        msg = __build_cover(tmpl, total_nr, msg_id, options)
        from_addr, to_addr_list = __parse_addresses(msg)

        # subsequent e-mails are seen as replies to the first one
        if not options.noreply:
            ref_id = msg_id

        if options.mbox:
            __write_mbox(from_addr, msg)
        else:
            print 'Sending the cover message...',
            sys.stdout.flush()
            __send_message(smtpserver, from_addr, to_addr_list, msg, sleep,
                           smtpuser, smtppassword)
            print 'done'

    # send the patches
    if options.template:
        tmpl = file(options.template).read()
    else:
        tmpl = templates.get_template('patchmail.tmpl')
        if not tmpl:
            raise CmdException, 'No e-mail template file found'

    for (p, patch_nr) in zip(patches, range(1, len(patches) + 1)):
        msg_id = email.Utils.make_msgid('stgit')
        msg = __build_message(tmpl, p, patch_nr, total_nr, msg_id, ref_id,
                              options)
        from_addr, to_addr_list = __parse_addresses(msg)

        # subsequent e-mails are seen as replies to the first one
        if not options.noreply and not ref_id:
            ref_id = msg_id

        if options.mbox:
            __write_mbox(from_addr, msg)
        else:
            print 'Sending patch "%s"...' % p,
            sys.stdout.flush()
            __send_message(smtpserver, from_addr, to_addr_list, msg, sleep,
                           smtpuser, smtppassword)
            print 'done'
