#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define MAX_AMP 1.0
#define CLIP 0.8
#define CLIP_A ((MAX_AMP - CLIP) * (MAX_AMP - CLIP))
#define CLIP_B (MAX_AMP - 2 * CLIP)

#define DECLIP_INPUT                   0
#define DECLIP_OUTPUT                  1

LADSPA_Descriptor *declipDescriptor = NULL;

typedef struct {
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data run_adding_gain;
} Declip;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return declipDescriptor;
	default:
		return NULL;
	}
}

void cleanupDeclip(LADSPA_Handle instance) {
	free(instance);
}

void connectPortDeclip(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Declip *plugin;

	plugin = (Declip *)instance;
	switch (port) {
	case DECLIP_INPUT:
		plugin->input = data;
		break;
	case DECLIP_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateDeclip(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Declip *plugin_data = (Declip *)malloc(sizeof(Declip));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runDeclip(LADSPA_Handle instance, unsigned long sample_count) {
	Declip *plugin_data = (Declip *)instance;

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;

	for (pos = 0; pos < sample_count; pos++) {
	        LADSPA_Data in = *(input++);

	        if((in < CLIP) && (in > -CLIP)) {
	                buffer_write(*(output++), in);
	        } else if (in > 0) {
	                buffer_write(*(output++), MAX_AMP - (CLIP_A / (CLIP_B + in)));
	        } else {
	                buffer_write(*(output++), -(MAX_AMP - (CLIP_A / (CLIP_B - in))));
	        }
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainDeclip(LADSPA_Handle instance, LADSPA_Data gain) {
	((Declip *)instance)->run_adding_gain = gain;
}

void runAddingDeclip(LADSPA_Handle instance, unsigned long sample_count) {
	Declip *plugin_data = (Declip *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;

	for (pos = 0; pos < sample_count; pos++) {
	        LADSPA_Data in = *(input++);

	        if((in < CLIP) && (in > -CLIP)) {
	                buffer_write(*(output++), in);
	        } else if (in > 0) {
	                buffer_write(*(output++), MAX_AMP - (CLIP_A / (CLIP_B + in)));
	        } else {
	                buffer_write(*(output++), -(MAX_AMP - (CLIP_A / (CLIP_B - in))));
	        }
	}
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	declipDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (declipDescriptor) {
		declipDescriptor->UniqueID = 1195;
		declipDescriptor->Label = strdup("declip");
		declipDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		declipDescriptor->Name =
		 strdup("Declipper");
		declipDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		declipDescriptor->Copyright =
		 strdup("GPL");
		declipDescriptor->PortCount = 2;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(2,
		 sizeof(LADSPA_PortDescriptor));
		declipDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(2,
		 sizeof(LADSPA_PortRangeHint));
		declipDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(2, sizeof(char*));
		declipDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Input */
		port_descriptors[DECLIP_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[DECLIP_INPUT] =
		 strdup("Input");
		port_range_hints[DECLIP_INPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[DECLIP_INPUT].LowerBound = -1;
		port_range_hints[DECLIP_INPUT].UpperBound = +1;

		/* Parameters for Output */
		port_descriptors[DECLIP_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[DECLIP_OUTPUT] =
		 strdup("Output");
		port_range_hints[DECLIP_OUTPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[DECLIP_OUTPUT].LowerBound = -1;
		port_range_hints[DECLIP_OUTPUT].UpperBound = +1;

		declipDescriptor->activate = NULL;
		declipDescriptor->cleanup = cleanupDeclip;
		declipDescriptor->connect_port = connectPortDeclip;
		declipDescriptor->deactivate = NULL;
		declipDescriptor->instantiate = instantiateDeclip;
		declipDescriptor->run = runDeclip;
		declipDescriptor->run_adding = runAddingDeclip;
		declipDescriptor->set_run_adding_gain = setRunAddingGainDeclip;
	}
}

void _fini() {
	int i;

	if (declipDescriptor) {
		free((char *)declipDescriptor->Label);
		free((char *)declipDescriptor->Name);
		free((char *)declipDescriptor->Maker);
		free((char *)declipDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)declipDescriptor->PortDescriptors);
		for (i = 0; i < declipDescriptor->PortCount; i++)
			free((char *)(declipDescriptor->PortNames[i]));
		free((char **)declipDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)declipDescriptor->PortRangeHints);
		free(declipDescriptor);
	}

}
