#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define FOLDOVER_DRIVE                 0
#define FOLDOVER_PUSH                  1
#define FOLDOVER_INPUT                 2
#define FOLDOVER_OUTPUT                3

LADSPA_Descriptor *foldoverDescriptor = NULL;

typedef struct {
	LADSPA_Data *drive;
	LADSPA_Data *push;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data run_adding_gain;
} Foldover;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return foldoverDescriptor;
	default:
		return NULL;
	}
}

void cleanupFoldover(LADSPA_Handle instance) {
	free(instance);
}

void connectPortFoldover(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Foldover *plugin;

	plugin = (Foldover *)instance;
	switch (port) {
	case FOLDOVER_DRIVE:
		plugin->drive = data;
		break;
	case FOLDOVER_PUSH:
		plugin->push = data;
		break;
	case FOLDOVER_INPUT:
		plugin->input = data;
		break;
	case FOLDOVER_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateFoldover(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Foldover *plugin_data = (Foldover *)malloc(sizeof(Foldover));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runFoldover(LADSPA_Handle instance, unsigned long sample_count) {
	Foldover *plugin_data = (Foldover *)instance;

	/* Drive (float value) */
	LADSPA_Data drive = *(plugin_data->drive);

	/* Skew (float value) */
	LADSPA_Data push = *(plugin_data->push);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;
	float x;

	drive += 1.0f;

	for (pos = 0; pos < sample_count; pos++) {
	  x = input[pos] * drive + push;
	  buffer_write(output[pos], 1.5f * x - 0.5f * x * x * x);
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainFoldover(LADSPA_Handle instance, LADSPA_Data gain) {
	((Foldover *)instance)->run_adding_gain = gain;
}

void runAddingFoldover(LADSPA_Handle instance, unsigned long sample_count) {
	Foldover *plugin_data = (Foldover *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Drive (float value) */
	LADSPA_Data drive = *(plugin_data->drive);

	/* Skew (float value) */
	LADSPA_Data push = *(plugin_data->push);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;
	float x;

	drive += 1.0f;

	for (pos = 0; pos < sample_count; pos++) {
	  x = input[pos] * drive + push;
	  buffer_write(output[pos], 1.5f * x - 0.5f * x * x * x);
	}
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	foldoverDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (foldoverDescriptor) {
		foldoverDescriptor->UniqueID = 1213;
		foldoverDescriptor->Label = strdup("foldover");
		foldoverDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		foldoverDescriptor->Name =
		 strdup("Foldover distortion");
		foldoverDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		foldoverDescriptor->Copyright =
		 strdup("GPL");
		foldoverDescriptor->PortCount = 4;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(4,
		 sizeof(LADSPA_PortDescriptor));
		foldoverDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(4,
		 sizeof(LADSPA_PortRangeHint));
		foldoverDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(4, sizeof(char*));
		foldoverDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Drive */
		port_descriptors[FOLDOVER_DRIVE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FOLDOVER_DRIVE] =
		 strdup("Drive");
		port_range_hints[FOLDOVER_DRIVE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FOLDOVER_DRIVE].LowerBound = 0;
		port_range_hints[FOLDOVER_DRIVE].UpperBound = 1;

		/* Parameters for Skew */
		port_descriptors[FOLDOVER_PUSH] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[FOLDOVER_PUSH] =
		 strdup("Skew");
		port_range_hints[FOLDOVER_PUSH].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[FOLDOVER_PUSH].LowerBound = 0;
		port_range_hints[FOLDOVER_PUSH].UpperBound = 1;

		/* Parameters for Input */
		port_descriptors[FOLDOVER_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[FOLDOVER_INPUT] =
		 strdup("Input");
		port_range_hints[FOLDOVER_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[FOLDOVER_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[FOLDOVER_OUTPUT] =
		 strdup("Output");
		port_range_hints[FOLDOVER_OUTPUT].HintDescriptor = 0;

		foldoverDescriptor->activate = NULL;
		foldoverDescriptor->cleanup = cleanupFoldover;
		foldoverDescriptor->connect_port = connectPortFoldover;
		foldoverDescriptor->deactivate = NULL;
		foldoverDescriptor->instantiate = instantiateFoldover;
		foldoverDescriptor->run = runFoldover;
		foldoverDescriptor->run_adding = runAddingFoldover;
		foldoverDescriptor->set_run_adding_gain = setRunAddingGainFoldover;
	}
}

void _fini() {
	int i;

	if (foldoverDescriptor) {
		free((char *)foldoverDescriptor->Label);
		free((char *)foldoverDescriptor->Name);
		free((char *)foldoverDescriptor->Maker);
		free((char *)foldoverDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)foldoverDescriptor->PortDescriptors);
		for (i = 0; i < foldoverDescriptor->PortCount; i++)
			free((char *)(foldoverDescriptor->PortNames[i]));
		free((char **)foldoverDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)foldoverDescriptor->PortRangeHints);
		free(foldoverDescriptor);
	}

}
