#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"

#define MAX_BSIZE 2000

inline int partition(LADSPA_Data array[], int left, int right);

inline void q_sort(LADSPA_Data array[], int left, int right) {
        int pivot = partition(array, left, right);

        if (left < pivot) {
                q_sort(array, left, pivot-1);
        }
        if (right > pivot) {
                q_sort(array, pivot+1, right);
        }
}

inline int partition(LADSPA_Data array[], int left, int right) {
        int pivot = array[left];

        while (left < right) {
                while (array[right] >= pivot && left < right) {
                        right--;
                }
                if (left != right) {
                        array[left] = array[right];
                        left++;
                }
                while (array[left] <= pivot && left < right) {
                        left++;
                }
                if (left != right) {
                        array[right] = array[left];
                        right--;
                }
        }
        array[left] = pivot;

        return left;
}

#define SIFTER_SIZE                    0
#define SIFTER_INPUT                   1
#define SIFTER_OUTPUT                  2

LADSPA_Descriptor *sifterDescriptor = NULL;

typedef struct {
	LADSPA_Data *size;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data *b1;
	long         b1ptr;
	LADSPA_Data *b2;
	long         b2ptr;
	LADSPA_Data *ob;
	LADSPA_Data *rc;
	LADSPA_Data run_adding_gain;
} Sifter;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return sifterDescriptor;
	default:
		return NULL;
	}
}

void activateSifter(LADSPA_Handle instance) {
	Sifter *plugin_data = (Sifter *)instance;
	LADSPA_Data *b1 = plugin_data->b1;
	long b1ptr = plugin_data->b1ptr;
	LADSPA_Data *b2 = plugin_data->b2;
	long b2ptr = plugin_data->b2ptr;
	LADSPA_Data *ob = plugin_data->ob;
	LADSPA_Data *rc = plugin_data->rc;
	b1ptr = 0;
	b2ptr = 0;
	memset(b1, 0, MAX_BSIZE * sizeof(LADSPA_Data));
	memset(b2, 0, MAX_BSIZE * sizeof(LADSPA_Data));
	memset(ob, 0, MAX_BSIZE * sizeof(LADSPA_Data));
	rc = rc;
	plugin_data->b1 = b1;
	plugin_data->b1ptr = b1ptr;
	plugin_data->b2 = b2;
	plugin_data->b2ptr = b2ptr;
	plugin_data->ob = ob;
	plugin_data->rc = rc;

}

void cleanupSifter(LADSPA_Handle instance) {
	free(instance);
}

void connectPortSifter(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Sifter *plugin;

	plugin = (Sifter *)instance;
	switch (port) {
	case SIFTER_SIZE:
		plugin->size = data;
		break;
	case SIFTER_INPUT:
		plugin->input = data;
		break;
	case SIFTER_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateSifter(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Sifter *plugin_data = (Sifter *)malloc(sizeof(Sifter));
	LADSPA_Data *b1 = NULL;
	long b1ptr;
	LADSPA_Data *b2 = NULL;
	long b2ptr;
	LADSPA_Data *ob = NULL;
	LADSPA_Data *rc = NULL;

	long i;
	float scla = (float)MAX_BSIZE * 0.5f;
	float sclb = (float)MAX_BSIZE;

	b1 = (LADSPA_Data *)calloc(MAX_BSIZE, sizeof(LADSPA_Data));
	b2 = (LADSPA_Data *)calloc(MAX_BSIZE, sizeof(LADSPA_Data));
	ob = (LADSPA_Data *)calloc(MAX_BSIZE, sizeof(LADSPA_Data));
	rc = (LADSPA_Data *)calloc(MAX_BSIZE, sizeof(LADSPA_Data));

	// Calculate raised cosine table, to build windowing function from
	for (i=0; i<MAX_BSIZE / 2; i++) {
	  rc[i] = cos((((float)i - scla) / sclb) * M_PI);
	  rc[i] *= rc[i];
	  rc[MAX_BSIZE - i] = rc[i];
	}
	rc[MAX_BSIZE / 2] = 1.0f;

	b1ptr = 0;
	b2ptr = 0;

	plugin_data->b1 = b1;
	plugin_data->b1ptr = b1ptr;
	plugin_data->b2 = b2;
	plugin_data->b2ptr = b2ptr;
	plugin_data->ob = ob;
	plugin_data->rc = rc;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runSifter(LADSPA_Handle instance, unsigned long sample_count) {
	Sifter *plugin_data = (Sifter *)instance;

	/* Sift size (float value) */
	LADSPA_Data size = *(plugin_data->size);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * b1 = plugin_data->b1;
	long b1ptr = plugin_data->b1ptr;
	LADSPA_Data * b2 = plugin_data->b2;
	long b2ptr = plugin_data->b2ptr;
	LADSPA_Data * ob = plugin_data->ob;
	LADSPA_Data * rc = plugin_data->rc;

	unsigned long pos, i;
	long bsize = f_round(LIMIT(size, 1, MAX_BSIZE));
	
	for (pos = 0; pos < sample_count; pos++) {
	        if (b1ptr >= bsize) {
	                float wstep = MAX_BSIZE / b1ptr, wpos = 0.0f;
	                q_sort(b1, 0, b1ptr);
	                for (i=0; i<b1ptr/2; i++) {
	                        ob[i] += b1[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                for (; i<b1ptr; i++) {
	                        ob[i] += b1[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                b1ptr = 0;
	                b2ptr = bsize / 2;
	        }
	
	        if (b2ptr >= bsize) {
	                float wstep = MAX_BSIZE / b2ptr, wpos = 0.0f;
	                int offset = b2ptr/2;
	
	                q_sort(b2, 0, b2ptr);
	                for (i=0; i<offset; i++) {
	                        ob[i + offset] += b2[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                for (; i<b2ptr; i++) {
	                        ob[i - offset] += b2[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                b2ptr = 0;
	        }
	
	        b1[b1ptr] = input[pos];
	        b2[b2ptr] = input[pos];
	        buffer_write(output[pos], ob[b1ptr]);
	        ob[b1ptr] = 0.0f;
	        b1ptr++;
	        b2ptr++;
	}
	
	plugin_data->b1ptr = b1ptr;
	plugin_data->b2ptr = b2ptr;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainSifter(LADSPA_Handle instance, LADSPA_Data gain) {
	((Sifter *)instance)->run_adding_gain = gain;
}

void runAddingSifter(LADSPA_Handle instance, unsigned long sample_count) {
	Sifter *plugin_data = (Sifter *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Sift size (float value) */
	LADSPA_Data size = *(plugin_data->size);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * b1 = plugin_data->b1;
	long b1ptr = plugin_data->b1ptr;
	LADSPA_Data * b2 = plugin_data->b2;
	long b2ptr = plugin_data->b2ptr;
	LADSPA_Data * ob = plugin_data->ob;
	LADSPA_Data * rc = plugin_data->rc;

	unsigned long pos, i;
	long bsize = f_round(LIMIT(size, 1, MAX_BSIZE));
	
	for (pos = 0; pos < sample_count; pos++) {
	        if (b1ptr >= bsize) {
	                float wstep = MAX_BSIZE / b1ptr, wpos = 0.0f;
	                q_sort(b1, 0, b1ptr);
	                for (i=0; i<b1ptr/2; i++) {
	                        ob[i] += b1[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                for (; i<b1ptr; i++) {
	                        ob[i] += b1[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                b1ptr = 0;
	                b2ptr = bsize / 2;
	        }
	
	        if (b2ptr >= bsize) {
	                float wstep = MAX_BSIZE / b2ptr, wpos = 0.0f;
	                int offset = b2ptr/2;
	
	                q_sort(b2, 0, b2ptr);
	                for (i=0; i<offset; i++) {
	                        ob[i + offset] += b2[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                for (; i<b2ptr; i++) {
	                        ob[i - offset] += b2[i] * rc[f_round(wpos)];
	                        wpos += wstep;
	                }
	                b2ptr = 0;
	        }
	
	        b1[b1ptr] = input[pos];
	        b2[b2ptr] = input[pos];
	        buffer_write(output[pos], ob[b1ptr]);
	        ob[b1ptr] = 0.0f;
	        b1ptr++;
	        b2ptr++;
	}
	
	plugin_data->b1ptr = b1ptr;
	plugin_data->b2ptr = b2ptr;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	sifterDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (sifterDescriptor) {
		sifterDescriptor->UniqueID = 1210;
		sifterDescriptor->Label = strdup("sifter");
		sifterDescriptor->Properties =
		 0;
		sifterDescriptor->Name =
		 strdup("Signal sifter");
		sifterDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		sifterDescriptor->Copyright =
		 strdup("GPL");
		sifterDescriptor->PortCount = 3;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(3,
		 sizeof(LADSPA_PortDescriptor));
		sifterDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(3,
		 sizeof(LADSPA_PortRangeHint));
		sifterDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(3, sizeof(char*));
		sifterDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Sift size */
		port_descriptors[SIFTER_SIZE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[SIFTER_SIZE] =
		 strdup("Sift size");
		port_range_hints[SIFTER_SIZE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[SIFTER_SIZE].LowerBound = 1;
		port_range_hints[SIFTER_SIZE].UpperBound = 2000;

		/* Parameters for Input */
		port_descriptors[SIFTER_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[SIFTER_INPUT] =
		 strdup("Input");
		port_range_hints[SIFTER_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[SIFTER_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[SIFTER_OUTPUT] =
		 strdup("Output");
		port_range_hints[SIFTER_OUTPUT].HintDescriptor = 0;

		sifterDescriptor->activate = activateSifter;
		sifterDescriptor->cleanup = cleanupSifter;
		sifterDescriptor->connect_port = connectPortSifter;
		sifterDescriptor->deactivate = NULL;
		sifterDescriptor->instantiate = instantiateSifter;
		sifterDescriptor->run = runSifter;
		sifterDescriptor->run_adding = runAddingSifter;
		sifterDescriptor->set_run_adding_gain = setRunAddingGainSifter;
	}
}

void _fini() {
	int i;

	if (sifterDescriptor) {
		free((char *)sifterDescriptor->Label);
		free((char *)sifterDescriptor->Name);
		free((char *)sifterDescriptor->Maker);
		free((char *)sifterDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)sifterDescriptor->PortDescriptors);
		for (i = 0; i < sifterDescriptor->PortCount; i++)
			free((char *)(sifterDescriptor->PortNames[i]));
		free((char **)sifterDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)sifterDescriptor->PortRangeHints);
		free(sifterDescriptor);
	}

}
