#!/usr/bin/perl -w
#
# Author:  Petter Reinholdtsen <pere@hungry.com>
# Date:    2002-04-02
# License: GPL
#
# Convert HW databases from one format to another.
# Should support the following formats:
#  Debian xviddetect
#  RedHat kudzu
#  RedHat Xconfigurator
#  Mandrake harddrake / libdetect

# Debian xviddetect /usr/share/xviddetect/
#  pci.ids
#  video.ids

# Mandrake Harddrake + libdetect /usr/share/detect/
#  isa.lst
#  pci.lst
#  pcmcia.lst
#  usb.lst

# RedHat kudzu /usr/share/kudzu/
#  pcitable
#  printertable (from http://www.picante.com/~gtaylor/pht/printer_list.cgi)

# RedHat Xconfigurator /usr/X11R6/share/Xconfigurator/
#  MonitorsDB
#  /usr/X11R6/lib/X11/Cards (?)
#
# Linux kernel > 2.4.17 hotplug pcimaps
#  /lib/modules/<version>/modules.pcimap
#
# pciutils (card names) (Latest on http://pciids.sourceforge.net/pci.ids.gz)
#  /usr/share/misc/pci.ids
use strict;

use vars qw(%pci $debug);

# $debug = 1;

sub load_libdetect_pci {
    my ($filename) = @_;

    my $vendorid;

    open(PCIFILE, "<$filename") || die "Unable to read $filename";
    while (<PCIFILE>) {
        chomp;

        # Remove comments
        s/^\#.+$//;

        # Skip empty lines
        next if /^\s*$/;

        if (m/^(\S{2,})\s+(.+)$/) {
            # Vendor info
            my ($id, $desc) = ($1, $2);

            $vendorid = $id;

            $pci{'vendor'}{$vendorid}{'desc'} = $desc;
        } elsif (m/^\t(\S+)\s+(.+)$/) {
            # Card info - including the vendor ID
            my ($id, $rest) = ($1, $2);

            my @f = split(/\t/, $rest);

            # Remove space at head and tail of module name
            $f[1] =~ s/^\s+(\S+)\s+$/$1/ if defined $f[1];


            # Remove or translate XFree86 server info
            if (defined $f[1] && $f[1] =~ /Card:/) {
                $f[1] = undef;
            }

            $pci{'vendor'}{$vendorid}{'cards'}{"$vendorid$id"} = $f[2];
            if ( ! exists $pci{'card'}{"$vendorid$id"}{'class'} && defined $f[0] &&
		 "unknown" ne $f[0]) {
                $pci{'card'}{"$vendorid$id"}{'class'}  = $f[0];
            }
            if ( ! exists $pci{'card'}{"$vendorid$id"}{'driver'} && defined $f[1] &&
		 $f[1] ne "unknown") {
                $pci{'card'}{"$vendorid$id"}{'driver'} = $f[1];
            }
            if ( ! exists $pci{'card'}{"$vendorid$id"}{'desc'} && defined $f[2]) {
                $pci{'card'}{"$vendorid$id"}{'desc'}   = $f[2];
            }
        }
    }
    close(PCIFILE);
}

# Load the database from xviddetect
sub load_xviddetect_info {
    my ($pcifile, $videofile) = @_;

    # First load the PCI information
    # This uses almost the same format as libdetect

    my $type; # card or device

    my $tmp;  # Store the current superclass/vendor

    open(PCIFILE, "<$pcifile") || die "Unable to read $pcifile";
    while (<PCIFILE>) {
        chomp;

        # Remove comments
        s/\#.+$//;

        # Skip empty lines
        next if /^\s*$/;

        if (m/^(\S{2,})\s+(.+)$/) {
            # Vendor info
            my ($id, $desc) = ($1, $2);
            $type = 'card';
            $id = "\L$id";

            if ( ! defined $pci{'vendor'}{$id}{'desc'} ) {
                $pci{'vendor'}{$id}{'desc'} = $desc;
            }

            $tmp = $id;
        } elsif (m/^(.) (..)\s+(.+)$/) {
            # Device clases
            my ($foo, $bar, $desc) = ($1, $2, $3);
            $type = 'device';

            my $id = "$foo|$bar";

            $pci{'class'}{$id} = $desc;

            $tmp = $id;
        } elsif (m/^\t(\S+)\s+(.+)$/) {
            # Card info - using the vendor ID or
            # Device subclass using the device class id
            my ($id, $desc) = ($1, $2);

            # Add the vendor id
            $id = "\L$tmp$id";

            if ("card" eq $type) {
                $pci{'vendor'}{$tmp}{'cards'}{$id} = $desc;
                
                if ( ! defined $pci{'card'}{$id}{'desc'} ) {
                    $pci{'card'}{$id}{'desc'}   = $desc;
                }
#               $pci{'card'}{$id}{'class'}  = "unknown";
#               $pci{'card'}{$id}{'driver'} = "unknown";
            } elsif ("device" eq $type) {
            } else {
                die "Unknown type";
            }
        }
    }
    close(PCIFILE);


    # Then load the video file
    open(VIDEOFILE, "<$videofile") || die "Unable to read $videofile";
    while (<VIDEOFILE>) {
        chomp;

        # Remove comments
        s/\#.+$//;

        # Skip empty lines
        next if /^\s*$/;

        my ($v3server, $arch, $id, $vendordesc, $carddesc) = split(/\|/);
        $id = "\L$id";

        if (defined $v3server && $v3server) {
            $pci{'card'}{$id}{'v4server'}{$arch} = $v3server;
            $pci{'card'}{$id}{'class'}    = "video";
            if ("all" eq $arch
                && defined $pci{'card'}{$id}{'driver'}
                && $pci{'card'}{$id}{'driver'} !~ /^Server:XFree86/ ) {
                $pci{'card'}{$id}{'driver'} = "Server:XF86_\U$v3server";
            }
        }
    }
    close(VIDEOFILE);
}

# This uses almost the same format as libdetect
sub load_pciutils_names {
    my ($filename) = @_;

    my $type; # card or device

    my $tmp;  # Store the current superclass/vendor

    open(PCIFILE, "<$filename") || die "Unable to read $filename";
    while (<PCIFILE>) {
        chomp;

        # Remove comments
        s/^\#.+$//;

        # Skip empty lines
        next if /^\s*$/;

        if (m/^(\S{2,})\s+(.+)$/) {
            # Vendor info
            my ($id, $desc) = ($1, $2);
            $type = 'card';

            if ( ! defined $pci{'vendor'}{$id}{'desc'} && defined $desc) {
                $pci{'vendor'}{$id}{'desc'} = $desc;
            }

            $tmp = $id;
        } elsif (m/^(.) (..)\s+(.+)$/) {
            # Device clases
            my ($foo, $bar, $desc) = ($1, $2, $3);
            $type = 'device';

            my $id = "$foo|$bar";

            $pci{'class'}{$id} = $desc;

            $tmp = $id;
        } elsif (m/^\t(\S+)\s+(.+)$/) {
            # Card info - using the vendor ID or
            # Device subclass using the device class id
            my ($id, $desc) = ($1, $2);

            # Add the vendor id
            $id = "$tmp$id";

            if ("card" eq $type) {
                $pci{'vendor'}{$tmp}{'cards'}{"$tmp$id"} = $desc;
                
                if ( ! defined $pci{'card'}{"$tmp$id"}{'desc'} && defined $desc) {
                    $pci{'card'}{"$tmp$id"}{'desc'}   = $desc;
                }
            } elsif ("device" eq $type) {
            } else {
                die "Unknown type";
            }
        }
    }
    close(PCIFILE);
}

sub load_redhat_info {
    my ($pcifilename, $xc_filename, $replace) = @_;

    # Load XFree86 module and server info
    my ($name, $server, $driver) = ("", "", "");
    my %xf86_info;
    open(CARDSFILE, "< $xc_filename") || die "Unable to read $xc_filename";
    while (<CARDSFILE>) {
        chomp;

        if (/^NAME (.+)$/) {
            if ($name) {
                my $info;
                if ($driver) {
                    $info = "Server:XFree86($driver)";
                } elsif ($server) {
                    $info = "Server:XF86_$server";
                } else {
                    $info = ""; # unsupported
                }

                print "$info\n" if $debug;
                $xf86_info{$name} = $info;
            }

            $name   = $1;
            $server = "";
            $driver = "";
        }
        $server = $1 if (/^SERVER (.+)$/);
        $driver = $1 if (/^DRIVER (.+)$/);
        if (/^SEE (.+)$/) {
            if (exists $xf86_info{$1}) {
                $xf86_info{$name} = $xf86_info{$1};
                $name = "";
                $server = "";
                $driver = "";
            }
        }
    }
    close(CARDSFILE);

    # Then load kudzu PCI table, using the Cards info to fill in the
    # XFree86 info.
    open(PCIFILE, "<$pcifilename") || die "Unable to read $pcifilename";
    while (<PCIFILE>) {
        chomp;

        # Remove comments
        s/^\#.+$//;

        # Skip empty lines
        next if /^\s*$/;

# 0x1011  0x0046  0x9005  0x1365  "aacraid"  "DEC|Dell PowerEdge RAID Control
# 0x1002  0x4144  "Card:ATI Radeon 9500 Pro"	"ATI|Radeon R300 AD [Radeon 9500 Pro]"

        my ($vendorid, $rest) = m/^0x(\S+)\s+(\S+.+)$/;
        $vendorid = "\L$vendorid";

        my @cardids;
        while ($rest =~ m/^0x(\S+)\s+(\S+.+)$/) {
            push(@cardids, "\L$1");
            $rest = $2;
        }

        my ($driverinfo, $desc) = $rest =~ m/^\"(.*)\"\s+\"(.*)\"$/;

        my $vendordesc;
        if ($desc =~ /^(.+)\|(.+)$/) {
            $vendordesc = $1;
            $desc = $2;
        }

        print "D1: $driverinfo\n" if $debug;
        if ($driverinfo =~ m/Card:(.+)/) {
            $driverinfo = $xf86_info{$1};
        }
        print "D2: $driverinfo\n" if $debug;

        if ( ! $pci{'vendor'}{$vendorid}{'desc'}
             && defined $vendordesc && "" ne $vendordesc ) {
            $pci{'vendor'}{$vendorid}{'desc'} = $vendordesc;
        }

        my $cardid;
        for $cardid (@cardids) {

            my $id = "$vendorid$cardid";

            $pci{'vendor'}{$vendorid}{'cards'}{$id} = $desc;
            if ( ! exists $pci{'card'}{$id}{'desc'} && defined $desc ) {
                $pci{'card'}{$id}{'desc'}   = $desc;
            }
	    if (defined $driverinfo && "" ne $driverinfo
		&& "unknown" ne $driverinfo ) {
		if ( ! exists $pci{'card'}{$id}{'driver'}) {
		    $pci{'card'}{$id}{'driver'} = "$driverinfo";
		} else {
		    if ($replace &&
			$pci{'card'}{$id}{'driver'} ne $driverinfo) {
			my $oldmodule = $pci{'card'}{$id}{'driver'};
			print STDERR "Replacing $id '$oldmodule' -> '$driverinfo'\n";
			$pci{'card'}{$id}{'driver'} = "$driverinfo";
		    }
		}
	    }
        }
    }
    close(PCIFILE);
}

## pci module         vendor     device     subvendor  subdevice  class      class_mask driver_data
#cciss                0x00000e11 0x0000b060 0x00000e11 0x00004070 0x00000000 0x00000000 0x00000000
#cciss                0x00000e11 0x0000b178 0x00000e11 0x00004080 0x00000000 0x00000000 0x00000000
sub load_kernel_modules {
    my ($filename) = @_;
    open(PCIFILE, "< $filename") || die "Unable to open $filename";
    while (<PCIFILE>) {
        chomp;
        next if (/^\#/);

        my ($module, $vendor, $device, $subvendor, $subdevice, $class,
            $class_mask, $driver_data) = split(/\s+/);
        $vendor =~ s/0x0000//;
        $device =~ s/0x0000//;
        my $cardid = "$vendor$device";

        # Avoid video drivers, because I give the kernel modules less
        # priority than the XFree86 info.
        if (defined $pci{'card'}{$cardid}{'class'} &&
            "video" ne $pci{'card'}{$cardid}{'class'}) {
	    next if ($module eq "eepro100");
            my $oldmodule = $pci{'card'}{$cardid}{'driver'};
#	    next if ($oldmodule);
            if ($oldmodule && $oldmodule ne $module) {
                print STDERR "Replacing $cardid '$oldmodule' -> '$module'\n";
            }
            $pci{'card'}{$cardid}{'driver'} = $module;
        }
    }
    close(PCIFILE);
}

sub save_libdetect_pci {
    my ($filename) = @_;
    open(PCIFILE, ">$filename") || die "Unable to write $filename";
    my $vendor;
    for $vendor (sort keys %{$pci{'vendor'}}) {
        my $vendordesc = $pci{'vendor'}{$vendor}{'desc'} || "unknown";
        print PCIFILE "$vendor $vendordesc\n";

        if (exists $pci{'vendor'}{$vendor}{'cards'}) {
            my $cardid;
            for $cardid (sort keys %{$pci{'vendor'}{$vendor}{'cards'}}) {
                my $class  = $pci{'card'}{$cardid}{'class'} || "unknown";
                my $driver = $pci{'card'}{$cardid}{'driver'} || "unknown";
                my $desc   = $pci{'card'}{$cardid}{'desc'} || "unknown";
                $cardid =~ s/^$vendor//;
                print PCIFILE "\t$cardid\t$class\t$driver\t$desc\n";
            }
        }
    }
    close(PCIFILE);
}

sub save_pciutils_names {
    my ($filename) = @_;
    open(PCIFILE, ">$filename") || die "Unable to write $filename";
    my $vendor;
    for $vendor (sort keys %{$pci{'vendor'}}) {
        my $vendordesc = $pci{'vendor'}{$vendor}{'desc'} || "unknown";
        print PCIFILE "$vendor  $vendordesc\n";

        if (exists $pci{'vendor'}{$vendor}{'cards'}) {
            my $cardid;
            for $cardid (sort keys %{$pci{'vendor'}{$vendor}{'cards'}}) {
                my $desc   = $pci{'card'}{$cardid}{'desc'} || "unknown";
                $cardid =~ s/^$vendor//;
                printf PCIFILE "\t$cardid  $desc\n";
            }
        }
    }
    close(PCIFILE);
}

load_pciutils_names("pci.ids");

load_libdetect_pci("pci.lst");

#load_pciutils_names("/usr/share/misc/pci.ids"); # from the pciutils package
#load_pciutils_names("pciutils-pci.ids");
#load_pciutils_names("short-pci.ids");

#load_libdetect_pci("detect-pci.lst");

#load_redhat_info("kudzu-pcitable", "xconfigurator-Cards", 1);

#load_xviddetect_info("xviddetect-pci.ids", "xviddetect-video.ids");

if (0) {
    my $file;
    for $file (</lib/modules/*/modules.pcimap>) {
        load_kernel_modules($file);
    }
}

save_libdetect_pci("new-pci.lst");
#save_pciutils_names("new-pci.ids");
